/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

#include "mod_dav_xdelta.h"
#include "xdfs_cpp.h"

MESSAGE_TYPE DEBUG_DAV_MESSAGE ("dav",  MCLASS_DEBUG);
#define DEBUG_DAV  MK_DEBUG_MESSAGE (DAV)

/* Need to fake this out like SVN does. */
struct dav_db {
    apr_pool_t *pool;
};

/* context needed to identify a resource */
struct dav_resource_private {
    apr_pool_t *pool;        /* memory storage pool associated with request */
    const char *pathname;    /* full pathname to resource */
};

/*
** The namespace URIs that we use. This list and the enumeration must
** stay in sync.
*/
static const char * const dav_xdelta_namespace_uris[] =
{
    "DAV:",
    "http://apache.org/dav/props/",

    NULL	/* sentinel */
};

enum {
    DAV_XDELTA_URI_DAV,		/* the DAV: namespace URI */
};

static const dav_liveprop_spec dav_xdelta_props[] =
{
    {
        DAV_XDELTA_URI_DAV,
        "creationdate",
        DAV_PROPID_creationdate,
        0
    },
    {
        DAV_XDELTA_URI_DAV,
        "getcontentlength",
        DAV_PROPID_getcontentlength,
        0
    },
    {
        DAV_XDELTA_URI_DAV,
        "getetag",
        DAV_PROPID_getetag,
        0
    },
    {
        DAV_XDELTA_URI_DAV,
        "getlastmodified",
        DAV_PROPID_getlastmodified,
        0
    },

    { 0 }	/* sentinel */
};

/* define the dav_stream structure for our use */
struct dav_stream {
    apr_pool_t *p;
    apr_file_t *f;
    const char *pathname;	/* we may need to remove it at close time */
};

#if 0
/* Note: picked up from ap_gm_timestr_822() */
/* NOTE: buf must be at least DAV_TIMEBUF_SIZE chars in size */
static void dav_format_time_xdelta (int style, apr_time_t sec, char *buf)
{
    apr_exploded_time_t tms;

    /* ### what to do if fails? */
    (void) apr_explode_gmt(&tms, sec);

    if (style == DAV_STYLE_ISO8601) {
	/* ### should we use "-00:00" instead of "Z" ?? */

	/* 20 chars plus null term */
	sprintf(buf, "%.4d-%.2d-%.2dT%.2d:%.2d:%.2dZ",
               tms.tm_year + 1900, tms.tm_mon + 1, tms.tm_mday,
               tms.tm_hour, tms.tm_min, tms.tm_sec);
        return;
    }

    /* RFC 822 date format; as strftime '%a, %d %b %Y %T GMT' */

    /* 29 chars plus null term */
    sprintf(buf,
	    "%s, %.2d %s %d %.2d:%.2d:%.2d GMT",
           apr_day_snames[tms.tm_wday],
           tms.tm_mday, apr_month_snames[tms.tm_mon],
           tms.tm_year + 1900,
           tms.tm_hour, tms.tm_min, tms.tm_sec);
}
#endif

/* --------------------------------------------------------------------
**
** REPOSITORY HOOK FUNCTIONS
*/

static apr_pool_t *__xdelta_pool;
static apr_hash_t *__xdelta_hash;
static char       *__xdelta_argv[] = { "<mod_dav_xdelta>" };

static int dav_xdelta_open_dbfs (const char *fspath, DBFS *& dbfsp)
{
    if (__xdelta_pool == NULL) {
	apr_pool_create (& __xdelta_pool, NULL);
    }

    if (__xdelta_hash == NULL) {
	__xdelta_hash = apr_hash_make (__xdelta_pool);
    }

    if ((dbfsp = (DBFS*) apr_hash_get (__xdelta_hash, fspath, APR_HASH_KEY_STRING)) == NULL) {

	int ret;

	dbfsp = new DBFS (1, __xdelta_argv);

	if ((ret = dbfsp->open (fspath, NULL, 0))) {
	    PROP_ERROR (ret) ("dbfs_open");
	    return ret;
	}

	apr_hash_set (__xdelta_hash, fspath, APR_HASH_KEY_STRING, dbfsp);
    }

    return 0;
}

static dav_error * dav_xdelta_get_resource(request_rec *r,
					   const char *root_dir,
					   const char *label,
					   int use_checked_in,
					   dav_resource **result_resource)
{
    const char *fspath = dav_xdelta_fspath (r);
    DBFS *dbfs;
    int ret;

    if (fspath == NULL) {
	return dav_new_error (r->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			      "No XdeltaFSPath directive given for this request");
    }

    if ((ret = dav_xdelta_open_dbfs (fspath, dbfs))) {
	return dav_new_error (r->pool, HTTP_INTERNAL_SERVER_ERROR, ret, dbfs_strerror (ret));
    }

    DEBUG_DAV ("get_resource: root=%s label=%s use_checked_in=%d", root_dir, label, use_checked_in);

    return dav_new_error (r->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_get_parent_resource(const dav_resource *resource,
						  dav_resource **result_parent)
{
    DEBUG_DAV ("get_parent_resource");
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static int dav_xdelta_is_same_resource (const dav_resource *res1,
					const dav_resource *res2)
{
    DEBUG_DAV ("is_same_resource");
    return 0;
}

static int dav_xdelta_is_parent_resource (const dav_resource *res1,
					  const dav_resource *res2)
{
    DEBUG_DAV ("is_parent_resource");
    return 0;
}

static dav_error * dav_xdelta_open_stream(const dav_resource *resource,
				      dav_stream_mode mode,
				      dav_stream **stream)
{
    DEBUG_DAV ("open_stream");
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_close_stream(dav_stream *stream, int commit)
{
    DEBUG_DAV ("close_stream");
    return dav_new_error (stream->p, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_read_stream(dav_stream *stream,
				      void *buf, apr_size_t *bufsize)
{
    DEBUG_DAV ("read_stream");
    return dav_new_error (stream->p, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_write_stream(dav_stream *stream,
				       const void *buf, apr_size_t bufsize)
{
    DEBUG_DAV ("write_stream");
    return dav_new_error (stream->p, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_seek_stream(dav_stream *stream, apr_off_t abs_pos)
{
    DEBUG_DAV ("seek_stream");
    return dav_new_error (stream->p, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_set_headers(request_rec *r,
				      const dav_resource *resource)
{
    DEBUG_DAV ("set_headers");
    return dav_new_error (r->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static const char * dav_xdelta_get_pathname(const dav_resource *resource,
					    void **free_handle_p)
{
    DEBUG_DAV ("get_pathname: %s", resource->info->pathname);
    return resource->info->pathname;
}

static void dav_xdelta_free_file(void *free_handle)
{
    /* nothing to free ... */
}

static dav_error * dav_xdelta_create_collection(dav_resource *resource)
{
    DEBUG_DAV ("create_collection");
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_copy_resource(const dav_resource *src,
					    dav_resource *dst,
					    int depth,
					    dav_response **response)
{
    DEBUG_DAV ("copy_resource");
    return dav_new_error (src->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_move_resource(dav_resource *src,
					    dav_resource *dst,
					    dav_response **response)
{
    DEBUG_DAV ("move_resource");
    return dav_new_error (src->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_remove_resource(dav_resource *resource,
                                          dav_response **response)
{
    DEBUG_DAV ("remove_resource");
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_walk(const dav_walk_params *params, int depth,
				   dav_response **response)
{
    DEBUG_DAV ("walk");
    return dav_new_error (params->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static const char *dav_xdelta_getetag(const dav_resource *resource)
{
    DEBUG_DAV ("get_etag");
    return "etag";
}

static dav_prop_insert
dav_xdelta_insert_prop(const dav_resource *resource,
		       int propid, dav_prop_insert what,
		       ap_text_header *phdr)
{
    DEBUG_DAV ("insert_prop");
    return what;
}

static int dav_xdelta_is_writable(const dav_resource *resource, int propid)
{
    DEBUG_DAV ("is_writeable");
    return 1;
}

static dav_error *dav_xdelta_patch_validate (const dav_resource *resource,
					     const ap_xml_elem *elem,
					     int operation,
					     void **context,
					     int *defer_to_dead)
{
    DEBUG_DAV ("patch_validate");
    // SVN: Don't need these until support for writable props
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error *dav_xdelta_patch_exec(const dav_resource *resource,
					const ap_xml_elem *elem,
					int operation,
					void *context,
					dav_liveprop_rollback **rollback_ctx)
{
    DEBUG_DAV ("patch_exec");
    // SVN: Don't need these until support for writable props
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static void dav_xdelta_patch_commit(const dav_resource *resource,
				    int operation,
				    void *context,
				    dav_liveprop_rollback *rollback_ctx)
{
    DEBUG_DAV ("patch_commit");
    // SVN: Don't need these until support for writable props
}

static dav_error *dav_xdelta_patch_rollback(const dav_resource *resource,
					int operation,
					void *context,
					dav_liveprop_rollback *rollback_ctx)
{
    DEBUG_DAV ("patch_rollback");
    // SVN: Don't need these until support for writable props
    return dav_new_error (resource->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

void dav_xdelta_gather_propsets(apr_array_header_t *uris)
{
    DEBUG_DAV ("gather_propsets");
}

int dav_xdelta_find_liveprop(const dav_resource *resource,
			     const char *ns_uri, const char *name,
			     const dav_hooks_liveprop **hooks)
{
    DEBUG_DAV ("find_liveprop");
    /* don't try to find any liveprops if this isn't "our" resource */
    if (resource->hooks != &dav_hooks_repository_xdelta)
        return 0;
    return dav_do_find_liveprop(ns_uri, name, &dav_xdelta_liveprop_group, hooks);
}

void dav_xdelta_insert_all_liveprops(request_rec *r, const dav_resource *resource,
                                 dav_prop_insert what, ap_text_header *phdr)
{
    DEBUG_DAV ("insert_all_liveprops");
    /* don't insert any liveprops if this isn't "our" resource */
    if (resource->hooks != &dav_hooks_repository_xdelta) {
        return;
    }

    if (!resource->exists) {
	/* a lock-null resource */
	/*
	** ### technically, we should insert empty properties. dunno offhand
	** ### what part of the spec said this, but it was essentially thus:
	** ### "the properties should be defined, but may have no value".
	*/
	return;
    }

    (void) dav_xdelta_insert_prop(resource, DAV_PROPID_creationdate,
				  what, phdr);
    (void) dav_xdelta_insert_prop(resource, DAV_PROPID_getcontentlength,
				  what, phdr);
    (void) dav_xdelta_insert_prop(resource, DAV_PROPID_getlastmodified,
				  what, phdr);
    (void) dav_xdelta_insert_prop(resource, DAV_PROPID_getetag,
				  what, phdr);

    /* ### we know the others aren't defined as liveprops */
}

//////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////

static dav_error * dav_xdelta_propdb_open(apr_pool_t * p, const dav_resource *resource,
                                int ro, dav_db **pdb)
{
    DEBUG_DAV ("propdb_open");
    return dav_new_error (p, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static void dav_xdelta_propdb_close(dav_db *db)
{
    DEBUG_DAV ("propdb_close");
}

static dav_error * dav_xdelta_propdb_fetch(dav_db *db, dav_datum key, dav_datum *pvalue)
{
    DEBUG_DAV ("propdb_fetch");
    return dav_new_error (db->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_propdb_store(dav_db *db, dav_datum key, dav_datum value)
{
    DEBUG_DAV ("propdb_store");
    return dav_new_error (db->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_propdb_delete(dav_db *db, dav_datum key)
{
    DEBUG_DAV ("propdb_delete");
    return dav_new_error (db->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static int dav_xdelta_propdb_exists(dav_db *db, dav_datum key)
{
    DEBUG_DAV ("propdb_exists");
    return 0;
}

static dav_error * dav_xdelta_propdb_firstkey(dav_db *db, dav_datum *pkey)
{
    DEBUG_DAV ("propdb_firstkey");
    return dav_new_error (db->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static dav_error * dav_xdelta_propdb_nextkey(dav_db *db, dav_datum *pkey)
{
    DEBUG_DAV ("propdb_nextkey");
    return dav_new_error (db->pool, HTTP_INTERNAL_SERVER_ERROR, 0,
			  "DAV_DEBUG: What's going on?");
}

static void dav_xdelta_propdb_freedatum(dav_db *db, dav_datum data)
{
}

//////////////////////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////////////////////

const dav_hooks_db dav_hooks_db_xdelta =
{
    dav_xdelta_propdb_open,
    dav_xdelta_propdb_close,
    dav_xdelta_propdb_fetch,
    dav_xdelta_propdb_store,
    dav_xdelta_propdb_delete,
    dav_xdelta_propdb_exists,
    dav_xdelta_propdb_firstkey,
    dav_xdelta_propdb_nextkey,
    dav_xdelta_propdb_freedatum,
};

const dav_hooks_repository dav_hooks_repository_xdelta =
{
    1, // Require GET handling
    dav_xdelta_get_resource,
    dav_xdelta_get_parent_resource,
    dav_xdelta_is_same_resource,
    dav_xdelta_is_parent_resource,
    dav_xdelta_open_stream,
    dav_xdelta_close_stream,
    dav_xdelta_read_stream,
    dav_xdelta_write_stream,
    dav_xdelta_seek_stream,
    dav_xdelta_set_headers,
    dav_xdelta_get_pathname,
    dav_xdelta_free_file,
    dav_xdelta_create_collection,
    dav_xdelta_copy_resource,
    dav_xdelta_move_resource,
    dav_xdelta_remove_resource,
    dav_xdelta_walk,
    dav_xdelta_getetag,
};

const dav_hooks_liveprop dav_hooks_liveprop_xdelta =
{
    dav_xdelta_insert_prop,
    dav_xdelta_is_writable,
    dav_xdelta_namespace_uris,
    dav_xdelta_patch_validate,
    dav_xdelta_patch_exec,
    dav_xdelta_patch_commit,
    dav_xdelta_patch_rollback
};

const dav_liveprop_group dav_xdelta_liveprop_group =
{
    dav_xdelta_props,
    dav_xdelta_namespace_uris,
    & dav_hooks_liveprop_xdelta
};

const dav_provider dav_xdelta_provider =
{
    & dav_hooks_repository_xdelta,
    & dav_hooks_db_xdelta,
    NULL,               /* locks */
    NULL,               /* vsn */
    NULL                /* binding */
};
