% $Id: Expr.lhs,v 1.9 2004/08/08 12:05:32 berrueta Exp $
%
% Copyright (c) 1999-2003, Wolfgang Lux
% Copyright (c) 2003-2004, Diego Berrueta
% See LICENSE for the full license.
%
\nwfilename{Expr.lhs}
\codesection{Free and bound variables in expressions}
\begin{lstlisting}

> module Expr where
> import CurrySyntax
> import Ident
> import Set(fromListSet,notElemSet)
> import Error(internalError)
> import TypeExpr

\end{lstlisting}
The compiler needs to compute the sets of free and bound variables for
various different entities. We will devote three type classes to that
purpose. The \texttt{QualExpr} class is expected to take into account
that it is possible to use a qualified name to refer to a function
defined in the current module and therefore \emph{M.x} and $x$, where
$M$ is the current module name, should be considered the same name.
However note that this is correct only after renaming all local
definitions as \emph{M.x} always denotes an entity defined at the
top-level.

The \texttt{Decl} instance of \texttt{QualExpr} returns all free
variables on the right hand side, regardless of whether they are bound
on the left hand side. This is more convenient as declarations are
usually processed in a declaration group where the set of free
variables cannot be computed independently for each declaration. Also
note that the operator in a unary minus expression is not a free
variable. This operator always refers to a global function from the
prelude.
\begin{lstlisting}

> class Expr e where
>   fv :: e -> [Ident]
> class QualExpr e where
>   qfv :: ModuleIdent -> e -> [Ident]
> class QuantExpr e where
>   bv :: e -> [Ident]

> instance Expr e => Expr [e] where
>   fv = concat . map fv
> instance QualExpr e => QualExpr [e] where
>   qfv m = concat . map (qfv m)
> instance QuantExpr e => QuantExpr [e] where
>   bv = concat . map bv

> instance QualExpr Decl where
>   qfv m (FunctionDecl _ f eqs) = qfv m eqs
>   qfv m (PatternDecl _ t rhs) = qfv m rhs
>   qfv _ _ = []

> instance QuantExpr Decl where
>   bv (TypeSig _ vs _) = vs
>   bv (EvalAnnot _ fs _) = fs
>   bv (FunctionDecl _ f _) = [f]
>   bv (ExternalDecl _ _ _ f _) = [f]
>   bv (PatternDecl _ t _) = bv t
>   bv (ExtraVariables _ vs) = vs
>   bv _ = []

> instance QualExpr Equation where
>   qfv m (Equation _ ts rhs) = [v | v <- qfv m rhs, v `notElemSet` bvs]
>     where bvs = fromListSet (bv ts)

> instance QualExpr Rhs where
>   qfv m (SimpleRhs _ e ds) =
>     [v | v <- qfv m e ++ qfv m ds, v `notElemSet` bvs]
>     where bvs = fromListSet (bv ds)
>   qfv m (GuardedRhs es ds) =
>     [v | v <- qfv m es ++ qfv m ds, v `notElemSet` bvs]
>     where bvs = fromListSet (bv ds)

> instance QualExpr CondExpr where
>   qfv m (CondExpr _ g e) = qfv m g ++ qfv m e

> instance QualExpr Expression where
>   qfv _ (Literal _) = []
>   qfv m (Variable v) = maybe [] return (localIdent m v)
>   qfv _ (Constructor _) = []
>   qfv m (Paren e) = qfv m e
>   qfv m (Typed e _) = qfv m e
>   qfv m (Tuple es) = qfv m es
>   qfv m (List es) = qfv m es
>   qfv m (ListCompr e []) = qfv m e
>   qfv m (ListCompr e (q:qs)) =
>     qfv m q ++ [v | v <- qfv m (ListCompr e qs), v `notElemSet` bvs]
>     where bvs = fromListSet (bv q)
>   qfv m (EnumFrom e) = qfv m e
>   qfv m (EnumFromThen e1 e2) = qfv m e1 ++ qfv m e2
>   qfv m (EnumFromTo e1 e2) = qfv m e1 ++ qfv m e2
>   qfv m (EnumFromThenTo e1 e2 e3) = qfv m e1 ++ qfv m e2 ++ qfv m e3
>   qfv m (UnaryMinus _ e) = qfv m e
>   qfv m (Apply e1 e2) = qfv m e1 ++ qfv m e2
>   qfv m (InfixApply e1 op e2) = qfv m op ++ qfv m e1 ++ qfv m e2
>   qfv m (LeftSection e op) = qfv m op ++ qfv m e
>   qfv m (RightSection op e) = qfv m op ++ qfv m e
>   qfv m (Lambda ts e) = [v | v <- qfv m e, v `notElemSet` bvs]
>     where bvs = fromListSet (bv ts)
>   qfv m (Let ds e) = [v | v <- qfv m ds ++ qfv m e, v `notElemSet` bvs]
>     where bvs = fromListSet (bv ds)
>   qfv m (Do []) = []
>   qfv m (Do (st:sts)) =
>     qfv m st ++ [v | v <- qfv m (Do sts), v `notElemSet` bvs]
>     where bvs = fromListSet (bv st)
>   qfv m (IfThenElse e1 e2 e3) = qfv m e1 ++ qfv m e2 ++ qfv m e3
>   qfv m (Case e alts) = qfv m e ++ qfv m alts

> instance QualExpr Statement where
>   qfv m (StmtExpr e) = qfv m e
>   qfv m (StmtDecl ds) = [v | v <- qfv m ds, v `notElemSet` bvs]
>     where bvs = fromListSet (bv ds)
>   qfv m (StmtBind t e) = qfv m e

> instance QualExpr Alt where
>   qfv m (Alt _ t rhs) = [v | v <- qfv m rhs, v `notElemSet` bvs]
>     where bvs = fromListSet (bv t)

> instance QuantExpr Statement where
>   bv (StmtExpr e) = []
>   bv (StmtBind t e) = bv t
>   bv (StmtDecl ds) = bv ds

> instance QualExpr InfixOp where
>   qfv m (InfixOp op) = qfv m (Variable op)
>   qfv _ (InfixConstr _) = []

> instance QuantExpr ConstrTerm where
>   bv (LiteralPattern _) = []
>   bv (VariablePattern v) = [v]
>   bv (ConstructorPattern c ts) = bv ts
>   bv (InfixPattern t1 op t2) = bv t1 ++ bv t2
>   bv (TuplePattern ts) = bv ts
>   bv (ListPattern ts) = bv ts
>   bv (AsPattern v t) = v : bv t
>   bv (LazyPattern t) = bv t

> instance Expr TypeExpr where
>   fv (TypeExprConstructor _) = []
>   fv (TypeExprVariable tv)
>     | tv == anonId = []
>     | otherwise = [tv]
>   fv (TypeExprApplication ty1 ty2) = fv ty1 ++ fv ty2

\end{lstlisting}
