# Copyright (c) 2003-2006 Ingeniweb SAS

# This software is subject to the provisions of the GNU General Public
# License, Version 2.0 (GPL).  A copy of the GPL should accompany this
# distribution.  THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL
# EXPRESS OR IMPLIED WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF TITLE, MERCHANTABILITY,
# AGAINST INFRINGEMENT, AND FITNESS FOR A PARTICULAR PURPOSE

# More details in the ``LICENSE`` file included in this package.

"""PloneArticleTool main class
$Id: PloneArticleTool.py,v 1.26 2006/10/05 14:08:27 encolpe Exp $
"""

# Python imports
import StringIO
import os

# Zope imports
from AccessControl import ClassSecurityInfo
from OFS.PropertyManager import PropertyManager
from OFS.SimpleItem import SimpleItem, Item
from Products.CMFCore.utils import UniqueObject
from Globals import InitializeClass
from Products.PageTemplates.PageTemplateFile import PageTemplateFile
from Acquisition import Implicit
from AccessControl import Role 

# CMF imports
from Products.CMFCore import CMFCorePermissions
from Products.CMFCore.utils import getToolByName
from Products.CMFCore.ActionProviderBase import ActionProviderBase
from Products.Archetypes.public import *
from Products.Archetypes.BaseObject import BaseObject
from Products.Archetypes.interfaces.base import IBaseUnit

# PloneArticle imports
from Products.PloneArticle.migration.Migrator import Migrator
from config import *

# CMFEditions
if HAS_CMFEDITIONS:
    from Products.PloneArticle.CMFEditionsModifier import PloneArticleV3CMFEModifier

import TypelessPloneArticleProperties

_www = os.path.join(os.path.dirname(__file__), 'www')

class PloneArticleTool(PropertyManager, UniqueObject, SimpleItem, ActionProviderBase):
    """
    PloneArticleTool tool
    """

    id = 'portal_article'
    meta_type = 'PloneArticle Tool'
    _actions = []
    security = ClassSecurityInfo()

    manage_options=(
        (
        { 'label' : 'Overview'
        , 'action' : 'manage_overview'
        },
        { 'label' : 'Migrate'
        , 'action' : 'manage_migration'
        },
##        { 'label' : 'Properties',
##          'action': 'manage_plonearticle_properties',
##          },
        ) + 
        ActionProviderBase.manage_options + 
##        PropertyManager.manage_options + 
        SimpleItem.manage_options
        )

    security.declareProtected(CMFCorePermissions.ManagePortal, 'manage_overview')
    manage_overview = PageTemplateFile('manage_overview', _www)
    
    security.declareProtected(CMFCorePermissions.ManagePortal, 'manage_migration')
    manage_migration = PageTemplateFile('manage_migration', _www)


    #
    #   ZMI methods
    #

    def __init__(self):
        # Properties to be edited by site manager
        # Properties are initialized in manage_afterAdd
        pass


    security.declarePrivate('manage_afterAdd')
    def manage_afterAdd(self, item, container):
        """Update plonearticle properties"""
        self.initProperties()


    #                                                                                   #
    #                                Properties management                              #
    #                                                                                   #

    security.declareProtected(CMFCorePermissions.ManagePortal, "getPropertySheet")
    def getPropertySheet(self, ):
        """createTemporaryUser(self, ) => temp property sheet

        This method will create a temporary prop sheet in memory
        It will return the object itself.
        """
        target = self.portal_url.getPortalObject().__of__(self)
        return TypelessPloneArticleProperties.getTypelessPloneArticleProperties(
            target,
            )

    security.declareProtected(CMFCorePermissions.ManagePortal, "setPreferences")
    def setPreferences(self, data):
        """setPreferences(self, data) => set preferences

        Data is a dictionary. We use the archetypes stuff to set things,
        by calling each mutator method.
        """
        self.updateCMFEditionsSupport(data.get('cmfeditions_support'))
        return TypelessPloneArticleProperties.updateTypelessPloneArticleProperties(
            self,
            data,
            )

    security.declarePrivate('initProperties')
    def initProperties(self):
        """Initialize properties"""
        # Default value for image
        image_types = self.getAvailableImageTypes()
        default_image_type = 'undefined'
        wished_image_types = ('Image', 'ATImage')
        
        for image_type in wished_image_types:
            if image_type in image_types:
                default_image_type = image_type
                break
        
        # Default value for attachment
        attachment_types = self.getAvailableAttachmentTypes()
        default_attachment_type = 'undefined'
        wished_attachment_types = ('PloneExFile', 'File', 'ATFile')
        
        for attachment_type in wished_attachment_types:
            if attachment_type in attachment_types:
                default_attachment_type = attachment_type
                break
        
        # Default value for link
        link_types = self.getAvailableLinkTypes()
        default_link_type = 'undefined'
        wished_link_types = ('Link', 'ATLink')
        
        for link_type in wished_link_types:
            if link_type in link_types:
                default_link_type = link_type
                break
        
        self.safeEditProperty(self, 'max_attachment_size', 0, data_type='int')
        self.safeEditProperty(self, 'max_image_size', 0, data_type='int')
        self.safeEditProperty(self, 'preview_activated', 1, data_type='boolean')
        ## self.safeEditProperty(self, 'site_text_formats', 'getAvailableTextFormats', data_type='multiple selection', new_value=['text/html',])
        self.safeEditProperty(self, 'site_models', 'getAvailableModels', data_type='multiple selection', new_value=self.getAvailableModels())
        self.safeEditProperty(self, 'allowed_image_type', 'getAvailableImageTypesWithUndefined', data_type='selection', new_value=default_image_type)
        self.safeEditProperty(self, 'allowed_image_types', 'getAvailableImageTypes', data_type='multiple selection', new_value=image_types)
        self.safeEditProperty(self, 'allowed_attachment_type', 'getAvailableAttachmentTypesWithUndefined', data_type='selection', new_value=default_attachment_type)
        self.safeEditProperty(self, 'allowed_attachment_types', 'getAvailableAttachmentTypes', data_type='multiple selection', new_value=attachment_types)
        self.safeEditProperty(self, 'allowed_link_type', 'getAvailableLinkTypesWithUndefined', data_type='selection', new_value=default_link_type)
        self.safeEditProperty(self, 'allowed_link_types', 'getAvailableLinkTypes', data_type='multiple selection', new_value=link_types)
        self.safeEditProperty(self, 'lock_support', 0, data_type='boolean')

    security.declarePublic("hasLockSupport")
    def hasLockSupport(self,):
        """Return the lock support availability
        """
        ret = getattr(self, "lock_support", 0)
        return ret

##    def setLockSupport(self, value):
##        """
##        Set the lock support feature
##        """
##        Log(LOG_DEBUG, "Setting lock support to", int(value), "on", self)
##        self.lock_support = not not int(value)
##        Log(LOG_DEBUG, "lock support set to:", self.hasLockSupport())

    security.declarePublic("hasLockSupport")
    def hasCMFEditionsSupport(self,):
        """Return the CMFEditions support availability
        """
        ret = getattr(self, 'cmfeditions_support', 0)
        return ret

    security.declareProtected(CMFCorePermissions.ManagePortal, 'updateCMFEditionsSupport')
    def updateCMFEditionsSupport(self, value, **kwargs):
        """Update CMFEditions configuration for PloneArticle

        'on_edit' version policy don't have sense for plonearticle since
        attachments, images and links modifications don't go through edit method.
        """
        if not HAS_CMFEDITIONS:
            Log(LOG_ERROR, "CMFEditions cannot be imported")
            return

        rep_tool = getToolByName(self, 'portal_repository')
        mod_tool = getToolByName(self, 'portal_modifier')
        if value != getattr(self, 'cmfeditions_support', None):
            if value:
                # Instanciate and configure PloneArticle modifier
                PloneArticleV3CMFEModifier.install(mod_tool)
                modifier = mod_tool.get('PloneArticleV3Modifier')
                modifier.edit(enabled=True,
                              condition="python: hasattr(object, 'portal_type') and object.portal_type in ('PloneArticle', 'PloneArticleMultiPage')",
                          )

                # Enable PloneArticle versioning
                rep_tool.setVersionableContentTypes(['PloneArticle', 'PloneArticleMultiPage'])
                rep_tool.manage_setTypePolicies({
                    'PloneArticle': ['version_on_revert'],
                    'PloneArticleMultiPage': ['version_on_revert'],
                })

        # finally we set value
        setattr(self, 'cmfeditions_support', value)

    security.declarePublic("getMaxAttachmentSize")
    def getMaxAttachmentSize(self):
        """ Return the max size of attachments"""
        try:
            max_attachment_size=self.max_attachment_size
        except:
            max_attachment_size=0
        return max_attachment_size

##    def setMaxAttachmentSize(self, max_attachment_size):
##        """ Set the max size of attachments"""
##        if max_attachment_size>0:
##            self.max_attachment_size=int(max_attachment_size)
##        else:
##            self.max_attachment_size=0

    security.declarePublic("getMaxImageSize")
    def getMaxImageSize(self):
        """ Return the max size of images"""
        try:
            max_image_size=self.max_image_size
        except:
            max_image_size=0
        return max_image_size

##    def setMaxImageSize(self, max_image_size):
##        """ Set the max size of images"""
##        self.max_image_size = max_image_size

    security.declarePublic("isPreviewActivated")
    def isPreviewActivated(self):
        """ Returns 1 of preview is set, otherwise 0"""
        try:
            preview_activated = self.preview_activated
        except:
            preview_activated = 1
        return preview_activated

##    def setPreviewActivated(self, value):
##        """Sets preview activation state"""
##        self.preview_activated = not not int(value)

##    def activatePreview(self):
##        """ Sets the preview """
##        self.preview_activated=1

##    def desactivatePreview(self):
##        """ Sets the preview """
##        self.preview_activated=0

##    security.declarePublic("getAvailableTextFormats")
##    def getAvailableTextFormats(self):
##        """ get the availeble text formats """
##        return ('text/structured',
##                'text/restructured',
##                'text/html',
##                'text/plain',
##                'text/plain-pre',
##                'text/python-source',)

##    security.declarePublic("getSiteTextFormats")
##    def getSiteTextFormats(self):
##        """ return the text formats for this site """
##        try:
##            site_text_formats = self.site_text_formats
##        except:
##            site_text_formats = ['html',]
##        return site_text_formats

##    def setSiteTextFormats(self, site_text_formats):
##        """ set the site text formats"""
##        checked_text_formats=[]
##        available_text_formats=self.getAvailableTextFormats()
##        for text_format in site_text_formats:
##            if text_format in available_text_formats:
##                checked_text_formats.append(text_format)
##            else:
##                raise ValueError
        
##        self.site_text_formats = site_text_formats

    # Copied from CMFPlone/migrations/migration_util.py
    security.declarePrivate("safeEditProperty")
    def safeEditProperty(self, obj, key, value, data_type='string', new_value=None):
        """ An add or edit function, surprisingly useful :) """
        
        if obj.hasProperty(key):
            for property in self._properties:
                if property['id'] == key:
                    property['data_type'] = data_type
                    if data_type in ('selection', 'multiple selection'):
                        property['select_variable'] = value
                    break
        else:
            obj._setProperty(key, value, data_type)
            
            if new_value is not None:
                obj._updateProperty(key, new_value)

    security.declarePublic("checkImageSize")
    def checkImageSize(self, image):
        image_size = len(image.read()) / 1024
        max_image_size = self.getMaxImageSize()
        if not max_image_size or image_size<=max_image_size:
            return 1
        else:
            return 0

    security.declarePublic("checkAttachmentSize")
    def checkAttachmentSize(self, attachment):
        attachment_size = len(attachment.read()) / 1024
        max_attachment_size = self.getMaxAttachmentSize()
        if not max_attachment_size or attachment_size<=max_attachment_size:
            return 1
        else:
            return 0

    security.declarePublic("getAvailableTextFormats")
    def getAvailableModelsWithTitle(self,):
        """
        getAvailableModels(self,) => list of strings

        Search portal_skins for all page templates that can be used as a model
        """
        # Plone1 Vs. Plone2 criterias
        search_term = '"plonearticle_model"'

        # Find the templates
        portal_skins = getToolByName(self, "portal_skins")
        templates = portal_skins.ZopeFind(portal_skins,
                                          obj_searchterm = search_term,
                                          search_sub = 1,)

        # Reduce the list
        models = {}
        for (path, obj) in templates:
            title = obj.title_or_id()
            id_ = obj.getId()
            if title and title != id_:
                title = "%s (%s)" % (title, id_, )
            else:
                title = id_
            models[obj.getId()] = title

        # Sort the list
        ret = models.items()
        ret.sort()
        return ret

    security.declarePublic("getAvailableModels")
    def getAvailableModels(self,):
        """
        getAvailableModels(self,) => list of strings

        Search portal_skins for all page templates that can be used as a model
        """
        # Plone1 Vs. Plone2 criterias
        search_term = '"plonearticle_model"'

        # Find the templates
        portal_skins = getToolByName(self, "portal_skins")
        templates = portal_skins.ZopeFind(portal_skins,
                                          obj_searchterm = search_term,
                                          search_sub = 1,)

        # Reduce & return the list
        models = {}
        for (path, obj) in templates:
            models[obj.getId()] = 1

        return models.keys()

    security.declarePublic("getSiteModels")
    def getSiteModels(self):
        """ return the text formats for this site """
        models=list(self.site_models)
        models.sort()
        return models

##    def setSiteModels(self, site_models):
##        """ set the site models"""
##        checked_models=[]
##        available_models=self.getAvailableModels()
##        for model in site_models:
##            if model in available_models:
##                checked_models.append(model)
##            else:
##                raise ValueError
                
##        self.site_models = site_models

    security.declarePublic("getAvailableAttachmentTypes")
    def getAvailableAttachmentTypes(self):
        """
        Get list of available attachment types
        """
        return self.getATTypes(['file'])

    security.declarePublic("getAvailableAttachmentTypesWithUndefined")
    def getAvailableAttachmentTypesWithUndefined(self):
        """
        Get list of attachment types with possibility to select undefined value
        """
        results = ['undefined']
        results.extend(self.getAvailableAttachmentTypes())
        return results

    security.declarePublic("getAllowedAttachmentTypes")
    def getAllowedAttachmentTypes(self):
        """
        Get allowed attachment types
        """
        Log(LOG_DEBUG, "Getting Allowed attach type", self.allowed_attachment_types)
        return self.allowed_attachment_types
        
    security.declarePublic("getAllowedAttachmentType")
    def getAllowedAttachmentType(self):
        """
        Get allowed attachment type
        """
        return self.allowed_attachment_type
        
    security.declarePublic("getAvailableImageTypesWithUndefined")
    def getAvailableImageTypesWithUndefined(self):
        """
        Get list of available image types with possibility to select undefined value
        """        
        results = ['undefined']
        results.extend(self.getAvailableImageTypes())
        return results
        
    security.declarePublic("getAvailableImageTypes")
    def getAvailableImageTypes(self):
        """
        Get list of available image types
        """
        return self.getATTypes(['image'])

    security.declarePublic("getAllowedImageTypes")
    def getAllowedImageTypes(self):
        """
        Get allowed image types
        """        
        return self.allowed_image_types
        
    security.declarePublic("getAllowedImageType")
    def getAllowedImageType(self):
        """
        Get allowed image type
        """
        return self.allowed_image_type
            
    security.declarePublic("getAvailableLinkTypes")
    def getAvailableLinkTypes(self):
        """
        Get list of available link types
        """        
        return self.getATTypes()

    security.declarePublic("getAvailableLinkTypesWithUndefined")
    def getAvailableLinkTypesWithUndefined(self):
        """
        Get list of link types with possibility to select undefined value
        """        
        results = ['undefined']
        results.extend(self.getATTypes(['remoteUrl']))
        return results

    security.declarePrivate("getATTypes")
    def getATTypes(self, field_filters=None):
        """
        Return AT content types implicitly addable
        """
        portal_types = getToolByName(self, 'portal_types')
        atool = getToolByName(self, 'archetype_tool')
        
        # Build dictionnary meta_type -> schema
        at_meta_types = dict([(x['meta_type'], x['schema']) for x in atool.listRegisteredTypes()])
        type_infos = portal_types.listTypeInfo()
        types = []
        
        for type_info in type_infos:
            meta_type = type_info.Metatype()
            if meta_type in at_meta_types.keys():
                schema = at_meta_types[meta_type]
                b_ok = True
                
                if field_filters:
                    b_ok = False
                    
                    for field_filter in field_filters:
                        if schema.has_key(field_filter):
                            b_ok = True
                            break
                
                if b_ok:
                    types.append(type_info.getId())
        
        return types

    security.declarePublic("getAllowedLinkTypes")
    def getAllowedLinkTypes(self):
        """
        Get allowed link types
        """
        return self.allowed_link_types

    security.declarePublic("getAllowedLinkType")
    def getAllowedLinkType(self):
        """
        Get allowed link type
        """
        return self.allowed_link_type


    #                                                                                   #
    #                                        Misc                                       #
    #                                                                                   #

    security.declarePublic("getSmallFilename")
    def getSmallFilename(self, longfn):
        """
        get a small filename from a long version
        """
        return os.path.split(longfn)[1]
        

    #                                                                                   #
    #                                Migrations management                              #
    #                                                                                   #

    
    security.declareProtected(CMFCorePermissions.ManagePortal, 'manage_migrate')
    def manage_migrate(self, REQUEST=None):
        """Migration script"""
        
        request = self.REQUEST
        options = {}
        options['dry_run'] = 0
        options['external_images'] = 0
        options['external_attachments'] = 0
        options['convert_limit'] = 0
        stdout = StringIO.StringIO()
        
        if request is not None:
            options['dry_run'] = request.get('dry_run', 0)
            options['external_images'] = request.get('external_images', 0)
            options['external_attachments'] = request.get('external_attachments', 0)
            options['convert_limit'] = int(request.get('convert_limit', 0))
        
        Migrator().migrate(self, stdout, options)
        
        if request is not None:
            message = "Migration completed."
            logs = stdout.getvalue()
            logs = '<br />'.join(logs.split('\r\n'))
            return self.manage_migration(self, manage_tabs_message = message, logs=logs)
    
    security.declarePublic('getReferenceBreadcrumbs')
    def getBreadcrumbs(self, context):
        """Get Breadcrumbs. Returns list of object. First is far from context"""
        
        request = self.REQUEST
        mtool = getToolByName(self, 'portal_membership')
        utool = getToolByName(self, 'portal_url')
        portal = utool.getPortalObject()
        portal_path = utool.getPortalPath()
        portal_path_len = len(portal_path.split('/'))
        parents = request.PARENTS
        parents = parents[:-portal_path_len]
        
        # Test permissions
        parents = [x for x in parents \
                   if mtool.checkPermission(CMFCorePermissions.View, x) \
                   and mtool.checkPermission(CMFCorePermissions.ListFolderContents, x)]
        
        parents.append(portal)
        parents.reverse()
        
        return parents
        
    security.declarePublic('getFolderContents')
    def getFolderContents(self, context):
        """Get folder contents. List Folder contents doesn't work with Plone site so fix it"""
        
        mtool = getToolByName(self, 'portal_membership')
        object_ids = context.contentIds()
        objects = []
        
        # Remove unauthorize objects
        for object_id in object_ids:
            try:
                object = getattr(context, object_id)
                
                if not mtool.checkPermission(CMFCorePermissions.View, object):
                    continue
                
                objects.append(object)
            except:
                pass # Unauthorize object
        
        return objects
    
    security.declarePublic("getFieldFilename")
    def getFieldFilename(self, instance, field):
        """Returns filename of field."""
        
        try:
            data = field.get(instance)
            
            if IBaseUnit.isImplementedBy(data) or \
                hasattr(data.aq_explicit, 'filename'):
                return data.filename
            elif hasattr(data.aq_explicit, 'getFilename'):
                return data.getFilename()
        except:
            pass # Never crash in this method
        
        return ''
    
InitializeClass(PloneArticleTool)

