/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: proxyvars.c,v 1.15 2004/07/02 10:03:33 bazsi Exp $
 *
 * Author  : Bazsi
 * Auditor : kisza, bazsi
 * Last audited version: 1.9, 1.39
 * Notes:
 *
 ***************************************************************************/

#include <zorp/proxyvars.h>
#include <zorp/proxy.h>
#include <zorp/policy.h>
#include <zorp/log.h>
#include <zorp/pysockaddr.h>
#include <zorp/dimhash.h>

/*
 * Generic attribute - kind of 'Variant' :)
 * 
 * It has a name, a type (flags), a pointer to its value (so it can be of any
 * size), methods for getting and setting the value, and a destructor.
 *
 * The 'flags' specify both the type (Z_VAR_TYPE() can be used to extract it):
 *   Z_VAR_TYPE_INT, Z_VAR_TYPE_STRING, Z_VAR_TYPE_SOCKADDR, Z_VAR_TYPE_OBJECT,
 *   Z_VAR_TYPE_HASH, Z_VAR_TYPE_METHOD, Z_VAR_TYPE_CUSTOM, Z_VAR_TYPE_DIMHASH
 *   Z_VAR_TYPE_ALIAS, Z_VAR_TYPE_OBSOLETE
 * and the allowed access methods of the attribute:
 *   Z_VAR_GET, Z_VAR_SET                 if it can be read/written
 *   Z_VAR_GET_CONFIG, Z_VAR_SET_CONFIG   if it can be read/written from the config?
 *   Z_VAR_GET_VERBOSE, Z_VAR_SET_VERBOSE FIXME: I found no occurence of these

 * The 'flags' is a binary-OR-ed combination of them.
 */
typedef struct _ZPyVarDesc
{
  gchar *name;
  guint flags;
  gpointer value;
  void (*free)(gpointer value);
  ZProxySetVarFunc setvar;
  ZProxyGetVarFunc getvar;
} ZPyVarDesc;


/**
 * z_proxy_vars_lookup:
 * @self: this
 * @name: Name of variable
 *
 * Searches for variable @name in the hash table of @self. First it tries to
 * find the variable in the session-scope hash session_vars->vars, and in the
 * case of failure tries to find it in the instance-scope hash vars.
 * This enables session-level settings (eg. remote port address, username, etc.)
 * to override the config-specified ones.
 *
 * Returns:
 * ZPyVarDesc variable if found, otherwise NULL
 */
static ZPyVarDesc *
z_proxy_vars_lookup(ZProxyVars *self, gchar *name)
{
  ZPyVarDesc *p = NULL;
  
  if (self->session_vars)
    p = g_hash_table_lookup(self->session_vars->vars, name);
  if (!p)
    p = g_hash_table_lookup(self->vars, name);
  return p;
}

/**
 * z_proxy_vars_setattr:
 * @self: this
 * @name: Name of the variable to set
 * @new: New value
 *
 * Sets the value of a variable if write access is granted for it.
 *
 * Returns:
 * 1 on success, 0 on error
 */
gint
z_proxy_vars_setattr(ZProxy *self, gchar *name, PyObject *new)
{
  ZPyVarDesc *p;

  z_enter();
  if (z_proxy_get_state(self) == ZPS_INITIAL || z_proxy_get_state(self) >= ZPS_DESTROYING)
    {
      return 0;
    }

  p = z_proxy_vars_lookup(self->vars, name);
  if (p)
    {
      if (((z_proxy_get_state(self) == ZPS_CONFIG) && (p->flags & Z_VAR_SET_CONFIG)) ||
	  ((z_proxy_get_state(self) != ZPS_CONFIG) && (p->flags & Z_VAR_SET)))
	{
	  z_leave();
	  return p->setvar(self, name, p->value, new);
	}
      else
	{
	  PyErr_SetString(PyExc_AttributeError, "SET access is not allowed for this attribute");
	  /* FIXME: correct error handling */
	  /*LOG
	    This message indicates that SET access for the given attribute is forbidden. You might be
	    trying to set a config-only attribute outside of the config event,
	    or set a read-only attribute.
	   */
	  z_log(self->session_id, CORE_POLICY, 3, "SET access is not allowed; attribute='%s'", name);
	}
    }
  z_leave();
  return 0;
}

/**
 * z_proxy_vars_getattr:
 * @self: this
 * @name: Name of the variable to get
 *
 * Gets the value of a variable if read access is granted to it.
 * NOTE: z_enter() & z_leave() is not present, as it generates a _LOT_ of messages 
 *
 * Returns:
 * NULL on error, the value otherwise
 */
PyObject *
z_proxy_vars_getattr(ZProxy *self, gchar *name)
{
  ZPyVarDesc *p;

  if (z_proxy_get_state(self) == ZPS_INITIAL || z_proxy_get_state(self) >= ZPS_DESTROYING)
    {
      return NULL;
    }

  p = z_proxy_vars_lookup(self->vars, name);

  if (p)
    {
      if (((z_proxy_get_state(self) == ZPS_CONFIG) && (p->flags & Z_VAR_GET_CONFIG)) ||
	  ((z_proxy_get_state(self) != ZPS_CONFIG) && (p->flags & Z_VAR_GET)))
	{
	  return p->getvar(self, name, p->value);
	}
      else
	{
	  PyErr_SetString(PyExc_AttributeError, "GET access is not allowed for this attribute");
	  /*LOG
	    This message indicates that GET access for the given attribute is not permitted at this time.
	   */
	  z_log(self->session_id, CORE_POLICY, 3, "GET access is not allowed; attribute='%s'", name);
	}
    }
  return NULL;
}


/******************************************************************************
 * int attributes support 
 */

/**
 * z_proxy_vars_int_get:
 * @self: not used
 * @name: not used
 * @value: (gint*) pointer to the value
 *
 * Gets the value of an integer variable
 *
 * Returns:
 * PyInt value
 */
static PyObject *
z_proxy_vars_int_get(ZProxy *self G_GNUC_UNUSED, gchar *name G_GNUC_UNUSED, gpointer value)
{
  z_enter();
  z_leave();
  return z_policy_var_build("i", *(gint *) value);
}

/**
 * z_proxy_vars_int_set:
 * @self: not used
 * @name: not used
 * @value: (gint*) pointer to the value
 * @new: New value
 *
 * Sets the value of an integer variable
 *
 * Returns:
 * 1 on success, otherwise 0
 */
static gint
z_proxy_vars_int_set(ZProxy *self G_GNUC_UNUSED, gchar *name G_GNUC_UNUSED, gpointer value, PyObject *new)
{
  z_enter();
  if (!z_policy_var_parse(new, "i", value))
    {
      z_leave();
      return 0;
    }
  z_leave();
  return 1;
}

/******************************************************************************
 * string attributes support
 */

/**
 * z_proxy_vars_string_get:
 * @self: not used
 * @name: not used
 * @value: (GString*) pointer to the value
 *
 * Gets the value of a GString variable
 *
 * Returns:
 * PyString value
 */
static PyObject *
z_proxy_vars_string_get(ZProxy *self G_GNUC_UNUSED, gchar *name G_GNUC_UNUSED, gpointer value)
{
  z_enter();
  z_leave();
  return PyString_FromString(((GString *) value)->str);
}

/**
 * z_proxy_vars_string_set:
 * @self: not used
 * @name: not used
 * @value: (GString*) pointer to the value
 * @new: New value
 *
 * Sets the value of an integer variable
 *
 * Returns:
 * 1 on success, otherwise 0
 */
static gint 
z_proxy_vars_string_set(ZProxy *self G_GNUC_UNUSED, gchar *name G_GNUC_UNUSED, gpointer value, PyObject *new)
{
  gchar *str;
  
  z_enter();
  if (!PyArg_Parse(new, "s", &str))
    {
      z_leave();
      return 0;
    }
  g_string_assign((GString *) value, str);
  z_leave();
  return 1;
}

/**
 * z_proxy_vars_string_free:
 * @value: (GString*) pointer to the value
 *
 * Deallocates a string variable
 */
static void
z_proxy_vars_string_free(gpointer value)
{
  z_enter();
  g_string_free((GString *) value, TRUE);
  z_leave();
}

/******************************************************************************
 * sockaddr attributes support
 */

/**
 * z_proxy_vars_sockaddr_get:
 * @self: not used
 * @name: not used
 * @value: (ZSockAddr**) pointer to the value
 *
 * Gets the value of a ZSockAddr variable
 *
 * Returns:
 * Zorp.SockAddr value
 */
static PyObject *
z_proxy_vars_sockaddr_get(ZProxy *self G_GNUC_UNUSED, gchar *name G_GNUC_UNUSED, gpointer value)
{
  ZSockAddr **v = value;
  PyObject *res;

  z_enter();
  res = z_py_zorp_sockaddr_new(*v);
  z_leave();
  return res;
}

/**
 * z_proxy_vars_sockaddr_set:
 * @self: not used
 * @name: not used
 * @value: (ZSockAddr**) pointer 
 * @new: New value
 *
 * Sets the value of a ZSockAddr variable
 *
 * Returns:
 * 1 on success, otherwise 0
 */
static gint 
z_proxy_vars_sockaddr_set(ZProxy *self G_GNUC_UNUSED, gchar *name G_GNUC_UNUSED, gpointer value, PyObject *new)
{
  z_enter();
  if (z_py_zorp_sockaddr_check(new))
    {
      ZSockAddr **v = value;
    
      z_sockaddr_unref(*v);
      *v = z_sockaddr_ref(((ZorpSockAddr *) new)->sa);
      z_leave();
      return 1;
    }
  else
    {
      PyErr_SetObject(PyExc_TypeError, new);
      z_leave();
      return 0;
    }
  z_leave();
}

/**
 * z_proxy_vars_sockaddr_free:
 * @value: (ZSockAddr**) pointer to the value
 *
 * Decrements the reference counter of a ZSockAddr variable
 */
static void
z_proxy_vars_sockaddr_free(gpointer value)
{
  z_enter();
  z_sockaddr_unref(*((ZSockAddr **) value));
  z_leave();
}

/******************************************************************************
 * object attributes support
 */

/**
 * z_proxy_vars_object_free:
 * @value: (PyObject**) pointer to the value
 *
 * Decrements the reference counter of a PyObject variable
 */
static void
z_proxy_vars_object_free(gpointer value)
{
  z_enter();
  Py_XDECREF(*((PyObject **) value));
  z_leave();
}

/**
 * z_proxy_vars_object_get:
 * @self: not used
 * @name: not used
 * @value: (PyObject**) pointer to the value
 *
 * Gets the value of a PyObject variable
 *
 * Returns:
 * The PyObject value
 */
static PyObject *
z_proxy_vars_object_get(ZProxy *self G_GNUC_UNUSED, gchar *name G_GNUC_UNUSED, gpointer value)
{
  PyObject *r = *((PyObject **) value);
  
  z_enter();
  Py_XINCREF(r);
  z_leave();
  return r;
}

/**
 * z_proxy_vars_object_set:
 * @self: not used
 * @name: not used
 * @value: (PyObject**) pointer to the value
 * @new: New value
 *
 * Sets the value of a PyObject variable
 *
 * Returns:
 * 1 (always succeeds)
 */
static gint 
z_proxy_vars_object_set(ZProxy *self G_GNUC_UNUSED, gchar *name G_GNUC_UNUSED, gpointer value, PyObject *new)
{
  z_enter();
  z_proxy_vars_object_free(value);
  *((PyObject **) value) = new;
  Py_XINCREF(new);
  
  z_leave();
  return 1;
}

/******************************************************************************
 * method attributes support
 */

typedef struct _ZorpMethod
{
  PyObject_HEAD
  ZProxy *proxy;
  ZProxyMethodFunc method;
} ZorpMethod;


/* FIXME?: z_py_zorp_method_type is defined in this file, so i think there's no need for 'extern' here */
extern PyTypeObject z_py_zorp_method_type;


/**
 * z_py_zorp_method_new:
 * @proxy: ZProxy instance
 * @method: Method function
 *
 * Constructor of ZorpMethod - a Python type that encapsulates a method of a class
 * derived from ZProxy
 *
 * Returns:
 * The new instance
 */
static ZorpMethod *
z_py_zorp_method_new(ZProxy *proxy, ZProxyMethodFunc method)
{
  ZorpMethod *self;
  
  self = PyObject_NEW(ZorpMethod, &z_py_zorp_method_type);
  if (!self)
    return NULL;
  self->proxy = proxy;
  self->method = method;
  return self;
}


/**
 * z_py_zorp_method_call:
 * @self: this
 * @args: Arguments to call the method with
 * @kw: not used
 *
 * Call the ZProxy method referred to by @self
 *
 * Returns:
 * The value returned by the method
 */
static PyObject *
z_py_zorp_method_call(ZorpMethod *self, PyObject *args, PyObject *kw G_GNUC_UNUSED)
{
  return self->method(self->proxy, args);
}

/**
 * z_py_zorp_method_free:
 * @self: this
 *
 * Destructor of ZorpMethod
 */
static void
z_py_zorp_method_free(ZorpMethod *self)
{
  PyObject_Del(self);
}

PyTypeObject z_py_zorp_method_type = 
{
  PyObject_HEAD_INIT(NULL)
  0,
  "Zorp method",
  sizeof(ZorpMethod),
  0,
  (destructor) z_py_zorp_method_free, 
  0,                                  /* tp_print */
  0,                                  /* tp_getattr */
  0,                                  /* tp_setattr */
  0,                                  /* tp_compare */
  0,                                  /* tp_repr */
  0,                                  /* tp_as_number */
  0,                                  /* tp_as_sequence */
  0,                                  /* tp_as_mapping */
  0,                                  /* tp_hash */
  (ternaryfunc) z_py_zorp_method_call,/* tp_call */
  0,                                  /* tp_str */
  0,                                  /* tp_getattro */
  0,                                  /* tp_setattro */
  0,                                  /* tp_as_buffer */
  0,                                  /* flags */
  "ZorpMethod class for Zorp",        /* docstring */
  0, 0, 0, 0,
  Z_PYTYPE_TRAILER
};

/**
 * z_proxy_vars_method_get:
 * @self: not used
 * @name: not used
 * @value: (PyObject*) pointer to the value
 *
 * Get the value of a ZorpMethod variable.
 * Note that this is not the same as _object_get(), since there the argument is
 * a pointer to a pointer to an object, while here only a casting to PyObject* is done.
 *
 * Returns:
 * The PyObject value
 */
static PyObject *
z_proxy_vars_method_get(ZProxy *self G_GNUC_UNUSED, gchar *name G_GNUC_UNUSED, gpointer value)
{
  PyObject *res = (PyObject *) value;
  Py_XINCREF(res);
  return res;
}

/**
 * z_proxy_vars_method_free:
 * @value: this
 *
 * Decrements the reference counter of a ZorpMethod variable
 */
static void
z_proxy_vars_method_free(gpointer value)
{
  /* z_policy_lock(z_python_global_state); */
  Py_XDECREF((PyObject *) value);
  /* z_policy_unlock(z_python_global_state); */
}

/******************************************************************************
 * hash-table attributes support
 */

typedef struct _ZorpHash
{
  PyObject_HEAD
  GHashTable *hash;
} ZorpHash;

extern PyTypeObject z_py_zorp_hash_type;

/**
 * z_py_zorp_hash_new:
 * @hash: Hash table to create a ZorpHash around
 *
 * Constructor of ZorpHash - the Python type that encapsulates a glib hash table
 *
 * Returns:
 * 
 */
static ZorpHash *
z_py_zorp_hash_new(GHashTable *hash)
{
  ZorpHash *self = PyObject_NEW(ZorpHash, &z_py_zorp_hash_type);
  
  self->hash = hash;
  return self;
}

/**
 * z_py_zorp_hash_unref_items:
 * @key: not used
 * @value: this
 * @user_data: not used
 *
 * Decrement the reference counter of a ZorpHash instance
 *
 * Returns:
 * 
 */
static gboolean
z_py_zorp_hash_unref_items(gpointer key G_GNUC_UNUSED, gpointer value, gpointer user_data G_GNUC_UNUSED)
{
  Py_XDECREF((PyObject *) value);
  return TRUE;
}

/**
 * z_py_zorp_hash_free:
 * @self: this
 *
 * Destructor of ZorpHash. Removes all items in the hash, and the hash itself, too.
 */
static void
z_py_zorp_hash_free(ZorpHash *self)
{
  g_hash_table_foreach_remove(self->hash, z_py_zorp_hash_unref_items, NULL);
  g_hash_table_destroy(self->hash);
  PyObject_Del(self);
}

/**
 * z_py_zorp_hash_subscript:
 * @self: this
 * @k: PyString key
 *
 * Performs a lookup for @k in the hash table.
 *
 * Returns:
 * The value if found, otherwise NULL
 */
static PyObject *
z_py_zorp_hash_subscript(ZorpHash *self, PyObject *k)
{
  gchar *key;
  PyObject *res;
  
  if (!PyArg_Parse(k, "s", &key))
    return NULL;
  res = g_hash_table_lookup(self->hash, key);
  if (res)
    {
      Py_INCREF(res);
      return res;
    }
  else
    {
      PyErr_SetObject(PyExc_KeyError, k);
      return NULL;
    }
}

/**
 * z_py_zorp_hash_ass_subscript:
 * @self: this
 * @u: PyString key
 * @v: PyObject value
 *
 * Insert-or-update the value @v for the key @u in the hash table.
 * If there was a previous value for the key, its reference counter
 * is decremented.
 *
 * Returns:
 * 0 on success, -1 otherwise
 */
static gint 
z_py_zorp_hash_ass_subscript(ZorpHash *self, PyObject *u, PyObject *v)
{
  gchar *key;
  PyObject *res;
  
  if (!PyArg_Parse(u, "s", &key))
    return -1;
    
  res = g_hash_table_lookup(self->hash, key);
  if (v == NULL)
    {
      /* delete item */
      if (!res)
        {
          PyErr_SetObject(PyExc_KeyError, u);
          return -1;
        }
      g_hash_table_remove(self->hash, key);
      Py_XDECREF(res);
      return 0;
    }
  else
    {
      Py_XINCREF(v);
      g_hash_table_insert(self->hash, key, v);
      Py_XDECREF(res);
      return 0;
    }
}

PyMappingMethods z_py_zorp_hash_mapping = 
{
  NULL,
  (binaryfunc) z_py_zorp_hash_subscript,
  (objobjargproc) z_py_zorp_hash_ass_subscript
};

PyTypeObject z_py_zorp_hash_type = 
{
  PyObject_HEAD_INIT(NULL)
  0,
  "Zorp hash",
  sizeof(ZorpHash),
  0,
  (destructor) z_py_zorp_hash_free, 
  0,                                  /* tp_print */
  0,                                  /* tp_getattr */
  0,                                  /* tp_setattr */
  0,                                  /* tp_compare */
  0,                                  /* tp_repr */
  0,                                  /* tp_as_number */
  0,                                  /* tp_as_sequence */
  &z_py_zorp_hash_mapping,            /* tp_as_mapping */
  0,                                  /* tp_hash */
  0,  				      /* tp_call */
  0,                                  /* tp_str */
  0,                                  /* tp_getattro */
  0,                                  /* tp_setattro */
  0,                                  /* tp_as_buffer */
  0,                                  /* flags */
  "ZorpHash class for Zorp",          /* docstring */
  0, 0, 0, 0,
  Z_PYTYPE_TRAILER
};

/**
 * z_proxy_vars_hash_get:
 * @self: not used
 * @name: not used
 * @value: (ZorpHash*) pointer to the value
 *
 * Get the value to a ZorpHash variable - cast to PyObject, just like at
 * z_proxy_vars_method_get.
 *
 * Returns:
 * The ZorpHash value
 */
static PyObject *
z_proxy_vars_hash_get(ZProxy *self G_GNUC_UNUSED, gchar *name G_GNUC_UNUSED, gpointer value)
{
  PyObject *res = (PyObject *) value;
  
  Py_XINCREF(res);
  return res;
}

/**
 * z_proxy_vars_hash_free:
 * @value: this
 *
 * Destructor of ZorpHash
 */
static void
z_proxy_vars_hash_free(gpointer value)
{
  /* z_policy_lock(z_python_global_state); */
  Py_XDECREF((PyObject *) value);
  /* z_policy_unlock(z_python_global_state); */
}

/******************************************************************************
 * Multidimensional hash table attributes support
 */

typedef struct _ZorpDimHash
{
  PyObject_HEAD
  ZDimHashTable *hash;
} ZorpDimHash;

extern PyTypeObject z_py_zorp_dimhash_type;

/**
 * z_py_zorp_dimhash_new:
 * @hash: ZDimHashTable to create a ZorpDimHash around
 *
 * Constructor of ZorpDimHash - a Python class that encapsulates a
 * ZDimHashTable.
 *
 * Returns:
 * The new instance
 */
static ZorpDimHash *
z_py_zorp_dimhash_new(ZDimHashTable *hash)
{
  ZorpDimHash *self = PyObject_NEW(ZorpDimHash, &z_py_zorp_dimhash_type);
  
  self->hash = hash;
  return self;
}

/**
 * z_py_zorp_dimhash_unref_items:
 * @value: this
 *
 * Decrements the reference counter
 *
 * Returns:
 * TRUE
 */
static gboolean
z_py_zorp_dimhash_unref_items(gpointer value)
{
  Py_XDECREF((PyObject *) value);
  return TRUE;
}

/**
 * z_py_zorp_dimhash_free:
 * @self: this
 *
 * Destructor of ZorpDimHash
 */
static void
z_py_zorp_dimhash_free(ZorpDimHash *self)
{
  z_dim_hash_table_free(self->hash, z_py_zorp_dimhash_unref_items);
  PyObject_Del(self);
}

/**
 * z_py_zorp_dimhash_subscript:
 * @self: this
 * @k: key sequence
 *
 * Assembles a composite key ("key0::key1::...:keyN") from the sequence @k and
 * looks up its value from the hash.
 *
 * Returns:
 * The value if found, otherwise NULL
 */
static PyObject *
z_py_zorp_dimhash_subscript(ZorpDimHash *self, PyObject *k)
{
  gchar **keys;
  gchar *key;
  guint keynum;
  PyObject *item;
  PyObject *stritem;
  PyObject *res;
  guint i;
  
  
  if (PyArg_Parse(k, "s", &key))
    {
      keynum = 1;
      keys = g_new0(gchar *, keynum);
      keys[0] = g_strdup(key);
    }
  else
    {
      PyErr_Clear();
      if (z_policy_seq_check(k))
        {
          keynum = z_policy_seq_length(k);
      
          keys = g_new0(gchar *, keynum);
      
          for (i = 0; i < keynum; i++)
            {
              item = z_policy_seq_getitem(k, i);
              stritem = z_policy_var_str(item);
              z_policy_var_unref(item);
          
              key = z_policy_str_as_string(stritem);
              keys[i] = g_new0(gchar, strlen(key)+1);
              strcpy(keys[i], key);
              z_policy_var_unref(stritem);
            }
        }
      else
        return NULL;
    }
  res = z_dim_hash_table_lookup(self->hash, keynum, keys);
  z_dim_hash_key_free(keynum, keys);

  if (res)
    {
      Py_INCREF(res);
      return res;
    }
  else
    {
      PyErr_SetObject(PyExc_KeyError, k);
      return NULL;
    }
}

/**
 * z_py_zorp_dimhash_ass_subscript:
 * @self: this
 * @u: key sequence
 * @v: new value
 *
 * Assemble a composite key from the sequence @u, and assign @v to it
 * in the hash, creating a new entry if the key was a new one, or replacing
 * the previous value if it wasn't.
 * 
 * Returns:
 * 0 on success, -1 on error
 */
static gint 
z_py_zorp_dimhash_ass_subscript(ZorpDimHash *self, PyObject *u, PyObject *v)
{
  gchar **keys;
  gchar *key;
  guint keynum;
  PyObject *res;
  PyObject *item;
  PyObject *stritem;
  guint i;
  
  
  if (PyArg_Parse(u, "s", &key))
    {
      keynum = 1;
      keys = g_new0(gchar *, keynum);
      keys[0] = g_new0(gchar, strlen(key)+1);
      strcpy(keys[0], key);
    }
  else
    {
      PyErr_Clear();
      if (z_policy_seq_check(u))
        {
          keynum = z_policy_seq_length(u);
      
          keys = g_new0(gchar *, keynum);
      
          for (i = 0; i < keynum; i++)
            {
              item = z_policy_seq_getitem(u, i);
              stritem = z_policy_var_str(item);
              z_policy_var_unref(item);
          
              key = z_policy_str_as_string(stritem);
              keys[i] = g_new0(gchar, strlen(key)+1);
              strcpy(keys[i], key);
              z_policy_var_unref(stritem);
            }
        }
      else
        return -1;
    }
  res = z_dim_hash_table_lookup(self->hash, keynum, keys);

  if (v == NULL)
    {
      /* delete item */
      if (!res)
        {
          PyErr_SetObject(PyExc_KeyError, u);
          z_dim_hash_key_free(keynum, keys);
          return -1;
        }
      z_dim_hash_table_delete(self->hash, keynum, keys, z_py_zorp_dimhash_unref_items);
      z_dim_hash_key_free(keynum, keys);
      Py_XDECREF(res);
      return 0;
    }
  else
    {
      Py_XINCREF(v);
      z_dim_hash_table_insert(self->hash, v, keynum, keys);
      z_dim_hash_key_free(keynum, keys);
      Py_XDECREF(res);
      return 0;
    }
}

PyMappingMethods z_py_zorp_dimhash_mapping = 
{
  NULL,
  (binaryfunc) z_py_zorp_dimhash_subscript,
  (objobjargproc) z_py_zorp_dimhash_ass_subscript
};

PyTypeObject z_py_zorp_dimhash_type = 
{
  PyObject_HEAD_INIT(NULL)
  0,
  "Zorp Multidimensional hash",
  sizeof(ZorpDimHash),
  0,
  (destructor) z_py_zorp_dimhash_free, 
  0,                                  /* tp_print */
  0,                                  /* tp_getattr */
  0,                                  /* tp_setattr */
  0,                                  /* tp_compare */
  0,                                  /* tp_repr */
  0,                                  /* tp_as_number */
  0,                                  /* tp_as_sequence */
  &z_py_zorp_dimhash_mapping,        /* tp_as_mapping */
  0,                                  /* tp_hash */
  0,  				      /* tp_call */
  0,                                  /* tp_str */
  0,                                  /* tp_getattro */
  0,                                  /* tp_setattro */
  0,                                  /* tp_as_buffer */
  0,                                  /* flags */
  "ZorpDimHash class for Zorp",       /* docstring */
  0, 0, 0, 0,
  Z_PYTYPE_TRAILER
};

/**
 * z_proxy_vars_dimhash_get:
 * @self: not used
 * @name: not used
 * @value: (ZorpDimHash*) pointer to the value
 *
 * Get the value to a ZorpDimHash variable - cast to PyObject.
 *
 * Returns:
 * The value
 */
static PyObject *
z_proxy_vars_dimhash_get(ZProxy *self G_GNUC_UNUSED, gchar *name G_GNUC_UNUSED, gpointer value)
{
  PyObject *res = (PyObject *) value;
  
  Py_XINCREF(res);
  return res;
}

/**
 * z_proxy_vars_dimhash_free:
 * @value: this
 *
 * Destructor of ZorpDimHash
 */
static void
z_proxy_vars_dimhash_free(gpointer value)
{
  /* z_policy_lock(z_python_global_state); */
  Py_XDECREF((PyObject *) value);
  /* z_policy_unlock(z_python_global_state); */
}


/******************************************************************************
 * alias support
 */

/**
 * z_proxy_vars_alias_get:
 * @self: this
 * @name: not used
 * @value: Name of the alias
 *
 * Gets the real name of an alias-variable
 *
 * Returns:
 * The real name of the variable 
 */
static PyObject *
z_proxy_vars_alias_get(ZProxy *self, gchar *name G_GNUC_UNUSED, gpointer value)
{
  return PyObject_GetAttrString(self->handler, (char *) value);
}

/**
 * z_proxy_vars_alias_set:
 * @self: this
 * @name: not used
 * @value: Name of the alias
 * @new: Name of the real variable
 *
 * Sets the alias @value to refer to @new
 *
 * Returns:
 * ???
 */
static gint
z_proxy_vars_alias_set(ZProxy *self, gchar *name G_GNUC_UNUSED, gpointer value, PyObject *new)
{
  return PyObject_SetAttrString(self->handler, (char *) value, new);
}

/******************************************************************************
 * obsolete support
 */

/**
 * z_proxy_vars_obsolete_get:
 * @self: this
 * @name: not used
 * @value: Name of the alias
 *
 * Gets the real name of an obsolete-alias-variable
 * Obsolete variables differ only in one thing from aliases: they produce log
 * entries on every access to warn the user that this option will be removed.
 *
 * Returns:
 * The real name of the variable 
 */
static PyObject *
z_proxy_vars_obsolete_get(ZProxy *self, gchar *name, gpointer value)
{
  /*LOG
    This message indicates that the use of this variable is obsolete, check the reference guide for further details.
   */
  z_log(self->session_id, CORE_POLICY, 3, "This variable is obsolete; variable='%s'", name);
  return PyObject_GetAttrString(self->handler, (char *) value);
}

/**
 * z_proxy_vars_obsolete_set:
 * @self: this
 * @name: not used
 * @value: Name of the alias
 * @new: Name of the real variable
 *
 * Sets the obsolete-alias @value to refer to @new
 *
 * Returns:
 * ???
 */
static gint
z_proxy_vars_obsolete_set(ZProxy *self, gchar *name G_GNUC_UNUSED, gpointer value, PyObject *new)
{
  /*LOG
    This message indicates that the use of this variable is obsolete, check the reference guide for further details.
   */
  z_log(self->session_id, CORE_POLICY, 3, "This variable is obsolete; variable='%s'", name);
  return PyObject_SetAttrString(self->handler, (char *) value, new);
}


/******************************************************************************
 * attribute support
 */


/**
 * z_proxy_vars_var_new:
 * @self: GHashTable to insert the new ZPyVarDesc into
 * @name: name of the variable
 * @flags: type&access flags, see at the definition of ZPyVarDesc
 * @ ...: type-specific arg list
 *
 * Constructor of ZPyVarDesc, create new instance, set its getter and setter
 * function, and add it to the hash table @self
 */
void
z_proxy_vars_var_new(GHashTable *self, gchar *name, guint flags, ...)
{
  ZPyVarDesc *e = g_new0(ZPyVarDesc, 1);
  va_list l;
  
  e->name = name;
  e->flags = flags;

  va_start(l, flags);
  switch (Z_VAR_TYPE(flags))
    {
    case Z_VAR_TYPE_INT:
      e->value = va_arg(l, gpointer);
      e->getvar = z_proxy_vars_int_get;
      e->setvar = z_proxy_vars_int_set;
      break;
    case Z_VAR_TYPE_STRING:
      e->value = va_arg(l, gpointer);
      e->getvar = z_proxy_vars_string_get;
      e->setvar = z_proxy_vars_string_set;
      e->free = z_proxy_vars_string_free;
      break;
    case Z_VAR_TYPE_SOCKADDR:
      e->value = va_arg(l, gpointer);
      e->getvar = z_proxy_vars_sockaddr_get;
      e->setvar = z_proxy_vars_sockaddr_set;
      e->free = z_proxy_vars_sockaddr_free;
      break;
    case Z_VAR_TYPE_OBJECT:
      e->value = va_arg(l, gpointer);
      e->getvar = z_proxy_vars_object_get;
      e->setvar = z_proxy_vars_object_set;
      e->free = z_proxy_vars_object_free;
      break;
    case Z_VAR_TYPE_METHOD:
      {
	ZProxy *proxy;

	proxy = va_arg(l, ZProxy *);
	z_python_lock();
	e->value = z_py_zorp_method_new(proxy, va_arg(l, ZProxyMethodFunc));
	z_python_unlock();
	e->free = z_proxy_vars_method_free;
	e->getvar = z_proxy_vars_method_get;
	break;
      }
    case Z_VAR_TYPE_HASH:

      z_python_lock();
      e->value = z_py_zorp_hash_new((GHashTable *) va_arg(l, gpointer));
      z_python_unlock();
      e->free = z_proxy_vars_hash_free;
      e->getvar = z_proxy_vars_hash_get;
      break;
    case Z_VAR_TYPE_CUSTOM:

      e->value = va_arg(l, gpointer);
      e->getvar = va_arg(l, ZProxyGetVarFunc);
      e->setvar = va_arg(l, ZProxySetVarFunc);
      e->free = va_arg(l, gpointer);
      break;
    case Z_VAR_TYPE_DIMHASH:
      z_python_lock();
      e->value = z_py_zorp_dimhash_new((ZDimHashTable *) va_arg(l, gpointer));
      z_python_unlock();
      e->free = z_proxy_vars_dimhash_free;
      e->getvar = z_proxy_vars_dimhash_get;
      break;
    case Z_VAR_TYPE_ALIAS:
      e->value = va_arg(l, gpointer);
      e->getvar = z_proxy_vars_alias_get;
      e->setvar = z_proxy_vars_alias_set;
      break;
    case Z_VAR_TYPE_OBSOLETE:
      e->value = va_arg(l, gpointer);
      e->getvar = z_proxy_vars_obsolete_get;
      e->setvar = z_proxy_vars_obsolete_set;
      break;
    }
  va_end(l);
  g_hash_table_insert(self, name, e);
}


/**
 * z_proxy_var_destroy:
 * @key: not used
 * @value: this
 * @user_data: not used
 *
 * Descriptor of ZPyVarDesc
 *
 * Returns:
 * TRUE
 */
static gboolean
z_proxy_var_destroy(gchar *key G_GNUC_UNUSED, ZPyVarDesc *value, gpointer user_data G_GNUC_UNUSED)
{
  if (value->free)
    (*value->free)(value->value);
  g_free(value);
  return TRUE;
}


/**
 * z_proxy_vars_new:
 *
 * Constructor of ZProxyVars, creates the hash table for ZPyVarDesc instances.
 *
 * Returns:
 * The new instances
 */
ZProxyVars *
z_proxy_vars_new(void)
{
  ZProxyVars *self = g_new0(ZProxyVars, 1);
  
  self->vars = g_hash_table_new(g_str_hash, g_str_equal);
  return self;
}


/**
 * z_proxy_vars_destroy:
 * @self: this
 *
 * Destructor of ZProxyVars.
 * NOTE: we don't free session specific variables here, it is up to the
 * proxy to free them as the subsession ends.
 */
void
z_proxy_vars_destroy(ZProxyVars *self)
{
  g_hash_table_foreach_remove(self->vars, (GHRFunc) z_proxy_var_destroy, NULL);
  g_hash_table_destroy(self->vars);
  g_free(self);
}


/**
 * z_proxy_var_dump:
 * @key: key string to print in the dump
 * @value: ZPyVarDesc variable to dump
 * @self: ZProxy to obtain the session id from
 *
 * Logs the variable @value along with its key and the session id.
 *
 * Returns:
 * TRUE
 */
static gboolean
z_proxy_var_dump(gchar *key, ZPyVarDesc *value, ZProxy *self)
{
  ZPolicyObj *res;
  ZPolicyObj *repr;
  
  z_enter();
  if (value->flags & (Z_VAR_GET_CONFIG|Z_VAR_SET_CONFIG) &&
      Z_VAR_TYPE(value->flags) != Z_VAR_TYPE_ALIAS &&
      Z_VAR_TYPE(value->flags) != Z_VAR_TYPE_OBSOLETE)
    {
      res = value->getvar(self, key, value->value);
      
      repr = z_policy_var_repr(res);
      /*LOG
        This message reports the configured value of the given proxy attribute.
       */
      z_log(self->session_id, CORE_DUMP, 6, "Config dump, attribute value; name='%s', value='%s'", key, z_policy_str_as_string(repr));
      z_policy_var_unref(repr);
      z_policy_var_unref(res);
    }
  z_leave();
  return TRUE;
}


/**
 * z_proxy_vars_dump_values:
 * @self: ZProxyVars that contains the hash of variables
 * @proxy: ZProxy to obtain the session id from
 *
 * Dumps the entire variable set @self, using z_proxy_var_dump.
 * NOTE: caller must lock the current thread state
 */
void
z_proxy_vars_dump_values(ZProxyVars *self, ZProxy *proxy)
{
  z_enter();
  if (z_log_enabled(CORE_DUMP, 6))
    {
      if (self->session_vars && self->session_vars->vars)
        {
          /*LOG
            This message reports that the following dumped attributes are session specific.
           */
          z_log(proxy->session_id, CORE_DUMP, 6, "Session specific attributes follow;");
          g_hash_table_foreach(self->session_vars->vars, (GHFunc) z_proxy_var_dump, proxy);
        }
      g_hash_table_foreach(self->vars, (GHFunc) z_proxy_var_dump, proxy);
    }
  z_leave();
}


/**
 * z_proxy_vars_set_active_session:
 * @self: this
 * @vars: Session-scope variable overriders
 *
 * Sets the session-scope variable overriders of @self to @vars.
 * See z_proxy_vars_lookup for details.
 */
void 
z_proxy_vars_set_active_session(ZProxy *self, struct _ZSessionVars *vars)
{
  self->vars->session_vars = vars;
}


/**
 * z_session_vars_new:
 *
 * Constructor of ZSessionVars, creates the hash of variables.
 *
 * Returns:
 * The new instance 
 */
ZSessionVars *
z_session_vars_new(void)
{
  ZSessionVars *self = g_new0(ZSessionVars, 1);

  self->vars = g_hash_table_new(g_str_hash, g_str_equal);
  
  return self;
}


/**
 * z_session_vars_destroy:
 * @self: this
 *
 * Destructor of ZSessionVars.
 */
void
z_session_vars_destroy(ZSessionVars *self)
{
  g_hash_table_foreach_remove(self->vars, (GHRFunc) z_proxy_var_destroy, NULL);
  g_hash_table_destroy(self->vars);
  g_free(self);
}
