/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: proxy.h,v 1.82 2004/06/11 12:57:39 bazsi Exp $
 *
 ***************************************************************************/

#ifndef ZORP_BLOB_H_INCLUDED
#define ZORP_BLOB_H_INCLUDED

#include <zorp/zorp.h>

/*
 * ZBlobSystem - Central management of blobs
 */
struct ZBlob;

typedef struct ZBlobSystem
{
  guint         ref_cnt;                    /* reference counter */
  gchar         *dir;                       /* directory to store the blobs in */
  gsize         disk_max, disk_used;        /* maximal and current disk usage */
  gsize         mem_max, mem_used;          /* maximal and current memory usage */
  gsize         lowat, hiwat, noswap_max;   /* control limits - see spec */
  
  GMutex        *mtx_blobsys;               /* gadgets used for signalling request like allocation, etc. */
  GCond         *cond_thread_started;
  
  GThread       *thr_management;            /* management thread */
  GError        *thread_error;              /* error structure for creating @thr_management */
  GList         *blobs;                     /* blobs created in this blob system */

  GAsyncQueue   *req_queue;                 /* queue of blobs who have pending requests */
  GList         *waiting_list;              /* list of blobs whose requests weren't approved immediately */
	gboolean			active;											/* false if the blobsys is 'under destruction' */
} ZBlobSystem;

/* global default instance */
extern ZBlobSystem  *z_blob_system_default;
extern const gchar* z_blob_system_default_tmpdir; /* directory to store the blobs in */
extern gsize z_blob_system_default_max_disk_usage; /* max disk usage = 1 GB */
extern gsize z_blob_system_default_max_mem_usage;   /* max mem usage = 256 MB */
extern gsize z_blob_system_default_lowat;            /* lowat = 96 MB */
extern gsize z_blob_system_default_hiwat;           /* hiwat = 128 MB */
extern gsize z_blob_system_default_noswap_max;             /* noswap_max = 16 kB */

/* constructor, ref, unref, destructor */
ZBlobSystem* z_blob_system_new(const char* dir, gsize dmax, gsize mmax, gsize low, gsize hiw, gsize nosw);
void z_blob_system_ref(ZBlobSystem *self);
void z_blob_system_unref(ZBlobSystem *self);
/* create and destroy the default instance */
void z_blob_system_default_init(void);
void z_blob_system_default_destroy(void);


/*
 * ZBlobStatistic - usage statistics for a blob
 */
typedef struct ZBlobStatistic
{
    gint                req_rd, req_wr, req_map;  /* performed read, write and mapping requests */
    gint                swap_count;               /* swapout counter */
    gint                alloc_count;              /* alloc modification counter */
    unsigned long long  total_rd, total_wr;       /* total bytes read and written */
    time_t              created, last_accessed;   /* time of creation and last access */
} ZBlobStatistic;

/* initialise a blob stat - counters set to zero, timestamps to 'now' */
void z_blob_statistic_init(ZBlobStatistic *self);

/*
 * ZBlobAccessType - hint of what type of accesses will be performed on the blob
 */
typedef enum ZBlobAccessType
{
  Z_BLOB_STORE,
  Z_BLOB_RETRIEVE
} ZBlobAccessType;

/* ZBlobAccessMode - hint of how the requests will access the blob */
typedef enum ZBlobAccessMode
{
  Z_BLOB_SEQUENTIAL,
  Z_BLOB_RANDOM
} ZBlobAccessMode;

/* ZBlobRequestCodes - blob sends it to the maintenance thread for approval or as notification */ 
typedef enum ZBlobRequestCode
{
  Z_BLOB_REQ_NONE,                          /* none - default value, does nothing */
  Z_BLOB_REQ_REGISTER,                      /* new blob wants to register itself */
  Z_BLOB_REQ_UNREGISTER,                    /* blob wants to unregister itself */
  Z_BLOB_REQ_ALLOC                          /* blob asks for approval on modification of its allocation */
} ZBlobRequestCode;

/* ZBlob - the blob itself */
typedef struct ZBlob
{
  guint             ref_cnt;                /* reference counter */
  ZBlobAccessType   access;                 /* access type hint */
  ZBlobAccessMode   mode;                   /* access mode hint */
  gsize             size, alloc_size;       /* actual size of the blob and the allocated space */
  gboolean          is_in_file;             /* is the blob swapped out */
  gchar             *filename;              /* swapfile name */
  gint              fd;                     /* swapfile descriptor */
  gchar             *data;                  /* memory image pointer */
  ZBlobSystem       *system;                /* blob system it belongs to */
  GMutex            *mtx_lock;              /* for being able to downgrade @lock */
  GStaticRWLock     lock;                   /* lock for concurrent accesses */
  ZBlobStatistic    stat;                   /* statistics */

  GMutex            *mtx_reply;             /* mutex and conditional for waiting for reply */
  GCond             *cond_reply;
  
  gchar             *mapped_ptr;            /* addr and length of the mapped area */
  gsize             mapped_length;          /* (when multiple mappings will be implemented, replace with ?GHash?) */
} ZBlob;

/* constructor, ref, unref, destructor */
ZBlob *z_blob_new(ZBlobSystem *sys, gsize initial_size);
ZBlob	*z_blob_ref(ZBlob *self);
void z_blob_unref(ZBlob *self);

/* get/set access type and mode */
void z_blob_set_access_type(ZBlob *self, ZBlobAccessType access);
ZBlobAccessType z_blob_get_access_type(ZBlob *self);
void z_blob_set_access_mode(ZBlob *self, ZBlobAccessMode mode);
ZBlobAccessMode z_blob_get_access_mode(ZBlob *self);

/* swap out and get the filename - WARNING - the blob will be locked for reading (only) */
const gchar *z_blob_get_file(ZBlob *self, gint timeout);
void z_blob_release_file(ZBlob *self);

/* truncate/expand a blob to a specified size */
gboolean z_blob_truncate(ZBlob *self, gsize pos, gint timeout);

/* write and read */
gsize z_blob_add_copy(ZBlob *self, gsize pos, const gchar* data, gsize req_datalen, gint timeout);
gsize z_blob_get_copy(ZBlob *self, gsize pos, gchar* data, gsize req_datalen, gint timeout);

/* lock and unlock part of the blob and return pointer to it - WARNING - the blob will be locked for reading (only) */
gchar *z_blob_get_ptr(ZBlob *self, gsize pos, gsize *req_datalen, gint timeout);
void z_blob_free_ptr(ZBlob *self, gchar *data);

/* locking functions - needed by ZStreamBlob */
gboolean z_blob_lock_read(ZBlob *self, gint timeout);
void z_blob_unlock_read(ZBlob *self);
gboolean z_blob_lock_write(ZBlob *self, gint timeout);
void z_blob_unlock_write(ZBlob *self);

#endif

