/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008-2010 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include "GribPlot.h"
#include <QMessageBox>
#include <QObject>

#include "Font.h"

//----------------------------------------------------
GribPlot::GribPlot(const GribPlot &model)
{
	initNewGribPlot (model.mustInterpolateValues, model.drawWindArrowsOnGrid);	
	loadFile (model.fileName);
	duplicateFirstCumulativeRecord (model.mustDuplicateFirstCumulativeRecord);
}

//----------------------------------------------------
GribPlot::GribPlot()
{
	initNewGribPlot();
}

//----------------------------------------------------
void GribPlot::initNewGribPlot(bool interpolateValues, bool windArrowsOnGribGrid)
{
    gribReader = NULL;
    
	this->mustInterpolateValues = interpolateValues;
	this->drawWindArrowsOnGrid = windArrowsOnGribGrid;
}
//----------------------------------------------------
GribPlot::~GribPlot() {
}

//----------------------------------------------------
void GribPlot::loadFile(QString fileName)
{
	this->fileName = fileName;
	listDates.clear();
    
    if (gribReader != NULL) {
    	delete gribReader;
        gribReader = NULL;
    }
	
	gribReader = new GribReader(qPrintable(fileName));
    if (gribReader != NULL  &&  gribReader->isOk())
    {
		listDates = gribReader->getListDates();
		setCurrentDate ( listDates.size()>0 ? *(listDates.begin()) : 0);
	}
}

//----------------------------------------------------
void GribPlot::duplicateFirstCumulativeRecord ( bool mustDuplicate )
{
	mustDuplicateFirstCumulativeRecord = mustDuplicate;
    if (gribReader != NULL  &&  gribReader->isOk())
    {
		if (mustDuplicate) {
			gribReader->copyFirstCumulativeRecord();
		}
		else {
			gribReader->removeFirstCumulativeRecord();
		}
	}
}

//----------------------------------------------------
void GribPlot::setCurrentDate (time_t t)
{
    currentDate = t;
}
 
//-------------------------------------------------------------
// Grille GRIB
void GribPlot::draw_GridPoints (QPainter &pnt, const Projection *proj)
{
    if (gribReader == NULL) {
        return;
    }
    GribRecord *rec = gribReader->getFirstGribRecord();
    if (rec == NULL)
        return;
    int px,py, i,j, dl=2;
    for (i=0; i<rec->getNi(); i++)
        for (j=0; j<rec->getNj(); j++)
        {
            if (rec->hasValue(i,j))
            {
                proj->map2screen(rec->getX(i), rec->getY(j), &px,&py);
                pnt.drawLine(px-dl,py, px+dl,py);
                pnt.drawLine(px,py-dl, px,py+dl);
                proj->map2screen(rec->getX(i)-360.0, rec->getY(j), &px,&py);
                pnt.drawLine(px-dl,py, px+dl,py);
                pnt.drawLine(px,py-dl, px,py+dl);
            }
        }
}


//==================================================================================
// Flèches de direction du vent
//==================================================================================
void GribPlot::draw_WIND_Arrows (
				Altitude altitude, 
				bool barbules, QColor arrowsColor, 
				QPainter &pnt, const Projection *proj )
{
    if (gribReader == NULL) {
        return;
    }
	windAltitude = altitude;
    windArrowColor = arrowsColor;

    GribRecord *recx = gribReader->getRecord
								(DataCode(GRB_WIND_VX,altitude),currentDate);
    GribRecord *recy = gribReader->getRecord 
								(DataCode(GRB_WIND_VY,altitude),currentDate);
    if (recx == NULL || recy == NULL)
        return;        
    int i, j;
    double x, y, vx, vy;
    int W = proj->getW();
    int H = proj->getH();
    
	int space;    
    if (barbules)
	    space =  drawWindArrowsOnGrid ? windBarbuleSpaceOnGrid : windBarbuleSpace;
    else
	    space =  drawWindArrowsOnGrid ? windArrowSpaceOnGrid : windArrowSpace;
    
    if (drawWindArrowsOnGrid)
    {	// Flèches uniquement sur les points de la grille
    	int oldi=-1000, oldj=-1000;
    	for (int gi=0; gi<recx->getNi(); gi++)
    	{
			x = recx->getX(gi);
			y = recx->getY(0);
			proj->map2screen(x,y, &i,&j);
			if (abs(i-oldi)>=space)
			{
				oldi = i;
				for (int gj=0; gj<recx->getNj(); gj++)
				{
					x = recx->getX(gi);
					y = recx->getY(gj);
					proj->map2screen(x,y, &i,&j);
					
						//----------------------------------------------------------------------
						if (! recx->isXInMap(x))
							x += 360.0;   // tour du monde ?

						if (recx->isPointInMap(x,y)) {
							if (abs(j-oldj)>=space)
							{
								oldj = j;
								vx = recx->getInterpolatedValue(x, y, mustInterpolateValues);
								vy = recy->getInterpolatedValue(x, y, mustInterpolateValues);
								if (vx != GRIB_NOTDEF && vy != GRIB_NOTDEF)
								{
									if (barbules)
										drawWindArrowWithBarbs(pnt, i,j, vx,vy, (y<0), arrowsColor);
									else
										drawWindArrow(pnt, i,j, vx,vy);
								}
							}
						}
				}
			}
    	}
    }
    else
    {	// Flèches uniformément réparties sur l'écran
		for (i=0; i<W; i+=space)
		{
			for (j=0; j<H; j+=space)
			{
				proj->screen2map(i,j, &x,&y);
				//----------------------------------------------------------------------    			
				if (! recx->isXInMap(x))
					x += 360.0;   // tour du monde ?
				if (recx->isPointInMap(x,y)) {
					vx = recx->getInterpolatedValue(x, y, mustInterpolateValues);
					vy = recy->getInterpolatedValue(x, y, mustInterpolateValues);
					if (vx != GRIB_NOTDEF && vy != GRIB_NOTDEF)
					{
						if (barbules)
							drawWindArrowWithBarbs(pnt, i,j, vx,vy, (y<0), arrowsColor);
						else
							drawWindArrow(pnt, i,j, vx,vy);
					}
				}
				//----------------------------------------------------------------------    			
			}
		}
	}
}

//--------------------------------------------------------------------------
void GribPlot::draw_ColoredMapPlain (
						DataCode dtc,
						bool smooth,
						QPainter &pnt, 
						const Projection *proj)
{
    if (gribReader == NULL)
        return;

	switch (dtc.dataType) {
		case GRB_WIND_XY2D :
			windAltitude = dtc.getAltitude ();
			drawColorMapGeneric_2D (pnt,proj,smooth, 
							DataCode (GRB_WIND_VX, dtc.levelType,dtc.levelValue),
							DataCode (GRB_WIND_VY, dtc.levelType,dtc.levelValue),
							&GribPlot::getWindColor );
			break;
		case GRB_DIFF_TEMPDEW :
			drawColorMapGeneric_Abs_Delta_Data (pnt,proj,smooth,
							DataCode (GRB_TEMP, dtc.levelType,dtc.levelValue),
							DataCode (GRB_DEWPOINT, dtc.levelType,dtc.levelValue),
							&GribPlot::getDeltaTemperaturesColor );
			break;
		case GRB_CLOUD_TOT : 
			drawColorMapGeneric_1D(pnt,proj,smooth, dtc, &GribPlot::getCloudColor );
			break;
		case GRB_PRECIP_TOT :
			drawColorMapGeneric_1D(pnt,proj,smooth, dtc, &GribPlot::getRainColor );
			break;
		case GRB_HUMID_REL :
			drawColorMapGeneric_1D(pnt,proj,smooth, dtc, &GribPlot::getHumidColor );
			break;
		case GRB_TEMP :
			drawColorMapGeneric_1D(pnt,proj,smooth, dtc, &GribPlot::getTemperatureColor );
			break;
		case GRB_TEMP_POT :
			drawColorMapGeneric_1D(pnt,proj,smooth, dtc, &GribPlot::getTemperatureColor );
			break;
		case GRB_DEWPOINT :
			drawColorMapGeneric_1D(pnt,proj,smooth, dtc, &GribPlot::getTemperatureColor );
			break;
		case GRB_SNOW_DEPTH :
			drawColorMapGeneric_1D(pnt,proj,smooth, dtc, &GribPlot::getSnowDepthColor );
			break;
		case GRB_SNOW_CATEG :
			drawColorMapGeneric_1D(pnt,proj,smooth, dtc, &GribPlot::getSnowDepthColor );
			break;
		case GRB_FRZRAIN_CATEG :
			drawColorMapGeneric_1D(pnt,proj,smooth, dtc, &GribPlot::getSnowDepthColor );
			break;
		case GRB_CAPE :
			drawColorMapGeneric_1D(pnt,proj,smooth, dtc, &GribPlot::getCAPEColor );
			break;
		default :
			break;
	}
}


