/*
 *   meix439n.h -- MEIO device dependant definitions
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <port_types.h> 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include <meio_api.h>          /* include general MEIO info */
#include <meiomacs.h>          /* include vendor driver types and macros */
#include <meiogtwy.h>
#define TIGERSHARK2_DEF
#include "meix0439.h"          /* get resource defs and function protos */
#include "meiwmix.h"
#include "drv50439.h"
#include <stdio.h>
#include <stdlib.h>

extern char *ini_file;

#ifdef WORLDTRADE_VERSION
#include <mwwttbl.h>
#endif

#define MAX_MEIO_WAIT 32767    /* wait maximum of 100 times longer than usual */
#define POWER_OFF 0x0
#define POWER_ON 0x1

USHORT Mixer_Connected=0x0;
USHORT MicCount=0x0;
USHORT CdDacCount=0x0;
ULONG  ulWtMicGainMin=0x0;
ULONG  ulWtMicGainNom=0x0;
ULONG  ulWtMicGainMax=55;
USHORT WtMicVolMax=55;
USHORT WtMicVolMin=0;
HANDLE hMixerEvent=0x0;
HWND  hChildMixerThread=0x0;
USHORT MicR1Gain=0x0;
USHORT MicL1Gain=0x0;

VOID CloseMixerAndThread(VOID);


/*-------------------------------------------------------------------------
 *
 *  CdDacMethods() - special handling of UPDATE GAIN.
 *
 *------------------------------------------------------------------------- */
ULONG FAR CdDacMethods   (RF_ARGS)    /* CD DAC has gain control */
{
  ULONG ulRC = MEIO_NOERROR;
  short sGain;
  
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdDacMethods entry\n");
  if ( CHECK_UPDATE_CONNECTION ) {
    TelephonyCodecUpdated++;
    
    /*set mute bit based on lParam2  - if ON turn off mute bit */
    if (lParam2 == ON)
      TelephonyCodec16.Ctl.Mute = 0;
    else if (lParam2 == OFF)
      TelephonyCodec16.Ctl.Mute = 1;
    
  } else {
    if ( CHECK_QUERY_CONNECTION ) {
      if (TelephonyCodec16.Ctl.Mute == 0) {
	*((LPLONG)lParam2) = ON;    /* DAC port is on */
      } else {
	*((LPLONG)lParam2) = OFF;   /* DAC port is off */
      }
    } else {
      if ( CHECK_UPDATE_OCGAIN ) {         /* special purpose handling of update gain */
	sGain = (short)bound(-465,(LONG)lParam2,0);  /* range is 0 -> -46.5 dB */
	sGain = sGain - (sGain%15);     /* 1.5 dB granularity; keep it the same, */
	/*   or move it closer to 0. */
	
	((ANALOG_USER_PORT_RESOURCE FAR *)lpResource)->data.gain = sGain;  /* save object's gain */
	
	sGain = sGain / -15;   /* convert to bit value; make it positive by */
	/* dividing by a negative number because the absolute value is what's */
	/* really needed.  For example, if you desire -45 dB then that is */
	/* (-450 / 15) = (-30 * 1.5 dB increments).  However, you don't want */
	/* -30 increments; you want 30 decrements. */
	
	switch (lpResource->usHandle)   /* who are we? */
	  {
	  case CDDAC_L1:                  /* left channel? */
	  case CDDAC_R1:                  /* right channel? */
	    TelephonyCodec16.Ctl.OutputRight= (USHORT)sGain;
	    TelephonyCodecUpdated++;
	    
	    break;
	  default:                        /* unknown! return some bad news */
	    return MEIO_ERROR;           /* this should not occur! */
	  } /* endswitch */ 
      } else {
	  ulRC = AnalogOwnerDefault(RF_ARGLIST);
      }
    }
  }
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdDacMethods exit\n");
  return ulRC;
}

/*-------------------------------------------------------------------------
 *
 *  CdAdcMethods() - special handling of UPDATE GAIN.
 *
 *------------------------------------------------------------------------- */
ULONG FAR CdAdcMethods   (RF_ARGS)    /* CD ADC has gain control */
{
  ULONG ulRC = MEIO_NOERROR;
  short sGain;
  
  
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdAdcMethods entry\n");
  if ( CHECK_UPDATE_UCGAIN ) {         /* special purpose handling of update gain */
    sGain = (short)bound(0,(LONG)lParam2,225);  /* range is 0 -> 22.5 dB */
    sGain = sGain - (sGain%15);     /* 1.5 dB granularity */
    
    if (lpResource->usHandle != CDADC_S1)   /* stereo?             */
      ((ANALOG_USER_PORT_RESOURCE FAR *)lpResource)->data.gain = sGain;  /* save object's gain */
    
    sGain = sGain / 15;             /* convert to bit value */
    
    switch (lpResource->usHandle)   /* who are we? */
      {
      case CDADC_S1:                  /* both channels?              */
      case CDADC_L1:                  /* left channel? */
      case CDADC_R1:                  /* right channel? */
	TelephonyCodec16.Ctl.InputRight= (USHORT)sGain;
	TelephonyCodecUpdated++;
	break;
      default:                            /* unknown! return some bad news */
	return MEIO_ERROR;           /* this should not occur! */
      } 
  } else { 
    if ( CHECK_QUERY_CONNECTION ) {
      *((LPLONG)lParam2) = ON;    /* ADC port is on */
    } else {
      ulRC = AnalogUserDefault(RF_ARGLIST);
    }
  }
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meix439n::CdAdcMethods exit ulRC %lx\n",ulRC);
  return ulRC;
}

ULONG SetCrystalCodecPower(LPARAM lPowerRequirements)
{
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::SetCrystalCodecPower entry+exit\n");
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::SetCrystalCodecPower, unported function\n");
  return 0;
}

VOID CloseMixerAndThread(VOID)
{
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CloseMixerAndThread entry\n");
  SetCrystalCodecPower (POWER_OFF);
  /* if hChildMixerThread is valid, then call the mixer */
  if (hChildMixerThread){
    /* clean up the child thread, post message to the child and have him exit */
    /* normally. */
    hChildMixerThread = 0x0;
    MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CloseMixerAndThread, unported message\n");
  }
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CloseMixerAndThread exit\n");
}

/* =======================================================================
 *
 * begin connection functions
 *
 * ======================================================================= */
ULONG FAR MicS1_CdAdcS1        (RF_ARGS)
{
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicS1_CdAdcS1 entry+exit\n");
  return generic_scf(RF_ARGLIST);
}


ULONG FAR MicFns(RF_ARGS)
{
  ULONG ulRC = MEIO_NOERROR;
  USHORT     usCount;
  USHORT     usGainL=0;
  USHORT     usGainR=0;
  
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicFns entry\n");
  
  if ( CHECK_QUERY_CONNECTION )
    *((LPLONG)lParam2) = ON;    /* Assume all ports are ON for now */
  
  
  switch (usMessage)
    {
    case mRESERVEmsg:
      
      
      if (!(MicConnectCountInc(&usCount))) {
	if (usCount == 0x1) {
	  SetCrystalCodecPower (POWER_ON);
	  /* initialize gains from persistent data */
#ifdef WORLDTRADE_VERSION /* @PORT */
	  if(MEIO_NOERROR != WtQueryItem((char FAR *)0,
					 WT_MIC2_GAIN_MIN, &ulWtMicGainMin))
	    ulWtMicGainMin = 0;
	    
	  ulWtMicGainMin = ulWtMicGainMin * 10;      /* shift to include decimal */
	    
	  if(MEIO_NOERROR != WtQueryItem((char FAR *)0,
					 WT_MIC2_GAIN_MAX, &ulWtMicGainMax))
	    ulWtMicGainMax = 55;
	  
	  ulWtMicGainMax = ulWtMicGainMax * 10;              /* shift to include decimal */
	  
	  if(MEIO_NOERROR == WtQueryItem((char FAR *)0,
					 WT_MIC2_GAIN_NOM, &ulWtMicGainNom)) {
	    usGainL = (USHORT)ulWtMicGainNom * 10;
	    usGainR = (USHORT)ulWtMicGainNom * 10;
	  } else {
	    usGainL = 32000/MIC_SCALING_FACTOR;
	    usGainR = usGainL;
	  }
#endif
	  MicL1.data.gain = (USHORT)bound(ulWtMicGainMin,usGainL,ulWtMicGainMax);
	  MicR1.data.gain = (USHORT)bound(ulWtMicGainMin,usGainR,ulWtMicGainMax);
	    
	  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicFns, unported message\n");
	}
      }
      if (!ulRC)
	ulRC = generic_cf(RF_ARGLIST);    /* go through normal methods */
      
      break;

   case mRELEASEmsg:
     ulRC = generic_cf(RF_ARGLIST);    /* go through normal methods */
     if (!(MicConnectCountDec(&usCount))) {
       if (usCount == 0x0) {
	 MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicFns, unported message\n");
	 if (CdDacCount == 0x0) {
	   CloseMixerAndThread();
	 }
       }
     }
     break;

   default:
      ulRC = generic_cf(RF_ARGLIST);    /* go through normal methods */
      break;
   } /* endswitch */

   MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meix439n::MicFns exit ulRC %lx\n",ulRC);
   return(ulRC);
}

ULONG FAR MicL1_CdAdcL1 (RF_ARGS)
{
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicL1_CdAdcL1 entry\n");
  if (CHECK_UPDATE_OCGAIN) {
    MicL1.data.gain = (USHORT)bound(0,lParam2,550);   /* limit between 0 and 55dB */
    MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicL1_CdAdcL1, unported message\n");
  }
  MicFns(RF_ARGLIST);
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicL1_CdAdcL1 exit\n");
  return (MEIO_NOERROR);
}

ULONG FAR MicR1_CdAdcR1 (RF_ARGS)
{
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicR1_CdAdcR1 entry\n");
  if (CHECK_UPDATE_OCGAIN) {
    MicR1.data.gain = (USHORT)bound(0,lParam2,550);   /* limit between 0 and 55dB */
    MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicR1_CdAdcR1, unported message\n");
  }
  MicFns(RF_ARGLIST);
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicR1_CdAdcR1 exit\n");
  return (MEIO_NOERROR);
}


ULONG FAR CdDacS1_LineOutS1(RF_ARGS)
{
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdDacS1_LineOutS1 entry+exit\n");
  return generic_cf(RF_ARGLIST);    /* go through normal methods */
}


ULONG FAR CdDacL1_LineOutL1 (RF_ARGS)
{
  ULONG ulRC = MEIO_NOERROR;
  USHORT     usCount;
  
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdDacL1_LineOutL1 entry\n");
  
  if ( CHECK_QUERY_CONNECTION )
    *((LPLONG)lParam2) = ON;    /* Assume all ports are ON for now */
  
  
  switch (usMessage)
    {
      
    case mRESETmsg:
      ulRC = generic_rf(RF_ARGLIST);    /* go through normal methods */
      break;
      
    case mREFRESHmsg:
      ulRC = generic_rf(RF_ARGLIST);    /* go through normal methods */
      break;
      
    case mRESERVEmsg:
      SetCrystalCodecPower (POWER_ON);
      MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"CdacL1_LineOutL1, unported function \n");
      ulRC = generic_rf(RF_ARGLIST);    /* go through normal methods */
      break;
      
    case mRELEASEmsg:
      ulRC = generic_rf(RF_ARGLIST);    /* go through normal methods */
      if (!(CdDacConnectCountDec(&usCount))) {
	if (usCount == 0x0) {
	  /* The following is a patch to fix crystal bug that serial port is controlled by midi */
	  CDCodec16.Ctl.Mute = 1;
	  CDCodec16.Ctl.ModemNotInUse = 1;
	  ulRC = WriteMEIOData(pMEIODSP, WRITE_CDC, CDCodec16.Data);
	  if (MicCount == 0x0) {
	    CloseMixerAndThread();               						
	  }
	}
      }
      break;
      
    default:
      break;
    } /* endswitch */
  
  ulRC=MEIO_NOERROR;
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meix439n::CdDacL1_LineOutL1 exit ulRC %lx\n",ulRC);
  
  return(ulRC);
}

ULONG FAR CdDacR1_LineOutR1 (RF_ARGS)
{
  
  ULONG ulRC = MEIO_NOERROR;
  USHORT     usCount;
  
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdDacR1_LineOutR1 entry\n");
  
  if ( CHECK_QUERY_CONNECTION )
    *((LPLONG)lParam2) = ON;    /* Assume all ports are ON for now */
  
  ulRC = generic_rf(RF_ARGLIST);    /* go through normal methods */
  
  switch (usMessage)
    {
      
    case mRESETmsg:
      ulRC = generic_rf(RF_ARGLIST);    /* go through normal methods */
      break;
      
    case mREFRESHmsg:
      ulRC = generic_rf(RF_ARGLIST);    /* go through normal methods */
      break;
      
    case mRESERVEmsg:
      SetCrystalCodecPower (POWER_ON);
      MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdacR1_LineOutR1 unported function \n");
      ulRC = generic_rf(RF_ARGLIST);    /* go through normal methods */
      break;

   case mRELEASEmsg:
      ulRC = generic_rf(RF_ARGLIST);    /* go through normal methods */
      if (!(CdDacConnectCountDec(&usCount))) {
	if (usCount == 0x0) {
	  /* The following is a patch to fix crystal bug that serial port is controlled by midi */
	  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdDacR1_LineOutR1, unported message\n");
	  
	  if (MicCount == 0x0) {
	    CloseMixerAndThread();
	  }
	  CDCodec16.Ctl.Mute = 1;
	  CDCodec16.Ctl.ModemNotInUse = 1;
	  ulRC = WriteMEIOData(pMEIODSP, WRITE_CDC, CDCodec16.Data);
	}
      }
      break;
      
    default:
      break;
    } /* endswitch */
  
  ulRC=MEIO_NOERROR;
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meix439n::CdDacR1_LineOutR1 exit ulRC %lx\n",ulRC);
  return(ulRC);
  
}


ULONG FAR CdDacConnectCountInc(LPUSHORT lpCount)
{
  ULONG ulRC = MEIO_NOERROR;
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdDacConnectCountInc entry\n");
  CdDacCount++;
  (*lpCount)=CdDacCount;
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdDacConnectCountInc exit\n");
  return(ulRC);
}

ULONG FAR CdDacConnectCountDec(LPUSHORT lpCount)
{
  ULONG ulRC = MEIO_NOERROR;
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdDacConnectCountDec entry\n");
  CdDacCount--;
  (*lpCount)=CdDacCount;
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::CdDacConnectCountDec exit\n");
  return(ulRC);
};

ULONG FAR MicConnectCountInc(LPUSHORT lpCount)
{
  ULONG ulRC = MEIO_NOERROR;
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicConnectCountInc entry\n");
  MicCount++;
  (*lpCount)=MicCount;
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meix439n::MicConnectCountInc exit ulRC %lx\n",ulRC);
  return(ulRC);
};

ULONG FAR MicConnectCountDec(LPUSHORT lpCount)
{
  ULONG ulRC = MEIO_NOERROR;
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicConnectCountDec entry\n");
  MicCount--;
  (*lpCount)=MicCount;
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MicConnectCountDec exit\n");
  return(ulRC);
};

LRESULT MEIO_SendMessage( HWND hWnd, UINT Msg, WPARAM wParam,
			  LPARAM lParam)
{ 
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MEIO_SendMessage entry1+exit1\n");
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::MEIO_SendMessage, unported function\n");
  return MEIO_NOERROR;
}

 
/*--------------------------------------------------------------------
 *
 * define local, special purpose methods
 *
 *-------------------------------------------------------------------- */
ULONG determineCouplerID(HDSP hDSP)
{
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meix439n::determineCouplerID entry hDSP %x\n",(int)hDSP);
  usCouplerID = 0;
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::determineCouplerID exit usCouplerID = 0\n");
  
  return(MEIO_NOERROR);
}

/*-------------------------------------------------------------------------
 *
 * WriteMEIOData() - send some data and command down to MEIO Task
 *
 *------------------------------------------------------------------------- */
ULONG WriteMEIOData(PMEIO_DSP pMeioDsp, const USHORT cusCommand, ULONG data)
{
  ULONG  ulRC = MEIO_NOERROR;
  USHORT usCommand;
  USHORT i;                      /* index counter for timeout on task */
  HDSP   hDsp;
  ULONG  ulDSP_MEIO_CMD;         /* DSP Address of MEIO_CMD */
  ULONG  ulDSP_MEIODATA;         /* DSP Address of MEIODATA */
  
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meix439n::WriteMEIOData entry cusCommand %x\n",cusCommand);
  
  /* get dsp specifics from Meio Handle */
  hDsp           = pMeioDsp->hDsp;
  ulDSP_MEIO_CMD = pMeioDsp->ulMEIO_Command;
  ulDSP_MEIODATA = pMeioDsp->ulMEIO_Data;
  
  usCommand = cusCommand;  /* reset command back to intended value */
  
  /*---------------------------------------------------------------------*/
  /* Write Data to DSP..............                                     */
  /*---------------------------------------------------------------------*/
  
  ulRC = dspMemTransfer(hDsp,
			ulDSP_MEIODATA,
			&data,
			2,                 /* ULONG is 2 x 16 Bits       */
			DSP_MEMXFER_DATA_WRITE);
  
  /*---------------------------------------------------------------------*/
  /* Write Command to DSP...........                                     */
  /*---------------------------------------------------------------------*/
  if (ulRC == DSP_NOERROR)
    ulRC = dspMemTransfer(hDsp,
			  ulDSP_MEIO_CMD,
			  &usCommand,
			  1,                 /* USHORT = 1 x 16 Bits       */
			  DSP_MEMXFER_DATA_WRITE);
  
  /*--------------------------------------------------------------------*/
  /* Wait for MEIO_CMD to go back to a 0 or i to decrement to 0         */
  /*--------------------------------------------------------------------*/
  
  for ( i=MAX_MEIO_WAIT; usCommand && i && ulRC==DSP_NOERROR; i--) {
    ulRC = dspMemTransfer(hDsp,
			  ulDSP_MEIO_CMD,
			  &usCommand,
			  1,
			  DSP_MEMXFER_DATA_READ);
    
  }
  
  if (ulRC)        ulRC = 0x10000000 | (0x0fffffff & ulRC);
  else if (i == 0) ulRC = MEIO_TASK_ERROR;
  
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meix439n::WriteMEIOData exit ulRC %lx\n",ulRC);
  return(ulRC);
}


/*--------------------------------------------------------------------
 *
 * SendScoreBoards() - function to send any updated PIO scoreboard entries
 *
 *-------------------------------------------------------------------- */
ULONG FAR SendScoreBoards(PMEIO_DSP pMeioDsp)
{
  ULONG  ulRC=MEIO_NOERROR;      /* Local Return Code, assume good */
  
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::SendScoreBoards entry\n");
  if (CDCodecUpdated) {
    ulRC = WriteMEIOData(pMeioDsp, WRITE_CDC, CDCodec16.Data);
    if (ulRC==MEIO_NOERROR)
      CDCodecUpdated = 0;
    ulRC=0;
  }
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::SendScoreBoards exit\n");
  return(ulRC);
}

/*========================================================================*/
/* FUNCTION : Initialize MEIO                                             */
/*========================================================================*/
/* Name   : InitializeMEIODsp                                             */
/*                                                                        */
/* Purpose: This function is called by MEIO.C to initialize the MEIO      */
/*          task and any associated variables.                            */
/*          This function :                                               */
/*                                                                        */
/*              (1) Loads MEIO Task                                       */
/*              (2) Gets address of label 'MEIO_CMD'                      */
/*              (3) Gets address of label 'MEIODATA'                      */
/*                                                                        */
/* Format : InitializeMEIODsp(PMEIO_DSP pMeioDsp, USHORT reset)           */
/*            pMeioDsp : pointer to MEIO data area for this DSP           */
/*                                                                        */
/*            reset    : reset all objects and local status               */
/*                                                                        */
/*            ulMEIO_Command = Function sets this variable equal to the   */
/*                             address of the label 'MEIO_CMD'            */
/*                                                                        */
/*            ulMEIO_Data = Function sets this variable equal to the      */
/*                             address of the label 'MEIODATA'            */
/*                                                                        */
/* Return : MEIO_NOERROR - successful                                     */
/*------------------------------------------------------------------------*/
static struct
{
  RDSPINFO data;
  ULONG    reserved[30];    /* room for Task and frame manager info */
} InfoBuf;


void delay(ULONG delay_time) {
  
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::delay, unported function\n");
  
}

ULONG SetupUART( PMEIO_DSP pD);

#define MW_USE_REG_FOR_INI_SETTINGS
//#include <ini2reg.h>

ULONG FAR InitializeMEIODsp(PMEIO_DSP pMeioDsp, USHORT reset)
{
  USHORT InfoSize;
  char    FileName[256];
  ULONG  ulRC=MEIO_NOERROR;
  int    i;
  
  MW_SYSLOG_1(TRACE_MEIO_SPECIFIC,"meix439n::InitializeMEIODsp entry\n");
  pMEIODSP = pMeioDsp;
  if (reset != 0 || pMeioDsp->hMod == 0) {
    ulRC = determineCouplerID(pMeioDsp->hDsp);
    if (ulRC != MEIO_NOERROR)
      return MEIO_ERROR;
    
    /* reset Resource Owner array */
    for (i=0; i<NUM_OWNERS; i++) {
      roOwners[ i ].owner = 0;
      roOwners[ i ].next  = 0;
    } 
    
    /* reset all resource sets */
    for (i=0; AllResources[i]; i++) {
      RS_SendMsg((PMEIO_CONNECTION)0,    /* send reset message to all objects */
		 GETRP(AllResources[i]),
		 mRESETmsg,
		 0,0);
    } 
  } 
  
  
  /*----------------------------------------------------------------------
   *
   *   Get the DSP info so we can get the Mwave OS hMod for future BIOS ops
   *
   *---------------------------------------------------------------------- */
  InfoSize = sizeof(InfoBuf);
  
  ulRC = dspQueryDSPInfo( pMeioDsp->hDsp, &InfoSize, &InfoBuf);
  
  if (ulRC != DSP_NOERROR) {
    ulRC = MEIO_FAIL_REQ_ISPOS_MHANDLE;
    pMeioDsp -> hMod = 0;
    return(ulRC);
  }
  pMeioDsp->hMwaveMod = InfoBuf.data.DSPINFO_hMod_MwaveOS;
  
  if (ulRC == DSP_NOERROR) { 
    GetPrivateProfileString("Mwave,Board","MEIODSP","MEIX0437.DSP",FileName,64,ini_file);
    ulRC = dspLoadModule(pMeioDsp->hDsp, FileName,"MEIO",
			 DSP_ACTIVATE_ACTIVE,&pMeioDsp->hMod);     /* loads meix0460.mwa,  */
  }                                                                /* MEIO's DSP task     */
  if (ulRC == DSP_NOERROR)
    ulRC = dspNameToTaskHandle(pMeioDsp->hMod,"MEIO",&pMeioDsp->hmTask);
  
  /* if manager error, tell the world */
  if (ulRC) {
    ulRC = 0x10000000 | (0x0fffffff & ulRC);
    return ulRC;
  }
  
  ulRC = dspLabelToAddress(pMeioDsp->hmTask, MEIO_COMMAND_LABEL,
			   &pMeioDsp->ulMEIO_Command);
  
  if (ulRC==DSP_NOERROR)
    ulRC = dspLabelToAddress(pMeioDsp->hmTask, MEIO_DATA_LABEL,
			     &pMeioDsp->ulMEIO_Data);
  
  if (ulRC==DSP_NOERROR)
    ulRC = dspLabelToAddress(pMeioDsp->hmTask, MEIO_TAIO_ITCB_LABEL,
			     &pMeioDsp->ulMEIO_TaioAddr);
  
  if ((ulRC==DSP_NOERROR) && (TaioBios.data.State == MEIO_OBJ_USED))
    ulRC = SetupTAIO(pMeioDsp);
  
  if (ulRC == MEIO_NOERROR)
    ulRC = SetupUART( pMeioDsp );
  
  if (ulRC==DSP_NOERROR)
    CDCodec16.Ctl.ModemNotInUse = 1;
  
  CDCodec16.Ctl.Mute = 1;
  ulRC = SendScoreBoards(pMeioDsp);
  
  MW_SYSLOG_2(TRACE_MEIO_SPECIFIC,"meix439n::InitializeMEIODsp exit ulRC %lx\n",ulRC);
  return(ulRC);
}
