/*
 *   dspfixlw.c -- Functions for fixing up DSP segments and TCBs 
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "port_types.h"

#include "dspmgr.h"

#include "dspbios.h"
#include "dspfixup.h"
#include "dspfxutl.h"

RC FMLabelToAddress(PRTSK prtsk,PSZ pszLabel,PULONG pulFMAddress);

/****************************************************************************/
/*                                                                          */
/* The following Data Structure is utilized to facilate isolating and       */
/* inserting operands into ISP Instructions.  The table is indexed by       */
/* the operand type which is represented by the least significant 3         */
/* bits of the FIX_ulflgtype field.                                         */
/*                                                                          */
/* The Opmask entry is a bit mask that has a 1 for each bit that            */
/* represents and immediate operand field within the instruction.           */
/*                                                                          */
/* The Sxbmask field is used to test a single bit which represents          */
/* the sign bit of the operand.                                             */
/*                                                                          */
/* The Sxfmask field is used to test and remove the sign extention from     */
/* the targeted DSP address.  If the bit represented by Sxbmask is set,     */
/* and all of the bits in the sign extention field are set, then the sign   */
/* extention field is cleared and the sign extention bit is set so that     */
/* the target DSP address will fit within the operand field of its opcode.  */
/*                                                                          */
/* Bit 23 movement, which is used in certain ISP instruction is             */
/* supported by using this sign extension scheme in a somewhat bogus        */
/* manner.                                                                  */
/*                                                                          */
/****************************************************************************/

#define SUPPORTED_FIXTYPES 9  /* types 0-8 supported. Only MSP-1.1 uses 7&8 */
static struct {
      LONG       lOpmask;              /* Operand Mask                      */
      LONG       lSxbmask;             /* Sign Extend Bit Mask              */
      LONG       lSxfmask;             /* Sign Extend Field Mask            */
} ExtractInfo[ SUPPORTED_FIXTYPES ] =  {
      {0x0000FFFFL,   0x00000000L,      0xFFFF0000L}, /* 0  CH02 No Sxtn   */
      {0x0000FFFEL,   0x00000000L,      0xFFFF0000L}, /* 1  CH02 No Sxtn   */
      {0x00001FFEL,   0x00001000L,      0xFFFFF000L}, /* 2  CH01 MTS2162   */
      {0x00000FFFL,   0x00000800L,      0xFFFFF800L}, /* 3                 */
      {0x000400FFL,   0x00040000L,      0xFFFFFF00L}, /* 4                 */
      {0x000000FFL,   0x00000001L,      0x00000100L}, /* 5                 */
      {0x000007FEL,   0x00000400l,      0xFFFFFC00L}, /* 6                 */
      {0x000003FEL,   0x00000200L,      0xFFFFFE00L}, /* 7  CH03 MSP11     */
      {0x0000007EL,   0x00000000L,      0xFFFFFFC0L}  /* 8  CH03 MSP11     */
} ;
/*    Operand_Mask   Sign_Extend_Mask  Sign_Extend_Field_Mask            */
/*    ------------   ----------------  ----------------------            */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FixupTCB                                                */
/*                                                                          */
/* FUNCTION: Fixup task control block (TCB) image                           */
/*                                                                          */
/* INPUT: ptr to task structure                                             */
/*                                                                          */
/* OUTPUT: error code                                                       */
/*                                                                          */
/* SIDE EFFECTS: none                                                       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FixupTCB(PRTSK prtsk)
{
   RTCB       rtcb;
   PRTCB      prtcb;
   PRSEG      prseg;
   RC         ulRC;
   PSZ        pszSegName;
   PSZ        pszLabel;
   ULONG      ulAddr;
   USHORT     usAddrFlg;

   /* define strings for frame mgr return address labels                    */

   static char aszLabels[4][9] =
   {
      "FMRTN",
         "AFMRTN",
         "NRTM1RTN",
         "NRTM2RTN"
   }
   ;

   prtcb = &rtcb;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixlw::FixupTCB entry prtsk %x\n",(int)prtsk);

   /* write task's cycles per frame                                         */
   prtcb->TCB_ulCycles = prtsk->TSK_ulCPF+2;

   if ((pszLabel = prtsk->TSK_prtsktmp->TSK_rslALUEntry.pszLabel) == NULL)
      prtcb->TCB_usAluReset = 0;
   else {
      pszSegName = prtsk->TSK_prtsktmp->TSK_rslALUEntry.pszSegment;
      if ((ulRC = SegLabelToPointer(pszSegName, prtsk, &prseg)) != DSP_NOERROR
         )
         return (ulRC);
      usAddrFlg = DSP_ABSOLUTE;
      if ((ulRC = LabelToOffset(prseg, pszLabel, &ulAddr, &usAddrFlg)) !=
         DSP_NOERROR)
      {
	MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspfixlw::FixupTCB error %lx ALU OV-flow entry point %s.%s.%s\n",ulRC,
	       prseg->SEG_prtsk->TSK_pszVirtName, pszSegName, pszLabel);
         return (ulRC);
      }
      prtcb->TCB_usAluReset = (USHORT)ulAddr;
   }

   /* write standby address and codeptr                                     */

   pszSegName = prtsk->TSK_prtsktmp->TSK_rslStandbyEntry.pszSegment;
   pszLabel = prtsk->TSK_prtsktmp->TSK_rslStandbyEntry.pszLabel;

   if ((ulRC = SegLabelToPointer(pszSegName, prtsk, &prseg)) != DSP_NOERROR)
      return (ulRC);

   usAddrFlg = DSP_ABSOLUTE;
   if ((ulRC = LabelToOffset(prseg, pszLabel, &ulAddr, &usAddrFlg)) !=
      DSP_NOERROR)
   {
     MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspfixlw::FixupTCP error ulRC %lx Standby entry point for %s.%s.%s\n",ulRC,
	    prseg->SEG_prtsk->TSK_pszVirtName, pszSegName, pszLabel);
      return (ulRC);
   }
   prtcb->TCB_usStandbyAddr = (USHORT)ulAddr;
   prtcb->TCB_usCodePtr = prtcb->TCB_usStandbyAddr;

   /* write main code entry point                                           */

   pszSegName = prtsk->TSK_prtsktmp->TSK_rslActiveEntry.pszSegment;
   pszLabel = prtsk->TSK_prtsktmp->TSK_rslActiveEntry.pszLabel;

   if ((ulRC = SegLabelToPointer(pszSegName, prtsk, &prseg)) != DSP_NOERROR)

      return (ulRC);
   usAddrFlg = DSP_ABSOLUTE;
   if ((ulRC = LabelToOffset(prseg, pszLabel, &ulAddr, &usAddrFlg)) !=
      DSP_NOERROR)
   {
     MW_SYSLOG_5(TRACE_MANAGER_CORE,"FixupTCB error ulRC %lx Active entry point for %s.%s.%s\n",ulRC,
	    prseg->SEG_prtsk->TSK_pszVirtName, pszSegName, pszLabel);
      return (ulRC);
   }
   prtcb->TCB_usMainAddr = (USHORT)ulAddr;


   if ((ulRC = IsposLabelToAddress(prtsk->TSK_prdsp, aszLabels
      [prtsk->TSK_usflgType&(USHORT)TSK_TYPEMASK], &ulAddr)) != DSP_NOERROR)
   {
     MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfixlw::FixupTCB error ulRC %lx to ISPOS label: %s\n", ulRC,
	    aszLabels[prtsk->TSK_usflgType&(USHORT)TSK_TYPEMASK]);
      return (ulRC);
   }
   else
      prtcb->TCB_usRtnAddr = (USHORT)(ulAddr);

   /* setup pointer to default data seg                                     */

   prseg = prtsk->TSK_prsegDTail->SEG_prsegNext;

   /* write TCB template into DSP memory                                    */

   ulAddr = prseg->SEG_prmem->MEM_ulSegAddr-(ULONG)sizeof(rtcb);

   PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspfixlw::FixupTCB");
   PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)\n", prtsk);
   PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
        "WRITE34:  TCB template in DSP memory", EOF, (ULONG)(sizeof(rtcb)/2),
        prtsk->TSK_prdsp, ulAddr, (ULONG)(sizeof(rtcb)/2), prtcb);

   ulRC = DspBIOS_D_Write(prtsk->TSK_prdsp, ulAddr, (ULONG)(sizeof(rtcb)/2),
      prtcb);
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixlw::FixupTCB exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FMLabelToAddress                                        */
/*                                                                          */
/* FUNCTION: Finds address of FrameMgr label                                */
/*                                                                          */
/* INPUT:                                                                   */
/*   PRTSK prtsk  - Task to search                                          */
/*   PSZ pszLabel - Zero terminated string containing the FM label to       */
/*               search for                                                 */
/* OUTPUT:                                                                  */
/*   PULONG pulFMAddress - FrameMgr Address                                 */
/*      or error code                                                       */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FMLabelToAddress(PRTSK prtsk,PSZ pszLabel,PULONG pulFMAddress)
{
   RC         ulRC;
   ULONG      ulTmpAddr;
   USHORT     usIndexNRTNames;

/* define strings for non-realtime framemgr control block names             */

   static PSZ pszNRTNames[] =  {
      "NRT1",
      "NRT2"
   }
   ;

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfixlw::FMLabelToAddress entry prtsk %x pszLabel %s\n",(int)prtsk,pszLabel);
   if ((ulRC = FMLabelToOffset(pszLabel, pulFMAddress)) != DSP_NOERROR) {
       MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixlw::FMLabelToAddress exit ulRC %lx\n",ulRC);

      return (ulRC);
   }

/* acquire address of FrameMgr Control Block                                */

   if (prtsk->TSK_prfm == NULL) {
      if ((prtsk->TSK_usflgType&TSK_TYPEMASK) == TSK_NRT1)
         usIndexNRTNames = 0;
      else
         if ((prtsk->TSK_usflgType&TSK_TYPEMASK) == TSK_NRT2)

            usIndexNRTNames = 1;
         else
            return (DSP_INV_TASK_TYPE);

   /* for non-realtime tasks, FMCB is in ISPOS                              */

      if ((ulRC = IsposLabelToAddress(prtsk->TSK_prdsp, pszNRTNames
         [usIndexNRTNames], &ulTmpAddr)) != DSP_NOERROR)
      {
	MW_SYSLOG_3(TRACE_MANAGER_CORE,"Fixuplw::FMLabelToAddress error ulRC %lx Mwave/OS label: %s\n",ulRC,
	       pszNRTNames[usIndexNRTNames]);  
         return (ulRC);
      }
      *pulFMAddress += ulTmpAddr;
   }
   else
      *pulFMAddress += prtsk->TSK_prfm->FM_prmemFMCB->MEM_ulSegAddr;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspfixlw::FMLabelToAddress exit ulRC %lx\n",ulRC);

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FixupSegImage                                           */
/*                                                                          */
/* FUNCTION: Fixup the references for the given segment.                    */
/*                                                                          */
/* INPUT: ptrs to rSeg and Image                                            */
/*                                                                          */
/* OUTPUT: error code                                                       */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/************************** END OF SPECIFICATIONS ***************************/

RC FixupSegImage(PRSEG prseg,PVOID primg)
{
   RC         ulRC;
   PARSTBL    parstbl;
   PARFIX     prfix;
   PRDSP      prdsp;
   PRSEG      prRef;
   PUSHORT    pusDspData;
   PULONG     pulDspInst;
   PSZ        pszModName;
   PSZ        pszSegName;
   PSZ        pszLabel;
   LONG       lDspAddr;
   ULONG      ulTmpAddr;
   USHORT     usDataImage;
   USHORT     usTmp;
   USHORT     usFixupCount;
   USHORT     usAddrFlg;
   USHORT     usFixType;

/* setup ptrs to various struct used in fixing up a segment                 */

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfixlw::FixupSegImage entry prseg %x primg %x\n",(int)prseg,(int)primg);

   parstbl = prseg->SEG_prsegtmp->SEG_parstblDiscard;
   prfix = prseg->SEG_prsegtmp->SEG_parfix;
   prdsp = prseg->SEG_prtsk->TSK_prdsp;

/* setup initial condition for while loop                                   */

   usFixupCount = prseg->SEG_prsegtmp->SEG_usFixupCount;

   while (usFixupCount > 0) {

   /* if fixup not done for this record, then do it                         */

      if ((prfix->FIX_usflg&FIX_DONEMASK) != FIX_FINISHED) {/* begin not    */
                                       /* done                              */

      /* setup ptrs to module, segment and label names                      */

         FindStr(parstbl, prfix->FIX_usModName, &pszModName);
         FindStr(parstbl, prfix->FIX_usSegment, &pszSegName);
         FindStr(parstbl, prfix->FIX_usLabel, &pszLabel);

      /* is this special case where reference is to an ISPOS label          */

         if ((*(PUSHORT)pszSegName) == *(PUSHORT)pcMinus2) {
            if ((ulRC = FMLabelToAddress(prseg->SEG_prtsk, pszLabel,
               &ulTmpAddr)) != DSP_NOERROR)
               {

            /* OK, it's not an FrameMgr label, is it a ISPOS label?         */

               if ((ulRC = IsposLabelToAddress(prdsp, pszLabel, &ulTmpAddr))
                  != DSP_NOERROR)
		 {
		   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfixlw::FixupSegImage error ulRC %lx FAR label: %s\n", ulRC,pszLabel);
                  return (ulRC);
               }
            }
            lDspAddr = ulTmpAddr;
         /* add offset field to label's address for case: label+n    CH01   */
            lDspAddr += prfix->FIX_lLabOff;                       /* CH01   */
         }

      /* or is it special case where reference is to a FrameMgr label       */
      /* NOTE: this test for pcMinus3 and the call to FMLabelToAddress      */
      /*       are to support old sytle .dsp files and should be            */
      /*       removed later.                                               */

         else
            if ((*(PUSHORT)pszSegName) == *(PUSHORT)pcMinus3) {
               if ((ulRC = FMLabelToAddress(prseg->SEG_prtsk, pszLabel,
                  &ulTmpAddr)) != DSP_NOERROR)
               {
		 MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspfixlw::FixupSegImage error ulRC %lx Label: %s\n", ulRC,pszLabel);
                  return (ulRC);
               }
               lDspAddr = ulTmpAddr;
            }

/*                                                                          */
/* Absolute Local Label                                                     */
/*                                                                          */
/* Note: Absolute Local Labels always have the FIX_VALID bit set in         */
/* FIX_usflg                                                                */
/*                                                                          */

            else
               if ((*(PUSHORT)pszLabel) == *(PUSHORT)pcMinus1)

                  lDspAddr = prseg->SEG_prmem->MEM_ulSegAddr+
                     prfix->FIX_lLabOff;

/*                                                                          */
/* Normal Label                                                             */
/*                                                                          */

               else {
                  if ((ulRC = SegLabelToPointer(pszSegName, prseg->SEG_prtsk,
                     &prRef)) != DSP_NOERROR)
                     return (ulRC);

/*                                                                          */
/* Offset supplied by Module Compiler.  Also, add in the base address       */
/* of the referenced segment when required.                                 */
/*                                                                          */

                  if ((prfix->FIX_usflg&FIX_VALIDMASK) == FIX_VALID) {
                     lDspAddr = prfix->FIX_lLabOff;
                     if ((prfix->FIX_usflg&FIX_PUBEQUMASK) == FIX_NOTPUBEQU &&
                        (prfix->FIX_ulflgType&FIX_STYLEMASK) !=
                        FIX_STYLE_OFFSET)
                        lDspAddr += prRef->SEG_prmem->MEM_ulSegAddr;
                  }

/*                                                                          */
/* Find Label's Offset                                                      */
/*                                                                          */

                  else {

                     if ((prfix->FIX_ulflgType&FIX_STYLEMASK) ==
                        FIX_STYLE_OFFSET)
                        usAddrFlg = DSP_RELATIVE;/* we want offset of the   */
                                       /* label                             */
                     else
                        usAddrFlg = DSP_ABSOLUTE;/* we want address of the  */
                                       /* label                             */
                     if ((ulRC = LabelToOffset(prRef, pszLabel, &ulTmpAddr,
                        &usAddrFlg)) != DSP_NOERROR)

                     {
		       MW_SYSLOG_7(TRACE_MANAGER_CORE,"dspfixlw::FixupSegImage error ulRC %lx In task %s.%s at offset: 0x%04lX to %s.%s \n",ulRC,
                           prseg->SEG_prtsk->TSK_pszVirtName,
                           prseg->SEG_pszVirtName, prfix->FIX_ulOffset,
                           prRef->SEG_pszVirtName, pszLabel);
                        return (ulRC);
                     }
                     lDspAddr = ulTmpAddr;
                  }
               }

/*                                                                          */
/* Select Instruction or Data Fixup                                         */
/*                                                                          */

         if ((prfix->FIX_ulflgType&FIX_SEGMENTMASK) == FIX_INST) {/* Begin  */
                                       /* Instruction Fixup                 */

/*                                                                          */
/* Subtract Address of Instruction for ILR Relative addressing              */
/*                                                                          */

            if ((prfix->FIX_ulflgType&FIX_STYLEMASK) == FIX_STYLE_ILR)
               lDspAddr -= (LONG)(prseg->SEG_prmem->MEM_ulSegAddr+
                  prfix->FIX_ulOffset);

/*                                                                          */
/* Locate Instruction being Fixed up                                        */
/*                                                                          */

            pulDspInst = (PULONG)primg+prfix->FIX_ulOffset;

/*                                                                          */
/* The following line assumes that the bits in FIX_CONTROLMASK are the      */
/* least significant bits of FIX_ulflgtype.                                 */
/*                                                                          */
/* Get the operand location in the instruction                              */
/*                                                                          */

            usFixType = (USHORT)(prfix->FIX_ulflgType&FIX_CONTROLMASK);

            if (usFixType >= SUPPORTED_FIXTYPES) { /* CH03 MSP-1.1 check */
               ulRC = DSP_IO_INV_DSPFILE;
	       MW_SYSLOG_6(TRACE_MANAGER_CORE,"dspfixlw::FixupSegImage error ulRC %lx Invalid fixup type (%d) in DSPfile in %s.%s offset 0x%04lX\n",ulRC,
                  usFixType,
                  prseg->SEG_prtsk->TSK_pszVirtName, prseg->SEG_pszVirtName,
                  prfix->FIX_ulOffset);
               return (ulRC);
            } /* endif */

/*                                                                          */
/* If (operand & lSxbmask) && ((operand & lSxfmask) != lSxfmask)            */
/*   operand out of range error                                             */
/* If (operand & lSxfmask) = lSxfmask                                       */
/*   operand = (operand & ~lSxfmask) | lSxbmask                             */
/* If (operand & ~lOpmask) is non-zero                                      */
/*   operand out of range error                                             */
/* else                                                                     */
/*   fixed up Instruction = Instruction | operand                           */
/*                                                                          */

/* CH02 Begin add code for MTS # 2162                                       */
         /* if the sign extention bit is set all bits in the sign extention */
         /* field must also be set or we have an out of range address       */
            if ((lDspAddr&ExtractInfo[usFixType].lSxbmask) &&
                ((lDspAddr&ExtractInfo[usFixType].lSxfmask) !=
                           ExtractInfo[usFixType].lSxfmask))
            {
               ulRC = DSP_FIX_ADDR_ERR;
	       MW_SYSLOG_6(TRACE_MANAGER_CORE,"dspfixlw::FixupSegImage error ulRC %lx DSP address 0x%04lX out of bounds in %s.%s offset 0x%04lX\n",ulRC,
                  lDspAddr,
                  prseg->SEG_prtsk->TSK_pszVirtName, prseg->SEG_pszVirtName,
                  prfix->FIX_ulOffset);
               return (ulRC);
            }
/* CH02 End code for MTS # 2162                                             */

            /* if all sign extention bits in DSP address are on */
            if ((lDspAddr&ExtractInfo[usFixType].lSxfmask) ==
                          ExtractInfo[usFixType].lSxfmask)
               /* strip off sign extention from lDspAddr */
               lDspAddr = (lDspAddr&~ExtractInfo[usFixType].lSxfmask)|
                  ExtractInfo[usFixType].lSxbmask;

            if ((lDspAddr&~ExtractInfo[usFixType].lOpmask) != 0L) {
               ulRC = DSP_FIX_ADDR_ERR;
	       MW_SYSLOG_6(TRACE_MANAGER_CORE,"dspfixlw::FixupSegImage error ulRC %lx DSP address 0x%04lX out of bounds in %s.%s offset 0x%04lX\n",ulRC,
                 lDspAddr,
                 prseg->SEG_prtsk->TSK_pszVirtName, prseg->SEG_pszVirtName,
                 prfix->FIX_ulOffset);
               return (ulRC);
            }
            else
               *pulDspInst |= lDspAddr;

         }                             /* End Instruction Fixup             */
         else {                        /* Begin Data Fixup                  */

/*                                                                          */
/* Locate Data Reference to be Fixed up                                     */
/*                                                                          */

            pusDspData = (PUSHORT)((ULONG)primg+prfix->FIX_ulOffset);

/*                                                                          */
/* Ensure Data is in range                                                  */
/*                                                                          */

            if ((lDspAddr <= 65535L) && (lDspAddr >= 0L)) {
               usDataImage = (USHORT)lDspAddr;

/*                                                                          */
/* If R4 Table Addressing, rotate address right on bit                      */
/*                                                                          */

               if ((prfix->FIX_ulflgType&FIX_STYLEMASK) == FIX_STYLE_R4) {
                  usTmp = (USHORT)(usDataImage&R4_BIT0);
                  usDataImage >>= 1;
                  if (usTmp)
                     usDataImage |= R4_BIT15;
               }

               *pusDspData = usDataImage;
            }
            else {
               ulRC = DSP_FIX_ADDR_ERR;
	       MW_SYSLOG_6(TRACE_MANAGER_CORE,"dspfixlw::FixupSegImage error ulRC %lx DSP address 0x%04lX out of bounds in %s.%s offset 0x%04lX\n",ulRC,
                  lDspAddr,
                  prseg->SEG_prtsk->TSK_pszVirtName, prseg->SEG_pszVirtName,
                  prfix->FIX_ulOffset);
               return (ulRC);
            }

         }                             /* End of Data Fixup                 */
      }                                /* end of not done                   */
      prfix++;                         /* point to next fixup record        */
      usFixupCount--;                  /* eventually, this will get us      */

                              /* out of this while loop                     */

   }

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspfixlw::FixupSegImage exit ulRC=0\n");

   return (DSP_NOERROR);

}                                      /* end of FixupSegImage              */
