/*
 * tuxadmin: present (X,Y,Z) coordinates along with node attributes.
 *
 * Its name derives from dog food, http://www.purina.co.nz/tux/about.htm
 */
#include <basil_alps.h>
#include "basil_mysql.h"
#include "cmdline.h"

#define FIELD_SIZEOF(t, f) (sizeof(((t*)0)->f))
#define LINESIZE	64
#define ALPS_CLIENT	"/usr/bin/apbasil"

/**
 * struct node_data - Node record entry
 * @nid:	Cray XT/XE node ID
 * @name:	node name in Cray XT/XE convention (c#-#c#s#n#)
 * @gemini:	gemini name in Cray XE convention  (c#-#c#s#g#)
 * @x:		X coordinate (real)
 * @y:		Y coordinate (may be faked on Gemini systems)
 * @real_y:	Y coordinate as in the Cray SDB database
 * @z:		Z coordinate (real)
 * @hilbert:	Hilbert number (based on either @y or @real_y)
 */
struct node_data {
	uint32_t nid;
	char	 name[32],
		 gemini[32];
	uint8_t	 x,
		 y,
		 real_y,
		 z;
	int 	 hilbert;
	char 	 line[LINESIZE];
};

extern void ns_print_pdsh_format(const struct nodespec *head, FILE *fp);
extern int hilbert_integer(uint8_t x, uint8_t y, uint8_t z);

static int cmp_hilbert(const void *a, const void *b)
{
	return ((struct node_data *)a)->hilbert - ((struct node_data *)b)->hilbert;
}

static int cmp_nid(const void *a, const void *b)
{
	return ((struct node_data *)a)->nid - ((struct node_data *)b)->nid;
}

static char enc_toruscoord(uint8_t tc)
{
	return tc + (tc < 10 ? '0' : 'A' - 10);
}

int main(int argc, char **argv)
{
	char query[2048] = "SELECT processor_id, x_coord, y_coord, z_coord, "
			   "       cab_position, cab_row, cage, slot, cpu,  "
			   "       processor_type, processor_status, alloc_mode "
			   "FROM  processor";
	enum query_columns {
			COL_NID,
			COL_X, COL_Y, COL_Z,
			COL_CAB, COL_ROW,
			COL_CAGE, COL_SLOT, COL_CPU,
			COL_TYPE,
			COL_STATE,
			COL_ALLOC,
			COLUMN_COUNT	/* sentinel */
	};
	MYSQL_BIND	bind_cols[COLUMN_COUNT];
	int		int_data[COL_CPU + 1];
	char 		c_data[COL_ALLOC-COL_TYPE + 1][BASIL_STRING_SIZE];
	my_bool		is_null[COLUMN_COUNT];
	my_bool		error[COLUMN_COUNT];

	MYSQL		*handle;
	MYSQL_STMT	*stmt;
	int		i = 0, k, n, is_gemini;
	struct gengetopt_args_info args;
	struct node_data *node_array;
	struct nodespec *free_nodes = NULL;

	/*
	 * Argument handling
	 */
	if (cmdline_parser(argc, argv, &args) != 0)
		exit(EXIT_FAILURE);

	if (!args.slurm_conf_given) {
		/* Some nodes (e.g. boot node) may not have apbasil installed */
		if (access(ALPS_CLIENT, X_OK) == 0)
			free_nodes = get_free_nodes(get_basil_version());
		if (args.alps_nidorder_given) {
			size_t len = strlen(query);

			snprintf(query + len, sizeof(query) - len, " %s",
				 current_alps_nidorder());
		}
	}

	handle = cray_connect_sdb();
	if (handle == NULL)
		errx(1, "can not connect to XTAdmin database on the SDB");

	is_gemini = cray_is_gemini_system(handle);
	if (is_gemini < 0) {
		errx(1, "can not determine type of interconnect (SeaStar/Gemini)");
	} else if (!is_gemini && args.real_given && !args.slurm_conf_given) {
		warnx("This is a SeaStar system, where TUX coordinates (XYZ) "
		      "always equal the real XYZ coordinates.");
		warnx("Hence the `-r' option is not needed on this system.\n");
		args.real_given = false;
	}

	memset(bind_cols, 0, sizeof(bind_cols));
	for (i = 0; i <= COL_CPU; i++) {
		bind_cols[i].buffer_type = MYSQL_TYPE_LONG;
		bind_cols[i].buffer	 = (char *)&int_data[i];
		bind_cols[i].is_null	 = &is_null[i];
		bind_cols[i].error	 = &error[i];
	}
	for (i = COL_TYPE; i <= COL_ALLOC; i++) {
		bind_cols[i].buffer_type   = MYSQL_TYPE_STRING;
		bind_cols[i].buffer	   = c_data[i - COL_TYPE];
		bind_cols[i].buffer_length = BASIL_STRING_SIZE;
		bind_cols[i].is_null	   = &is_null[i];
		bind_cols[i].error	   = &error[i];
	}

	stmt = exec_query(handle, query, bind_cols, COLUMN_COUNT);
	if (stmt == NULL) {
		mysql_close(handle);
		return EXIT_FAILURE;
	}

	n = mysql_stmt_num_rows(stmt);
	if (n == 0)
		errx(0, "no results");

	if (!args.slurm_conf_given) {
		node_array = calloc(n, sizeof(*node_array));
		if (node_array == NULL)
			errx(1, "calloc(%d)", n);
	}

	for (k = 0; k < n && mysql_stmt_fetch(stmt) == 0; ) {
		bool unavail = strcmp("compute", c_data[0]);

		/* The slurm.conf expression ignores only non-compute nodes. */
		if (args.slurm_conf_given) {
			if (unavail)
				continue;
			if (ns_add_node(&free_nodes, int_data[COL_NID]) < 0)
				errx(1, "can not add node %u", int_data[COL_NID]);
			k++;
			continue;
		}

		unavail |= strcmp("up", c_data[1]) || strcmp("batch", c_data[2]);
		if (unavail && !args.all_given && !args.xtorder2db_given)
			continue;

		node_array[k].nid = int_data[COL_NID];
		snprintf(node_array[k].name,
			 FIELD_SIZEOF(struct node_data, name),
			 "c%d-%dc%ds%dn%d ", int_data[COL_CAB],
			 int_data[COL_ROW], int_data[COL_CAGE],
			 int_data[COL_SLOT], int_data[COL_CPU]);

		node_array[k].x	     = int_data[COL_X];
		node_array[k].real_y = int_data[COL_Y];
		node_array[k].z	     = int_data[COL_Z];

		/* Hilbert number is always based on the real (X,Y,Z) coordinates */
		node_array[k].hilbert = hilbert_integer(node_array[k].x,
							node_array[k].real_y,
							node_array[k].z);
		if (is_gemini) {
			snprintf(node_array[k].gemini,
				 FIELD_SIZEOF(struct node_data, gemini),
				 "c%d-%dc%ds%dg%d ", int_data[COL_CAB],
				 int_data[COL_ROW], int_data[COL_CAGE],
				 int_data[COL_SLOT], int_data[COL_CPU] >> 1);
			/*
			 * Gemini system: create "virtual" Y coordinate, so that
			 * the Y dimension is populated as in a SeaStar system.
			 */
			node_array[k].y = 4 * int_data[COL_CAGE] + int_data[COL_CPU];
			/*
			 * Further constraint on XE/Gemini systems:
			 * Need to make sure that  the lower NID always must come
			 * last when producing an ordering. Otherwise ALPS hangs
			 * with the error:
			 *   "apsched expects  NIDs sharing a network chip to be in
			 *    ascending NIC address order"
			 * This is called the "Gilbert" variant of Hilbert ordering.
			 */
			if (!args.real_given || args.xtorder2db_given) {
				node_array[k].hilbert <<= 1;
				node_array[k].hilbert |= node_array[k].nid & 1;
			}
		} else {
			node_array[k].y = node_array[k].real_y;
		}

		if (unavail)
			snprintf(node_array[k].line,
				 FIELD_SIZEOF(struct node_data, line),
				 "%8s  %5.5s  %5.5s! ",
				 c_data[0], c_data[1], c_data[2]);
		else if (free_nodes)
			snprintf(node_array[k].line,
				 FIELD_SIZEOF(struct node_data, line),
			 	 "%8s  %5.5s  %5.5s  ", c_data[0],
			 	 ns_in_range(free_nodes, node_array[k].nid)
				 	? c_data[1] : "alloc", c_data[2]);
		else
			snprintf(node_array[k].line,
				 FIELD_SIZEOF(struct node_data, line),
			 	 "%8s  %5.5s  %5.5s  ",
				 c_data[0], c_data[1], c_data[2]);
		k++;
	}
	if (mysql_stmt_close(stmt))
		warnx("error closing statement: %s", mysql_stmt_error(stmt));
	mysql_close(handle);

	if (k == 0) {
		if (!args.all_given && !args.slurm_conf_given)
			errx(0, "no usable node found - try the --all/-a flag?");
		else
			errx(1, "no nodes found - check configuration");
	} else if (args.xtorder2db_given) {
		qsort(node_array, k, sizeof(*node_array), cmp_hilbert);
		for (i = 0; i < k; i++)
			printf("%d  %d\n", node_array[i].nid,
					   node_array[i].hilbert);
		goto done;
	} else if (args.slurm_conf_given) {
		printf("slurm can use %d compute nodes. These should be listed "
		       "in slurm.conf as:\n  NodeName=", k);
		ns_print_pdsh_format(free_nodes, stdout);
		printf("\n  ...\n  PartitionName=DEFAULT Shared=EXCLUSIVE State=UP Nodes=");
		ns_print_pdsh_format(free_nodes, stdout);
		printf("  MaxNodes=%d\n", k);
		goto done;
	} else if (args.Hilbert_given) {
		qsort(node_array, k, sizeof(*node_array), cmp_hilbert);
	} else if (!args.alps_nidorder_given) {
		qsort(node_array, k, sizeof(*node_array), cmp_nid);
	}

	printf(" NID   %s%s    NODENAME%s       TYPE  STATE   MODE%s\n",
		args.real_given || !is_gemini ? "XYZ  " : "",
		is_gemini ?                     "TUX  " : "",
		is_gemini ?             "       GEMINI" : "",
		args.Hilbert_given ?
			(args.real_given || !is_gemini ? "   HILBERT"  :
							 "   GILBERT") : "");
	for (i = 0; i < k; i++) {
		printf("%4u   ", node_array[i].nid);
  		if (args.real_given || !is_gemini)
			printf("%c%c%c  ",
 				enc_toruscoord(node_array[i].x),
				enc_toruscoord(node_array[i].real_y),
				enc_toruscoord(node_array[i].z));
		if (is_gemini)
			printf("%c%c%c  ",
				enc_toruscoord(node_array[i].x),
				enc_toruscoord(node_array[i].y),
				enc_toruscoord(node_array[i].z));
		printf("%13s", node_array[i].name);

		if (is_gemini)
			printf(" %12s", node_array[i].gemini);
		printf("  %s", node_array[i].line);

		if (args.Hilbert_given)
			printf(" %7d", node_array[i].hilbert);
		printf("\n");
	}
done:
	free_nodespec(free_nodes);
	return EXIT_SUCCESS;
}
