/*
 * petvia.c - VIA emulation in the PET.
 *
 * Written by
 *  Andre Fachat <fachat@physik.tu-chemnitz.de>
 *  Andreas Boose <viceteam@t-online.de>
 *  Marco van den Heuvel <blackystardust68@yahoo.com>
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *  02111-1307  USA.
 *
 */

#include "vice.h"

#include <stdio.h>

#include "crtc.h"
#include "datasette.h"
#include "drive.h"
#include "interrupt.h"
#include "joystick.h"
#include "keyboard.h"
#include "lib.h"
#include "log.h"
#include "maincpu.h"
#include "parallel.h"
#include "pet.h"
#include "petsound.h"
#include "petvia.h"
#include "tapeport.h"
#include "types.h"
#include "userport.h"
#include "via.h"


void via_store(uint16_t addr, uint8_t data)
{
    viacore_store(machine_context.via, addr, data);
}

uint8_t via_read(uint16_t addr)
{
    return viacore_read(machine_context.via, addr);
}

uint8_t via_peek(uint16_t addr)
{
    return viacore_peek(machine_context.via, addr);
}

/* switching PET charrom with CA2 */
static void set_ca2(via_context_t *via_context, int state)
{
    crtc_set_chargen_offset(state ? 256 : 0);
}

/* switching userport strobe with CB2 */
static void set_cb2(via_context_t *via_context, int state, int offset)
{
    store_userport_pa2((uint8_t)state);
    petsound_store_manual(state, *via_context->clk_ptr - offset);
}

static void set_int(via_context_t *via_context, unsigned int int_num,
                    int value, CLOCK rclk)
{
    interrupt_set_irq(maincpu_int_status, int_num, value, rclk);
}

static void restore_int(via_context_t *via_context, unsigned int int_num, int value)
{
    interrupt_restore_irq(maincpu_int_status, int_num, value);
}

static void undump_pra(via_context_t *via_context, uint8_t byte)
{
    store_userport_pbx(byte, USERPORT_NO_PULSE);
}

static void store_pra(via_context_t *via_context, uint8_t byte, uint8_t myoldpa,
                      uint16_t addr)
{
    store_userport_pbx(byte, USERPORT_NO_PULSE);
}

static void undump_prb(via_context_t *via_context, uint8_t byte)
{
    parallel_cpu_set_nrfd((uint8_t)(!(byte & 0x02)));
    parallel_cpu_restore_atn((uint8_t)(!(byte & 0x04)));
}

static void store_prb(via_context_t *via_context, uint8_t byte, uint8_t myoldpb,
                      uint16_t addr)
{
    if ((addr == VIA_DDRB) && (via_context->via[addr] & 0x20)) {
        log_warning(via_context->log,
                    "PET: Killer POKE! might kill a real PET!\n");
    }
    parallel_cpu_set_nrfd((uint8_t)(!(byte & 0x02)));
    parallel_cpu_set_atn((uint8_t)(!(byte & 0x04)));
    if ((byte ^ myoldpb) & 0x8) {
        tapeport_toggle_write_bit(TAPEPORT_PORT_1, (~(via_context->via[VIA_DDRB]) | byte) & 0x8);
        tapeport_toggle_write_bit(TAPEPORT_PORT_2, (~(via_context->via[VIA_DDRB]) | byte) & 0x8);
    }
    tapeport_set_motor(TAPEPORT_PORT_2, ((~(via_context->via[VIA_DDRB]) | byte) & 0x10) ? 0 : 1);
}

static void undump_pcr(via_context_t *via_context, uint8_t byte)
{
#if 0
    register uint8_t tmp = byte;
    /* first set bit 1 and 5 to the real output values */
    if ((tmp & 0x0c) != 0x0c) {
        tmp |= 0x02;
    }
    if ((tmp & 0xc0) != 0xc0) {
        tmp |= 0x20;
    }
    crtc_set_char(byte & 2); /* switching PET charrom with CA2 */
                             /* switching userport strobe with CB2 */
#endif
    petsound_store_manual((byte & 0xe0) == 0xe0, *via_context->clk_ptr);   /* Manual control of CB2 sound */
}

static uint8_t store_pcr(via_context_t *via_context, uint8_t byte, uint16_t addr)
{
#if 0
    if (byte != via_context->via[VIA_PCR]) {
        register uint8_t tmp = byte;
        /* first set bit 1 and 5 to the real output values */
        if ((tmp & 0x0c) != 0x0c) {
            tmp |= 0x02;
        }
        if ((tmp & 0xc0) != 0xc0) {
            tmp |= 0x20;
        }
        crtc_set_char(byte & 2); /* switching PET charrom with CA2 */
                                 /* switching userport strobe with CB2 */
        store_userport_pa2((byte & 0x20) >> 5);
    }
#endif
    return byte;
}

static inline bool IS_SR_SHIFT_OUT_BY_T2(uint8_t acr)
{
    return (((acr) & VIA_ACR_SR_CONTROL) == VIA_ACR_SR_OUT_FREE_T2 ||
            ((acr) & VIA_ACR_SR_CONTROL) == VIA_ACR_SR_OUT_T2);
}

/*
 * Cut out extremely high frequencies that are not properly
 * suppressed by the low-pass filter, by requiring a minimum
 * value for T2L of 0.
 * TODO: make the low-pass filter more effective.
 */
static inline bool SOUND_ACTIVE(uint8_t acr, uint8_t t2ll)
{
    return !IS_SR_SHIFT_OUT_BY_T2(acr) || t2ll;
}

/*
 * Help the sound to avoid background noise.
 */

static void undump_acr(via_context_t *via_context, uint8_t byte)
{
    petsound_store_onoff(SOUND_ACTIVE(byte,
                                      via_context->via[VIA_T2LL]));
}

static void store_acr(via_context_t *via_context, uint8_t byte)
{
    petsound_store_onoff(SOUND_ACTIVE(byte,
                                      via_context->via[VIA_T2LL]));
}

static void store_sr(via_context_t *via_context, uint8_t byte)
{
}

static void store_t2l(via_context_t *via_context, uint8_t byte)
{
    petsound_store_onoff(SOUND_ACTIVE(via_context->via[VIA_ACR],
                                      byte));
}

static void reset(via_context_t *via_context)
{
    /* set IEC output lines */
    parallel_cpu_set_atn(0);
    parallel_cpu_set_nrfd(0);

    store_userport_pbx(0xff, USERPORT_NO_PULSE);
    store_userport_pa2(1);
}

inline static uint8_t read_pra(via_context_t *via_context, uint16_t addr)
{
    uint8_t byte = 0xff;

    byte = read_userport_pbx(byte);

    /* joystick always pulls low, even if high output, so no
       masking with DDRA */
    /*return ((j & ~(via_context->via[VIA_DDRA]))
        | (via_context->via[VIA_PRA] & via_context->via[VIA_DDRA]));*/
    return byte;
}

static uint8_t read_prb(via_context_t *via_context)
{
    uint8_t byte;

    drive_cpu_execute_all(maincpu_clk);

    /* read parallel IEC interface line states */
    byte = 255
           - (parallel_nrfd ? 64 : 0)
           - (parallel_ndac ? 1 : 0)
           - (parallel_dav ? 128 : 0);
    /* vertical retrace */
    byte -= crtc_offscreen() ? 32 : 0;

    /* none of the load changes output register value -> std. masking */
    byte = ((byte & ~(via_context->via[VIA_DDRB]))
            | (via_context->via[VIA_PRB] & via_context->via[VIA_DDRB]));
    return byte;
}

void via_init(via_context_t *via_context)
{
    viacore_init(machine_context.via, maincpu_alarm_context,
                 maincpu_int_status);
}

void petvia_setup_context(machine_context_t *machinecontext)
{
    via_context_t *via;

    machinecontext->via = lib_malloc(sizeof(via_context_t));
    via = machinecontext->via;

    via->prv = NULL;
    via->context = NULL;

    via->rmw_flag = &maincpu_rmw_flag;
    via->clk_ptr = &maincpu_clk;

    via->myname = lib_msprintf("Via");
    via->my_module_name = lib_msprintf("VIA");

    viacore_setup_context(via);

    via->irq_line = IK_IRQ;

    via->undump_pra = undump_pra;
    via->undump_prb = undump_prb;
    via->undump_pcr = undump_pcr;
    via->undump_acr = undump_acr;
    via->store_pra = store_pra;
    via->store_prb = store_prb;
    via->store_pcr = store_pcr;
    via->store_acr = store_acr;
    via->store_sr = store_sr;
    via->store_t2l = store_t2l;
    via->read_pra = read_pra;
    via->read_prb = read_prb;
    via->set_int = set_int;
    via->restore_int = restore_int;
    via->set_ca2 = set_ca2;
    via->set_cb2 = set_cb2;
    via->reset = reset;
}
