---
myst:
  html_meta:
    keywords: LaTeX, composition, tableaux, flottants, modifier l'apparence des tableaux,
              mise en forme des tableaux, épaisseur des lignes
---

# Comment modifier l'épaisseur des traits dans les tableaux ?

Les filets d'un tableau LaTeX ont par défaut une épaisseur de `0.4pt` ; 
il s'agit d'une valeur prédéfinie au niveau le plus bas, qui s'applique 
à tous les traits (y compris ceux qui séparent les blocs de texte, notamment).

Parfois, cependant, en regardant un tableau, on voudrait faire ressortir 
certains filets, par exemple pour mieux séparer le contenu du tableau du corps
du texte, ou pour délimiter les différentes parties du tableau. Cependant, 
un examen rapide de n'importe quel livre sur LaTeX ne révèle aucune technique
permettant d'épaissir un filet en particulier, et une petite expérimentation 
montre qu'il est en effet assez difficile d'éviter qu'une modification 
ne "déborde" et affecte tous les filets du tableau.

Si vous lisez la question "[](/3_composition/tableaux/mes_tableaux_sont_moches)", 
ailleurs dans cette FAQ, vous avez dû sentir que lors de sa conception, 
LaTeX a tout simplement ignoré la plupart des questions esthétiques liées 
aux tableaux : c'est probablement la raison pour laquelle il ne fournit rien 
pour modifier ponctuellement l'épaisseur des filets.

Plus précisément, si la longueur `\arrayrulewidth` affecte l'épaisseur 
des traits (à la fois horizontaux et verticaux) dans les environnements 
`tabular` et `array`, elle la modifie de façon *globale*. Par exemple :

```
\documentclass[french]{article}
  \usepackage[T1]{fontenc}    % Encodage T1 (adapté au français)
  \usepackage{lmodern}        % Caractères plus lisibles
  \usepackage{babel}          % Réglages linguistiques (avec french)
  \pagestyle{empty}           % N'affiche pas de numéro de page

\begin{document}
Un tableau avec l'épaisseur par défaut : \\[0.5cm]
\begin{tabular}{|lc|}
  \hline
  Peintre & Années \\
  \hline
  Manet  & 1832--1883 \\
  Ingres & 1780--1867 \\
  Goya   & 1746--1828 \\
  \hline
\end{tabular} \\[0.5cm]

\setlength{\arrayrulewidth}{0.05cm}
Un tableau avec une épaisseur de 0,05cm : \\[0.5cm]
\begin{tabular}{|lc|}
  \hline
  Peintre & Années \\
  \hline
  Manet  & 1832--1883 \\
  Ingres & 1780--1867 \\
  Goya   & 1746--1828 \\
  \hline
\end{tabular}
\end{document}
```

Le changement est bien visible, mais remarquez que la jonction 
entre les trais verticaux et horizontaux se fait mal quand leur épaisseur 
est modifiée, tellement l'épaisseur par défaut est ancrée dans le code. 
La programmation au niveau utilisateur, même pointue, s'avère incapable 
de modifier un seul trait : il est nécessaire de se plonger dans le code 
(plutôt délicat) de `\hline` et `\cline` pour y arriver.

Heureusement, ce travail a déjà été fait pour la communauté : l'extension
<ctanpkg:booktabs> définit trois catégories de filets distinctes : 
`\toprule`, `\midrule` et `\bottomrule`, et la 
[documentation (en français)](texdoc:booktabs-fr) de l'extension offre 
des conseils sur la façon de les utiliser. Nous vous conseillons vivement
de la lire attentivement.

```
\documentclass[french]{article}
  \usepackage[T1]{fontenc}    % Encodage T1 (adapté au français)
  \usepackage{lmodern}        % Caractères plus lisibles
  \usepackage{babel}          % Réglages linguistiques (avec french)
  \pagestyle{empty}           % N'affiche pas de numéro de page
  \usepackage{booktabs}

\begin{document}
\begin{tabular}{lc}
\toprule
  Peintre & Années \\
\midrule
  Manet  & 1832--1883 \\
  Ingres & 1780--1867 \\
  Goya   & 1746--1828 \\
\bottomrule
\end{tabular}

\begin{tabular}{lc}
\toprule
  Peintre & Années \\
\cmidrule(lr){1-2}
  Manet  & 1832--1883 \\
  Ingres & 1780--1867 \\
  Goya   & 1746--1828 \\
\bottomrule
\end{tabular}
\end{document}
```

Dans l'exemple ci-dessus, la commande `\cmidrule` raccourcit légèrement 
les filets internes par rapport aux autres, ce qui génère un tableau
un peu plus soigné.

La classe <ctanpkg:memoir> inclut l'extension <ctanpkg:booktabs> 
et reprend la teneur de sa documentation dans son 
[manuel très détaillé](texdoc:memoir) (et pour lequel il existe 
un [projet de traduction](https://github.com/jejust/memoir-fr)).

Notez qu'on a retiré les traits verticaux dans les derniers exemples 
de tableaux. Pour en connaître les raisons, consultez la 
[documentation](texdoc:booktabs-fr) de <ctanpkg:booktabs> 
(encore une fois) ; en résumé, les traits verticaux sont à éviter, 
pour des raisons à la fois historiques et esthétiques. On peut 
toujours s'en passer, en jouant sur l'alignement du contenu des colonnes.


## Pour les bricoleurs

### Épaissir des traits horizontaux à la demande

Nous avons vu que la longueur `\arrayrulewidth` définit l'épaisseur 
des filets de séparation horizontale *et* verticale. Cela ne permet 
donc de changer que *globalement* l'épaisseur des filets.

Maintenant, si, dans un tableau, vous mettez deux filets de séparation 
horizontaux successifs, ceux-ci sont séparées par la longueur 
`\doublerulesep`. Il suffit donc de donner la bonne valeur 
à cette longueur pour que les deux filets soient collés, et semblent 
former un unique filet, plus épais. Vous écrirez donc :

```{noedit}
\setlength{\doublerulesep}{\arrayrulewidth}
```

Ensuite, il suffit d'exécuter plusieurs fois consécutivement 
la commande `\hline` pour épaissir le filet autant que désiré.

Vous pouvez aussi, en se fondant sur la définition de `\hline`, 
créer une commande `\hlinewd` dont l'épaisseur sera passée en argument. 
Voici un exemple :

```
\documentclass[french]{article}
  \usepackage[T1]{fontenc}    % Encodage T1 (adapté au français)
  \usepackage{lmodern}        % Caractères plus lisibles
  \usepackage{babel}          % Réglages linguistiques (avec french)
  \pagestyle{empty}           % N'affiche pas de numéro de page

  \makeatletter
  \newcommand\hlinewd[1]{%
    \noalign{\ifnum0=`}\fi\hrule \@height #1%
    \futurelet\reserved@a\@xhline}
  \makeatother

\begin{document}
\begin{tabular}{|l|r|}
  \hline
  lundi    & 8  \\ \hlinewd{.5\arrayrulewidth}
  mardi    & 4  \\ \hlinewd{.5\arrayrulewidth}
  mercredi & 9  \\ \hlinewd{.5\arrayrulewidth}
  jeudi    & 13 \\ \hlinewd{.5\arrayrulewidth}
  vendredi & 5  \\ \hlinewd{2\arrayrulewidth}
  total    & 39 \\ \hline
\end{tabular}
\end{document}
```


### Épaissir des traits verticaux à la demande

Pour augmenter l'épaisseur d'un filet de séparation vertical, 
nous allons supprimer le filet vertical par défaut, et en insérer 
un "à la main", avec la commande `\vrule`. Voici comment faire :

```
\documentclass[french]{article}
  \usepackage[T1]{fontenc}    % Encodage T1 (adapté au français)
  \usepackage{lmodern}        % Caractères plus lisibles
  \usepackage{babel}          % Réglages linguistiques (avec french)
  \pagestyle{empty}           % N'affiche pas de numéro de page
  \usepackage{array}

\begin{document}
\begin{tabular}{|c<{\global\addtolength{\arrayrulewidth}{1pt}}|
  >{\global\addtolength{\arrayrulewidth}{-1pt}}c|
  c|c@{\hskip\arraycolsep\vrule width 1.4pt}}
  \hline
  a & b & c & d \\
  \hline
  a & b & c & d \\
  \hline
\end{tabular}
\end{document}
```

Notez que les extensions <ctanpkg:array> et <ctanpkg:mdwtab> définissent 
un nouveau type de séparateur de colonnes, semblable à `@`, 
mais qui ne supprime pas l'espace inter-colonnes. 
L'extension <ctanpkg:mdwtab> fournit en outre une commande, `\vline`, 
qui prend en argument optionnel l'épaisseur du filet.


:::{sources}
[The thickness of rules in LaTeX tables](faquk:FAQ-rulethk)
:::