/**
 * Authors: Frederik Leyvraz, David Degenhardt
 * License: GNU General Public License v3.0 only
 * Version: 1.0.1
 */

package ch.bfh.ti.latexindexer;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import static org.junit.jupiter.api.Assertions.*;

import java.io.File;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.Scanner;

class PlotterTest {

    List<Word> words;
    Plotter plotter;

    @BeforeEach
    public void setUp(){
        words = new LinkedList<>();
        words.add(new Word("The", 5));
        words.add(new Word("be", 4));
        words.add(new Word("to", 3));
        words.add(new Word("of", 2));
        words.add(new Word("and", 1));
        plotter = new Plotter(words);
    }

    @Test
    void testSimpleCLI(){
        assertEquals(plotter.print(new UI.Argument[]{}),
                "The...........5\nbe............4\nto............3\nof............2\nand...........1\n");
    }

    @Test
    void testLimitedNumberCLI(){
        assertEquals(plotter.print(new UI.Argument[]{new UI.Argument("-n", "3")}),
                "The...........5\nbe............4\nto............3\n");
    }

    @Test
    void testAlphabeticalCLI(){
        assertEquals(plotter.print(new UI.Argument[]{new UI.Argument("-c", "a")}),
                "and...........1\nbe............4\nof............2\nThe...........5\nto............3\n");
    }

    @Test
    void testFrequencyCLI(){
        assertEquals(plotter.print(new UI.Argument[]{new UI.Argument("-c", "f")}),
                "The...........5\nbe............4\nto............3\nof............2\nand...........1\n");
    }

    @Test
    void testReverseCLI(){
        assertEquals(plotter.print(new UI.Argument[]{new UI.Argument("-r", "true")}),
                "and...........1\nof............2\nto............3\nbe............4\nThe...........5\n");
    }

    @Test
    void testEmptyCLI(){
        List<Word> wordsEmpty = new LinkedList<>();
        Plotter plotterEmpty = new Plotter(wordsEmpty);
        assertEquals(plotterEmpty.print(new UI.Argument[]{}), "");
    }

    @Test
    void testAllCLI(){
        assertEquals(plotter.print(new UI.Argument[]{new UI.Argument("-c", "a"), new UI.Argument("-r", "true"), new UI.Argument("-n", "3")}),
                "of............2\nbe............4\nand...........1\n");
    }

    @Test
    void testIllegalArgs(){
        System.out.println("[INFO] The following output is part of a test:");
        assertEquals(plotter.print(new UI.Argument[]{new UI.Argument("-x", "cool"), new UI.Argument("-n", "-50"), new UI.Argument("-c", "l"), new UI.Argument("-r", "extraTrue")}),
                "The...........5\nbe............4\nto............3\nof............2\nand...........1\n");
    }

    @Test
    void testPlotTEX(){
        List<Word> words = new LinkedList<>();
        words.add(new Word("The", 5));
        words.add(new Word("be", 4));
        words.add(new Word("to", 3));
        words.add(new Word("of", 2));
        words.add(new Word("and", 1));
        Plotter plotter = new Plotter(words);
        try {
            File expect = new File("src/test/resources/PGFplots/ExpectPlot.tex");
            Scanner scannerExpect = new Scanner(expect);
            System.out.println("[INFO] The following output is part of a test:");
            plotter.generatePlot("src/test/resources/PGFplots", new UI.Argument[]{new UI.Argument("-f", "IsPlot")});
            File is = new File("src/test/resources/PGFplots/IsPlot.tex");
            Scanner scannerIs = new Scanner(is);
            while (scannerExpect.hasNext()){
                assertEquals(scannerExpect.nextLine(), scannerIs.nextLine());
            }
        } catch (IOException e){
            System.out.println("Could not create plot tex");
            e.printStackTrace();
        } catch (InterruptedException e){
            System.out.println("Could not render plot pdf");
            e.printStackTrace();
        }
    }

}
