// Copyright (c) 1991 by Parag Patel.  All Rights Reserved.
static const char rcsid[] = "$Header: readfont.C,v 1.22 91/05/20 16:12:46 hmgr Exp $";

// read font data from the METAFONT GF (generic font) files
//
// by Parag Patel

#include "defs.h"


// maximum number of file that we can keep open at one time - we try to
// keep as many font files open at one time as we can instead of
// reading the entire font into memory - this should gain a little
// speed if only a few characters are loaded from many different fonts
// (which is the normal case)
//
#ifdef _SC_OPEN_MAX
const long MAXFILES = sysconf(_SC_OPEN_MAX) - NUMOPENFILES;
#else
const long MAXFILES = _NFILE - NUMOPENFILES;
#endif

static int numfiles = 0;		// number of files currently open


// try looking for a particular type of font file - this is called 
// multiple times until we succeeding in finding a font we can use
//
static int tryfile(font &f, char *dir, char *mag, char *base, char *ext)
{
    static char buf[MAXPATHLEN + 1];
    sprintf(buf, "%s/%s/%s%s", dir, mag, base, ext);
    char *path = buf;

    debug(3, "looking for font file %s", path);
    FILE *fp = fopen(path, F_READ);
    if (fp == NULL)
	return FALSE;

    // first byte is a preamble byte - same for GF and PK files
    if (getuval(1, fp) != FPRE)
    {
	fclose(fp);
	return FALSE;
    }

    // second byte is the font version ID field
    f.type = (int)getuval(1, fp);
    char *old = f.path;
    f.path = path;
    switch (f.type)
    {
    Case GFID:
	debug(2, "GF font file %s", path);
	setupgffont(f, fp);
    Case PKID:
	debug(2, "PK font file %s", path);
	setuppkfont(f, fp);
    Default:
	fclose(fp);
	f.path = old;
	return FALSE;
    }

    delete old;
    f.path = strdup(path);
    old = f.basename;
    sprintf(buf, "%s/%s%s", mag, base, ext);
    f.basename = strdup(buf);
    delete old;
    debug(6, "font basename = %s", f.basename);

    // keep only a MAXFILES number of font files open at one time
    if (numfiles >= MAXFILES)
    {
	fclose(fp);
	f.fp = NULL;
    }
    else
    {
	f.fp = fp;
	numfiles++;
    }

    return TRUE;
}


// setup a new font in memory - the font is already initialized with
// the information from the DVI file - we now have to actually read the
// font file (GF or PK) to get the rest of the font's description
// 
void setupfont(font &f)
{
    if (f.path != NULL)
	if (tryfile(f, f.path, "", f.basename, ""))
	    return;

    // setup the font directory cache if we need to
    if (pathlist.size() < 1)
	setupdirs(fontpath);

    for (int i = 0; i < pathlist.size(); i++)
    {
	Pathent & p = pathlist[i];

	if (tryfile(f, p.path, pkmagdir(p.dirs, f.mag), f.basename, ".pk"))
	    return;
	if (tryfile(f, p.path, gfmagdir(p.dirs, f.mag), f.basename, ".gf"))
	    return;
	if (tryfile(f, p.path, gfmagdir(p.dirs, f.mag), f.basename, ""))
	    return;
	if (tryfile(f, p.path, pkmagdir(p.dirs, f.mag), f.basename, ""))
	    return;

	char ext[30];
	sprintf(ext, ".%dpk", RESOLUTION);
	if (tryfile(f, p.path, pkmagdir(p.dirs, f.mag), f.basename, ext))
	    return;
	sprintf(ext, ".%dgf", RESOLUTION);
	if (tryfile(f, p.path, pkmagdir(p.dirs, f.mag), f.basename, ext))
	    return;
    }
    quit("Cannot find font file \"%s\" (at or near magnification %ld.%03ld) anywhere",
	    f.basename, f.mag / 1000, f.mag % 1000);
}


// dump the current "fontbits" on screen in a human-readable
// format - this is really for debugging only
// 
void dumpbits(int ch)
{
    register long i, j;
    font & f = *currfont;
    fontchar & g = f.chr[ch];

    // print a row of 2-digit X coordinate values across the top
    fprintf(stderr, "     ");
    for (j = 0; j <= g.maxm - g.minm; j++)
	fprintf(stderr, "%2ld", j + g.minm);
    fprintf(stderr, "\n\n");

    // print the Y coordinate on both sides of each row of pixels
    for (i = g.maxn - g.minn; i >= 0; i--)
    {
	fprintf(stderr, "%3ld  ", i + g.minn);
	for (j = 0; j <= g.maxm - g.minm; j++)
	    fprintf(stderr, "%s", fontbits[i]->isin(j) ? "[]" : " .");
	fprintf(stderr, "  %3ld\n", i + g.minn);
    }

    // and finally another row of X-coords across the bottom
    fprintf(stderr, "\n     ");
    for (j = 0; j <= g.maxm - g.minm; j++)
	fprintf(stderr, "%2ld", j + g.minm);
    fprintf(stderr, "\n");
}


// down load the specified character "ch" to the device - we read this
// character's GF paint commands to build a bitmap of the character in
// memory - then we download this image to the printer
// 
void downchar(int ch, boolean toomany)
{
    font & f = *currfont;
    fontchar & g = f.chr[ch];

    // is this font file open?
    if (f.fp == NULL)
    {
	// we need to close another font file if we already have the
	// maximum number open
	if (numfiles >= MAXFILES)
	{
	    register int i;
	    int fn = -1;
	    long u = MAXLONG;

	    // look for the least-recently used font
	    for (i = 0; i < fontlist.size(); i++)
		if (fontlist[i] != NULL && fontlist[i]->fp != NULL
			&& fontlist[i]->use < u)
		{
		    fn = i;
		    u = fontlist[i]->use;
		}
	    if (fn < 0)
		quit("Cannot open another font file - all fonts are in use");

	    // close this font file
	    fclose(fontlist[fn]->fp);
	    fontlist[fn]->fp = NULL;
	    numfiles--;
	    mesg("@");
	}

	// open the new font file
	f.fp = fopen(f.path, F_READ);
	if (f.fp == NULL)
	    quit("Cannot re-open font file \"%s\"", f.path);
	numfiles++;
    }

    switch (f.type)
    {
    Case GFID:
	getgfchar(f, g, ch);
    Case PKID:
	getpkchar(f, g, ch);
    Default:
	quit("?!BUG? Internal font type id changed");
    }

    // show what this character looks like
    if (debuglevel > 12)
	dumpbits(ch);

    // now download this character to the printer...

    long width = g.maxm - g.minm + 1;
    long height = g.maxn - g.minn + 1;
    if (toomany || g.minm < FHMIN || g.minm > FHMAX || width > FWIDTH
	    || g.maxn < FVMIN || g.maxn > FVMAX || height > FHEIGHT)
    {
	// char is too big - send it down as a graphics
	// image but do NOT mark it as downloaded
	if (!toomany)
	    g.charbig = TRUE;
	dev_bigchar(f, g, ch);
	return;
    }

    // this is the usual method...
    g.downloaded = TRUE;
    dev_downchar(f, g, ch);
}
