/*****************************************************************
**
** MathSpad 0.60
**
** Copyright 1996, Eindhoven University of Technology (EUT)
** 
** Permission to use, copy, modify and distribute this software
** and its documentation for any purpose is hereby granted
** without fee, provided that the above copyright notice appear
** in all copies and that both that copyright notice and this
** permission notice appear in supporting documentation, and
** that the name of EUT not be used in advertising or publicity
** pertaining to distribution of the software without specific,
** written prior permission.  EUT makes no representations about
** the suitability of this software for any purpose. It is provided
** "as is" without express or implied warranty.
** 
** EUT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
** SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL EUT
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
** DAMAGES OR ANY DAMAGE WHATSOEVER RESULTING FROM
** LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
** OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
** OF THIS SOFTWARE.
** 
** 
** Roland Backhouse & Richard Verhoeven.
** Department of Mathematics and Computing Science.
** Eindhoven University of Technology.
**
********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <dirent.h>
#include <sys/stat.h>
#include <pwd.h>
#include <unistd.h>
#include "mathpad.h"
#include "system.h"
#include "funcs.h"
#include "debug.h"
#ifdef SK
#include "sun_kludge.h"
#endif

static char *currentwd=NULL;
static int currentwdlen=0;
int user_id=0, group_id=0;
Bool failure=False;

int mystrtoi(char *s, char **endp, int base)
{
    char *h = s;
    int n=0;
    char c;
    char sn='0', en='0', sl='A', el='@';

#define isbasenumber(A) (sn<=(A) && ((A)<=en || (sl<=(A) && (A)<=el)))

    if (!s) {
        endp = NULL;
        return 0;
    }
    while (isspace(*h)) h++;
    if (!base) {
        if (*h!='0') base=10;
        else {
            h++;
            if (*h=='X' || *h=='x') {
                base=16;
                h++;
            } else
                base=8;
        }
    }
    en = '0'+base-1;
    if (base>10) {
        en = '9';
        el = 'A'+base-1;
    }
    c = toupper(*h);
    while (isbasenumber(c)) {
        if (c<='9')
            n = n*base+c-'0';
        else
            n = n*base+c-'A'+10;
        h++;
        c = toupper(*h);
    }
    if (endp)
        *endp = h;
    return n;
}

char *begins_with(char *s, char *t)
{
    while (s && *s)
        if (!(toupper(*s++) == toupper(*t++)))
            return NULL;
    while (isspace(*t)) t++;
    return t;
}

char *strip_name(char *name)
{
    char *c;

    c = strrchr(name, '/');
    if (c)
        return c+1;
    else
        return name;
}

void concat_in(char *dest, char *s1, char *s2)
{
    if (s1) {
        while ((*dest++ = *s1++));
        dest--;
    }
    if (s2)
        while ((*dest++ = *s2++));
    else *dest = '\0';
}

char *concat(char *s1, char *s2)
{
    char *temp;
    temp = (char *) malloc((s1?strlen(s1):0) +
                           (s2?strlen(s2):0) + 1);
    concat_in(temp, s1, s2);
    return temp;
}

FILE *open_file(char *dir, char *filename, char *mode)
{
    char tchar[500];
    FILE *tfile;

    concat_in(tchar, dir, "/");
    concat_in(tchar, tchar, filename);
    tfile = fopen(tchar, mode);
    return tfile;
}

static Bool can_access_file(struct stat *stbuf)
{
    if (!user_id || !group_id) return True;
    if (stbuf->st_uid==user_id) {
        if (!(stbuf->st_mode & S_IRUSR)) return False;
        if (((stbuf->st_mode & S_IFMT) == S_IFDIR) &&
            !(stbuf->st_mode & S_IXUSR)) return False;
        return True;
    }
    if (stbuf->st_gid==group_id) {
        if (!(stbuf->st_mode & S_IRGRP)) return False;
        if (((stbuf->st_mode & S_IFMT) == S_IFDIR) &&
            !(stbuf->st_mode & S_IXGRP)) return False;
        return True;
    }
    if (!(stbuf->st_mode & S_IROTH)) return False;
    if (((stbuf->st_mode & S_IFMT) == S_IFDIR) &&
        !(stbuf->st_mode & S_IXOTH)) return False;
    return True;
}

Bool is_directory(char *name)
{
    struct stat stbuf;
    if (stat(name, &stbuf) != -1 && can_access_file(&stbuf))
        return ((stbuf.st_mode & S_IFMT)==S_IFDIR);
    else
        return 0;
}

static Bool is_file(char *name)
{
    struct stat stbuf;
    if (stat(name, &stbuf) != -1 && can_access_file(&stbuf))
        return ((stbuf.st_mode & S_IFMT)!=S_IFDIR);
    else
        return 0;
}

static int file_size(char *name)
{
    struct stat stbuf;
    if (stat(name, &stbuf) != -1 && can_access_file(&stbuf))
        return (stbuf.st_size);
    else
        return 0;
}

static char search_result[1000];

char *search_through_dirs(char **dirs, int nr, char *filename)
{
    int i,j,k,n;
    Bool fname;
    Bool f = False;
    char *buffer=search_result;

    for (i=0; !f && i<nr; i++) {
        fname=False;
        k=0;
        for (j=0; dirs[i][j]; j++) {
            if (dirs[i][j]=='%' && !fname) {
                for (n=0; filename[n]; n++,k++)
                    buffer[k]=filename[n];
                fname=True;
            } else
                buffer[k++]=dirs[i][j];
        }
        if (!fname) {
            if (buffer[k-1]!='/') buffer[k++]='/';
            for (j=0; filename[j]; buffer[k++]=filename[j++]);
        }
        buffer[k]='\0';
        f = is_file(buffer);
    }
    if (f)
        return buffer;
    else
        return NULL;
}

static int qsortstrcmp(const void *e1,const  void *e2)
{
    return strcmp( *((char**) e1), *((char**) e2)); 
}

static char checkname[1024];

int read_dir_contents(char *dirname, Bool only_files,
		      Bool (*mask_check)(char*,char*),
		      char *mask,
		      char **dirs, int *nr_dirs, char **files, int *nr_files)
{
    DIR *dr;
    struct dirent *de;
    struct stat stbuf;
    char *ldp;
    int i,n,ld=0,lf=0;
    char *buf;
    char *buffer2;
    int idx;
    char **found_files;

    *nr_dirs = 0;
    *nr_files = 0;
    if (!(dr = opendir(dirname))) return 0;

    strcpy(checkname, dirname);
    ldp = checkname+(strlen(checkname)-1);
    if (*ldp!='/') {
        ldp[1] = '/';
        ldp[2] = '\0';
        ldp += 2;
    } else
        ldp++;
    n=file_size(dirname);
    buffer2 = (char *) malloc(n);
    found_files = (char **) malloc(n*sizeof(char*));
    idx = 0;
    while ((de = readdir(dr))) {
        strcpy(ldp,de->d_name);
        i = strlen(de->d_name);
        if (stat(checkname, &stbuf) != -1 && can_access_file(&stbuf)) {
            if ((stbuf.st_mode & S_IFMT)==S_IFDIR) {
                if (strcmp(de->d_name,".") && !only_files) {
                    found_files[n/2+*nr_dirs] = buffer2+idx;
                    (*nr_dirs)++;
                    strcpy(buffer2+idx, de->d_name);
                    idx += i+1;
                    ld+=i+1;
                }
            } else {
                if (mask_check(mask,de->d_name)) {
                    found_files[*nr_files] = buffer2+idx;
                    (*nr_files)++;
                    strcpy(buffer2+idx, de->d_name);
                    idx += i+1;
                    lf+=i+1;
                }
            }
        }
    }
    closedir(dr);
    if (*nr_dirs) {
        qsort(found_files+n/2, *nr_dirs, sizeof(char *), qsortstrcmp);
        buf = (char *) malloc(ld);
        buf[0] = '\0';
        ldp = buf;
        i = n/2;
        while (i<n/2+*nr_dirs) {
            strcpy(ldp, found_files[i]);
            ldp += strlen(ldp);
            *ldp++ = '\n';
            i++;
        }
        ldp--;
        *ldp='\0';
        *dirs = buf;
    }
    if (*nr_files) {
        qsort(found_files, *nr_files, sizeof(char *), qsortstrcmp);
        buf = (char*) malloc(lf);
        buf[0]='\0';
        ldp=buf;
        i=0;
        while (i<*nr_files) {
            strcpy(ldp, found_files[i]);
            ldp += strlen(ldp);
            *ldp++='\n';
            i++;
        }
        ldp--;
        *ldp='\0';
        *files = buf;
    }
    myfree(buffer2);
    myfree(found_files);
    return *nr_files+ *nr_dirs;
}

static void expand_user(char *name, char *result)
{
    struct passwd *pinfo=NULL;

    if (!name || !result) return;
    setpwent();
    if (*name)
        pinfo = getpwnam(name);
    else
        pinfo = getpwuid(user_id);
    endpwent();
    if (pinfo)
        strcpy(result, pinfo->pw_dir);
    else if (*name) {
	char *c=getenv(name);
	if (c) strcpy(result, c);
    }
}

void get_currentwd(void)
{
    char *p;
    char buf[2048];
    struct stat fs1,fs2;

    p=getenv("PWD");
    if (!p || stat(p,&fs1) ||
        (!stat(".",&fs2) && fs1.st_ino != fs2.st_ino)) {
        p=NULL;
        if (getcwd((char *)buf, 2048))
            p=buf;
    }
    user_id=getuid();
    group_id=getgid();
    currentwd = concat(p,"");
    currentwdlen=strlen(currentwd);
}

static char *expand_environment(char *name)
{
    int i=0,j=0,expanded=0;
    char buffer[2000];
    if (!name) return NULL;
    do {
        if (name[j]=='$' && isalpha(name[j+1])) {
            int k=j+1;
            char *en;
            char c;
            j++;
            expanded=1;
            while (isalnum(name[j]) || name[j]=='_') j++;
            c=name[j];
            name[j]='\0';
            en = getenv(name+k);
            name[j]=c;
            if (en) while (*en) buffer[i++]=*en++;
        }
        buffer[i++]=name[j];
    } while (name[j++]);
    if (expanded) {
        if (j<i) {
            free(name);
            name = concat(buffer,"");
        } else
            strcpy(name, buffer);
    }
    return name;
}
            
static int pointstate[7] = { 0, 2, 3, 0, 5, 5, 0 };
static int tildestate[7] = { 0, 4, 0, 0, 5, 5, 4 };
static int defaultstate[7] = {0, 0, 0, 0, 5, 5, 0 };

char *standard_dir(char *dirname)
{
    int i = 0, state = 6, j = 0;
    Bool ok;
    char buffer[2000];
    char *c = expand_environment(dirname);

    if (!c) return NULL;
    dirname=c;
    /* meaning of state:
    ** 1:  ending in /
    ** 2:  ending in /.
    ** 3:  ending in /..
    ** 4:  ending in /~
    ** 5:  ending in /~[^/]*
    ** 0:  everything else.
    */
    while (*c) {
        switch (*c) {
        case '.':
            state = pointstate[state];
            buffer[i++]='.';
            break;
        case '/':
            switch (state) {
            case 1:
                break;
            case 2:
                i--;
                state = 1;
                break;
            case 3:
                buffer[i]='/';
                i -= 2;
                ok = (i>=2 && buffer[i-2]!='.') ||
                     (i>=3 && buffer[i-3]!='.') ||
                     (i>=4 && buffer[i-4]!='/');
                if (ok) {
                    i-=2;
                    while (i>0 && buffer[i]!='/') i--;
                    if (buffer[i]=='/') i++;
                } else
                    i+=3;
                state = i>0;
                break;
            case 4:
            case 5:
                buffer[i]='\0';
                while (i>0 && buffer[i]!='~') i--;
                if (buffer[i]=='~') {
                    i++;
                    expand_user(buffer+i,buffer);
                    i = strlen(buffer);
                    if (!i || buffer[i-1]!='/') buffer[i++]='/';
                } else {
                    i = strlen(buffer);
                    buffer[i++]='/';
                }
                state = 1;
                break;
            default:
                buffer[i++]='/';
                state = 1;
                break;
            }
            break;
        case '~':
            state = tildestate[state];
            buffer[i++]='~';
            break;
        default:
            state = defaultstate[state];
            buffer[i++]= *c;
            break;
        }
        c++;
        j++;
    }
    switch (state) {
    case 2:
        i--;
        break;
    case 3:
        i -= 2;
        ok = (i>=2 && buffer[i-2]!='.') ||
            (i>=3 && buffer[i-3]!='.') ||
            (i>=4 && buffer[i-4]!='/');
        if (ok) {
            i-=2;
            while (i>0 && buffer[i]!='/') i--;
            if (buffer[i]=='/') i++;
        } else
            i+=2;
        break;
    case 4:
    case 5:
        buffer[i]='\0';
        while (i>0 && buffer[i]!='~') i--;
        if (buffer[i]=='~') {
            i++;
            expand_user(buffer+i,buffer);
            i = strlen(buffer);
            if (!i || buffer[i-1]!='/') buffer[i++]='/';
        } else {
            i = strlen(buffer);
            buffer[i++]='/';
        }
        break;
    default:
        break;
    }
    if (!i) {
        strcpy(buffer, currentwd);
        i=currentwdlen;
    }   
    buffer[i]='\0';
    if (j<i) {
        free(dirname);
        dirname = concat(buffer,"");
    } else
        strcpy(dirname, buffer);
    return dirname;
}

void remove_file(char *name)
{
    unlink(name);
}

int mystrlen(char *s)
{
    int i=0;

    if (s) while (*s && *s!='\n') s++,i++;
    return i;
}

int charstrlen(Char *s)
{
    int i=0;

    if (!s) return 0;
    while (s[i]) i++;
    return i;
}

void charstrcpy(Char *s, Char *t)
{
    if (!t) return;
    if (!s) {
        t[0]=0;
        return;
    }
    while ((*t++=*s++));
}

Char *char2Char(char *txt)
{
    Char *temp;

    temp = (Char*) malloc(sizeof(Char) * (strlen(txt)+1));
    if (temp) {
        Index i = 0;
        while (txt[i]) {
            if (txt[i]=='\n')
                temp[i] = Newline;
            else if (txt[i]=='\t')
                temp[i] = Rtab;
            else
                temp[i] = txt[i];
            i++;
        }
        temp[i] = 0;
    }
    return temp;
}

int skip_fontpart(FILE *f)
{
    
#define get_new_string (not_finished = ((fgets(buffer, 2000, f)!=NULL) && \
					!begins_with("STOP", buffer)))

    int nr;
    Bool not_finished;
    char *ind;
    char buffer[2000];

    nr=0;
    get_new_string;
    while (not_finished && nr<MAXFONTS) {
	if ((ind = begins_with("FONT:", buffer)) ||
	    (ind = begins_with("FONTGROUP:", buffer)))
	    nr++;
	else
	    if ((ind = begins_with("BUTTONFONT:", buffer)))
		nr++;
	get_new_string;
    }
    if (begins_with("STOP", buffer))
	return nr;
    else return 0;
}
