"""
Pour pouvoir fabriquer un PDF incluant la compilation des exemples
"""

from sphinx.writers.latex import CR, LaTeXTranslator
from sphinx.builders.latex.nodes import  captioned_literal_block
from docutils import nodes
import hashlib

# On n'utilise pas de shell-escape, aussi pour les documents externalisés
latexmkopt = {
    'latexmkpdf': '-pdf',
    'latexmklua': '-lualatex',
    'latexmkxe' : '-xelatex',
}

class FAQLaTeXGUTranslator(LaTeXTranslator):

    def append_srccode(self, node):
        # Ici on implémente une couche logique pour
        # 1) déterminer si c'est du code latex
        # 2) s'il s'agit bien d'un document complet
        # 3) s'il doit être compilé avec lualatex ou pdflatex
        #
        # À utiliser par
        #
        #     make latex
        #     make -C build/latex FAQLaTeXGUTenberg.pdf LATEXOPTS="--shell-escape"
        #
        # On dispose au 31 mai d'un "make latexpdf-shell-escape".
        #
        # TODO: 1) traduire en français les commentaires encore en anglais.
        #       2) s'occuper des longues lignes de code s'il en reste.
        lang = node.get('language', 'default')
        if lang == "latex" and 'noedit' not in node.get('classes', []):
            # MEMO: 1) idéalement tout code sans le "noedit" est censé être compilable
            #       2) lorsque ce sera le cas il suffira ici de vérifier la présence d'un
            #          '%!TEX engine=' pour utiliser lualatex ou xelatex et non pas le
            #          pdflatex de défaut.
            #       3) on pourrait faire par latex+dvips+ps2pdf mais pour le moment
            #          tous les exemples PSTricks sont compilés par lualatex
            # MEMO: le document principal est compilé par lualatex mais presque tous les
            #       documents extraits le sont par pdflatex.  Actuellement la taille du
            #       PDF final est environ 38M mais ps2pdf le fait descendre à moins de 17M.
            #       (au tout début on compilait les documents extraits par lualatex,
            #        cela donnait de l'ordre de 27M mais on n'a jamais fait ps2pdf dessus).
            srccode = node.rawsource.strip()
            srccodelines = srccode.splitlines()
            # CALCUL D'UN MD5, qu'on utilisera en entier si on arrive pas
            # à retrouver un nom de section et sinon seulement ses 8
            # premiers chiffres hexa.
            #
            # Après on va modifier srccode pour lui ajouter un \pagestyle{empty}
            # Faudrait-il ici faire le hash (MD5 maintenant) du srccode modifié ?
            # Je ne vois pas trop l'intérêt que cela pourrait avoir
            #
            hashedcontent = hashlib.md5(srccode.encode()).hexdigest()
            # TODO: est-ce que le séparateur est bien toujours / ici même sur Windows?
            sanitizeddocname = ""
            nodeparent = node  # pas encore parent...
            while sanitizeddocname == "":
                nodeparent = nodeparent.parent
                sanitizeddocname = nodeparent.get('docname', "").replace('/', '_')
            # pas de façon hyper simple de faire du -1, -2, -3, ... pour
            # les exemples avec même docname ; 8 premiers chiffres du hash
            # devraient suffire.
            examplename = sanitizeddocname + '-' + hashedcontent[:8]

            if '\\documentclass' not in srccode:
                print("************************************************************************")
                print("Dans le fichier :")
                print()
                print("%s.md" % nodeparent.get('docname', ""))
                print()
                if len(srccodelines) == 0:
                    print("Il y a un exemple avec 0 ligne, voici pour le localiser :")
                    print()
                    print(node)
                    print()
                    print(node.parent)
                    print()
                    print("Au boulot Denis !")
                    assert False, "Sources corrompues"
                elif len(srccodelines) < 3:
                    print("Le fragment ci-dessous n'est pas marqué {noedit}:")
                    print()
                    print(srccode)
                    print()
                    print("Il n'a pas de \\documentclass et ne sera pas externalisé.")
                else:
                    print("Le fragment débutant par :")
                    print()
                    for j in range(3):
                        print(srccodelines[j])
                    print()
                    print("n'est pas marqué {noedit}.")
                    print("Il n'a pas de \\documentclass et ne sera pas externalisé.")
                print()
                return None

            # ON UTILISE LATEXMK POUR COMPILER LES EXEMPLES
            # (mais ça veut dire au moins deux compils pour chaque)
            # il va falloir empêcher latexmk d'utiliser le latexmkrc prévu pour Sphinx
            # Le compilateur par défaut sera pdflatex
            # Pour le moment pas de mécanisme pour faire latex+dvips (PSTricks)
            compiler = 'latexmkpdf'
            # MEMO: toujours utiliser la balise %!TEX engine=lualatex pour signaler
            #       que le document doit utiliser lualatex.  À terme on pourra supprimer
            #       les tests de présence de fontspec, et pour PSTricks on n'aura pas
            #       à tester pour '\usepackage{ps'.
            if (srccodelines[0].rstrip().endswith(('lualatex','LuaLaTeX')) or
                r'\directlua' in srccode or
                r'\usepackage{ps' in srccode or # PSTricks (pstricks, pst-plot, psgo, etc...)
                r'\usepackage{fontspec}' in srccode or
                r'\usepackage{unicode-math}' in srccode or
                r'\usepackage[domains]{pgfmolbio}' in srccode):
                compiler = 'latexmklua'
            if srccodelines[0].rstrip().endswith('xelatex'):
                compiler = 'latexmkxe'
            if (r'\documentclass{ctexart}' in srccode or
                r'\documentclass{ctexbeamer}' in srccode or
                r'\usepackage[heading = true]{ctex}' in srccode):
                compiler = 'latexmkxe'  # parce que "lualatex could not find Kaiti SC"

            # On a toute l'architecture pour faire du
            # minted avec shell-escape que ce soit en pdflatex, lualatex,
            # ou xelatex.  Mais le balisage est partagé avec la sortie
            # HTML et latexcgi ne compile pas les documents nécessitant
            # shell-escape.  Donc actuellement (31 mai 2024) ces documents
            # sont {noedit} et n'arrivent pas ici.
            # Finalement le 2 juin on a tout commenté-out pour le -shell-escape
            # de compilation des fichiers extraits, pour les mêmes raisons
            # qu'on ne fait plus la compilation globale en -shell-escape
            # if '{minted' in srccode:
            #     compiler = compiler + 'sh'

            # Externalisation de l'engin à utiliser
            self.body.append(r'\begin{filecontents*}[overwrite,nowarn,nosearch]'
                             '{codedir/%s.latexmkopt}' % examplename + CR)
            self.body.append(latexmkopt[compiler])
            self.body.append(CR + r'\end{filecontents*}' + CR)

            # Externalisation du code LaTeX à compiler.  On ne peut pas utiliser
            # filecontents* car la source à externaliser pourrait elle l'avoir fait.
            self.body.append(r'\begin{faqGUTfilecontents}[overwrite,nowarn,nosearch]'
                             '{codedir/%s}' % examplename + CR)
            # Bien que cela n'ait pas pour le moment une grande
            # importance, puisque les fichiers externalisés dans codedir
            # sont surtout à usage interne, (peut-être un jour on voudra
            # "attacher" ces sources au PDF, et d'ailleurs on aura alors
            # peut-être trouvé le moyen de les externaliser par Python,
            # pas par l'intermédiaire de filecontents dont tout ce qui est
            # fait ici sera obsolète), on fait en sorte que les ascii 9
            # externalisés donnent 4 espaces et non pas 1 seul si on
            # laissait faire le filecontents de LaTeX (qu'il est pour
            # ainsi dire à peu près impossible de customiser, car est une
            # unique grosse macro et le \space est hard-codé à
            # l'intérieur).
            srccode_edited = srccode.replace('\t', 4 * ' ')
            # On essaye d'empêcher un  numéro de bas de page, en vue de PDFCROP.
            if (
                ('pagestyle' not in srccode)
                and (r'\tableofcontents' not in srccode)
                and (r'\chapter' not in srccode)
            ):
                srccode_edited = srccode_edited.replace(
                    r'\begin{document}',
                    r'\begin{document}\pagestyle{empty}'
                )
            self.body.append(srccode_edited)
            self.body.append(CR + r'\end{faqGUTfilecontents}' + CR)

            # INCLUSION par \sphinxincludegraphics et al.
            # Voir FAQoutputbox.sty pour \faqGUTincludepdf
            self.body.append(r'\faqGUTincludepdf{%s}' % examplename + CR)

    def visit_literal_block(self, node):
        ## On pourrait préfixer par
        ##     if node.rawsource == node.astext():
        ## Mais de toute façons l'exception nodes.SkipNode n'est levée
        ## que dans le cas où un environnement sphinxVerbatim est produit.
        ## Néanmoins, le fait que le visit_literal_block originel ait deux
        ## branches et que nous ne voulions intervenir que dans la branche
        ## donnant un sphinxVerbatim induit un très léger risque en cas
        ## de modifications "upstream", très improbable cependant.
        ##    (le SkipNode c'est parce que depart_literal_block()
        ##     n'est utilisé par Sphinx que lorsqu'il utilise un alltt)
        try:
            super().visit_literal_block(node)
        except nodes.SkipNode:
            self.append_srccode(node)
            raise nodes.SkipNode
