      SUBROUTINE DSYRK( UPLO, TRANS, N, K, ALPHA, A, LDA,
     $                   BETA, C, LDC )
*     .. Scalar Arguments ..
      CHARACTER*1        UPLO, TRANS
      INTEGER            N, K, LDA, LDC
      DOUBLE PRECISION   ALPHA, BETA
*     .. Array Arguments ..
      DOUBLE PRECISION   A( LDA, * ), C( LDC, * )
*     ..
*
*  Purpose
*  =======
*
*  DSYRK  performs one of the symmetric rank k operations
*
*     C := alpha*A*A' + beta*C,
*
*  or
*
*     C := alpha*A'*A + beta*C,
*
*  where  alpha and beta  are scalars, C is an  n by n  symmetric matrix
*  and  A  is an  n by k  matrix in the first case and a  k by n  matrix
*  in the second case.
*
*  Parameters
*  ==========
*
*  UPLO   - CHARACTER*1.
*           On  entry,   UPLO  specifies  whether  the  upper  or  lower
*           triangular  part  of the  array  C  is to be  referenced  as
*           follows:
*
*              UPLO = 'U' or 'u'   Only the  upper triangular part of  C
*                                  is to be referenced.
*
*              UPLO = 'L' or 'l'   Only the  lower triangular part of  C
*                                  is to be referenced.
*
*           Unchanged on exit.
*
*  TRANS  - CHARACTER*1.
*           On entry,  TRANS  specifies the operation to be performed as
*           follows:
*
*              TRANS = 'N' or 'n'   C := alpha*A*A' + beta*C.
*
*              TRANS = 'T' or 't'   C := alpha*A'*A + beta*C.
*
*              TRANS = 'C' or 'c'   C := alpha*A'*A + beta*C.
*
*           Unchanged on exit.
*
*  N      - INTEGER.
*           On entry,  N specifies the order of the matrix C.  N must be
*           at least zero.
*           Unchanged on exit.
*
*  K      - INTEGER.
*           On entry with  TRANS = 'N' or 'n',  K  specifies  the number
*           of  columns   of  the   matrix   A,   and  on   entry   with
*           TRANS = 'T' or 't' or 'C' or 'c',  K  specifies  the  number
*           of rows of the matrix  A.  K must be at least zero.
*           Unchanged on exit.
*
*  ALPHA  - DOUBLE PRECISION.
*           On entry, ALPHA specifies the scalar alpha.
*           Unchanged on exit.
*
*  A      - DOUBLE PRECISION array of DIMENSION ( LDA, ka ), where ka is
*           k  when  TRANS = 'N' or 'n',  and is  n  otherwise.
*           Before entry with  TRANS = 'N' or 'n',  the  leading  n by k
*           part of the array  A  must contain the matrix  A,  otherwise
*           the leading  k by n  part of the array  A  must contain  the
*           matrix A.
*           Unchanged on exit.
*
*  LDA    - INTEGER.
*           On entry, LDA specifies the first dimension of A as declared
*           in  the  calling  (sub)  program.   When  TRANS = 'N' or 'n'
*           then  LDA must be at least  max( 1, n ), otherwise  LDA must
*           be at least  max( 1, k ).
*           Unchanged on exit.
*
*  BETA   - DOUBLE PRECISION.
*           On entry, BETA specifies the scalar beta.
*           Unchanged on exit.
*
*  C      - DOUBLE PRECISION array of DIMENSION ( LDC, n ).
*           Before entry  with  UPLO = 'U' or 'u',  the leading  n by n
*           upper triangular part of the array C must contain the upper
*           triangular part  of the  symmetric matrix  and the strictly
*           lower triangular part of C is not referenced.  On exit, the
*           upper triangular part of the array  C is overwritten by the
*           upper triangular part of the updated matrix.
*           Before entry  with  UPLO = 'L' or 'l',  the leading  n by n
*           lower triangular part of the array C must contain the lower
*           triangular part  of the  symmetric matrix  and the strictly
*           upper triangular part of C is not referenced.  On exit, the
*           lower triangular part of the array  C is overwritten by the
*           lower triangular part of the updated matrix.
*
*  LDC    - INTEGER.
*           On entry, LDC specifies the first dimension of C as declared
*           in  the  calling  (sub)  program.   LDC  must  be  at  least
*           max( 1, n ).
*           Unchanged on exit.
*
*
*  Level 3 Blas routine.
*
*  -- Written on 8-February-1989.
*     Jack Dongarra, Argonne National Laboratory.
*     Iain Duff, AERE Harwell.
*     Jeremy Du Croz, Numerical Algorithms Group Ltd.
*     Sven Hammarling, Numerical Algorithms Group Ltd.
*
*  -- Rewritten in December-1993.
*     GEMM-Based Level 3 BLAS.
*     Per Ling, Institute of Information Processing,
*     University of Umea, Sweden.
*
*  -- Rewritten in Mars-1995.
*     Superscalar GEMM-Based Level 3 BLAS (Version 0.1).
*     Per Ling, Department of Computing Science,
*     University of Umea, Sweden.
*
*
*     .. Local Scalars ..
      INTEGER            INFO, NROWA
      INTEGER            I, II, IX, ISEC, J, JX, UJ, L, LL, LSEC
      INTEGER            UISEC, RISEC, UJSEC, ULSEC
      LOGICAL            UPPER, NOTR
      DOUBLE PRECISION   FR1, FR2, FR3, FR4, FR5, FR6, FR7, FR8
      DOUBLE PRECISION   DELTA
*     .. Intrinsic Functions ..
      INTRINSIC          MAX, MIN, MOD, SQRT
*     .. External Functions ..
      LOGICAL            LSAME
      EXTERNAL           LSAME
*     .. External Subroutines ..
      EXTERNAL           XERBLA
      EXTERNAL           DGEMM
*     .. Parameters ..
      DOUBLE PRECISION   ONE, ZERO
      PARAMETER        ( ONE = 1.0D+0, ZERO = 0.0D+0 )
*     .. User specified parameters for DSYRK ..
      INTEGER            RB, CB
      PARAMETER        ( RB = 32, CB = 32 )
*     .. Local Arrays ..
      DOUBLE PRECISION   T1( RB, CB )
*     ..
*     .. Executable Statements ..
*
*     Test the input parameters.
*
      UPPER = LSAME( UPLO, 'U' )
      NOTR = LSAME( TRANS, 'N' )
      IF( NOTR )THEN
         NROWA = N
      ELSE
         NROWA = K
      END IF
      INFO = 0
      IF( ( .NOT.UPPER ).AND.( .NOT.LSAME( UPLO, 'L' ) ) )THEN
         INFO = 1
      ELSE IF( ( .NOT.NOTR ).AND.( .NOT.LSAME( TRANS, 'T' ) ).AND.
     $                                ( .NOT.LSAME( TRANS, 'C' ) ) )THEN
         INFO = 2
      ELSE IF( N.LT.0 )THEN
         INFO = 3
      ELSE IF( K.LT.0 )THEN
         INFO = 4
      ELSE IF( LDA.LT.MAX( 1, NROWA ) )THEN
         INFO = 7
      ELSE IF( LDC.LT.MAX( 1, N ) )THEN
         INFO = 10
      END IF
      IF( INFO.NE.0 )THEN
         CALL XERBLA( 'DSYRK ', INFO )
         RETURN
      END IF
*
*     Quick return if possible.
*
      IF( ( N.EQ.0 ).OR.
     $    ( ( ( ALPHA.EQ.ZERO ).OR.( K.EQ.0 ) ).AND.( BETA.EQ.ONE ) ) )
     $   RETURN
*
*     And when alpha.eq.zero or k.eq.0.
*
      IF( ( ALPHA.EQ.ZERO ).OR.( K.EQ.0 ) )THEN
         IF( UPPER )THEN
            IF( BETA.EQ.ZERO )THEN
               DO 30, J = 1, N
                  UISEC = J-MOD( J, 4 )
                  DO 10, I = 1, UISEC, 4
                     C( I, J ) = ZERO
                     C( I+1, J ) = ZERO
                     C( I+2, J ) = ZERO
                     C( I+3, J ) = ZERO
   10             CONTINUE
                  DO 20, I = UISEC+1, J
                     C( I, J ) = ZERO
   20             CONTINUE
   30          CONTINUE
            ELSE
               DO 60, J = 1, N
                  UISEC = J-MOD( J, 4 )
                  DO 40, I = 1, UISEC, 4
                     C( I, J ) = BETA*C( I, J )
                     C( I+1, J ) = BETA*C( I+1, J )
                     C( I+2, J ) = BETA*C( I+2, J )
                     C( I+3, J ) = BETA*C( I+3, J )
   40             CONTINUE
                  DO 50, I = UISEC+1, J
                     C( I, J ) = BETA*C( I, J )
   50             CONTINUE
   60          CONTINUE
            END IF
         ELSE
            IF( BETA.EQ.ZERO )THEN
               DO 100, J = 1, N
                  RISEC = MOD( N-J, 4 )+1
                  DO 80, I = J, J+RISEC-1
                     C( I, J ) = ZERO
   80             CONTINUE
                  DO 90, I = J+RISEC, N, 4
                     C( I, J ) = ZERO
                     C( I+1, J ) = ZERO
                     C( I+2, J ) = ZERO
                     C( I+3, J ) = ZERO
   90             CONTINUE
  100          CONTINUE
            ELSE
               DO 130, J = 1, N
                  RISEC = MOD( N-J, 4 )+1
                  DO 110, I = J, J+RISEC-1
                     C( I, J ) = BETA*C( I, J )
  110             CONTINUE
                  DO 120, I = J+RISEC, N, 4
                     C( I, J ) = BETA*C( I, J )
                     C( I+1, J ) = BETA*C( I+1, J )
                     C( I+2, J ) = BETA*C( I+2, J )
                     C( I+3, J ) = BETA*C( I+3, J )
  120             CONTINUE
  130          CONTINUE
            END IF
         END IF
         RETURN
      END IF
*
*     Start the operations.
*
      IF( UPPER )THEN
         IF( NOTR )THEN
*
*           Form  C := alpha*A*A' + beta*C, C is stored in the
*           upper triangular part.
*
            DO 240, II = 1, N, CB
               ISEC = MIN( CB, N-II+1 )
*
*              C := alpha*A*A' + beta*C, general matrix multiply for
*              upper rectangular vertical blocks of C.
*
               IF( II.GT.1 )THEN
                  CALL DGEMM ( 'N', 'T', II-1, ISEC, K, ALPHA,
     $                                  A( 1, 1 ), LDA, A( II, 1 ), LDA,
     $                                           BETA, C( 1, II ), LDC )
               END IF
*
*              C := alpha*A*A' + C, update a upper triangular diagonal
*              block of C with alpha*A*A'.
*
               DO 230, LL = 1, K, RB
                  LSEC = MIN( RB, K-LL+1 )
                  ULSEC = LSEC-MOD( LSEC, 2 )
*
*                 Determine if the block of C should be updated with
*                 beta or not.
*
                  DELTA = ONE
                  IF( LL.EQ.1 ) DELTA = BETA
*
*                 T1 := A', copy the transpose of a rectangular
*                 block of A to T1.
*
                  UISEC = ISEC-MOD( ISEC, 2 )
                  DO 150, L = LL, LL+ULSEC-1, 2
                     DO 140, I = II, II+UISEC-1, 2
                        T1( L-LL+1, I-II+1 ) = A( I, L )
                        T1( L-LL+2, I-II+1 ) = A( I, L+1 )
                        T1( L-LL+1, I-II+2 ) = A( I+1, L )
                        T1( L-LL+2, I-II+2 ) = A( I+1, L+1 )
  140                CONTINUE
                     IF( UISEC.LT.ISEC )THEN
                        T1( L-LL+1, ISEC ) = A( II+ISEC-1, L )
                        T1( L-LL+2, ISEC ) = A( II+ISEC-1, L+1 )
                     END IF
  150             CONTINUE
                  IF( ULSEC.LT.LSEC )THEN
                     DO 160, I = II, II+ISEC-1
                        T1( LSEC, I-II+1 ) = A( I, LL+LSEC-1 )
  160                CONTINUE
                  END IF
*
*                 C := alpha*T1'*T1 + C, a upper triangular diagonal
*                 block of C is updated with beta.
*
                  DO 220, JX = II+ISEC-1, II, -2
                     UJ = JX-2+1
                     UJSEC = JX+1-MAX( UJ, II )
                     UISEC = UJ-II-MOD( UJ-II, 4 )
*
*                    Four by two unrolling.
*
                     IF( UJSEC.EQ.2 )THEN
                        IF( UISEC.GE.4. )THEN
                           DO 180, I = II, II+UISEC-1, 4
                              FR1 = ZERO
                              FR2 = ZERO
                              FR3 = ZERO
                              FR4 = ZERO
                              FR5 = ZERO
                              FR6 = ZERO
                              FR7 = ZERO
                              FR8 = ZERO
                              DO 170, L = 1, LSEC
                                 FR1 = FR1 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+1 )
                                 FR2 = FR2 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+2 )
                                 FR3 = FR3 + T1( L, I-II+2 )*
     $                                                  T1( L, UJ-II+1 )
                                 FR4 = FR4 + T1( L, I-II+2 )*
     $                                                  T1( L, UJ-II+2 )
                                 FR5 = FR5 + T1( L, I-II+3 )*
     $                                                  T1( L, UJ-II+1 )
                                 FR6 = FR6 + T1( L, I-II+3 )*
     $                                                  T1( L, UJ-II+2 )
                                 FR7 = FR7 + T1( L, I-II+4 )*
     $                                                  T1( L, UJ-II+1 )
                                 FR8 = FR8 + T1( L, I-II+4 )*
     $                                                  T1( L, UJ-II+2 )
  170                         CONTINUE
                              C( I, UJ ) = DELTA*C( I, UJ ) + ALPHA*FR1
                              C( I, UJ+1 ) = DELTA*C( I, UJ+1 ) +
     $                                                         ALPHA*FR2
                              C( I+1, UJ ) = DELTA*C( I+1, UJ ) +
     $                                                         ALPHA*FR3
                              C( I+1, UJ+1 ) = DELTA*C( I+1, UJ+1 ) +
     $                                                         ALPHA*FR4
                              C( I+2, UJ ) = DELTA*C( I+2, UJ ) +
     $                                                         ALPHA*FR5
                              C( I+2, UJ+1 ) = DELTA*C( I+2, UJ+1 ) +
     $                                                         ALPHA*FR6
                              C( I+3, UJ ) = DELTA*C( I+3, UJ ) +
     $                                                         ALPHA*FR7
                              C( I+3, UJ+1 ) = DELTA*C( I+3, UJ+1 ) +
     $                                                         ALPHA*FR8
  180                      CONTINUE
                        END IF
                        DO 200, I = II+UISEC, UJ
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 190, L = 1, ULSEC, 2
                              FR1 = FR1 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+1 )
                              FR1 = FR1 + T1( L+1, I-II+1 )*
     $                                                T1( L+1, UJ-II+1 )
                              FR2 = FR2 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+2 )
                              FR2 = FR2 + T1( L+1, I-II+1 )*
     $                                                T1( L+1, UJ-II+2 )
  190                      CONTINUE
                           IF( LSEC.GT.ULSEC )THEN
                              FR1 = FR1 + T1( LSEC, I-II+1 )*
     $                                               T1( LSEC, UJ-II+1 )
                              FR2 = FR2 + T1( LSEC, I-II+1 )*
     $                                               T1( LSEC, UJ-II+2 )
                           END IF
                           C( I, UJ ) = DELTA*C( I, UJ ) + ALPHA*FR1
                           C( I, UJ+1 ) = DELTA*C( I, UJ+1 ) + ALPHA*FR2
  200                   CONTINUE
                     END IF
                     FR1 = ZERO
                     DO 210, L = 1, ULSEC, 2
                        FR1 = FR1 + T1( L, UJ-II+2 )* T1( L, UJ-II+2 )
                        FR1 = FR1 + T1( L+1, UJ-II+2 )*
     $                                                T1( L+1, UJ-II+2 )
  210                CONTINUE
                     IF( LSEC.GT.ULSEC )THEN
                        FR1 = FR1 + T1( LSEC, UJ-II+2 )*
     $                                               T1( LSEC, UJ-II+2 )
                     END IF
                     C( UJ+1, UJ+1 ) =  DELTA*C( UJ+1, UJ+1 ) +
     $                                                         ALPHA*FR1
  220             CONTINUE
  230          CONTINUE
  240       CONTINUE
         ELSE
*
*           Form  C := alpha*A'*A + beta*C, C is stored in the
*           upper triangular part.
*
            DO 350, II = 1, N, CB
               ISEC = MIN( CB, N-II+1 )
*
*              C := alpha*A'*A + beta*C, general matrix multiply for
*              upper rectangular vertical blocks of C.
*
               IF( II.GT.1 )THEN
                  CALL DGEMM ( 'T', 'N', II-1, ISEC, K, ALPHA,
     $                                  A( 1, 1 ), LDA, A( 1, II ), LDA,
     $                                           BETA, C( 1, II ), LDC )
               END IF
*
*              C := alpha*A'*A + C, update a upper triangular diagonal
*              block of C with alpha*A'*A.
*
               DO 340, LL = 1, K, RB
                  LSEC = MIN( RB, K-LL+1 )
                  ULSEC = LSEC-MOD( LSEC, 2 )
*
*                 Determine if the block of C should be updated with
*                 beta or not.
*
                  DELTA = ONE
                  IF( LL.EQ.1 ) DELTA = BETA
*
*                 T1 := A, copy a rectangular block of A to T1.
*
                  UISEC = ISEC-MOD( ISEC, 2 )
                  DO 260, I = II, II+UISEC-1, 2
                     DO 250, L = LL, LL+ULSEC-1, 2
                        T1( L-LL+1, I-II+1 ) = A( L, I )
                        T1( L-LL+1, I-II+2 ) = A( L, I+1 )
                        T1( L-LL+2, I-II+1 ) = A( L+1, I )
                        T1( L-LL+2, I-II+2 ) = A( L+1, I+1 )
  250                CONTINUE
                     IF( ULSEC.LT.LSEC )THEN
                        T1( LSEC, I-II+1 ) = A( LL+LSEC-1, I )
                        T1( LSEC, I-II+2 ) = A( LL+LSEC-1, I+1 )
                     END IF
  260             CONTINUE
                  IF( UISEC.LT.ISEC )THEN
                     DO 270, L = LL, LL+LSEC-1
                        T1( L-LL+1, ISEC ) = A( L, II+ISEC-1 )
  270                CONTINUE
                  END IF
*
*                 C := alpha*T1'*T1 + C, a upper triangular diagonal
*                 block of C is updated with beta.
*
                  DO 330, JX = II+ISEC-1, II, -2
                     UJ = JX-2+1
                     UJSEC = JX+1-MAX( UJ, II )
                     UISEC = UJ-II-MOD( UJ-II, 4 )
*
*                    Four by two unrolling.
*
                     IF( UJSEC.EQ.2 )THEN
                        IF( UISEC.GE.4. )THEN
                           DO 290, I = II, II+UISEC-1, 4
                              FR1 = ZERO
                              FR2 = ZERO
                              FR3 = ZERO
                              FR4 = ZERO
                              FR5 = ZERO
                              FR6 = ZERO
                              FR7 = ZERO
                              FR8 = ZERO
                              DO 280, L = 1, LSEC
                                 FR1 = FR1 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+1 )
                                 FR2 = FR2 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+2 )
                                 FR3 = FR3 + T1( L, I-II+2 )*
     $                                                  T1( L, UJ-II+1 )
                                 FR4 = FR4 + T1( L, I-II+2 )*
     $                                                  T1( L, UJ-II+2 )
                                 FR5 = FR5 + T1( L, I-II+3 )*
     $                                                  T1( L, UJ-II+1 )
                                 FR6 = FR6 + T1( L, I-II+3 )*
     $                                                  T1( L, UJ-II+2 )
                                 FR7 = FR7 + T1( L, I-II+4 )*
     $                                                  T1( L, UJ-II+1 )
                                 FR8 = FR8 + T1( L, I-II+4 )*
     $                                                  T1( L, UJ-II+2 )
  280                         CONTINUE
                              C( I, UJ ) = DELTA*C( I, UJ ) + ALPHA*FR1
                              C( I, UJ+1 ) = DELTA*C( I, UJ+1 ) +
     $                                                         ALPHA*FR2
                              C( I+1, UJ ) = DELTA*C( I+1, UJ ) +
     $                                                         ALPHA*FR3
                              C( I+1, UJ+1 ) = DELTA*C( I+1, UJ+1 ) +
     $                                                         ALPHA*FR4
                              C( I+2, UJ ) = DELTA*C( I+2, UJ ) +
     $                                                         ALPHA*FR5
                              C( I+2, UJ+1 ) = DELTA*C( I+2, UJ+1 ) +
     $                                                         ALPHA*FR6
                              C( I+3, UJ ) = DELTA*C( I+3, UJ ) +
     $                                                         ALPHA*FR7
                              C( I+3, UJ+1 ) = DELTA*C( I+3, UJ+1 ) +
     $                                                         ALPHA*FR8
  290                      CONTINUE
                        END IF
                        DO 310, I = II+UISEC, UJ
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 300, L = 1, ULSEC, 2
                              FR1 = FR1 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+1 )
                              FR1 = FR1 + T1( L+1, I-II+1 )*
     $                                                T1( L+1, UJ-II+1 )
                              FR2 = FR2 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+2 )
                              FR2 = FR2 + T1( L+1, I-II+1 )*
     $                                                T1( L+1, UJ-II+2 )
  300                      CONTINUE
                           IF( LSEC.GT.ULSEC )THEN
                              FR1 = FR1 + T1( LSEC, I-II+1 )*
     $                                               T1( LSEC, UJ-II+1 )
                              FR2 = FR2 + T1( LSEC, I-II+1 )*
     $                                               T1( LSEC, UJ-II+2 )
                           END IF
                           C( I, UJ ) = DELTA*C( I, UJ ) + ALPHA*FR1
                           C( I, UJ+1 ) = DELTA*C( I, UJ+1 ) + ALPHA*FR2
  310                   CONTINUE
                     END IF
                     FR1 = ZERO
                     DO 320, L = 1, ULSEC, 2
                        FR1 = FR1 + T1( L, UJ-II+2 )* T1( L, UJ-II+2 )
                        FR1 = FR1 + T1( L+1, UJ-II+2 )*
     $                                                T1( L+1, UJ-II+2 )
  320                CONTINUE
                     IF( LSEC.GT.ULSEC )THEN
                        FR1 = FR1 + T1( LSEC, UJ-II+2 )*
     $                                               T1( LSEC, UJ-II+2 )
                     END IF
                     C( UJ+1, UJ+1 ) = DELTA*C( UJ+1, UJ+1 ) + ALPHA*FR1
  330             CONTINUE
  340          CONTINUE
  350       CONTINUE
         END IF
      ELSE
         IF( NOTR )THEN
*
*           Form  C := alpha*A*A' + beta*C, C is stored in the
*           lower triangular part.
*
            DO 460, IX = N, 1, -CB
               II = MAX( 1, IX-CB+1 )
               ISEC = IX-II+1
*
*              C := alpha*A*A' + C, update a lower triangular diagonal
*              block of C with alpha*A*A'.
*
               DO 450, LL = 1, K, RB
                  LSEC = MIN( RB, K-LL+1 )
                  ULSEC = LSEC-MOD( LSEC, 2 )
*
*                 Determine if the block of C should be updated with
*                 beta or not.
*
                  DELTA = ONE
                  IF( LL.EQ.1 ) DELTA = BETA
*
*                 T1 := A', copy the transpose of a rectangular
*                 block of A to T1.
*
                  UISEC = ISEC-MOD( ISEC, 2 )
                  DO 370, L = LL, LL+ULSEC-1, 2
                     DO 360, I = II, II+UISEC-1, 2
                        T1( L-LL+1, I-II+1 ) = A( I, L )
                        T1( L-LL+2, I-II+1 ) = A( I, L+1 )
                        T1( L-LL+1, I-II+2 ) = A( I+1, L )
                        T1( L-LL+2, I-II+2 ) = A( I+1, L+1 )
  360                CONTINUE
                     IF( UISEC.LT.ISEC )THEN
                        T1( L-LL+1, ISEC ) = A( II+ISEC-1, L )
                        T1( L-LL+2, ISEC ) = A( II+ISEC-1, L+1 )
                     END IF
  370             CONTINUE
                  IF( ULSEC.LT.LSEC )THEN
                     DO 380, I = II, II+ISEC-1
                        T1( LSEC, I-II+1 ) = A( I, LL+LSEC-1 )
  380                CONTINUE
                  END IF
*
*                 C := alpha*T1'*T1 + C, a lower triangular diagonal
*                 block of C is updated with beta.
*
                  DO 440, UJ = II, II+ISEC-1, 2
                     UJSEC = MIN( 2, II+ISEC-UJ )
                     RISEC = MOD( II+ISEC-UJ-1, 4 )+1
*
*                    Four by two unrolling.
*
                     FR1 = ZERO
                     DO 390, L = 1, ULSEC, 2
                        FR1 = FR1 + T1( L, UJ-II+1 )*T1( L, UJ-II+1 )
                        FR1 = FR1 + T1( L+1, UJ-II+1 )*
     $                                                T1( L+1, UJ-II+1 )
  390                CONTINUE
                     IF( LSEC.GT.ULSEC )THEN
                        FR1 = FR1 + T1( LSEC, UJ-II+1 )*
     $                                               T1( LSEC, UJ-II+1 )
                     END IF
                     C( UJ, UJ ) = DELTA*C( UJ, UJ ) + ALPHA*FR1
                     IF( UJSEC.EQ.2 )THEN
                        DO 410, I = UJ+1, UJ+RISEC-1
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 400, L = 1, ULSEC, 2
                              FR1 = FR1 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+1 )
                              FR1 = FR1 + T1( L+1, I-II+1 )*
     $                                                T1( L+1, UJ-II+1 )
                              FR2 = FR2 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+2 )
                              FR2 = FR2 + T1( L+1, I-II+1 )*
     $                                                T1( L+1, UJ-II+2 )
  400                      CONTINUE
                           IF( LSEC.GT.ULSEC )THEN
                              FR1 = FR1 + T1( LSEC, I-II+1 )*
     $                                               T1( LSEC, UJ-II+1 )
                              FR2 = FR2 + T1( LSEC, I-II+1 )*
     $                                               T1( LSEC, UJ-II+2 )
                           END IF
                           C( I, UJ ) = DELTA*C( I, UJ ) + ALPHA*FR1
                           C( I, UJ+1 ) = DELTA*C( I, UJ+1 ) + ALPHA*FR2
  410                   CONTINUE
                        DO 430, I = UJ+RISEC, II+ISEC-1, 4
                           FR1 = ZERO
                           FR2 = ZERO
                           FR3 = ZERO
                           FR4 = ZERO
                           FR5 = ZERO
                           FR6 = ZERO
                           FR7 = ZERO
                           FR8 = ZERO
                           DO 420, L = 1, LSEC
                              FR1 = FR1 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+1 )
                              FR2 = FR2 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+2 )
                              FR3 = FR3 + T1( L, I-II+2 )*
     $                                                  T1( L, UJ-II+1 )
                              FR4 = FR4 + T1( L, I-II+2 )*
     $                                                  T1( L, UJ-II+2 )
                              FR5 = FR5 + T1( L, I-II+3 )*
     $                                                  T1( L, UJ-II+1 )
                              FR6 = FR6 + T1( L, I-II+3 )*
     $                                                  T1( L, UJ-II+2 )
                              FR7 = FR7 + T1( L, I-II+4 )*
     $                                                  T1( L, UJ-II+1 )
                              FR8 = FR8 + T1( L, I-II+4 )*
     $                                                  T1( L, UJ-II+2 )
  420                      CONTINUE
                           C( I, UJ ) = DELTA*C( I, UJ ) + ALPHA*FR1
                           C( I, UJ+1 ) = DELTA*C( I, UJ+1 ) + ALPHA*FR2
                           C( I+1, UJ ) = DELTA*C( I+1, UJ ) + ALPHA*FR3
                           C( I+1, UJ+1 ) = DELTA*C( I+1, UJ+1 ) +
     $                                                         ALPHA*FR4
                           C( I+2, UJ ) = DELTA*C( I+2, UJ ) + ALPHA*FR5
                           C( I+2, UJ+1 ) = DELTA*C( I+2, UJ+1 ) +
     $                                                         ALPHA*FR6
                           C( I+3, UJ ) = DELTA*C( I+3, UJ ) + ALPHA*FR7
                           C( I+3, UJ+1 ) = DELTA*C( I+3, UJ+1 ) +
     $                                                         ALPHA*FR8
  430                   CONTINUE
                     END IF
  440             CONTINUE
  450          CONTINUE
*
*              C := alpha*A*A' + beta*C, general matrix multiply for
*              lower rectangular vertical blocks of C.
*
               IF( II+ISEC.LE.N )THEN
                  CALL DGEMM ( 'N', 'T', N-II-ISEC+1, ISEC, K,
     $                          ALPHA, A( II+ISEC, 1 ), LDA, A( II, 1 ),
     $                                LDA, BETA, C( II+ISEC, II ), LDC )
               END IF
  460       CONTINUE
         ELSE
*
*           Form  C := alpha*A'*A + beta*C, C is stored in the
*           lower triangular part.
*
            DO 570, IX = N, 1, -CB
               II = MAX( 1, IX-CB+1 )
               ISEC = IX-II+1
*
*              C := alpha*A*'A + C, update a lower triangular diagonal
*              block of C with alpha*A'*A.
*
               DO 560, LL = 1, K, RB
                  LSEC = MIN( RB, K-LL+1 )
                  ULSEC = LSEC-MOD( LSEC, 2 )
*
*                 Determine if the block of C should be updated with
*                 beta or not.
*
                  DELTA = ONE
                  IF( LL.EQ.1 ) DELTA = BETA
*
*                 T1 := A, copy a rectangular block of A to T1.
*
                  UISEC = ISEC-MOD( ISEC, 2 )
                  DO 480, I = II, II+UISEC-1, 2
                     DO 470, L = LL, LL+ULSEC-1, 2
                        T1( L-LL+1, I-II+1 ) = A( L, I )
                        T1( L-LL+1, I-II+2 ) = A( L, I+1 )
                        T1( L-LL+2, I-II+1 ) = A( L+1, I )
                        T1( L-LL+2, I-II+2 ) = A( L+1, I+1 )
  470                CONTINUE
                     IF( ULSEC.LT.LSEC )THEN
                        T1( LSEC, I-II+1 ) = A( LL+LSEC-1, I )
                        T1( LSEC, I-II+2 ) = A( LL+LSEC-1, I+1 )
                     END IF
  480             CONTINUE
                  IF( UISEC.LT.ISEC )THEN
                     DO 490, L = LL, LL+LSEC-1
                        T1( L-LL+1, ISEC ) = A( L, II+ISEC-1 )
  490                CONTINUE
                  END IF
*
*                 C := alpha*T1'*T1 + C, a lower triangular diagonal
*                 block of C is updated with beta.
*
                  DO 550, UJ = II, II+ISEC-1, 2
                     UJSEC = MIN( 2, II+ISEC-UJ )
                     RISEC = MOD( II+ISEC-UJ-1, 4 )+1
*
*                    Four by two unrolling.
*
                     FR1 = ZERO
                     DO 500, L = 1, ULSEC, 2
                        FR1 = FR1 + T1( L, UJ-II+1 )*T1( L, UJ-II+1 )
                        FR1 = FR1 + T1( L+1, UJ-II+1 )*
     $                                                T1( L+1, UJ-II+1 )
  500                CONTINUE
                     IF( LSEC.GT.ULSEC )THEN
                        FR1 = FR1 + T1( LSEC, UJ-II+1 )*
     $                                               T1( LSEC, UJ-II+1 )
                     END IF
                     C( UJ, UJ ) = DELTA*C( UJ, UJ ) + ALPHA*FR1
                     IF( UJSEC.EQ.2 )THEN
                        DO 520, I = UJ+1, UJ+RISEC-1
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 510, L = 1, ULSEC, 2
                              FR1 = FR1 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+1 )
                              FR1 = FR1 + T1( L+1, I-II+1 )*
     $                                                T1( L+1, UJ-II+1 )
                              FR2 = FR2 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+2 )
                              FR2 = FR2 + T1( L+1, I-II+1 )*
     $                                                T1( L+1, UJ-II+2 )
  510                      CONTINUE
                           IF( LSEC.GT.ULSEC )THEN
                              FR1 = FR1 + T1( LSEC, I-II+1 )*
     $                                               T1( LSEC, UJ-II+1 )
                              FR2 = FR2 + T1( LSEC, I-II+1 )*
     $                                               T1( LSEC, UJ-II+2 )
                           END IF
                           C( I, UJ ) = DELTA*C( I, UJ ) + ALPHA*FR1
                           C( I, UJ+1 ) = DELTA*C( I, UJ+1 ) + ALPHA*FR2
  520                   CONTINUE
                        DO 540, I = UJ+RISEC, II+ISEC-1, 4
                           FR1 = ZERO
                           FR2 = ZERO
                           FR3 = ZERO
                           FR4 = ZERO
                           FR5 = ZERO
                           FR6 = ZERO
                           FR7 = ZERO
                           FR8 = ZERO
                           DO 530, L = 1, LSEC
                              FR1 = FR1 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+1 )
                              FR2 = FR2 + T1( L, I-II+1 )*
     $                                                  T1( L, UJ-II+2 )
                              FR3 = FR3 + T1( L, I-II+2 )*
     $                                                  T1( L, UJ-II+1 )
                              FR4 = FR4 + T1( L, I-II+2 )*
     $                                                  T1( L, UJ-II+2 )
                              FR5 = FR5 + T1( L, I-II+3 )*
     $                                                  T1( L, UJ-II+1 )
                              FR6 = FR6 + T1( L, I-II+3 )*
     $                                                  T1( L, UJ-II+2 )
                              FR7 = FR7 + T1( L, I-II+4 )*
     $                                                  T1( L, UJ-II+1 )
                              FR8 = FR8 + T1( L, I-II+4 )*
     $                                                  T1( L, UJ-II+2 )
  530                      CONTINUE
                           C( I, UJ ) = DELTA*C( I, UJ ) + ALPHA*FR1
                           C( I, UJ+1 ) = DELTA*C( I, UJ+1 ) + ALPHA*FR2
                           C( I+1, UJ ) = DELTA*C( I+1, UJ ) + ALPHA*FR3
                           C( I+1, UJ+1 ) = DELTA*C( I+1, UJ+1 ) +
     $                                                         ALPHA*FR4
                           C( I+2, UJ ) = DELTA*C( I+2, UJ ) + ALPHA*FR5
                           C( I+2, UJ+1 ) = DELTA*C( I+2, UJ+1 ) +
     $                                                         ALPHA*FR6
                           C( I+3, UJ ) = DELTA*C( I+3, UJ ) + ALPHA*FR7
                           C( I+3, UJ+1 ) = DELTA*C( I+3, UJ+1 ) +
     $                                                         ALPHA*FR8
  540                   CONTINUE
                     END IF
  550             CONTINUE
  560          CONTINUE
*
*              C := alpha*A'*A + beta*C, general matrix multiply for
*              lower rectangular vertical blocks of C.
*
               IF( II+ISEC.LE.N )THEN
                  CALL DGEMM ( 'T', 'N', N-II-ISEC+1, ISEC, K,
     $                          ALPHA, A( 1, II+ISEC ), LDA, A( 1, II ),
     $                                LDA, BETA, C( II+ISEC, II ), LDC )
               END IF
  570       CONTINUE
         END IF
      END IF
*
      RETURN
*
*     End of DSYRK.
*
      END
