      SUBROUTINE DTRSM( SIDE, UPLO, TRANSA, DIAG, M, N, ALPHA, A, LDA,
     $                   C, LDC )
*     .. Scalar Arguments ..
      CHARACTER*1        SIDE, UPLO, TRANSA, DIAG
      INTEGER            M, N, LDA, LDC
      DOUBLE PRECISION   ALPHA
*     .. Array Arguments ..
      DOUBLE PRECISION   A( LDA, * ), C( LDC, * )
*     ..
*
*  Purpose
*  =======
*
*  DTRSM  solves one of the matrix equations
*
*     op( A )*X = alpha*C,   or   X*op( A ) = alpha*C,
*
*  where alpha is a scalar, X and C are m by n matrices, A is a unit, or
*  non-unit,  upper or lower triangular matrix  and  op( A )  is one  of
*
*     op( A ) = A   or   op( A ) = A'.
*
*  The matrix X is overwritten on C.
*
*  Parameters
*  ==========
*
*  SIDE   - CHARACTER*1.
*           On entry, SIDE specifies whether op( A ) appears on the left
*           or right of X as follows:
*
*              SIDE = 'L' or 'l'   op( A )*X = alpha*C.
*
*              SIDE = 'R' or 'r'   X*op( A ) = alpha*C.
*
*           Unchanged on exit.
*
*  UPLO   - CHARACTER*1.
*           On entry, UPLO specifies whether the matrix A is an upper or
*           lower triangular matrix as follows:
*
*              UPLO = 'U' or 'u'   A is an upper triangular matrix.
*
*              UPLO = 'L' or 'l'   A is a lower triangular matrix.
*
*           Unchanged on exit.
*
*  TRANSA - CHARACTER*1.
*           On entry, TRANSA specifies the form of op( A ) to be used in
*           the matrix multiplication as follows:
*
*              TRANSA = 'N' or 'n'   op( A ) = A.
*
*              TRANSA = 'T' or 't'   op( A ) = A'.
*
*              TRANSA = 'C' or 'c'   op( A ) = A'.
*
*           Unchanged on exit.
*
*  DIAG   - CHARACTER*1.
*           On entry, DIAG specifies whether or not A is unit triangular
*           as follows:
*
*              DIAG = 'U' or 'u'   A is assumed to be unit triangular.
*
*              DIAG = 'N' or 'n'   A is not assumed to be unit
*                                  triangular.
*
*           Unchanged on exit.
*
*  M      - INTEGER.
*           On entry, M specifies the number of rows of C. M must be at
*           least zero.
*           Unchanged on exit.
*
*  N      - INTEGER.
*           On entry, N specifies the number of columns of C.  N must be
*           at least zero.
*           Unchanged on exit.
*
*  ALPHA  - DOUBLE PRECISION.
*           On entry,  ALPHA specifies the scalar  alpha. When  alpha is
*           zero then  A is not referenced and  C need not be set before
*           entry.
*           Unchanged on exit.
*
*  A      - DOUBLE PRECISION array of DIMENSION ( LDA, k ), where k is m
*           when  SIDE = 'L' or 'l'  and is  n  when  SIDE = 'R' or 'r'.
*           Before entry  with  UPLO = 'U' or 'u',  the  leading  k by k
*           upper triangular part of the array  A must contain the upper
*           triangular matrix  and the strictly lower triangular part of
*           A is not referenced.
*           Before entry  with  UPLO = 'L' or 'l',  the  leading  k by k
*           lower triangular part of the array  A must contain the lower
*           triangular matrix  and the strictly upper triangular part of
*           A is not referenced.
*           Note that when  DIAG = 'U' or 'u',  the diagonal elements of
*           A  are not referenced either,  but are assumed to be  unity.
*           Unchanged on exit.
*
*  LDA    - INTEGER.
*           On entry, LDA specifies the first dimension of A as declared
*           in the calling (sub) program.  When  SIDE = 'L' or 'l'  then
*           LDA  must be at least  max( 1, m ),  when  SIDE = 'R' or 'r'
*           then LDA must be at least max( 1, n ).
*           Unchanged on exit.
*
*  C      - DOUBLE PRECISION array of DIMENSION ( LDC, n ).
*           Before entry,  the leading  m by n part of the array  C must
*           contain  the  right-hand  side  matrix  C,  and  on exit  is
*           overwritten by the solution matrix  X.
*
*  LDC    - INTEGER.
*           On entry, LDC specifies the first dimension of C as declared
*           in  the  calling  (sub)  program.   LDC  must  be  at  least
*           max( 1, m ).
*           Unchanged on exit.
*
*
*  Level 3 Blas routine.
*
*  -- Written on 8-February-1989.
*     Jack Dongarra, Argonne National Laboratory.
*     Iain Duff, AERE Harwell.
*     Jeremy Du Croz, Numerical Algorithms Group Ltd.
*     Sven Hammarling, Numerical Algorithms Group Ltd.
*
*  -- Rewritten in December-1993.
*     GEMM-Based Level 3 BLAS.
*     Per Ling, Institute of Information Processing,
*     University of Umea, Sweden.
*
*  -- Rewritten in Mars-1995.
*     Superscalar GEMM-Based Level 3 BLAS (Version 0.1).
*     Per Ling, Department of Computing Science,
*     University of Umea, Sweden.
*
*
*     .. Local Scalars ..
      INTEGER            INFO, NROWA
      LOGICAL            LSIDE, UPPER, NOTR, NOUNIT
      DOUBLE PRECISION   GAMMA, IALPHA
      INTEGER            I, II, IX, ISEC, J, JJ, JX, JY, JSEC
      INTEGER            L, UJ, UJSEC, UISEC, RI, RISEC, RJSEC
      DOUBLE PRECISION   FR1, FR2, FR3, FR4, FR5, FR6, FR7, FR8

*     .. Intrinsic Functions ..
      INTRINSIC          MAX, MIN, MOD
*     .. External Functions ..
      LOGICAL            LSAME
      EXTERNAL           LSAME
*     .. External Subroutines ..
      EXTERNAL           XERBLA
      EXTERNAL           DGEMM
*     .. Parameters ..
      DOUBLE PRECISION   ZERO, ONE
      PARAMETER        ( ZERO = 0.0D+0, ONE = 1.0D+0 )
*     .. User specified parameters for DTRSM ..
      INTEGER            RCB, RB, CB
      PARAMETER        ( RCB = 32, RB = 32, CB = 32 )
      DOUBLE PRECISION   T1( RB, CB ), T2( RB, RB ), T3( RCB, RCB )
      DOUBLE PRECISION   T4( RB )
*     ..
*     .. Executable Statements ..
*
*     Test the input parameters.
*
      LSIDE = LSAME( SIDE, 'L' )
      UPPER = LSAME( UPLO, 'U' )
      NOTR = LSAME( TRANSA, 'N' )
      NOUNIT = LSAME( DIAG, 'N' )
      IF( LSIDE )THEN
         NROWA = M
      ELSE
         NROWA = N
      END IF
      INFO = 0
      IF( ( .NOT.LSIDE ).AND.( .NOT.LSAME( SIDE, 'R' ) ) )THEN
         INFO = 1
      ELSE IF( ( .NOT.UPPER ).AND.( .NOT.LSAME( UPLO, 'L' ) ) )THEN
         INFO = 2
      ELSE IF( ( .NOT.NOTR ).AND.( .NOT.LSAME( TRANSA, 'T' ) ).AND.
     $                               ( .NOT.LSAME( TRANSA, 'C' ) ) )THEN
         INFO = 3
      ELSE IF( ( .NOT.NOUNIT ).AND.( .NOT.LSAME( DIAG, 'U' ) ) )THEN
         INFO = 4
      ELSE IF( M.LT.0 )THEN
         INFO = 5
      ELSE IF( N.LT.0 )THEN
         INFO = 6
      ELSE IF( LDA.LT.MAX( 1, NROWA ) )THEN
         INFO = 9
      ELSE IF( LDC.LT.MAX( 1, M ) )THEN
         INFO = 11
      END IF
      IF( INFO.NE.0 )THEN
         CALL XERBLA( 'DTRSM ', INFO )
         RETURN
      END IF
*
*     Quick return if possible.
*
      IF( ( M.EQ.0 ).OR.( N.EQ.0 ) )
     $   RETURN
*
*     And when alpha.eq.zero.
*
      IF( ALPHA.EQ.ZERO )THEN
*
*        C := zero, set the elements of the rectangular M*N matrix C
*        to zero. Unroll to the depth of 4 to facilitate multiple word
*        load and store instructions.
*
         UISEC = M-MOD( M, 4 )
         DO 30, J = 1, N
            DO 10, I = 1, UISEC, 4
               C( I, J ) = ZERO
               C( I+1, J ) = ZERO
               C( I+2, J ) = ZERO
               C( I+3, J ) = ZERO
   10       CONTINUE
            DO 20, I = UISEC+1, M
               C( I, J ) = ZERO
   20       CONTINUE
   30    CONTINUE
         RETURN
      END IF
*
*     Start the operations.
*
      IF( LSIDE )THEN
         IF( UPPER )THEN
            IF( NOTR )THEN
*
*              Form  C := alpha*A*C. Left, Upper, No transpose.
*
               IALPHA = ONE/ALPHA
               DO 160, II = M-MOD( M-1, RCB ), 1, -RCB
                  ISEC = MIN( RCB, M-II+1 )
*
*                 C := -1.0/alpha*A*C + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  IF( II+ISEC.LE.M )THEN
                     CALL DGEMM ( 'N', 'N', ISEC, N, M-II-ISEC+1,
     $                  -IALPHA, A( II, II+ISEC ), LDA, C( II+ISEC, 1 ),
     $                                       LDC, ONE, C( II, 1 ), LDC )
                  END IF
*
*                 If diag = 'N', T3 := -1.0/alpha*A', except for the
*                 diagonal elements of T3, T3(i,i) := alpha/A(i,i).
*                 If diag = 'U', T3 := -1.0*A', except for the
*                 diagonal elements of T3, T3(i,i) := alpha. Unroll to
*                 the depth of 2 by 2 to facilitate multiple word load
*                 and store instructions.
*
                  DO 50, L = II+ISEC-2, II-1, -2
                     UISEC = L-II-MOD( L-II, 2 )
                     DO 40, I = II, II+UISEC-1, 2
                        T3( L-II+1, I-II+1 ) = -A( I, L )
                        T3( L-II+2, I-II+1 ) = -A( I, L+1 )
                        T3( L-II+1, I-II+2 ) = -A( I+1, L )
                        T3( L-II+2, I-II+2 ) = -A( I+1, L+1 )
   40                CONTINUE
                     IF( MOD( L-II, 2 ).EQ.1 )THEN
                        T3( L-II+1, L-II ) = -A( L-1, L )
                        T3( L-II+2, L-II ) = -A( L-1, L+1 )
                     END IF
                     IF( L-II.GE.0 )THEN
                        T3( L-II+1, L-II+1 ) = ALPHA
                        T3( L-II+2, L-II+1 ) = -A( L, L+1 )
                     END IF
                     T3( L-II+2, L-II+2 ) = ALPHA
   50             CONTINUE
                  IF( NOUNIT )THEN
                     DO 80, L = 1, ISEC
                        GAMMA = ONE/A( L+II-1, L+II-1 )
                        T3( L, L ) = GAMMA*ALPHA
                        RISEC = MOD( ISEC-L, 4 )+1
                        DO 60, I = L+1, L+RISEC-1
                           T3( I, L ) = GAMMA*T3( I, L )
   60                   CONTINUE
                        DO 70, I = L+RISEC, ISEC, 4
                           T3( I,   L ) = GAMMA*T3( I, L )
                           T3( I+1, L ) = GAMMA*T3( I+1, L )
                           T3( I+2, L ) = GAMMA*T3( I+2, L )
                           T3( I+3, L ) = GAMMA*T3( I+3, L )
   70                   CONTINUE
   80                CONTINUE
                  END IF
*
*                 C := alpha*T3'*C, update a rectangular block of C
*                 using the transpose of the upper triangular diagonal
*                 block of A stored in T3.
*
                  UISEC = ISEC-MOD( ISEC, 4 )
                  DO 150, JX = N, 1, -2
                     UJ = MAX( 1, JX-2+1 )
                     UJSEC = JX-UJ+1
*
*                    Four by two unrolling.
*
                     IF( UJSEC.EQ.2 )THEN
                        DO 100, RI = II+ISEC-1, II+UISEC, -1
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 90, L = RI, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, RI-II+1 )*
     $                                                      C( L, UJ+1 )
   90                      CONTINUE
                           C( RI, UJ ) = FR1
                           C( RI, UJ+1 ) = FR2
  100                   CONTINUE
                        DO 120, I = II+UISEC-4, II, -4
                           FR1 = ZERO
                           FR2 = ZERO
                           FR3 = ZERO
                           FR4 = ZERO
                           FR5 = ZERO
                           FR6 = ZERO
                           FR7 = ZERO
                           FR8 = ZERO
                           DO 110, L = I+4, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, I-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, I-II+1 )*
     $                                                      C( L, UJ+1 )
                              FR3 = FR3 + T3( L-II+1, I-II+2 )*
     $                                                        C( L, UJ )
                              FR4 = FR4 + T3( L-II+1, I-II+2 )*
     $                                                      C( L, UJ+1 )
                              FR5 = FR5 + T3( L-II+1, I-II+3 )*
     $                                                        C( L, UJ )
                              FR6 = FR6 + T3( L-II+1, I-II+3 )*
     $                                                      C( L, UJ+1 )
                              FR7 = FR7 + T3( L-II+1, I-II+4 )*
     $                                                        C( L, UJ )
                              FR8 = FR8 + T3( L-II+1, I-II+4 )*
     $                                                      C( L, UJ+1 )
  110                      CONTINUE
                           FR7 = FR7 + T3( I-II+4, I-II+4 )*C( I+3, UJ )
                           FR8 = FR8 + T3( I-II+4, I-II+4 )*
     $                                                    C( I+3, UJ+1 )
                           C( I+3, UJ ) = FR7
                           C( I+3, UJ+1 ) = FR8
                           FR5 = FR5 + T3( I-II+3, I-II+3 )*C( I+2, UJ )
                           FR6 = FR6 + T3( I-II+3, I-II+3 )*
     $                                                    C( I+2, UJ+1 )
                           FR5 = FR5 + T3( I-II+4, I-II+3 )*C( I+3, UJ )
                           FR6 = FR6 + T3( I-II+4, I-II+3 )*
     $                                                    C( I+3, UJ+1 )
                           C( I+2, UJ ) = FR5
                           C( I+2, UJ+1 ) = FR6
                           FR3 = FR3 + T3( I-II+2, I-II+2 )*C( I+1, UJ )
                           FR4 = FR4 + T3( I-II+2, I-II+2 )*
     $                                                    C( I+1, UJ+1 )
                           FR3 = FR3 + T3( I-II+4, I-II+2 )*C( I+3, UJ )
                           FR4 = FR4 + T3( I-II+4, I-II+2 )*
     $                                                    C( I+3, UJ+1 )
                           FR3 = FR3 + T3( I-II+3, I-II+2 )*C( I+2, UJ )
                           FR4 = FR4 + T3( I-II+3, I-II+2 )*
     $                                                    C( I+2, UJ+1 )
                           C( I+1, UJ ) = FR3
                           C( I+1, UJ+1 ) = FR4
                           FR1 = FR1 + T3( I-II+1, I-II+1 )*C( I, UJ )
                           FR2 = FR2 + T3( I-II+1, I-II+1 )*C( I, UJ+1 )
                           FR1 = FR1 + T3( I-II+3, I-II+1 )*C( I+2, UJ )
                           FR2 = FR2 + T3( I-II+3, I-II+1 )*
     $                                                    C( I+2, UJ+1 )
                           FR1 = FR1 + T3( I-II+4, I-II+1 )*C( I+3, UJ )
                           FR2 = FR2 + T3( I-II+4, I-II+1 )*
     $                                                    C( I+3, UJ+1 )
                           FR1 = FR1 + T3( I-II+2, I-II+1 )*C( I+1, UJ )
                           FR2 = FR2 + T3( I-II+2, I-II+1 )*
     $                                                    C( I+1, UJ+1 )
                           C( I, UJ ) = FR1
                           C( I, UJ+1 ) = FR2
  120                   CONTINUE
                     ELSE IF( UJSEC.EQ.1 )THEN
                        DO 140, RI = II+ISEC-1, II, -1
                           FR1 = ZERO
                           DO 130, L = RI, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
  130                      CONTINUE
                           C( RI, UJ ) = FR1
  140                   CONTINUE
                     END IF
  150             CONTINUE
  160          CONTINUE
            ELSE
*
*              Form  C := alpha*A'*C. Left, Upper, Transpose.
*
               IALPHA = ONE/ALPHA
               DO 300, II = 1, M, RCB
                  ISEC = MIN( RCB, M-II+1 )
*
*                 C := alpha*A'*C + C, general matrix multiply
*                 involving the transpose of a rectangular block
*                 of A.
*
                  IF( II.GT.1 )THEN
                     CALL DGEMM ( 'T', 'N', ISEC, N, II-1, -IALPHA,
     $                                  A( 1, II ), LDA, C( 1, 1 ), LDC,
     $                                            ONE, C( II, 1 ), LDC )
                  END IF
*
*                 T3 := alpha*A, copy a upper triangular diagonal block
*                 of alpha*A to T3. Unroll to the depth of 4 to
*                 facilitate multiple word load and store instructions.
*
                  IF( NOUNIT )THEN
                     DO 190, L = II, II+ISEC-1
                        GAMMA = ONE/A( L, L )
                        UISEC = L-II-MOD( L-II, 4 )
                        DO 170, I = II, II+UISEC-1, 4
                           T3( I-II+1, L-II+1 ) = -GAMMA*A( I, L )
                           T3( I-II+2, L-II+1 ) = -GAMMA*A( I+1, L )
                           T3( I-II+3, L-II+1 ) = -GAMMA*A( I+2, L )
                           T3( I-II+4, L-II+1 ) = -GAMMA*A( I+3, L )
  170                   CONTINUE
                        DO 180, I = II+UISEC, L-1
                           T3( I-II+1, L-II+1 ) = -GAMMA*A( I, L )
  180                   CONTINUE
                        T3( L-II+1, L-II+1 ) = GAMMA*ALPHA
  190                CONTINUE
                  ELSE
                     DO 220, L = II, II+ISEC-1
                        UISEC = L-II-MOD( L-II, 4 )
                        DO 200, I = II, II+UISEC-1, 4
                           T3( I-II+1, L-II+1 ) = -A( I, L )
                           T3( I-II+2, L-II+1 ) = -A( I+1, L )
                           T3( I-II+3, L-II+1 ) = -A( I+2, L )
                           T3( I-II+4, L-II+1 ) = -A( I+3, L )
  200                   CONTINUE
                        DO 210, I = II+UISEC, L-1
                           T3( I-II+1, L-II+1 ) = -A( I, L )
  210                   CONTINUE
                        T3( L-II+1, L-II+1 ) = ALPHA
  220                CONTINUE
                  END IF
*
*                 C := alpha*T3'*C, update a rectangular block of C
*                 using the upper triangular diagonal block of A
*                 stored in T3'.
*
                  RISEC = MOD( ISEC, 4 )
                  DO 290, JX = N, 1, -2
                     UJ = MAX( 1, JX-2+1 )
                     UJSEC = JX-UJ+1
*
*                    Four by two unrolling.
*
                     IF( UJSEC.EQ.2 )THEN
                        DO 240, RI = II, II+RISEC-1
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 230, L = II, RI
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, RI-II+1 )*
     $                                                      C( L, UJ+1 )
  230                      CONTINUE
                           C( RI, UJ ) = FR1
                           C( RI, UJ+1 ) = FR2
  240                   CONTINUE
                        DO 260, I = II+RISEC, II+ISEC-1, 4
                           FR1 = ZERO
                           FR2 = ZERO
                           FR3 = ZERO
                           FR4 = ZERO
                           FR5 = ZERO
                           FR6 = ZERO
                           FR7 = ZERO
                           FR8 = ZERO
                           DO 250, L = II, I-1
                              FR1 = FR1 + T3( L-II+1, I-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, I-II+1 )*
     $                                                      C( L, UJ+1 )
                              FR3 = FR3 + T3( L-II+1, I-II+2 )*
     $                                                        C( L, UJ )
                              FR4 = FR4 + T3( L-II+1, I-II+2 )*
     $                                                      C( L, UJ+1 )
                              FR5 = FR5 + T3( L-II+1, I-II+3 )*
     $                                                        C( L, UJ )
                              FR6 = FR6 + T3( L-II+1, I-II+3 )*
     $                                                      C( L, UJ+1 )
                              FR7 = FR7 + T3( L-II+1, I-II+4 )*
     $                                                        C( L, UJ )
                              FR8 = FR8 + T3( L-II+1, I-II+4 )*
     $                                                      C( L, UJ+1 )
  250                      CONTINUE
                           FR1 = FR1 + T3( I-II+1, I-II+1 )*C( I, UJ )
                           FR2 = FR2 + T3( I-II+1, I-II+1 )*C( I, UJ+1 )
                           C( I, UJ ) = FR1
                           C( I, UJ+1 ) = FR2
                           FR3 = FR3 + T3( I-II+2, I-II+2 )*C( I+1, UJ )
                           FR4 = FR4 + T3( I-II+2, I-II+2 )*
     $                                                    C( I+1, UJ+1 )
                           FR3 = FR3 + T3( I-II+1, I-II+2 )*C( I, UJ )
                           FR4 = FR4 + T3( I-II+1, I-II+2 )*C( I, UJ+1 )
                           C( I+1, UJ ) = FR3
                           C( I+1, UJ+1 ) = FR4
                           FR5 = FR5 + T3( I-II+3, I-II+3 )*C( I+2, UJ )
                           FR6 = FR6 + T3( I-II+3, I-II+3 )*
     $                                                    C( I+2, UJ+1 )
                           FR5 = FR5 + T3( I-II+2, I-II+3 )*C( I+1, UJ )
                           FR6 = FR6 + T3( I-II+2, I-II+3 )*
     $                                                    C( I+1, UJ+1 )
                           FR5 = FR5 + T3( I-II+1, I-II+3 )*C( I, UJ )
                           FR6 = FR6 + T3( I-II+1, I-II+3 )*C( I, UJ+1 )
                           C( I+2, UJ ) = FR5
                           C( I+2, UJ+1 ) = FR6
                           FR7 = FR7 + T3( I-II+4, I-II+4 )*C( I+3, UJ )
                           FR8 = FR8 + T3( I-II+4, I-II+4 )*
     $                                                    C( I+3, UJ+1 )
                           FR7 = FR7 + T3( I-II+3, I-II+4 )*C( I+2, UJ )
                           FR8 = FR8 + T3( I-II+3, I-II+4 )*
     $                                                    C( I+2, UJ+1 )
                           FR7 = FR7 + T3( I-II+2, I-II+4 )*C( I+1, UJ )
                           FR8 = FR8 + T3( I-II+2, I-II+4 )*
     $                                                    C( I+1, UJ+1 )
                           FR7 = FR7 + T3( I-II+1, I-II+4 )*C( I, UJ )
                           FR8 = FR8 + T3( I-II+1, I-II+4 )*C( I, UJ+1 )
                           C( I+3, UJ ) = FR7
                           C( I+3, UJ+1 ) = FR8
  260                   CONTINUE
                     ELSE IF( UJSEC.EQ.1 )THEN
                        DO 280, RI = II, II+ISEC-1
                           FR1 = ZERO
                           DO 270, L = II, RI
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
  270                      CONTINUE
                           C( RI, UJ ) = FR1
  280                   CONTINUE
                     END IF
  290             CONTINUE
  300          CONTINUE
            END IF
         ELSE
            IF( NOTR )THEN
*
*              Form  C := alpha*A'*C. Left, Lower, No transpose.
*
               IALPHA = ONE/ALPHA
               DO 430, IX = MOD( M-1, RCB )+1, M, RCB
                  II = MAX( 1, IX-RCB+1 )
                  ISEC = IX-II+1
*
*                 C := alpha*A'*C + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  IF( II.GT.1 )THEN
                     CALL DGEMM ( 'N', 'N', ISEC, N, II-1, -IALPHA,
     $                                  A( II, 1 ), LDA, C( 1, 1 ), LDC,
     $                                            ONE, C( II, 1 ), LDC )
                  END IF
*
*                 T3 := alpha*A', copy the transpose of a lower
*                 triangular diagonal block of alpha*A to T3. Unroll to
*                 the depth of 2 by 2 to facilitate multiple word load
*                 and store instructions.
*
                  DO 320, L = II, II+ISEC-1, 2
                     T3( L-II+1, L-II+1 ) = ALPHA
                     RISEC = MOD( II+ISEC-L-2, 2 )+2
                     IF( RISEC.GE.2 )THEN
                        T3( L-II+1, L-II+2 ) = -A( L+1, L )
                        T3( L-II+2, L-II+2 ) = ALPHA
                     END IF
                     IF( RISEC.EQ.3 )THEN
                        T3( L-II+1, L-II+3 ) = -A( L+2, L )
                        T3( L-II+2, L-II+3 ) = -A( L+2, L+1 )
                     END IF
                     DO 310, I = L+RISEC, II+ISEC-1, 2
                        T3( L-II+1, I-II+1 ) = -A( I, L )
                        T3( L-II+2, I-II+1 ) = -A( I, L+1 )
                        T3( L-II+1, I-II+2 ) = -A( I+1, L )
                        T3( L-II+2, I-II+2 ) = -A( I+1, L+1 )
  310                CONTINUE
  320             CONTINUE
                  IF( NOUNIT )THEN
                     DO 350, L = 1, ISEC
                        GAMMA = ONE/A( L+II-1, L+II-1 )
                        UISEC = L-1-MOD( L-1, 4 )
                        DO 330, I = 1, UISEC, 4
                           T3( I,   L ) = GAMMA*T3( I, L )
                           T3( I+1, L ) = GAMMA*T3( I+1, L )
                           T3( I+2, L ) = GAMMA*T3( I+2, L )
                           T3( I+3, L ) = GAMMA*T3( I+3, L )
  330                   CONTINUE
                        DO 340, I = 1+UISEC, ISEC-1
                           T3( I, L ) = GAMMA*T3( I, L )
  340                   CONTINUE
                        T3( L, L ) = GAMMA*ALPHA
  350                CONTINUE
                  END IF
*
*                 C := alpha*T3*C, update a rectangular block of C
*                 using the transpose of the lower triangular diagonal
*                 block of A stored in T3.
*
                  RISEC = MOD( ISEC, 4 )
                  DO 420, JX = N, 1, -2
                     UJ = MAX( 1, JX-2+1 )
                     UJSEC = JX-UJ+1
*
*                    Four by two unrolling.
*
                     IF( UJSEC.EQ.2 )THEN
                        DO 370, RI = II, II+RISEC-1
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 360, L = II, RI
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, RI-II+1 )*
     $                                                      C( L, UJ+1 )
  360                      CONTINUE
                           C( RI, UJ ) = FR1
                           C( RI, UJ+1 ) = FR2
  370                   CONTINUE
                        DO 390, I = II+RISEC, II+ISEC-1, 4
                           FR1 = ZERO
                           FR2 = ZERO
                           FR3 = ZERO
                           FR4 = ZERO
                           FR5 = ZERO
                           FR6 = ZERO
                           FR7 = ZERO
                           FR8 = ZERO
                           DO 380, L = II, I-1
                              FR1 = FR1 + T3( L-II+1, I-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, I-II+1 )*
     $                                                      C( L, UJ+1 )
                              FR3 = FR3 + T3( L-II+1, I-II+2 )*
     $                                                        C( L, UJ )
                              FR4 = FR4 + T3( L-II+1, I-II+2 )*
     $                                                      C( L, UJ+1 )
                              FR5 = FR5 + T3( L-II+1, I-II+3 )*
     $                                                        C( L, UJ )
                              FR6 = FR6 + T3( L-II+1, I-II+3 )*
     $                                                      C( L, UJ+1 )
                              FR7 = FR7 + T3( L-II+1, I-II+4 )*
     $                                                        C( L, UJ )
                              FR8 = FR8 + T3( L-II+1, I-II+4 )*
     $                                                      C( L, UJ+1 )
  380                      CONTINUE
                           FR1 = FR1 + T3( I-II+1, I-II+1 )*C( I, UJ )
                           FR2 = FR2 + T3( I-II+1, I-II+1 )*C( I, UJ+1 )
                           C( I, UJ ) = FR1
                           C( I, UJ+1 ) = FR2
                           FR3 = FR3 + T3( I-II+2, I-II+2 )*C( I+1, UJ )
                           FR4 = FR4 + T3( I-II+2, I-II+2 )*
     $                                                    C( I+1, UJ+1 )
                           FR3 = FR3 + T3( I-II+1, I-II+2 )*C( I, UJ )
                           FR4 = FR4 + T3( I-II+1, I-II+2 )*C( I, UJ+1 )
                           C( I+1, UJ ) = FR3
                           C( I+1, UJ+1 ) = FR4
                           FR5 = FR5 + T3( I-II+3, I-II+3 )*C( I+2, UJ )
                           FR6 = FR6 + T3( I-II+3, I-II+3 )*
     $                                                    C( I+2, UJ+1 )
                           FR5 = FR5 + T3( I-II+2, I-II+3 )*C( I+1, UJ )
                           FR6 = FR6 + T3( I-II+2, I-II+3 )*
     $                                                    C( I+1, UJ+1 )
                           FR5 = FR5 + T3( I-II+1, I-II+3 )*C( I, UJ )
                           FR6 = FR6 + T3( I-II+1, I-II+3 )*C( I, UJ+1 )
                           C( I+2, UJ ) = FR5
                           C( I+2, UJ+1 ) = FR6
                           FR7 = FR7 + T3( I-II+4, I-II+4 )*C( I+3, UJ )
                           FR8 = FR8 + T3( I-II+4, I-II+4 )*
     $                                                    C( I+3, UJ+1 )
                           FR7 = FR7 + T3( I-II+3, I-II+4 )*C( I+2, UJ )
                           FR8 = FR8 + T3( I-II+3, I-II+4 )*
     $                                                    C( I+2, UJ+1 )
                           FR7 = FR7 + T3( I-II+2, I-II+4 )*C( I+1, UJ )
                           FR8 = FR8 + T3( I-II+2, I-II+4 )*
     $                                                    C( I+1, UJ+1 )
                           FR7 = FR7 + T3( I-II+1, I-II+4 )*C( I, UJ )
                           FR8 = FR8 + T3( I-II+1, I-II+4 )*C( I, UJ+1 )
                           C( I+3, UJ ) = FR7
                           C( I+3, UJ+1 ) = FR8
  390                   CONTINUE
                     ELSE IF( UJSEC.EQ.1 )THEN
                        DO 410, RI = II, II+ISEC-1
                           FR1 = ZERO
                           DO 400, L = II, RI
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
  400                      CONTINUE
                           C( RI, UJ ) = FR1
  410                   CONTINUE
                     END IF
  420             CONTINUE
  430          CONTINUE
            ELSE
*
*              Form  C := alpha*A'*C. Left, Lower, Transpose.
*
               IALPHA = ONE/ALPHA
               DO 570, IX = M, 1, -RCB
                  II = MAX( 1, IX-RCB+1 )
                  ISEC = IX-II+1
*
*                 C := -1.0/alpha*A'*C + C, general matrix multiply
*                 involving the transpose of a rectangular block
*                 of A.
*
                  IF( II+ISEC.LE.M )THEN
                     CALL DGEMM ( 'T', 'N', ISEC, N, M-II-ISEC+1,
     $                  -IALPHA, A( II+ISEC, II ), LDA, C( II+ISEC, 1 ),
     $                                       LDC, ONE, C( II, 1 ), LDC )
                  END IF
*
*                 If diag = 'N', T3 := -1.0/alpha*A, except for the
*                 diagonal elements of T3, T3(i,i) := alpha/A(i,i).
*                 If diag = 'U', T3 := -1.0*A, except for the
*                 diagonal elements of T3, T3(i,i) := alpha. Unroll to
*                 the depth of 4 to facilitate multiple word load
*                 and store instructions.
*
                  IF( NOUNIT )THEN
                     DO 460, L = II, II+ISEC-1
                        GAMMA = ONE/A( L, L )
                        T3( L-II+1, L-II+1 ) = GAMMA*ALPHA
                        RISEC = MOD( II+ISEC-L-1, 4 )+1
                        DO 440, I = L+1, L+RISEC-1
                           T3( I-II+1, L-II+1 ) = -GAMMA*A( I, L )
  440                   CONTINUE
                        DO 450, I = L+RISEC, II+ISEC-1, 4
                           T3( I-II+1, L-II+1 ) = -GAMMA*A( I, L )
                           T3( I-II+2, L-II+1 ) = -GAMMA*A( I+1, L )
                           T3( I-II+3, L-II+1 ) = -GAMMA*A( I+2, L )
                           T3( I-II+4, L-II+1 ) = -GAMMA*A( I+3, L )
  450                   CONTINUE
  460                CONTINUE
                  ELSE
                     DO 490, L = II, II+ISEC-1
                        T3( L-II+1, L-II+1 ) = ALPHA
                        RISEC = MOD( II+ISEC-L-1, 4 )+1
                        DO 470, I = L+1, L+RISEC-1
                           T3( I-II+1, L-II+1 ) = -A( I, L )
  470                   CONTINUE
                        DO 480, I = L+RISEC, II+ISEC-1, 4
                           T3( I-II+1, L-II+1 ) = -A( I, L )
                           T3( I-II+2, L-II+1 ) = -A( I+1, L )
                           T3( I-II+3, L-II+1 ) = -A( I+2, L )
                           T3( I-II+4, L-II+1 ) = -A( I+3, L )
  480                   CONTINUE
  490                CONTINUE
                  END IF
*
*                 C := alpha*T3'*C, update a rectangular block of C
*                 using the lower triangular diagonal block of A
*                 stored in T3'.
*
                  UISEC = ISEC-MOD( ISEC, 4 )
                  DO 560, JX = N, 1, -2
                     UJ = MAX( 1, JX-2+1 )
                     UJSEC = JX-UJ+1
*
*                    Four by two unrolling.
*
                     IF( UJSEC.EQ.2 )THEN
                        DO 510, RI = II+ISEC-1, II+UISEC, -1
                           FR1 = ZERO
                           FR2 = ZERO
                           DO 500, L = RI, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, RI-II+1 )*
     $                                                      C( L, UJ+1 )
  500                      CONTINUE
                           C( RI, UJ ) = FR1
                           C( RI, UJ+1 ) = FR2
  510                   CONTINUE
                        DO 530, I = II+UISEC-4, II, -4
                           FR1 = ZERO
                           FR2 = ZERO
                           FR3 = ZERO
                           FR4 = ZERO
                           FR5 = ZERO
                           FR6 = ZERO
                           FR7 = ZERO
                           FR8 = ZERO
                           DO 520, L = I+4, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, I-II+1 )*
     $                                                        C( L, UJ )
                              FR2 = FR2 + T3( L-II+1, I-II+1 )*
     $                                                      C( L, UJ+1 )
                              FR3 = FR3 + T3( L-II+1, I-II+2 )*
     $                                                        C( L, UJ )
                              FR4 = FR4 + T3( L-II+1, I-II+2 )*
     $                                                      C( L, UJ+1 )
                              FR5 = FR5 + T3( L-II+1, I-II+3 )*
     $                                                        C( L, UJ )
                              FR6 = FR6 + T3( L-II+1, I-II+3 )*
     $                                                      C( L, UJ+1 )
                              FR7 = FR7 + T3( L-II+1, I-II+4 )*
     $                                                        C( L, UJ )
                              FR8 = FR8 + T3( L-II+1, I-II+4 )*
     $                                                      C( L, UJ+1 )
  520                      CONTINUE
                           FR7 = FR7 + T3( I-II+4, I-II+4 )*C( I+3, UJ )
                           FR8 = FR8 + T3( I-II+4, I-II+4 )*
     $                                                    C( I+3, UJ+1 )
                           C( I+3, UJ ) = FR7
                           C( I+3, UJ+1 ) = FR8
                           FR5 = FR5 + T3( I-II+3, I-II+3 )*C( I+2, UJ )
                           FR6 = FR6 + T3( I-II+3, I-II+3 )*
     $                                                    C( I+2, UJ+1 )
                           FR5 = FR5 + T3( I-II+4, I-II+3 )*C( I+3, UJ )
                           FR6 = FR6 + T3( I-II+4, I-II+3 )*
     $                                                    C( I+3, UJ+1 )
                           C( I+2, UJ ) = FR5
                           C( I+2, UJ+1 ) = FR6
                           FR3 = FR3 + T3( I-II+2, I-II+2 )*C( I+1, UJ )
                           FR4 = FR4 + T3( I-II+2, I-II+2 )*
     $                                                    C( I+1, UJ+1 )
                           FR3 = FR3 + T3( I-II+4, I-II+2 )*C( I+3, UJ )
                           FR4 = FR4 + T3( I-II+4, I-II+2 )*
     $                                                    C( I+3, UJ+1 )
                           FR3 = FR3 + T3( I-II+3, I-II+2 )*C( I+2, UJ )
                           FR4 = FR4 + T3( I-II+3, I-II+2 )*
     $                                                    C( I+2, UJ+1 )
                           C( I+1, UJ ) = FR3
                           C( I+1, UJ+1 ) = FR4
                           FR1 = FR1 + T3( I-II+1, I-II+1 )*C( I, UJ )
                           FR2 = FR2 + T3( I-II+1, I-II+1 )*C( I, UJ+1 )
                           FR1 = FR1 + T3( I-II+3, I-II+1 )*C( I+2, UJ )
                           FR2 = FR2 + T3( I-II+3, I-II+1 )*
     $                                                    C( I+2, UJ+1 )
                           FR1 = FR1 + T3( I-II+4, I-II+1 )*C( I+3, UJ )
                           FR2 = FR2 + T3( I-II+4, I-II+1 )*
     $                                                    C( I+3, UJ+1 )
                           FR1 = FR1 + T3( I-II+2, I-II+1 )*C( I+1, UJ )
                           FR2 = FR2 + T3( I-II+2, I-II+1 )*
     $                                                    C( I+1, UJ+1 )
                           C( I, UJ ) = FR1
                           C( I, UJ+1 ) = FR2
  530                   CONTINUE
                     ELSE IF( UJSEC.EQ.1 )THEN
                        DO 550, RI = II+ISEC-1, II, -1
                           FR1 = ZERO
                           DO 540, L = RI, II+ISEC-1
                              FR1 = FR1 + T3( L-II+1, RI-II+1 )*
     $                                                        C( L, UJ )
  540                      CONTINUE
                           C( RI, UJ ) = FR1
  550                   CONTINUE
                     END IF
  560             CONTINUE
  570          CONTINUE
            END IF
         END IF
      ELSE
         IF( UPPER )THEN
            IF( NOTR )THEN
*
*              Form  C := alpha*C*A. Right, Upper, No transpose.
*
               IALPHA = ONE/ALPHA
               DO 710, JJ = 1, N, RB
                  JSEC = MIN( RB, N-JJ+1 )
*
*                 C := alpha*C*A + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  IF( II.GT.1 )THEN
                     CALL DGEMM ( 'N', 'N', M, JSEC, JJ-1, -IALPHA,
     $                                  C( 1, 1 ), LDC, A( 1, JJ ), LDA,
     $                                            ONE, C( 1, JJ ), LDC )
                  END IF
                  IF( NOUNIT )THEN
                     DO 580, J = JJ, JJ+JSEC-2, 2
                        T4( J-JJ+1 ) =  -ONE/A( J, J )
                        T4( J-JJ+2 ) =  -ONE/A( J+1, J+1 )
  580                CONTINUE
                     T4( JSEC ) =  -ONE/A( JJ+JSEC-1, JJ+JSEC-1 )
                  ELSE
                     DO 590, J = JJ, JJ+JSEC-2, 2
                        T4( J-JJ+1 ) =  -ONE
                        T4( J-JJ+2 ) =  -ONE
  590                CONTINUE
                     T4( JSEC ) =  -ONE
                  END IF
                  DO 700, II = 1, M, CB
                     ISEC = MIN( CB, M-II+1 )
*
*                    T1 := C', copy the transpose of a rectangular
*                    block of C to T1. Unroll to the depth of 2 by 2 to
*                    facilitate multiple word load and store
*                    instructions.
*
                     UISEC = ISEC-MOD( ISEC, 2 )
                     UJSEC = JSEC-MOD( JSEC, 2 )
                     DO 610, J = JJ, JJ+UJSEC-1, 2
                        DO 600, I = II, II+UISEC-1, 2
                           T1( J-JJ+1, I-II+1 ) = C( I, J )
                           T1( J-JJ+2, I-II+1 ) = C( I, J+1 )
                           T1( J-JJ+1, I-II+2 ) = C( I+1, J )
                           T1( J-JJ+2, I-II+2 ) = C( I+1, J+1 )
  600                   CONTINUE
                        IF( UISEC.LT.ISEC )THEN
                           T1( J-JJ+1, ISEC ) = C( II+ISEC-1, J )
                           T1( J-JJ+2, ISEC ) = C( II+ISEC-1, J+1 )
                        END IF
  610                CONTINUE
                     IF( UJSEC.LT.JSEC )THEN
                        DO 620, I = II, II+ISEC-1
                           T1( JSEC, I-II+1 ) = C( I, JJ+JSEC-1 )
  620                   CONTINUE
                     END IF
*
*                    C := alpha*T1'*A, update a rectangular block of C
*                    using the transpose of a rectangular block of C
*                    stored in T1'.
*
                     UISEC = ISEC-MOD( ISEC, 4 )
                     DO 690, JX = MOD( JJ+JSEC-2, 2 )+1, JJ+JSEC-1, 2
                        UJ = MAX( JJ, JX-2+1 )
                        UJSEC = JX-UJ+1
*
*                       Four by two unrolling.
*
                        IF( UJSEC.EQ.1 )THEN
                           DO 640, RI = II, II+ISEC-1
                              FR1 = ZERO
                              DO 630, L = JJ, UJ-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                                        A( L, UJ )
  630                         CONTINUE
                              FR1 = FR1 - ALPHA*T1( UJ-JJ+1, RI-II+1 )
                              FR1 = FR1*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, RI-II+1 ) = FR1
                              C( RI, UJ ) = FR1
  640                      CONTINUE
                        ELSE IF( UJSEC.EQ.2 )THEN
                           DO 660, I = II, II+UISEC-1, 4
                              FR1 = ZERO
                              FR2 = ZERO
                              FR3 = ZERO
                              FR4 = ZERO
                              FR5 = ZERO
                              FR6 = ZERO
                              FR7 = ZERO
                              FR8 = ZERO
                              DO 650, L = JJ, UJ-1
                                 FR1 = FR1 + T1( L-JJ+1, I-II+1 )*
     $                                                        A( L, UJ )
                                 FR2 = FR2 + T1( L-JJ+1, I-II+1 )*
     $                                                      A( L, UJ+1 )
                                 FR3 = FR3 + T1( L-JJ+1, I-II+2 )*
     $                                                        A( L, UJ )
                                 FR4 = FR4 + T1( L-JJ+1, I-II+2 )*
     $                                                      A( L, UJ+1 )
                                 FR5 = FR5 + T1( L-JJ+1, I-II+3 )*
     $                                                        A( L, UJ )
                                 FR6 = FR6 + T1( L-JJ+1, I-II+3 )*
     $                                                      A( L, UJ+1 )
                                 FR7 = FR7 + T1( L-JJ+1, I-II+4 )*
     $                                                        A( L, UJ )
                                 FR8 = FR8 + T1( L-JJ+1, I-II+4 )*
     $                                                      A( L, UJ+1 )
  650                         CONTINUE
                              FR1 = FR1 - ALPHA*T1( UJ-JJ+1, I-II+1 )
                              FR1 = FR1*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+1 ) = FR1
                              FR2 = FR2 + T1( UJ-JJ+1, I-II+1 )*
     $                                                     A( UJ, UJ+1 )
                              FR2 = FR2 - ALPHA*T1( UJ-JJ+2, I-II+1 )
                              FR2 = FR2*T4( UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+1 ) = FR2
                              FR3 = FR3 - ALPHA*T1( UJ-JJ+1, I-II+2 )
                              FR3 = FR3*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+2 ) = FR3
                              FR4 = FR4 + T1( UJ-JJ+1, I-II+2 )*
     $                                                     A( UJ, UJ+1 )
                              FR4 = FR4 - ALPHA*T1( UJ-JJ+2, I-II+2 )
                              FR4 = FR4*T4( UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+2 ) = FR4
                              FR5 = FR5 - ALPHA*T1( UJ-JJ+1, I-II+3 )
                              FR5 = FR5*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+3 ) = FR5
                              FR6 = FR6 + T1( UJ-JJ+1, I-II+3 )*
     $                                                     A( UJ, UJ+1 )
                              FR6 = FR6 - ALPHA*T1( UJ-JJ+2, I-II+3 )
                              FR6 = FR6*T4( UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+3 ) = FR6
                              FR7 = FR7 - ALPHA*T1( UJ-JJ+1, I-II+4 )
                              FR7 = FR7*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+4 ) = FR7
                              FR8 = FR8 + T1( UJ-JJ+1, I-II+4 )*
     $                                                     A( UJ, UJ+1 )
                              FR8 = FR8 - ALPHA*T1( UJ-JJ+2, I-II+4 )
                              FR8 = FR8*T4( UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+4 ) = FR8
                              C( I, UJ ) = FR1
                              C( I, UJ+1 ) = FR2
                              C( I+1, UJ ) = FR3
                              C( I+1, UJ+1 ) = FR4
                              C( I+2, UJ ) = FR5
                              C( I+2, UJ+1 ) = FR6
                              C( I+3, UJ ) = FR7
                              C( I+3, UJ+1 ) = FR8
  660                      CONTINUE
                           DO 680, RI = II+UISEC, II+ISEC-1
                              FR1 = ZERO
                              FR2 = ZERO
                              DO 670, L = JJ, UJ-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                                        A( L, UJ )
                                 FR2 = FR2 + T1( L-JJ+1, RI-II+1 )*
     $                                                      A( L, UJ+1 )
  670                         CONTINUE
                              FR1 = FR1 - ALPHA*T1( UJ-JJ+1, RI-II+1 )
                              FR1 = FR1*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, RI-II+1 ) = FR1
                              FR2 = FR2 + T1( UJ-JJ+1, RI-II+1 )*
     $                                                     A( UJ, UJ+1 )
                              FR2 = FR2 - ALPHA*T1( UJ-JJ+2, RI-II+1 )
                              FR2 = FR2*T4( UJ-JJ+2 )
                              T1( UJ-JJ+2, RI-II+1 ) = FR2
                              C( RI, UJ ) = FR1
                              C( RI, UJ+1 ) = FR2
  680                      CONTINUE
                        END IF
  690                CONTINUE
  700             CONTINUE
  710          CONTINUE
            ELSE
*
*              Solve  X*A' = alpha*C. Right, Upper, Transpose.
*
               IALPHA = ONE/ALPHA
               DO 880, JX = N, 1, -RB
                  JJ = MAX( 1, JX-RB+1 )
                  JSEC = JX-JJ+1
*
*                 C := alpha*C*A + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  IF( JJ+JSEC.LE.N )THEN
                     CALL DGEMM ( 'N', 'T', M, JSEC, N-JJ-JSEC+1,
     $                                    -IALPHA, C( 1, JJ+JSEC ), LDC,
     $                                       A( JJ, JJ+JSEC ), LDA, ONE,
     $                                                 C( 1, JJ ), LDC )
                  END IF
*
*                 If diag = 'N', T3 := -1.0/alpha*A', except for the
*                 diagonal elements of T3, T3(i,i) := alpha/A(i,i).
*                 If diag = 'U', T3 := -1.0*A', except for the
*                 diagonal elements of T3, T3(i,i) := alpha. Unroll to
*                 the depth of 2 by 2 to facilitate multiple word load
*                 and store instructions.
*
                  DO 730, L = JJ+JSEC-2, JJ-1, -2
                     UJSEC = L-JJ-MOD( L-JJ, 2 )
                     DO 720, J = JJ, JJ+UJSEC-1, 2
                        T2( L-JJ+1, J-JJ+1 ) = -A( J, L )
                        T2( L-JJ+2, J-JJ+1 ) = -A( J, L+1 )
                        T2( L-JJ+1, J-JJ+2 ) = -A( J+1, L )
                        T2( L-JJ+2, J-JJ+2 ) = -A( J+1, L+1 )
  720                CONTINUE
                     IF( MOD( L-JJ, 2 ).EQ.1 )THEN
                        T2( L-JJ+1, L-JJ ) = -A( L-1, L )
                        T2( L-JJ+2, L-JJ ) = -A( L-1, L+1 )
                     END IF
                     IF( L-JJ.GE.0 )THEN
                        T2( L-JJ+1, L-JJ+1 ) = ALPHA
                        T2( L-JJ+2, L-JJ+1 ) = -A( L, L+1 )
                     END IF
                     T2( L-JJ+2, L-JJ+2 ) = ALPHA
  730             CONTINUE
                  IF( NOUNIT )THEN
                     DO 760, L = 1, JSEC
                        GAMMA = ONE/A( L+JJ-1, L+JJ-1 )
                        T2( L, L ) = GAMMA*ALPHA
                        RJSEC = MOD( JSEC-L, 4 )+1
                        DO 740, J = L+1, L+RJSEC-1
                           T2( J, L ) = GAMMA*T2( J, L )
  740                   CONTINUE
                        DO 750, J = L+RJSEC, JSEC, 4
                           T2( J,   L ) = GAMMA*T2( J, L )
                           T2( J+1, L ) = GAMMA*T2( J+1, L )
                           T2( J+2, L ) = GAMMA*T2( J+2, L )
                           T2( J+3, L ) = GAMMA*T2( J+3, L )
  750                   CONTINUE
  760                CONTINUE
                  END IF
                  DO 870, II = 1, M, CB
                     ISEC = MIN( CB, M-II+1 )
*
*                    T1 := C', copy the transpose of a rectangular
*                    block of C to T1. Unroll to the depth of 2 by 2
*                    to facilitate multiple word load and store
*                    instructions.
*
                     UISEC = ISEC-MOD( ISEC, 2 )
                     UJSEC = JSEC-MOD( JSEC, 2 )
                     DO 780, J = JJ, JJ+UJSEC-1, 2
                        DO 770, I = II, II+UISEC-1, 2
                           T1( J-JJ+1, I-II+1 ) = C( I, J )
                           T1( J-JJ+2, I-II+1 ) = C( I, J+1 )
                           T1( J-JJ+1, I-II+2 ) = C( I+1, J )
                           T1( J-JJ+2, I-II+2 ) = C( I+1, J+1 )
  770                   CONTINUE
                        IF( UISEC.LT.ISEC )THEN
                           T1( J-JJ+1, ISEC ) = C( II+ISEC-1, J )
                           T1( J-JJ+2, ISEC ) = C( II+ISEC-1, J+1 )
                        END IF
  780                CONTINUE
                     IF( UJSEC.LT.JSEC )THEN
                        DO 790, I = II, II+ISEC-1
                           T1( JSEC, I-II+1 ) = C( I, JJ+JSEC-1 )
  790                   CONTINUE
                     END IF
*
*                    C := alpha*T1'*A, update a rectangular block of C
*                    using the transpose of a rectangular block of C
*                    stored in T1'.
*
                     RISEC = MOD( ISEC, 4 )
                     DO 860, JY = JJ+JSEC-1, JJ, -2
                        UJ = MAX( JJ, JY-2+1 )
                        UJSEC = JY-UJ+1
*
*                       Four by two unrolling.
*
                        IF( UJSEC.EQ.1 )THEN
                           DO 810, RI = II+ISEC-1, II, -1
                              FR1 = ZERO
                              DO 800, L = UJ, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
  800                         CONTINUE
                              T1( UJ-JJ+1, RI-II+1 ) = FR1
                              C( RI, UJ ) = FR1
  810                      CONTINUE
                        ELSE IF( UJSEC.EQ.2 )THEN
                           DO 830, I = II+ISEC-4, II+RISEC, -4
                              FR1 = ZERO
                              FR2 = ZERO
                              FR3 = ZERO
                              FR4 = ZERO
                              FR5 = ZERO
                              FR6 = ZERO
                              FR7 = ZERO
                              FR8 = ZERO
                              DO 820, L = UJ+2, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, I-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR2 = FR2 + T1( L-JJ+1, I-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR3 = FR3 + T1( L-JJ+1, I-II+2 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR4 = FR4 + T1( L-JJ+1, I-II+2 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR5 = FR5 + T1( L-JJ+1, I-II+3 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR6 = FR6 + T1( L-JJ+1, I-II+3 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR7 = FR7 + T1( L-JJ+1, I-II+4 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR8 = FR8 + T1( L-JJ+1, I-II+4 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
  820                         CONTINUE
                              FR2 = FR2 + T1( UJ-JJ+2, I-II+1 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+1 ) = FR2
                              FR1 = FR1 + T1( UJ-JJ+2, I-II+1 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+1 )
                              FR1 = FR1 + T1( UJ-JJ+1, I-II+1 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+1 ) = FR1
                              FR4 = FR4 + T1( UJ-JJ+2, I-II+2 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+2 ) = FR4
                              FR3 = FR3 + T1( UJ-JJ+2, I-II+2 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+1 )
                              FR3 = FR3 + T1( UJ-JJ+1, I-II+2 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+2 ) = FR3
                              FR6 = FR6 + T1( UJ-JJ+2, I-II+3 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+3 ) = FR6
                              FR5 = FR5 + T1( UJ-JJ+2, I-II+3 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+1 )
                              FR5 = FR5 + T1( UJ-JJ+1, I-II+3 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+3 ) = FR5
                              FR8 = FR8 + T1( UJ-JJ+2, I-II+4 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+4 ) = FR8
                              FR7 = FR7 + T1( UJ-JJ+2, I-II+4 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+1 )
                              FR7 = FR7 + T1( UJ-JJ+1, I-II+4 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+4 ) = FR7
                              C( I, UJ ) = FR1
                              C( I, UJ+1 ) = FR2
                              C( I+1, UJ ) = FR3
                              C( I+1, UJ+1 ) = FR4
                              C( I+2, UJ ) = FR5
                              C( I+2, UJ+1 ) = FR6
                              C( I+3, UJ ) = FR7
                              C( I+3, UJ+1 ) = FR8
  830                      CONTINUE
                           DO 850, RI = II+RISEC-1, II, -1
                              FR1 = ZERO
                              FR2 = ZERO
                              DO 840, L = UJ+2, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR2 = FR2 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
  840                         CONTINUE
                              FR2 = FR2 + T1( UJ-JJ+2, RI-II+1 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              T1( UJ-JJ+2, RI-II+1 ) = FR2
                              FR1 = FR1 + T1( UJ-JJ+2, RI-II+1 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+1 )
                              FR1 = FR1 + T1( UJ-JJ+1, RI-II+1 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              T1( UJ-JJ+1, RI-II+1 ) = FR1
                              C( RI, UJ ) = FR1
                              C( RI, UJ+1 ) = FR2
  850                      CONTINUE
                        END IF
  860                CONTINUE
  870             CONTINUE
  880          CONTINUE
            END IF
         ELSE
            IF( NOTR )THEN
*
*              Solve  X*A = alpha*C. Right, Lower, No transpose.
*
               IALPHA = ONE/ALPHA
               DO 1020, JX = N, 1, -RB
                  JJ = MAX( 1, JX-RB+1 )
                  JSEC = JX-JJ+1
*
*                 C := alpha*C*A + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  IF( JJ+JSEC.LE.N )THEN
                     CALL DGEMM ( 'N', 'N', M, JSEC, N-JJ-JSEC+1,
     $                                    -IALPHA, C( 1, JJ+JSEC ), LDC,
     $                                       A( JJ+JSEC, JJ ), LDA, ONE,
     $                                                 C( 1, JJ ), LDC )
                  END IF
                  IF( NOUNIT )THEN
                     DO 890, J = JJ, JJ+JSEC-2, 2
                        T4( J-JJ+1 ) =  -ONE/A( J, J )
                        T4( J-JJ+2 ) =  -ONE/A( J+1, J+1 )
  890                CONTINUE
                     T4( JSEC ) =  -ONE/A( JJ+JSEC-1, JJ+JSEC-1 )
                  ELSE
                     DO 900, J = JJ, JJ+JSEC-2, 2
                        T4( J-JJ+1 ) =  -ONE
                        T4( J-JJ+2 ) =  -ONE
  900                CONTINUE
                     T4( JSEC ) =  -ONE
                  END IF
                  DO 1010, II = 1, M, CB
                     ISEC = MIN( CB, M-II+1 )
*
*                    T1 := C', copy the transpose of a rectangular
*                    block of C to T1. Unroll to the depth of 2 by 2
*                    to facilitate multiple word load and store
*                    instructions.
*
                     UISEC = ISEC-MOD( ISEC, 2 )
                     UJSEC = JSEC-MOD( JSEC, 2 )
                     DO 920, J = JJ, JJ+UJSEC-1, 2
                        DO 910, I = II, II+UISEC-1, 2
                           T1( J-JJ+1, I-II+1 ) = C( I, J )
                           T1( J-JJ+2, I-II+1 ) = C( I, J+1 )
                           T1( J-JJ+1, I-II+2 ) = C( I+1, J )
                           T1( J-JJ+2, I-II+2 ) = C( I+1, J+1 )
  910                   CONTINUE
                        IF( UISEC.LT.ISEC )THEN
                           T1( J-JJ+1, ISEC ) = C( II+ISEC-1, J )
                           T1( J-JJ+2, ISEC ) = C( II+ISEC-1, J+1 )
                        END IF
  920                CONTINUE
                     IF( UJSEC.LT.JSEC )THEN
                        DO 930, I = II, II+ISEC-1
                           T1( JSEC, I-II+1 ) = C( I, JJ+JSEC-1 )
  930                   CONTINUE
                     END IF
*
*                    C := alpha*T1'*A, update a rectangular block of C
*                    using the transpose of a rectangular block of C
*                    stored in T1'.
*
                     RISEC = MOD( ISEC, 4 )
                     DO 1000, JY = JJ+JSEC-1, JJ, -2
                        UJ = MAX( JJ, JY-2+1 )
                        UJSEC = JY-UJ+1
*
*                       Four by two unrolling.
*
                        IF( UJSEC.EQ.1 )THEN
                           DO 950, RI = II+ISEC-1, II, -1
                              FR1 = ZERO
                              DO 940, L = UJ+1, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                                        A( L, UJ )
  940                         CONTINUE
                              FR1 = FR1 - ALPHA*T1( UJ-JJ+1, RI-II+1 )
                              FR1 = FR1*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, RI-II+1 ) = FR1
                              C( RI, UJ ) = FR1
  950                      CONTINUE
                        ELSE IF( UJSEC.EQ.2 )THEN
                           DO 970, I = II+ISEC-4, II+RISEC, -4
                              FR1 = ZERO
                              FR2 = ZERO
                              FR3 = ZERO
                              FR4 = ZERO
                              FR5 = ZERO
                              FR6 = ZERO
                              FR7 = ZERO
                              FR8 = ZERO
                              DO 960, L = UJ+2, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, I-II+1 )*
     $                                                        A( L, UJ )
                                 FR2 = FR2 + T1( L-JJ+1, I-II+1 )*
     $                                                      A( L, UJ+1 )
                                 FR3 = FR3 + T1( L-JJ+1, I-II+2 )*
     $                                                        A( L, UJ )
                                 FR4 = FR4 + T1( L-JJ+1, I-II+2 )*
     $                                                      A( L, UJ+1 )
                                 FR5 = FR5 + T1( L-JJ+1, I-II+3 )*
     $                                                        A( L, UJ )
                                 FR6 = FR6 + T1( L-JJ+1, I-II+3 )*
     $                                                      A( L, UJ+1 )
                                 FR7 = FR7 + T1( L-JJ+1, I-II+4 )*
     $                                                        A( L, UJ )
                                 FR8 = FR8 + T1( L-JJ+1, I-II+4 )*
     $                                                      A( L, UJ+1 )
  960                         CONTINUE
                              FR2 = FR2 - ALPHA*T1( UJ-JJ+2, I-II+1 )
                              FR2 = FR2*T4( UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+1 ) = FR2
                              FR1 = FR1 + T1( UJ-JJ+2, I-II+1 )*
     $                                                     A( UJ+1, UJ )
                              FR1 = FR1 - ALPHA*T1( UJ-JJ+1, I-II+1 )
                              FR1 = FR1*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+1 ) = FR1
                              FR4 = FR4 - ALPHA*T1( UJ-JJ+2, I-II+2 )
                              FR4 = FR4*T4( UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+2 ) = FR4
                              FR3 = FR3 + T1( UJ-JJ+2, I-II+2 )*
     $                                                     A( UJ+1, UJ )
                              FR3 = FR3 - ALPHA*T1( UJ-JJ+1, I-II+2 )
                              FR3 = FR3*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+2 ) = FR3
                              FR6 = FR6 - ALPHA*T1( UJ-JJ+2, I-II+3 )
                              FR6 = FR6*T4( UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+3 ) = FR6
                              FR5 = FR5 + T1( UJ-JJ+2, I-II+3 )*
     $                                                     A( UJ+1, UJ )
                              FR5 = FR5 - ALPHA*T1( UJ-JJ+1, I-II+3 )
                              FR5 = FR5*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+3 ) = FR5
                              FR8 = FR8 - ALPHA*T1( UJ-JJ+2, I-II+4 )
                              FR8 = FR8*T4( UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+4 ) = FR8
                              FR7 = FR7 + T1( UJ-JJ+2, I-II+4 )*
     $                                                     A( UJ+1, UJ )
                              FR7 = FR7 - ALPHA*T1( UJ-JJ+1, I-II+4 )
                              FR7 = FR7*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+4 ) = FR7
                              C( I, UJ ) = FR1
                              C( I, UJ+1 ) = FR2
                              C( I+1, UJ ) = FR3
                              C( I+1, UJ+1 ) = FR4
                              C( I+2, UJ ) = FR5
                              C( I+2, UJ+1 ) = FR6
                              C( I+3, UJ ) = FR7
                              C( I+3, UJ+1 ) = FR8
  970                      CONTINUE
                           DO 990, RI = II+RISEC-1, II, -1
                              FR1 = ZERO
                              FR2 = ZERO
                              DO 980, L = UJ+2, JJ+JSEC-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                                        A( L, UJ )
                                 FR2 = FR2 + T1( L-JJ+1, RI-II+1 )*
     $                                                      A( L, UJ+1 )
  980                         CONTINUE
                              FR2 = FR2 - ALPHA*T1( UJ-JJ+2, RI-II+1 )
                              FR2 = FR2*T4( UJ-JJ+2 )
                              T1( UJ-JJ+2, RI-II+1 ) = FR2
                              FR1 = FR1 + T1( UJ-JJ+2, RI-II+1 )*
     $                                                     A( UJ+1, UJ )
                              FR1 = FR1 - ALPHA*T1( UJ-JJ+1, RI-II+1 )
                              FR1 = FR1*T4( UJ-JJ+1 )
                              T1( UJ-JJ+1, RI-II+1 ) = FR1
                              C( RI, UJ ) = FR1
                              C( RI, UJ+1 ) = FR2
  990                      CONTINUE
                        END IF
 1000                CONTINUE
 1010             CONTINUE
 1020          CONTINUE
            ELSE
*
*              Solve  X*A' = alpha*C. Right, Lower, Transpose.
*
               IALPHA = ONE/ALPHA
               DO 1190, JY = MOD( N-1, RB )+1, N, RB
                  JJ = MAX( 1, JY-RB+1 )
                  JSEC = JY-JJ+1
*
*                 C := alpha*C*A + C, general matrix multiply
*                 involving a rectangular block of A.
*
                  IF( II.GT.1 )THEN
                     CALL DGEMM ( 'N', 'T', M, JSEC, JJ-1, -IALPHA,
     $                                  C( 1, 1 ), LDC, A( JJ, 1 ), LDA,
     $                                            ONE, C( 1, JJ ), LDC )
                  END IF
*
*                 T2 := alpha*A', copy the transpose of a lower
*                 triangular diagonal block of alpha*A' to T2. Unroll to
*                 the depth of 2 by 2 to facilitate multiple word load
*                 and store instructions.
*
*
                  DO 1040, L = JJ, JJ+JSEC-1, 2
                     T2( L-JJ+1, L-JJ+1 ) = ALPHA
                     RJSEC = MOD( JJ+JSEC-L-2, 2 )+2
                     IF( RJSEC.GE.2 )THEN
                        T2( L-JJ+1, L-JJ+2 ) = -A( L+1, L )
                        T2( L-JJ+2, L-JJ+2 ) = ALPHA
                     END IF
                     IF( RJSEC.EQ.3 )THEN
                        T2( L-JJ+1, L-JJ+3 ) = -A( L+2, L )
                        T2( L-JJ+2, L-JJ+3 ) = -A( L+2, L+1 )
                     END IF
                     DO 1030, J = L+RJSEC, JJ+JSEC-1, 2
                        T2( L-JJ+1, J-JJ+1 ) = -A( J, L )
                        T2( L-JJ+2, J-JJ+1 ) = -A( J, L+1 )
                        T2( L-JJ+1, J-JJ+2 ) = -A( J+1, L )
                        T2( L-JJ+2, J-JJ+2 ) = -A( J+1, L+1 )
 1030                CONTINUE
 1040             CONTINUE
                  IF( NOUNIT )THEN
                     DO 1070, L = 1, JSEC
                        GAMMA = ONE/A( L+JJ-1, L+JJ-1 )
                        UJSEC = L-1-MOD( L-1, 4 )
                        DO 1050, J = 1, UJSEC, 4
                           T2( J,   L ) = GAMMA*T2( J, L )
                           T2( J+1, L ) = GAMMA*T2( J+1, L )
                           T2( J+2, L ) = GAMMA*T2( J+2, L )
                           T2( J+3, L ) = GAMMA*T2( J+3, L )
 1050                   CONTINUE
                        DO 1060, J = 1+UJSEC, JSEC-1
                           T2( J, L ) = GAMMA*T2( J, L )
 1060                   CONTINUE
                        T2( L, L ) = GAMMA*ALPHA
 1070                CONTINUE
                  END IF
                  DO 1180, II = 1, M, CB
                     ISEC = MIN( CB, M-II+1 )
*
*                    T1 := C', copy the transpose of a rectangular
*                    block of C to T1. Unroll to the depth of 2 by 2
*                    to facilitate multiple word load and store
*                    instructions.
*
                     UISEC = ISEC-MOD( ISEC, 2 )
                     UJSEC = JSEC-MOD( JSEC, 2 )
                     DO 1090, J = JJ, JJ+UJSEC-1, 2
                        DO 1080, I = II, II+UISEC-1, 2
                           T1( J-JJ+1, I-II+1 ) = C( I, J )
                           T1( J-JJ+2, I-II+1 ) = C( I, J+1 )
                           T1( J-JJ+1, I-II+2 ) = C( I+1, J )
                           T1( J-JJ+2, I-II+2 ) = C( I+1, J+1 )
 1080                   CONTINUE
                        IF( UISEC.LT.ISEC )THEN
                           T1( J-JJ+1, ISEC ) = C( II+ISEC-1, J )
                           T1( J-JJ+2, ISEC ) = C( II+ISEC-1, J+1 )
                        END IF
 1090                CONTINUE
                     IF( UJSEC.LT.JSEC )THEN
                        DO 1100, I = II, II+ISEC-1
                           T1( JSEC, I-II+1 ) = C( I, JJ+JSEC-1 )
 1100                   CONTINUE
                     END IF
*
*                    C := alpha*T1'*A, update a rectangular block of C
*                    using the transpose of a rectangular block of C
*                    stored in T1'.
*
                     UISEC = ISEC-MOD( ISEC, 4 )
                     DO 1170, JX = MOD( JJ+JSEC-2, 2 )+1, JJ+JSEC-1, 2
                        UJ = MAX( JJ, JX-2+1 )
                        UJSEC = JX-UJ+1
*
*                       Four by two unrolling.
*
                        IF( UJSEC.EQ.1 )THEN
                           DO 1120, RI = II, II+ISEC-1
                              FR1 = ZERO
                              DO 1110, L = JJ, UJ
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
 1110                         CONTINUE
                              T1( UJ-JJ+1, RI-II+1 ) = FR1
                              C( RI, UJ ) = FR1
 1120                      CONTINUE
                        ELSE IF( UJSEC.EQ.2 )THEN
                           DO 1140, I = II, II+UISEC-1, 4
                              FR1 = ZERO
                              FR2 = ZERO
                              FR3 = ZERO
                              FR4 = ZERO
                              FR5 = ZERO
                              FR6 = ZERO
                              FR7 = ZERO
                              FR8 = ZERO
                              DO 1130, L = JJ, UJ-1
                                 FR1 = FR1 + T1( L-JJ+1, I-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR2 = FR2 + T1( L-JJ+1, I-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR3 = FR3 + T1( L-JJ+1, I-II+2 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR4 = FR4 + T1( L-JJ+1, I-II+2 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR5 = FR5 + T1( L-JJ+1, I-II+3 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR6 = FR6 + T1( L-JJ+1, I-II+3 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
                                 FR7 = FR7 + T1( L-JJ+1, I-II+4 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR8 = FR8 + T1( L-JJ+1, I-II+4 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
 1130                         CONTINUE
                              FR1 = FR1 + T1( UJ-JJ+1, I-II+1 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+1 ) = FR1
                              FR2 = FR2 + T1( UJ-JJ+1, I-II+1 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+2 )
                              FR2 = FR2 + T1( UJ-JJ+2, I-II+1 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+1 ) = FR2
                              FR3 = FR3 + T1( UJ-JJ+1, I-II+2 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+2 ) = FR3
                              FR4 = FR4 + T1( UJ-JJ+1, I-II+2 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+2 )
                              FR4 = FR4 + T1( UJ-JJ+2, I-II+2 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+2 ) = FR4
                              FR5 = FR5 + T1( UJ-JJ+1, I-II+3 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+3 ) = FR5
                              FR6 = FR6 + T1( UJ-JJ+1, I-II+3 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+2 )
                              FR6 = FR6 + T1( UJ-JJ+2, I-II+3 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+3 ) = FR6
                              FR7 = FR7 + T1( UJ-JJ+1, I-II+4 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              T1( UJ-JJ+1, I-II+4 ) = FR7
                              FR8 = FR8 + T1( UJ-JJ+1, I-II+4 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+2 )
                              FR8 = FR8 + T1( UJ-JJ+2, I-II+4 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              T1( UJ-JJ+2, I-II+4 ) = FR8
                              C( I, UJ ) = FR1
                              C( I, UJ+1 ) = FR2
                              C( I+1, UJ ) = FR3
                              C( I+1, UJ+1 ) = FR4
                              C( I+2, UJ ) = FR5
                              C( I+2, UJ+1 ) = FR6
                              C( I+3, UJ ) = FR7
                              C( I+3, UJ+1 ) = FR8
 1140                      CONTINUE
                           DO 1160, RI = II+UISEC, II+ISEC-1
                              FR1 = ZERO
                              FR2 = ZERO
                              DO 1150, L = JJ, UJ-1
                                 FR1 = FR1 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+1 )
                                 FR2 = FR2 + T1( L-JJ+1, RI-II+1 )*
     $                                             T2( L-JJ+1, UJ-JJ+2 )
 1150                         CONTINUE
                              FR1 = FR1 + T1( UJ-JJ+1, RI-II+1 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+1 )
                              T1( UJ-JJ+1, RI-II+1 ) = FR1
                              FR2 = FR2 + T1( UJ-JJ+1, RI-II+1 )*
     $                                            T2( UJ-JJ+1, UJ-JJ+2 )
                              FR2 = FR2 + T1( UJ-JJ+2, RI-II+1 )*
     $                                            T2( UJ-JJ+2, UJ-JJ+2 )
                              T1( UJ-JJ+2, RI-II+1 ) = FR2
                              C( RI, UJ ) = FR1
                              C( RI, UJ+1 ) = FR2
 1160                      CONTINUE
                        END IF
 1170                CONTINUE
 1180             CONTINUE
 1190          CONTINUE
            END IF
         END IF
      END IF
*
      RETURN
*
*     End of DTRSM.
*
      END
