/* morph3d_sub.c  --  Morph3DSub main procedure  */

/* contains: Morph3DSub() */

/* dependencies: SetUpOp(), AllocBuf(), DeAllocBuf(),                */
/*               InputImage(), OutputImage(), ThresholdImg(),        */
/*               BlockMove(), BinBinErode(), BinBinDilate(),         */
/*               BinSetMaxMin(), BinSetMinMax(), BinGrayErode(),     */
/*               BinGrayDilate(), BinRankFilt(), GraySetMaxMin(),    */
/*               GraySetMinMax(), GrayBinErode(), GrayBinDilate(),   */
/*               GrayGraySetErode(), GrayGraySetDilate(),            */
/*               GrayGrayFctErode(), GrayGrayFctDilate(),            */
/*               GrayRankFilt(), GrayFctMaxMin(), GrayFctMinMax()    */
/*               LUMFilt(), LUMSmooth(), LUMSharp()                  */



/* morph3d version 4.1    12 July 1993             */
/* 3D image morphology program                     */
/*                                                 */
/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */
/* Nashville, TN 37235                             */
/* rap2@vuse.vanderbilt.edu                        */
/*                                                 */
/* This software is freely redistributable if      */
/* the author's name and affiliation are included. */


#include "morph3d_sub.h"
#include "m3dse.h"


/* 4.1 change: rewrote main loop using a new delay strategy to permit */
/* use of 3D SE's with even z-depths or a depth of 1.*/


   /* call Morph3DSub with complete SE info (I.e call another SR to get it) */
   /*                  also with addresses of file_i/o procs for callback */

void Morph3DSub( MOp, InputImage, InStruct, OutputImage, OutStruct, 
                 Ix, Iy, ImgType, NZPad,LThresh, UThresh, 
                 SE, sX, sY, sZ, sV, sorgx, sorgy, sorgz, SEType, SorF, 
                 Rank ) 

   int MOp;               /* Morphological operation to perform */

   int (*InputImage)();   /* address of image input function  */
   int *InStruct;         /* address of structure to pass to InputImage */
   int (*OutputImage)();  /* address of image output function */
   int *OutStruct;        /* address of structure to pass to OutputImage */
   int Ix, Iy;            /* image horizontal, vertical dimensions */
   int ImgType;           /* image type (gray-level or binary) */
   int NZPad ;            /* flag.  F => zeropadding of input */
   int LThresh;           /* lower binary threshold value */
   int UThresh;           /* upper binary threshold value */

   int *SE;               /* SE in plane-row-major order */
   int sX,sY,sZ,sV;       /* SE x, y, and z support dims and max gray-lev (v) */
   int sorgx,sorgy,sorgz; /* SE origin coordinates */
   int SEType;            /* Binary SE or gray-level SE */
   int SorF;              /* Set operation or Function operation */
   int Rank;              /* rank for rank filter */

   {

   int MorphOp;           /* morphological operation specification variable */

   int iX,iY;             /* image dimensions (w/ zero padding) */
   int tX,tY;             /* transform scan dimensions */
   int bX,bY;             /* x and y image border (zeropad) widths */
   int oX,oY;             /* image offsets */
   int ssize;             /* SE support size */
   int Opt;               /* optional argument */

   void (*morph1)();    /* addresses of morphological function 1 */
   void (*morph2)();    /* addresses of morphological function 2 */
   int LUMparam[3];     /* LUM filter parameters (k,l,med) */

   int InDelay1,InDelay2;   /* wait for this many images to be input before */
                            /* starting morphological operations */
   int EndDelay1,EndDelay2; /* wait for this many zero images to be scrolled */
                            /* into the buffers before quitting */ 

   int eos;              /* (End Of Sequence) T => last file has been read */
   byte *In;             /* a pointer to a locally allocated image buffer.   */
                         /* The call-back function, InputImage passes this  */
                         /* as its first argument */
   byte *Out;            /* a pointer to a locally allocated image buffer   */
                         /* The call-back function, OutputImage passes this */
                         /* as its first argument */
   byte **Buf1,**Buf2;   /* pointer to a list of pointers, each of which is */
                         /* an image */
   int PixSize;          /* 2/1=> intermediate images have 16/8 bit pixels */
   int NumInReqs;        /* number of image input requests */
   int NumOutReqs;       /* number of image output requests */
   int InCount;          /* number of images scrolled into the buffer */
   int EndCount;         /* number of zero image requests after EOS */


   /* begin */

   /* check to see that the essentials have been passed */
   if ( !(InputImage && OutputImage && SE) )
      {
      fprintf( stderr,"NULL address passed to Morph3DSub\n");
      exit( 0 );
      }
   if ( !(Ix && Iy && sX && sY && sZ) )
      {
      fprintf( stderr,"Zero image or SE dimension passed to Morph3DSub\n");
      exit( 0 );
      }

   /* initialize everything */
   eos = FALSE;
   morph1=morph2=NULL;
   Buf1=Buf2=(byte **)NULL;
   iX=iY=tX=tY=oX=oY=ssize=0;
   NumInReqs=NumOutReqs=0;
   InCount=EndCount=0;

   /* select appropriate functions given Op specification */
   MorphOp = SetUpOp( &morph1,&morph2,MOp,SorF,SEType,ImgType,&LThresh,
                      &UThresh,SE,sX,sY,sZ,sV,&ssize,Rank,&Opt,LUMparam );

   /* determine if word image or byte image procedure */
   PixSize = ((MorphOp & SFMASK) == FUNCT  &&   
              (MorphOp & SEMASK) == GRASE  &&
              (MorphOp & IMMASK) == GRAIMG   )  ? WORDPX : BYTEPX ;

   /* calculate image zeropad border widths */  
   bX = sX - 1;
   bY = sY - 1;

   /* calculate working image dimensions and transform scan dimensions */
   if ( NZPad )
      {
      iX = Ix;            /* dimensions of image actually scanned by prog. */
      iY = Iy;            /* including zero padding (if any) */
      tX = iX - bX;       /* actual number of neighborhoods scanned per row */
      tY = iY - bY;       /* actual number of neighborhoods scanned per col */
      oX = oY = 0;        /* offset between i/o images and working images */
      }
   else
      {
      iX = Ix + 2*bX;     /* dimensions of image actually scanned by prog. */
      iY = Iy + 2*bY;     /* including zero padding (if any) */
      tX = Ix + bX;       /* actual number of neighborhoods scanned per row */
      tY = Iy + bY;       /* actual number of neighborhoods scanned per col */
      oX = bX;            /* offset between i/o images and working images */
      oY = bY;
      }

   /* allocate and initialize buffers to contain all zeros */
   Buf1 = AllocBuf( iX, iY, sZ, PixSize );
   if ( morph2 ) Buf2 = AllocBuf( iX, iY, sZ, PixSize );
   In  = (byte *)AllocBuf( iX, iY, 0, PixSize );
   Out = (byte *)AllocBuf( iX, iY, 0, PixSize );
    
   /* setup input and EOS delays */
   switch ( MorphOp & OPMASK )
      {
      /* if erode or rank need sorgz zero-images in buffer 1  */
      /*          and set input delay 1 to sZ - sorgz - 1     */
      case ERODE  :
      case RANK   :
      case LUM    :
      case LUMSMO :
      case LUMSHA :
      case MINMAX :
      case MAXMIN :
         InDelay1  = sZ - sorgz;
         InDelay2  = 0;
         EndDelay1 = InDelay1 - 1;
         EndDelay2 = 0;
         break;

      /* if dilate need sZ - sorgz - 1 zero-images in buffer 1 */
      /*           and set input delay 1 to sorgz              */
      case DILATE :
         InDelay1  = sorgz + 1;
         InDelay2  = 0;
         EndDelay1 = InDelay1 - 1;
         EndDelay2 = 0;
         break;

      /* if open need sorgz zero-images in buffer 1            */
      /*         set input delay 1 to sZ - sorgz - 1           */
      /*         need sZ - sorgz - 1 zero-images in buffer 2   */
      /*         set input delay 2 to input delay 1 plus sorgz */
      case OPEN :
         InDelay1  = sZ - sorgz;
         InDelay2  = sZ;
         EndDelay1 = InDelay1 - 1;
         EndDelay2 = sZ-1;
         break;

      /* if close need sZ - sorgz - 1 zero-images in buffer 1            */
      /*          set input delay 1 to sorgz                             */
      /*          need sorgz zero-images in buffer 2                     */
      /*          set input delay 2 to input delay 1 plus sZ - sorgz - 1 */
      case CLOSE :
         InDelay1  = sorgz + 1;
         InDelay2  = sZ;
         EndDelay1 = InDelay1 - 1;
         EndDelay2 = sZ-1;
         break;
      /* this error should never occur */
      default:
         fprintf(stderr,
         "Warning: The flurby map has overwritten the lamoid pointer!\n");
         break;
      }


   do /* the main sequence processing loop */
      {

      if ( !eos )      /* (the next attempt to get an      */ 
         {             /* image may result in eos == TRUE) */
         ++NumInReqs;

         /* call back for the next image in the sequence */
         if ( InputImage( In, NumInReqs, &eos, InStruct ) )
            {
            fprintf(stderr,"Morph3DSub: Input procedure returned error.\n" );
            exit(0);
            }
         }

      if ( eos ) /* we've reached or passed the end of the sequence */
         {
         if ( !InCount )
            {
            fprintf(stderr,"Morph3DSub: There are no images in the sequence.\n" );
            exit(0);
            }
         ++EndCount;
         if ( EndCount > EndDelay1 )  /* we've passed the morph1 output delay */
            {
            if ( !morph2  || (EndCount > EndDelay2) ) 
               {        /* if there is no morph2, or if there is but we have     */
               break;   /* passed it's output delay, break out of the main loop. */
               }        /* We're done! */
            }
         }

      /* scroll image into buffer 1 (note: if eos, will scroll in zero-img */
      ScrollBuf(Buf1,iX,iY,PixSize,sZ,oX,oY,(eos?NULL:In),Ix,Iy,BYTEPX );

      /* increment scroll-in count */
      ++InCount;

      /* if thresholding required, do it */
      if ( (!eos)  &&  (((MorphOp & SFMASK) == SET   ) &&  UThresh) ) 
         {
         if (LThresh > UThresh) OptErr( 7, NULL, NULL );
         ThresholdImg( Buf1[sZ-1], iX, iY, LThresh, UThresh );
         }

      /* if less than input delay 1 go to the begining of the do-loop */
      if ( InCount < InDelay1 ) continue;

      /* Do first pass morphop */
      morph1( Buf1,Out,iX,tX,tY,SE,sX,sY,sZ,sorgx,sorgy,sorgz,Opt ); 

      /* if there is no 2nd pass, */
      if ( !morph2 )
         {
         /* move image into position for output */
            BlockMove( Out,PixSize,iX,iY,oX,oY,Ix,Iy,Out,BYTEPX,Ix,Iy,0,0 );
         }
      else /* if there is a second pass */
         {
         ScrollBuf( Buf2, iX, iY, PixSize, sZ, 0, 0, Out, iX, iY, PixSize );

         /* if not beyond input delay 2 go to begining of do-loop */
         if ( InCount < InDelay2 ) continue;

         /* Do the second pass morphop */
         morph2( Buf2,Out,iX,tX,tY,SE,sX,sY,sZ,sorgx,sorgy,sorgz,Opt );

         /* move image into position for output */
         BlockMove( Out,PixSize,iX,iY,oX,oY,Ix,Iy,Out,BYTEPX,Ix,Iy,0,0 );
         }

      ++NumOutReqs;

      /* call back to output the image and repeat the loop */
      if ( OutputImage( Out, NumOutReqs, OutStruct ) ) 
         {
         fprintf(stderr,"Morph3DSub: Output procedure returned error.\n");
         exit(0);
         }

      }   /* end of main do-loop */
   while ( TRUE );

   DeAllocBuf( Buf1, sZ );
   DeAllocBuf( Buf2, sZ );
   DeAllocBuf( In,  0 );
   DeAllocBuf( Out, 0 );

   return;
   }

