/* rasterio.c  --  routine to read and write Sun raster format images   */
/*                 (rasterfiles). These routines are limited to 8-bit   */
/*                 type RT_STANDARD rasterfiles.                        */
/* includes: GetRasHeader, GetRasColorMap, GetRasImage, ReadRasterFile, */
/*           WriteRasterFile, ExtractLuminance, OpenFile                */

/* version 2.0      13 Jan 1993 */

/* by Richard Alan Peters II                       */
/* Department of Electrical Engineering            */
/* Vanderbilt University School of Engineering     */ 
/* Nashville, TN 37235                             */ 
/* rap2@vuse.vanderbilt.edu                        */ 
/*                                                 */ 
/* This software is freely redistributable if      */ 
/* the author's name and affiliation are included. */

/* changes in version 2.0:                                         */ 
/*                                                                 */
/* (1) OpenFile now decompresses unix-compressed (.Z) rasterfiles. */
/*                                                                 */
/* (2) ReadRasterFile and WriteRasterFile have been rewritten to   */
/*     handle properly images with an odd number of pixels per row */
/*                                                                 */
/* (3) ExtractLuminance has been modified to round the results.    */
/*     In the previous version, truncation led to some erroneous   */
/*     conversions.                                                */

/* includes */

#include <ctype.h>
#include <malloc.h>			
#include <stdio.h>	
#include <stdlib.h>
#include <string.h>
#include "rasterio.h"



/* Get rasimage header */
/* this routine restricts the type of rasterfiles to 8-bit */
/* with or without colormap. These routines will not read  */
/* binary (1-bit) or color (24-bit) rasterfiles            */

int GetRasHeader( stream, RasHd )
   FILE *stream;              /* address of open ("r") input file  */
   struct rasterfile *RasHd;  /* address of existing struct */
   {

   if ( fread( RasHd, sizeof(int), 8, stream ) != 8 )
      {
      fprintf(stderr,"GetRasHeader: Error reading image header\n");
      return( ERROR );
      }

   if ( RasHd->ras_magic != RAS_MAGIC )
      {
      fprintf(stderr,"GetRasHeader: Input image must have sun rasterfile format\n");
      return( ERROR );
      }

   if ( RasHd->ras_depth != 8 )
      {
      fprintf(stderr,"GetRasHeader: Input image must be 8 bits/pixel\n");
      fprintf(stderr,"This one is %d bits/pixel\n",RasHd->ras_depth);
      return( ERROR );
      }

   if ( RasHd->ras_type != RT_STANDARD )
      {
      fprintf(stderr,"GetRasHeader: Input image must be type RT_STANDARD\n");
      return( ERROR );
      }

   return( NOERR );
   }



/* Allocate space for, and load, colormap if there is one */
/* Note, this routine allocates space for a full size (CMAPSZ) */
/* whenever RasHd->ras_maplength > 0. This permits the extract */
/* luminance routine to fill-out the cmap. */
/* If RasHd->ras_maplength == 0, return without allocating CMap */

int GetRasColorMap( stream, RasHd, CMap, Alloc )
   FILE *stream;              /* address of open ("r") input file  */
   struct rasterfile *RasHd;  /* address of existing struct */
   byte **CMap;               /* address of pointer (space allocated here) */
   int  Alloc;                /* T => allocate space */
   {

   if ( RasHd->ras_maplength )
      {
      if ( Alloc ) {
         if ( !(*CMap = (unsigned char *)calloc( sizeof(byte), CMAPSZ )) )
            {
            fprintf(stderr,"GetRasColorMap: Unable to allocate space for colormap\n");
            return( ERROR );
            } }
      if ( fread( *CMap, sizeof(byte), RasHd->ras_maplength, stream )
            != RasHd->ras_maplength )
         {
         fprintf(stderr,"GetRasColorMap: Error reading color map\n");
         return( ERROR );
         }
      }
   else
      if ( Alloc )  *CMap = NULL;

   return( NOERR );
   }



/* allocate memory for, and load, a raster image, with zero padding.        */
/* this assumes GetrRasHead and GetColorMap have already been run on stream */

int GetRasImage( stream, RasHd, Img, ulx, uly, lrx, lry, Alloc )
   FILE *stream;              /* address of open ("r") input file  */
   struct rasterfile *RasHd;  /* address of existing struct */
   byte **Img;                /* address of pointer (space allocated here) */
   int ulx, uly;              /* upper left  hand offset for zero padding */
   int lrx, lry;              /* lower right hand offset for zero padding */
   int  Alloc;                /* T => allocate space */
   {
   int X,Y;    /* image horiz., vert. dimensions with zero-padding */
   int wid;    /* image data-structure row width if picture width is odd */
   int i;      /* index */

   /* this field might be zero for some images, so compute it */
   if ( !RasHd->ras_length )
      {
      if ( RasHd->ras_width % 2 ) /* if there are an odd # pix / line */
         {
         RasHd->ras_length = (RasHd->ras_width+1) * RasHd->ras_height *
                          RasHd->ras_depth / 8;
         }
      else /* there are an even # of pixels / line */
         {
         RasHd->ras_length = RasHd->ras_width * RasHd->ras_height *
                             RasHd->ras_depth / 8;
         }
      }

   /* compute width, height with zero padding */
   X = RasHd->ras_width  + ulx + lrx;
   Y = RasHd->ras_height + uly + lry;

   /* if requested, allocate space for the image (incl. zero padding). */
   /* The "+1" is to assure that the fread below has enough room to    */
   /* write the final line if there are an odd # of pixels per row     */
   if ( Alloc ) 
      {
      if ( !(*Img = (byte *)calloc( (X*Y)+1, sizeof(byte) )) )
         {
         fprintf(stderr,"GetRasImg: Unable to allocate space for image\n");
         return( ERROR );
         } 
      }

   if ( !( ulx || uly || lrx || lry ) &&  /* There is no zero padding and */
         ( RasHd->ras_length == RasHd->ras_width * RasHd->ras_height ) )
         /* the length of the image data structure equals the number  */
         /* of pixels per line times the number of lines, then we can */
         /* can read the image in as one block */
      {
      if ( fread( *Img, sizeof(byte), RasHd->ras_length, stream ) 
           != RasHd->ras_length )
         {
         fprintf(stderr,"GetRasImg: Image is smaller than expected.\n");
         return( ERROR );
         }
      }
   else   /* there is zero padding and/or there are an odd # of pixels per */
          /* row padded to make the # pix / row even in the data structure */
      {
      wid = RasHd->ras_width;
      if ( wid % 2 )
         ++wid;
      if ( (wid * RasHd->ras_height) != RasHd->ras_length )
         {
         fprintf(stderr,"GetRasImg: stated width * height != length.\n");
         return( ERROR );
         }

      /* Sun's rasterfile standard demands that each image data-structure */
      /* row have an even number of bytes. Thus, if the visual image has  */
      /* an odd number of 1-byte pixels in a row, many programs append a  */
      /* empty byte to the end of each row before writing the file. This  */
      /* routine checks for that.  If RasHd->ras_width is odd, and if     */
      /* RasHd->ras_length = (RasHd->ras_width+1) * RasHd->ras_height),   */
      /* then this program advances RasHd->ras_width+1 pixels per line    */
      /* for for every line read out of the file, but transfers only      */
      /* RasHd->ras_width pixels per line to the internal "Img" data      */ 
      /* structure. The transfer does include, however, program specifed  */
      /* zero padding (ulx,uly,lrx,lry). The WriteRasterFile routine pads */
      /* the image rows on output if necessary.                           */

      for ( i=0; i<RasHd->ras_height; ++i )
         {
         if ( fread( *Img + (i+uly)*X + ulx, sizeof(byte), wid, stream ) 
              != wid )
            {
            fprintf(stderr,"GetRasImg: Error reading rasterfile.\n");
            return( ERROR );
            }
         }
      }
   return( NOERR );
   }




int ReadRasterFile( stream, RasHd, CMap, Img, ulx, uly, lrx, lry, Alloc )
   FILE *stream;              /* address of open ("r") input file  */
   struct rasterfile *RasHd;  /* address of existing struct */
   byte **CMap;               /* address of pointer (space allocated here) */
   byte **Img;                /* address of pointer (space allocated here) */
   int ulx, uly;              /* upper left  hand offset for zero padding */
   int lrx, lry;              /* lower right hand offset for zero padding */
   int  Alloc;                /* T => allocate space */
   {

   if ( GetRasHeader( stream, RasHd ) ) return( ERROR );
   if ( GetRasColorMap( stream, RasHd, CMap, Alloc ) ) return( ERROR );
   if ( GetRasImage( stream, RasHd, Img, ulx, uly, lrx, lry, Alloc ) ) 
      return( ERROR );

   return( NOERR );
   }



int WriteRasterFile( stream, RasHd, CMap, Img, ulx, uly, lrx, lry )
   FILE *stream;              /* address of open ("w") output file  */
   struct rasterfile *RasHd;  /* address of existing struct */
   byte *CMap;                /* address of existing color map (or NULL) */
   byte *Img;                 /* address of existing image array */
   int ulx, uly;              /* upper left  hand offset for zero padding */
   int lrx, lry;              /* lower right hand offset for zero padding */
   {
   byte *buf;  /* output buffer */
   int wid;    /* number of pixels per line in visible image */
   int X;      /* image horiz. dim. including zero padding */
   int i;      /* index */

   /* verify that we can handle this */
   if ( (RasHd->ras_type != RT_STANDARD) || (RasHd->ras_depth != 8) )
      {
      fprintf( stdout, "WriteRasterFile: can only write 8-bit RT_STANDARD images. \n");
      return( ERROR );
      }

   /* make sure the file is open */
   if ( !stream )
      {
      fprintf( stdout, "WriteRasterFile: Output stream / file not open\n");
      return( ERROR );
      }

   /* compute the data-structure length */
   if ( (RasHd->ras_width % 2) )
      {
      RasHd->ras_length = (RasHd->ras_width+1) * RasHd->ras_height;
      }
   else
      {
      RasHd->ras_length = RasHd->ras_width * RasHd->ras_height;
      }

   /* check for zero-length images */
   if ( !RasHd->ras_length ) 
      {
      fprintf(stderr,"WriteRasterFile: Image has zero length -- aborting\n");
      return( ERROR );
      }

   if ( fwrite( RasHd, sizeof(int), 8, stream ) != 8 )
      {
      fprintf(stderr,"WriteRasterFile: Error writing raster file header\n");
      return( ERROR );
      }
 
   if ( RasHd->ras_maplength )
      {
      if ( fwrite( CMap, sizeof(byte), RasHd->ras_maplength, stream )
            != RasHd->ras_maplength )
         {
         fprintf(stderr,"WriteRasterFile: Error writing raster file color map\n");
         return( ERROR );
         }
      }

   if ( !( ulx || uly || lrx || lry ) &&  /* There is no zero padding and */
         ( RasHd->ras_length == RasHd->ras_width * RasHd->ras_height ) )
         /* the length of the image data structure equals the number  */
         /* of pixels per line times the number of lines, then we can */
         /* can read the image in as one block */
      {
      if ( fwrite( Img, sizeof(byte), RasHd->ras_length, stream ) 
           != RasHd->ras_length )
         {
         fprintf(stderr,"WriteRasterFile: Error writing raster file image\n");
         return( ERROR );
         }
      }
   else   /* there is zero padding and/or there are an odd # of pixels per */
          /* row. In the latter case, this program will pad the image data */
          /* structure rows to make the # pix / row even in the file       */
      {
      wid = RasHd->ras_width;
      X = wid + ulx + lrx;
      if ( wid % 2 )  /* there are an odd number of pixels / line */
         {
         ++wid;
         if ( !(buf = (byte *)calloc( wid, sizeof( byte ) )) )
            {
            fprintf(stderr,"WriteRasterFile: Error allocating buf.\n");
            return( ERROR );
            }
         for ( i=0; i<RasHd->ras_height; ++i )
            {
            memcpy( buf, Img + (i+uly)*X + ulx, RasHd->ras_width );
            if ( fwrite( buf, sizeof(byte), wid, stream ) != wid )
               {
               fprintf(stderr,"WriteRasterFile: Error writing rasterfile.\n");
               return( ERROR );
               }
            }
         free( buf );
         }
      else /* there is zero padding and there are an even # of pixels / line */
         {
         for ( i=0; i<RasHd->ras_height; ++i )
            {
            if ( fwrite( Img+(i+uly)*X+ulx, sizeof(byte), wid, stream ) != wid )
               {
               fprintf(stderr,"WriteRasterFile: Error writing rasterfile.\n");
               return( ERROR );
               }
            }
         }
      }

   return( NOERR );
   }


/* extract the luminance component from a color mapped iamge */
/* This expects that the CMap array is CMAPSZ bytes long (as */
/* allocated by GetRasColorMap) even though */
/* RasHd->ras_maplength may be < CMAPSZ. The CMap output by this */
/* routine is always CMAPSZ bytes long. */


int ExtractLuminance( RasHd, CMap, Img, ulx, uly, lrx, lry )
   struct rasterfile *RasHd;  /* address of header   */
   byte *CMap;                /* address of colormap */
   byte *Img;                 /* address of image    */
   int ulx, uly;              /* upper left  hand offset for zero padding */
   int lrx, lry;              /* lower right hand offset for zero padding */
   {
   byte *R,*G,*B;             /* cmap component pointers */
   byte *I;                   /* image pointer  */
   byte pix;                  /* pixel value    */
   int i;                     /* index variable */
   int N;                     /* color map length */

   if ( CMap == NULL ) return( NOERR );  /* no cmap ==> already is grayscale */

   N = RasHd->ras_maplength / 3;

   R = CMap;
   G = R + N;
   B = G + N;


   /* remap the image */
   I = Img;
   for ( i=RasHd->ras_width*RasHd->ras_height; i; --i )
      {
      pix = (byte) 
            (0.3*(float)R[*I] + 0.59*(float)G[*I] + 0.11*(float)B[*I] + 0.5);
      *(I++) = pix;
      }


   /* remake the color map */

   I = CMap;
   N = CMAPSZ / 3;
   for ( i=0; i<N; ++i )
      {
      *I = i;
      *(I+N) = i;
      *(I+2*N) = i;
      ++I;
      }
   RasHd->ras_maplength = CMAPSZ;

   return( NOERR );
   }






FILE *OpenFile( name, envpre, opt )
   char *name;    /* file name */
   char *envpre;  /* name of environment variable containg path prefix */
   char *opt;     /* option string, either "r" or "w" */
   {
   int n;
   int compressed;
   char *prefix;
   char *suffix;
   char pathname[256];
   char compname[256];
   FILE *fp;

   pathname[0] = '\0';
   compname[0] = '\0';
   compressed  = 0;
   if ( envpre ) /* not null => there may be a pathname */
      {          /* prefix in this environment variable */
      prefix = getenv( envpre ); /* get the path prefix, if there is one. */
      if ( prefix && (n = strlen(prefix))  ) /* if there is a non null    */
         {                                   /* prefix, prepend it to     */
         strcpy(pathname,prefix);            /* the pathname.             */
         if ( pathname[n-1] != '/' )
            {
            pathname[n] = '/';
            pathname[n+1] = '\0';
            }
         }
      }
   strcat( pathname, name );

   if ( !(strcmp( opt , "r")) )  /* if this is an open-for-read */
      {
      if ( (suffix = strrchr(pathname, '.')) )/* find the loc. of the suffix */
         compressed = !strcmp(suffix, ".Z");  /* true if suffix == ".Z" */

      if ( !compressed )  /* if the supplied file name does not end */
         {                /* in ".Z", see if the real filename does */
         if ( !( fp = fopen( pathname, "r" )) ) /* try to open the file */
            {  
            sprintf(compname, "%s.Z", pathname );  /* if unsuccessful, try */
            if ( !( fp = fopen( compname, "r" )) ) /* pathname.Z */
               {                             /* if thet doesn't work, quit */
               fprintf( stderr, "OpenFile: Unable to open %s\n", pathname );
               exit( 0 );
               }
            else  /* actual filename has a ".Z" on the end */
               {
               compressed = TRUE;
               strcpy( pathname, compname );
               close( fp );
               }
            }
         }

      if ( compressed ) /* open and uncompress */
         {
	     sprintf(compname, "zcat %s", pathname );
         if ( !(fp = popen( compname, "r" )) )
            {
            fprintf( stderr, "Cannot uncompress %s.\n",pathname);
            exit(0);
            }
         }
      }     /* end of open-for-read */
   else     /* open-for-write */
      {
    if ( !( fp = fopen( pathname, "w" )) )
         {
         fprintf( stderr, "OpenFile: Unable to open %s\n", pathname );
         exit( 0 );
         }
      }


   return ( fp );
   }

