/*
 *	Author:  Christopher G. Phillips
 *	Copyright (C) 1993 All Rights Reserved
 *
 *	NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * The author makes no representations about the suitability of this
 * software for any purpose.  This software is provided ``as is''
 * without express or implied warranty.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "pvm3.h"

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 100
#endif

static int	*tids;
static int	nodes;
static int	me;
static int	parent;
extern int	pvm_errno;
static char	*program_name;

static void
usage(void)
{
	fprintf(stderr, "Usage: %s nodes ...\n", program_name);
	exit(1);
}

static void
say_hello(void)
{
	char	hostname[MAXHOSTNAMELEN];

	fflush(stdout);
	pvm_setiomode(fileno(stdout), PvmIomodeCommon);
	pvm_setiomode(fileno(stderr), PvmIomodeCommon);
	if (gethostname(hostname, sizeof hostname) == -1)
		strcpy(hostname, "Unknown");
	printf("Hello world from node %d running on %s!\n", me, hostname);
	fflush(stdout);
}

static void
shutdown(int num_spawned, char *s)
{
	int		i;
	extern int	pvmmyfstid;

	pvmmyfstid = 0;
	if (s)
		pvm_perror(s);
	else if (num_spawned == 0)
		pvm_perror("pvm_spawn");

	for (i = 1; i < num_spawned; i++)
		if (tids[i] < 0) {
			pvm_errno = tids[i];
			pvm_perror("pvm_spawn");
		} else if (pvm_kill(tids[i]) < 0)
			pvm_perror("pvm_kill");

	exit(1);
}

#define GSYNC_TYPE	0x11000000

void
gsync(void)
{
	int	i;

	if (nodes > 1) {
		if (me == 0) {
			for (i = 1; i < nodes; i++)
				pvm_recv(-1, GSYNC_TYPE);
			pvm_initsend(PvmDataDefault);
			pvm_mcast(&tids[1], nodes - 1, GSYNC_TYPE + 1);
		} else {
			pvm_initsend(PvmDataDefault);
			pvm_send(parent, GSYNC_TYPE);
			pvm_recv(parent, GSYNC_TYPE + 1);
		}
	}
}

int
startup(int *argcp, char **argv)
{
	int		mytid;
	char		*slash;
	int		argc = *argcp;

	pvm_usefs(1);

	if (slash = strrchr(argv[0], '/'))
		argv[0] = slash + 1;
	program_name = argv[0];

	if ((mytid = pvm_mytid()) < 0) {
		pvm_perror("pvm_mytid");
		exit(1);
	}

	if ((parent = pvm_parent()) < 0) {
		if (argc < 2)
			usage();

		if ((nodes = atoi(argv[1])) <= 0)
			usage();
		if (nodes > 1) {
			int		done = 1;
			int		i;
			int		j;
			int		nhost;
			int		this_host = pvm_tidtohost(mytid);
			struct hostinfo	*hip;

			if ((tids = malloc(nodes * sizeof(*tids))) == NULL) {
				perror(argv[0]);
				exit(1);
			}
			tids[0] = mytid;

			if (pvm_config(&nhost, &j, &hip) < 0) {
				pvm_perror("pvm_config");
				exit(1);
			}

			for (i = 0; i < nhost; i++) {
				j = nodes / nhost; 
				if (i < nodes % nhost)
					j++;
				if (hip[i].hi_tid == this_host)
					j--;
				if (j <= 0)
					continue;

				if (pvm_spawn(argv[0], argv + 2,
 				  PvmTaskHost, hip[i].hi_name, j, &tids[done])
				  != j)
					shutdown(done, NULL);

				done += j;
			}

			for (i = 1; i < argc; i++)
				argv[i] = argv[i + 1];
			(*argcp)--;

			if (pvm_packf("%+ %d %*d", PvmDataDefault, nodes, nodes,
			  tids) < 0)
				shutdown(done, "pvm_packf");
			if (pvm_mcast(&tids[1], nodes - 1, 32767) < 0)
				shutdown(done, "pvm_mcast");
		}
		me = 0;
	} else {
		if (pvm_recv(parent, 32767) < 0)
			shutdown(0, "pvm_recv");
		if (pvm_unpackf("%d", &nodes) < 0)
			shutdown(0, "pvm_unpackf");

		if ((tids = malloc(nodes * sizeof(*tids))) == NULL) {
			perror("malloc");
			exit(1);
		}
		if (pvm_unpackf("%*d", nodes, tids) < 0)
			shutdown(0, "pvm_unpackf");

		for (me = nodes; me > 0; me--)
			if (tids[me] == mytid)
				break;
		/*assert(me);*/
	}

	say_hello();

	gsync();

	return me;
}
