/*
 *                 Author:  Christopher G. Phillips
 *              Copyright (C) 1993 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * The author makes no representations about the suitability of this
 * software for any purpose.  This software is provided ``as is''
 * without express or implied warranty.
 */

#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/uio.h>
#include <sys/select.h>
#include "syscall-fake.h"

#ifdef TRACE
char	hex[] = "0123456789abcdef";

/*
 * A minimal sprintf...
 */
int
minsprintf(char *s, const char *fmt, ...)
{
        va_list		ap;
        int		i;
        int		num;
	unsigned long	ul;
	int		long_variety;
	long		l;
        char		*begin = s;
        char		*cp;

        va_start(ap, fmt);

        fmt--;

        while (*++fmt) {
                if (*fmt == '%') {
			long_variety = 0;
			if (*++fmt == 'l')
				long_variety = 1;
			else
				fmt--;
                        switch (*++fmt) {
                        case 'p':
                                ul = va_arg(ap, void *);
                                for (cp = s; ul >= 16; ul /= 16)
                                        *s++ = hex[ul % 16];
                                *s++ = hex[ul];
                                num = s - cp;
                                for (ul = 0; ul < num / 2; ul++) {
                                        char    tmp = cp[ul];
                                        cp[ul] = cp[num - ul - 1];
                                        cp[num - ul - 1] = tmp;
                                }
                                break;
                        case 'd':
				if (long_variety)
					l = va_arg(ap, long);
				else
	                                l = (long)va_arg(ap, int);
                                if (l < 0) {
                                        *s++ = '-';
                                        l = -l;
                                }
                                for (cp = s; l >= 10; l /= 10)
                                        *s++ = (l % 10) + '0';
                                *s++ = l + '0';
                                num = s - cp;
                                for (l = 0; l < num / 2; l++) {
                                        char    tmp = cp[l];
                                        cp[l] = cp[num - l - 1];
                                        cp[num - l - 1] = tmp;
                                }
                                break;
                        case 's':
                                cp = va_arg(ap, char *);
                                cp--; s--;
                                while (*++s = *++cp)
                                        ;
                                break;
                        }
                } else if (*fmt == '\\')
                        *s++ = *++fmt;
                else
                        *s++ = *fmt;
        }
        return s - begin;
}

#define dprint(a)	\
{	\
	struct iovec	iovec;	\
	char		buf3[1000];	\
	int		n;	\
	\
	n = minsprintf a;	\
	iovec.iov_base = buf3;	\
	iovec.iov_len = n;	\
	kwritev(2, &iovec, 1, 1);	\
}
#else
#define dprint(a)
#endif

int
syscall(int num, ...)
{
	va_list		ap;
	int		retval;
	int		i1, i2, i3;
	int		*ip;
	const char	*ccp1, *ccp2;
	char		*cp1, *cp2;
	const void	*vp1;
	struct iovec	*iovp1;
	struct iovec	iovec;
	struct flock	*flp;
	struct stat	*stp1;
	struct timeval	*tvp;
	struct rlimit	*rlp;
	short		*sp1;
	size_t		s1;
	unsigned	u1;
	off_t		off1, off2;
	dev_t		dev1;
	uid_t		uid1;
	gid_t		gid1;
	mode_t		mode1;
	fd_set		*fdsp1, *fdsp2, *fdsp3;

	va_start(ap, num);

	switch (num) {
	case SYS_access:
		cp1 = va_arg(ap, char *);
		i1 = va_arg(ap, int);
		retval = access(cp1, i1);
		dprint((buf3, "syscall(SYS_access, %p, %d) = %d\n",
		  cp1, i1, retval));
		break;
	case SYS_accessx:
		cp1 = va_arg(ap, char *);
		i1 = va_arg(ap, int);
		i2 = va_arg(ap, int);
		retval = accessx(cp1, i1, i2);
		dprint((buf3, "syscall(SYS_accessx, %p, %d, %d) = %d\n",
		  cp1, i1, i2, retval));
		break;
	case SYS_chdir:
		cp1 = va_arg(ap, char *);
		retval = chdir(cp1);
		dprint((buf3, "syscall(SYS_chdir, %p) = %d\n", cp1, retval));
		break;
	case SYS_chmod:
		cp1 = va_arg(ap, char *);
		mode1 = va_arg(ap, mode_t);
		retval = chmod(cp1, mode1);
		break;
	case SYS_chown:
		cp1 = va_arg(ap, char *);
		uid1 = va_arg(ap, uid_t);
		gid1 = va_arg(ap, gid_t);
		retval = chown(cp1, uid1, gid1);
		break;
	case SYS_chownx:
		cp1 = va_arg(ap, char *);
		uid1 = va_arg(ap, uid_t);
		gid1 = va_arg(ap, gid_t);
		i1 = va_arg(ap, int);
		retval = chownx(cp1, uid1, gid1, i1);
		break;
	case SYS_chroot:
		cp1 = va_arg(ap, char *);
		retval = chroot(cp1);
		break;
	case SYS_close:
		i1 = va_arg(ap, int);
		retval = close(i1);
		dprint((buf3, "syscall(SYS_close, %d) = %d\n", i1, retval));
		break;
	case SYS_creat:
		cp1 = va_arg(ap, char *);
		mode1 = va_arg(ap, mode_t);
		retval = creat(cp1, mode1);
		dprint((buf3, "syscall(SYS_creat, \"%s\", %ld, %d) = %d\n",
		  cp1, (long)mode1, retval));
		break;
	case SYS_dup:
		i1 = va_arg(ap, int);
		retval = kfcntl(i1, F_DUPFD, 0);
		dprint((buf3, "syscall(SYS_dup, %d) = %d\n", i1, retval));
		break;
	case SYS_dup2:
		i1 = va_arg(ap, int);
		i2 = va_arg(ap, int);
		/* Should make sure i1 is valid, i2 <= OPEN_MAX, ... */
		close(i2);
		retval = kfcntl(i1, F_DUPFD, i2);
		dprint((buf3, "syscall(SYS_dup2, %d, %d) = %d\n",
		  i1, i2, retval));
		break;
	case SYS_faccessx:
		i1 = va_arg(ap, int);
		i2 = va_arg(ap, int);
		i3 = va_arg(ap, int);
		retval = faccessx(i1, i2, i3);
		break;
	case SYS_fchdir:
		retval = -1;
		errno = ENOSYS;
		break;
	case SYS_fchmod:
		i1 = va_arg(ap, int);
		mode1 = va_arg(ap, mode_t);
		retval = fchmod(i1, mode1);
		break;
	case SYS_fchown:
		i1 = va_arg(ap, int);
		uid1 = va_arg(ap, uid_t);
		gid1 = va_arg(ap, gid_t);
		retval = fchown(i1, uid1, gid1);
		break;
	case SYS_fchownx:
		i1 = va_arg(ap, int);
		uid1 = va_arg(ap, uid_t);
		gid1 = va_arg(ap, gid_t);
		i2 = va_arg(ap, int);
		retval = fchownx(i1, uid1, gid1, i2);
		break;
	case SYS_fcntl:
	case SYS_kfcntl:
		i1 = va_arg(ap, int);
		i2 = va_arg(ap, int);
		if (i2 == F_GETLK || F_SETLK || F_SETLKW) {
			flp = va_arg(ap, struct flock *);
			retval = kfcntl(i1, i2, flp);
		} else {
			i3 = va_arg(ap, int);
			retval = kfcntl(i1, i2, i3);
		}
		break;
	case SYS_fstat:
		i1 = va_arg(ap, int);
		stp1 = va_arg(ap, struct stat *);
		retval = fstatx(i1, stp1, STATSIZE, STX_NORMAL);
		dprint((buf3, "syscall(SYS_fstat, %d, %p) = %d\n",
		  i1, stp1, retval));
		break;
	case SYS_fstatx:
		i1 = va_arg(ap, int);
		stp1 = va_arg(ap, struct stat *);
		i1 = va_arg(ap, int);
		i2 = va_arg(ap, int);
		retval = fstatx(i1, stp1, i1, i2);
		break;
	case SYS_fsync:
		i1 = va_arg(ap, int);
		retval = fsync(i1);
		break;
	case SYS_ftruncate:
		i1 = va_arg(ap, int);
		off1 = va_arg(ap, off_t);
		retval = ftruncate(i1, off1);
		break;
	case SYS_getrlimit:
		i1 = va_arg(ap, int);
		rlp = va_arg(ap, struct rlimit *);
		retval = getrlimit(i1, rlp);
#if 0
		dprint((buf3,
		  "syscall(SYS_getrlimit, %d, %p) = %d <%lu, %lu>\n", i1,
		  (void *)rlp, retval,
		  (unsigned long)rlp->rlim_cur, (unsigned long)rlp->rlim_max));
#endif
		break;
	case SYS_kreadv:
		i1 = va_arg(ap, int);
		iovp1 = va_arg(ap, struct iovec *);
		i2 = va_arg(ap, int);
		i3 = va_arg(ap, int);
		retval = kreadv(i1, iovp1, i2, i3);
		dprint((buf3,
		  "syscall(SYS_kreadv, %d, <%p, %d>, %d, %d) = %d\n",
		  i1, iovp1->iov_base, iovp1->iov_len, i2, i3, retval));
		break;
	case SYS_kwritev:
		i1 = va_arg(ap, int);
		iovp1 = va_arg(ap, struct iovec *);
		i2 = va_arg(ap, int);
		i3 = va_arg(ap, int);
		retval = kwritev(i1, iovp1, i2, i3);
		dprint((buf3,
		  "syscall(SYS_kwritev, %d, <%p, %d>, %d, %d) = %d\n",
		  i1, iovp1->iov_base, iovp1->iov_len, i2, i3, retval));
		break;
	case SYS_link:
		cp1 = va_arg(ap, char *);
		cp2 = va_arg(ap, char *);
		retval = link(cp1, cp2);
		break;
	case SYS_lseek:
		i1 = va_arg(ap, int);
		off1 = va_arg(ap, off_t);
		i2 = va_arg(ap, int);
		retval = lseek(i1, off1, i2);
		break;
	case SYS_lstat:
		cp1 = va_arg(ap, char *);
		stp1 = va_arg(ap, struct stat *);
		retval = statx(cp1, stp1, STATSIZE, STX_LINK);
		break;
	case SYS_mkdir:
		cp1 = va_arg(ap, char *);
		mode1 = va_arg(ap, mode_t);
		retval = mkdir(cp1, mode1);
		break;
	case SYS_mknod:
		ccp1 = va_arg(ap, const char *);
		i1 = va_arg(ap, int);
		dev1 = va_arg(ap, dev_t);
		retval = mknod(ccp1, i1, dev1);
		break;
	case SYS_open:
		cp1 = va_arg(ap, char *);
		i1 = va_arg(ap, int);
		if (i1 & O_CREAT) {
			mode1 = va_arg(ap, int);
			retval = open(cp1, i1, mode1);
			dprint((buf3,
			  "syscall(SYS_open, \"%s\", %d, %d) = %d\n",
			    cp1, i1, mode1, retval));
		} else {
			retval = open(cp1, i1);
			dprint((buf3, "syscall(SYS_open, \"%s\", %d) = %d\n",
			  cp1, i1, retval));
		}
		break;
	case SYS_openx:
		cp1 = va_arg(ap, char *);
		i1 = va_arg(ap, int);
		i2 = va_arg(ap, int);
		i3 = va_arg(ap, int);
		retval = openx(cp1, i1, i2, i3);
		break;
	case SYS_read:
		i1 = va_arg(ap, int);
		cp1 = va_arg(ap, char *);
		u1 = va_arg(ap, unsigned);
		iovec.iov_base = cp1;
		iovec.iov_len = u1;
		retval = kreadv(i1, &iovec, 1, 1);
		dprint((buf3, "syscall(SYS_read, %d, %p, %d) = %d\n",
		  i1, cp1, u1, retval));
		break;
	case SYS_readx:
		i1 = va_arg(ap, int);
		cp1 = va_arg(ap, char *);
		u1 = va_arg(ap, unsigned);
		i2 = va_arg(ap, int);
		iovec.iov_base = cp1;
		iovec.iov_len = u1;
		retval = kreadv(i1, &iovec, 1, i2);
		break;
	case SYS_readv:
		i1 = va_arg(ap, int);
		iovp1 = va_arg(ap, struct iovec *);
		i2 = va_arg(ap, int);
		retval = kreadv(i1, &iovec, i2, 1);
		break;
	case SYS_readvx:
		i1 = va_arg(ap, int);
		iovp1 = va_arg(ap, struct iovec *);
		i2 = va_arg(ap, int);
		i3 = va_arg(ap, int);
		retval = kreadv(i1, iovp1, i2, i3);
		break;
	case SYS_readlink:
		ccp1 = va_arg(ap, const char *);
		cp1 = va_arg(ap, char *);
		i1 = va_arg(ap, int);
		retval = readlink(ccp1, cp1, i1);
		break;
	case SYS_rename:
		ccp1 = va_arg(ap, const char *);
		ccp2 = va_arg(ap, const char *);
		retval = rename(ccp1, ccp2);
		break;
	case SYS_rmdir:
		cp1 = va_arg(ap, char *);
		retval = rmdir(cp1);
		break;
	case SYS_setrlimit:
		i1 = va_arg(ap, int);
		rlp = va_arg(ap, struct rlimit *);
		retval = setrlimit(i1, rlp);
#if 0
		dprint((buf3,
		  "syscall(SYS_setrlimit, %d, <%lu, %lu>) = %d\n", i1,
		  (unsigned long)rlp->rlim_cur, (unsigned long)rlp->rlim_max,
		  retval));
#endif
		break;
	case SYS_stat:
		cp1 = va_arg(ap, char *);
		stp1 = va_arg(ap, struct stat *);
		retval = statx(cp1, stp1, STATSIZE, STX_NORMAL);
		dprint((buf3, "syscall(SYS_stat, \"%s\", %p) = %d\n",
		  cp1, stp1, retval));
		break;
	case SYS_statx:
		cp1 = va_arg(ap, char *);
		stp1 = va_arg(ap, struct stat *);
		i1 = va_arg(ap, int);
		i2 = va_arg(ap, int);
		retval = statx(cp1, stp1, i1, i2);
		break;
	case SYS_symlink:
		ccp1 = va_arg(ap, const char *);
		ccp2 = va_arg(ap, const char *);
		retval = symlink(ccp1, ccp2);
		break;
	case SYS_sync:
		retval = sync();
		break;
	case SYS_truncate:
		ccp1 = va_arg(ap, const char *);
		off1 = va_arg(ap, off_t);
		retval = truncate(ccp1, off1);
		break;
	case SYS_umask:
		mode1 = va_arg(ap, mode_t);
		retval = umask(mode1);
		break;
	case SYS_unlink:
		cp1 = va_arg(ap, char *);
		retval = unlink(cp1);
		break;
	case SYS_utimes:
		ccp1 = va_arg(ap, const char *);
		tvp = va_arg(ap, struct timeval *);
		retval = utimes(mode1, tvp);
		break;
	case SYS_write:
		i1 = va_arg(ap, int);
		cp1 = va_arg(ap, char *);
		u1 = va_arg(ap, unsigned);
		iovec.iov_base = cp1;
		iovec.iov_len = u1;
		retval = kwritev(i1, &iovec, 1, 1);
		dprint((buf3, "syscall(SYS_write, %d, %p, %d) = %d\n",
		  i1, cp1, u1, retval));
		break;
	case SYS_writex:
		i1 = va_arg(ap, int);
		cp1 = va_arg(ap, char *);
		u1 = va_arg(ap, unsigned);
		i2 = va_arg(ap, int);
		iovec.iov_base = cp1;
		iovec.iov_len = u1;
		retval = kwritev(i1, &iovec, 1, i2);
		break;
	case SYS_writev:
		i1 = va_arg(ap, int);
		iovp1 = va_arg(ap, struct iovec *);
		i2 = va_arg(ap, int);
		retval = kwritev(i1, iovp1, i2, 1);
		break;
	case SYS_writevx:
		i1 = va_arg(ap, int);
		iovp1 = va_arg(ap, struct iovec *);
		i2 = va_arg(ap, int);
		i3 = va_arg(ap, int);
		retval = kwritev(i1, iovp1, i2, i3);
		break;
	case SYS_select:
		i1 = va_arg(ap, int);
		fdsp1 = va_arg(ap, fd_set *);
		fdsp2 = va_arg(ap, fd_set *);
		fdsp3 = va_arg(ap, fd_set *);
		tvp = va_arg(ap, struct timeval *);
		retval = select(i1, fdsp1, fdsp2, fdsp3, tvp);
		break;
	case SYS_pipe:
		ip = va_arg(ap, int *);
		retval = pipe(ip);
		break;
	case SYS_ioctl:
	case SYS_kioctl:
	default:
		retval = -1;
		break;
	}

	va_end(ap);

	return retval;
}
