/*
 *                 Author:  Christopher G. Phillips
 *              Copyright (C) 1993 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * The author makes no representations about the suitability of this
 * software for any purpose.  This software is provided ``as is''
 * without express or implied warranty.
 */

/*
 * This implementation of malloc and friends assumes that all of
 * the memory we give out is in a static block.  Because of this,
 * we can keep a sorted doubly-linked list of free blocks.
 * (Other than perhaps using pointer comparison and checking for the
 * presence of the highest address in the free list, everything
 * should be ok if another mechanism is used to add memory to
 * the free list.)
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>

#define N		1000000

typedef union ugh {
	struct {
		size_t		_nughs;
		union ugh	*_prev;
		union ugh	*_next;
	}	h;
	double	d;
	long	l;
} UGH;

#define size	h._nughs
#define prev	h._prev
#define next	h._next

#define PATTERN		((void *)0xa5a5a5a5)
#define SETUSED(p)	{ (p)->prev = NULL; (p)->next = PATTERN; }
#define ISFREE(p)	((p)->prev != NULL && (p)->next != PATTERN)

static UGH	space[N];
static UGH	*freelist;
static UGH	*lowest;
static int	init = 0;

static void
initialize(UGH *up, size_t upsize)
{
	/*
	 * Free list only contains this one block.
	 */
	up->size = upsize;
	lowest = freelist = up->next = up->prev = up;
}

static void
delete(UGH *ptr)
{
	/*
	 * Remove a block from the free list.
	 * Adjust its neighbors' pointers.
	 */
	if (ptr->next == ptr)
		lowest = freelist = NULL;
	else {
		freelist = ptr->prev->next = ptr->next;
		ptr->next->prev = ptr->prev;
		if (ptr == lowest)
			if (ptr->prev < ptr)
				lowest = ptr->prev;
			else
				lowest = ptr->next;
	}
}

static void
add(UGH *ptr, UGH *left, UGH *right)
{
	/*
	 * Insert block ``ptr'' between blocks ``left'' and ``right''.
	 */

	/*
	 * We know that right == left->next
	 * and right->prev == left.
	 */
	ptr->next = right;
	ptr->prev = left;
	left->next = right->prev = ptr;
	if (ptr < left)
		lowest = ptr;
}

static void
combine(UGH *ptr)
{
	 UGH	*higher = ptr + ptr->size + 1;

	 /*
	  * Have ``ptr'' swallow up ``higher''.
	  */
	 ptr->size += higher->size + 1;
	 ptr->next = higher->next;
	 higher->next->prev = ptr;
	 if (freelist == higher)
		freelist = ptr;
}

void *
malloc(size_t nbytes)
{
	UGH	*ptr;
	size_t	nughs;

	if (!init) {
		initialize(space, N - 1);
		init = 1;
	}

	if (!nbytes)
		return NULL;

	nughs = (nbytes + sizeof(UGH) - 1) / sizeof(UGH);

	if (ptr = freelist)
		do {
			if (ptr->size >= nughs) {
				if (ptr->size > nughs + 1) {
					/*
					 * Split the block and
					 * return the top part.
					 */
					ptr->size -= nughs + 1;
					ptr += ptr->size + 1;
					ptr->size = nughs;
				} else
					delete(ptr);
				SETUSED(ptr);
				return ptr + 1;
			}
			ptr = ptr->next;
		} while (ptr != freelist);

	return NULL;
}

static size_t
min3(size_t s1, size_t s2, size_t s3)
{
	size_t	min;

	min = s1 < s2 ? s1 : s2;
	return s3 < min ? s3 : min;
}

static UGH *
findleft(UGH *target, UGH *base)
{
	/*
	 * Starting from ``base'', determine the
	 * first free block that is left of ``target''.
	 */
	if (base < target)
		while (base->next  < target)
			base = base->next;
	else
		do
			base = base->prev;
		while (base > target);
	return base;
}

void
free(void *p)
{
	UGH	*up;
	UGH	*left, *right;
	size_t	diff1, diff2, diff3;
	size_t	minimum;

	if (!p)
		return;

	up = (UGH *)p - 1;
	if (ISFREE(up))
		return;

	if (freelist) {
		/*
		 * Determine the free blocks ``left'' and ``right''
		 * that ``up'' will go between.
		 */
		if (up < lowest) {
			left = NULL;
			right = lowest;
		} else if (up > lowest->prev) {
			left = lowest->prev;
			right = NULL;
		} else {
			/*
			 * We try to start from the somewhere close
			 * to ``up'' that we know is on the free list.
			 */
			diff1 = up - lowest;
			diff2 = lowest->prev - up;
			if (freelist > up)
				diff3 = freelist - up;
			else
				diff3 = up - freelist;
			if ((minimum = min3(diff1, diff2, diff3)) == diff1)
				left = findleft(up, lowest);
			else if (minimum == diff2)
				left = findleft(up, lowest->next);
			else
				left = findleft(up, freelist);
			right = left->next;
		}

		/*
		 * Insert ``up''.  Adjust prev and next pointers
		 * ignoring whether the blocks can be combined.
		 */
		add(up, left ? left : right->prev, right ? right : left->next);

		/*
		 * Check if ``left'' and ``right'' are contiguous
		 * to ``up''.  Update ``freelist''.
		 */
		if (right && up + up->size + 1 == right)
			combine(up);
		freelist = up;
		if (left && left + left->size + 1 == up) {
			freelist = left;
			combine(left);
		}
	} else
		initialize(up, up->size);
}

void *
realloc(void *p, size_t nbytes)
{
	UGH	*up;
	UGH	*higher;
	void	*vp;
	size_t	nughs;

	if (!init) {
		initialize(space, N - 1);
		init = 1;
	}

	if (!p)
		return malloc(nbytes);
	if (!nbytes) {
		free(p);
		return NULL;
	}

	up = (UGH *)p - 1;
	if (ISFREE(up))
		return NULL;
	nughs = (nbytes + sizeof(UGH) - 1) / sizeof(UGH);
	if (up->size >= nughs)
		return p;
	else if (lowest && (higher = up + up->size + 1) <= lowest->prev
	  && ISFREE(higher) && up->size + higher->size + 1 >= nughs) {
		/*
		 * If we get here, then the neighbor contigous to ``up''
		 * to the right is free and will give us the space we need.
		 */
		up->size += higher->size + 1;
		delete(higher);
		if (up->size > nughs + 1) {
			UGH	*newhigher;

			/*
			 * Split the ``higher'' block
			 * and put it on the free list.
			 */
			newhigher = up + nughs + 1;
			newhigher->size = up->size - nughs - 1;
			SETUSED(newhigher);
			free(newhigher + 1);
			up->size = nughs;
		}
		return p;
	} else if ((vp = malloc(nbytes)) == NULL)
		return NULL;

		/*
		 * Perhaps just above here, if malloc fails,
		 * we should check if the neighbor contiguous to ``up''
		 * to the left is free and would give us enough space.
		 */

	/*
	 * Move the data to the new location.
	 */
	memcpy(vp, p, up->size * sizeof(UGH));
	free(p);
	return vp;
}

void *
calloc(size_t s1, size_t s2)
{
	size_t	nbytes = s1 * s2;
	void	*p = malloc(nbytes);

	if (p)
		memset(p, '\0', nbytes);

	return p;
}
