/*
 *                 Author:  Christopher G. Phillips
 *              Copyright (C) 1993 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * The author makes no representations about the suitability of this
 * software for any purpose.  This software is provided ``as is''
 * without express or implied warranty.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <dirent.h>
#include <termios.h>
#include "pvm3.h"
#include "fs.h"
#include "fsp.h"
#ifndef IMA_RS6K
#include <syscall.h>
#include <sys/mman.h>
#ifndef IMA_SUN4
#include <sys/ioctl.h>
#endif
#else
#include "syscall-fake.h"
#endif

#ifdef TRACE
/*
 * Debugging and trace stuff.
 */
extern int	pvmdebug;

#define dprint(s)	do {	\
				if (pvmdebug) {	\
					sprintf s;	\
					syscall(SYS_write, 2, buf2, \
					  strlen(buf2)); \
				}	\
			} while (0)
#define dprintl(s)	dprint(s)

static char buf2[100];

#else
#define dprint(s)
#define dprintl(s)
#endif

extern int	lpvmerr __ProtoGlarp__((const char *, int));
extern int	pvmmytid;
extern int	pvmmyfstid;
extern int	errno;

extern int	pvm_local_errno __ProtoGlarp__((int));
extern int	pvm_tcflow_action __ProtoGlarp__((int));
extern int	pvm_tcflush_queue __ProtoGlarp__((int));
extern int	pvm_tcsetattr_opt __ProtoGlarp__((int));
extern long	pvm_local_tcattr_iflag __ProtoGlarp__((long));
extern long	pvm_local_tcattr_oflag __ProtoGlarp__((long));
extern long	pvm_local_tcattr_cflag __ProtoGlarp__((long));
extern long	pvm_local_tcattr_lflag __ProtoGlarp__((long));
extern long	pvm_local_tcattr_ispeed __ProtoGlarp__((long));
extern long	pvm_local_tcattr_ospeed __ProtoGlarp__((long));
extern long	pvm_tcattr_iflag __ProtoGlarp__((long));
extern long	pvm_tcattr_oflag __ProtoGlarp__((long));
extern long	pvm_tcattr_cflag __ProtoGlarp__((long));
extern long	pvm_tcattr_lflag __ProtoGlarp__((long));
extern long	pvm_tcattr_ispeed __ProtoGlarp__((long));
extern long	pvm_tcattr_ospeed __ProtoGlarp__((long));
extern int	pvm_pathconf_name __ProtoGlarp__((int));

/*
 * Local POSIX directory stuff.
 * We map pointers to integers.
 */
struct dirmap {
	int		dindex;
	struct dirent	dirent;
};

static struct dirmap	*dirmap = NULL;
static int		numdirs = 0;

DIR *
opendir(const char *pathname)
{
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_OPENDIR;
	int	nargs = 1;
	int	i;

	if (pvmmyfstid <= 0) {
		lpvmerr("opendir", PvmNoFileServer);
		errno = ENOSYS;
		return NULL;
	}

	for (i = 0; i < numdirs; i++)
		if (dirmap[i].dindex == -1)
			break;
	if (i == numdirs) {
		struct dirmap	*tmp;

#if defined(IMA_SUN3) || defined(IMA_SUN4)
		if (dirmap == NULL) {
			if ((tmp = malloc(sizeof(*dirmap))) == NULL)
				return NULL;
		} else
#endif
		if ((tmp = realloc(dirmap, (numdirs + 1) * sizeof(*dirmap)))
		  == NULL)
			return NULL;
		numdirs++;
		dirmap = tmp;
	}
	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s", PvmDataDefault, nargs, pathname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &dirmap[i].dindex);
	if (dirmap[i].dindex == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x opendir(\"%s\") = NULL (pvmerrno = %d)\n", pvmmytid,
		  pathname, errno));
	} else
		dprint((buf2, "tid t%x opendir(\"%s\") = ok (index %d)\n",
		  pvmmytid, pathname, i));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return (DIR *)(dirmap[i].dindex + 1);
}

struct dirent *
readdir(DIR *dp)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	dindex;
	int	request = PVMFS_READDIR;
	int	nargs = 1;
	int	saved_errno = errno;

	if (pvmmyfstid <= 0) {
		lpvmerr("readdir", PvmNoFileServer);
		errno = ENOSYS;
		return NULL;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	dindex = (unsigned int)dp - 1;
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, dindex);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		/*
		 * If we get pvmerrno == -2, we've
		 * reached the end of the directory.
		 */
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		if (errno > 0)
			dprint((buf2,
			  "tid t%x readdir(index %d) = NULL (errno = %d)\n",
			  pvmmytid, (int)dp, errno));
		else
			dprint((buf2, "tid t%x readdir(index %d) = NULL\n",
			  pvmmytid, (int)dp));
	} else {
		pvm_unpackf("%s", dirmap[(unsigned int)dp - 1].dirent.d_name);
		dprint((buf2, "tid t%x readdir(index %d) = ok\n",
		  pvmmytid, (int)dp));
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	if (cc == -1)
		if (pvmerrno < 0)
			errno = saved_errno;
		else
			errno = pvm_local_errno(pvmerrno);

	return (cc == -1) ? NULL : &dirmap[(unsigned int)dp - 1].dirent;
}

int
closedir(DIR *dp)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	dindex;
	int	request = PVMFS_CLOSEDIR;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		lpvmerr("closedir", PvmNoFileServer);
		errno = ENOSYS;
		return -1;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	dindex = (int)dp - 1;
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, dindex);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x closedir(index %d) = -1 (errno = %d)\n",
		  pvmmytid, (int)dp, errno));
	} else
		dprint((buf2, "tid t%x closedir(index %d) = %d\n",
		  pvmmytid, (int)dp, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

/*
 * Not sure if this trick really helps...
 */
#ifdef rewinddir
#define rewinddir_is_defined
static void
REWINDDIR(DIR *dp)
{
	rewinddir(dp);
}
#undef rewinddir
#endif

void
rewinddir(DIR *dp)
{
	int	cc;
	int	rbuf;
	int	sbuf;
	int	dindex;
	int	request = PVMFS_REWINDDIR;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
#ifdef rewinddir_is_defined
		REWINDDIR(dp);
#else
		lpvmerr("rewinddir", PvmNoFileServer);
		errno = ENOSYS;
#endif
		return;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	dindex = (int)dp - 1;
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, dindex);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1)
		dprint((buf2, "tid t%x rewinddir(index %d) - bad index\n",
		  pvmmytid, (int)dp));
	else
		dprint((buf2, "tid t%x rewinddir(index %d) ok\n",
		  pvmmytid, (int)dp));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return;
}

void
seekdir(DIR *dp, long loc)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	dindex;
	int	request = PVMFS_SEEKDIR;
	int	nargs = 2;

	if (pvmmyfstid <= 0) {
		lpvmerr("seekdir", PvmNoFileServer);
		errno = ENOSYS;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	dindex = (int)dp - 1;
	pvm_packf("%+ %d %d %ld", PvmDataDefault, nargs, dindex, loc);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%ld", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x seekdir(index %d, %ld) - bad index\n",
		  pvmmytid, (int)dp, loc));
	} else
		dprint((buf2, "tid t%x seekdir(index %d, %ld) ok\n",
		  pvmmytid, (int)dp, loc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));
}

long
telldir(DIR *dp)
{
	int	lcc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	dindex;
	int	request = PVMFS_TELLDIR;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		lpvmerr("telldir", PvmNoFileServer);
		errno = ENOSYS;
		return -1;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	dindex = (int)dp - 1;
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, dindex);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%ld", &lcc);
	if (lcc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x telldir(index %d) = -1 (errno = %d)\n",
		  pvmmytid, (int)dp, errno));
	} else
		dprint((buf2, "tid t%x telldir(index %d) = %ld\n",
		  pvmmytid, (int)dp, lcc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return lcc;
}

char *
ttyname(int fd)
{
	int		cc;
	int		pvmerrno;
	int		rbuf;
	int		sbuf;
	int		request = PVMFS_TTYNAME;
	int		nargs = 1;
	char		*cresult;
	static char	*tty = NULL;

	if (pvmmyfstid <= 0) {
		lpvmerr("ttyname", PvmNoFileServer);
		errno = ENOSYS;
		return NULL;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == 0 ||
#if defined(IMA_SUN3) || defined(IMA_SUN4)
	  !tty && (tty = malloc(cc + 1)) == NULL || tty &&
#endif
	  (tty = realloc(tty, cc + 1)) == NULL) {
		if (cc == 0) {
			pvm_unpackf("%d", &pvmerrno);
			errno = pvm_local_errno(pvmerrno);
		}
		cresult = NULL;
		dprint((buf2, "tid t%x ttyname(%d) = NULL (errno = %d)\n",
		  pvmmytid, fd, errno));
	} else {
		pvm_unpackf("%s", tty);
		cresult = tty;
		dprint((buf2, "tid t%x ttyname(%d) = %s\n",
		  pvmmytid, fd, tty));
	}

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cresult;
}

#ifdef tcdrain
#define tcdrain_is_defined
static int
TCDRAIN(int fd)
{
	return tcdrain(fd);
}
#undef tcdrain
#endif

int
tcdrain(int fd)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_TCDRAIN;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
#ifdef tcdrain_is_defined
		return TCDRAIN(fd);
#else
		lpvmerr("tcdrain", PvmNoFileServer);
		errno = ENOSYS;
		return -1;
#endif
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x tcdrain(%d) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, pvmerrno));
	} else
		dprint((buf2, "tid t%x tcdrain(%d) = %d\n",
		  pvmmytid, fd, cc));

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

#ifdef tcflow
#define tcflow_is_defined
static int
TCFLOW(int fd, int action)
{
	return tcflow(fd, action);
}
#undef tcflow
#endif

int
tcflow(int fd, int action)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_TCFLOW;
	int	nargs = 2;
	int	pvmaction;

	if (pvmmyfstid <= 0) {
#ifdef tcflow_is_defined
		return TCFLOW(fd, action);
#else
		lpvmerr("tcflow", PvmNoFileServer);
		errno = ENOSYS;
		return -1;
#endif
	}

	pvmaction = pvm_tcflow_action(action);
	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %d", PvmDataDefault, nargs, fd, pvmaction);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x tcflow(%d, %d) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, action, pvmerrno));
	} else
		dprint((buf2, "tid t%x tcflow(%d, %d) = %d\n",
		  pvmmytid, fd, action, cc));

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

#ifdef tcflush
#define tcflush_is_defined
static int
TCFLUSH(int fd, int queue)
{
	return tcflush(fd, queue);
}
#undef tcflush
#endif

int
tcflush(int fd, int queue)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_TCFLUSH;
	int	nargs = 2;
	int	pvmqueue;

	if (pvmmyfstid <= 0) {
#ifdef tcflush_is_defined
		return TCFLUSH(fd, queue);
#else
		lpvmerr("tcflush", PvmNoFileServer);
		errno = ENOSYS;
		return -1;
#endif
	}

	pvmqueue = pvm_tcflush_queue(queue);
	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %d", PvmDataDefault, nargs, fd, pvmqueue);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x tcflush(%d, %d) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, queue, pvmerrno));
	} else
		dprint((buf2, "tid t%x tcflush(%d, %d) = %d\n",
		  pvmmytid, fd, queue, cc));

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
tcsendbreak(int fd, int duration)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_TCSENDBREAK;
	int	nargs = 2;

	if (pvmmyfstid <= 0) {
		lpvmerr("tcsendbreak", PvmNoFileServer);
		errno = ENOSYS;
		return -1;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %d", PvmDataDefault, nargs, fd, duration);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x tcsendbreak(%d, %d) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, duration, pvmerrno));
	} else
		dprint((buf2, "tid t%x tcsendbreak(%d, %d) = %d\n",
		  pvmmytid, fd, duration, cc));

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

#if 0

/*
 * Give up on tc[gs]etattr for now.  Too much wierdness.
 */
int	pvmfs_lccindex[] = {
	VEOF, VEOL, VERASE, VINTR, VKILL, VQUIT, VSTART, VSTOP, VSUSP,
#ifdef VLNEXT
	VLNEXT,
#else
	-1,
#endif
#ifdef VWERASE
	VWERASE,
#else
	-1,
#endif
#ifdef VREPRINT
	VREPRINT,
#else
	-1,
#endif
#ifdef VDISCARD
	VDISCARD,
#else
	-1,
#endif
#ifdef VDSUSP
	VDSUSP,
#else
	-1,
#endif
#ifdef VEOL2
	VEOL2,
#else
	-1,
#endif
#ifdef VSTATUS
	VSTATUS,
#else
	-1,
#endif
};

int	pvmfs_fsccindex[] = {
	PVMFS_VEOF, PVMFS_VEOL, PVMFS_VERASE, PVMFS_VINTR, PVMFS_VKILL,
	PVMFS_VQUIT, PVMFS_VSTART, PVMFS_VSTOP, PVMFS_VSUSP, PVMFS_VLNEXT,
	PVMFS_VWERASE, PVMFS_VREPRINT, PVMFS_VDISCARD, PVMFS_VDSUSP,
	PVMFS_VEOL2, PVMFS_VSTATUS,
};

#define PVMFS_NCCS	sizeof pvmfs_fsccindex / sizeof pvmfs_fsccindex[0]
static long	ctl_chars[PVMFS_NCCS];

int
tcgetattr(int fd, struct termios *tp)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_TCGETATTR;
	int	nargs = 1;
	long	iflag;
	long	oflag;
	long	cflag;
	long	lflag;
	long	speed;

	if (pvmmyfstid <= 0) {
		lpvmerr("tcgetattr", PvmNoFileServer);
		errno = ENOSYS;
		return -1;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x tcgetattr(%d, %p) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, tp, pvmerrno));
	} else {
		(void)memset(tp, '\0', sizeof(*tp));
		pvm_unpackf("%ld %ld %ld %ld", &iflag, &oflag, &cflag, &lflag);
		tp->iflag = pvm_local_tcattr_iflag(iflag);
		tp->oflag = pvm_local_tcattr_oflag(oflag);
		tp->cflag = pvm_local_tcattr_cflag(cflag);
		tp->lflag = pvm_local_tcattr_lflag(lflag);
		pvm_unpackf("%ld", &speed);
		cfsetispeed(tp, pvm_local_tcattr_ispeed(speed));
		pvm_unpackf("%ld", &speed);
		cfsetospeed(tp, pvm_local_tcattr_ospeed(speed));
		for (i = 0; i < PVMFS_NCCS; i++) {
			pvm_unpackf("%ld", &ctl_chars[i]);
			if (pvmfs_lccindex[i] >= 0)
				tp->c_cc[pvmfs_lccindex[i]] = ctl_chars[i];
		}
		dprint((buf2, "tid t%x tcgetattr(%d, %p) = %d\n",
		  pvmmytid, fd, tp, cc));
	}

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
tcsetattr(int fd, int opt, struct termios *tp)
{
	int		cc;
	int		pvmerrno;
	int		rbuf;
	int		sbuf;
	int		request = PVMFS_TCSETATTR;
	int		nargs = 2;
	int		pvmopt;
	long		iflag;
	long		oflag;
	long		cflag;
	long		lflag;
	long		speed;
	long		ctl_char;
	static int	getattr_called = 0;

	if (pvmmyfstid <= 0) {
		lpvmerr("tcsetattr", PvmNoFileServer);
		errno = ENOSYS;
		return -1;
	}

	if (!getattr_called) {
		struct termios	term;

		if (tcgetattr(fd, &term) == -1)
			return -1;
		getattr_called = 1;
	}

	pvmopt = pvm_tcsetattr_opt(opt);
	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %d", PvmDataDefault, nargs, fd, pvmopt);
	iflag = pvm_tcattr_iflag(tp->iflag);
	oflag = pvm_tcattr_oflag(tp->oflag);
	cflag = pvm_tcattr_cflag(tp->cflag);
	lflag = pvm_tcattr_lflag(tp->lflag);
	pvm_packf("%ld %ld %ld %ld", iflag, oflag, cflag, lflag);
	speed = cfgetispeed(tp);
	pvm_packf("%ld", speed);
	speed = cfgetospeed(tp);
	pvm_packf("%ld", speed);
	for (i = 0; i < PVMFS_NCCS; i++) {
		ctl_char = (pvmfs_lccindex[i] >= 0)
		  ? tp->c_cc[pvmfs_lccindex[i]] : ctl_chars[i];
		pvm_packf("%ld", ctl_chars[i]);
	}
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x tcgetattr(%d, %p) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, tp, pvmerrno));
	} else
		dprint((buf2, "tid t%x tcgetattr(%d, %p) = %d\n",
		  pvmmytid, fd, tp, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}
#endif

long
pathconf(const char *pathname, int name)
{
	long	lcc;
	int	pvmerrno;
	int	pvmname;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_PATHCONF;
	int	nargs = 2;

	if (pvmmyfstid <= 0) {
		if (name < 0) {
#ifdef SYS_sysconf
			lcc = syscall(SYS_sysconf, _SC_OPEN_MAX);
			if (lcc == -1L)
				dprintl((buf2,
			    "tid t%x sysconf(_SC_OPEN_MAX) = -1 (errno = %d)\n",
			    pvmmytid, errno));
			else
				dprintl((buf2,
				  "tid t%x sysconf(_SC_OPEN_MAX) = %ld\n",
				  pvmmytid, lcc));
#else
			lpvmerr("sysconf", PvmNoFileServer);
			errno = ENOSYS;
			return -1L;
#endif
		} else {
#ifdef SYS_pathconf
			lcc = syscall(SYS_pathconf, pathname, name);
			if (lcc == -1L)
				dprintl((buf2,
			    "tid t%x pathconf(\"%s\", %d) = %ld (errno = %d)\n",
			    pvmmytid, pathname, name, lcc, errno));
			else
				dprintl((buf2,
				  "tid t%x pathconf(\"%s\", %d) = %ld\n",
				  pvmmytid, pathname, name, lcc));
			return lcc;
#else
			lpvmerr("pathconf", PvmNoFileServer);
			errno = ENOSYS;
			return -1L;
#endif
		}
	}

	pvmname = pvm_pathconf_name(name);
	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %d", PvmDataDefault, nargs, pathname, pvmname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%ld", &lcc);
	if (lcc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		if (name < 0)
			dprint((buf2,
			 "tid t%x sysconf(_SC_OPEN_MAX) = -1 (pvmerrno = %d)\n",
			 pvmmytid, pvmerrno));
		else
			dprint((buf2,
			  "tid t%x pathconf(\"%s\", %d) = -1 (pvmerrno = %d)\n",
			  pvmmytid, pathname, name, pvmerrno));
	} else if (name < 0)
		dprint((buf2, "tid t%x sysconf(_SC_OPEN_MAX) = %ld\n",
		  pvmmytid, lcc));
	else
		dprint((buf2, "tid t%x pathconf(\"%s\", %d) = %ld\n",
		  pvmmytid, pathname, name, lcc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return lcc;
}

long
fpathconf(int fd, int name)
{
	int	lcc;
	int	pvmerrno;
	int	pvmname;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_FPATHCONF;
	int	nargs = 2;

	if (pvmmyfstid <= 0) {
		if (name < 0) {
#ifdef SYS_sysconf
			lcc = syscall(SYS_sysconf, _SC_OPEN_MAX);
			if (lcc == -1L)
				dprintl((buf2,
			    "tid t%x sysconf(_SC_OPEN_MAX) = -1 (errno = %d)\n",
			    pvmmytid, errno));
			else
				dprintl((buf2,
				  "tid t%x sysconf(_SC_OPEN_MAX) = %ld\n",
				  pvmmytid, lcc));
#else
			lpvmerr("sysconf", PvmNoFileServer);
			errno = ENOSYS;
			return -1L;
#endif
		} else {
#ifdef SYS_fpathconf
			lcc = syscall(SYS_fpathconf, fd, name);
			if (lcc == -1)
				dprintl((buf2,
			       "tid t%x fpathconf(%d, %d) = %ld (errno = %d)\n",
			       pvmmytid, fd, name, lcc, errno));
			else
				dprintl((buf2,
				  "tid t%x fpathconf(%d, %d) = %ld\n",
				  pvmmytid, fd, name, lcc));
			return lcc;
#else
			lpvmerr("fpathconf", PvmNoFileServer);
			errno = ENOSYS;
			return NULL;
#endif
		}
	}

	pvmname = pvm_pathconf_name(name);
	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %d", PvmDataDefault, nargs, fd, pvmname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%ld", &lcc);
	if (lcc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x fpathconf(%d, %d) = %ld (pvmerrno = %d)\n", pvmmytid,
		  fd, name, lcc, pvmerrno));
	} else
		dprint((buf2, "tid t%x fpathconf(%d, %d) = %ld\n",
		  pvmmytid, fd, name, lcc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return lcc;
}

#if 0
/*
 * This getcwd stuff doesn't really help
 * unless we implement the POSIX directory functions.
 */

/* reverse reverse a string so that "string" becomes "gnirts" */
static char *
reverse(char *s)
{
	size_t	len;
	size_t	i;
	char	c;
	char	*t;
	char	*result = s;

	len = strlen(s);
	t = &s[len - 1];

	for (i = 0; i < len / 2; i++) {
		c = *s;
		*s++ = *t;
		*t-- = c;
	}

	return result;
}

static char *
mygetcwd(char *buf, size_t size)
{
	ino_t		inode;
	dev_t		dev;
	struct stat	stbuf;
	DIR		*dfd;
	struct dirent	*dp;
	struct stat	rootbuf;
	ino_t		rootinode;
	dev_t		rootdev;
	char		dots[PATH_MAX];
	size_t		len;
	char		*end;

	if (size == 0) {
		errno = EINVAL;
		return NULL;
	}

	if (stat("/", &rootbuf) == -1)
		return NULL;
	rootinode = rootbuf.st_ino;
	rootdev = rootbuf.st_dev;

	(void)strcpy(dots, ".");
	buf[0] = '\0';

	while (1) {
		if (stat(dots, &stbuf) == -1)
			return NULL;
		inode = stbuf.st_ino;
		dev = stbuf.st_dev;
		if (inode == rootinode && dev == rootdev) {
			if (strlen(buf) + 1 >= size) {
				errno = ERANGE;
				return NULL;
			} else
				return reverse(strcat(buf, "/"));
		}
	
		if ((len = strlen(dots) + 3) >= size) {
			errno = ERANGE;
			return NULL;
		}
		(void)strcat(dots, "/..");
		if ((dfd = opendir(dots)) == NULL)
			return NULL;
		if (++len >= size) {
			errno = ERANGE;
			return NULL;
		}
		(void)strcat(dots, "/");
		while (dp = readdir(dfd)) {
			if (len + dp->d_namlen >= size)
				continue;
			if (stat(strcat(dots, dp->d_name), &stbuf) == -1) {
				dots[len] = '\0';
				continue;
			}
			if (stbuf.st_ino == inode && stbuf.st_dev == dev) {
				size_t	buflen = strlen(buf);

				if (buflen + dp->d_namlen + 1 >= size) {
					errno = ERANGE;
					return NULL;
				}
				if (buflen)
					(void)strcat(buf, "/");
				(void)strcat(buf, reverse(dp->d_name));
				break;
			}
			dots[len] = '\0';
		}
		if (!dp)
			return NULL;	/* errno = ??? */
		dots[len - 1] = '\0';
	}
}
#endif

char *
getcwd(char *buf, size_t size)
{
	char		*result;
	int		pvmerrno;
	int		rbuf;
	int		sbuf;
	int		request = PVMFS_GETCWD;
	int		nargs = 0;
	unsigned long	ul;

	if (size == 0) {
		errno = ERANGE;
		return NULL;
	}

	if (pvmmyfstid <= 0) {
#if 0
		result = mygetcwd(buf, size);
		if (result == NULL)
			dprintl((buf2,
			  "tid t%x getcwd(%p, %lu) = NULL (errno = %d)\n",
			  pvmmytid, (void *)buf, (unsigned long)size, errno));
		else
			dprintl((buf2, "tid t%x getcwd(%p, %lu) = %p\n",
			  pvmmytid, (void *)buf, (unsigned long)size,
			  (void *)result));
		return result;
#else
		lpvmerr("getcwd", PvmNoFileServer);
		errno = ENOSYS;
		return NULL;
#endif
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d", PvmDataDefault, nargs);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_upkulong(&ul, 1, 1);	/* ul = strlen() + 1 */
	result = buf;
	if (ul > 0 && buf == NULL)
		result = malloc(ul);
	if (ul == 0 || result == NULL || ul > size) {
		if (ul == 0) {
			pvm_unpackf("%d", &pvmerrno);
			errno = pvm_local_errno(pvmerrno);
		} else if (ul > size)
			errno = ERANGE;
		result = NULL;
		dprintl((buf2,
		  "tid t%x getcwd(%p, %lu) = NULL (errno = %d)\n",
		  pvmmytid, (void *)buf, (unsigned long)size, errno));
	} else {
		result = buf;
		pvm_unpackf("%s", buf);
		dprintl((buf2, "tid t%x getcwd(%p, %lu) = \"%s\"\n",
		  pvmmytid, (void *)buf, (unsigned long)size, result));
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return result;
}

caddr_t
mmap(caddr_t addr, size_t len, int prot, int flags, int fd, off_t offset)
{
	caddr_t	cp;

#ifdef SYS_mmap
	if (pvmmyfstid <= 0 || fd < 0) {
		cp = (caddr_t)syscall(SYS_mmap, addr, len, prot, flags, fd,
		  offset);
		if (cp == (caddr_t)-1)
			dprintl((buf2,
		  "tid t%x mmap(%p, %lu, %d, %d, %d, %ld) = -1 (errno = %d)\n",
			  pvmmytid, (void *)addr, (unsigned long)len, prot,
			  flags, fd, (long)offset, errno));
		else
			dprintl((buf2,
			  "tid t%x mmap(%p, %lu, %d, %d, %d, %ld) = %p\n",
			  pvmmytid, (void *)addr, (unsigned long)len, prot,
			  flags, fd, (long)offset, (void *)cp));
		return cp;
	} else {
#endif
		/*lpvmerr("mmap", PvmNotImpl);*/
		errno = ENOSYS;
		return NULL;
#ifdef SYS_mmap
	}
#endif
}

#ifdef IMA_CNVXN
#include <stdarg.h>
#endif

int
#ifdef IMA_CNVXN
ioctl(int d, int request, ...)
#else
ioctl(int d, unsigned long request, char *argp)
#endif
{
#ifdef IMA_CNVXN
	va_list	ap;
	char	*argp;

	va_start(ap, request);
	argp = va_arg(ap, char *);
#endif

#ifdef SYS_ioctl
	if (pvmmyfstid <= 0)
		return syscall(SYS_ioctl, d, request, argp);
	else {
#endif
		lpvmerr("ioctl", PvmNotImpl);
		errno = ENOSYS;
		return -1;
#ifdef SYS_ioctl
	}
#endif
}

#ifdef IMA_CNVXN

/*
 * Again, why do these exist?
 */

DIR *
__ap$opendir(const char *pathname)
{
	return opendir(pathname);
}

struct dirent *
__ap$readdir(DIR *dp)
{
	return readdir(dp);
}

int
__ap$closedir(DIR *dp)
{
	return closedir(dp);
}

void
__ap$rewinddir(DIR *dp)
{
	rewinddir(dp);
}

void
__ap$seekdir(DIR *dp, long loc)
{
	seekdir(dp, loc);
}

long
__ap$telldir(DIR *dp)
{
	return telldir(dp);
}

char *
__ap$ttyname(int fd)
{
	return ttyname(fd);
}

int
__ap$tcdrain(int fd)
{
	return tcdrain(fd);
}

int
__ap$tcflow(int fd, int action)
{
	return tcflow(fd, action);
}

int
__ap$tcflush(int fd, int queue)
{
	return tcflush(fd, queue);
}

int
__ap$tcsendbreak(int fd, int duration)
{
	return tcsendbreak(fd, duration);
}

long
__ap$pathconf(const char *pathname, int name)
{
	return pathconf(pathname, name);
}

long
__ap$fpathconf(int fd, int name)
{
	return fpathconf(fd, name);
}

char *
__ap$getcwd(char *buf, size_t size)
{
	return getcwd(buf, size);
}

caddr_t
__ap$mmap(caddr_t addr, size_t len, int prot, int flags, int fd, off_t offset)
{
	return mmap(addr, len, prot, flags, fd, offset);
}

int
__ap$ioctl(int d, int request, ...)
{
	va_list	ap;
	char	*argp;

	va_start(ap, request);
	argp = va_arg(ap, char *);

#ifdef SYS_ioctl
	if (pvmmyfstid <= 0)
		return syscall(SYS_ioctl, d, request, argp);
	else {
#endif
		lpvmerr("ioctl", PvmNotImpl);
		errno = ENOSYS;
		return -1;
#ifdef SYS_ioctl
	}
#endif
}

#endif /* IMA_CNVXN */
