/*
 *                 Author:  Christopher G. Phillips
 *              Copyright (C) 1993 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * The author makes no representations about the suitability of this
 * software for any purpose.  This software is provided ``as is''
 * without express or implied warranty.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#if defined(__STDC__) || defined(IMA_CNVXN) || defined(IMA_RS6K)
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <errno.h>
#include <fcntl.h>
#include <sys/uio.h>
#include <utime.h>
#include <termios.h>
#include "pvm3.h"
#include "fs.h"
#include "fsp.h"
#include "pipe.h"
#ifdef SYSCALL_PIPE_PROBLEM
#include <sys/socket.h>
#endif
#ifndef IMA_RS6K
#include <syscall.h>
#else
#include <sys/select.h>
#include "syscall-fake.h"
#endif
#ifndef IMA_SUN4
#include <sys/ioctl.h>
#endif
#include "argtypes.h"

#ifdef MIN
#undef MIN
#endif
#define MIN(a,b)	((a) < (b) ? (a) : (b))

#ifdef TRACE
/*
 * Debugging and trace junk follows...
 */
int	pvmdebug = 0;

#define dprint(s)	do {	\
				if (pvmdebug) {	\
					sprintf s;	\
					syscall(SYS_write, 2, buf2, \
					  strlen(buf2)); \
				}	\
			} while (0)
#define dprintl(s)	dprint(s)

static char buf2[100];

#else
#define dprint(s)
#define dprintl(s)
#endif /* TRACE */

extern int	lpvmerr __ProtoGlarp__((const char *, int));
extern int	pvm_local_errno __ProtoGlarp__((int));
extern int	pvmmytid;
extern int	pvmmyfstid;
extern int	errno;

/*
 * Mapping from PVMFS constants to local constants.
 */

int
pvm_local_open_oflag(int oflag)
{
	int	result = 0;

	switch (oflag & (PVMFS_O_RDONLY | PVMFS_O_WRONLY | PVMFS_O_RDWR)) {
	case PVMFS_O_RDONLY:
		result |= O_RDONLY;
		break;
	case PVMFS_O_WRONLY:
		result |= O_WRONLY;
		break;
	case PVMFS_O_RDWR:
		result |= O_RDWR;
		break;
	default:
		return -1;
	}

	if (oflag & PVMFS_O_APPEND)	result |= O_APPEND;
	if (oflag & PVMFS_O_CREAT)	result |= O_CREAT;
	if (oflag & PVMFS_O_EXCL)	result |= O_EXCL;
	if (oflag & PVMFS_O_TRUNC)	result |= O_TRUNC;
#ifdef O_NOCTTY
	if (oflag & PVMFS_O_NOCTTY)	result |= O_NOCTTY;
#endif
#ifdef O_NONBLOCK
	if (oflag & PVMFS_O_NONBLOCK)	result |= O_NONBLOCK;
#endif
#ifdef O_SYNC
	if (oflag & PVMFS_O_SYNC)	result |= O_SYNC;
#endif
#ifdef O_ASYNC
	if (oflag & PVMFS_O_ASYNC)	result |= O_ASYNC;
#endif
#ifdef O_LARGEFILE
	if (oflag & PVMFS_O_LARGEFILE)	result |= O_LARGEFILE;
#endif
	return result;
}

mode_t
pvm_local_open_mode(mode_t mode)
{
	mode_t	result = 0;

	if (mode & PVMFS_S_IRUSR)	result |= S_IRUSR;
	if (mode & PVMFS_S_IWUSR)	result |= S_IWUSR;
	if (mode & PVMFS_S_IXUSR)	result |= S_IXUSR;
	if (mode & PVMFS_S_IRGRP)	result |= S_IRGRP;
	if (mode & PVMFS_S_IWGRP)	result |= S_IWGRP;
	if (mode & PVMFS_S_IXGRP)	result |= S_IXGRP;
	if (mode & PVMFS_S_IROTH)	result |= S_IROTH;
	if (mode & PVMFS_S_IWOTH)	result |= S_IWOTH;
	if (mode & PVMFS_S_IXOTH)	result |= S_IXOTH;

	return result;
}

int
pvm_local_lseek_whence(int whence)
{
	switch (whence) {
	case PVMFS_SEEK_SET:	return SEEK_SET;
	case PVMFS_SEEK_CUR:	return SEEK_CUR;
	case PVMFS_SEEK_END:	return SEEK_END;
	default:		return -1;
	}
}

int
pvm_local_fcntl_cmd(int command)
{
	switch (command) {
	case PVMFS_F_DUPFD:	return F_DUPFD;
	case PVMFS_F_GETFD:	return F_GETFD;
	case PVMFS_F_SETFD:	return F_SETFD;
	case PVMFS_F_GETFL:	return F_GETFL;
	case PVMFS_F_SETFL:	return F_SETFL;
	case PVMFS_F_GETOWN:	return F_GETOWN;
	case PVMFS_F_SETOWN:	return F_SETOWN;
	case PVMFS_F_GETLK:	return F_GETLK;
	case PVMFS_F_SETLK:	return F_SETLK;
	case PVMFS_F_SETLKW:	return F_SETLKW;
	default:		return -1;
	}
}

short
pvm_local_fcntl_ltype(short type)
{
	switch (type) {
	case PVMFS_F_RDLCK:	return F_RDLCK;
	case PVMFS_F_WRLCK:	return F_WRLCK;
	case PVMFS_F_UNLCK:	return F_UNLCK;
	default:		return -1;
	}
}

int
pvm_local_access_mode(int mode)
{
	int	result = 0;

	if (mode & PVMFS_R_OK)	result |= R_OK;
	if (mode & PVMFS_W_OK)	result |= W_OK;
	if (mode & PVMFS_X_OK)	result |= X_OK;
	if (mode & PVMFS_F_OK)	result |= F_OK;

	return result;
}

int
pvm_local_tcflow_action(int action)
{
	switch (action) {
	case PVMFS_TCOOFF:	return TCOOFF;
	case PVMFS_TCOON:	return TCOON;
	case PVMFS_TCIOFF:	return TCIOFF;
	case PVMFS_TCION:	return TCION;
	default:		return -1;
	}
}

int
pvm_local_tcflush_queue(int queue)
{
	switch (queue) {
	case PVMFS_TCIFLUSH:	return TCIFLUSH;
	case PVMFS_TCOFLUSH:	return TCOFLUSH;
	case PVMFS_TCIOFLUSH:	return TCIOFLUSH;
	default:		return -1;
	}
}

int
pvm_local_tcsetattr_opt(int opt)
{
	switch (opt) {
	case PVMFS_TCSANOW:	return TCSANOW;
	case PVMFS_TCSADRAIN:	return TCSADRAIN;
	case PVMFS_TCSAFLUSH:	return TCSAFLUSH;
	default:		return -1;
	}
}

long
pvm_local_tcattr_iflag(long iflag)
{
	long	result = 0;

	if (iflag & PVMFS_BRKINT)	result |= BRKINT;
	if (iflag & PVMFS_ICRNL)	result |= ICRNL;
	if (iflag & PVMFS_IGNBRK)	result |= IGNBRK;
	if (iflag & PVMFS_IGNCR)	result |= IGNCR;
	if (iflag & PVMFS_IGNPAR)	result |= IGNPAR;
#ifdef IMAXBEL
	if (iflag & PVMFS_IMAXBEL)	result |= IMAXBEL;
#endif
	if (iflag & PVMFS_INLCR)	result |= INLCR;
	if (iflag & PVMFS_INPCK)	result |= INPCK;
	if (iflag & PVMFS_ISTRIP)	result |= ISTRIP;
#ifdef IUCLC
	if (iflag & PVMFS_IUCLC)	result |= IUCLC;
#endif
#ifdef IXANY
	if (iflag & PVMFS_IXANY)	result |= IXANY;
#endif
	if (iflag & PVMFS_IXOFF)	result |= IXOFF;
	if (iflag & PVMFS_IXON)		result |= IXON;
	if (iflag & PVMFS_PARMRK)	result |= PARMRK;

	return result;
}

long
pvm_local_tcattr_oflag(long oflag)
{
	long	result = 0;

#ifdef BSDLY
	result &= ~BSDLY;
	if (oflag & PVMFS_BS0)
		result |= BS0;
	else if (oflag & PVMFS_BS1)
		result |= BS1;
#endif
#ifdef CRDLY
	result &= ~CRDLY;
	if (oflag & PVMFS_CR0)
		result |= CR0;
	else if (oflag & PVMFS_CR1)
		result |= CR1;
	else if (oflag & PVMFS_CR2)
		result |= CR2;
	else if (oflag & PVMFS_CR3)
		result |= CR3;
#endif
#ifdef FFDLY
	result &= ~FFDLY;
	if (oflag & PVMFS_FF0)
		result |= FF0;
	else if (oflag & PVMFS_FF1)
		result |= FF1;
#endif
#ifdef NLDLY
	result &= ~NLDLY;
	if (oflag & PVMFS_NL0)
		result |= NL0;
	else if (oflag & PVMFS_NL1)
		result |= NL1;
#endif
#ifdef OCRNL
	if (oflag & PVMFS_OCRNL)	result |= OCRNL;
#endif
#ifdef OFDEL
	if (oflag & PVMFS_OFDEL)	result |= OFDEL;
#endif
#ifdef OFILL
	if (oflag & PVMFS_OFILL)	result |= OFILL;
#endif
#ifdef OLCUC
	if (oflag & PVMFS_OLCUC)	result |= OLCUC;
#endif
#ifdef ONLCR
	if (oflag & PVMFS_ONLCR)	result |= ONLCR;
#endif
#ifdef ONLRET
	if (oflag & PVMFS_ONLRET)	result |= ONLRET;
#endif
#ifdef ONOCR
	if (oflag & PVMFS_ONOCR)	result |= ONOCR;
#endif
#ifdef ONOEOT
	if (oflag & PVMFS_ONOEOT)	result |= ONOEOT;
	if (oflag & PVMFS_OPOST)	result |= OPOST;
#endif
#ifdef OXTABS
	if (oflag & PVMFS_OXTABS)	result |= OXTABS;
#endif
#ifdef TABDLY
	result &= ~TABDLY;
	if (oflag & PVMFS_TAB0)
		result |= TAB0;
	else if (oflag & PVMFS_TAB1)
		result |= TAB1;
	else if (oflag & PVMFS_TAB2)
		result |= TAB2;
	else if (oflag & PVMFS_TAB3)
		result |= TAB3;
#endif
#ifdef VTDLY
	result &= ~VTDLY;
	if (oflag & PVMFS_VT0)
		result |= VT0;
	else if (oflag & PVMFS_VT1)
		result |= VT1;
#endif

	return result;
}

long
pvm_local_tcattr_cflag(long cflag)
{
	long	result = 0;

#ifdef CCTS_OFLOW
	if (cflag & PVMFS_CCTS_OFLOW)	result |= CCTS_OFLOW;
#endif
#ifdef CIGNORE
	if (cflag & PVMFS_CIGNORE)	result |= CIGNORE;
#endif
	if (cflag & PVMFS_CLOCAL)	result |= CLOCAL;
	if (cflag & PVMFS_CREAD)	result |= CREAD;
#ifdef CRTS_IFLOW
	if (cflag & PVMFS_CRTS_IFLOW)	result |= CRTS_IFLOW;
#endif
	if (cflag & PVMFS_CSIZE)	result |= CSIZE;
	if (cflag & PVMFS_CSTOPB)	result |= CSTOPB;
	if (cflag & PVMFS_HUPCL)	result |= HUPCL;
#ifdef MDMBUF
	if (cflag & PVMFS_MDMBUF)	result |= MDMBUF;
#endif
	if (cflag & PVMFS_PARENB)	result |= PARENB;
	if (cflag & PVMFS_PARODD)	result |= PARODD;

	return result;
}

long
pvm_local_tcattr_lflag(long lflag)
{
	long	result = 0;

#ifdef ALTWERASE
	if (lflag & PVMFS_ALTWERASE)	result |= ALTWERASE;
#endif
	if (lflag & PVMFS_ECHO)	result |= ECHO;
#ifdef ECHOCTL
	if (lflag & PVMFS_ECHOCTL)	result |= ECHOCTL;
#endif
	if (lflag & PVMFS_ECHOE)	result |= ECHOE;
	if (lflag & PVMFS_ECHOK)	result |= ECHOK;
#ifdef ECHOKE
	if (lflag & PVMFS_ECHOKE)	result |= ECHOKE;
#endif
	if (lflag & PVMFS_ECHONL)	result |= ECHONL;
#ifdef ECHOPRT
	if (lflag & PVMFS_ECHOPRT)	result |= ECHOPRT;
#endif
#ifdef FLUSHO
	if (lflag & PVMFS_FLUSHO)	result |= FLUSHO;
#endif
	if (lflag & PVMFS_ICANON)	result |= ICANON;
	if (lflag & PVMFS_IEXTEN)	result |= IEXTEN;
	if (lflag & PVMFS_ISIG)		result |= ISIG;
	if (lflag & PVMFS_NOFLSH)	result |= NOFLSH;
#ifdef NOKERNINFO
	if (lflag & PVMFS_NOKERNINFO)	result |= NOKERNINFO;
#endif
#ifdef PENDIN
	if (lflag & PVMFS_PENDIN)	result |= PENDIN;
#endif
	if (lflag & PVMFS_TOSTOP)	result |= TOSTOP;
#ifdef XCASE
	if (lflag & PVMFS_XCASE)	result |= XCASE;
#endif

	return result;
}

long
pvm_local_tcattr_speed(long speed)
{
	switch (speed) {
	case PVMFS_B0:		return B0;
	case PVMFS_B50:		return B50;
	case PVMFS_B75:		return B75;
	case PVMFS_B110:	return B110;
	case PVMFS_B134:	return B134;
	case PVMFS_B150:	return B150;
	case PVMFS_B200:	return B200;
	case PVMFS_B300:	return B300;
	case PVMFS_B600:	return B600;
	case PVMFS_B1200:	return B1200;
	case PVMFS_B1800:	return B1800;
	case PVMFS_B2400:	return B2400;
	case PVMFS_B4800:	return B4800;
	case PVMFS_B9600:	return B9600;
	case PVMFS_B19200:	return B19200;
	case PVMFS_B38400:	return B38400;
	default:		return -1;
	}
}

int
pvm_local_pathconf_name(int name)
{
	switch (name) {
	case PVMFS__PC_LINK_MAX:		return _PC_LINK_MAX;
	case PVMFS__PC_MAX_CANON:		return _PC_MAX_CANON;
	case PVMFS__PC_MAX_INPUT:		return _PC_MAX_INPUT;
	case PVMFS__PC_NAME_MAX:		return _PC_NAME_MAX;
	case PVMFS__PC_PATH_MAX:		return _PC_PATH_MAX;
	case PVMFS__PC_PIPE_BUF:		return _PC_PIPE_BUF;
	case PVMFS__PC_CHOWN_RESTRICTED:	return _PC_CHOWN_RESTRICTED;
	case PVMFS__PC_NO_TRUNC:		return _PC_NO_TRUNC;
	case PVMFS__PC_VDISABLE:		return _PC_VDISABLE;
	default:				return -1;
	}
}

int
pvm_local_rlimit_resource(int resource)
{
	switch (resource) {
	case PVMFS_RLIMIT_CORE:		return RLIMIT_CORE;
	case PVMFS_RLIMIT_FSIZE:	return RLIMIT_FSIZE;
#ifdef RLIMIT_OFILE
	case PVMFS_RLIMIT_OFILE:	return RLIMIT_OFILE;
#endif
#ifdef RLIMIT_NOFILE
	case PVMFS_RLIMIT_NOFILE:	return RLIMIT_NOFILE;
#endif
	default:			return -1;
	}
}

/*
 * Mapping from local constants to PVMFS constants.
 */

int
pvm_open_oflag(int oflag)
{
	int	result = 0;

	switch (oflag & O_ACCMODE) {
	case O_RDONLY:	result |= PVMFS_O_RDONLY;	break;
	case O_WRONLY:	result |= PVMFS_O_WRONLY;	break;
	case O_RDWR:	result |= PVMFS_O_RDWR;		break;
	default:
		return -1;
	}
	oflag &= ~O_ACCMODE;

	if (oflag & O_APPEND) {
		result |= PVMFS_O_APPEND;
		oflag &= ~O_APPEND;
	}
	if (oflag & O_CREAT) {
		result |= PVMFS_O_CREAT;
		oflag &= ~O_CREAT;
	}
	if (oflag & O_EXCL) {
		result |= PVMFS_O_EXCL;
		oflag &= ~O_EXCL;
	}
	if (oflag & O_TRUNC) {
		result |= PVMFS_O_TRUNC;
		oflag &= ~O_TRUNC;
	}
#ifdef O_NOCTTY
	if (oflag & O_NOCTTY) {
		result |= PVMFS_O_NOCTTY;
		oflag &= ~O_NOCTTY;
	}
#endif
#ifdef O_NONBLOCK
	if (oflag & O_NONBLOCK) {
		result |= PVMFS_O_NONBLOCK;
		oflag &= ~O_NONBLOCK;
	}
#endif
#ifdef O_SYNC
	if (oflag & O_SYNC) {
		result |= PVMFS_O_SYNC;
		oflag &= ~O_SYNC;
	}
#endif
#ifdef O_ASYNC
	if (oflag & O_ASYNC) {
		result |= PVMFS_O_ASYNC;
		oflag &= ~O_ASYNC;
	}
#endif
#ifdef O_LARGEFILE
	if (oflag & O_LARGEFILE) {
		result |= PVMFS_O_LARGEFILE;
		oflag &= ~O_LARGEFILE;
	}
#endif

	dprint((buf2, "PVM oflag is %x\n", result));
	return oflag ? -1 : result;
}

mode_t
pvm_open_mode(mode_t mode)
{
	mode_t	result = 0;

	if (mode & S_IRUSR)	result |= PVMFS_S_IRUSR;
	if (mode & S_IRGRP)	result |= PVMFS_S_IRGRP;
	if (mode & S_IROTH)	result |= PVMFS_S_IROTH;
	if (mode & S_IWUSR)	result |= PVMFS_S_IWUSR;
	if (mode & S_IWGRP)	result |= PVMFS_S_IWGRP;
	if (mode & S_IWOTH)	result |= PVMFS_S_IWOTH;
	if (mode & S_IXUSR)	result |= PVMFS_S_IXUSR;
	if (mode & S_IXGRP)	result |= PVMFS_S_IXGRP;
	if (mode & S_IXOTH)	result |= PVMFS_S_IXOTH;

	return result;
}

int
pvm_lseek_whence(int whence)
{
	switch (whence) {
	case SEEK_SET:	return PVMFS_SEEK_SET;
	case SEEK_CUR:	return PVMFS_SEEK_CUR;
	case SEEK_END:	return PVMFS_SEEK_END;
	default:	return -1;
	}
}

int
pvm_fcntl_cmd(int command)
{
	switch (command) {
	case F_DUPFD:	return PVMFS_F_DUPFD;
	case F_GETFD:	return PVMFS_F_GETFD;
	case F_SETFD:	return PVMFS_F_SETFD;
	case F_GETFL:	return PVMFS_F_GETFL;
	case F_SETFL:	return PVMFS_F_SETFL;
	case F_GETOWN:	return PVMFS_F_GETOWN;
	case F_SETOWN:	return PVMFS_F_SETOWN;
	case F_GETLK:	return PVMFS_F_GETLK;
	case F_SETLK:	return PVMFS_F_SETLK;
	case F_SETLKW:	return PVMFS_F_SETLKW;
	default:	return -1;
	}
}

short
pvm_fcntl_ltype(short type)
{
	switch (type) {
	case F_RDLCK:	return PVMFS_F_RDLCK;
	case F_WRLCK:	return PVMFS_F_WRLCK;
	case F_UNLCK:	return PVMFS_F_UNLCK;
	default:	return -1;
	}
}

int
pvm_access_mode(int mode)
{
	int	result = 0;

	if (mode & R_OK)	result |= PVMFS_R_OK;
	if (mode & W_OK)	result |= PVMFS_W_OK;
	if (mode & X_OK)	result |= PVMFS_X_OK;
	if (mode & F_OK)	result |= PVMFS_F_OK;

	return result;
}

void
pvm_select_fdset_pack(int maxfdp1, fd_set *setp)
{
	int	numset = 0;
	int	numdone = 0;
	int	fd;

	if (!setp) {
		pvm_packf("%d", numset);
		dprint((buf2, "fdset_pack: packing %d fd's\n", numset));
		return;
	}

	for (fd = 0; fd < maxfdp1; fd++)
		if (FD_ISSET(fd, setp))
			numset++;

	pvm_packf("%d", numset);
	dprint((buf2, "fdset_pack: packing %d fd's\n", numset));

	for (fd = 0; fd < maxfdp1; fd++)
		if (FD_ISSET(fd, setp)) {
			pvm_packf("%d", fd);
			dprint((buf2, "fdset_pack: packing fd %d\n", fd));
			if (++numdone == numset)
				break;
		}
}

int
pvm_select_fdset_unpack(fd_set *setp)
{
	int	numset;
	int	fd;
	int	i;

	pvm_unpackf("%d", &numset);
	dprint((buf2, "fdset_unpack: unpacking %d fd's\n", numset));
	if (!setp)
		return;

	FD_ZERO(setp);
	for (i = 0; i < numset; i++) {
		pvm_unpackf("%d", &fd);
		dprint((buf2, "fdset_unpack: unpacking fd %d\n", fd));
		FD_SET(fd, setp);
	}

	return numset;
}

int
pvm_tcflow_action(int action)
{
	switch (action) {
	case TCOOFF:	return PVMFS_TCOOFF;
	case TCOON:	return PVMFS_TCOON;
	case TCIOFF:	return PVMFS_TCIOFF;
	case TCION:	return PVMFS_TCION;
	default:	return -1;
	}
}

int
pvm_tcflush_queue(int queue)
{
	switch (queue) {
	case TCIFLUSH:	return PVMFS_TCIFLUSH;
	case TCOFLUSH:	return PVMFS_TCOFLUSH;
	case TCIOFLUSH:	return PVMFS_TCIOFLUSH;
	default:	return -1;
	}
}

int
pvm_tcsetattr_opt(int opt)
{
	switch (opt) {
	case TCSANOW:	return PVMFS_TCSANOW;
	case TCSADRAIN:	return PVMFS_TCSADRAIN;
	case TCSAFLUSH:	return PVMFS_TCSAFLUSH;
	default:	return -1;
	}
}

long
pvm_tcattr_iflag(long iflag)
{
	long	result = 0;

	if (iflag & BRKINT)	result |= PVMFS_BRKINT;
	if (iflag & ICRNL)	result |= PVMFS_ICRNL;
	if (iflag & IGNBRK)	result |= PVMFS_IGNBRK;
	if (iflag & IGNCR)	result |= PVMFS_IGNCR;
	if (iflag & IGNPAR)	result |= PVMFS_IGNPAR;
#ifdef IMAXBEL
	if (iflag & IMAXBEL)	result |= PVMFS_IMAXBEL;
#endif
	if (iflag & INLCR)	result |= PVMFS_INLCR;
	if (iflag & INPCK)	result |= PVMFS_INPCK;
	if (iflag & ISTRIP)	result |= PVMFS_ISTRIP;
#ifdef IUCLC
	if (iflag & IUCLC)	result |= PVMFS_IUCLC;
#endif
#ifdef IXANY
	if (iflag & IXANY)	result |= PVMFS_IXANY;
#endif
	if (iflag & IXOFF)	result |= PVMFS_IXOFF;
	if (iflag & IXON)	result |= PVMFS_IXON;
	if (iflag & PARMRK)	result |= PVMFS_PARMRK;

	return result;
}

long
pvm_tcattr_oflag(long oflag)
{
	long	result = 0;

#ifdef BSDLY
	if (oflag & BSDLY)	result |= PVMFS_BSDLY;
#endif
#ifdef CRDLY
	if (oflag & CRDLY)	result |= PVMFS_CRDLY;
#endif
#ifdef FFDLY
	if (oflag & FFDLY)	result |= PVMFS_FFDLY;
#endif
#ifdef NLDLY
	if (oflag & NLDLY)	result |= PVMFS_NLDLY;
#endif
#ifdef OCRNL
	if (oflag & OCRNL)	result |= PVMFS_OCRNL;
#endif
#ifdef OFDEL
	if (oflag & OFDEL)	result |= PVMFS_OFDEL;
#endif
#ifdef OFILL
	if (oflag & OFILL)	result |= PVMFS_OFILL;
#endif
#ifdef OLCUC
	if (oflag & OLCUC)	result |= PVMFS_OLCUC;
#endif
#ifdef ONLCR
	if (oflag & ONLCR)	result |= PVMFS_ONLCR;
#endif
#ifdef ONLRET
	if (oflag & ONLRET)	result |= PVMFS_ONLRET;
#endif
#ifdef ONOCR
	if (oflag & ONOCR)	result |= PVMFS_ONOCR;
#endif
#ifdef ONOEOT
	if (oflag & ONOEOT)	result |= PVMFS_ONOEOT;
	if (oflag & OPOST)	result |= PVMFS_OPOST;
#endif
#ifdef OXTABS
	if (oflag & OXTABS)	result |= PVMFS_OXTABS;
#endif
#ifdef TABDLY
	if (oflag & TABDLY)	result |= PVMFS_TABDLY;
#endif
#ifdef VTDLY
	if (oflag & VTDLY)	result |= PVMFS_VTDLY;
#endif

	return result;
}

long
pvm_tcattr_cflag(long cflag)
{
	long	result = 0;

#ifdef CCTS_OFLOW
	if (cflag & CCTS_OFLOW)	result |= PVMFS_CCTS_OFLOW;
#endif
#ifdef CIGNORE
	if (cflag & CIGNORE)	result |= PVMFS_CIGNORE;
#endif
	if (cflag & CLOCAL)	result |= PVMFS_CLOCAL;
	if (cflag & CREAD)	result |= PVMFS_CREAD;
#ifdef CRTS_IFLOW
	if (cflag & CRTS_IFLOW)	result |= PVMFS_CRTS_IFLOW;
#endif
	if (cflag & CSIZE)	result |= PVMFS_CSIZE;
	if (cflag & CSTOPB)	result |= PVMFS_CSTOPB;
	if (cflag & HUPCL)	result |= PVMFS_HUPCL;
#ifdef MDMBUF
	if (cflag & MDMBUF)	result |= PVMFS_MDMBUF;
#endif
	if (cflag & PARENB)	result |= PVMFS_PARENB;
	if (cflag & PARODD)	result |= PVMFS_PARODD;

	return result;
}

long
pvm_tcattr_lflag(long lflag)
{
	long	result = 0;

#ifdef ALTWERASE
	if (lflag & ALTWERASE)	result |= PVMFS_ALTWERASE;
#endif
	if (lflag & ECHO)	result |= PVMFS_ECHO;
#ifdef ECHOCTL
	if (lflag & ECHOCTL)	result |= PVMFS_ECHOCTL;
#endif
	if (lflag & ECHOE)	result |= PVMFS_ECHOE;
	if (lflag & ECHOK)	result |= PVMFS_ECHOK;
#ifdef ECHOKE
	if (lflag & ECHOKE)	result |= PVMFS_ECHOKE;
#endif
	if (lflag & ECHONL)	result |= PVMFS_ECHONL;
#ifdef ECHOPRT
	if (lflag & ECHOPRT)	result |= PVMFS_ECHOPRT;
#endif
#ifdef FLUSHO
	if (lflag & FLUSHO)	result |= PVMFS_FLUSHO;
#endif
	if (lflag & ICANON)	result |= PVMFS_ICANON;
	if (lflag & IEXTEN)	result |= PVMFS_IEXTEN;
	if (lflag & ISIG)	result |= PVMFS_ISIG;
	if (lflag & NOFLSH)	result |= PVMFS_NOFLSH;
#ifdef NOKERNINFO
	if (lflag & NOKERNINFO)	result |= PVMFS_NOKERNINFO;
#endif
#ifdef PENDIN
	if (lflag & PENDIN)	result |= PVMFS_PENDIN;
#endif
	if (lflag & TOSTOP)	result |= PVMFS_TOSTOP;
#ifdef XCASE
	if (lflag & XCASE)	result |= PVMFS_XCASE;
#endif

	return result;
}

long
pvm_tcattr_speed(long speed)
{
	switch (speed) {
	case B0:	return PVMFS_B0;
	case B50:	return PVMFS_B50;
	case B75:	return PVMFS_B75;
	case B110:	return PVMFS_B110;
	case B134:	return PVMFS_B134;
	case B150:	return PVMFS_B150;
	case B200:	return PVMFS_B200;
	case B300:	return PVMFS_B300;
	case B600:	return PVMFS_B600;
	case B1200:	return PVMFS_B1200;
	case B1800:	return PVMFS_B1800;
	case B2400:	return PVMFS_B2400;
	case B4800:	return PVMFS_B4800;
	case B9600:	return PVMFS_B9600;
	case B19200:	return PVMFS_B19200;
	case B38400:	return PVMFS_B38400;
	default:	return -1;
	}
}

int
pvm_pathconf_name(int name)
{
	switch (name) {
	case _PC_LINK_MAX:		return PVMFS__PC_LINK_MAX;
	case _PC_MAX_CANON:		return PVMFS__PC_MAX_CANON;
	case _PC_MAX_INPUT:		return PVMFS__PC_MAX_INPUT;
	case _PC_NAME_MAX:		return PVMFS__PC_NAME_MAX;
	case _PC_PATH_MAX:		return PVMFS__PC_PATH_MAX;
	case _PC_PIPE_BUF:		return PVMFS__PC_PIPE_BUF;
	case _PC_CHOWN_RESTRICTED:	return PVMFS__PC_CHOWN_RESTRICTED;
	case _PC_NO_TRUNC:		return PVMFS__PC_NO_TRUNC;
	case _PC_VDISABLE:		return PVMFS__PC_VDISABLE;
	default:			return -1;
	}
}

int
pvm_rlimit_resource(int resource)
{
	switch (resource) {
	case RLIMIT_CORE:	return PVMFS_RLIMIT_CORE;
	case RLIMIT_FSIZE:	return PVMFS_RLIMIT_FSIZE;
#ifdef RLIMIT_OFILE
	case RLIMIT_OFILE:	return PVMFS_RLIMIT_OFILE;
#endif
#ifdef RLIMIT_NOFILE
	case RLIMIT_NOFILE:	return PVMFS_RLIMIT_NOFILE;
#endif
	default:		return -1;
	}
}

static int	defiomode = PvmIomodeIndividual;

int
pvm_setdefiomode(int mode)
{
	if (pvmmyfstid <= 0)
		return lpvmerr("pvm_setdefiomode", PvmNoFileServer);

	switch (mode) {
	case PvmIomodeIndividual:
	case PvmIomodeCommon:
	case PvmIomodeIndependent:
	case PvmIomodeSyncBC:
	case PvmIomodeSyncSeq:
		defiomode = mode;
		return 0;
	default:
		return lpvmerr("pvm_setdefiomode", PvmBadParam);
	}
}

int
pvm_getdefiomode(void)
{
	if (pvmmyfstid <= 0)
		return lpvmerr("pvm_getdefiomode", PvmNoFileServer);

	return defiomode;
}

int
pvm_setioorder(int order)
{
	int	cc;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_SETIOORDER;
	int	nargs = 1;

	if (pvmmyfstid <= 0)
		return lpvmerr("pvm_setioorder", PvmNoFileServer);

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, order);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return (cc == -1) ? lpvmerr("pvm_setioorder", PvmBadParam) : cc;
}

int
pvm_getioorder(void)
{
	int	cc;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_GETIOORDER;
	int	nargs = 0;

	if (pvmmyfstid <= 0)
		return lpvmerr("pvm_getioorder", PvmNoFileServer);

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d", PvmDataDefault, nargs);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return (cc == -1) ? lpvmerr("pvm_getioorder", PvmSysErr) : cc;
}

int
pvm_setiomode(int fd, int mode)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_SETIOMODE;
	int	nargs = 2;

	if (pvmmyfstid <= 0)
		return lpvmerr("pvm_setiomode", PvmNoFileServer);

	switch (mode) {
	case PvmIomodeIndividual:
	case PvmIomodeCommon:
	case PvmIomodeIndependent:
	case PvmIomodeSyncBC:
	case PvmIomodeSyncSeq:
		break;
	default:
		return PvmBadParam;
	}
	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %d", PvmDataDefault, nargs, fd, mode);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1)  {
		pvm_unpackf("%d", &pvmerrno);
		lpvmerr("pvm_setiomode", PvmBadParam);
		errno = pvm_local_errno(pvmerrno);
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
pvm_getiomode(int fd)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_GETIOMODE;
	int	nargs = 1;

	if (pvmmyfstid <= 0)
		return lpvmerr("pvm_getiomode", PvmNoFileServer);

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1)  {
		pvm_unpackf("%d", &pvmerrno);
		lpvmerr("pvm_getiomode", PvmBadParam);
		errno = pvm_local_errno(pvmerrno);
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
#if defined(__STDC__) || defined(IMA_CNVXN) || defined(IMA_RS6K)
open(const char *pathname, int oflag, ...)
#else
open(va_alist) va_dcl
#endif
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_OPEN;
	int	nargs = 3;
	va_list	ap;
	long	mode;

#if defined(__STDC__) || defined(IMA_CNVXN) || defined(IMA_RS6K)
	va_start(ap, oflag);
#else
	char	*pathname;
	int	oflag;

	va_start(ap);
	pathname = va_arg(ap, char *);
	oflag = va_arg(ap, int);
#endif

	if (pvmmyfstid <= 0) {
		if (oflag & O_CREAT) {
			if (sizeof(mode_t) < sizeof(int))
				mode = (long)va_arg(ap, int);
			else
				mode = (long)va_arg(ap, mode_t);
			cc = syscall(SYS_open, pathname, oflag, mode);
			if (cc == -1)
				dprintl((buf2,
			    "tid t%x open(\"%s\", %d, %ld) = %d (errno = %d)\n",
			    pvmmytid, pathname, oflag, (long)mode, cc, errno));
			else
				dprintl((buf2,
				  "tid t%x open(\"%s\", %d, %ld) = %d\n",
				  pvmmytid, pathname, oflag, (long)mode, cc));
		} else {
			cc = syscall(SYS_open, pathname, oflag);
			if (cc == -1)
				dprintl((buf2,
				 "tid t%x open(\"%s\", %d) = %d (errno = %d)\n",
				 pvmmytid, pathname, oflag, cc, errno));
			else
				dprintl((buf2,
				  "tid t%x open(\"%s\", %d) = %d\n",
				  pvmmytid, pathname, oflag, cc));
		}
		va_end(ap);
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	if ((oflag = pvm_open_oflag(oflag)) == -1) {
		errno = EINVAL;
		va_end(ap);
		dprint((buf2,
		  "tid t%x open(\"%s\", %d, ...) = -1 (errno = EINVAL)\n",
		  pvmmytid, pathname, oflag));
		return -1;
	}
	if (oflag & PVMFS_O_CREAT) {
		nargs = 4;
		if (sizeof(mode_t) < sizeof(int))
			mode = (long)va_arg(ap, int);
		else
			mode = (long)va_arg(ap, mode_t);
	}
	pvm_packf("%+ %d %s %d", PvmDataDefault, nargs, pathname, oflag);
	if (oflag & PVMFS_O_CREAT)
		pvm_packf("%ld", mode);
	pvm_packf("%d", defiomode);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		if (nargs == 3)
			dprint((buf2,
			  "tid t%x open(\"%s\", %d) = %d (pvmerrno = %d)\n",
			  pvmmytid, pathname, oflag, cc, pvmerrno));
		else
			dprint((buf2,
			 "tid t%x open(\"%s\", %d, %ld) = %d (pvmerrno = %d)\n",
			 pvmmytid, pathname, oflag, (long)mode, cc, pvmerrno));
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	if (nargs == 3)
		dprint((buf2, "tid t%x open(\"%s\", %d) = %d\n",
		  pvmmytid, pathname, oflag, cc));
	else
		dprint((buf2, "tid t%x open(\"%s\", %d, %ld) = %d\n",
		  pvmmytid, pathname, oflag, (long)mode, cc));
	va_end(ap);
	return cc;
}

#ifdef IMA_ALPHA

/*
 * Why does the ALPHA have this?
 * Should probably be integrated with the open above (and the __ap$open below).
 */
int
_open(const char *pathname, int oflag, ...)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_OPEN;
	int	nargs = 3;
	va_list	ap;
	long	mode;

	va_start(ap, oflag);

	if (pvmmyfstid <= 0) {
		if (oflag & O_CREAT) {
			if (sizeof(mode_t) < sizeof(int))
				mode = (long)va_arg(ap, int);
			else
				mode = (long)va_arg(ap, mode_t);
			cc = syscall(SYS_open, pathname, oflag, mode);
			if (cc == -1)
				dprintl((buf2,
			    "tid t%x open(\"%s\", %d, %ld) = %d (errno = %d)\n",
			    pvmmytid, pathname, oflag, (long)mode, cc, errno));
			else
				dprintl((buf2,
				  "tid t%x open(\"%s\", %d, %ld) = %d\n",
				  pvmmytid, pathname, oflag, (long)mode, cc));
		} else {
			cc = syscall(SYS_open, pathname, oflag);
			if (cc == -1)
				dprintl((buf2,
				 "tid t%x open(\"%s\", %d) = %d (errno = %d)\n",
				 pvmmytid, pathname, oflag, cc, errno));
			else
				dprintl((buf2,
				  "tid t%x open(\"%s\", %d) = %d\n",
				  pvmmytid, pathname, oflag, cc));
		}
		va_end(ap);
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	if ((oflag = pvm_open_oflag(oflag)) == -1) {
		errno = EINVAL;
		va_end(ap);
		dprint((buf2,
		  "tid t%x open(\"%s\", %d, ...) = -1 (errno = EINVAL)\n",
		  pvmmytid, pathname, oflag));
		return -1;
	}
	if (oflag & PVMFS_O_CREAT) {
		nargs = 4;
		if (sizeof(mode_t) < sizeof(int))
			mode = (long)va_arg(ap, int);
		else
			mode = (long)va_arg(ap, mode_t);
	}
	pvm_packf("%+ %d %s %d", PvmDataDefault, nargs, pathname, oflag);
	if (oflag & PVMFS_O_CREAT)
		pvm_packf("%ld", mode);
	pvm_packf("%d", defiomode);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		if (nargs == 3)
			dprint((buf2,
			  "tid t%x open(\"%s\", %d) = %d (pvmerrno = %d)\n",
			  pvmmytid, pathname, oflag, cc, pvmerrno));
		else
			dprint((buf2,
			 "tid t%x open(\"%s\", %d, %ld) = %d (pvmerrno = %d)\n",
			 pvmmytid, pathname, oflag, (long)mode, cc, pvmerrno));
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	if (nargs == 3)
		dprint((buf2, "tid t%x open(\"%s\", %d) = %d\n",
		  pvmmytid, pathname, oflag, cc));
	else
		dprint((buf2, "tid t%x open(\"%s\", %d, %ld) = %d\n",
		  pvmmytid, pathname, oflag, (long)mode, cc));
	va_end(ap);
	return cc;
}
#endif

int
creat(const char *pathname, mode_t mode)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_CREAT;
	int	nargs = 3;
	long	lmode = pvm_open_mode(mode);

	if (pvmmyfstid <= 0) {
#ifdef SYS_creat
		cc = syscall(SYS_creat, pathname, mode);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x creat(\"%s\", %ld) = %d (errno = %d)\n",
			  pvmmytid, pathname, (long)mode, cc, errno));
		else
			dprintl((buf2, "tid t%x creat(\"%s\", %ld) = %d\n",
			  pvmmytid, pathname, (long)mode, cc));
#else
		cc = syscall(SYS_open, pathname, O_WRONLY | O_CREAT | O_TRUNC,
		  mode);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x creat(\"%s\", %ld) = %d (errno = %d)\n",
			  pvmmytid, pathname, (long)mode, cc, errno));
		else
			dprintl((buf2, "tid t%x creat(\"%s\", %ld) = %d\n",
			  pvmmytid, pathname, (long)mode, cc));
#endif
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %ld %d", PvmDataDefault, nargs, pathname, lmode,
	  defiomode);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x creat(\"%s\", %ld) = %d (pvmerrno = %d)\n", pvmmytid,
		  pathname, (long)mode, cc, pvmerrno));
	} else
		dprint((buf2, "tid t%x creat(\"%s\", %ld) = %d\n",
		  pvmmytid, pathname, (long)mode, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
close(int fd)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_CLOSE;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_close, fd);
		if (cc == -1)
			dprintl((buf2, "tid t%x close(%d) = %d (errno = %d)\n",
			  pvmmytid, fd, cc, errno));
		else
			dprintl((buf2, "tid t%x close(%d) = %d\n",
			  pvmmytid, fd, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x close(%d) = %d (pvmerrno = %d)\n",
		  pvmmytid, fd, cc, pvmerrno));
	} else
		dprint((buf2, "tid t%x close(%d) = %d\n",
		  pvmmytid, fd, cc));

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

off_t
lseek(int fd, off_t offset, int whence)
{
	long	lcc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_LSEEK;
	int	nargs = 3;
	int	fswhence = pvm_lseek_whence(whence);
	long	loffset = (long)offset;

	if (pvmmyfstid <= 0) {
		lcc = (off_t)syscall(SYS_lseek, fd, offset, whence);
		if (lcc == -1L)
			dprintl((buf2,
			  "tid t%x lseek(%d, %ld, %d) = %ld (errno = %d)\n",
			  pvmmytid, fd, (long)offset, whence, lcc, errno));
		else
			dprintl((buf2, "tid t%x lseek(%d, %ld, %d) = %ld\n",
			  pvmmytid, fd, (long)offset, whence, lcc));
		return lcc;
	}

	if (fswhence == -1) {
		errno = EINVAL;
		dprint((buf2,
		  "tid t%x lseek(%d, %ld, %d) = -1 (errno = EINVAL)\n",
		  pvmmytid, fd, (long)offset, whence));
		return (off_t)-1;
	}
	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %ld %d", PvmDataDefault, nargs, fd, loffset,
	  fswhence);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%ld", &lcc);
	if (lcc == -1L) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x lseek(%d, %ld, %d) = %ld (pvmerrno = %d)\n",
		  pvmmytid, fd, (long)offset, whence, lcc, pvmerrno));
	} else
		dprint((buf2, "tid t%x lseek(%d, %ld, %d) = %d\n",
		  pvmmytid, fd, (long)offset, whence, lcc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return (off_t)lcc;
}

READ_RETURN_TYPE
read(int fd, READ_ARG2_TYPE buf, READ_ARG3_TYPE nbytes)
{
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_READ;
	int	nargs = 2;
	long	lnbytes = (long)nbytes;

	if (pvmmyfstid <= 0) {
		lnbytes = (long)syscall(SYS_read, fd, buf, nbytes);
		if (lnbytes == -1L)
			dprintl((buf2,
			  "tid t%x read(%d, %p, %ld) = %ld (errno = %d)\n",
			  pvmmytid, fd, buf, (long)nbytes, lnbytes, errno));
		else
			dprintl((buf2, "tid t%x read(%d, %p, %ld) = %ld\n",
			  pvmmytid, fd, buf, (long)nbytes, lnbytes));
		return (READ_RETURN_TYPE)lnbytes;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %ld", PvmDataDefault, nargs, fd, lnbytes);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%ld", &lnbytes);
	if (lnbytes == -1L) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x read(%d, %p, %ld) = %ld (errno = %d)\n", pvmmytid,
		  fd, buf, (long)nbytes, lnbytes, errno));
	} else {
		if (lnbytes > 0)
			pvm_unpackf("%*c", (int)lnbytes, (char *)buf);
		dprint((buf2, "tid t%x read(%d, %p, %ld) = %ld\n",
		  pvmmytid, fd, buf, (long)nbytes, lnbytes));
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return (READ_RETURN_TYPE)lnbytes;
}

READV_RETURN_TYPE
readv(int fd, READV_CONST struct iovec *iovp, int count)
{
	READ_RETURN_TYPE	scc;
	int			pvmerrno;
	int			rbuf;
	int			sbuf;
	int			request = PVMFS_READV;
	int			nargs = 2;
	int			i;
	long			nbytes;
	long			left;
	long			nread;

	if (pvmmyfstid <= 0) {
		scc = (READ_RETURN_TYPE)syscall(SYS_readv, fd, iovp, count);
		if (scc == -1)
			dprintl((buf2,
			  "tid t%x readv(%d, %p, %d) = %ld (errno = %d)\n",
			  pvmmytid, fd, iovp, count, (long)scc, errno));
		else
			dprintl((buf2, "tid t%x readv(%d, %p, %d) = %ld\n",
			  pvmmytid, fd, iovp, count, (long)scc));
		return (READV_RETURN_TYPE)scc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));

	nbytes = 0;
	for (i = 0; i < count; i++)
		nbytes += iovp[i].iov_len;
	pvm_packf("%+ %d %d %ld", PvmDataDefault, nargs, fd, nbytes);
	for (i = 0; i < count; i++) {
		nbytes = iovp[i].iov_len;
		pvm_packf("%ld", nbytes);
	}
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%ld", &nbytes);
	if (nbytes == -1L) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x readv(%d, %p, %d) = %ld (errno = %d)\n",
		  pvmmytid, fd, iovp, count, nbytes, errno));
	} else {
		for (left = nbytes, i = 0; left && i < count;
		  left -= nread, i++)
			if (nread = MIN(left, iovp[i].iov_len)) {
				pvm_unpackf("%*c", (int)nread,
				  (char *)iovp[i].iov_base);
				dprint((buf2,
				  "tid t%x unpacked \"%*.*s\" at %p\n",
				  pvmmytid, (int)nread, (int)nread,
				  (char *)iovp[i].iov_base,
				  (char *)iovp[i].iov_base));
			}
		if (left) {
			/* shouldn't happen */
			errno = 0;
			return -1;
		}
		dprint((buf2, "tid t%x readv(%d, %p, %d) = %ld\n",
		  pvmmytid, fd, iovp, count, nbytes));
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return (READV_RETURN_TYPE)nbytes;
}

WRITE_RETURN_TYPE
write(int fd, WRITE_ARG2_TYPE buf, WRITE_ARG3_TYPE nbytes)
{
	WRITE_RETURN_TYPE	scc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_WRITE;
	int	nargs = 3;
	long	lnbytes = (long)nbytes;

#ifdef TRACE
int olddebug = pvmdebug;
#endif /* TRACE */
	if (pvmmyfstid <= 0) {
		scc = (WRITE_RETURN_TYPE)syscall(SYS_write, fd, buf, nbytes);
		if (scc == -1)
#if 0
			dprintl((buf2, "tid t%x write(%d, %p, %ld) = -1"
			  " (errno = %d) \"%*.*s\"\n", pvmmytid, fd, buf,
			  (long)nbytes, errno, (int)nbytes, (int)nbytes,
	  (isprint(((char *)buf)[0]) && isprint(((char *)buf)[1])) ? buf : ""));
#else
			dprintl((buf2,
			  "tid t%x write(%d, %p, %ld) = -1 (errno = %d)\n",
			  pvmmytid, fd, buf, (long)nbytes, errno));
#endif
		else
#if 0
			dprintl((buf2,
			  "tid t%x write(%d, %p, %ld) = %ld \"%*.*s\"\n",
			  pvmmytid, fd, buf, (long)nbytes, scc, (int)nbytes,
			  (int)nbytes,
	  (isprint(((char *)buf)[0]) && isprint(((char *)buf)[1])) ? buf : ""));
#else
			dprintl((buf2,
			  "tid t%x write(%d, %p, %ld) = %ld\n",
			  pvmmytid, fd, buf, (long)nbytes, scc));
#endif
		return (WRITE_RETURN_TYPE)scc;
	}
#ifdef TRACE
pvmdebug = 0;
#endif /* TRACE */

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %ld %*c", PvmDataDefault, nargs, fd, lnbytes,
	  (int)lnbytes, (char *)buf);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%ld", &lnbytes);
	if (lnbytes == -1L) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
#if 0
		dprint((buf2,
		  "tid t%x write(%d, %p, %ld) = -1 (pvmerrno = %d) \"%*.*s\"\n",
		  pvmmytid, fd, buf, (long)nbytes, pvmerrno, nbytes, nbytes,
	  (isprint(((char *)buf)[0]) && isprint(((char *)buf)[1])) ? buf : ""));
#else
		dprint((buf2,
		  "tid t%x write(%d, %p, %ld) = -1 (pvmerrno = %d)\n", pvmmytid,
		  fd, buf, (long)nbytes, pvmerrno));
#endif
	} else
#if 0
		dprint((buf2, "tid t%x write(%d, %p, %ld) = %ld \"%*.*s\"\n",
		  pvmmytid, fd, buf, (long)nbytes, lnbytes, nbytes, nbytes, ""));
	  (isprint(((char *)buf)[0]) && isprint(((char *)buf)[1])) ? buf : ""));
#else
		dprint((buf2, "tid t%x write(%d, %p, %ld) = %ld\n",
		  pvmmytid, fd, buf, (long)nbytes, lnbytes));
#endif
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

#ifdef TRACE
pvmdebug = olddebug;
#endif /* TRACE */
	return (WRITE_RETURN_TYPE)lnbytes;
}

WRITEV_RETURN_TYPE
writev(int fd, WRITEV_CONST struct iovec *iovp, int count)
{
	WRITE_RETURN_TYPE	scc;
	int			pvmerrno;
	int			rbuf;
	int			sbuf;
	int			request = PVMFS_WRITEV;
	int			nargs = 3;
	long			nbytes;
	int			i;
#ifdef TRACE
int olddebug = pvmdebug;
pvmdebug = 0;
#endif /* TRACE */

	if (pvmmyfstid <= 0) {
		scc = (WRITE_RETURN_TYPE)syscall(SYS_writev, fd, iovp, count);
		if (scc == -1)
			dprintl((buf2,
			  "tid t%x writev(%d, %p, %d) = -1 (errno = %d)\n",
			  pvmmytid, fd, iovp, count, errno));
		else
			dprintl((buf2,
			  "tid t%x writev(%d, %p, %d) = %ld\n",
			  pvmmytid, fd, iovp, count, (long)scc));
#ifdef TRACE
pvmdebug = olddebug;
#endif /* TRACE */
		return (WRITEV_RETURN_TYPE)scc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));

	nbytes = 0;
	for (i = 0; i < count; i++)
		nbytes += iovp[i].iov_len;
	pvm_packf("%+ %d %d %ld", PvmDataDefault, nargs, fd, nbytes);
	for (i = 0; i < count; i++) {
		nbytes = iovp[i].iov_len;
		pvm_packf("%ld %*c", nbytes, (int)iovp[i].iov_len,
		  (char *)iovp[i].iov_base);
	}
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%ld", &nbytes);
	if (nbytes == -1L) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x writev(%d, %p, %d) = -1 (pvmerrno = %d)\n", pvmmytid,
		  fd, iovp, count, pvmerrno));
	} else
		dprint((buf2, "tid t%x writev(%d, %p, %d) = %ld\n",
		  pvmmytid, fd, iovp, count, nbytes));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

#ifdef TRACE
pvmdebug = olddebug;
#endif /* TRACE */
	return (WRITEV_RETURN_TYPE)nbytes;
}

int
dup(int fd)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_DUP;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_dup, fd);
		if (cc == -1)
			dprintl((buf2, "tid t%x dup(%d) = %d (errno = %d)\n",
			  pvmmytid, fd, cc, errno));
		else
			dprintl((buf2, "tid t%x dup(%d) = %d\n",
			  pvmmytid, fd, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x dup(%d) = %d (pvmerrno = %d)\n",
		  pvmmytid, fd, cc, pvmerrno));
	} else
		dprint((buf2, "tid t%x dup(%d) = %d\n",
		  pvmmytid, fd, cc));

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
dup2(int fd1, int fd2)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_DUP2;
	int	nargs = 2;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_dup2, fd1, fd2);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x dup2(%d, %d) = %d (errno = %d)\n",
			  pvmmytid, fd1, fd2, cc, errno));
		else
			dprintl((buf2, "tid t%x dup2(%d, %d) = %d\n",
			  pvmmytid, fd1, fd2, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %d", PvmDataDefault, nargs, fd1, fd2);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x dup2(%d, %d) = %d (pvmerrno = %d)\n",
		  pvmmytid, fd1, fd2, cc, pvmerrno));
	} else
		dprint((buf2, "tid t%x dup2(%d, %d) = %d\n",
		  pvmmytid, fd1, fd2, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

static void
fillstat(struct stat *stp)
{
	long	l;

#ifdef __STDC__
#define get(mem)	pvm_unpackf("%ld", &l); stp->st_##mem = l
	get(dev); get(ino); get(mode); get(nlink); get(uid); get(gid);
	get(rdev); get(size); get(atime); get(ctime); get(mtime); get(blksize);
	get(blocks);
#undef get
#else
	pvm_unpackf("%ld", &l);	stp->st_dev = l;
	pvm_unpackf("%ld", &l);	stp->st_ino = l;
	pvm_unpackf("%ld", &l);	stp->st_mode = l;
	pvm_unpackf("%ld", &l);	stp->st_nlink = l;
	pvm_unpackf("%ld", &l);	stp->st_uid = l;
	pvm_unpackf("%ld", &l);	stp->st_gid = l;
	pvm_unpackf("%ld", &l);	stp->st_rdev = l;
	pvm_unpackf("%ld", &l);	stp->st_size = l;
	pvm_unpackf("%ld", &l);	stp->st_atime = l;
	pvm_unpackf("%ld", &l);	stp->st_ctime = l;
	pvm_unpackf("%ld", &l);	stp->st_mtime = l;
	pvm_unpackf("%ld", &l);	stp->st_blksize = l;
	pvm_unpackf("%ld", &l);	stp->st_blocks = l;
#endif
}

int
stat(const char *pathname, struct stat *stp)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_STAT;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_stat, pathname, stp);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x stat(\"%s\", %p) = %d (errno = %d)\n",
			  pvmmytid, pathname, (void *)stp, cc, errno));
		else
			dprintl((buf2, "tid t%x stat(\"%s\", %p) = %d\n",
			  pvmmytid, pathname, (void *)stp, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s", PvmDataDefault, nargs, pathname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x stat(\"%s\", %p) = %d (pvmerrno = %d)\n",
		  pvmmytid, pathname, (void *)stp, cc, pvmerrno));
	} else {
		dprint((buf2, "tid t%x stat(\"%s\", %p) = %d\n",
		  pvmmytid, pathname, (void *)stp, cc));
		fillstat(stp);
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
lstat(const char *pathname, struct stat *stp)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_LSTAT;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_lstat, pathname, stp);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x lstat(\"%s\", %p) = %d (errno = %d)\n",
			  pvmmytid, pathname, (void *)stp, cc, errno));
		else
			dprintl((buf2, "tid t%x lstat(\"%s\", %p) = %d\n",
			  pvmmytid, pathname, (void *)stp, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s", PvmDataDefault, nargs, pathname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x lstat(\"%s\", %p) = %d (pvmerrno = %d)\n", pvmmytid,
		  pathname, (void *)stp, cc, pvmerrno));
	} else {
		dprint((buf2, "tid t%x lstat(\"%s\", %p) = %d\n",
		  pvmmytid, pathname, (void *)stp, cc));
		fillstat(stp);
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
fstat(int fd, struct stat *stp)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_FSTAT;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_fstat, fd, stp);
if (fd < 20) {
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x fstat(%d, %p) = %d (errno = %d)\n",
			    pvmmytid, fd, (void *)stp, cc, errno));
		else
			dprintl((buf2, "tid t%x fstat(%d, %p) = %d\n",
			  pvmmytid, fd, (void *)stp, cc));
}
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x fstat(%d, %p) = %d (pvmerrno = %d)\n", pvmmytid, fd,
		  (void *)stp, cc, pvmerrno));
	} else {
		dprint((buf2, "tid t%x fstat(%d, %p) = %d\n",
		  pvmmytid, fd, (void *)stp, cc));
		fillstat(stp);
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
#if defined(__STDC__) || defined(IMA_CNVXN) || defined(IMA_RS6K)
fcntl(int fd, int command, ...)
#else
fcntl(va_alist) va_dcl
#endif
{
	va_list		ap;
	int		iarg = 0;
	struct flock	*lockp;
	int		cc;
	int		pvmerrno;
	int		rbuf;
	int		sbuf;
	int		request = PVMFS_FCNTL;
	long		l1, l2;
	int		nargs = 3;

#if defined(__STDC__) || defined(IMA_CNVXN) || defined(IMA_RS6K)
	va_start(ap, command);
#else
	int	fd;
	int	command;

	va_start(ap);
	fd = va_arg(ap, int);
	command = va_arg(ap, int);
#endif

	if (pvmmyfstid <= 0) {
		switch (command) {
		case F_GETLK:
		case F_SETLK:
		case F_SETLKW:
			lockp = va_arg(ap, struct flock *);
			cc = syscall(SYS_fcntl, fd, command, lockp);
			if (cc == -1)
				dprintl((buf2,
				"tid t%x fcntl(%d, %d, %p) = -1 (errno = %d)\n",
				pvmmytid, fd, command, (void *)lockp, errno));
			else
				dprintl((buf2,
				  "tid t%x fcntl(%d, %d, %p) = %d\n",
				  pvmmytid, fd, command, (void *)lockp, cc));
			break;
		default:
			iarg = va_arg(ap, int);
			cc = syscall(SYS_fcntl, fd, command, iarg);
			if (cc == -1)
				dprintl((buf2,
				"tid t%x fcntl(%d, %d, %d) = -1 (errno = %d)\n",
				pvmmytid, fd, command, iarg, errno));
			else
				dprintl((buf2,
				  "tid t%x fcntl(%d, %d, %d) = %d\n",
				  pvmmytid, fd, command, iarg, cc));
			break;
		}
		va_end(ap);
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));

	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);

	switch (command) {
	case F_GETLK:
	case F_SETLK:
	case F_SETLKW:
		lockp = va_arg(ap, struct flock *);
		if ((lockp->l_type = pvm_fcntl_ltype(lockp->l_type)) == -1L
		  || (lockp->l_whence = pvm_lseek_whence(lockp->l_whence))
		  == -1L) {
			errno = EINVAL;
			va_end(ap);
			return -1;
		}
		command = pvm_fcntl_cmd(command);
		pvm_packf("%d %d %ld %d %ld", command, lockp->l_type,
		  (long)lockp->l_start, lockp->l_whence, (long)lockp->l_len);
		break;
	case F_DUPFD:
	case F_SETFD:
	case F_SETFL:
	case F_SETOWN:
		iarg = va_arg(ap, int);
		if (command == F_SETFL)
			iarg = pvm_open_oflag(iarg);
		/* FALLTHROUGH */
	default:
		command = pvm_fcntl_cmd(command);
		pvm_packf("%d %d", command, iarg);
		break;
	}
	va_end(ap);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		switch (command) {
		case F_GETLK:
		case F_SETLK:
		case F_SETLKW:
			dprint((buf2,
			  "tid t%x fcntl(%d, %d, %p) = -1 (pvmerrno = %d)\n",
			  pvmmytid, fd, command, (void *)lockp, pvmerrno));
			break;
		default:
			dprint((buf2,
			  "tid t%x fcntl(%d, %d, %d) = -1 (pvmerrno = %d)\n",
			  pvmmytid, fd, command, iarg, pvmerrno));
			break;
		}
	} else
		switch (command) {
		case F_GETLK:
			pvm_unpackf("%hd", &lockp->l_type);
			lockp->l_type =
			  (short)pvm_local_fcntl_ltype(lockp->l_type);
			if (lockp->l_type != F_UNLCK) {
				pvm_unpackf("%ld %hd %ld", &l1,
				  &lockp->l_whence, &l2);
				lockp->l_start = (off_t)l1;
				lockp->l_whence = (short)
				  pvm_local_lseek_whence(lockp->l_whence);
				lockp->l_len = (off_t)l2;
			}
		case F_SETLK:
		case F_SETLKW:
			dprint((buf2, "tid t%x fcntl(%d, %d, %p) = %d\n",
			  pvmmytid, fd, command, (void *)lockp, cc));
			break;
		case F_GETFL:
			cc = pvm_local_open_oflag(cc);
		default:
			dprint((buf2, "tid t%x fcntl(%d, %d, %d) = %d\n",
			  pvmmytid, fd, command, iarg, cc));
			break;
		}

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
unlink(const char *pathname)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_UNLINK;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_unlink, pathname);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x unlink(\"%s\") = -1 (errno = %d)\n",
			  pvmmytid, pathname, errno));
		else
			dprintl((buf2, "tid t%x unlink(\"%s\") = %d\n",
			  pvmmytid, pathname, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s", PvmDataDefault, nargs, pathname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x unlink(\"%s\") = -1 (pvmerrno = %d)\n",
		  pvmmytid, pathname, pvmerrno));
	} else
		dprint((buf2, "tid t%x unlink(\"%s\") = %d\n",
		  pvmmytid, pathname, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
rename(const char *oldpathname, const char *newpathname)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_RENAME;
	int	nargs = 2;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_rename, oldpathname, newpathname);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x rename(\"%s\", \"%s\") = -1 (errno = %d)\n",
			  pvmmytid, oldpathname, newpathname, errno));
		else
			dprintl((buf2, "tid t%x rename(\"%s\", \"%s\") = %d\n",
			  pvmmytid, oldpathname, newpathname, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %s", PvmDataDefault, nargs, oldpathname,
	  newpathname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x rename(\"%s\", \"%s\") = -1 (pvmerrno = %d)\n",
		  pvmmytid, oldpathname, newpathname, pvmerrno));
	} else
		dprint((buf2, "tid t%x rename(\"%s\", \"%s\") = %d\n",
		  pvmmytid, oldpathname, newpathname, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
mkdir(const char *pathname, mode_t mode)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_MKDIR;
	int	nargs = 2;
	long	lmode = pvm_open_mode(mode);

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_mkdir, pathname, mode);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x mkdir(\"%s\") = -1 (errno = %d)\n", pvmmytid,
			  pathname, errno));
		else
			dprintl((buf2, "tid t%x mkdir(\"%s\") = %d\n",
			  pvmmytid, pathname, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %ld", PvmDataDefault, nargs, pathname, lmode);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x mkdir(\"%s\", mode) = -1 (pvmerrno = %d)\n",
		  pvmmytid, pathname, (long)mode, pvmerrno));
	} else
		dprint((buf2, "tid t%x mkdir(\"%s\", %ld) = %d\n",
		  pvmmytid, pathname, (long)mode, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
rmdir(const char *pathname)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_RMDIR;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_rmdir, pathname);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x rmdir(\"%s\") = -1 (errno = %d)\n", pvmmytid,
			  pathname, errno));
		else
			dprintl((buf2, "tid t%x rmdir(\"%s\") = %d\n",
			  pvmmytid, pathname, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s", PvmDataDefault, nargs, pathname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x rmdir(\"%s\") = -1 (pvmerrno = %d)\n",
		  pvmmytid, pathname, pvmerrno));
	} else
		dprint((buf2, "tid t%x rmdir(\"%s\") = %d\n",
		  pvmmytid, pathname, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
chroot(const char *pathname)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_CHROOT;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_chroot, pathname);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x chroot(\"%s\") = -1 (errno = %d)\n",
			  pvmmytid, pathname, errno));
		else
			dprintl((buf2, "tid t%x chroot(\"%s\") = %d\n",
			  pvmmytid, pathname, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s", PvmDataDefault, nargs, pathname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x chroot(\"%s\") = -1 (pvmerrno = %d)\n",
		  pvmmytid, pathname, pvmerrno));
	} else
		dprint((buf2, "tid t%x chroot(\"%s\") = %d\n",
		  pvmmytid, pathname, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
fchroot(int fd)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_FCHDIR;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
#ifdef SYS_fchroot
		cc = syscall(SYS_fchroot, fd);
#else
		cc = -1;
		errno = ENOSYS;
#endif
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x fchroot(%d) = -1 (errno = %d)\n",
			  pvmmytid, fd, errno));
		else
			dprintl((buf2, "tid t%x fchroot(%d) = %d\n",
			  pvmmytid, fd, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x fchroot(%d) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, pvmerrno));
	} else
		dprint((buf2, "tid t%x fchroot(%d) = %d\n", pvmmytid, fd, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
chdir(const char *pathname)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_CHDIR;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_chdir, pathname);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x chdir(\"%s\") = -1 (errno = %d)\n", pvmmytid,
			  pathname, errno));
		else
			dprintl((buf2, "tid t%x chdir(\"%s\") = %d\n",
			  pvmmytid, pathname, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s", PvmDataDefault, nargs, pathname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x chdir(\"%s\") = -1 (pvmerrno = %d)\n",
		  pvmmytid, pathname, pvmerrno));
	} else
		dprint((buf2, "tid t%x chdir(\"%s\") = %d\n",
		  pvmmytid, pathname, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
fchdir(int fd)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_FCHDIR;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
#ifdef SYS_fchdir
		cc = syscall(SYS_fchdir, fd);
#else
		cc = -1;
		errno = ENOSYS;
#endif
		if (cc == -1)
			dprintl((buf2, "tid t%x fchdir(%d) = -1 (errno = %d)\n",
			  pvmmytid, fd, errno));
		else
			dprintl((buf2, "tid t%x fchdir(%d) = %d\n",
			  pvmmytid, fd, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x fchdir(%d) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, pvmerrno));
	} else
		dprint((buf2, "tid t%x fchdir(%d) = %d\n", pvmmytid, fd, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
chmod(const char *pathname, mode_t mode)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_CHMOD;
	int	nargs = 2;
	long	lmode = pvm_open_mode(mode);

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_chmod, pathname, mode);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x chmod(\"%s\", %ld) = -1 (errno = %d)\n",
			  pvmmytid, pathname, (long)mode,
			  errno));
		else
			dprintl((buf2, "tid t%x chmod(\"%s\", %ld) = %d\n",
			  pvmmytid, pathname, (long)mode, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %ld", PvmDataDefault, nargs, pathname, lmode);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x chmod(\"%s\", %ld) = -1 (pvmerrno = %d)\n", pvmmytid,
		  pathname, (long)mode, pvmerrno));
	} else
		dprint((buf2, "tid t%x chmod(\"%s\", %ld) = %d\n",
		  pvmmytid, pathname, (long)mode, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

#ifndef IMA_RS6K
int
fchmod(int fd, mode_t mode)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_FCHMOD;
	int	nargs = 2;
	long	lmode = pvm_open_mode(mode);

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_fchmod, fd, mode);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x fchmod(%d, %ld) = -1 (errno = %d)\n",
			  pvmmytid, fd, (long)mode, errno));
		else
			dprintl((buf2, "tid t%x fchmod(%d, %ld) = %d\n",
			  pvmmytid, fd, (long)mode, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %ld", PvmDataDefault, nargs, fd, lmode);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x fchmod(%d, %ld) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, (long)mode, pvmerrno));
	} else
		dprint((buf2, "tid t%x fchmod(%d, %ld) = %d\n", pvmmytid, fd,
		  (long)mode, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}
#endif

int
chown(const char *pathname, uid_t uid, gid_t gid)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_CHOWN;
	int	nargs = 3;
	long	luid = uid;
	long	lgid = gid;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_chown, pathname, uid, gid);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x chown(\"%s\", %ld, %ld) = -1 (errno = %d)\n",
			  pvmmytid, pathname, (long)uid, (long)gid, errno));
		else
			dprintl((buf2, "tid t%x chown(\"%s\", %ld, %ld) = %d\n",
			  pvmmytid, pathname, (long)uid, (long)gid, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %ld %ld", PvmDataDefault, nargs, pathname, luid,
	  lgid);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x chown(\"%s\", %ld, %ld) = -1 (pvmerrno = %d)\n",
		  pvmmytid, pathname, (long)uid, (long)gid, pvmerrno));
	} else
		dprint((buf2, "tid t%x chown(\"%s\", %ld, %ld) = %d\n",
		  pvmmytid, pathname, (long)uid, (long)gid, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
fchown(int fd, uid_t uid, gid_t gid)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_FCHOWN;
	int	nargs = 3;
	long	luid = uid;
	long	lgid = gid;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_fchown, fd, uid, gid);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x fchown(%d, %ld, %ld) = -1 (errno = %d)\n",
			  pvmmytid, fd, (long)uid, (long)gid, errno));
		else
			dprintl((buf2, "tid t%x fchown(%d, %ld, %ld) = %d\n",
			  pvmmytid, fd, (long)uid, (long)gid, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %ld %ld", PvmDataDefault, nargs, fd, luid, lgid);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x fchown(%d, %ld, %ld) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, (long)uid, (long)gid, pvmerrno));
	} else
		dprint((buf2, "tid t%x fchown(%d, %ld, %ld) = %d\n",
		  pvmmytid, fd, (long)uid, (long)gid, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
lchown(const char *pathname, uid_t uid, gid_t gid)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_LCHOWN;
	int	nargs = 3;
	long	luid = uid;
	long	lgid = gid;

	if (pvmmyfstid <= 0) {
#ifdef SYS_lchown
		cc = syscall(SYS_lchown, pathname, uid, gid);
#else
		cc = -1;
		errno = ENOSYS;
#endif
		if (cc == -1)
			dprintl((buf2,
			 "tid t%x lchown(\"%s\", %ld, %ld) = -1 (errno = %d)\n",
			 pvmmytid, pathname, (long)uid, (long)gid, errno));
		else
			dprintl((buf2,
			  "tid t%x lchown(\"%s\", %ld, %ld) = %d\n",
			  pvmmytid, pathname, (long)uid, (long)gid, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %ld %ld", PvmDataDefault, nargs, pathname, luid,
	  lgid);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x lchown(\"%s\", %ld, %ld) = -1 (pvmerrno = %d)\n",
		  pvmmytid, pathname, (long)uid, (long)gid, pvmerrno));
	} else
		dprint((buf2, "tid t%x lchown(\"%s\", %ld, %ld) = %d\n",
		  pvmmytid, pathname, (long)uid, (long)gid, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
link(const char *oldpathname, const char *newpathname)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_LINK;
	int	nargs = 2;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_link, oldpathname, newpathname);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x link(\"%s\", \"%s\") = -1 (errno = %d)\n",
			  pvmmytid, oldpathname, newpathname, errno));
		else
			dprintl((buf2, "tid t%x link(\"%s\", \"%s\") = %d\n",
			  pvmmytid, oldpathname, newpathname, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %s", PvmDataDefault, nargs, oldpathname,
	  newpathname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x link(\"%s\", \"%s\") = -1 (pvmerrno = %d)\n",
		  pvmmytid, oldpathname, newpathname, pvmerrno));
	} else
		dprint((buf2, "tid t%x link(\"%s\", \"%s\") = %d\n",
		  pvmmytid, oldpathname, newpathname, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
symlink(const char *oldpathname, const char *newpathname)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_SYMLINK;
	int	nargs = 2;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_symlink, oldpathname, newpathname);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x symlink(\"%s\", \"%s\") = -1 (errno = %d)\n",
			  pvmmytid, oldpathname, newpathname, errno));
		else
			dprintl((buf2, "tid t%x symlink(\"%s\", \"%s\") = %d\n",
			  pvmmytid, oldpathname, newpathname, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %s", PvmDataDefault, nargs, oldpathname,
	  newpathname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x symlink(\"%s\", \"%s\") = -1 (pvmerrno = %d)\n",
		  pvmmytid, oldpathname, newpathname, pvmerrno));
	} else
		dprint((buf2, "tid t%x symlink(\"%s\", \"%s\") = %d\n",
		  pvmmytid, oldpathname, newpathname, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
mkfifo(const char *pathname, mode_t mode)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_MKFIFO;
	int	nargs = 2;
	long	lmode = pvm_open_mode((long)mode);

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_mknod, pathname, mode | S_IFIFO);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x mkfifo(\"%s\", %ld) = -1 (errno = %d)\n",
			  pvmmytid, pathname, (long)mode | S_IFIFO, errno));
		else
			dprintl((buf2,
			  "tid t%x mkfifo(\"%s\", %ld) = %d\n",
			  pvmmytid, pathname, (long)mode | S_IFIFO, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %ld", PvmDataDefault, nargs, pathname, lmode);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x mkfifo(\"%s\", %ld) = -1 (pvmerrno = %d)\n",
		  pvmmytid, pathname, (long)mode, pvmerrno));
	} else
		dprint((buf2, "tid t%x mkfifo(\"%s\", %ld) = %d\n",
		  pvmmytid, pathname, (long)mode, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
mknod(const char *pathname, int mode, dev_t dev)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_MKNOD;
	int	nargs = 3;
	long	lmode = pvm_open_mode((long)mode);
	long	ldev = dev;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_mknod, pathname, mode, dev);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x mknod(\"%s\", %ld, %ld) = -1 (errno = %d)\n",
			  pvmmytid, pathname, (long)mode, (long)dev, errno));
		else
			dprintl((buf2, "tid t%x mknod(\"%s\", %ld, %ld) = %d\n",
			  pvmmytid, pathname, (long)mode, (long)dev, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %ld %ld", PvmDataDefault, nargs, pathname, lmode,
	  ldev);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x mknod(\"%s\", %ld, %ld) = -1 (pvmerrno = %d)\n",
		  pvmmytid, pathname, (long)mode, (long)dev, pvmerrno));
	} else
		dprint((buf2, "tid t%x mknod(\"%s\", %ld, %ld) = %d\n",
		  pvmmytid, pathname, (long)mode, (long)dev, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
pipe(int fd[2])
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_PIPE;
	int	nargs = 0;

	if (pvmmyfstid <= 0) {
#ifdef SYSCALL_PIPE_PROBLEM
		if ((cc = socketpair(AF_UNIX, SOCK_STREAM, 0, fd)) == 0) {
			(void)shutdown(fd[0], 1);
			(void)shutdown(fd[1], 0);
		}
#else
		/*
		 * Why doesn't this work?
		 */
		cc = syscall(SYS_pipe, fd);
#endif
		if (cc == -1)
			dprintl((buf2, "tid t%x pipe(%p) = -1 (errno = %d)\n",
			  pvmmytid, fd, errno));
		else
			dprintl((buf2, "tid t%x pipe(%p) = %d <%d, %d>\n",
			  pvmmytid, fd, cc, fd[0], fd[1]));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d", PvmDataDefault, nargs);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x pipe(%p) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, pvmerrno));
	} else {
		pvm_unpackf("%d %d", &fd[0], &fd[1]);
		dprint((buf2, "tid t%x pipe(%p) = %d <%d, %d>\n",
		  pvmmytid, fd, cc, fd[0], fd[1]));
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
readlink(const char *pathname, char *buf, int length)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_READLINK;
	int	nargs = 2;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_readlink, pathname, buf, length);
		if (cc == -1)
			dprintl((buf2,
			 "tid t%x readlink(\"%s\", %p, %d) = -1 (errno = %d)\n",
			 pvmmytid, pathname, buf, length, errno));
		else
			dprintl((buf2,
			  "tid t%x readlink(\"%s\", %p, %d) = %d\n",
			  pvmmytid, pathname, buf, length, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %d", PvmDataDefault, nargs, pathname, length);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x readlink(\"%s\", %p, %d) = -1 (pvmerrno = %d)\n",
		  pvmmytid, pathname, buf, length, pvmerrno));
	} else if (cc == length) {
		pvm_unpackf("%*c", cc, buf);
		dprint((buf2,
		  "tid t%x readlink(\"%s\", %p, %d) = %d \"%*.*s\"\n",
		  pvmmytid, pathname, buf, length, cc, cc, cc, buf));
	} else {
		pvm_unpackf("%s", buf);
		dprint((buf2, "tid t%x readlink(\"%s\", %p, %d) = %d \"%s\"\n",
		  pvmmytid, pathname, buf, length, cc, buf));
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

#if 0
static void
fillstatfs(struct statfs *stp)
{
	long	l;

#define get(mem)	pvm_unpackf("%ld", &l); stp->f_##mem = l;
	get(type);
	get(bsize);
	get(blocks);
	get(bfree);
	get(bavail);
	get(files);
	get(ffree);
#if 0
	/* these don't seem to be very uniform */
	get(fsid);
	get(fname);
	get(name_max);
#endif
#undef get
}

int
statfs(const char *pathname, struct statfs *stp, int length)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_STATFS;
	int	nargs = 2;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_statfs, pathname, stp);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x statfs(\"%s\", %p) = %d (errno = %d)\n",
			  pvmmytid, pathname, (void *)stp, cc, errno));
		else
			dprintl((buf2, "tid t%x statfs(\"%s\", %p) = %d\n",
			  pvmmytid, pathname, (void *)stp, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s", PvmDataDefault, nargs, pathname);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x statfs(\"%s\", %p) = %d (pvmerrno = %d)\n", pvmmytid,
		  pathname, (void *)stp, cc, pvmerrno));
	} else {
		dprint((buf2, "tid t%x statfs(\"%s\", %p) = %d\n",
		  pvmmytid, pathname, (void *)stp, cc));
		fillstatfs(stp);
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
fstatfs(int fd, struct statfs *stp)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_FSTAT;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_fstatfs, fd, stp);
if (fd < 100) {
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x fstatfs(%d, %p) = %d (errno = %d)\n",
			  pvmmytid, fd, (void *)stp, cc, errno));
		else
			dprintl((buf2, "tid t%x fstatfs(%d, %p) = %d\n",
			  pvmmytid, fd, (void *)stp, cc));
}
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x fstatfs(%d, %p) = %d (pvmerrno = %d)\n",
		  pvmmytid, fd, (void *)stp, cc, pvmerrno));
	} else {
		dprint((buf2, "tid t%x fstatfs(%d, %p) = %d\n",
		  pvmmytid, fd, (void *)stp, cc));
		fillstat(stp);
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}
#endif

int
truncate(const char *pathname, off_t len)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_TRUNCATE;
	int	nargs = 2;
	long	llen = len;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_truncate, pathname, len);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x truncate(\"%s\", %ld) = -1 (errno = %d)\n",
			  pvmmytid, pathname, (long)len, errno));
		else
			dprintl((buf2, "tid t%x truncate(\"%s\", %ld) = %d\n",
			  pvmmytid, pathname, (long)len, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %ld", PvmDataDefault, nargs, pathname, llen);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x truncate(\"%s\", %ld) = -1 (pvmerrno = %d)\n",
		  pvmmytid, pathname, (long)len, pvmerrno));
	} else
		dprint((buf2, "tid t%x truncate(\"%s\", %ld) = %d\n",
		  pvmmytid, pathname, (long)len, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
ftruncate(int fd, off_t len)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_FTRUNCATE;
	int	nargs = 2;
	long	llen = len;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_ftruncate, fd, len);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x ftruncate(%d, %ld) = -1 (errno = %d)\n",
			  pvmmytid, fd, (long)len, errno));
		else
			dprintl((buf2, "tid t%x ftruncate(%d, %ld) = %d\n",
			  pvmmytid, fd, (long)len, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %ld", PvmDataDefault, nargs, fd, llen);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x ftruncate(%d, %ld) = -1 (pvmerrno = %d)\n", pvmmytid,
		  fd, (long)len, pvmerrno));
	} else
		dprint((buf2, "tid t%x ftruncate(%d, %ld) = %d\n", pvmmytid, fd,
		  (long)len, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

mode_t
umask(mode_t mask)
{
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_UMASK;
	int	nargs = 1;
	long	lmask = pvm_open_mode(mask);
	mode_t	oldmask;

	if (pvmmyfstid <= 0) {
		oldmask = syscall(SYS_umask, mask);
		dprintl((buf2, "tid t%x umask(%ld) = %ld\n", pvmmytid,
		  (long)mask, (long)oldmask));
		return oldmask;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %ld", PvmDataDefault, nargs, lmask);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%ld", &lmask);
	oldmask = pvm_local_open_mode(lmask);
	dprint((buf2, "tid t%x umask(%ld) = %d\n", pvmmytid, (long)mask,
	  (long)oldmask));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return oldmask;
}

int
utime(const char *pathname, const struct utimbuf *utp)
{
	int		cc;
	int		pvmerrno;
	int		rbuf;
	int		sbuf;
	int		request = PVMFS_UTIME;
	int		nargs = 1;
	long		l1, l2;

	if (pvmmyfstid <= 0) {
		struct timeval	tv[2];

		tv[0].tv_usec = tv[1].tv_usec = 0;
		if (utp) {
			tv[0].tv_sec = utp->actime;
			tv[1].tv_sec = utp->modtime;
		} else
			tv[0].tv_sec = tv[1].tv_sec = time(NULL);
		cc = syscall(SYS_utimes, pathname, tv);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x utime(\"%s\", %p) = -1 (errno = %d)\n",
			  pvmmytid, pathname, utp, errno));
		else
			dprintl((buf2, "tid t%x utime(\"%s\", %p) = %d\n",
			  pvmmytid, pathname, utp, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));

	if (utp) {
		nargs = 3;
		l1 = utp->actime;
		l2 = utp->modtime;
	}
	pvm_packf("%+ %d %s", PvmDataDefault, nargs, pathname);
	if (nargs == 3)
		pvm_packf("%ld %ld", l1, l2);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x utime(\"%s\", %p) = -1 (pvmerrno = %d)\n", pvmmytid,
		  pathname, utp, pvmerrno));
	} else
		dprint((buf2, "tid t%x utime(\"%s\", %p) = %d\n",
		  pvmmytid, pathname, utp, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
utimes(const char *pathname, struct timeval tv[2])
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_UTIMES;
	int	nargs = 5;
	long	l1, l2, l3, l4;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_utimes, pathname, tv);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x utimes(\"%s\", %p) = -1 (errno = %d)\n",
			  pvmmytid, pathname, tv, errno));
		else
			dprintl((buf2, "tid t%x utimes(\"%s\", %p) = %d\n",
			  pvmmytid, pathname, tv, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	l1 = tv[0].tv_sec;
	l2 = tv[0].tv_usec;
	l3 = tv[1].tv_sec;
	l4 = tv[1].tv_usec;
	pvm_packf("%+ %d %s %ld %ld %ld %ld", PvmDataDefault, nargs, pathname,
	  l1, l2, l3, l4);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x utimes(\"%s\", %p) = -1 (pvmerrno = %d)\n", pvmmytid,
		  pathname, tv, pvmerrno));
	} else
		dprint((buf2, "tid t%x utimes(\"%s\", %p) = %d\n",
		  pvmmytid, pathname, tv, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

void
sync(void)
{
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_SYNC;
	int	nargs = 0;

	if (pvmmyfstid <= 0) {
		syscall(SYS_sync);
		dprintl((buf2, "tid t%x sync()\n"));
		return;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d", PvmDataDefault, nargs);
	pvm_send(pvmmyfstid, request);
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));
}

int
fsync(int fd)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_FSYNC;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_fsync, fd);
		if (cc == -1)
			dprintl((buf2, "tid t%x fsync(%d) = -1 (errno = %d)\n",
			  pvmmytid, fd, errno));
		else
			dprintl((buf2, "tid t%x fsync(%d) = %d\n",
			  pvmmytid, fd, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x fsync(%d) = -1 (pvmerrno = %d)\n",
		  pvmmytid, fd, pvmerrno));
	} else
		dprint((buf2, "tid t%x fsync(%d) = %d\n", pvmmytid, fd, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
access(const char *pathname, int mode)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_ACCESS;
	int	nargs = 2;
	int	pvmmode = pvm_access_mode(mode);

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_access, pathname, mode);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x access(\"%s\", %d) = -1 (errno = %d)\n",
			  pvmmytid, pathname, mode, errno));
		else
			dprintl((buf2, "tid t%x access(\"%s\", %d) = %d\n",
			  pvmmytid, pathname, mode, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %s %d", PvmDataDefault, nargs, pathname, pvmmode);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x access(\"%s\", %d) = -1 (pvmerrno = %d)\n", pvmmytid,
		  pathname, mode, pvmerrno));
	} else
		dprint((buf2, "tid t%x access(\"%s\", %d) = %d\n", pvmmytid,
		  pathname, mode, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
select(SELECT_ARG1_TYPE maxfdp1, SELECT_SETP_TYPE rsetp, SELECT_SETP_TYPE wsetp, SELECT_SETP_TYPE esetp, struct timeval *tvp)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_SELECT;
	int	nargs = tvp ? 5 : 4;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_select, maxfdp1, rsetp, wsetp, esetp, tvp);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x select(%d, ...) = -1 (errno = %d)\n",
			  pvmmytid, maxfdp1, errno));
		else
			dprintl((buf2, "tid t%x select(%d, ...) = %d\n",
			  pvmmytid, maxfdp1, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, (int)maxfdp1);
	pvm_select_fdset_pack(maxfdp1, rsetp);
	pvm_select_fdset_pack(maxfdp1, wsetp);
	pvm_select_fdset_pack(maxfdp1, esetp);
	if (tvp) {
		long	l;

		l = tvp->tv_sec; pvm_packf("%ld", &l);
		l = tvp->tv_usec; pvm_packf("%ld", &l);
	}
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x select(%d, ...) = -1 (pvmerrno = %d)\n",
		  pvmmytid, maxfdp1, pvmerrno));
	} else if (cc > 0) {
		if (rsetp)
			pvm_select_fdset_unpack(rsetp);
		if (wsetp)
			pvm_select_fdset_unpack(wsetp);
		if (esetp)
			pvm_select_fdset_unpack(esetp);
		dprint((buf2, "tid t%x select(%d, ...) = %d\n", pvmmytid,
		  maxfdp1, cc));
	} else
		dprint((buf2, "tid t%x select(%d, ...) = %d\n", pvmmytid,
		  maxfdp1, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
getrlimit(int resource, struct rlimit *rlp)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_GETRLIMIT;
	int	nargs = 1;

	if (pvmmyfstid <= 0 || (resource != RLIMIT_CORE
#ifdef RLIMIT_NOFILE
	  && resource != RLIMIT_NOFILE
#endif
#ifdef RLIMIT_OFILE
	  && resource != RLIMIT_OFILE
#endif
	  && resource != RLIMIT_FSIZE)) {
		cc = syscall(SYS_getrlimit, resource, rlp);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x getrlimit(%d, %p) = %d (errno = %d)\n",
			    pvmmytid, resource, (void *)rlp, cc, errno));
		else
			dprintl((buf2, "tid t%x getrlimit(%d, %p) = %d\n",
			  pvmmytid, resource, (void *)rlp, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs,
	  pvm_rlimit_resource(resource));
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x getrlimit(%d, %p) = %d (pvmerrno = %d)\n", pvmmytid,
		  resource, (void *)rlp, cc, pvmerrno));
	} else {
		unsigned long	ul;
		int		infinity;

		dprint((buf2, "tid t%x getrlimit(%d, %p) = %d\n", pvmmytid,
		  resource, (void *)rlp, cc));
		pvm_unpackf("%d", &infinity);
		if (infinity)
			rlp->rlim_cur = RLIM_INFINITY;
		else {
			pvm_upkulong(&ul, 1, 1);
			rlp->rlim_cur = ul;
		}
		pvm_unpackf("%d", &infinity);
		if (infinity)
			rlp->rlim_max = RLIM_INFINITY;
		else {
			pvm_upkulong(&ul, 1, 1);
			rlp->rlim_max = ul;
		}
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
setrlimit(int resource, SETRLIMIT_CONST struct rlimit *rlp)
{
	int		cc;
	int		pvmerrno;
	int		rbuf;
	int		sbuf;
	int		request = PVMFS_SETRLIMIT;
	int		nargs = 3;
	unsigned long	ul;

	if (pvmmyfstid <= 0 || (resource != RLIMIT_CORE
#ifdef RLIMIT_NOFILE
	  && resource != RLIMIT_NOFILE
#endif
#ifdef RLIMIT_OFILE
	  && resource != RLIMIT_OFILE
#endif
	  && resource != RLIMIT_FSIZE)) {
		cc = syscall(SYS_setrlimit, resource, rlp);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x setrlimit(%d, %p) = %d (errno = %d)\n",
			    pvmmytid, resource, (void *)rlp, cc, errno));
		else
			dprintl((buf2, "tid t%x setrlimit(%d, %p) = %d\n",
			  pvmmytid, resource, (void *)rlp, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %d", PvmDataDefault, nargs,
	  pvm_rlimit_resource(resource), rlp->rlim_cur == RLIM_INFINITY);
	if (rlp->rlim_cur != RLIM_INFINITY) {
		ul = rlp->rlim_cur;
		pvm_pkulong(&ul, 1, 1);
	}
	pvm_packf("%d", rlp->rlim_max == RLIM_INFINITY);
	if (rlp->rlim_max != RLIM_INFINITY) {
		ul = rlp->rlim_max;
		pvm_pkulong(&ul, 1, 1);
	}
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2,
		  "tid t%x setrlimit(%d, %p <%lu, %lu>) = %d (pvmerrno = %d)\n",
		  pvmmytid, resource, (void *)rlp, (unsigned long)rlp->rlim_cur,
		  (unsigned long)rlp->rlim_max, cc, pvmerrno));
	} else
		dprint((buf2, "tid t%x setrlimit(%d, %p <%lu, %lu>) = %d\n",
		  pvmmytid, resource, (void *)rlp, (unsigned long)rlp->rlim_cur,
		  (unsigned long)rlp->rlim_max, cc));
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
getdtablesize(void)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_GETDTABLESIZE;
	int	nargs = 0;

	if (pvmmyfstid <= 0) {
#ifdef SYS_getdtablesize
		cc = syscall(SYS_getdtablesize);
#else
		cc = (int)openmax();
#endif
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x getdtablesize() = %d (errno = %d)\n",
			  pvmmytid, cc, errno));
		else
			dprintl((buf2, "tid t%x getdtablesize() = %d\n",
			  pvmmytid, cc));
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d", PvmDataDefault, nargs);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x getdtablesize() = %d (pvmerrno = %d)\n",
		  pvmmytid, cc, pvmerrno));
	} else
		dprint((buf2, "tid t%x getdtablesize() = %d\n", pvmmytid, cc));

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
getdirentries(int fd, char *buf, int nbytes, long *basep)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_GETDIRENTRIES;
	int	nargs = 2;

	if (pvmmyfstid <= 0) {
#ifdef SYS_getdirentries
		cc = syscall(SYS_getdirentries, fd, buf, nbytes, basep);
		if (cc == -1)
			dprintl((buf2,
		    "tid t%x getdirentries(%d, %p, %d, %p) = %d (errno = %d)\n",
			 pvmmytid, fd, (void *)buf, nbytes, (void *)basep, cc,
			 errno));
		else
			dprintl((buf2,
			  "tid t%x getdirentries(%d, %p, %d, %p) = %d\n",
			  pvmmytid, fd, (void *)buf, nbytes, (void *)basep,
			  cc));
		return cc;
#endif
	}
#if 1
	errno = ENOSYS;
	return -1;

#else
	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d %d %ld", PvmDataDefault, nargs, fd, nbytes, *basep);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprintl((buf2,
		 "tid t%x getdirentries(%d, %p, %d, %p) = %d (pvmerrno = %d)\n",
		  pvmmytid, fd, (void *)buf, nbytes, (void *)basep, cc,
		  pvmerrno));
	} else {
		if (cc) {
			/* ? */
		}
		dprintl((buf2, "tid t%x getdirentries(%d, %p, %d, %p) = %d\n",
		  pvmmytid, fd, (void *)buf, nbytes, (void *)basep, cc,
		  pvmerrno));
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
#endif
}

int
isatty(int fd)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_ISATTY;
	int	nargs = 1;

	if (pvmmyfstid <= 0) {
		struct winsize	ws;

		return ioctl(fd, TIOCGWINSZ, &ws) == 0;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == 0) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		dprint((buf2, "tid t%x isatty(%d) = %d (pvmerrno = %d)\n",
		  pvmmytid, fd, cc, pvmerrno));
	} else
		dprint((buf2, "tid t%x isatty(%d) = %d\n",
		  pvmmytid, fd, cc));

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

char *
ctermid(char *cp)
{
	int		cc;
	int		pvmerrno;
	int		rbuf;
	int		sbuf;
	int		request = PVMFS_CTERMID;
	int		nargs = 0;
	static char	*termid = NULL;
	static char	*saved = "/dev/tty";

	if (termid)
		free(termid);

	if (pvmmyfstid <= 0) {
		if (cp)
			return strcpy(cp, saved);
		else
			return saved;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	pvm_packf("%+ %d", PvmDataDefault, nargs);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == 0 || cp == NULL && (termid = malloc(cc + 1)) == NULL) {
		if (cc == 0) {
			pvm_unpackf("%d", &pvmerrno);
			errno = pvm_local_errno(pvmerrno);
		}
		dprint((buf2, "tid t%x ctermid(%p) = \"\" (errno = %d)\n",
		  pvmmytid, (void *)cp, errno));
	} else {
		pvm_unpackf("%s", cp ? cp : termid);
		dprint((buf2, "tid t%x ctermid(%p) = %s\n",
		  pvmmytid, (void *)cp, cp ? cp : termid));
	}

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cp ? cp : termid;
}

/*
 * With inspiration from Stevens' _Advanced Programming in the UNIX Environment_
 */

#undef MIN
#include <limits.h>
#include <unistd.h>
#include <sys/param.h>

#ifdef OPEN_MAX
static int	result = OPEN_MAX;
#else
static int	result = 0;
#endif

static int
openmax(void)
{
	if (result)
		return result;

	if ((result = sysconf(_SC_OPEN_MAX)) == -1)
#ifdef NOFILE
		result = NOFILE;
#else
		result = 64;
#endif

	return result;
}

#ifdef IMA_CNVXN

/*
 * CONVEX functions for 64-bit file offsets.
 * We make them fail unless they're in the 32-bit range.
 */
off64_t
lseek64(int fd, off64_t offset, int whence)
{
	off64_t	result;
	long	high;
	long	low;

	if (pvmmyfstid <= 0) {
		result = (off64_t)syscall(SYS_lseek64, fd, offset, whence);
		if (result == -1)
			dprintl((buf2,
			  "tid t%x lseek64(%d, %ll, %d) = %ll (errno = %d)\n",
			  pvmmytid, fd, offset, whence, result, errno));
		else
			dprintl((buf2, "tid t%x lseek64(%d, %ll, %d) = %ll\n",
			  pvmmytid, fd, offset, whence, result));
		return result;
	}

	high = offset >> 32;
	low = offset & 0xffffffff;
	if (high)
		result = -1;
	else
		result = lseek(fd, low, whence);
	if (result == -1)
		dprintl((buf2,
		  "tid t%x lseek64(%d, %ll, %d) = %ll (errno = %d)\n",
		  pvmmytid, fd, offset, whence, result, errno));
	else
		dprintl((buf2, "tid t%x lseek64(%d, %ll, %d) = %ll\n",
		  pvmmytid, fd, offset, whence, result));

	return result;
}

int
truncate64(const char *pathname, off64_t len)
{
	int	cc;
	long	high;
	long	low;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_truncate64, pathname, len);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x truncate64(\"%s\", %ll) = %d (errno = %d)\n",
			  pvmmytid, pathname, len, cc, errno));
		else
			dprintl((buf2, "tid t%x truncate64(\"%s\", %ll) = %d\n",
			  pvmmytid, pathname, len, cc));
		return cc;
	}

	high = len >> 32;
	low = len & 0xffffffff;
	if (high) {
		errno = EINVAL;
		cc = -1;
	} else
		cc = truncate(pathname, low);
	if (cc == -1)
		dprintl((buf2,
		  "tid t%x truncate64(\"%s\", %ll) = %d (errno = %d)\n",
		  pvmmytid, pathname, len, cc, errno));
	else
		dprintl((buf2, "tid t%x truncate64(\"%s\", %ll) = %d\n",
		  pvmmytid, pathname, len, cc));

	return cc;
}

int
ftruncate64(int fd, off64_t len)
{
	int	cc;
	long	high;
	long	low;

	if (pvmmyfstid <= 0) {
		cc = syscall(SYS_ftruncate64, fd, len);
		if (cc == -1)
			dprintl((buf2,
			  "tid t%x ftruncate64(%d, %ll) = %d (errno = %d)\n",
			  pvmmytid, fd, len, cc, errno));
		else
			dprintl((buf2, "tid t%x ftruncate64(%d, %ll) = %d\n",
			  pvmmytid, fd, len, cc));
		return cc;
	}

	high = len >> 32;
	low = len & 0xffffffff;
	if (high) {
		errno = EINVAL;
		cc = -1;
	} else
		cc = ftruncate(fd, low);
	if (cc == -1)
		dprintl((buf2,
		  "tid t%x ftruncate64(%d, %ll) = %d (errno = %d)\n",
		  pvmmytid, fd, len, cc, errno));
	else
		dprintl((buf2, "tid t%x ftruncate64(%d, %ll) = %d\n",
		  pvmmytid, fd, len, cc));

	return cc;
}

/*
 * What's the purpose for the new names???
 */
int
__ap$open(const char *pathname, int oflag, ...)
{
	int	cc;
	int	pvmerrno;
	int	rbuf;
	int	sbuf;
	int	request = PVMFS_OPEN;
	int	nargs = 3;
	va_list	ap;
	long	mode;

	va_start(ap, oflag);

	if (pvmmyfstid <= 0) {
		if (oflag & O_CREAT) {
			if (sizeof(mode_t) < sizeof(int))
				mode = (long)va_arg(ap, int);
			else
				mode = (long)va_arg(ap, mode_t);
			cc = syscall(SYS_open, pathname, oflag, mode);
			if (cc == -1)
				dprintl((buf2,
			    "tid t%x open(\"%s\", %d, %ld) = %d (errno = %d)\n",
			    pvmmytid, pathname, oflag, (long)mode, cc, errno));
			else
				dprintl((buf2,
				  "tid t%x open(\"%s\", %d, %ld) = %d\n",
				  pvmmytid, pathname, oflag, (long)mode, cc));
		} else {
			cc = syscall(SYS_open, pathname, oflag);
			if (cc == -1)
				dprintl((buf2,
				 "tid t%x open(\"%s\", %d) = %d (errno = %d)\n",
				 pvmmytid, pathname, oflag, cc, errno));
			else
				dprintl((buf2,
				  "tid t%x open(\"%s\", %d) = %d\n",
				  pvmmytid, pathname, oflag, cc));
		}
		va_end(ap);
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));
	if ((oflag = pvm_open_oflag(oflag)) == -1) {
		errno = EINVAL;
		va_end(ap);
		dprint((buf2,
		  "tid t%x open(\"%s\", %d, ...) = -1 (errno = EINVAL)\n",
		  pvmmytid, pathname, oflag));
		return -1;
	}
	if (oflag & PVMFS_O_CREAT) {
		nargs = 4;
		if (sizeof(mode_t) < sizeof(int))
			mode = (long)va_arg(ap, int);
		else
			mode = (long)va_arg(ap, mode_t);
	}
	pvm_packf("%+ %d %s %d", PvmDataDefault, nargs, pathname, oflag);
	if (oflag & PVMFS_O_CREAT)
		pvm_packf("%ld", mode);
	pvm_packf("%d", defiomode);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		if (nargs == 3)
			dprint((buf2,
			  "tid t%x open(\"%s\", %d) = %d (pvmerrno = %d)\n",
			  pvmmytid, pathname, oflag, cc, pvmerrno));
		else
			dprint((buf2,
			 "tid t%x open(\"%s\", %d, %ld) = %d (pvmerrno = %d)\n",
			 pvmmytid, pathname, oflag, (long)mode, cc, pvmerrno));
	}
	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	if (nargs == 3)
		dprint((buf2, "tid t%x open(\"%s\", %d) = %d\n",
		  pvmmytid, pathname, oflag, cc));
	else
		dprint((buf2, "tid t%x open(\"%s\", %d, %ld) = %d\n",
		  pvmmytid, pathname, oflag, (long)mode, cc));
	va_end(ap);
	return cc;
}

int
__ap$creat(const char *pathname, mode_t mode)
{
	return creat(pathname, mode);
}

int
__ap$close(int fd)
{
	return close(fd);
}

off_t
__ap$lseek(int fd, off_t offset, int whence)
{
	return lseek(fd, offset, whence);
}

READ_RETURN_TYPE
__ap$read(int fd, READ_ARG2_TYPE buf, READ_ARG3_TYPE nbytes)
{
	return read(fd, buf, nbytes);
}

READV_RETURN_TYPE
__ap$readv(int fd, READV_CONST struct iovec *iovp, int count)
{
	return readv(fd, iovp, count);
}

WRITE_RETURN_TYPE
__ap$write(int fd, WRITE_ARG2_TYPE buf, WRITE_ARG3_TYPE nbytes)
{
	return write(fd, buf, nbytes);
}

WRITEV_RETURN_TYPE
__ap$writev(int fd, WRITEV_CONST struct iovec *iovp, int count)
{
	return writev(fd, iovp, count);
}

int
__ap$dup(int fd)
{
	return dup(fd);
}

int
__ap$dup2(int fd1, int fd2)
{
	return dup2(fd1, fd2);
}

int
__ap$stat(const char *pathname, struct stat *stp)
{
	return stat(pathname, stp);
}

int
__ap$lstat(const char *pathname, struct stat *stp)
{
	return lstat(pathname, stp);
}

int
__ap$fstat(int fd, struct stat *stp)
{
	return fstat(fd, stp);
}

int
__ap$fcntl(int fd, int command, ...)
{
	va_list		ap;
	int		iarg = 0;
	struct flock	*lockp;
	int		cc;
	int		pvmerrno;
	int		rbuf;
	int		sbuf;
	int		request = PVMFS_FCNTL;
	long		l1, l2;
	int		nargs = 3;

	va_start(ap, command);

	if (pvmmyfstid <= 0) {
		switch (command) {
		case F_GETLK:
		case F_SETLK:
		case F_SETLKW:
			lockp = va_arg(ap, struct flock *);
			cc = syscall(SYS_fcntl, fd, command, lockp);
			if (cc == -1)
				dprintl((buf2,
				"tid t%x fcntl(%d, %d, %p) = -1 (errno = %d)\n",
				pvmmytid, fd, command, (void *)lockp, errno));
			else
				dprintl((buf2,
				  "tid t%x fcntl(%d, %d, %p) = %d\n",
				  pvmmytid, fd, command, (void *)lockp, cc));
			break;
		default:
			iarg = va_arg(ap, int);
			cc = syscall(SYS_fcntl, fd, command, iarg);
			if (cc == -1)
				dprintl((buf2,
				"tid t%x fcntl(%d, %d, %d) = -1 (errno = %d)\n",
				pvmmytid, fd, command, iarg, errno));
			else
				dprintl((buf2,
				  "tid t%x fcntl(%d, %d, %d) = %d\n",
				  pvmmytid, fd, command, iarg, cc));
			break;
		}
		va_end(ap);
		return cc;
	}

	rbuf = pvm_setrbuf(pvm_mkbuf(PvmDataDefault));
	sbuf = pvm_setsbuf(pvm_mkbuf(PvmDataDefault));

	pvm_packf("%+ %d %d", PvmDataDefault, nargs, fd);

	switch (command) {
	case F_GETLK:
	case F_SETLK:
	case F_SETLKW:
		lockp = va_arg(ap, struct flock *);
		if ((lockp->l_type = pvm_fcntl_ltype(lockp->l_type)) == -1L
		  || (lockp->l_whence = pvm_lseek_whence(lockp->l_whence))
		  == -1L) {
			errno = EINVAL;
			va_end(ap);
			return -1;
		}
		command = pvm_fcntl_cmd(command);
		pvm_packf("%d %d %ld %d %ld", command, lockp->l_type,
		  (long)lockp->l_start, lockp->l_whence, (long)lockp->l_len);
		break;
	case F_DUPFD:
	case F_SETFD:
	case F_SETFL:
	case F_SETOWN:
		iarg = va_arg(ap, int);
		if (command == F_SETFL)
			iarg = pvm_open_oflag(iarg);
		/* FALLTHROUGH */
	default:
		command = pvm_fcntl_cmd(command);
		pvm_packf("%d %d", command, iarg);
		break;
	}
	va_end(ap);
	pvm_send(pvmmyfstid, request);
	pvm_recv(pvmmyfstid, request);
	pvm_unpackf("%d", &cc);
	if (cc == -1) {
		pvm_unpackf("%d", &pvmerrno);
		errno = pvm_local_errno(pvmerrno);
		switch (command) {
		case F_GETLK:
		case F_SETLK:
		case F_SETLKW:
			dprint((buf2,
			  "tid t%x fcntl(%d, %d, %p) = -1 (pvmerrno = %d)\n",
			  pvmmytid, fd, command, (void *)lockp, pvmerrno));
			break;
		default:
			dprint((buf2,
			  "tid t%x fcntl(%d, %d, %d) = -1 (pvmerrno = %d)\n",
			  pvmmytid, fd, command, iarg, pvmerrno));
			break;
		}
	} else
		switch (command) {
		case F_GETLK:
			pvm_unpackf("%hd", &lockp->l_type);
			lockp->l_type =
			  (short)pvm_local_fcntl_ltype(lockp->l_type);
			if (lockp->l_type != F_UNLCK) {
				pvm_unpackf("%ld %hd %ld", &l1,
				  &lockp->l_whence, &l2);
				lockp->l_start = (off_t)l1;
				lockp->l_whence = (short)
				  pvm_local_lseek_whence(lockp->l_whence);
				lockp->l_len = (off_t)l2;
			}
		case F_SETLK:
		case F_SETLKW:
			dprint((buf2, "tid t%x fcntl(%d, %d, %p) = %d\n",
			  pvmmytid, fd, command, (void *)lockp, cc));
			break;
		case F_GETFL:
			cc = pvm_local_open_oflag(cc);
		default:
			dprint((buf2, "tid t%x fcntl(%d, %d, %d) = %d\n",
			  pvmmytid, fd, command, iarg, cc));
			break;
		}

	pvm_freebuf(pvm_setrbuf(rbuf));
	pvm_freebuf(pvm_setsbuf(sbuf));

	return cc;
}

int
__ap$unlink(const char *pathname)
{
	return unlink(pathname);
}

int
__ap$rename(const char *oldpathname, const char *newpathname)
{
	return rename(oldpathname, newpathname);
}

int
__ap$mkdir(const char *pathname, mode_t mode)
{
	return mkdir(pathname, mode);
}

int
__ap$rmdir(const char *pathname)
{
	return rmdir(pathname);
}

int
__ap$chroot(const char *pathname)
{
	return chroot(pathname);
}

int
__ap$fchroot(int fd)
{
	return fchroot(fd);
}

int
__ap$chdir(const char *pathname)
{
	return chdir(pathname);
}

int
__ap$fchdir(int fd)
{
	return fchdir(fd);
}

int
__ap$chmod(const char *pathname, mode_t mode)
{
	return chmod(pathname, mode);
}

int
__ap$fchmod(int fd, mode_t mode)
{
	return fchmod(fd, mode);
}

int
__ap$chown(const char *pathname, uid_t uid, gid_t gid)
{
	return chown(pathname, uid, gid);
}

int
__ap$fchown(int fd, uid_t uid, gid_t gid)
{
	return fchown(fd, uid, gid);
}

int
__ap$lchown(const char *pathname, uid_t uid, gid_t gid)
{
	return lchown(pathname, uid, gid);
}

int
__ap$link(const char *oldpathname, const char *newpathname)
{
	return link(oldpathname, newpathname);
}

int
__ap$symlink(const char *oldpathname, const char *newpathname)
{
	return symlink(oldpathname, newpathname);
}

int
__ap$mkfifo(const char *pathname, mode_t mode)
{
	return mkfifo(pathname, mode);
}

int
__ap$mknod(const char *pathname, int mode, dev_t dev)
{
	return mknod(pathname, mode, dev);
}

int
__ap$pipe(int fd[2])
{
	return pipe(fd);
}

int
__ap$readlink(const char *pathname, char *buf, int length)
{
	return readlink(pathname, buf, length);
}

int
__ap$truncate(const char *pathname, off_t len)
{
	return truncate(pathname, len);
}

int
__ap$ftruncate(int fd, off_t len)
{
	return ftruncate(fd, len);
}

mode_t
__ap$umask(mode_t mask)
{
	return umask(mask);
}

int
__ap$utime(const char *pathname, const struct utimbuf *utp)
{
	return utime(pathname, utp);
}

int
__ap$utimes(const char *pathname, struct timeval tv[2])
{
	return utimes(pathname, tv);
}

void
__ap$sync(void)
{
	sync();
}

int
__ap$fsync(int fd)
{
	return fsync(fd);
}

int
__ap$access(const char *pathname, int mode)
{
	return access(pathname, mode);
}

int
__ap$select(SELECT_ARG1_TYPE maxfdp1, SELECT_SETP_TYPE rsetp, SELECT_SETP_TYPE wsetp, SELECT_SETP_TYPE esetp, struct timeval *tvp)
{
	return select(maxfdp1, rsetp, wsetp, esetp, tvp);
}

int
__ap$getrlimit(int resource, struct rlimit *rlp)
{
	return getrlimit(resource, rlp);
}

int
__ap$setrlimit(int resource, SETRLIMIT_CONST struct rlimit *rlp)
{
	return setrlimit(resource, rlp);
}

int
__ap$getdtablesize(void)
{
	return getdtablesize();
}

int
__ap$getdirentries(int fd, char *buf, int nbytes, long *basep)
{
	return getdirentries(fd, buf, nbytes, basep);
}

int
__ap$isatty(int fd)
{
	return isatty(fd);
}

char *
__ap$ctermid(char *cp)
{
	return ctermid(cp);
}

#endif /* IMA_CNVXN */
