/*
 *                 Author:  Christopher G. Phillips
 *              Copyright (C) 1993 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * The author makes no representations about the suitability of this
 * software for any purpose.  This software is provided ``as is''
 * without express or implied warranty.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <pvm3.h>
#include "fsp.h"
#include "map.h"

#ifndef IMA_RS6K
#include <syscall.h>
#else
#include "syscall-fake.h"
#endif

#ifdef TRACE
/*
 * Debugging stuff follows.  You can get a neat trace using these
 * (although they can mess up errno).  This stuff could be taken out
 * if warranted.
 */
 
extern int      pvmdebug;

#define dprint(s)	do {	\
				if (pvmdebug) {	\
					sprintf s;	\
					syscall(SYS_write, 2, buf2, \
					  strlen(buf2)); \
				}	\
			} while (0)

char	buf2[BUFSIZ];

#else
#define dprint(s)
#endif

extern size_t	nummapsused;

int		mytid;
extern int	pvm_errno;

extern struct lmap	*lmap_add __ProtoGlarp__((int));
extern struct lmap	*lmap_find __ProtoGlarp__((int));
extern void		lmap_delete __ProtoGlarp__((int));
extern int		lmap_init __ProtoGlarp__((int, int, int));

#define FS_ARGS	__ProtoGlarp__((struct lmap *, int, int, int))

extern void	fs_open FS_ARGS;
extern void	fs_close FS_ARGS;
extern void	fs_lseek FS_ARGS;
extern void	fs_read FS_ARGS;
extern void	fs_write FS_ARGS;
extern void	fs_dup FS_ARGS;
extern void	fs_setioorder FS_ARGS;
extern void	fs_getioorder FS_ARGS;
extern void	fs_setiomode FS_ARGS;
extern void	fs_getiomode FS_ARGS;
extern void	fs_stat FS_ARGS;
extern void	fs_fstat FS_ARGS;
extern void	fs_lstat FS_ARGS;
extern void	fs_fcntl FS_ARGS;
extern void	fs_unlink FS_ARGS;
extern void	fs_rename FS_ARGS;
extern void	fs_mkdir FS_ARGS;
extern void	fs_rmdir FS_ARGS;
extern void	fs_chroot FS_ARGS;
extern void	fs_chdir FS_ARGS;
extern void	fs_fchdir FS_ARGS;
extern void	fs_fchroot FS_ARGS;
extern void	fs_chmod FS_ARGS;
extern void	fs_fchmod FS_ARGS;
extern void	fs_chown FS_ARGS;
extern void	fs_fchown FS_ARGS;
extern void	fs_lchown FS_ARGS;
extern void	fs_link FS_ARGS;
extern void	fs_symlink FS_ARGS;
extern void	fs_mkfifo FS_ARGS;
extern void	fs_mknod FS_ARGS;
extern void	fs_pipe FS_ARGS;
extern void	fs_readlink FS_ARGS;
extern void	fs_truncate FS_ARGS;
extern void	fs_ftruncate FS_ARGS;
extern void	fs_umask FS_ARGS;
extern void	fs_utime FS_ARGS;
extern void	fs_utimes FS_ARGS;
extern void	fs_sync FS_ARGS;
extern void	fs_fsync FS_ARGS;
extern void	fs_access FS_ARGS;
extern void	fs_select FS_ARGS;
extern void	fs_opendir FS_ARGS;
extern void	fs_readdir FS_ARGS;
extern void	fs_closedir FS_ARGS;
extern void	fs_rewinddir FS_ARGS;
extern void	fs_isatty FS_ARGS;
extern void	fs_ttyname FS_ARGS;
extern void	fs_ctermid FS_ARGS;
extern void	fs_tcdrain FS_ARGS;
extern void	fs_tcflow FS_ARGS;
extern void	fs_tcflush FS_ARGS;
extern void	fs_tcsendbreak FS_ARGS;
extern void	fs_pathconf FS_ARGS;
extern void	fs_fpathconf FS_ARGS;
extern void	fs_getrlimit FS_ARGS;
extern void	fs_setrlimit FS_ARGS;
extern void	fs_getdtablesize FS_ARGS;
extern void	fs_getdirentries FS_ARGS;
extern void	fs_getcwd FS_ARGS;
extern void	fs_seekdir FS_ARGS;
extern void	fs_telldir FS_ARGS;

int
main(int argc, char **argv)
{
	int		fsver = FSPROTOCOL;
	int		authfd;
	int		bufid;
	int		request;
	int		nargs;
	int		requestor;
	int		parents_sock;
	int		iresult;
	struct lmap	*lmp;
	int		pfd[2];
#ifdef TRACE
extern int pvmdebug;
pvmdebug = 1;
#endif

#ifdef DEBUG
	if (argc != 2) {
#else
	if (argc != 5) {
#endif
		fprintf(stderr,
#ifdef DEBUG
		  "Usage: fs tmpfilename\n");
#else
		  "Usage: fs tmpfilename parents_sock pfd[0] pfd[1]\n");
#endif
		exit(1);
	}
	if ((authfd = open(argv[1], O_WRONLY | O_CREAT | O_TRUNC, 0600))
	  == -1) {
		fprintf(stderr, "fs: can't open %s\n", argv[1]);
		exit(1);
	}

#ifdef DEBUG
#define PIPE	"/tmp/fspipe"
	if ((pfd[1] = open(PIPE, O_WRONLY)) == -1
	  || (pfd[0] = open(PIPE, O_RDONLY)) == -1) {
		perror(PIPE);
		exit(1);
	}
#else

	/*
	 * Get file descriptors from the arguments.
	 * Close parent's pvm socket.
	 */
	if ((parents_sock = atoi(argv[2])) < 0) {
		fprintf(stderr, "fs: parents_sock == %s???\n", argv[2]);
		exit(1);
	}
	(void)close(parents_sock);
	if ((pfd[0] = atoi(argv[3])) < 0) {
		fprintf(stderr, "fs: pipe read descriptor == %s???\n", argv[3]);
		exit(1);
	}
	if ((pfd[1] = atoi(argv[4])) < 0) {
		fprintf(stderr, "fs: pipe write descriptor == %s???\n",
		  argv[4]);
		exit(1);
	}
#endif

	/*
	 * Figure out what file descriptors are initially valid.
	 */
	if (lmap_init(pfd[0], pfd[1], authfd) == -1) {
		perror("fs: can't lmap_init\n");
		exit(1);
	}

	if ((mytid = pvm_mytid()) < 0) {
		pvm_perror("fs: pvm_mytid");
		exit(1);
	}

	/*
	 * Tell parent our tid via authfd
	 * and then wake up the parent by writing to the pipe.
	 */
	if (write(authfd, &mytid, sizeof mytid) == -1) {
		fprintf(stderr, "fs: can't write to fd-auth file %s\n",
		  argv[1]);
		exit(1);
	}
	if (write(pfd[1], &fsver, sizeof fsver) != sizeof fsver) {
		pvm_perror("fs: can't continue parent");
		exit(1);
	}

	/*
	 * Close the pipe descriptors and the authfd used for handshaking.
	 */
	(void)close(pfd[0]);
	(void)close(pfd[1]);
	(void)close(authfd);

	dprint((buf2, "fs: my tid is %d\n", mytid));

	(void)signal(SIGHUP, SIG_IGN);

	while (1) {
		/*
		 * Wait for a message.
		 */
		if ((bufid = pvm_recv(-1, -1)) < 0) {
			pvm_perror("pvm_recv");
			if (pvm_errno != PvmSysErr)
				continue;
		}
		if (pvm_bufinfo(bufid, NULL, &request, &requestor) < 0) {
			pvm_perror("pvm_bufinfo");
			if (pvm_errno != PvmSysErr)
				continue;
		}
		pvm_initsend(PvmDataDefault);
		if (request != PVMFS_TASK_EXIT) {
			if ((lmp = lmap_find(requestor)) == NULL) {
				if (request != PVMFS_NEW_TASK)
					fprintf(stderr,
					  "fs: tid %d not initialized\n",
					  requestor);
				else {
					/*
					 * New client registering.
					 */
					dprint((buf2,
					  "fs: initializing tid %d\n",
					  requestor));
					if (lmp = lmap_add(requestor)) {
						iresult = 0;
						pvm_packf("%+ %d",
						  PvmDataDefault, iresult);
						pvm_send(requestor, PVMFS_INIT);
						(void)pvm_notify(PvmTaskExit,
						  PVMFS_TASK_EXIT, 1,
						  &requestor);
						dprint((buf2,
						  "fs: initialized tid %d ok\n",
						  requestor));
					} else {
						iresult = -1;
						dprint((buf2,
					  "fs: couldn't initialize tid %d\n",
						  requestor));
					}
				}
				continue;
			}
		}

		pvm_unpackf("%d", &nargs);

		/*
		 * Megaswitch for services requested.
		 */
		switch (request) {
		case PVMFS_TASK_EXIT:
			/*
			 * A task exited.
			 * Delete it from our list of clients.
			 */
                        dprint((buf2, "fs: task %d exited! %d to go\n", nargs,
			  nummapsused - 1));
                        lmap_delete(nargs);
                        if (nummapsused == 0) {
	                        dprint((buf2, "fs exiting!\n"));
                                pvm_exit();
                                exit(0);
                        }
			break;
		case PVMFS_OPEN:
		case PVMFS_CREAT:
			fs_open(lmp, requestor, request, nargs);
			break;
		case PVMFS_CLOSE:
			fs_close(lmp, requestor, request, nargs);
			break;
		case PVMFS_LSEEK:
			fs_lseek(lmp, requestor, request, nargs);
			break;
		case PVMFS_READ:
		case PVMFS_READV:
			fs_read(lmp, requestor, request, nargs);
			break;
		case PVMFS_WRITE:
		case PVMFS_WRITEV:
			fs_write(lmp, requestor, request, nargs);
			break;
		case PVMFS_DUP:
			fs_dup(lmp, requestor, request, nargs);
			break;
		case PVMFS_DUP2:
			fs_dup(lmp, requestor, request, nargs);
			break;
		case PVMFS_SETIOORDER:
			fs_setioorder(lmp, requestor, request, nargs);
			break;
		case PVMFS_GETIOORDER:
			fs_getioorder(lmp, requestor, request, nargs);
			break;
		case PVMFS_SETIOMODE:
			fs_setiomode(lmp, requestor, request, nargs);
			break;
		case PVMFS_GETIOMODE:
			fs_getiomode(lmp, requestor, request, nargs);
			break;
		case PVMFS_STAT:
			fs_stat(lmp, requestor, request, nargs);
			break;
		case PVMFS_FSTAT:
			fs_fstat(lmp, requestor, request, nargs);
			break;
		case PVMFS_LSTAT:
			fs_lstat(lmp, requestor, request, nargs);
			break;
		case PVMFS_FCNTL:
			fs_fcntl(lmp, requestor, request, nargs);
			break;
		case PVMFS_UNLINK:
			fs_unlink(lmp, requestor, request, nargs);
			break;
		case PVMFS_RENAME:
			fs_rename(lmp, requestor, request, nargs);
			break;
		case PVMFS_MKDIR:
			fs_mkdir(lmp, requestor, request, nargs);
			break;
		case PVMFS_RMDIR:
			fs_rmdir(lmp, requestor, request, nargs);
			break;
		case PVMFS_CHROOT:
			fs_chroot(lmp, requestor, request, nargs);
			break;
		case PVMFS_FCHROOT:
			fs_fchroot(lmp, requestor, request, nargs);
			break;
		case PVMFS_CHDIR:
			fs_chdir(lmp, requestor, request, nargs);
			break;
		case PVMFS_FCHDIR:
			fs_fchdir(lmp, requestor, request, nargs);
			break;
		case PVMFS_CHMOD:
			fs_chmod(lmp, requestor, request, nargs);
			break;
		case PVMFS_FCHMOD:
			fs_fchmod(lmp, requestor, request, nargs);
			break;
		case PVMFS_CHOWN:
			fs_chown(lmp, requestor, request, nargs);
			break;
		case PVMFS_FCHOWN:
			fs_fchown(lmp, requestor, request, nargs);
			break;
		case PVMFS_LCHOWN:
			fs_lchown(lmp, requestor, request, nargs);
			break;
		case PVMFS_LINK:
			fs_link(lmp, requestor, request, nargs);
			break;
		case PVMFS_SYMLINK:
			fs_symlink(lmp, requestor, request, nargs);
			break;
		case PVMFS_MKFIFO:
			fs_mkfifo(lmp, requestor, request, nargs);
			break;
		case PVMFS_MKNOD:
			fs_mknod(lmp, requestor, request, nargs);
			break;
		case PVMFS_PIPE:
			fs_pipe(lmp, requestor, request, nargs);
			break;
		case PVMFS_READLINK:
			fs_readlink(lmp, requestor, request, nargs);
			break;
		case PVMFS_TRUNCATE:
			fs_truncate(lmp, requestor, request, nargs);
			break;
		case PVMFS_FTRUNCATE:
			fs_ftruncate(lmp, requestor, request, nargs);
			break;
		case PVMFS_UMASK:
			fs_umask(lmp, requestor, request, nargs);
			break;
		case PVMFS_UTIME:
			fs_utime(lmp, requestor, request, nargs);
			break;
		case PVMFS_UTIMES:
			fs_utimes(lmp, requestor, request, nargs);
			break;
		case PVMFS_SYNC:
			fs_sync(lmp, requestor, request, nargs);
			break;
		case PVMFS_FSYNC:
			fs_fsync(lmp, requestor, request, nargs);
			break;
		case PVMFS_ACCESS:
			fs_access(lmp, requestor, request, nargs);
			break;
		case PVMFS_SELECT:
			fs_select(lmp, requestor, request, nargs);
			break;
		case PVMFS_OPENDIR:
			fs_opendir(lmp, requestor, request, nargs);
			break;
		case PVMFS_READDIR:
			fs_readdir(lmp, requestor, request, nargs);
			break;
		case PVMFS_CLOSEDIR:
			fs_closedir(lmp, requestor, request, nargs);
			break;
		case PVMFS_REWINDDIR:
			fs_rewinddir(lmp, requestor, request, nargs);
			break;
		case PVMFS_ISATTY:
			fs_isatty(lmp, requestor, request, nargs);
			break;
		case PVMFS_TTYNAME:
			fs_ttyname(lmp, requestor, request, nargs);
			break;
		case PVMFS_CTERMID:
			fs_ctermid(lmp, requestor, request, nargs);
			break;
		case PVMFS_TCDRAIN:
			fs_tcdrain(lmp, requestor, request, nargs);
			break;
		case PVMFS_TCFLOW:
			fs_tcflow(lmp, requestor, request, nargs);
			break;
		case PVMFS_TCFLUSH:
			fs_tcflush(lmp, requestor, request, nargs);
			break;
		case PVMFS_TCSENDBREAK:
			fs_tcsendbreak(lmp, requestor, request, nargs);
			break;
		case PVMFS_PATHCONF:
			fs_pathconf(lmp, requestor, request, nargs);
			break;
		case PVMFS_FPATHCONF:
			fs_fpathconf(lmp, requestor, request, nargs);
			break;
		case PVMFS_GETRLIMIT:
			fs_getrlimit(lmp, requestor, request, nargs);
			break;
		case PVMFS_SETRLIMIT:
			fs_setrlimit(lmp, requestor, request, nargs);
			break;
		case PVMFS_GETDTABLESIZE:
			fs_getdtablesize(lmp, requestor, request, nargs);
			break;
		case PVMFS_GETDIRENTRIES:
			fs_getdirentries(lmp, requestor, request, nargs);
			break;
		case PVMFS_GETCWD:
			fs_getcwd(lmp, requestor, request, nargs);
			break;
		case PVMFS_SEEKDIR:
			fs_seekdir(lmp, requestor, request, nargs);
			break;
		case PVMFS_TELLDIR:
			fs_telldir(lmp, requestor, request, nargs);
			break;
		default:
			fprintf(stderr, "fs: bad request: %d\n", request);
			dprint((buf2,
			  "fs: bad nargs in request %d from t%xn", request,
			  requestor));
			break;
		}
	}
}
