/*
 *                 Author:  Christopher G. Phillips
 *              Copyright (C) 1993 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * The author makes no representations about the suitability of this
 * software for any purpose.  This software is provided ``as is''
 * without express or implied warranty.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <utime.h>
#include <dirent.h>
#include <termios.h>
#include <sys/uio.h>
#include <pvm3.h>
#include "fs.h"
#include "fsp.h"
#include "map.h"
#include "bitops.h"
#ifndef PATH_MAX
#define PATH_MAX 1024
#endif

#ifndef IMA_RS6K
#include <syscall.h>
#else
#include <sys/select.h>
#include "syscall-fake.h"
#endif

#ifdef TRACE
/*
 * Debugging and trace stuff.
 */
extern int	pvmdebug;

#define dprint(s)	do {	\
				if (pvmdebug) {	\
					sprintf s;	\
					syscall(SYS_write, 2, buf2, \
					  strlen(buf2)); \
				}	\
			} while (0)
#define dprintrw(s)	do {	\
				if (1) { \
					sprintf s;	\
					syscall(SYS_write, 2, buf2, \
					  strlen(buf2)); \
				}	\
			} while (0)
extern char	buf2[];

#else
#define dprint(s)
#define dprintrw(s)
#endif

extern struct gmap	globalmap;
extern struct lmap	*lmap;
extern struct gop	*gop;

extern size_t	nummaps;
extern size_t	nummapsused;
extern int	*tidorder;
extern int	errno;

int	defiomode = PvmIomodeIndividual;

extern int		fdtorealfd __ProtoGlarp__((struct lmap *, int));
extern int		pvm_local_open_oflag __ProtoGlarp__((int));
extern int		pvm_open_oflag __ProtoGlarp__((int));
extern mode_t		pvm_local_open_mode __ProtoGlarp__((long mode));
extern mode_t		pvm_open_mode __ProtoGlarp__((long mode));
extern int		pvm_local_lseek_whence __ProtoGlarp__((int));
extern int		pvm_lseek_whence __ProtoGlarp__((int));
extern int		pvm_local_fcntl_cmd __ProtoGlarp__((int));
extern short		pvm_local_fcntl_ltype __ProtoGlarp__((short));
extern short		pvm_fcntl_ltype __ProtoGlarp__((short));
extern int		pvm_local_access_mode __ProtoGlarp__((int));
extern void		pvm_select_fdset_pack __ProtoGlarp__((int, fd_set *));
extern int		pvm_select_fdset_unpack __ProtoGlarp__((fd_set *));
extern int		pvm_local_tcflow_action __ProtoGlarp__((int));
extern int		pvm_local_tcflush_queue __ProtoGlarp__((int));
extern int		pvm_local_pathconf_name __ProtoGlarp__((int));
extern int		pvm_local_rlimit_resource __ProtoGlarp__((int));
extern int		pvmerrno __ProtoGlarp__((int));
extern struct gop	*gop_add __ProtoGlarp__((int, int, ino_t, dev_t, long));
extern int		gop_delete __ProtoGlarp__((struct gop *));
extern int		gop_doclose __ProtoGlarp__((int));
extern struct gop	*gop_find __ProtoGlarp__((int, int, ino_t, dev_t, int));
extern struct gop	*gop_first __ProtoGlarp__((int));
extern int		allset __ProtoGlarp__((unsigned char *));
extern void		gop_queue_add __ProtoGlarp__((struct gop *,
			  struct queued *));
extern struct queued	*gop_queue_find __ProtoGlarp__((struct gop *, int));
extern void		gop_queue_delete __ProtoGlarp__((struct gop *, int));
extern int		tid2order __ProtoGlarp__((int));
extern int		tid2bitpos __ProtoGlarp__((int));

void
fatal(const char *s)
{
	char	buf[PATH_MAX + BUFSIZ];

	sprintf(buf, "FS FATAL: %s", s);
	perror(buf);
	abort();
}

void
fs_open(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		i;
	int		iresult;
	int		oflag;
	mode_t		mode;
	long		pvmmode;
	char		pathname[PATH_MAX];
	int		iomode;
	int		doit = 0;
	struct stat	stbuf;
	struct gop	*gopp;

	if (!(nargs == 3 || nargs == 4 && request == PVMFS_OPEN)) {
		dprint((buf2, "FS: bad nargs in %s request from t%x\n",
		  request == PVMFS_OPEN ? "open" : "creat", requestor));
		return;
	}

	/*
	 * Find an available client file descriptor.
	 */
	for (i = 0; i < globalmap.numfds; i++)
		if (!lmp->lfd[i].fd)
			break;
	if (i == globalmap.numfds) {
		iresult = -1;
		errno = pvmerrno(EMFILE);
		dprint((buf2, "FS: tid t%x open(...) = -1 (errno = EMFILE)\n",
		  requestor));
		pvm_packf("%+ %d %d", PvmDataDefault, iresult, errno);
		pvm_send(requestor, request);
		return;
	}

	/*
	 * Get the arguments and determine if really need to call open.
	 */
	pvm_unpackf("%s", pathname);
	if (request == PVMFS_OPEN) {
		pvm_unpackf("%d", &oflag);
		oflag = pvm_local_open_oflag(oflag);
	}
	if (request != PVMFS_OPEN || nargs == 4) {
		pvm_unpackf("%ld", &pvmmode);
		mode = pvm_local_open_mode(pvmmode);
	}
	pvm_unpackf("%d", &iomode);
	switch (iomode) {
	case PvmIomodeIndividual:
		doit = 1;
		break;
	case PvmIomodeCommon:
	case PvmIomodeIndependent:
	case PvmIomodeSyncBC:
	case PvmIomodeSyncSeq:
		if (stat(pathname, &stbuf) == 0 && (gopp = gop_find(
		  request, -1, stbuf.st_ino, stbuf.st_dev, lmp - lmap)))
			iresult = gopp->result;
		else
			doit = 1;
		break;
	default:
		iresult = -1;
		errno = EINVAL;
		break;
	}

	if (doit) {
		if (request == PVMFS_OPEN) {
			if (nargs == 3)
				iresult = open(pathname, oflag);
			else
				iresult = open(pathname, oflag, mode);
		} else
			iresult = creat(pathname, mode);
	}

	if (iresult == -1) {
		pvm_packf("%+ %d", PvmDataDefault, iresult);
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		if (request == PVMFS_CREAT)
			dprint((buf2,
			"FS: tid t%x creat(\"%s\", %ld) = -1 (pvmerrno = %d)\n",
			requestor, pathname, mode, iresult));
		else if (nargs == 3)
			dprint((buf2,
			  "FS: tid t%x open(\"%s\", %d) = -1 (pvmerrno = %d)\n",
			  requestor, pathname, oflag, iresult));
		else
			dprint((buf2,
		     "FS: tid t%x open(\"%s\", %d, %ld) = -1 (pvmerrno = %d)\n",
		     requestor, pathname, oflag, (long)mode, iresult));
	} else {
		/*
		 * Update the global file descriptor
		 * and global operation mappings.
		 */
		switch (iomode) {
		case PvmIomodeIndividual:
			globalmap.fd[iresult].nrefs = 1;
			globalmap.fd[iresult].iomode = iomode;
			break;
		case PvmIomodeCommon:
		case PvmIomodeIndependent:
		case PvmIomodeSyncBC:
		case PvmIomodeSyncSeq:
			if (doit) {
				if (stat(pathname, &stbuf) == -1) {
					char	s[PATH_MAX + 32];

					sprintf(s, "can't stat %s", pathname);
					fatal(s);
				}
				globalmap.fd[iresult].st_ino = stbuf.st_ino;
				globalmap.fd[iresult].st_dev = stbuf.st_dev;
				globalmap.fd[iresult].nrefs = 1;
				if ((gopp = gop_add(request, iresult,
				  stbuf.st_ino, stbuf.st_dev, iresult))
				  == NULL)
					fatal("malloc");
				globalmap.fd[iresult].iomode = iomode;
			} else
				globalmap.fd[iresult].nrefs++;
			if (iomode == PvmIomodeIndependent)
				lmp->lfd[i].offset = 0;
			setbit(gopp->done, lmp - lmap);
			gopp->numdone++;
			if (allset(gopp->done))
				gop_delete(gopp);
			break;
		}

		/*
		 * Update the client's file descriptor mapping.
		 */
		globalmap.fd[iresult].from_start = 0;
		/*globalmap.fd[iresult].iomode = defiomode;*/
		lmp->lfd[i].fd = &globalmap.fd[iresult];
		lmp->lfd[i].iomode = iomode;
		pvm_packf("%+ %d", PvmDataDefault, i);
		if (request == PVMFS_CREAT)
			dprint((buf2,
			  "FS: tid t%x creat(\"%s\", %ld) = %d\n",
			  requestor, pathname, mode, iresult));
		else if (nargs == 3)
			dprint((buf2,
			  "FS: tid t%x open(\"%s\", %d) = %d\n",
			  requestor, pathname, oflag, iresult));
		else
			dprint((buf2,
			  "FS: tid t%x open(\"%s\", %d, %ld) = %d\n",
			  requestor, pathname, oflag, (long)mode,
			  iresult));
	}
	pvm_send(requestor, request);
}
	
void
fs_close(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	gfd;
	int	iresult;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in close request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &fd);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
		dprint((buf2, "FS: tid t%x close(%d) = -1 (errno = EBADF)\n",
		  requestor, gfd));
	} else {
		int		doit = 0;
		struct fd	*fdp = &globalmap.fd[gfd];

		/*
		 * Determine if should really close the file descriptor.
		 */
		switch (lmp->lfd[fd].iomode) {
		case PvmIomodeIndividual:
			if (--fdp->nrefs == 0 && !fdp->from_start)
				doit = 1;
			else
				iresult = 0;
			dprint((buf2,
			  "FS: global fd %d now has %d references\n", gfd,
			  fdp->nrefs));
			break;
		case PvmIomodeCommon:
		case PvmIomodeIndependent:
		case PvmIomodeSyncBC:
		case PvmIomodeSyncSeq:
			if (--fdp->nrefs == 0 && !fdp->from_start
			  && gop_doclose(gfd))
				doit = 1;
			else
				iresult = 0;
			dprint((buf2,
			  "FS: global fd %d now has %d references\n", gfd,
			  fdp->nrefs));
			break;
		}
		lmp->lfd[fd].fd = NULL;
		
		if (doit) {
			iresult = close(gfd);
			globalmap.fd[gfd].from_start = 0;
		}
	}
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2, "FS: tid t%x close(%d) = -1 (pvmerrno = %d)\n",
		  requestor, gfd, iresult));
	} else {
		globalmap.fd[gfd].from_start
		  = globalmap.fd[gfd].iomode = 0;
		lmp->lfd[fd].fd = NULL;
		dprint((buf2, "FS: tid t%x close(%d) = %d\n",
		  requestor, gfd, iresult));
	}
	pvm_send(requestor, request);
}

void
fs_lseek(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	gfd;
	int	whence;
	long	offset;
	int	iresult;
	long	lresult;

	if (nargs != 3) {
		dprint((buf2, "FS: bad nargs in lseek request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d %ld %d", &fd, &offset, &whence);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		lresult = -1L;
		errno = EBADF;
		dprint((buf2,
		  "FS: tid t%x lseek(%d, %ld, %d) = -1 (errno = EBADF)\n",
		  requestor, fd, offset, whence));
	} else if ((whence = pvm_local_lseek_whence(whence)) == -1) {
		lresult = -1L;
		errno = EINVAL;
		dprint((buf2,
		  "FS: tid t%x lseek(%d, %ld, %d) = -1 (errno = EINVAL)\n",
		  requestor, fd, offset, whence));
	} else {
		int		doit = 0;

		switch (lmp->lfd[fd].iomode) {
		case PvmIomodeIndividual:
		case PvmIomodeCommon:
		case PvmIomodeIndependent:
		case PvmIomodeSyncBC:
		case PvmIomodeSyncSeq:
			doit = 1;
			break;
		}

		if (doit)
			lresult = lseek(gfd, (off_t)offset, whence);

		if (lmp->lfd[fd].iomode == PvmIomodeIndependent && lresult >= 0)
			lmp->lfd[fd].offset = lresult;
	}
	pvm_packf("%+ %ld", PvmDataDefault, lresult);
	if (lresult == -1L) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x lseek(%d, %ld, %d) = -1 (pvmerrno = %d)\n",
		  requestor, fd, offset, whence, iresult));
	} else
		dprint((buf2, "FS: tid t%x lseek(%d, %ld, %d) = %ld\n",
		  requestor, fd, offset, whence, lresult));
	pvm_send(requestor, request);
}

static char	*read_buffer = NULL;
static size_t	read_buffer_size = 0;

long
do_read(int fd, long nbytes)
{
	char	*tmp;

	/*
	 * Go ahead and call read.
	 */
	if (nbytes > read_buffer_size) {
#if defined(IMA_SUN3) || defined(IMA_SUN4)
		if (read_buffer == NULL) {
			if ((tmp = malloc(nbytes)) == NULL)
				return -1L;
		} else
#endif
		if ((tmp = realloc(read_buffer, nbytes)) == NULL)
			return -1L;

		read_buffer = tmp;
		read_buffer_size = nbytes;
	}
	return (long)read(fd, read_buffer, nbytes);
}

void
send_read(int requestor, int request, int gfd, long nbytes, long lresult, char *buf, void **counts)
{
	int	iresult;
	long	left;
	long	count;

	/*
	 * Reply to the client.
	 */
	pvm_packf("%+ %ld", PvmDataDefault, lresult);
	if (lresult == -1L) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprintrw((buf2,
		  "FS: tid t%x %s(%d, ..., %s%ld) = -1 (pvmerrno = %d)\n",
		  requestor, request == PVMFS_READ ? "read" : "readv", gfd,
		  request == PVMFS_READ ? "" : "total bytes = ", nbytes,
		  lresult));
	} else if (lresult > 0L) {
		if (request == PVMFS_READV) {
			for (left = lresult; left > 0; left -= count) {
				if (counts)
					count = *(long *)*counts++;
				else
					pvm_unpackf("%ld", &count);
				dprintrw((buf2,
				  "FS: left is %ld and count is %ld\n", left,
				  count));
				if (left < count)
					count = left;
				dprintrw((buf2,
				  "FS: Now left is %ld and count is %ld\n",
				  left, count));
				pvm_packf("%*c", (int)count,
				  buf + lresult - left);
				dprintrw((buf2, "FS: Packing \"%*.*s\"\n",
				  (int)count, (int)count,
				  buf + lresult - left));
			}
			dprintrw((buf2,
			  "FS: finished with left = %ld\n", left));
			if (left)
				abort();
		} else
			pvm_packf("%*c", (int)lresult, buf);
		dprintrw((buf2, "FS: tid t%x %s(%d, ..., %s%ld) = %ld\n",
		  requestor, request == PVMFS_READ ? "read" : "readv", gfd,
		  request == PVMFS_READ ? "" : "total bytes = ", nbytes,
		  lresult));
	} else
		dprintrw((buf2, "FS: tid t%x %s(%d, ..., %s%ld) = 0\n",
		  requestor, request == PVMFS_READ ? "read" : "readv", gfd,
		  request == PVMFS_READ ? "" : "total bytes = ", nbytes));
	pvm_send(requestor, request);
}

int
flush_read(struct gop *gp, int min, int maxp1, int request)
{
	int		i;
	long		nbytes;
	long		lresult;
	struct queued	*qp;

	/*
	 * Check if any clients waiting for a SyncSeq read
	 * should now be responded to.
	 */
	for (i = min; i < maxp1; i++) {
		if (lmap[i].tid == 0)
			continue;
		if (qp = gop_queue_find(gp, i)) {
			nbytes = *(long *)qp->args[0];
			lresult = do_read(qp->fd, nbytes);
			dprintrw((buf2, "FSR: %d READING - got %ld\n",
			  qp->number, lresult));
			send_read(tidorder[i], request, qp->fd, nbytes, lresult,
			  read_buffer, request == PVMFS_READ ? NULL
			  : &qp->args[1]);
			setbit(gp->done, tid2bitpos(tidorder[i]));
			gp->numdone++;
			gop_queue_delete(gp, i);
		} else
			break;
	}

	return i;
}

void
fs_read(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		fd;
	int		gfd;
	long		lresult;
	long		nbytes;
	char		*buf;
	int		me;
	struct queued	*qp;
	struct gop	*gopp;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in %s request from t%x\n",
		  request == PVMFS_READ ? "read" : "readv", requestor));
		return;
	}

	pvm_unpackf("%d %ld", &fd, &nbytes);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		lresult = -1L;
		errno = EBADF;
	} else {
		int		queueit = 0;

		/*
		 * What really happens during a read?
		 * Read below for all the gory details!
		 */
		switch (lmp->lfd[fd].iomode) {
		case PvmIomodeIndependent:
			if (lseek(gfd, lmp->lfd[fd].offset, SEEK_SET) == -1) {
				lresult = -1L;
				break;
			}
			/* FALLTHROUGH */
		case PvmIomodeIndividual:
		case PvmIomodeCommon:
			if ((lresult = do_read(gfd, nbytes)) > 0)
				lmp->lfd[fd].offset += lresult;
			buf = read_buffer;
			break;
		case PvmIomodeSyncBC:
			if (gopp = gop_find(request, gfd,
			  globalmap.fd[gfd].st_ino, globalmap.fd[gfd].st_dev,
			  lmp - lmap)) {
				buf = gopp->buf;
				lresult = gopp->result;
			} else {
				lresult = do_read(gfd, nbytes);
				buf = read_buffer;
				if ((gopp = gop_add(request, gfd,
				  globalmap.fd[gfd].st_ino,
				  globalmap.fd[gfd].st_dev, lresult)) == NULL)
					fatal("malloc");
				if (lresult > 0) {
					if ((gopp->buf = malloc(lresult))
					  == NULL)
						fatal("malloc");
					memcpy(gopp->buf, read_buffer, lresult);
				}
			}
			break;
		case PvmIomodeSyncSeq:
			if ((gopp = gop_find(request, gfd,
			  globalmap.fd[gfd].st_ino, globalmap.fd[gfd].st_dev,
			  lmp - lmap)) == NULL
			  && (gopp = gop_add(request, gfd,
			  globalmap.fd[gfd].st_ino,
			  globalmap.fd[gfd].st_dev, 0)) == NULL)
				fatal("malloc");
			if ((me = tid2order(requestor)) == -1)
				fatal("tid2order");
			if (gopp != gop_first(request))
				queueit = 1;
			else {
				/*
				 * A previous global read hasn't finished.
				 */
				(void)flush_read(gopp, gopp->numdone, me,
				  request);
				if (me == gopp->numdone) {
					/*
					 * It's this client's turn!
					 */
					lresult = do_read(gfd, nbytes);
					dprintrw((buf2,
					  "FSR: %d READING - got %ld\n", me,
					  lresult));
					send_read(requestor, request, gfd,
					  nbytes, lresult, read_buffer, NULL);
					if (qp = gop_queue_find(gopp, me))
						gop_queue_delete(gopp, me);
					/*
					 * See if we can now do other reads.
					 */
					if (flush_read(gopp, me + 1, nummaps,
					  request) == nummaps) {
						struct gop	*gop2;

						for (gop2 = gopp->next;
						  gop2
						  && gop2->type != request;
						  gop2 = gop2->next)
							;
						if (gop2
						  && gop2->type == request)
							(void)flush_read(gop2,
							  gop2->numdone, me,
							  request);
					}
				} else
					queueit = 1;
			}
			if (queueit) {
				if ((qp = malloc(sizeof(*qp))) == NULL)
					fatal("malloc");
				qp->number = me;
				qp->tid = lmp->tid;
				qp->fd = gfd;
				dprintrw((buf2, "FSR: %d READ BEING QUEUED\n",
				  me));
				qp->nargs = 1;
				if ((qp->args = malloc(sizeof(*qp->args)))
				  == NULL
				  || (qp->args[0] = malloc(sizeof(long)))
				  == NULL)
					fatal("malloc");
				*(long *)qp->args[0] = nbytes;
				if ((qp->type = request) == PVMFS_READV) {
					long	count;
					long	left;

					for (left = nbytes; left > 0;
					  left -= count) {
						pvm_unpackf("%ld", &count);
if ((qp->args = realloc(qp->args, ++qp->nargs * sizeof(*qp->args))) == NULL
  || (qp->args[qp->nargs - 1] = malloc(sizeof(long))) == NULL)
							fatal("malloc");
						*(long *)qp->args[qp->nargs - 1]
						  = count;
					}
				}
				gop_queue_add(gopp, qp);
				return;
			}
			break;
		}

		if (lmp->lfd[fd].iomode == PvmIomodeSyncBC
		  || lmp->lfd[fd].iomode == PvmIomodeSyncSeq) {
			setbit(gopp->done, lmp - lmap);
			gopp->numdone++;
			if (lmp->lfd[fd].iomode == PvmIomodeSyncBC)
				send_read(requestor, request, gfd, nbytes,
				  lresult, buf, NULL);
			if (allset(gopp->done))
				gop_delete(gopp);
			return;
		}
	}

	send_read(requestor, request, gfd, nbytes, lresult, buf, NULL);
}

static char	*write_buffer = NULL;
static size_t	write_buffer_size = 0;

long
unpack_write(int request, long nbytes)
{
	char	*tmp;
	long	n;
	long	count;

	/*
	 * Unpack what they want to write.
	 */
	if (nbytes > write_buffer_size) {
#if defined(IMA_SUN3) || defined(IMA_SUN4)
		if (write_buffer == NULL) {
			if ((tmp = malloc(nbytes)) == NULL)
				return -1L;
		} else
#endif
		if ((tmp = realloc(write_buffer, nbytes)) == NULL)
			return -1L;

		write_buffer = tmp;
		write_buffer_size = nbytes;
	}
	if (nbytes > 0)
		if (request == PVMFS_WRITEV) {
			for (n = 0; n < nbytes; n += count) {
				pvm_unpackf("%ld", &count);
				pvm_unpackf("%*c", (int)count,
				  write_buffer + n);
			}
		} else
			pvm_unpackf("%*c", (int)nbytes, write_buffer);
#if 0
	dprintrw((buf2, "FS: unpack_write got \"%*.*s\"\n", (int)nbytes,
	  (int)nbytes, write_buffer));
#endif

	return 0L;
}

void
send_write(int requestor, int request, int gfd, long nbytes, long lresult)
{
	int	iresult;

	/*
	 * Respond to a client's write request.
	 */
	pvm_packf("%+ %ld", PvmDataDefault, lresult);
	if (lresult == -1L) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprintrw((buf2,
		  "FS: tid t%x %s(%d, ..., %s%ld) = -1 (pvmerrno = %d)\n",
		  requestor, request == PVMFS_WRITE ? "write" : "writev", gfd,
		  request == PVMFS_WRITE ? "" : "total bytes = ", nbytes,
		  lresult));
	} else
		dprintrw((buf2, "FS: tid t%x %s(%d, ..., %s%ld) = %ld\n",
		  requestor, request == PVMFS_WRITE ? "write" : "writev", gfd,
		  request == PVMFS_WRITE ? "" : "total bytes = ", nbytes,
		  lresult));
	pvm_send(requestor, request);
}

int
flush_write(struct gop *gp, int min, int maxp1, int request)
{
	int		i;
	long		nbytes;
	long		lresult;
	struct queued	*qp;

	/*
	 * Analogous to flush_read...
	 */
	for (i = min; i < maxp1; i++) {
		if (lmap[i].tid == 0)
			continue;
		if (qp = gop_queue_find(gp, i)) {
			nbytes = *(long *)qp->args[0];
			lresult = write(qp->fd, qp->args[1], nbytes);
			dprintrw((buf2, "FSR: %d WRITING - got %ld\n",
			  qp->number, lresult));
			send_write(tidorder[i], request, qp->fd, nbytes,
			  lresult);
			setbit(gp->done, tid2bitpos(tidorder[i]));
			gp->numdone++;
			gop_queue_delete(gp, i);
		} else
			break;
	}

	return i;
}

void
fs_write(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		fd;
	int		gfd;
	int		iresult;
	long		lresult;
	long		nbytes;
	int		me;
	struct queued	*qp;
	struct gop	*gopp;
	int		i;

	if (nargs != 3) {
		dprint((buf2, "FS: bad nargs in %s request from t%x\n",
		  request == PVMFS_WRITE ? "write" : "writev", requestor));
		return;
	}

	pvm_unpackf("%d %ld", &fd, &nbytes);
	if (unpack_write(request, nbytes) == -1)
		lresult = -1L;
	else if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		lresult = -1L;
		errno = EBADF;
	} else {
		int		queueit = 0;

		/*
		 * Much of this is also analogous to fs_read...
		 */
		switch (lmp->lfd[fd].iomode) {
		case PvmIomodeIndependent:
			if (lseek(gfd, lmp->lfd[fd].offset, SEEK_SET) == -1) {
				lresult = -1L;
				break;
			}
			/* FALLTHROUGH */
		case PvmIomodeIndividual:
		case PvmIomodeCommon:
			if ((lresult = write(gfd, write_buffer, nbytes)) > 0)
				lmp->lfd[fd].offset += lresult;
			break;
		case PvmIomodeSyncBC:
			if ((gopp = gop_find(request, gfd,
			  globalmap.fd[gfd].st_ino, globalmap.fd[gfd].st_dev,
			  lmp - lmap)) == NULL
			  && (gopp = gop_add(request, gfd,
			  globalmap.fd[gfd].st_ino, globalmap.fd[gfd].st_dev,
			  0)) == NULL)
				fatal("malloc");
			if ((me = tid2order(requestor)) == -1)
				fatal("tid2order");
			else if (me == 0) {
				lresult = write(gfd, write_buffer, nbytes);
				gopp->result = lresult;
				for (i = 0; i < nummaps; i++) {
					if (lmap[i].tid == 0)
						continue;
					if (qp = gop_queue_find(gopp, i)) {
						nbytes = *(long *)qp->args[0];
						send_write(tidorder[i], request,
						  qp->fd, nbytes, lresult);
						setbit(gopp->done,
						  tid2bitpos(tidorder[i]));
						gopp->numdone++;
						gop_queue_delete(gopp, i);
					}
				}
			/*
			 * I suppose we should queue these
			 * but let's go ahead and act like it worked.
			 */
#if 0
                        } else if (gopp->result == -2) {
                                /* queue it */
                                if ((qp = malloc(sizeof(*qp))) == NULL)
                                        fatal("malloc");
                                qp->number = me;
                                dprintrw((buf2, "FSR: %d WRITE BEING QUEUED\n",
                                  me));
                                qp->type = request;
                                qp->fd = gfd;
                                qp->nargs = 1;
                                if ((qp->args = malloc(2 * sizeof(*qp->args)))
                                  == NULL
                                  || (qp->args[0] = malloc(sizeof(long)))
                                  == NULL)
                                        fatal("malloc");
                                *(long *)qp->args[0] = nbytes;
                                qp->tid = lmp->tid;
                                gop_queue_add(gopp, qp);
                                return;
#endif
			} else 
				lresult = gopp->result;
			break;
		case PvmIomodeSyncSeq:
			if ((gopp = gop_find(request, gfd,
			  globalmap.fd[gfd].st_ino, globalmap.fd[gfd].st_dev,
			  lmp - lmap)) == NULL
			  && (gopp = gop_add(request, gfd,
			  globalmap.fd[gfd].st_ino, globalmap.fd[gfd].st_dev,
			  0)) == NULL)
				fatal("malloc");
			if ((me = tid2order(requestor)) == -1)
				fatal("tid2order");
			if (gopp != gop_first(request))
				queueit = 1;
			else {
				/*
				 * A previous global write hasn't finished.
				 */
				(void)flush_write(gopp, gopp->numdone, me,
				  request);
				if (me == gopp->numdone) {
					/*
					 * It's this client's turn!
					 */
					lresult = write(gfd, write_buffer,
					  nbytes);
					dprintrw((buf2,
					  "FSR: %d WRITING - got %ld\n", me,
					  lresult));
					send_write(requestor, request, gfd,
					  nbytes, lresult);
					if (qp = gop_queue_find(gopp, me))
						gop_queue_delete(gopp, me);
					/*
					 * See if we can now do other writes.
					 */
					if (flush_write(gopp, me + 1, nummaps,
					  request) == nummaps) {
						struct gop	*gop2;

						for (gop2 = gopp->next;
						  gop2
						  && gop2->type != request;
						  gop2 = gop2->next)
							;
						if (gop2
						  && gop2->type == request)
							(void)flush_write(gop2,
							  gop2->numdone, me,
							  request);
					}
				} else
					queueit = 1;
			}
			if (queueit) {
				if ((qp = malloc(sizeof(*qp))) == NULL)
					fatal("malloc");
				qp->number = me;
				dprintrw((buf2, "FSR: %d WRITE BEING QUEUED\n",
				  me));
				qp->type = request;
				qp->fd = gfd;
				qp->nargs = 2;
				if ((qp->args = malloc(2 * sizeof(*qp->args)))
				  == NULL
				  || (qp->args[0] = malloc(sizeof(long)))
				  == NULL
				  || nbytes
				  && (qp->args[1] = malloc(nbytes)) == NULL)
					fatal("malloc");
				*(long *)qp->args[0] = nbytes;
				if (nbytes)
					memcpy(qp->args[1], write_buffer,
					  nbytes);
				else
					qp->args[1] = NULL;
				qp->tid = lmp->tid;
				gop_queue_add(gopp, qp);
				return;
			}
			break;
		}

		if (lmp->lfd[fd].iomode == PvmIomodeSyncBC
		  || lmp->lfd[fd].iomode == PvmIomodeSyncSeq) {
			setbit(gopp->done, lmp - lmap);
			gopp->numdone++;
			if (lmp->lfd[fd].iomode == PvmIomodeSyncBC)
				send_write(requestor, request, gfd, nbytes,
				  lresult);
			if (allset(gopp->done))
				gop_delete(gopp);
			return;
		}
	}

	send_write(requestor, request, gfd, nbytes, lresult);
}

void
fs_dup(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	gfd;
	int	fd2;
	int	gfd2;
	int	iresult = 0;
	int	i;

	if (nargs != 1 && request == PVMFS_DUP
	  || nargs != 2 && request == PVMFS_DUP2) {
		dprint((buf2, "FS: bad nargs in %s request from t%x\n",
		  request == PVMFS_DUP ? "dup" : "dup2", requestor));
		return;
	}

	pvm_unpackf("%d", &fd);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else if (request == PVMFS_DUP) {
		for (i = 0; i < globalmap.numfds; i++)
			if (!lmp->lfd[i].fd)
				break;
		if (i == globalmap.numfds) {
			iresult = -1;
			errno = EMFILE;
		} else {
			lmp->lfd[i] = lmp->lfd[fd];
			lmp->lfd[i].fd->nrefs++;
			iresult = i;
		}
	} else {
		/* request == PVMFS_DUP2 */
		pvm_unpackf("%d", &fd2);
		if (fd2 < 0 || fd2 >= globalmap.numfds) {
			iresult = -1;
			errno = EBADF;
		} else if (fd != fd2) {
			struct fd	*fdp;

			if ((gfd2 = fdtorealfd(lmp, fd2)) >= 0) {
				fdp = &globalmap.fd[gfd2];
				if (fdp->nrefs)
					if (--fdp->nrefs == 0)
						close(gfd2);
			}
			lmp->lfd[fd2] = lmp->lfd[fd];
			lmp->lfd[fd2].fd->nrefs++;
			iresult = fd2;
		} else
			iresult = fd;
	}

	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2, "FS: tid t%x dup(%d) = -1 (pvmerrno = %d)\n",
		  requestor, fd, iresult));
	} else
		dprint((buf2, "FS: tid t%x dup(%d) = %d\n",
		  requestor, fd, iresult));
	pvm_send(requestor, request);
}

void
fs_setioorder(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	order;
	int	iresult;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in setioorder request from t%x\n",
		  requestor));
		return;
	}

	/*
	 * Always works as long as there are at least ``order'' clients.
	 */
	pvm_unpackf("%d", &order);
	if (order < nummapsused) {
		tidorder[order] = requestor;
		iresult = 0;
	} else
		iresult = -1;
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	pvm_send(requestor, request);
}

void
fs_getioorder(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	i;
	int	iresult;

	if (nargs != 0) {
		dprint((buf2, "FS: bad nargs in getioorder request from t%x\n",
		  requestor));
		return;
	}

	iresult = -1;
	for (i = 0; i < nummaps; i++)
		if (lmap[i].tid > 0 && tidorder[i] == requestor) {
			iresult = i;
			break;
		}

	pvm_packf("%+ %d", PvmDataDefault, iresult);
	pvm_send(requestor, request);
}

void
fs_setiomode(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	gfd;
	int	mode;
	int	iresult;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in setiomode request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d %d", &fd, &mode);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else {
		switch (globalmap.fd[gfd].iomode) {
		case PvmIomodeIndividual:
			if (mode == PvmIomodeIndividual)
				iresult = 0;
			else {
				/*
				 * Can't change from PvmIomodeIndividual
				 * (not right now anyway).
				 */
				iresult = -1;
				errno = EINVAL;
			}
			break;
		case PvmIomodeCommon:
		case PvmIomodeIndependent:
		case PvmIomodeSyncBC:
		case PvmIomodeSyncSeq:
			/*
			 * Go ahead and allow the change.  Whether things
			 * make sense after this is their problem.
			 */
			iresult = 0;
			lmp->lfd[fd].iomode = mode;
			break;
		default:
			iresult = -1;
			errno = EINVAL;
			break;
		}
	}
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
	}
	pvm_send(requestor, request);
}

void
fs_getiomode(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	iresult;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in getiomode request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &fd);
	if (lmp->lfd[fd].fd == NULL) {
		iresult = -1;
		errno = pvmerrno(EBADF);
		pvm_packf("%+ %d %d", PvmDataDefault, iresult, errno);
	} else
		pvm_packf("%+ %d", PvmDataDefault, lmp->lfd[fd].iomode);
	pvm_send(requestor, request);
}

static void
putstat(struct stat *stp)
{
	long	l;

#ifdef __STDC__
#define put(member)	l = (long)stp->st_##member; pvm_packf("%ld", l);
	put(dev);
	put(ino);
	put(mode);
	put(nlink);
	put(uid);
	put(gid);
	put(rdev);
	put(size);
	put(atime);
	put(ctime);
	put(mtime);
	put(blksize);
	put(blocks);
#undef put
#else
	l = (long)stp->st_dev;		pvm_packf("%ld", l);
	l = (long)stp->st_ino;		pvm_packf("%ld", l);
	l = (long)stp->st_mode;		pvm_packf("%ld", l);
	l = (long)stp->st_nlink;	pvm_packf("%ld", l);
	l = (long)stp->st_uid;		pvm_packf("%ld", l);
	l = (long)stp->st_gid;		pvm_packf("%ld", l);
	l = (long)stp->st_rdev;		pvm_packf("%ld", l);
	l = (long)stp->st_size;		pvm_packf("%ld", l);
	l = (long)stp->st_atime;	pvm_packf("%ld", l);
	l = (long)stp->st_ctime;	pvm_packf("%ld", l);
	l = (long)stp->st_mtime;	pvm_packf("%ld", l);
	l = (long)stp->st_blksize;	pvm_packf("%ld", l);
	l = (long)stp->st_blocks;	pvm_packf("%ld", l);
#endif
}

void
fs_stat(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		iresult;
	char		pathname[PATH_MAX];
	struct stat	stbuf;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in stat request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s", pathname);
	iresult = stat(pathname, &stbuf);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x stat(\"%s\", %p) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, &stbuf, iresult));
	} else {
		putstat(&stbuf);
		dprint((buf2, "FS: tid t%x stat(\"%s\", %p) = %d\n",
		  requestor, pathname, &stbuf, iresult));
	}
	pvm_send(requestor, request);
}

void
fs_lstat(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		iresult;
	char		pathname[PATH_MAX];
	struct stat	stbuf;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in lstat request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s", pathname);
	iresult = lstat(pathname, &stbuf);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x lstat(\"%s\", %p) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, &stbuf, iresult));
	} else {
		putstat(&stbuf);
		dprint((buf2, "FS: tid t%x lstat(\"%s\", %p) = %d\n",
		  requestor, pathname, &stbuf, iresult));
	}
	pvm_send(requestor, request);
}

void
fs_fstat(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		fd;
	int		gfd;
	int		iresult;
	struct stat	stbuf;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in fstat request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &fd);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else
		iresult = fstat(gfd, &stbuf);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x fstat(%d, %p) = -1 (pvmerrno = %d)\n", requestor,
		  fd, &stbuf, iresult));
	} else {
		putstat(&stbuf);
		dprint((buf2, "FS: tid t%x fstat(%d, %p) = %d\n",
		  requestor, fd, &stbuf, iresult));
	}
	pvm_send(requestor, request);
}

void
fs_fcntl(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		fd;
	int		fd2;
	int		gfd;
	int		command;
	int		iresult;
	int		flags;
	int		i;
	long		l1, l2;
	struct flock	lock;

	if (nargs != 3) {
		dprint((buf2, "FS: bad nargs in fcntl request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d %d", &fd, &command);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else if ((command = pvm_local_fcntl_cmd(command)) == -1) {
		iresult = -1;
		errno = EINVAL;
	} else
		/*
		 * This stuff may be here but it won't work
		 * for all cases because the local OS won't
		 * distinguish between the clients...
		 */
		switch (command) {
		case F_GETLK:
		case F_SETLK:
		case F_SETLKW:
			pvm_unpackf("%hd %ld %hd %ld", &lock.l_type, &l1,
			  &lock.l_whence, &l2);
			lock.l_start = (off_t)l1;
			lock.l_type = pvm_local_fcntl_ltype(lock.l_type);
			lock.l_len = (off_t)l2;
			lock.l_whence = pvm_local_lseek_whence(lock.l_whence);
			iresult = fcntl(gfd, command, &lock);
			break;
		case F_DUPFD:
			pvm_unpackf("%d", &fd2);
			for (i = fd2; i < globalmap.numfds; i++)
				if (!lmp->lfd[i].fd)
					break;
			if (i == globalmap.numfds) {
				iresult = -1;
				errno = EMFILE;
			} else {
				lmp->lfd[i] = lmp->lfd[fd];
				lmp->lfd[i].fd->nrefs++;
				iresult = i;
			}
			break;
		default:
			pvm_unpackf("%d", &flags);
			if (command == F_SETFL)
				flags = pvm_local_open_oflag(flags);
			iresult = fcntl(gfd, command, flags);
			if (iresult != -1 && command == F_GETFL)
				iresult = pvm_open_oflag(iresult);
			break;
		}

	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		switch (command) {
		case F_GETLK:
		case F_SETLK:
		case F_SETLKW:
			dprint((buf2,
			"FS: tid t%x fcntl(%d, %d, ...) = -1 (pvmerrno = %d)\n",
			requestor, fd, command, iresult));
			break;
		default:
			dprint((buf2, "FS: tid t%x fcntl(%d, %d, %d) = %d\n",
			  requestor, fd, command, flags, iresult));
			break;
		}
	} else
		switch (command) {
		case F_GETLK:
			lock.l_type = pvm_fcntl_ltype(lock.l_type);
			pvm_packf("%d", iresult);
			if (lock.l_type != F_UNLCK) {
				l1 = (long)lock.l_start;
				lock.l_whence = (short)
				  pvm_lseek_whence(lock.l_whence);
				l2 = (long)lock.l_len;
				pvm_packf("%ld %hd %ld", l1, lock.l_whence, l2);
			}
		case F_SETLK:
		case F_SETLKW:
			dprint((buf2,
			"FS: tid t%x fcntl(%d, %d, ...) = -1 (pvmerrno = %d)\n",
			requestor, fd, command, iresult));
			break;
		case F_DUPFD:
			/* convert fd */
		case F_GETFL:
		default:
			dprint((buf2, "FS: tid t%x fcntl(%d, %d, %d) = %d\n",
			  requestor, fd, command, flags, iresult));
			break;
		}

	pvm_send(requestor, request);
}

void
fs_unlink(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in unlink request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s", pathname);
	iresult = unlink(pathname);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x unlink(\"%s\") = -1 (pvmerrno = %d)\n",
		  requestor, pathname, iresult));
	} else
		dprint((buf2, "FS: tid t%x unlink(\"%s\") = %d\n",
		  requestor, pathname, iresult));
	pvm_send(requestor, request);
}

void
fs_rename(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	oldpathname[PATH_MAX];
	char	newpathname[PATH_MAX];

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in rename request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %s", oldpathname, newpathname);
	iresult = rename(oldpathname, newpathname);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x rename(\"%s\", \"%s\") = -1 (pvmerrno = %d)\n",
		  requestor, oldpathname, newpathname, iresult));
	} else
		dprint((buf2, "FS: tid t%x rename(\"%s\", \"%s\") = %d\n",
		  requestor, oldpathname, newpathname, iresult));
	pvm_send(requestor, request);
}

void
fs_mkdir(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];
	long	pvmmode;
	mode_t	mode;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in mkdir request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %ld", pathname, &pvmmode);
	mode = pvm_local_open_mode(pvmmode);
	iresult = mkdir(pathname, mode);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x mkdir(\"%s\", %ld) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, (long)mode, iresult));
	} else
		dprint((buf2, "FS: tid t%x mkdir(\"%s\", %ld) = %d\n",
		  requestor, pathname, (long)mode, iresult));
	pvm_send(requestor, request);
}

void
fs_rmdir(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in rmdir request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s", pathname);
	iresult = rmdir(pathname);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x rmdir(\"%s\") = -1 (pvmerrno = %d)\n", requestor,
		  pathname, iresult));
	} else
		dprint((buf2, "FS: tid t%x rmdir(\"%s\") = %d\n",
		  requestor, pathname, iresult));
	pvm_send(requestor, request);
}

void
fs_chroot(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in chroot request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s", pathname);
	iresult = chroot(pathname);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x chroot(\"%s\") = -1 (pvmerrno = %d)\n",
		  requestor, pathname, iresult));
	} else
		dprint((buf2, "FS: tid t%x chroot(\"%s\") = %d\n",
		  requestor, pathname, iresult));
	pvm_send(requestor, request);
}

void
fs_fchroot(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	int	fd;
	int	gfd;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in fchroot request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &fd);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else
		iresult = fchroot(gfd);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2, "FS: tid t%x fchroot(%d) = -1 (pvmerrno = %d)\n",
		  requestor, fd, iresult));
	} else
		dprint((buf2, "FS: tid t%x fchroot(%d) = %d\n",
		  requestor, fd, iresult));
	pvm_send(requestor, request);
}

void
fs_chdir(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in chdir request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s", pathname);
	iresult = chdir(pathname);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x chdir(\"%s\") = -1 (pvmerrno = %d)\n", requestor,
		  pathname, iresult));
	} else
		dprint((buf2, "FS: tid t%x chdir(\"%s\") = %d\n",
		  requestor, pathname, iresult));
	pvm_send(requestor, request);
}

void
fs_fchdir(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	int	fd;
	int	gfd;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in fchdir request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &fd);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else
		iresult = fchdir(gfd);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2, "FS: tid t%x fchdir(%d) = -1 (pvmerrno = %d)\n",
		  requestor, fd, iresult));
	} else
		dprint((buf2, "FS: tid t%x fchdir(%d) = %d\n",
		  requestor, fd, iresult));
	pvm_send(requestor, request);
}

void
fs_chmod(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];
	long	pvmmode;
	mode_t	mode;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in chmod request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %ld", pathname, &pvmmode);
	mode = pvm_local_open_mode(pvmmode);
	iresult = chmod(pathname, mode);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x chmod(\"%s\", %ld) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, (long)mode, iresult));
	} else
		dprint((buf2, "FS: tid t%x chmod(\"%s\", %ld) = %d\n",
		  requestor, pathname, (long)mode, iresult));
	pvm_send(requestor, request);
}

void
fs_fchmod(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	int	fd;
	int	gfd;
	long	pvmmode;
	mode_t	mode;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in fchmod request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d %ld", &fd, &pvmmode);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else {
		mode = pvm_local_open_mode(pvmmode);
		iresult = fchmod(gfd, mode);
	}
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x fchmod(%d, %ld) = -1 (pvmerrno = %d)\n",
		  requestor, fd, (long)mode, iresult));
	} else
		dprint((buf2, "FS: tid t%x fchmod(%d, %ld) = %d\n",
		  requestor, fd, (long)mode, iresult));
	pvm_send(requestor, request);
}

void
fs_chown(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];
	long	pvmowner;
	long	pvmgroup;
	uid_t	owner;
	gid_t	group;

	if (nargs != 3) {
		dprint((buf2, "FS: bad nargs in chown request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %ld %ld", pathname, &pvmowner, &pvmgroup);
	owner = pvmowner;
	group = pvmgroup;
	iresult = chown(pathname, owner, group);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x chown(\"%s\", %ld, %ld) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, (long)owner, (long)group, iresult));
	} else
		dprint((buf2, "FS: tid t%x chown(\"%s\", %ld, %ld) = %d\n",
		  requestor, pathname, (long)owner, (long)group, iresult));
	pvm_send(requestor, request);
}

void
fs_fchown(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	int	fd;
	int	gfd;
	long	pvmowner;
	long	pvmgroup;
	uid_t	owner;
	gid_t	group;

	if (nargs != 3) {
		dprint((buf2, "FS: bad nargs in fchown request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d %ld %ld", &fd, &pvmowner, &pvmgroup);
	owner = pvmowner;
	group = pvmgroup;
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else
		iresult = fchown(gfd, owner, group);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x fchown(%d, %ld, %ld) = -1 (pvmerrno = %d)\n",
		  requestor, fd, (long)owner, (long)group, iresult));
	} else
		dprint((buf2, "FS: tid t%x fchown(%d, %ld, %ld) = %d\n",
		  requestor, fd, (long)owner, (long)group, iresult));
	pvm_send(requestor, request);
}

void
fs_lchown(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];
	long	pvmowner;
	long	pvmgroup;
	uid_t	owner;
	gid_t	group;

	if (nargs != 3) {
		dprint((buf2, "FS: bad nargs in lchown request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %ld %ld", pathname, &pvmowner, &pvmgroup);
	owner = pvmowner;
	group = pvmgroup;
	iresult = lchown(pathname, owner, group);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x lchown(\"%s\", %ld, %ld) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, (long)owner, (long)group, iresult));
	} else
		dprint((buf2, "FS: tid t%x lchown(\"%s\", %ld, %ld) = %d\n",
		  requestor, pathname, (long)owner, (long)group, iresult));
	pvm_send(requestor, request);
}

void
fs_link(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	oldpathname[PATH_MAX];
	char	newpathname[PATH_MAX];

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in link request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %s", oldpathname, newpathname);
	iresult = link(oldpathname, newpathname);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x link(\"%s\", \"%s\") = -1 (pvmerrno = %d)\n",
		  requestor, oldpathname, newpathname, iresult));
	} else
		dprint((buf2, "FS: tid t%x link(\"%s\", \"%s\") = %d\n",
		  requestor, oldpathname, newpathname, iresult));
	pvm_send(requestor, request);
}

void
fs_symlink(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	oldpathname[PATH_MAX];
	char	newpathname[PATH_MAX];

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in symlink request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %s", oldpathname, newpathname);
	iresult = symlink(oldpathname, newpathname);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x symlink(\"%s\", \"%s\") = -1 (pvmerrno = %d)\n",
		  requestor, oldpathname, newpathname, iresult));
	} else
		dprint((buf2, "FS: tid t%x symlink(\"%s\", \"%s\") = %d\n",
		  requestor, oldpathname, newpathname, iresult));
	pvm_send(requestor, request);
}

void
fs_mkfifo(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];
	long	pvmmode;
	mode_t	mode;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in mkfifo request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %ld", pathname, &pvmmode);
	mode = pvm_local_open_mode(pvmmode);
	iresult = mkfifo(pathname, mode);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x mkfifo(\"%s\", %ld) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, mode, iresult));
	} else
		dprint((buf2, "FS: tid t%x mkfifo(\"%s\", %ld) = %d\n",
		  requestor, pathname, mode, iresult));
	pvm_send(requestor, request);
}

void
fs_mknod(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];
	long	pvmmode;
	mode_t	mode;
	long	pvmdev;
	dev_t	dev;

	if (nargs != 3) {
		dprint((buf2, "FS: bad nargs in mknod request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %ld %ld", pathname, &pvmmode, &pvmdev);
	mode = pvm_local_open_mode(pvmmode);
	dev = pvmdev;
	iresult = mknod(pathname, mode, dev);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x mknod(\"%s\", %ld, %ld) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, mode, iresult));
	} else
		dprint((buf2, "FS: tid t%x mknod(\"%s\", %ld, %ld) = %d\n",
		  requestor, pathname, mode, iresult));
	pvm_send(requestor, request);
}

void
fs_pipe(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd[2];
	int	gfd[2];
	int	iresult = 0;
	int	i;

	if (nargs != 0) {
		dprint((buf2, "FS: bad nargs in pipe request from t%x\n",
		  requestor));
		return;
	}

	/*
	 * Find 2 unused client file descriptors.
	 */
	for (i = 0; i <= 1; i++) {
		for (fd[i] = 0; fd[i] < globalmap.numfds; fd[i]++)
			if (!lmp->lfd[fd[i]].fd && (i == 0 || fd[i] != fd[0]))
				break;
		if (fd[i] == globalmap.numfds) {
			iresult = -1;
			errno = EMFILE;
			break;
		}
	}

	if (iresult == 0)
		iresult = pipe(gfd);

	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2, "FS: tid t%x pipe(%p) = -1 (pvmerrno = %d)\n",
		  requestor, gfd, iresult));
	} else {
		/*
		 * Update global and local file descriptor mappings.
		 * Pretty, huh?
		 */
		globalmap.fd[gfd[0]].nrefs = 1;
		globalmap.fd[gfd[0]].iomode = PvmIomodeIndividual;
		globalmap.fd[gfd[0]].from_start = 0;
		lmp->lfd[fd[0]].fd = &globalmap.fd[gfd[0]];
		lmp->lfd[fd[0]].iomode = PvmIomodeIndividual;
		globalmap.fd[gfd[1]].nrefs = 1;
		globalmap.fd[gfd[1]].iomode = PvmIomodeIndividual;
		globalmap.fd[gfd[1]].from_start = 0;
		lmp->lfd[fd[1]].fd = &globalmap.fd[gfd[1]];
		lmp->lfd[fd[1]].iomode = PvmIomodeIndividual;
		pvm_packf("%d %d", fd[0], fd[1]);
		dprint((buf2,
		  "FS: tid t%x pipe(%p) = %d <gfds: %d, %d fds: %d, %d>\n",
		  requestor, gfd, iresult, gfd[0], gfd[1], fd[0], fd[1]));
	}
	pvm_send(requestor, request);
}

void
fs_readlink(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];
	char	linkname[PATH_MAX];
	int	len;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in readlink request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %d", pathname, &len);
	iresult = readlink(pathname, linkname, len);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x readlink(\"%s\", %p, %d) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, linkname, len, iresult));
	} else if (iresult == len) {
		pvm_packf("%*.*c", iresult, iresult, linkname);
		dprint((buf2,
		  "FS: tid t%x readlink(\"%s\", %p, %d) = %d \"%*.*s\"\n",
		  requestor, pathname, linkname, len, len, len, len, linkname));
	} else {
		pvm_packf("%s", linkname);
		dprint((buf2,
		  "FS: tid t%x readlink(\"%s\", %p, %d) = %d \"%s\"\n",
		  requestor, pathname, linkname, len, iresult, linkname));
	}
	pvm_send(requestor, request);
}

#if 0

/*
 * Give up on statfs for now.  Not much consistency out there...
 */

#ifdef _AIX
#include <sys/statfs.h>
#endif
#ifdef __alpha
#include <sys/mount.h>
#endif
#ifdef __convex__
#include <sys/vfs.h>
#endif

static void
putstatfs(struct statfs *stp)
{
	long	l;

#define put(member)	l = (long)stp->f_##member; pvm_packf("%ld", l);
	put(type);
	put(bsize);
	put(blocks);
	put(bfree);
	put(bavail);
	put(files);
	put(ffree);
#if 0
	/* these don't seem to be very uniform */
	put(fsid);
	put(fname);
	put(name_max);
#endif
#undef put
}

void
fs_statfs(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		iresult;
	char		pathname[PATH_MAX];
	struct statfs	stbuf;
	int		len;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in statfs request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %d", pathname, &len);
	iresult = statfs(pathname, &stbuf, len);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x statfs(\"%s\", %p, %d) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, &stbuf, len, iresult));
	} else {
		putstatfs(&stbuf);
		dprint((buf2, "FS: tid t%x statfs(\"%s\", %p, %d) = %d\n",
		  requestor, pathname, &stbuf, len, iresult));
	}
	pvm_send(requestor, request);
}

void
fs_fstatfs(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		fd;
	int		gfd;
	int		iresult;
	struct statfs	stbuf;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in fstatfs request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &fd);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else
		iresult = fstatfs(gfd, &stbuf);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x fstatfs(%d, %p, %d) = -1 (pvmerrno = %d)\n",
		  requestor, fd, &stbuf, len, iresult));
	} else {
		putstatfs(&stbuf);
		dprint((buf2, "FS: tid t%x fstatfs(%d, %p, %d) = %d\n",
		  requestor, fd, &stbuf, len, iresult));
	}
	pvm_send(requestor, request);
}
#endif

void
fs_truncate(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];
	long	llen;
	off_t	len;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in truncate request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %ld", pathname, &llen);
	len = llen;;
	iresult = truncate(pathname, len);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x truncate(\"%s\", %ld) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, (long)len, iresult));
	} else
		dprint((buf2, "FS: tid t%x truncate(\"%s\", %ld) = %d\n",
		  requestor, pathname, (long)len, iresult));
	pvm_send(requestor, request);
}

void
fs_ftruncate(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	int	fd;
	int	gfd;
	long	llen;
	off_t	len;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in ftruncate request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d %ld", &fd, &llen);
	len = llen;
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else {
		struct fd	*fdp = &globalmap.fd[gfd];

		switch (lmp->lfd[fd].iomode) {
		case PvmIomodeIndividual:
			iresult = ftruncate(gfd, len);
			break;
		default:
			/*
			 * This is to stop FORTRAN libraries from doing
			 * write, lseek, ftruncate for files opened with
			 * mode 'unknown'.  One process could squeeze
			 * a write between the lseek and ftruncate of
			 * another process...
			 */
			if (fdp->nrefs == 1)
				iresult = ftruncate(gfd, len);
			else
				iresult = 0;
		}
	}
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x ftruncate(%d, %ld) = -1 (pvmerrno = %d)\n",
		  requestor, fd, (long)len, iresult));
	} else
		dprint((buf2, "FS: tid t%x ftruncate(%d, %ld) = %d\n",
		  requestor, fd, (long)len, iresult));
	pvm_send(requestor, request);
}

void
fs_umask(struct lmap *lmp, int requestor, int request, int nargs)
{
	mode_t	mode;
	long	oldmode;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in umask request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%ld", &oldmode);
	mode = pvm_local_open_mode(oldmode);
	oldmode = pvm_open_mode(umask(mode));
	pvm_packf("%+ %ld", PvmDataDefault, oldmode);
	dprint((buf2, "FS: tid t%x umask(%ld) = %ld\n", requestor, (long)mode,
	  oldmode));
	pvm_send(requestor, request);
}

void
fs_utime(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		iresult;
	char		pathname[PATH_MAX];
	struct utimbuf	ut;
	struct utimbuf	*utp = NULL;
	long		l1, l2;

	if (nargs != 1 && nargs != 3) {
		dprint((buf2, "FS: bad nargs in utime request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s", pathname);
	if (nargs == 3) {
		pvm_unpackf("%ld %ld", &l1, &l2);
		utp = &ut;
		utp->actime = l1;
		utp->modtime = l2;
	}
	iresult = utime(pathname, utp);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x utime(\"%s\", %p) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, utp, iresult));
	} else
		dprint((buf2, "FS: tid t%x utime(\"%s\", %p) = %d\n",
		  requestor, pathname, utp, iresult));
	pvm_send(requestor, request);
}

void
fs_utimes(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		iresult;
	char		pathname[PATH_MAX];
	struct timeval	tv[2];
	long		l1, l2, l3, l4;

	if (nargs != 5) {
		dprint((buf2, "FS: bad nargs in utimes request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %ld %ld %ld %ld", pathname, &l1, &l2, &l3, &l4);
	tv[0].tv_sec = l1;
	tv[0].tv_usec = l2;
	tv[1].tv_sec = l3;
	tv[1].tv_usec = l4;
	iresult = utimes(pathname, tv);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x utimes(\"%s\", %p) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, tv, iresult));
	} else
		dprint((buf2, "FS: tid t%x utimes(\"%s\", %p) = %d\n",
		  requestor, pathname, tv, iresult));
	pvm_send(requestor, request);
}

void
fs_sync(struct lmap *lmp, int requestor, int request, int nargs)
{
	dprint((buf2, "FS: tid t%x sync()\n", requestor));
	syscall(SYS_sync);
}

void
fs_fsync(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	gfd;
	int	iresult;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in fsync request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &fd);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else
		iresult = fsync(gfd);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2, "FS: tid t%x fsync(%d) = -1 (pvmerrno = %d)\n",
		  requestor, fd, iresult));
	} else
		dprint((buf2, "FS: tid t%x fsync(%d) = %d\n", requestor, fd,
		  iresult));
	pvm_send(requestor, request);
}

void
fs_access(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	char	pathname[PATH_MAX];
	int	pvmmode;
	int	mode;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in access request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %d", pathname, &pvmmode);
	mode = pvm_local_access_mode(pvmmode);
	iresult = access(pathname, mode);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x access(\"%s\", %d) = -1 (pvmerrno = %d)\n",
		  requestor, pathname, pvmmode, iresult));
	} else
		dprint((buf2, "FS: tid t%x access(\"%s\", %d) = %d\n",
		  requestor, pathname, pvmmode, iresult));
	pvm_send(requestor, request);
}

static int
fs_convert_fdset(struct lmap *lmp, int maxfdp1, int gmaxfdp1, fd_set *setp, fd_set *gsetp)
{
	int	numset = 0;
	int	numdone = 0;
	int	fd;
	int	gfd;
	fd_set	tmp;

	FD_ZERO(&tmp);

	for (fd = 0; fd < maxfdp1; fd++) {
		if (!FD_ISSET(fd, setp))
			continue;
		if ((gfd = fdtorealfd(lmp, fd)) >= 0 && FD_ISSET(gfd, gsetp)) {
			FD_SET(fd, &tmp);
			numset++;	
		}
	}

	FD_ZERO(setp);
	for (fd = 0; fd < maxfdp1; fd++)
		if (FD_ISSET(fd, &tmp)) {
			FD_SET(fd, setp);
			if (++numdone == numset)
				break;
		}

	return numset;
}

int
pvm_select_global_fdset(struct lmap *lmp, int maxfdp1, int *gmaxfdp1p, fd_set *setp, fd_set *gsetp)
{
	int	fd;
	int	gfd;

	FD_ZERO(gsetp);

	for (fd = 0; fd < maxfdp1; fd++)
		if (FD_ISSET(fd, setp)) {
			if ((gfd = fdtorealfd(lmp, fd)) == -1)
				return -1;
			FD_SET(gfd, gsetp);
			if (gfd + 1 > *gmaxfdp1p)
				*gmaxfdp1p = gfd + 1;
		}

	return 0;
}

void
fs_select(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		iresult;
	int		maxfdp1, gmaxfdp1;
	int		want_rset, want_wset, want_eset;
	fd_set		rset, wset, eset;
	fd_set		grset, gwset, geset;
	struct timeval	tv;
	struct timeval	*tvp;

	if (nargs != 4 && nargs != 5) {
		dprint((buf2, "FS: bad nargs in select request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &maxfdp1);
	if (maxfdp1 > globalmap.numfds)
		maxfdp1 = globalmap.numfds;
	if (maxfdp1 > FD_SETSIZE)
		maxfdp1 = FD_SETSIZE;
	want_rset = pvm_select_fdset_unpack(&rset);
	want_wset = pvm_select_fdset_unpack(&wset);
	want_eset = pvm_select_fdset_unpack(&eset);
	if (nargs == 5) {
		long	l;

		tvp = &tv;
		pvm_unpackf("%ld", &l);	tvp->tv_sec = l;
		pvm_unpackf("%ld", &l);	tvp->tv_usec = l;
	} else
		tvp = NULL;
	gmaxfdp1 = 0;
	if (want_rset && pvm_select_global_fdset(lmp, maxfdp1, &gmaxfdp1, &rset,
	  &grset) == -1 ||
	  want_wset && pvm_select_global_fdset(lmp, maxfdp1, &gmaxfdp1, &wset,
	  &gwset) == -1 ||
	  want_eset && pvm_select_global_fdset(lmp, maxfdp1, &gmaxfdp1, &eset,
	  &geset) == -1) {
		iresult = -1;
		errno = EBADF;
	} else
		iresult = select(gmaxfdp1, want_rset ? &grset : NULL,
		  want_wset ? &gwset : NULL, want_eset ? &geset : NULL, tvp);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x select(%d, ...) = -1 (pvmerrno = %d)\n",
		  requestor, maxfdp1, iresult));
	} else if (iresult > 0) {
		iresult = 0;
		if (want_rset) {
			iresult += fs_convert_fdset(lmp, maxfdp1, gmaxfdp1,
			  &rset, &grset);
			pvm_select_fdset_pack(maxfdp1, &rset);
		}
		if (want_wset) {
			iresult += fs_convert_fdset(lmp, maxfdp1, gmaxfdp1,
			  &wset, &gwset);
			pvm_select_fdset_pack(maxfdp1, &wset);
		}
		if (want_eset) {
			iresult += fs_convert_fdset(lmp, maxfdp1, gmaxfdp1,
			  &eset, &geset);
			pvm_select_fdset_pack(maxfdp1, &eset);
		}
		dprint((buf2, "FS: tid t%x select(%d, ...) = %d\n", requestor,
		  maxfdp1, iresult));
	} else
		dprint((buf2, "FS: tid t%x select(%d, ...) = 0\n", requestor,
		  maxfdp1));
	pvm_send(requestor, request);
}

/*
 * Mapping for POSIX directory functions.
 */
struct dirmap {
	DIR		*dp;
	struct dirent	*dirp;
};

static struct dirmap	*dirmap = NULL;
static int		numdirs = 0;

void
fs_opendir(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		i;
	int		iresult;
	char		pathname[PATH_MAX];

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in opendir request from t%x\n",
		  requestor));
		return;
	}

	for (i = 0; i < numdirs; i++)
		if (dirmap[i].dp == NULL)
			break;
	if (i == numdirs) {
		struct dirmap	*tmp;

#if defined(IMA_SUN3) || defined(IMA_SUN4)
		if (dirmap == NULL && (tmp = malloc(sizeof(*dirmap))) == NULL
		  || dirmap &&
#else
		if  (
#endif
		  (tmp = realloc(dirmap, (numdirs + 1) * sizeof(*dirmap)))
		  == NULL) {
			iresult = -1;
			errno = pvmerrno(errno);
			pvm_packf("%+ %d %d", PvmDataDefault, iresult, errno);
			dprint((buf2,
			  "FS: tid t%x opendir(...) = -1 (pvmerrno = %d)\n",
			  requestor, errno));
			pvm_send(requestor, request);
			return;
		}
		numdirs++;
		dirmap = tmp;
	}

	pvm_unpackf("%s", pathname);
	dirmap[i].dp = opendir(pathname);
	if (dirmap[i].dp == NULL) {
		iresult = -1;
		errno = pvmerrno(errno);
		pvm_packf("%+ %d %d", PvmDataDefault, iresult, errno);
		dprint((buf2,
		  "FS: tid t%x opendir(\"%s\") = NULL (pvmerrno = %d)\n",
		  requestor, pathname, errno));
	} else {
		pvm_packf("%+ %d", PvmDataDefault, i);
		dprint((buf2, "FS: tid t%x opendir(\"%s\") = %p (index %d)\n",
		  requestor, pathname, (void *)dirmap[i].dp, i));
	}
	pvm_send(requestor, request);
}

void
fs_readdir(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	i;
	int	iresult;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in readdir request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &i);
	if (i < 0 || i >= numdirs || dirmap[i].dp == NULL) {
		iresult = -1;
		errno = pvmerrno(EBADF);
		pvm_packf("%+ %d %d", PvmDataDefault, iresult, errno);
		dprint((buf2,
		  "FS: tid t%x readdir(index %d) = NULL (errno = EBADF)\n",
		  requestor, i + 1));
		pvm_send(requestor, request);
		return;
	}

	/*
	 * We set errno to -2.  If we get NULL and errno is still -2,
	 * we reset errno back to saved_errno.
	 */
	errno = -2;
	dirmap[i].dirp = readdir(dirmap[i].dp);
	if (dirmap[i].dirp == NULL) {
		iresult = -1;
		errno = pvmerrno(errno);
		pvm_packf("%+ %d %d", PvmDataDefault, iresult, errno);
		dprint((buf2,
		  "FS: tid t%x readdir(index %d) = NULL (pvmerrno = %d)\n",
		  requestor, i + 1, errno));
	} else {
		iresult = 0;
		pvm_packf("%+ %d %s", PvmDataDefault, iresult,
		  dirmap[i].dirp->d_name);
		dprint((buf2, "FS: tid t%x readdir(index %d) = %p\n",
		  requestor, i + 1, (void *)dirmap[i].dirp));
	}
	pvm_send(requestor, request);
}

void
fs_closedir(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	i;
	int	iresult;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in closedir request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &i);
	if (i < 0 || i >= numdirs || dirmap[i].dp == NULL) {
		iresult = -1;
		errno = pvmerrno(EBADF);
		pvm_packf("%+ %d %d", PvmDataDefault, iresult, errno);
		dprint((buf2,
		  "FS: tid t%x closedir(index %d) = -1 (errno = EBADF)\n",
		  requestor, i));
		pvm_send(requestor, request);
		return;
	}

	iresult = closedir(dirmap[i].dp);
	if (iresult == -1) {
		errno = pvmerrno(errno);
		pvm_packf("%+ %d %d", PvmDataDefault, iresult, errno);
		dprint((buf2,
		  "FS: tid t%x closedir(index %d) = -1 (pvmerrno = %d)\n",
		  requestor, i, errno));
	} else {
		dirmap[i].dp = NULL;
		pvm_packf("%+ %d", PvmDataDefault, iresult);
		dprint((buf2, "FS: tid t%x closedir(index %d) = %d\n",
		  requestor, i, iresult));
	}
	pvm_send(requestor, request);
}

void
fs_rewinddir(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	i;
	int	iresult;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in rewinddir request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &i);
	if (i < 0 || i >= numdirs || dirmap[i].dp == NULL) {
		iresult = -1;
		errno = pvmerrno(EBADF);
		pvm_packf("%+ %d %d", PvmDataDefault, iresult, errno);
		dprint((buf2, "FS: tid t%x rewinddir(index %d) - bad index\n",
		  requestor, i));
		pvm_send(requestor, request);
		return;
	}

	rewinddir(dirmap[i].dp);
	iresult = 0;
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	dprint((buf2, "FS: tid t%x rewinddir(index %d) ok\n",
	  requestor, i));
	pvm_send(requestor, request);
}

void
fs_isatty(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	gfd;
	int	iresult;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in isatty request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &fd);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = 0;
		errno = EBADF;
	} else
		iresult = isatty(gfd);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == 0) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2, "FS: tid t%x isatty(%d) = 0 (pvmerrno = %d)\n",
		  requestor, fd, iresult));
	} else
		dprint((buf2, "FS: tid t%x isatty(%d) = %d\n", requestor, fd,
		  iresult));
	pvm_send(requestor, request);
}

void
fs_ttyname(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	gfd;
	int	len;
	char	*cresult;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in ttyname request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &fd);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		cresult = NULL;
		errno = EBADF;
		len = 0;
	} else {
		cresult = ttyname(gfd);
		len = strlen(cresult);
	}
	pvm_packf("%+ %d", PvmDataDefault, len);
	if (len == 0) {
		len = pvmerrno(errno);
		pvm_packf("%d", len);
		dprint((buf2,
		  "FS: tid t%x ttyname(%d) = NULL (pvmerrno = %d)\n",
		  requestor, fd, len));
	} else {
		pvm_packf("%s", cresult);
		dprint((buf2, "FS: tid t%x ttyname(%d) = %s\n", requestor, fd,
		  cresult));
	}
	pvm_send(requestor, request);
}

void
fs_ctermid(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	len;
	char	*cresult;

	if (nargs != 0) {
		dprint((buf2, "FS: bad nargs in ctermid request from t%x\n",
		  requestor));
		return;
	}

	cresult = ctermid(NULL);
	len = strlen(cresult);

	pvm_packf("%+ %d", PvmDataDefault, len);
	if (len == 0) {
		len = pvmerrno(errno);
		pvm_packf("%d", len);
		dprint((buf2,
		  "FS: tid t%x ctermid(%d) = NULL (pvmerrno = %d)\n",
		  requestor, fd, len));
	} else {
		pvm_packf("%s", cresult);
		dprint((buf2, "FS: tid t%x ctermid(NULL) = %s\n", requestor,
		  cresult));
	}
	pvm_send(requestor, request);
}

void
fs_tcdrain(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	gfd;
	int	iresult;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in tcdrain request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &fd);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else
		iresult = tcdrain(gfd);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2, "FS: tid t%x tcdrain(%d) = -1 (pvmerrno = %d)\n",
		  requestor, fd, iresult));
	} else
		dprint((buf2, "FS: tid t%x tcdrain(%d) = %d\n", requestor, fd,
		  iresult));
	pvm_send(requestor, request);
}

void
fs_tcflow(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	gfd;
	int	action;
	int	iresult;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in tcflow request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d %d", &fd, &action);
	action = pvm_local_tcflow_action(action);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else
		iresult = tcflow(gfd, action);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x tcflow(%d, %d) = -1 (pvmerrno = %d)\n",
		  requestor, fd, action, iresult));
	} else
		dprint((buf2,
		  "FS: tid t%x tcflow(%d, %d) = %d\n", requestor, fd, action,
		  iresult));
	pvm_send(requestor, request);
}

void
fs_tcflush(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	gfd;
	int	queue;
	int	iresult;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in tcflush request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d %d", &fd, &queue);
	queue = pvm_local_tcflush_queue(queue);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else
		iresult = tcflush(gfd, queue);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x tcflush(%d, %d) = -1 (pvmerrno = %d)\n",
		  requestor, fd, queue, iresult));
	} else
		dprint((buf2,
		  "FS: tid t%x tcflush(%d, %d) = %d\n", requestor, fd, queue,
		  iresult));
	pvm_send(requestor, request);
}

void
fs_tcsendbreak(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	fd;
	int	gfd;
	int	duration;
	int	iresult;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in tcsendbreak request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d %d", &fd, &duration);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		iresult = -1;
		errno = EBADF;
	} else
		iresult = tcsendbreak(gfd, duration);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x tcsendbreak(%d, %d) = -1 (pvmerrno = %d)\n",
		  requestor, fd, duration, iresult));
	} else
		dprint((buf2,
		  "FS: tid t%x tcsendbreak(%d, %d) = %d\n", requestor, fd,
		  duration, iresult));
	pvm_send(requestor, request);
}

void
fs_pathconf(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	long	lresult;
	char	pathname[PATH_MAX];
	int	pvmname;
	int	name;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in pathconf request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%s %d", pathname, &pvmname);
	if (pvmname < 0)
		lresult = sysconf(_SC_OPEN_MAX);
	else {
		name = pvm_local_pathconf_name(pvmname);
		lresult = pathconf(pathname, name);
	}
	pvm_packf("%+ %ld", PvmDataDefault, lresult);
	if (lresult == -1L) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		if (pvmname < 0)
			dprint((buf2,
		     "FS: tid t%x sysconf(_SC_OPEN_MAX) = -1 (pvmerrno = %d)\n",
		     requestor, iresult));
		else
			dprint((buf2,
		      "FS: tid t%x pathconf(\"%s\", %d) = -1 (pvmerrno = %d)\n",
		      requestor, pathname, name, iresult));
	} else if (pvmname < 0)
		dprint((buf2, "FS: tid t%x sysconf(_SC_OPEN_MAX) = %ld\n",
		  requestor, lresult));
	else
		dprint((buf2, "FS: tid t%x pathconf(\"%s\", %d) = %ld\n",
		  requestor, pathname, name, lresult));
	pvm_send(requestor, request);
}

void
fs_fpathconf(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;
	long	lresult;
	int	fd;
	int	gfd;
	int	pvmname;
	uid_t	name;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in fpathconf request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d %d", &fd, &pvmname);
	if ((gfd = fdtorealfd(lmp, fd)) == -1) {
		lresult = -1L;
		errno = EBADF;
	} else if (pvmname < 0)
		/*
		 * Get maximum number of file descriptors.
		 */
		lresult = sysconf(_SC_OPEN_MAX);
	else {
		name = pvm_local_pathconf_name(pvmname);
		lresult = fpathconf(gfd, name);
	}
	pvm_packf("%+ %ld", PvmDataDefault, lresult);
	if (lresult == -1L) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		if (pvmname < 0)
			dprint((buf2,
		     "FS: tid t%x sysconf(_SC_OPEN_MAX) = -1 (pvmerrno = %d)\n",
		     requestor, iresult));
		else
			dprint((buf2,
			 "FS: tid t%x fpathconf(%d, %d) = -1 (pvmerrno = %d)\n",
			 requestor, fd, name, iresult));
	} else if (pvmname < 0)
		dprint((buf2, "FS: tid t%x sysconf(_SC_OPEN_MAX) = %ld\n",
		  requestor, fd, lresult));
	else
		dprint((buf2, "FS: tid t%x fpathconf(%d, %d) = %ld\n",
		  requestor, fd, name, lresult));
	pvm_send(requestor, request);
}

void
fs_getrlimit(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		resource;
	int		iresult;
	struct rlimit	rlim;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in getrlimit request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &resource);
	if ((resource = pvm_local_rlimit_resource(resource)) == -1) {
		iresult = -1;
		errno = EINVAL;
	} else
		iresult = getrlimit(resource, &rlim);
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x getrlimit(%d, %p) = -1 (pvmerrno = %d)\n",
		  requestor, resource, &rlim, iresult));
	} else {
		int		infinity;
		unsigned long	ul;

		infinity = rlim.rlim_cur == RLIM_INFINITY;
		pvm_packf("%d", infinity);
		if (!infinity) {
			ul = rlim.rlim_cur;
			pvm_pkulong(&ul, 1, 1);
		}
		infinity = rlim.rlim_max == RLIM_INFINITY;
		pvm_packf("%d", infinity);
		if (!infinity) {
			ul = rlim.rlim_max;
			pvm_pkulong(&ul, 1, 1);
		}
		dprint((buf2, "FS: tid t%x getrlimit(%d, %p) = %d\n",
		  requestor, resource, &rlim, iresult));
	}
	pvm_send(requestor, request);
}

void
fs_setrlimit(struct lmap *lmp, int requestor, int request, int nargs)
{
	int		resource;
	int		iresult;
	struct rlimit	rlim;

	if (nargs != 3) {
		dprint((buf2, "FS: bad nargs in setrlimit request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &resource);
	if ((resource = pvm_local_rlimit_resource(resource)) == -1) {
		iresult = -1;
		errno = EINVAL;
	} else {
		int		infinity;
		unsigned long	ul;

		pvm_unpackf("%d", &infinity);
		if (infinity)
			rlim.rlim_cur = RLIM_INFINITY;
		else {
			pvm_upkulong(&ul, 1, 1);
			rlim.rlim_cur = ul;
		}
		pvm_unpackf("%d", &infinity);
		if (infinity)
			rlim.rlim_max = RLIM_INFINITY;
		else {
			pvm_upkulong(&ul, 1, 1);
			rlim.rlim_max = ul;
		}
		iresult = setrlimit(resource, &rlim);
	}
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x setrlimit(%d, %p) = -1 (pvmerrno = %d)\n",
		  requestor, resource, (void *)&rlim, iresult));
	} else
		dprint((buf2, "FS: tid t%x setrlimit(%d, %p) = %d\n",
		  requestor, resource, (void *)&rlim, iresult));
	pvm_send(requestor, request);
}

void
fs_getdtablesize(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	iresult;

	if (nargs != 0) {
		dprint((buf2,
		  "FS: bad nargs in getdtablesize request from t%x\n",
		  requestor));
		return;
	}

	iresult = getdtablesize();
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	if (iresult == -1) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x getdtablesize() = -1 (pvmerrno = %d)\n",
		  requestor, iresult));
	} else
		dprint((buf2, "FS: tid t%x getdtablesize() = %d\n", requestor,
		  iresult));
	pvm_send(requestor, request);
}

void
fs_getdirentries(struct lmap *lmp, int requestor, int request, int nargs)
{
}

void
fs_getcwd(struct lmap *lmp, int requestor, int request, int nargs)
{
	unsigned long	ul;
	char		cwd[PATH_MAX];
	int		iresult;

	if (nargs != 0) {
		dprint((buf2, "FS: bad nargs in getcwd request from t%x\n",
		  requestor));
		return;
	}

	if (getcwd(cwd, sizeof cwd))
		ul = strlen(cwd) + 1;
	else
		ul = 0;
	pvm_packf("%+", PvmDataDefault);
	pvm_pkulong(&ul, 1, 1);
	if (ul == 0) {
		iresult = pvmerrno(errno);
		pvm_packf("%d", iresult);
		dprint((buf2,
		  "FS: tid t%x getcwd(...) = NULL (pvmerrno = %d)\n",
		  requestor, iresult));
	} else {
		pvm_packf("%s", cwd);
		dprint((buf2, "FS: tid t%x getcwd(...) = \"%s\"\n", requestor,
		  cwd));
	}
	pvm_send(requestor, request);
}

void
fs_seekdir(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	i;
	int	iresult;
	long	loc;

	if (nargs != 2) {
		dprint((buf2, "FS: bad nargs in seekdir request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d %ld", &i, &loc);
	if (i < 0 || i >= numdirs || dirmap[i].dp == NULL) {
		iresult = -1;
		errno = pvmerrno(EBADF);
		pvm_packf("%+ %d %d", PvmDataDefault, iresult, errno);
		dprint((buf2,
		  "FS: tid t%x seekdir(index %d, %ld) - bad index\n",
		  requestor, i, loc));
		pvm_send(requestor, request);
		return;
	}

	seekdir(dirmap[i].dp, loc);
	iresult = 0;
	pvm_packf("%+ %d", PvmDataDefault, iresult);
	dprint((buf2, "FS: tid t%x seekdir(index %d, %ld) ok\n",
	  requestor, i, loc));
	pvm_send(requestor, request);
}

void
fs_telldir(struct lmap *lmp, int requestor, int request, int nargs)
{
	int	i;
	long	lresult;

	if (nargs != 1) {
		dprint((buf2, "FS: bad nargs in telldir request from t%x\n",
		  requestor));
		return;
	}

	pvm_unpackf("%d", &i);
	if (i < 0 || i >= numdirs || dirmap[i].dp == NULL) {
		lresult = -1;
		errno = pvmerrno(EBADF);
		pvm_packf("%+ %ld %d", PvmDataDefault, lresult, errno);
		dprint((buf2,
		  "FS: tid t%x telldir(index %d) = -1 (errno = EBADF)\n",
		  requestor, i));
		pvm_send(requestor, request);
		return;
	}

	lresult = telldir(dirmap[i].dp);
	if (lresult == -1) {
		errno = pvmerrno(errno);
		pvm_packf("%+ %ld %d", PvmDataDefault, lresult, errno);
		dprint((buf2,
		  "FS: tid t%x telldir(index %d) = -1 (pvmerrno = %d)\n",
		  requestor, i, errno));
	} else {
		dirmap[i].dp = NULL;
		pvm_packf("%+ %ld", PvmDataDefault, lresult);
		dprint((buf2, "FS: tid t%x telldir(index %d) = %ld\n",
		  requestor, i, lresult));
	}
	pvm_send(requestor, request);
}
