/*
 *                 Author:  Christopher G. Phillips
 *              Copyright (C) 1993 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * The author makes no representations about the suitability of this
 * software for any purpose.  This software is provided ``as is''
 * without express or implied warranty.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <limits.h>
#include "map.h"
#include "bitops.h"
#include "fsp.h"

struct gop	*gop = NULL;

extern size_t		nummaps;
extern size_t		nummapsused;
extern struct lmap	*lmap;
extern int		*tidorder;
extern struct gmap	globalmap;

#ifndef IMA_RS6K
#include <syscall.h>
#else
#include "syscall-fake.h"
#endif
extern char	buf2[];
#define dprint(s)	do {	\
				sprintf s;	\
				syscall(SYS_write, 2, buf2, strlen(buf2)); \
			} while (0)
#if 1
#undef dprint
#define dprint(s)
#endif

int
gop_doclose(int fd)
{
	struct gop	*gp;

	/*
	 * Should we *really* close the file descriptor?
	 */
	for (gp = gop; gp; gp = gp->next)
		if ((gp->type == PVMFS_OPEN || gp->type == PVMFS_CREAT)
		  && gp->st_ino == globalmap.fd[fd].st_ino
		  && gp->st_dev == globalmap.fd[fd].st_dev) {
			dprint((buf2, "gop_doclose(%d) = 0\n", fd));
			return 0;
		}

	dprint((buf2, "gop_doclose(%d) = 1\n", fd));
	return 1;
}

struct gop *
gop_add(int type, int fd, ino_t st_ino, dev_t st_dev, long result)
{
	size_t		nbytes;
	struct gop	*glp;
	struct gop	*gp;

	/*
	 * A new global operation.  Add it at the tail.
	 */
	nbytes = (nummaps + CHAR_BIT - 1) / CHAR_BIT;
	if ((glp = malloc(sizeof(*glp))) == NULL
	  || (glp->done = malloc(nbytes)) == NULL) {
		if (glp)
			free(glp);
		return NULL;
	}
	(void)memset(glp->done, '\0', nbytes);
	glp->next = NULL;
	glp->type = type;
	glp->fd = fd;
	glp->st_ino = st_ino;
	glp->st_dev = st_dev;
	glp->result = result;
	glp->buf = NULL;
	glp->numdone = 0;
	glp->queue = NULL;

	if (gop) {
		for (gp = gop; gp->next; gp = gp->next)
			;
		gp->next = glp;
	} else
		gop = glp;
	
	return glp;
}

/* delete at head */
int
gop_delete(void)
{
	struct gop	*gp;

	if (!gop)
		return -1;
	gp = gop;
	gop = gop->next;
	free(gp->done);
	if (gp->buf)
		free(gp->buf);
	free(gp);
	return 0;
}

struct gop *
gop_find(int type, int fd, ino_t ino, dev_t dev, int bitpos)
{
	struct gop	*gp;

	/*
	 * Look for a global operation.
	 */
	for (gp = gop; gp; gp = gp->next)
		if (gp->type == type && isclr(gp->done, bitpos))
			if (type == PVMFS_OPEN || type == PVMFS_CREAT) {
				if (gp->st_ino == ino && gp->st_dev == dev)
					return gp;
			} else if (gp->fd == fd)
				return gp;

	return NULL;
}

struct gop *
gop_first(int type)
{
	struct gop	*gp;

	/*
	 * Find the first global operation of type ``type''.
	 */
	for (gp = gop; gp; gp = gp->next)
		if (gp->type == type)
			return gp;

	return NULL;
}

int
allset(unsigned char *vector)
{
	int	i;

	for (i = 0; i < nummaps; i++)
		if (lmap[i].tid && !isset(vector, i))
			return 0;

	return 1;
}

void
gop_queue_add(struct gop *gp, struct queued *qp)
{
	qp->next = gp->queue;
	gp->queue = qp;
}

struct queued *
gop_queue_find(struct gop *gp, int order)
{
	struct queued	*qp;

	for (qp = gp->queue; qp; qp = qp->next)
		if (qp->number == order)
			return qp;

	return NULL;
}

void
gop_queue_delete(struct gop *gp, int order)
{
	struct queued	*qp;
	struct queued	*prev;
	int		i;

	for (prev = NULL, qp = gp->queue; qp; prev = qp, qp = qp->next)
		if (qp->number == order) {
			if (prev)
				prev->next = qp->next;
			else
				gp->queue = qp->next;
			for (i = 0; i < qp->nargs; i++)
				if (qp->args[i])
					free(qp->args[i]);
			if (qp->args)
				free(qp->args);
			free(qp);
			break;
		}
}

int
tid2order(int tid)
{
	int	i;

	for (i = 0; i < nummaps; i++)
		if (tidorder[i] == tid)
			return i;
	
	return -1;
}
