
/*
 *         PVM version 3.3:  Parallel Virtual Machine System
 *               University of Tennessee, Knoxville TN.
 *           Oak Ridge National Laboratory, Oak Ridge TN.
 *                   Emory University, Atlanta GA.
 *      Authors:  A. L. Beguelin, J. J. Dongarra, G. A. Geist,
 *    W. C. Jiang, R. J. Manchek, B. K. Moore, and V. S. Sunderam
 *                   (C) 1992 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * Neither the Institutions (Emory University, Oak Ridge National
 * Laboratory, and University of Tennessee) nor the Authors make any
 * representations about the suitability of this software for any
 * purpose.  This software is provided ``as is'' without express or
 * implied warranty.
 *
 * PVM version 3 was funded in part by the U.S. Department of Energy,
 * the National Science Foundation and the State of Tennessee.
 */

/*
 *	cruft.c
 *
 *	Missing links and other wonk.
 *
$Log: pvmcruft.c,v $
 * Revision 1.9  1995/07/19  21:26:27  manchek
 * new function pvmnametag converts message tag to name or decimal string
 *
 * Revision 1.8  1995/07/18  17:01:35  manchek
 * added function pvmcrcappend to generate 32-bit crcs
 *
 * Revision 1.7  1995/05/24  16:58:42  manchek
 * whacked doofy coding style in pvmhdump
 *
 * Revision 1.6  1995/05/17  16:27:52  manchek
 * added support for CSPP subcomplexes (/tmp file naming)
 *
 * Revision 1.5  1994/11/07  22:48:09  manchek
 * added tweaks for SX3, SCO
 *
 * Revision 1.4  1994/07/18  19:21:05  manchek
 * pvmputenv() matched variable name substrings (need to compare = too)
 *
 * Revision 1.3  1994/06/03  20:38:22  manchek
 * version 3.3.0
 *
 * Revision 1.2  1993/10/04  20:28:27  manchek
 * added pvmdsockfile() function
 *
 * Revision 1.1  1993/08/30  23:26:50  manchek
 * Initial revision
 *
 */
#include <stdio.h>
#ifndef WIN32
#ifdef IMA_TITN
#include <bsd/sys/types.h>
#else
#include <sys/types.h>
#endif
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#else
#include "pvmwin.h"
#include <sys/types.h>
#include "winsock.h"
#include <time.h>
#endif

#include <ctype.h>
#ifdef	SYSVSTR
#include <string.h>
#define	CINDEX(s,c)	strchr(s,c)
#else
#include <strings.h>
#define	CINDEX(s,c)	index(s,c)
#endif
#include "global.h"
#include "protoglarp.h"
#include "pvmalloc.h"
#include "tdpro.h"
#include "ddpro.h"
#include "pvmsdpro.h"


/***************
 **  Globals  **
 **           **
 ***************/

char *getenv();

extern int pvm_useruid;				/* from pvmd.c or lpvm.c */
#ifdef WIN32 				 	/* no userid in win nt ... */

extern char *username;				/* from pvmd.c or lpvm.c */
#endif

/***************
 **  Private  **
 **           **
 ***************/

static char rcsid[] = "$Id: pvmcruft.c,v 1.9 1995/07/19 21:26:27 manchek Exp $";


int
pvmput32(p, n)
	char *p;
	int n;
{
	*p++ = n >> 24;
	*p++ = n >> 16;
	*p++ = n >> 8;
	*p = n;
	return 0;
}


int
pvmput16(p, n)
	char *p;
	int n;
{
	*p++ = n >> 8;
	*p = n;
	return 0;
}


int
pvmput8(p, n)
	char *p;
	int n;
{
	*p = n;
	return 0;
}


int
pvmget32(p)
	char *p;
{
	return ((0xff & p[0]) << 24)
	+ ((0xff & p[1]) << 16)
	+ ((0xff & p[2]) << 8)
	+ (0xff & p[3]);
}


int
pvmget16(p)
	char *p;
{
	return ((0xff & p[0]) << 8) + (0xff & p[1]);
}


int
pvmget8(p)
	char *p;
{
	return 0xff & *p;
}


#ifdef NOGETDTBLSIZ

#ifdef IMA_TITN

int
getdtablesize()
{
	return 64;
}

#else	/*IMA_TITN*/

#ifdef IMA_SX3

int
getdtablesize()
{
	return 64*8;
}

#else /*IMA_SX3*/
#ifdef WIN32
   
int
getdtablesize()
{
	return 8;
}
#else /* WIN32 */
#include <unistd.h>

int
getdtablesize()
{
	return sysconf(_SC_OPEN_MAX);
}

#endif /* WIN32 */
#endif /*IMA_SX3*/
#endif	/*IMA_TITN*/
#endif /*NOGETDTBLSIZ*/

#if defined(IMA_TITN) || defined(I860_NODE) || defined(I860_SRM) || defined(IMA_SCO) || defined(WIN32)
int
ffs(x)
	int x;
{
	int n = 1, m = 1;

	if (!x)
		return 0;
	while (!(x & m)) {
		m += m;
		n++;
	}
	return n;
}

#endif


extern char **environ;

/*	pvmputenv()
*
*	Replacement for systems that don't have putenv.
*	Tries to be conservative, i.e. doesn't assume that environ
*	points to malloc'd space.
*
*	String must never be freed...
*/

int
pvmputenv(s)
	char *s;	/* string of form "NAME=Value" */
{
	char *p;					/* gp */
	int l;						/* length of env var name or env */
	static char **myen = 0;		/* last environ that i created */
	char **newen;				/* new env if extending */
	char **ep, **ep2;			/* gp */

	/* sanity check the string and locate the '=' */

	if (!s)
		return -1;
	for (p = s; *p && *p != '='; p++);
	if (p == s || !*p)
		return -1;
	l = p - s + 1;	/* must include '=' */

	/* search current environ for name */

	for (ep = environ; *ep; ep++)
		if (!strncmp(*ep, s, l))
			break;

	if (*ep) {	/* already there, replace it */
		*ep = s;

	} else {	/* not there, must extend environ */

	/* length needed is? */

		for (l = 2, ep = environ; *ep; l++, ep++);

		if (environ == myen) {	/* mine, reallocate */
			if (!(newen = (char**)realloc((char *)environ,
					(unsigned)(l * sizeof(char*)))))
				return -1;

		} else {	/* not mine, alloc new and copy */
			if (!(newen = (char**)malloc((unsigned)(l * sizeof(char*)))))
				return -1;
			for (ep = environ, ep2 = newen; *ep2++ = *ep++; );
		}
		newen[l - 2] = s;
		newen[l - 1] = 0;
		environ = myen = newen;
	}
	return 0;
}


/*	pvmxtoi()
*
*	Yet another version of ascii hex to integer
*/

pvmxtoi(p)
	char *p;
{
	int i = 0;
	char c;

	while (isxdigit(c = *p++)) {
		i = (i << 4) + c - (isdigit(c) ? '0' : (isupper(c) ? 'A' : 'a') - 10);
	}
	return i;
}


#ifndef	I860_NODE

/*	hex_inadport()
*
*	Takes a string of format 00000000:0000 and returns a sockaddr_in.
*/

hex_inadport(s, sad)
	char *s;
	struct sockaddr_in *sad;
{
	sad->sin_addr.s_addr = htonl((unsigned)pvmxtoi(s));
	s = (s = CINDEX(s, ':')) ? s + 1 : "";
	sad->sin_port = htons((unsigned short)pvmxtoi(s));
}


/*	inadport_decimal()
*
*	Returns printable string corr. to sockaddr_in with addr in decimal.
*/

char *
inadport_decimal(sad)
	struct sockaddr_in *sad;
{
	static char buf[32];
	int a;

	a = ntohl(0xffffffff & sad->sin_addr.s_addr);
	sprintf(buf, "%d.%d.%d.%d:%d",
			0xff & (a >> 24),
			0xff & (a >> 16),
			0xff & (a >> 8),
			0xff & a,
			0xffff & (int)ntohs(sad->sin_port));
	return buf;
}


/*	inadport_hex()
*
*	Returns printable string corr. to sockaddr_in with addr in hex.
*/

char *
inadport_hex(sad)
	struct sockaddr_in *sad;
{
	static char buf[16];
	int a;

	a = ntohl(0xffffffff & sad->sin_addr.s_addr);
	sprintf(buf, "%08x:%04x", a, 0xffff & (int)ntohs(sad->sin_port));
	return buf;
}

#endif	/*I860_NODE*/


#ifdef	NOTMPNAM
char *
pvmtmpnam(buf)
	char *buf;
{
	static int n = 0;
	static char scratch[64];
	char tmptmp[32];

	if (!buf)
		buf = scratch;
	sprintf(buf,getenv("PVM_TMP"));
	sprintf(tmptmp, "/tmp%06d.%d", getpid(), n++ % 10000);
	strcat(buf,tmptmp);
	return buf;
}
#endif	/*NOTMPNAM*/


pvmhdump(p, n, pad)
	char *p;	/* bytes */
	int n;		/* length */
	char *pad;
{
	int i;
	pad = pad ? pad : "";
	for (i = 0; n-- > 0; i = (i + 1) & 15) {
		fprintf(stderr, "%s%02x%s",
			i ? "" : pad,
			0xff & *p++,
			n && i != 15 ? " " : "\n");
	}
	return 0;
}


/*	pvmgethome()
*
*	Return path of user homedir.
*/

char *
pvmgethome()
{
	static char *hd = 0;

	if (!hd) {
		if (hd = getenv("HOME")) {
			hd = STRALLOC(hd);

		} else {
			pvmlogerror("warning: can't getenv(HOME), using /\n");
			hd = "/";
		}
	}
	return hd;
}


/*	pvmgetroot()
*
*	Return absolute path of PVM top level directory.
*/

char *
pvmgetroot()
{
	static char *rd = 0;
	if (!rd && !(rd = getenv("PVM_ROOT"))) {
		pvmlogerror("PVM_ROOT environment variable not set.\n");
#ifdef IMA_CSPP
		{
			struct stat buf;

			rd = STRALLOC("/usr/convex/pvm");
			if (stat(rd, &buf) == -1) {
				pvmlogperror("Unable to default PVM_ROOT to /usr/convex/pvm");
				pvmbailout(0);
				exit(1);		/* the other meaning of bail out */
			}
			pvmputenv("PVM_ROOT=/usr/convex/pvm");
			pvmlogerror("Defaulting PVM_ROOT to /usr/convex/pvm");
		}
#endif
		pvmbailout(0);
		exit(1);		/* the other meaning of bail out */
	}
	return rd;
}


/*	pvmgetpvmd()
*
*	Return absolute path of pvmd startup script.
*/

char *
pvmgetpvmd()
{
	char *r;
	static char *pp = 0;

	if (!pp) {
		if (PVMDFILE[0] == '/')
			pp = PVMDFILE;
		else {
			r = pvmgetroot();
			pp = TALLOC(strlen(r) + strlen(PVMDFILE) + 2, char, "pvmdpath");
			sprintf(pp, "%s/%s", r, PVMDFILE);
		}
	}
	return pp;
}
 

/*	pvmdsockfile()
*
*	Returns name of file where pvmd advertises its sockaddr.
*	Assumes global pvm_useruid is set before calling.
*/

char *
pvmdsockfile()
{
	static char buf[160];
	char TDSOCKSPEC[128];
	char pvmtxt[128];
	char hna[128];
	char *p;
	char dir[64];
#ifdef	IMA_CSPP
	int scid = get_scid();	/* default (system) subcomplex ID is 1 */
#endif

#ifdef	SHAREDTMP
	if (gethostname(hna, sizeof(hna)-1) == -1) {
		pvmlogerror("pvmdsockfile() can't gethostname()\n");
		return (char*)0;
	}
	if (p = CINDEX(hna, '.'))
		*p = 0;
#ifdef	IMA_CSPP
	if (scid > 1)
		(void)sprintf(buf, TDSOCKNAME_CSPP, pvm_useruid, scid, hna);
	else
#endif
		(void)sprintf(buf, TDSOCKNAME, pvm_useruid, hna);

#else	/* no SHAREDTMP */
#ifdef	IMA_CSPP
	if (scid > 1)
		(void)sprintf(buf, TDSOCKNAME_CSPP, pvm_useruid, scid);
	else
 
#endif
#ifdef WIN32
		if (!getenv("PVM_TMP"))
		{
			(void)sprintf(buf, TDSOCKNAME, username);
			sprintf(pvmtxt,"Could not get PVM_TMP, continuing with %s \n",buf);
			pvmlogerror(pvmtxt);
		} else {
			
			strcpy(TDSOCKSPEC,getenv("PVM_TMP"));
			dir[0]=TDSOCKSPEC[0];
			dir[1]=TDSOCKSPEC[1];
			dir[2]=TDSOCKSPEC[2];
			dir[3]=TDSOCKSPEC[2];
			if (_chdir(getenv("PVM_TMP")) == -1) {
				pvmlogerror("Could Not retrieve PVM_TMP directory \n");
				pvmbailout(0);
			}
			sprintf(buf,"/pvmd.%s",username);
			strcat(TDSOCKSPEC,buf);
			sprintf(buf,TDSOCKSPEC);
		}
#else
		(void)sprintf(buf, TDSOCKNAME, pvm_useruid);
#endif
#endif
	
	return buf;
}


#ifdef IMA_CSPP
#include <sys/cnx_pattr.h>

/*	get_scid ()
*
*	(Convex CSPP only)
*	Returns the subcomplex identifier for the calling process
*/

int
get_scid ()
{
	static int my_scid = -1;

	int rc;
	struct cnx_pattributes pattrib;

	if (my_scid < 0) {
		my_scid = 0;
		if ((rc = cnx_getpattr(getpid(), CNX_PATTR_SCID, &pattrib)) != 0) {
			pvmlogperror ("get_scid: Unable to get scid");
			return 0;
		}
		my_scid = pattrib.pattr_scid;
	}
	return my_scid;
}
#endif	/*IMA_CSPP*/


#define	PVMCRCPOLY	0xedb88320

unsigned int
pvmcrcappend(buf, len, crc)
	char *buf;			/* buffer to append to crc value */
	int len;			/* length */
	unsigned int crc;	/* initial value */
{
	static unsigned int crctbl[256];
	static int once = 1;

	if (once) {
		int i, j;
		unsigned int v;

		for (i = 0; i < 256; i++) {
			v = i;
			for (j = 8; j--; )
				v = v & 1 ? (v >> 1) ^ PVMCRCPOLY : v >> 1;
			crctbl[i] = v;
		}
		once = 0;
	}

	while (len-- > 0)
		crc = ((crc >> 8) & 0x00ffffff) ^ crctbl[(crc ^ *buf++) & 0xff];
	return crc;
}


static char *taglist1[] = {
	"TM_CONNECT",
	"TM_CONN2",
	"TM_EXIT",
	"TM_ADDHOST",
	"TM_DELHOST",
	"TM_CONFIG",
	"TM_MSTAT",
	"TM_HALT",
	"TM_TICKLE",
	"TM_SPAWN",
	"TM_PSTAT",
	"TM_SENDSIG",
	"TM_TASK",
	"TM_MCA",
	"TM_NOTIFY",
	"TM_DB",
	"TM_SCHED",
	"TM_TASKER",
	"TM_HOSTER",
	"TM_HOSTSYNC",
	"TM_SETOPT",
	"TM_GETOPT",
};

static char *taglist2[] = {
	"DM_ADD", "DM_ADDACK",
	"DM_EXEC", "DM_EXECACK",
	"DM_SENDSIG",
	"DM_HTUPD", "DM_HTUPDACK", "DM_HTCOMMIT",
	"DM_SLCONF",
	"DM_STARTACK",
	"DM_TASK", "DM_TASKACK",
	"DM_DELHOST", "DM_DELHOSTACK",
	"DM_NULL",
	"DM_TASKOUT",
	"DM_PSTAT", "DM_PSTATACK",
	"DM_HALT",
	"DM_MCA",
	"DM_NOTIFY", "DM_NOTIFYACK",
	"DM_DB", "DM_DBACK",
	"DM_HTDEL",
	"DM_HOSTSYNC", "DM_HOSTSYNCACK",
};

static char *taglist3[] = {
	"TC_CONREQ",
	"TC_CONACK",
	"TC_TASKEXIT",
	"TC_NOOP",
	"TC_OUTPUT",
	"TC_SETTMASK",
	"TC_SHMAT",
};

static char *taglist4[] = {
	"SM_SPAWN",
	"SM_EXEC", "SM_EXECACK",
	"SM_TASK",
	"SM_CONFIG",
	"SM_ADDHOST",
	"SM_DELHOST",
	"SM_ADD", "SM_ADDACK",
	"SM_NOTIFY",
	"SM_TASKX",
	"SM_HOSTX",
	"SM_HANDOFF",
	"SM_SCHED",
	"SM_STHOST", "SM_STHOSTACK",
	"SM_STTASK",
};

struct nametag {
	int first;
	int last;
	char **names;
};

static struct nametag nametaglist[] = {
	{ TM_CONNECT, TM_GETOPT, taglist1 },
	{ DM_ADD, DM_HOSTSYNCACK, taglist2 },
	{ TC_CONREQ, TC_SHMAT, taglist3 },
	{ SM_SPAWN, SM_STTASK, taglist4 },
};


/*	pvmnametag()
*
*	Convert message tag to symbolic name for reserved tags.
*	Returns name string or decimal tag if not found.
*	If found is nonnull, return whether found.
*/

char *
pvmnametag(tag, found)
	int tag;
	int *found;
{
	static char buf[32];

	int i, j;

	for (i = sizeof(nametaglist)/sizeof(nametaglist[0]); i-- > 0; )
		if (tag >= nametaglist[i].first && tag <= nametaglist[i].last) {
			j = tag - nametaglist[i].first;
			if (found)
				*found = 1;
			return nametaglist[i].names[j];
		}
	sprintf(buf, "%d", tag);
	if (found)
		*found = 0;
	return buf;
}


