!***begin prologue  ftcb
!***date written   850801   (yymmdd)
!***revision date  900509   (yymmdd)
!***revision date  000928   (yymmdd)
!***category no.  j1a1
!***keywords  fast fourier transform, real periodic transform, 
!             fourier synthesis, backward transform, multiple sequences
!***authors  r.a. sweet, l.l. lindgren and r.f boisvert (nist)
!            s.r. clarke (s.clarke@maths.monash.edu.au)
!***purpose  backward real periodic transform, m sequences.
!***description
!
!  subroutine ftcb computes the synthesis (backward transform) of a
!  number of complex periodic sequences from their fourier coefficients. 
!  specifically, for each set of independent fourier coefficients
!  f(k), the corresponding real periodic sequence is computed. 
!
!  the array wsave which is used by subroutine ftrf must be
!  initialized by calling subroutine fti(wsave).
!
!
!  input parameters
!
!  m       the number of sets of coefficients.
!
!  n       the length of the sequences of coefficients to be 
!          transformed.  the method is most efficient when n is a
!          product of small primes, however n may be any positive 
!          integer.
!
!  c       a two-dimensional real array of size m x (2*n)  or a vector of
!          length 2*n (i.e. m=1) containing the complex
!          coefficients f to be transformed.  each row of c is stored as
!
!               c(l,:) = (/ ((real(f(l,k)),aimag(f(l,k))),k=1,n) /)
!
!  wsave   a real one-dimensional work array dimensioned n+15.
!          the wsave array must be initialized by 
!          calling subroutine ftri.  a different wsave array must be
!          used for each different value of n.  this initialization does
!          not have to be repeated so long as n remains unchanged.  the
!          same wsave array may be used by ftcb and ftcb.
!
!  output parameters
!
!  c       contains m complex sequences corresponding to the given
!          coefficients.  specifically, the l-th row of c contains the 
!          complex periodic sequence corresponding to the l-th set of
!          independent fourier coefficients f(l,k) stored as
!
!               c(l,:) = (/ ((real(x(l,j)),aimag(x(l,j))),j=1,n) /), where
!
!               x(l,j) = sqrt(1/n)*
!                           sum(k=1,n) f(l,k)*exp(2*i*(k-1)*(j-1)*pi/n)
!
!  wsave contains results which must not be destroyed between calls
!          to ftcf or ftcb.
!
!  -----------------------------------------------------------------
!
!  note  -  a call of ftcf followed immediately by a call of
!           of ftcb will return the original sequences r.  thus,
!           ftcb is the correctly normalized inverse of ftcf.
!
!
!***references  p. n. swarztrauber, vectorizing the ffts, in parallel
!               computations, (g. rodrigue, ed.), academic press, 1982,
!               pp. 51-83.
!***end prologue  ftcb
!
!     fftpk, version 3, sept 2000
!
module CmplxBack
  use RealBack
  implicit none
  private
  public ftcb
  interface ftcb
    module procedure sftcb,dftcb,svftcb,dvftcb
  end interface
contains
!
!	single precision scalar version
!
  subroutine sftcb(c,wsave)    
    integer, parameter :: kr=kind(0.)
    real(kr), dimension(:), intent(inout) :: c
    real(kr), dimension(:), intent(in) :: wsave
!
    real(kr), dimension(2,size(c)/2) :: r
    integer n,nh,j
!
    n = size(c)/2
    nh = (n-1)/2
    r(1,1) = c(1)
    r(2,1) = c(2)
    r(1,n) = c(2*(nh+2)-1)
    r(2,n) = c(2*(nh+2))
    do j = 1,nh
      r(1,2*j) = .5_kr*(c(2*(j+1)-1)+c(2*(n+1-j)-1))
      r(2,2*j) = .5_kr*(c(2*(j+1))+c(2*(n+1-j)))
      r(1,2*j+1) = .5_kr*(c(2*(j+1))-c(2*(n+1-j)))
      r(2,2*j+1) = -.5_kr*(c(2*(j+1)-1)-c(2*(n+1-j)-1))
    enddo
    call ftrb(r,wsave)
    c(1::2) = r(1,:)
    c(2::2) = r(2,:)
!
  end subroutine
!
!	double precision scalar version
!
  subroutine dftcb(c,wsave)    
    integer, parameter :: kr=kind(0.d0)
    real(kr), dimension(:), intent(inout) :: c
    real(kr), dimension(:), intent(in) :: wsave
!
    real(kr), dimension(2,size(c)/2) :: r
    integer n,nh,j
!
    n = size(c)/2
    nh = (n-1)/2
    r(1,1) = c(1)
    r(2,1) = c(2)
    r(1,n) = c(2*(nh+2)-1)
    r(2,n) = c(2*(nh+2))
    do j = 1,nh
      r(1,2*j) = .5_kr*(c(2*(j+1)-1)+c(2*(n+1-j)-1))
      r(2,2*j) = .5_kr*(c(2*(j+1))+c(2*(n+1-j)))
      r(1,2*j+1) = .5_kr*(c(2*(j+1))-c(2*(n+1-j)))
      r(2,2*j+1) = -.5_kr*(c(2*(j+1)-1)-c(2*(n+1-j)-1))
    enddo
    call ftrb(r,wsave)
    c(1::2) = r(1,:)
    c(2::2) = r(2,:)
!
  end subroutine
!
!	single precision vector version
!
  subroutine svftcb(c,wsave)    
    integer, parameter :: kr=kind(0.)
    real(kr), dimension(:,:), intent(inout) :: c
    real(kr), dimension(:), intent(in) :: wsave
!
    real(kr), dimension(2*size(c,1),size(c,2)/2) :: r
    integer n,nh,j,m
!
    n = size(c,2)/2
    nh = (n-1)/2
    m = size(c,1)
!
    r(1::2,1) = c(:,1)
    r(2::2,1) = c(:,2)
    r(1::2,n) = c(:,2*(nh+2)-1)
    r(2::2,n) = c(:,2*(nh+2))
    do j = 1,nh
      r(1::2,2*j) = .5_kr*(c(:,2*(j+1)-1)+c(:,2*(n+1-j)-1))
      r(2::2,2*j) = .5_kr*(c(:,2*(j+1))+c(:,2*(n+1-j)))
      r(1::2,2*j+1) = .5_kr*(c(:,2*(j+1))-c(:,2*(n+1-j)))
      r(2::2,2*j+1) = -.5_kr*(c(:,2*(j+1)-1)-c(:,2*(n+1-j)-1))
    enddo
    call ftrb(r,wsave)
    do j = 1,m
      c(j,1::2) = r(2*j-1,:)
      c(j,2::2) = r(2*j,:)
    enddo
!
  end subroutine
!
!	double precision scalar version
!
  subroutine dvftcb(c,wsave)    
    integer, parameter :: kr=kind(0.d0)
    real(kr), dimension(:,:), intent(inout) :: c
    real(kr), dimension(:), intent(in) :: wsave
!
    real(kr), dimension(2*size(c,1),size(c,2)/2) :: r
    integer n,nh,j,m
!
    n = size(c,2)/2
    nh = (n-1)/2
    m = size(c,1)
!
    r(1::2,1) = c(:,1)
    r(2::2,1) = c(:,2)
    r(1::2,n) = c(:,2*(nh+2)-1)
    r(2::2,n) = c(:,2*(nh+2))
    do j = 1,nh
      r(1::2,2*j) = .5_kr*(c(:,2*(j+1)-1)+c(:,2*(n+1-j)-1))
      r(2::2,2*j) = .5_kr*(c(:,2*(j+1))+c(:,2*(n+1-j)))
      r(1::2,2*j+1) = .5_kr*(c(:,2*(j+1))-c(:,2*(n+1-j)))
      r(2::2,2*j+1) = -.5_kr*(c(:,2*(j+1)-1)-c(:,2*(n+1-j)-1))
    enddo
    call ftrb(r,wsave)
    do j = 1,m
      c(j,1::2) = r(2*j-1,:)
      c(j,2::2) = r(2*j,:)
    enddo
!
  end subroutine
!
end module CmplxBack
