!***begin prologue  ftcf
!***date written   850801   (yymmdd)
!***revision date  900509   (yymmdd)
!***revision date  000928   (yymmdd)
!***category no.  j1a1
!***keywords  fast fourier transform, real periodic transform, 
!             fourier analysis, forward transform, multiple sequences
!***authors  r.a. sweet, l.l. lindgren and r.f boisvert (nist)
!            s.r. clarke (s.clarke@maths.monash.edu.au)
!***purpose  forward real periodic transform, m sequences.
!***description
!
!  subroutine ftcf computes the fourier coefficients (forward 
!  transform) of a number of complex periodic sequences.  specifically,
!  for each sequence the subroutine claculates the independent
!  fourier coefficients described below at output parameter r.
!
!  the array wsave which is used by subroutine ftcf must be
!  initialized by calling subroutine ftri(wsave).
!
!
!  input parameters
!
!  m       the number of sequences to be transformed.
!
!  n       the length of the sequences to be transformed.  the method
!          is most efficient when n is a product of small primes,
!          however n may be any positive integer.
!
!  c       a real two-dimensional array of size m x 2*n or a vector of
!          length 2*n (i.e. m=1) containing the complex
!          sequences x to be transformed.  the sequences are stored
!          in the rows of c.  thus, 
!
!             c(l,:) = (/ ((real(x(l,j)),aimag(x(l,j))),j=1,n) /)
!
!  wsave   a real one-dimensional work array dimensioned n+15.
!          the wsave array must be initialized by 
!          calling subroutine ftri.  a different wsave array must be
!          used for each different value of n.  this initialization does
!          not have to be repeated so long as n remains unchanged.  the
!          same wsave array may be used by ftcf and ftrb.
!
!  output parameters
!
!  c       contains the fourier coefficients f(k) for each of the m 
!          input sequences.  specifically, row l of c, c(l,j), 
!          j=1,2,..,n, contains the independent fourier coefficients
!          f(l,k), for the i-th input sequence stored as
!
!               c(l,:) = (/ ((real(f(l,k)),aimag(f(l,k))),j=1,n) /), where
!
!               f(l,k) = sqrt(1/n)*
!                           sum(j=1,n) x(l,j)*exp(-2*i*(k-1)*(j-1)*pi/n)
!
!  wsave contains results which must not be destroyed between calls
!          to ftcf or ftcb.
!
!  -----------------------------------------------------------------
!
!  note  -  a call of ftcf followed immediately by a call of
!           of ftcb will return the original sequences c.  thus,
!           ftcb is the correctly normalized inverse of ftcf.
!
!  -----------------------------------------------------------------
!
!
!***references  p. n. swarztrauber, vectorizing the ffts, in parallel
!               computations, (g. rodrigue, ed.), academic press, 1982,
!               pp. 51-83.
!***end prologue  ftcf
!
!     fftpk, version 3, sept 2000
!
module CmplxFwd
  use RealFwd
  implicit none
  private
  public ftcf
  interface ftcf
    module procedure sftcf,dftcf,svftcf,dvftcf
  end interface
contains
!
!	single precision scalar version
!
  subroutine sftcf(c,wsave)    
    integer, parameter :: kr=kind(0.)
    real(kr), dimension(:), intent(inout) :: c
    real(kr), dimension(:), intent(in) :: wsave
!
    real(kr), dimension(2,size(c)/2) :: r
    integer n,nh,j
!
    n = size(c)/2
    nh = (n-1)/2
    r(1,:) = c(1::2)
    r(2,:) = c(2::2)
    call ftrf(r,wsave)
    c(1) = r(1,1)
    c(2) = r(2,1)
    c(2*(nh+2)-1) = r(1,n)
    c(2*(nh+2)) = r(2,n)
    do j = 1,nh
      c(2*(j+1)-1) = r(1,2*j)-r(2,2*j+1)
      c(2*(j+1)) = r(1,2*j+1)+r(2,2*j)
      c(2*(n+1-j)-1) = r(1,2*j)+r(2,2*j+1)
      c(2*(n+1-j)) = -r(1,2*j+1)+r(2,2*j)
    enddo
!
  end subroutine
!
!	double precision scalar version
!
  subroutine dftcf(c,wsave)    
    integer, parameter :: kr=kind(0.d0)
    real(kr), dimension(:), intent(inout) :: c
    real(kr), dimension(:), intent(in) :: wsave
!
    real(kr), dimension(2,size(c)/2) :: r
    integer n,nh,j
!
    n = size(c)/2
    nh = (n-1)/2
    r(1,:) = c(1::2)
    r(2,:) = c(2::2)
    call ftrf(r,wsave)
    c(1) = r(1,1)
    c(2) = r(2,1)
    c(2*(nh+2)-1) = r(1,n)
    c(2*(nh+2)) = r(2,n)
    do j = 1,nh
      c(2*(j+1)-1) = r(1,2*j)-r(2,2*j+1)
      c(2*(j+1)) = r(1,2*j+1)+r(2,2*j)
      c(2*(n+1-j)-1) = r(1,2*j)+r(2,2*j+1)
      c(2*(n+1-j)) = -r(1,2*j+1)+r(2,2*j)
    enddo
!
  end subroutine
!
!	single precision vector version
!
  subroutine svftcf(c,wsave)    
    integer, parameter :: kr=kind(0.)
    real(kr), dimension(:,:), intent(inout) :: c
    real(kr), dimension(:), intent(in) :: wsave
!
    real(kr), dimension(2*size(c,1),size(c,2)/2) :: r
    integer n,nh,j,m
!
    n = size(c,2)/2
    nh = (n-1)/2
    m = size(c,1)
!
    do j = 1,m
      r(2*j-1,:) = c(j,1::2)
      r(2*j,:) = c(j,2::2)
    enddo
    call ftrf(r,wsave)
    c(:,1) = r(1::2,1)
    c(:,2) = r(2::2,1)
    c(:,2*(nh+2)-1) = r(1::2,n)
    c(:,2*(nh+2)) = r(2::2,n)
    do j = 1,nh
      c(:,2*(j+1)-1) = r(1::2,2*j)-r(2::2,2*j+1)
      c(:,2*(j+1)) = r(1::2,2*j+1)+r(2::2,2*j)
      c(:,2*(n+1-j)-1) = r(1::2,2*j)+r(2::2,2*j+1)
      c(:,2*(n+1-j)) = -r(1::2,2*j+1)+r(2::2,2*j)
    enddo
!
  end subroutine
!
!	double precision vector version
!
  subroutine dvftcf(c,wsave)    
    integer, parameter :: kr=kind(0.d0)
    real(kr), dimension(:,:), intent(inout) :: c
    real(kr), dimension(:), intent(in) :: wsave
!
    real(kr), dimension(2*size(c,1),size(c,2)/2) :: r
    integer n,nh,j,m
!
    n = size(c,2)/2
    nh = (n-1)/2
    m = size(c,1)
!
    do j = 1,m
      r(2*j-1,:) = c(j,1::2)
      r(2*j,:) = c(j,2::2)
    enddo
    call ftrf(r,wsave)
    c(:,1) = r(1::2,1)
    c(:,2) = r(2::2,1)
    c(:,2*(nh+2)-1) = r(1::2,n)
    c(:,2*(nh+2)) = r(2::2,n)
    do j = 1,nh
      c(:,2*(j+1)-1) = r(1::2,2*j)-r(2::2,2*j+1)
      c(:,2*(j+1)) = r(1::2,2*j+1)+r(2::2,2*j)
      c(:,2*(n+1-j)-1) = r(1::2,2*j)+r(2::2,2*j+1)
      c(:,2*(n+1-j)) = -r(1::2,2*j+1)+r(2::2,2*j)
    enddo

  end subroutine
end module CmplxFwd
