!***begin prologue  fthc
!***date written   860701   (yymmdd)
!***revision date  900509   (yymmdd)
!***revision date  000928   (yymmdd)
!***category no.  j1a3
!***keywords  fast fourier transform, cosine transform, multiple
!             sequences
!***authors  r.a. sweet, l.l. lindgren and r.f boisvert (nist)
!            s.r. clarke (s.clarke@maths.monash.edu.au)
!***purpose  cosine transform of one or more real, even sequences.
!***description
!
!  subroutine fthc computes the discrete fourier cosine transform
!  of m even sequences x(j,i), j=1,...,m.  the transform is defined
!  below at output parameter x.
!
!  the array wsave which is used by subroutine fthc must be
!  initialized by calling subroutine fthci(wsave).
!
!  input parameters
!
!  m       the number of sequences to be transformed.
!
!  n       the length of the sequence to be transformed.  n must be
!          greater than 1.  the method is most efficient when n-1 is
!          is a product of small primes.
!
!  x       an array of size x(m,n) or a vecror of length x(n) i.e. m=1
!          which contains the
!          the sequences to be transformed.  the sequences are stored
!          in the rows of x.  thus, the jth sequence is stored in
!          x(j,i), i=1,..,n.
!
!  wsave   work array dimensioned 2*n+15
!          in the program that calls fthc.  wsave  must be
!          initialized by calling subroutine fthci, and a
!          different arrays must be used for each different
!          value of n.  this initialization does not have to be
!          repeated so long as n remains unchanged.  thus subsequent
!          transforms can be obtained faster than the first.
!
!  output parameters
!
!  x       for i=1,...,n and j=1,...,m
!
!             x(j,i) = ( x(j,1)+(-1)**(i-1)*x(j,n)
!
!               + the sum from k=2 to k=n-1
!
!                 2*x(j,k)*cos((k-1)*(i-1)*pi/(n-1)) )/sqrt(2*(n-1))
!
!  wsave   contains initialization calculations which must not be
!          destroyed between calls of fthc.
!
!  -----------------------------------------------------------------
!
!  note  -  a call of fthc followed immediately by another call
!           of vfthc will return the original sequences x.  thus,
!           fthc is the correctly normalized inverse of itself.
!
!  -----------------------------------------------------------------
!
!  fthc is a straightforward extension of the subprogram cost to
!  handle m simultaneous sequences.  the scaling of the sequences
!  computed is different from that of the original cost. cost was
!  originally developed by p. n. swarztrauber of ncar.
!
!***references  p. n. swarztrauber, vectorizing the ffts, in parallel
!               computations, (g. rodrigue, ed.), academic press, 1982,
!               pp. 51-83.
!***end prologue  fthc
!
!     fftpk, version 3, sept 2000
!
module HalfCos
  use RealFwd
  implicit none
  private
  public fthc
  interface fthc
    module procedure sfthc,dfthc,svfthc,dvfthc
  end interface
contains
!
!     single precision scalar version
!
  subroutine sfthc(x,wsave)
    integer, parameter :: kr = kind(.0)
    real(kr), dimension(:), intent(inout), target :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(:), pointer :: xp
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x)-1) :: xt
    real(kr) t1,t2
    real(kr), parameter :: one=1._kr,half=.5_kr
    real(kr) :: scale,factor
    integer :: i,k,kc,modn,n,nm1,np1,ns2
!
    n=size(x)
    nm1=n-1
    np1=n+1
    ns2=n/2
!
    select case (n)
    case (2)
      scale = sqrt(half)
      t1 = scale*(x(1)+x(2))
      x(2) = scale*(x(1)-x(2))
      x(1) = t1
    case (3)
      scale = half
      t1 = x(1)+x(3)
      t2 = x(2)+x(2)
      x(2) = scale*(x(1)-x(3))
      x(1) = scale*(t1+t2)
      x(3) = scale*(t1-t2)
    case (4:)
      xt(1) = x(1)-x(n)
      x(1) = x(1)+x(n)
      do k=2,ns2
        kc = np1-k
        t1 = x(k)+x(kc)
        t2 = x(k)-x(kc)
        xt(1) = xt(1)+wsave(kc)*t2
        t2 = wsave(k)*t2
        x(k) = t1-t2
        x(kc) = t1+t2
      enddo
      modn = mod(n,2)
      if (modn/=0) then
        x(ns2+1) = x(ns2+1)+x(ns2+1)
      endif
      x(n) = xt(1)
      xp => x(:nm1)
      wp => wsave(n+2:)
      call ftrf (xp,wp)
      factor = one/sqrt(real(nm1,kr))
      xt(1) = x(2)
      x(2) = factor*x(n)
      do i=4,n,2
        t1 = x(i)
        x(i) = x(i-2)-x(i-1)
        x(i-1) = xt(1)
        xt(1) = t1
      enddo
      if (modn/=0) then
        x(n) = xt(1)
      endif
      scale = sqrt(half)
      x = scale*x
    end select
  end subroutine
!
!     double precision scalar version
!
  subroutine dfthc(x,wsave)
    integer, parameter :: kr = kind(.0d0)
    real(kr), dimension(:), intent(inout), target :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(:), pointer :: xp
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x)-1) :: xt
    real(kr) t1,t2
    real(kr), parameter :: one=1._kr,half=.5_kr
    real(kr) :: scale,factor
    integer :: i,k,kc,modn,n,nm1,np1,ns2
!
    n=size(x)
    nm1=n-1
    np1=n+1
    ns2=n/2
!
    select case (n)
    case (2)
      scale = sqrt(half)
      t1 = scale*(x(1)+x(2))
      x(2) = scale*(x(1)-x(2))
      x(1) = t1
    case (3)
      scale = half
      t1 = x(1)+x(3)
      t2 = x(2)+x(2)
      x(2) = scale*(x(1)-x(3))
      x(1) = scale*(t1+t2)
      x(3) = scale*(t1-t2)
    case (4:)
      xt(1) = x(1)-x(n)
      x(1) = x(1)+x(n)
      do k=2,ns2
        kc = np1-k
        t1 = x(k)+x(kc)
        t2 = x(k)-x(kc)
        xt(1) = xt(1)+wsave(kc)*t2
        t2 = wsave(k)*t2
        x(k) = t1-t2
        x(kc) = t1+t2
      enddo
      modn = mod(n,2)
      if (modn/=0) then
        x(ns2+1) = x(ns2+1)+x(ns2+1)
      endif
      x(n) = xt(1)
      xp => x(:nm1)
      wp => wsave(n+2:)
      call ftrf (xp,wp)
      factor = one/sqrt(real(nm1,kr))
      xt(1) = x(2)
      x(2) = factor*x(n)
      do i=4,n,2
        t1 = x(i)
        x(i) = x(i-2)-x(i-1)
        x(i-1) = xt(1)
        xt(1) = t1
      enddo
      if (modn/=0) then
        x(n) = xt(1)
      endif
      scale = sqrt(half)
      x = scale*x
    end select
  end subroutine
!
!     single precision vector version
!
  subroutine svfthc(x,wsave)
    integer, parameter :: kr = kind(.0)
    real(kr), dimension(:,:), intent(inout), target :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(:,:), pointer :: xp
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x,1),size(x,2)-1) :: xt
    real(kr), dimension(size(x,1)) :: t1,t2
    real(kr), parameter :: c1=1._kr,p5=.5_kr
    real(kr) :: scale,factor
    integer :: i,k,kc,modn,m,n,nm1,np1,ns2

    m=size(x,1)
    n=size(x,2)
    nm1=n-1
    np1=n+1
    ns2=n/2

    if (m<=0) return
    select case (n)
    case (2)
      scale = sqrt(p5)
      t1(:) = scale*(x(:,1)+x(:,2))
      x(:,2) = scale*(x(:,1)-x(:,2))
      x(:,1) = t1(:)
    case (3)
      scale = p5
      t1(:) = x(:,1)+x(:,3)
      t2(:) = x(:,2)+x(:,2)
      x(:,2) = scale*(x(:,1)-x(:,3))
      x(:,1) = scale*(t1(:)+t2(:))
      x(:,3) = scale*(t1(:)-t2(:))
    case (4:)
      xt(:,1) = x(:,1)-x(:,n)
      x(:,1) = x(:,1)+x(:,n)
      do k=2,ns2
        kc = np1-k
        t1(:) = x(:,k)+x(:,kc)
        t2(:) = x(:,k)-x(:,kc)
        xt(:,1) = xt(:,1)+wsave(kc)*t2(:)
        t2(:) = wsave(k)*t2(:)
        x(:,k) = t1(:)-t2(:)
        x(:,kc) = t1(:)+t2(:)
      enddo
      modn = mod(n,2)
      if (modn/=0) then
        x(:,ns2+1) = x(:,ns2+1)+x(:,ns2+1)
      endif
      x(:,n) = xt(:,1)
      xp => x(:,:nm1)
      wp => wsave(n+2:)
      call ftrf (xp,wp)
      factor = c1/sqrt(real(nm1,kr))
      xt(:,1) = x(:,2)
      x(:,2) = factor*x(:,n)
      do i=4,n,2
        t1(:) = x(:,i)
        x(:,i) = x(:,i-2)-x(:,i-1)
        x(:,i-1) = xt(:,1)
        xt(:,1) = t1(:)
      enddo
      if (modn/=0) then
        x(:,n) = xt(:,1)
      endif
      scale = sqrt(p5)
      x = scale*x
    end select
  end subroutine
!
!     doulbe precision vector version
!
  subroutine dvfthc(x,wsave)
    integer, parameter :: kr = kind(.0d0)
    real(kr), dimension(:,:), intent(inout), target :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(:,:), pointer :: xp
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x,1),size(x,2)-1) :: xt
    real(kr), dimension(size(x,1)) :: t1,t2
    real(kr), parameter :: c1=1._kr,p5=.5_kr
    real(kr) :: scale,factor
    integer :: i,k,kc,modn,m,n,nm1,np1,ns2

    m=size(x,1)
    n=size(x,2)
    nm1=n-1
    np1=n+1
    ns2=n/2

    if (m<=0) return
    select case (n)
    case (2)
      scale = sqrt(p5)
      t1(:) = scale*(x(:,1)+x(:,2))
      x(:,2) = scale*(x(:,1)-x(:,2))
      x(:,1) = t1(:)
    case (3)
      scale = p5
      t1(:) = x(:,1)+x(:,3)
      t2(:) = x(:,2)+x(:,2)
      x(:,2) = scale*(x(:,1)-x(:,3))
      x(:,1) = scale*(t1(:)+t2(:))
      x(:,3) = scale*(t1(:)-t2(:))
    case (4:)
      xt(:,1) = x(:,1)-x(:,n)
      x(:,1) = x(:,1)+x(:,n)
      do k=2,ns2
        kc = np1-k
        t1(:) = x(:,k)+x(:,kc)
        t2(:) = x(:,k)-x(:,kc)
        xt(:,1) = xt(:,1)+wsave(kc)*t2(:)
        t2(:) = wsave(k)*t2(:)
        x(:,k) = t1(:)-t2(:)
        x(:,kc) = t1(:)+t2(:)
      enddo
      modn = mod(n,2)
      if (modn/=0) then
        x(:,ns2+1) = x(:,ns2+1)+x(:,ns2+1)
      endif
      x(:,n) = xt(:,1)
      xp => x(:,:nm1)
      wp => wsave(n+2:)
      call ftrf (xp,wp)
      factor = c1/sqrt(real(nm1,kr))
      xt(:,1) = x(:,2)
      x(:,2) = factor*x(:,n)
      do i=4,n,2
        t1(:) = x(:,i)
        x(:,i) = x(:,i-2)-x(:,i-1)
        x(:,i-1) = xt(:,1)
        xt(:,1) = t1(:)
      enddo
      if (modn/=0) then
        x(:,n) = xt(:,1)
      endif
      scale = sqrt(p5)
      x = scale*x
    end select
  end subroutine
end module HalfCos
