!***begin prologue  fths
!***date written   860701   (yymmdd)
!***revision date  900509   (yymmdd)
!***revision date  000928   (yymmdd)
!***category no.  j1a3
!***keywords  fast fourier transform, sine transform, multiple
!             sequences
!***authors  r.a. sweet, l.l. lindgren and r.f boisvert (nist)
!            s.r. clarke (s.clarke@maths.monash.edu.au)
!***purpose  sine transform of one or more real, odd sequences.
!***description
!
!  subroutine fths computes the discrete fourier sine transform
!  of m odd sequences x(j,i), j=1,...,m.  the transform is defined
!  below at output parameter x.
!
!  the array wsave which is used by subroutine vfths must be
!  initialized by calling subroutine fthsi(wsave).
!
!  input parameters
!
!  m       the number of sequences to be transformed.
!
!  n       the length of the sequence to be transformed.  the method
!          is most efficient when n+1 is the product of small primes.
!
!  x       an array of size x(m,n) or a vector of length x(n) i.e. m=1
!          which contains the
!          the sequences to be transformed.  the sequences are stored
!          in the rows of x.  thus, the jth sequence is stored in
!          x(j,i), i=1,..,n.  
!
!  wsave   a work array dimensioned 2*n+15
!          in the program that calls fths.  these arrays must be
!          initialized by calling subroutine fthsi(wsave), and
!          different arrays must be used for each different
!          value of n.  this initialization does not have to be
!          repeated so long as n remains unchanged.
!
!  output parameters
!
!  x       for i=1,...,n and j=1,...,m
!
!               x(j,i)= the sum from k=1 to k=n
!
!                    2*x(j,k)*sin(k*i*pi/(n+1))/sqrt(2*(n+1))
!
!  wsave contains initialization calculations which must not be
!          destroyed between calls of fths.
!
!  -----------------------------------------------------------------
!
!  note  -  a call of fths followed immediately by another call
!           of fths will return the original sequences x.  thus,
!           fths is the correctly normalized inverse of itself.
!
!  -----------------------------------------------------------------
!
!  fths is a straightforward extension of the subprogram sint to
!  handle m simultaneous sequences.  the scaling of the sequences
!  computed is different than that of the original sint. sint was
!  originally developed by p. n. swarztrauber of ncar.
!
!***references  p. n. swarztrauber, vectorizing the ffths, in parallel
!               computations, (g. rodrigue, ed.), academic press, 1982,
!               pp. 51-83.
!***end prologue  fths
!
!     fftpk, version 3, sept 2000
!
module HalfSin
  use RealFwd
  implicit none
  private
  public fths
  interface fths
    module procedure sfths,dfths,svfths,dvfths
  end interface
contains
!
!      single precision scalar version
!
  subroutine sfths(x,wsave)
    integer, parameter :: kr=kind(.0)
    real(kr), dimension(:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x)+1) :: xt
    real(kr) t1,t2
    real(kr), parameter :: zero=.0_kr,half=.5_kr,four=4._kr
    real(kr)  sqrth,scale
    integer n,np1,ns2,k,kc,modn,i

    n=size(x)
    np1=n+1
    ns2=n/2
    select case (n)
    case (2)
      sqrth = sqrt(half)
      t1 = sqrth*(x(1)+x(2))
      x(2) = sqrth*(x(1)-x(2))
      x(1) = t1
    case (3:)
      xt(1) = zero
      do k=1,ns2
         kc = np1-k
         t1 = x(k)-x(kc)
         t2 = wsave(k)*(x(k)+x(kc))
         xt(k+1) = t1+t2
         xt(kc+1) = t2-t1
      enddo
      modn = mod(n,2)
      if (modn/=0) then
        xt(ns2+2) = four*x(ns2+1)
      endif
      wp => wsave(n:)
      call ftrf(xt,wp)
      x(1) = half*xt(1)
      do i=3,n,2
        x(i-1) = -xt(i)
        x(i) = x(i-2)+xt(i-1)
      enddo
      if (modn==0) then
        x(n) = -xt(n+1)
      endif
      scale = sqrt(half)
      x = scale*x
    end select
  end subroutine
!
!     double precision scalar version
!
  subroutine dfths(x,wsave)
    integer, parameter :: kr=kind(.0d0)
    real(kr), dimension(:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x)+1) :: xt
    real(kr) t1,t2
    real(kr), parameter :: zero=.0_kr,half=.5_kr,four=4._kr
    real(kr)  sqrth,scale
    integer n,np1,ns2,k,kc,modn,i

    n=size(x)
    np1=n+1
    ns2=n/2
    select case (n)
    case (2)
      sqrth = sqrt(half)
      t1 = sqrth*(x(1)+x(2))
      x(2) = sqrth*(x(1)-x(2))
      x(1) = t1
    case (3:)
      xt(1) = zero
      do k=1,ns2
         kc = np1-k
         t1 = x(k)-x(kc)
         t2 = wsave(k)*(x(k)+x(kc))
         xt(k+1) = t1+t2
         xt(kc+1) = t2-t1
      enddo
      modn = mod(n,2)
      if (modn/=0) then
        xt(ns2+2) = four*x(ns2+1)
      endif
      wp => wsave(n:)
      call ftrf(xt,wp)
      x(1) = half*xt(1)
      do i=3,n,2
        x(i-1) = -xt(i)
        x(i) = x(i-2)+xt(i-1)
      enddo
      if (modn==0) then
        x(n) = -xt(n+1)
      endif
      scale = sqrt(half)
      x = scale*x
    end select
  end subroutine
!
!      single precision vector version
!
  subroutine svfths(x,wsave)
    integer, parameter :: kr=kind(.0)
    real(kr), dimension(:,:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x,1),size(x,2)+1) :: xt
    real(kr), dimension(size(x,1)) :: t1,t2
    real(kr), parameter :: c0=.0_kr,p5=.5_kr,c4=4._kr
    real(kr)  sqrth,scale
    integer m,n,np1,ns2,k,kc,modn,i

    m=size(x,1) 
    n=size(x,2) 
    np1=n+1 
    ns2=n/2
    if (m<=0)  return
    select case (n)
    case (2)
      sqrth = sqrt(p5)
      t1(:) = sqrth*(x(:,1)+x(:,2))
      x(:,2) = sqrth*(x(:,1)-x(:,2))
      x(:,1) = t1(:)
    case (3:)
      xt(:,1) = c0
      do k=1,ns2
         kc = np1-k
         t1(:) = x(:,k)-x(:,kc)
         t2(:) = wsave(k)*(x(:,k)+x(:,kc))
         xt(:,k+1) = t1(:)+t2(:)
         xt(:,kc+1) = t2(:)-t1(:)
      enddo
      modn = mod(n,2)
      if (modn/=0) then
        xt(:,ns2+2) = c4*x(:,ns2+1)
      endif
      wp => wsave(n:)
      call ftrf(xt,wp)
      x(:,1) = p5*xt(:,1)
      do i=3,n,2
        x(:,i-1) = -xt(:,i)
        x(:,i) = x(:,i-2)+xt(:,i-1)
      enddo
      if (modn==0) then
        x(:,n) = -xt(:,n+1)
      endif
      scale = sqrt(p5)
      x = scale*x
    end select
  end subroutine
!
!     double precision vector version
!
  subroutine dvfths(x,wsave)
    integer, parameter :: kr=kind(.0d0)
    real(kr), dimension(:,:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x,1),size(x,2)+1) :: xt
    real(kr), dimension(size(x,1)) :: t1,t2
    real(kr), parameter :: c0=.0_kr,p5=.5_kr,c4=4._kr
    real(kr)  sqrth,scale
    integer m,n,np1,ns2,k,kc,modn,i

    m=size(x,1) 
    n=size(x,2) 
    np1=n+1 
    ns2=n/2
    if (m<=0)  return
    select case (n)
    case (2)
      sqrth = sqrt(p5)
      t1(:) = sqrth*(x(:,1)+x(:,2))
      x(:,2) = sqrth*(x(:,1)-x(:,2))
      x(:,1) = t1(:)
    case (3:)
      xt(:,1) = c0
      do k=1,ns2
         kc = np1-k
         t1(:) = x(:,k)-x(:,kc)
         t2(:) = wsave(k)*(x(:,k)+x(:,kc))
         xt(:,k+1) = t1(:)+t2(:)
         xt(:,kc+1) = t2(:)-t1(:)
      enddo
      modn = mod(n,2)
      if (modn/=0) then
        xt(:,ns2+2) = c4*x(:,ns2+1)
      endif
      wp => wsave(n:)
      call ftrf(xt,wp)
      x(:,1) = p5*xt(:,1)
      do i=3,n,2
        x(:,i-1) = -xt(:,i)
        x(:,i) = x(:,i-2)+xt(:,i-1)
      enddo
      if (modn==0) then
        x(:,n) = -xt(:,n+1)
      endif
      scale = sqrt(p5)
      x = scale*x
    end select
  end subroutine
end module HalfSin
