!***begin prologue  ftqcb
!***date written   860701   (yymmdd)
!***revision date  900509   (yymmdd)
!***revision date  000928   (yymmdd)
!***category no.  j1a3
!***keywords  fast fourier transform, cosine transform, odd wave
!             numbers, multiple sequences
!***authors  r.a. sweet, l.l. lindgren and r.f boisvert (nist)
!            s.r. clarke (s.clarke@maths.monash.edu.au)
!***purpose  forward cosine transform, odd wave numbers, m sequences.
!***description
!
!  subroutine ftqcb computes the backward fast fourier cosine transform
!  of m quarter wave sequences.  that is, cosine series representations
!  with only odd wave numbers.  the transform is defined below at output
!  parameter x.
!
!  the arrays wsave which are used by subroutine ftqcb must be
!  initialized by calling subroutine ftqci(wsave).
!
!
!  input parameters
!
!  m       the number of sequences to be transformed.
!
!  n       the length of the sequences to be transformed.  the method
!          is most efficient when n is a product of small primes.
!
!  x       an array of size x(m,n) or a scalar of length x(n) i.e. m=1
!          which contains the
!          the sequences to be transformed.  the sequences are stored
!          in the rows of x.  thus, the jth sequence is stored in
!          x(j,i), i=1,..,n.
!
!  wsave   work array which must be dimensioned 2*n+15
!          in the program that calls ftqcb.  the arrays must be
!          initialized by calling subroutine ftqci(wsave), and
!          different arrays must be used for each different
!          value of n.  this initialization does not have to be
!          repeated so long as n remains unchanged.
!
!  output parameters
!
!  x       for i=1,...,n and j=1,...,m
!
!               x(j,i)= the sum from k=1 to k=n of
!
!                 4*x(j,k)*cos((2*k-1)*(i-1)*pi/(2*n)) /sqrt(4*n)
!
!  wsave and fac  contain initialization calculations which must not
!          be destroyed between calls of ftqcf or ftqcb.
!
!  -----------------------------------------------------------------
!
!  note  -  a call of ftqcf followed immediately by a call of
!           of ftqcb will return the original sequences x.  thus,
!           ftqcb is the correctly normalized inverse ftqcf.
!
!  -----------------------------------------------------------------
!
!  ftqcb is a straightforward extension of the subprogram cosqb to
!  handle m simultaneous sequences.  cosqb was originally developed
!  by p. n. swarztrauber of ncar.
!
!***references  p. n. swarztrauber, vectorizing the ffts, in parallel
!               computations, (g. rodrigue, ed.), academic press, 1982,
!               pp. 51-83.
!***end prologue  ftqcb
!
!     fftpk, version 3, sept 2000
!
module QuartCosBack
  use RealBack
  implicit none
  private
  public ftqcb
  interface ftqcb
    module procedure sftqcb,dftqcb,svftqcb,dvftqcb
  end interface
contains
!
!    single precision scalar version
!
  subroutine sftqcb(x,wsave)
    integer, parameter :: kr=kind(.0)
    real(kr), dimension(:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(size(x)) :: xt
    real(kr), dimension(:), pointer :: wp,wr
    real(kr) t1
    real(kr), parameter :: two=2._kr,half=.5_kr
    real(kr) scale
    integer n,ns2,np2,na2,k,kc,i,modn

    n=size(x)
    ns2=(n+1)/2
    np2=n+2
    na2=np2-ns2
!
    select case (n)
    case (2)
      scale = sqrt(two)
      t1 = scale*(x(1)+x(2))
      x(2) = x(1)-x(2)
      x(1) = t1
    case (3:)
      do i=3,n,2
        t1 = x(i-1)+x(i)
        x(i) = x(i)-x(i-1)
        x(i-1) = t1
      enddo
      x(1) = 2*x(1)
      modn = mod(n,2)
      if (modn==0) x(n) = 2*x(n)
      wp => wsave(n+1:)
      call ftrb (x,wp)
      wp => wsave(1:ns2-1)
      wr => wsave(n-1:na2-1:-1)
      xt(2:ns2) = wp*x(n:na2:-1)+wr*x(2:ns2)
      xt(n:na2:-1) = wp*x(2:ns2)-wr*x(n:na2:-1)
      if (modn==0) x(ns2+1) = 2*wsave(ns2)*x(ns2+1)
      x(2:ns2) = xt(2:ns2)+xt(n:na2:-1)
      x(n:na2:-1) = xt(2:ns2)-xt(n:na2:-1)
      x(1) = 2*x(1)
      x = half*x
    end select
  end subroutine
!
!    double precision scalar version
!
  subroutine dftqcb(x,wsave)
    integer, parameter :: kr=kind(.0d0)
    real(kr), dimension(:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(size(x)) :: xt
    real(kr), dimension(:), pointer :: wp,wr
    real(kr) t1
    real(kr), parameter :: two=2._kr,half=.5_kr
    real(kr) scale
    integer n,ns2,np2,na2,k,kc,i,modn

    n=size(x)
    ns2=(n+1)/2
    np2=n+2
    na2=np2-ns2
!
    select case (n)
    case (2)
      scale = sqrt(two)
      t1 = scale*(x(1)+x(2))
      x(2) = x(1)-x(2)
      x(1) = t1
    case (3:)
      do i=3,n,2
        t1 = x(i-1)+x(i)
        x(i) = x(i)-x(i-1)
        x(i-1) = t1
      enddo
      x(1) = 2*x(1)
      modn = mod(n,2)
      if (modn==0) x(n) = 2*x(n)
      wp => wsave(n+1:)
      call ftrb (x,wp)
      wp => wsave(1:ns2-1)
      wr => wsave(n-1:na2-1:-1)
      xt(2:ns2) = wp*x(n:na2:-1)+wr*x(2:ns2)
      xt(n:na2:-1) = wp*x(2:ns2)-wr*x(n:na2:-1)
      if (modn==0) x(ns2+1) = 2*wsave(ns2)*x(ns2+1)
      x(2:ns2) = xt(2:ns2)+xt(n:na2:-1)
      x(n:na2:-1) = xt(2:ns2)-xt(n:na2:-1)
      x(1) = 2*x(1)
      x = half*x
    end select
  end subroutine
!
!    single precision vector version
!
  subroutine svftqcb(x,wsave)
    integer, parameter :: kr=kind(.0)
    real(kr), dimension(:,:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(size(x,1),size(x,2)) :: xt
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x,1)) :: t1
    real(kr), parameter :: c2=2._kr,p5=.5_kr
    real(kr) scale
    integer m,n,ns2,np2,na2,k,kc,i,modn

    m=size(x,1)
    n=size(x,2)
    ns2=(n+1)/2
    np2=n+2
    na2=np2-ns2
!
    if (m<=0)  return
    select case (n)
    case (2)
      scale = c2*sqrt(p5)
      t1(:) = scale*(x(:,1)+x(:,2))
      x(:,2) = x(:,1)-x(:,2)
      x(:,1) = t1(:)
    case (3:)
      do i=3,n,2
        t1(:) = x(:,i-1)+x(:,i)
        x(:,i) = x(:,i)-x(:,i-1)
        x(:,i-1) = t1(:)
      enddo
      x(:,1) = x(:,1)+x(:,1)
      modn = mod(n,2)
      if (modn==0) x(:,n) = x(:,n)+x(:,n)
      wp => wsave(n+1:)
      call ftrb (x,wp)
      do k=2,ns2
         kc = np2-k
         xt(:,k) = wsave(k-1)*x(:,kc)+wsave(kc-1)*x(:,k)
         xt(:,kc) = wsave(k-1)*x(:,k)-wsave(kc-1)*x(:,kc)
      enddo
      if (modn==0) x(:,ns2+1) = wsave(ns2)*(x(:,ns2+1)+x(:,ns2+1))
      x(:,2:ns2) = xt(:,2:ns2)+xt(:,n:na2:-1)
      x(:,n:na2:-1) = xt(:,2:ns2)-xt(:,n:na2:-1)
      x(:,1) = x(:,1)+x(:,1)
      x = p5*x
    end select
  end subroutine
!
!    double precision vector version
!
  subroutine dvftqcb(x,wsave)
    integer, parameter :: kr=kind(.0d0)
    real(kr), dimension(:,:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(size(x,1),size(x,2)) :: xt
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x,1)) :: t1
    real(kr), parameter :: c2=2._kr,p5=.5_kr
    real(kr) scale
    integer m,n,ns2,np2,na2,k,kc,i,modn

    m=size(x,1)
    n=size(x,2)
    ns2=(n+1)/2
    np2=n+2
    na2=np2-ns2
!
    if (m<=0)  return
    select case (n)
    case (2)
      scale = c2*sqrt(p5)
      t1(:) = scale*(x(:,1)+x(:,2))
      x(:,2) = x(:,1)-x(:,2)
      x(:,1) = t1(:)
    case (3:)
      do i=3,n,2
        t1(:) = x(:,i-1)+x(:,i)
        x(:,i) = x(:,i)-x(:,i-1)
        x(:,i-1) = t1(:)
      enddo
      x(:,1) = x(:,1)+x(:,1)
      modn = mod(n,2)
      if (modn==0) x(:,n) = x(:,n)+x(:,n)
      wp => wsave(n+1:)
      call ftrb (x,wp)
      do k=2,ns2
         kc = np2-k
         xt(:,k) = wsave(k-1)*x(:,kc)+wsave(kc-1)*x(:,k)
         xt(:,kc) = wsave(k-1)*x(:,k)-wsave(kc-1)*x(:,kc)
      enddo
      if (modn==0) x(:,ns2+1) = wsave(ns2)*(x(:,ns2+1)+x(:,ns2+1))
      x(:,2:ns2) = xt(:,2:ns2)+xt(:,n:na2:-1)
      x(:,n:na2:-1) = xt(:,2:ns2)-xt(:,n:na2:-1)
      x(:,1) = x(:,1)+x(:,1)
      x = p5*x
    end select
  end subroutine
end module QuartCosBack
