!***begin prologue  ftqcf
!***date written   860701   (yymmdd)
!***revision date  900509   (yymmdd)
!***revision date  000928   (yymmdd)
!***category no.  j1a3
!***keywords  fast fourier transform, cosine transform, odd wave
!             numbers, multiple sequences
!***authors  r.a. sweet, l.l. lindgren and r.f boisvert (nist)
!            s.r. clarke (s.clarke@maths.monash.edu.au)
!***purpose  forward cosine transform, odd wave numbers, m sequences.
!***description
!
!  subroutine ftqcf computes the forward fast fourier cosine transform
!  of m quarter wave sequences.  that is, cosine series representations
!  with only odd wave numbers.  the transform is defined below at output
!  parameter x.
!
!  the arrays wsave which are used by subroutine ftqcf must be
!  initialized by calling subroutine ftqci(wsave).
!
!
!  input parameters
!
!  m       the number of sequences to be transformed.
!
!  n       the length of the sequences to be transformed.  the method
!          is most efficient when n is a product of small primes.
!
!  x       an array of size x(m,n) or vector of length x(n) i.e. m=1
!          which contains the
!          the sequences to be transformed.  the sequences are stored
!          in the rows of x.  thus, the jth sequence is stored in
!          x(j,i), i=1,..,n.
!
!  wsave   work arrays which must be dimensioned 2*n+15 
!          in the program that calls ftqcf.  the arrays must be
!          initialized by calling subroutine ftqci(wsave), and
!          different arrays must be used for each different
!          value of n.  this initialization does not have to be
!          repeated so long as n remains unchanged.
!
!  output parameters
!
!  x       for i=1,...,n and j=1,...,m
!
!               x(j,i) = ( x(j,1) + the sum from k=2 to k=n of
!
!                  2*x(j,k)*cos((2*i-1)*(k-1)*pi/(2*n)) )/sqrt(4*n)
!
!  wsave  contains initialization calculations which must not
!          be destroyed between calls of ftqcf or ftqcb.
!
!  -----------------------------------------------------------------
!
!  note  -  a call of ftqcf followed immediately by a call of
!           of ftqcb will return the original sequences x.  thus,
!           ftqcb is the correctly normalized inverse ftqcf.
!
!  -----------------------------------------------------------------
!
!  ftqcf is a straightforward extension of the subprogram cosqf to
!  handle m simultaneous sequences.  cosqf was originally developed
!  by p. n. swarztrauber of ncar.
!
!***references  p. n. swarztrauber, vectorizing the ffts, in parallel
!               computations, (g. rodrigue, ed.), academic press, 1982,
!               pp. 51-83.
!***end prologue  ftqcf
module QuartCosFwd
  use RealFwd
  implicit none
  private
  public ftqcf
  interface ftqcf
    module procedure sftqcf,dftqcf,svftqcf,dvftqcf
  end interface
contains
!
!    single precision version
!
  subroutine sftqcf(x,wsave)
    integer, parameter :: kr=kind(.0)
    real(kr), dimension(:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(size(x)) :: xt
    real(kr), dimension(:), pointer :: wp,wr
    real(kr) t1
    real(kr), parameter :: half=.5_kr,two=2._kr
    real(kr) sqrt2,scale
    integer n,ns2,np2,na2,k,kc,i,modn

    n=size(x)
    ns2=(n+1)/2
    np2=n+2
    na2=np2-ns2

    select case (n)
    case (2)
      sqrt2 = sqrt(two)
      scale = half/sqrt2
      t1 = sqrt2*x(2)
      x(2) = scale*(x(1)-t1)
      x(1) = scale*(x(1)+t1)
    case (3:)
      xt(2:ns2) = x(2:ns2)+x(n:na2:-1)
      xt(n:na2:-1) = x(2:ns2)-x(n:na2:-1)
      modn = mod(n,2)
      if (modn==0) xt(ns2+1) = x(ns2+1)+x(ns2+1)
      wp => wsave(1:ns2-1)
      wr => wsave(n-1:na2-1:-1)
      x(2:ns2) = wp*xt(n:na2:-1)+wr*xt(2:ns2)
      x(n:na2:-1) = wp*xt(2:ns2)-wr*xt(n:na2:-1)
      if (modn==0) x(ns2+1) = wsave(ns2)*xt(ns2+1)
      wp => wsave(n+1:)
      call ftrf (x,wp)
      do i=3,n,2
        t1 = x(i-1)-x(i)
        x(i) = x(i-1)+x(i)
        x(i-1) = t1
      enddo
      x = half*x
    end select
  end subroutine
!
!    double precision version
!
  subroutine dftqcf(x,wsave)
    integer, parameter :: kr=kind(.0d0)
    real(kr), dimension(:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(size(x)) :: xt
    real(kr), dimension(:), pointer :: wp,wr
    real(kr) t1
    real(kr), parameter :: half=.5_kr,two=2._kr
    real(kr) sqrt2,scale
    integer n,ns2,np2,na2,k,kc,i,modn

    n=size(x)
    ns2=(n+1)/2
    np2=n+2
    na2=np2-ns2

    select case (n)
    case (2)
      sqrt2 = sqrt(two)
      scale = half/sqrt2
      t1 = sqrt2*x(2)
      x(2) = scale*(x(1)-t1)
      x(1) = scale*(x(1)+t1)
    case (3:)
      xt(2:ns2) = x(2:ns2)+x(n:na2:-1)
      xt(n:na2:-1) = x(2:ns2)-x(n:na2:-1)
      modn = mod(n,2)
      if (modn==0) xt(ns2+1) = x(ns2+1)+x(ns2+1)
      wp => wsave(1:ns2-1)
      wr => wsave(n-1:na2-1:-1)
      x(2:ns2) = wp*xt(n:na2:-1)+wr*xt(2:ns2)
      x(n:na2:-1) = wp*xt(2:ns2)-wr*xt(n:na2:-1)
      if (modn==0) x(ns2+1) = wsave(ns2)*xt(ns2+1)
      wp => wsave(n+1:)
      call ftrf (x,wp)
      do i=3,n,2
        t1 = x(i-1)-x(i)
        x(i) = x(i-1)+x(i)
        x(i-1) = t1
      enddo
      x = half*x
    end select
  end subroutine
!
!    single precision vector version
!
  subroutine svftqcf(x,wsave)
    integer, parameter :: kr=kind(.0)
    real(kr), dimension(:,:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(size(x,1),size(x,2)) :: xt
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x,1)) :: t1
    real(kr), parameter :: p5=.5_kr,c2=2._kr
    real(kr) sqrt2,scale
    integer m,n,ns2,np2,na2,k,kc,i,modn

    m=size(x,1)
    n=size(x,2)
    ns2=(n+1)/2
    np2=n+2
    na2=np2-ns2

    if (m<=0)  return
    select case (n)
    case (2)
      sqrt2 = sqrt(c2)
      scale = p5/sqrt2
      t1 = sqrt2*x(:,2)
      x(:,2) = scale*(x(:,1)-t1)
      x(:,1) = scale*(x(:,1)+t1)
    case (3:)
      xt(:,2:ns2) = x(:,2:ns2)+x(:,n:na2:-1)
      xt(:,n:na2:-1) = x(:,2:ns2)-x(:,n:na2:-1)
      modn = mod(n,2)
      if (modn==0) xt(:,ns2+1) = x(:,ns2+1)+x(:,ns2+1)
      do k = 2,ns2
        kc = np2-k
        x(:,k) = wsave(k-1)*xt(:,kc)+wsave(kc-1)*xt(:,k)
        x(:,kc) = wsave(k-1)*xt(:,k)-wsave(kc-1)*xt(:,kc)
      enddo
      if (modn==0) x(:,ns2+1) = wsave(ns2)*xt(:,ns2+1)
      wp => wsave(n+1:)
      call ftrf (x,wp)
      do i=3,n,2
        t1 = x(:,i-1)-x(:,i)
        x(:,i) = x(:,i-1)+x(:,i)
        x(:,i-1) = t1
      enddo
      x = p5*x
    end select
  end subroutine
!
!    double precision vector version
!
  subroutine dvftqcf(x,wsave)
    integer, parameter :: kr=kind(.0d0)
    real(kr), dimension(:,:), intent(inout) :: x
    real(kr), dimension(:), intent(in), target :: wsave
!
    real(kr), dimension(size(x,1),size(x,2)) :: xt
    real(kr), dimension(:), pointer :: wp
    real(kr), dimension(size(x,1)) :: t1
    real(kr), parameter :: p5=.5_kr,c2=2._kr
    real(kr) sqrt2,scale
    integer m,n,ns2,np2,na2,k,kc,i,modn

    m=size(x,1)
    n=size(x,2)
    ns2=(n+1)/2
    np2=n+2
    na2=np2-ns2

    if (m<=0)  return
    select case (n)
    case (2)
      sqrt2 = sqrt(c2)
      scale = p5/sqrt2
      t1(:) = sqrt2*x(:,2)
      x(:,2) = scale*(x(:,1)-t1(:))
      x(:,1) = scale*(x(:,1)+t1(:))
    case (3:)
      xt(:,2:ns2) = x(:,2:ns2)+x(:,n:na2:-1)
      xt(:,n:na2:-1) = x(:,2:ns2)-x(:,n:na2:-1)
      modn = mod(n,2)
      if (modn==0) xt(:,ns2+1) = x(:,ns2+1)+x(:,ns2+1)
      do k = 2,ns2
        kc = np2-k
        x(:,k) = wsave(k-1)*xt(:,kc)+wsave(kc-1)*xt(:,k)
        x(:,kc) = wsave(k-1)*xt(:,k)-wsave(kc-1)*xt(:,kc)
      enddo
      if (modn==0) x(:,ns2+1) = wsave(ns2)*xt(:,ns2+1)
      wp => wsave(n+1:)
      call ftrf (x,wp)
      do i=3,n,2
        t1(:) = x(:,i-1)-x(:,i)
        x(:,i) = x(:,i-1)+x(:,i)
        x(:,i-1) = t1(:)
      enddo
      x = p5*x
    end select
  end subroutine
end module QuartCosFwd
