!***begin prologue  ftqsf
!***date written   860701   (yymmdd)
!***revision date  900509   (yymmdd)
!***revision date  000928   (yymmdd)
!***category no.  j1a3
!***keywords  fast fourier transform, sine transform, odd wave
!             numbers, multiple sequences
!***authors  r.a. sweet, l.l. lindgren and r.f boisvert (nist)
!            s.r. clarke (s.clarke@maths.monash.edu.au)
!***purpose  normalized inverse of ftqsf.
!***description
!
!  subroutine ftqsf computes the backward fast fourier sine transform
!  of m quarter wave sequences.  that is, sine series representations
!  with only odd wave numbers.  the transform is defined below at output
!  parameter x.
!
!
!  input parameters
!
!  m       the number of sequences to be transformed.
!
!  n       the length of the sequences to be transformed.  the method
!          is most efficient when n is a product of small primes.
!
!  x       an array of size x(m,n) or a vector of length x(n) (i.e. m=1)
!          which contains the
!          the sequences to be transformed.  the sequences are stored
!          in the rows of x.  thus, the jth sequence is stored in
!          x(j,i), i=1,..,n.
!
!  wsave   a real work array which must be dimensioned 2*n+15
!          in the program that calls ftqsb.  the wsave array must be
!          initialized by calling subroutine ftqsi(wsave), and a
!          different wsave array must be used for each different
!          value of n.  this initialization does not have to be
!          repeated so long as n remains unchanged.
!
!  output parameters
!
!  x       for i=1,...,n and j=1,...,m
!
!               x(j,i) = ( (-1)**(i-1)*x(j,n)
!
!                        + the sum from k=1 to k=n-1 of
!
!                        2*x(j,k)*sin((2*i-1)*k*pi/(2*n)) )/sqrt(4*n)
!
!  wsave   contains initialization calculations which must not
!          be destroyed between calls of ftqsf or ftqsb.
!
!  -----------------------------------------------------------------
!
!  note  -  a call of ftqsf followed immediately by a call of
!           of ftqsb will return the original sequences x.  thus,
!           ftqsb is the correctly normalized inverse ftqsf.
!
!  -----------------------------------------------------------------
!
!  ftqsf is a straightforward extension of the subprogram sinqf to
!  handle m simultaneous sequences.  sinqf was originally developed
!  by p. n. swarztrauber of ncar.
!
!***end prologue  ftqsf
!
!     fftpk, version 3, sept 2000
!
module QuartSinFwd
  use QuartCosFwd
  implicit none 
  private
  public ftqsf
  interface ftqsf
    module procedure sftqsf,dftqsf,svftqsf,dvftqsf
  end interface
contains
!
!      single precision scalar version
!
  subroutine sftqsf(x,wsave)
    integer, parameter :: kr=kind(.0)
    real(kr), dimension(:), intent(inout) :: x
    real(kr), dimension(:), intent(in) :: wsave
!
    integer n

    n=size(x)
    if (n < 2) return
    x = x(n:1:-1)
    call ftqcf(x,wsave)
    x(2::2) = -x(2::2)
  end subroutine
!
!     double precision vector version
!
  subroutine dftqsf(x,wsave)
    integer, parameter :: kr=kind(.0d0)
    real(kr), dimension(:), intent(inout) :: x
    real(kr), dimension(:), intent(in) :: wsave
!
    integer n

    n=size(x)
    if (n < 2) return
    x = x(n:1:-1)
    call ftqcf(x,wsave)
    x(2::2) = -x(2::2)
  end subroutine
!
!      single precision vector version
!
  subroutine svftqsf(x,wsave)
    integer, parameter :: kr=kind(.0)
    real(kr), dimension(:,:), intent(inout) :: x
    real(kr), dimension(:), intent(in) :: wsave
!
    real(kr), dimension(size(x,1)) :: t1
    integer n,ns2,k,kc

    n=size(x,2) 
    ns2=n/2
    if (size(x,1)<=0) return
    do k=1,ns2
       kc = n-k
       t1(:) = x(:,k)
       x(:,k) = x(:,kc+1)
       x(:,kc+1) = t1(:)
    enddo
    call ftqcf(x,wsave)
    x(:,2:n:2) = -x(:,2:n:2)
  end subroutine
!
!     double precision vector version
!
  subroutine dvftqsf(x,wsave)
    integer, parameter :: kr=kind(.0d0)
    real(kr), dimension(:,:), intent(inout) :: x
    real(kr), dimension(:), intent(in) :: wsave
!
    real(kr), dimension(size(x,1)) :: t1
    integer n,ns2,k,kc

    n=size(x,2) 
    ns2=n/2
    if (size(x,1)<=0) return
    do k=1,ns2
       kc = n-k
       t1(:) = x(:,k)
       x(:,k) = x(:,kc+1)
       x(:,kc+1) = t1(:)
    enddo
    call ftqcf(x,wsave)
    x(:,2:n:2) = -x(:,2:n:2)
  end subroutine
end module QuartSinFwd
