!***begin prologue  ftrb
!***date written   850801   (yymmdd)
!***revision date  900509   (yymmdd)
!***revision date  000928   (yymmdd)
!***category no.  j1a1
!***keywords  fast fourier transform, real periodic transform, 
!             fourier synthesis, backward transform, multiple sequences
!***authors  r.a. sweet, l.l. lindgren and r.f boisvert (nist)
!            s.r. clarke (s.clarke@maths.monash.edu.au)
!***purpose  backward real periodic transform, m sequences.
!***description
!
!  subroutine ftrb computes the synthesis (backward transform) of a
!  number of real periodic sequences from their fourier coefficients. 
!  specifically, for each set of independent fourier coefficients
!  f(k), the corresponding real periodic sequence is computed. 
!
!  the array wsave which is used by subroutine ftrf must be
!  initialized by calling subroutine fti(wsave).
!
!
!  input parameters
!
!  m       the number of sets of coefficients.
!
!  n       the length of the sequences of coefficients to be 
!          transformed.  the method is most efficient when n is a
!          product of small primes, however n may be any positive 
!          integer.
!
!  c       a real two-dimensional array of size m x n  or a vector of
!          length n (i.e. m=1) containing the
!          coefficients to be transformed.  each set of coefficients
!          f(k), k\0,1,..,n-1, is stored as a row of c.  specifically,
!          the i-th set of independent fourier coefficients is stored
!
!                c(i,1) = real( f(i,0) ),
!
!                c(i,2*k) = real( f(i,k) )
!
!                c(i,2*k+1) = imag( f(i,k) )
!
!                   for k = 1, 2, . . . , m-1,
!
!                and, when n is even,
!
!                c(i,n) = real( f(i,n/2) ).
!
!  wsave   a real one-dimensional work array dimensioned n+15.
!          the wsave array must be initialized by 
!          calling subroutine ftri.  a different wsave array must be
!          used for each different value of n.  this initialization does
!          not have to be repeated so long as n remains unchanged.  the
!          same wsave array may be used by ftrb and ftrb.
!
!  output parameters
!
!  c       contains m real periodic sequences corresponding to the given
!          coefficients.  specifically, the i-th row of r contains the 
!          real periodic sequence corresponding to the i-th set of
!          independent fourier coefficients f(i,k) stored as
!
!               c(i,j) = x(i,j-1) ,   j = 1, 2, . . . , n, where
!
!               x(i,j) = sqrt(1/n)* f(i,0) + (-1)**j*f(i,n/2)
!                        + 2*sum(k=1,m)[ real(f(i,2k))*cos(2k*j*pi/n)
!                        - imag(f(i,2k+1))*sin(2k*j*pi/n) ]  ,
!
!                 when n is even, and
!
!               x(i,j) = sqrt(1/n)* f(i,0) +
!                        2*sum(k=1,m)[ real(f(i,2k))*cos(2k*j*pi/n)
!                        - imag(f(i,2k+1))*sin(2k*j*pi/n) ]  ,
!
!                 when n is odd.
!
!  wsave and fac contain results which must not be destroyed between calls
!          to ftrf or ftrb.
!
!  -----------------------------------------------------------------
!
!  note  -  a call of ftrf followed immediately by a call of
!           of ftrb will return the original sequences r.  thus,
!           ftrb is the correctly normalized inverse of ftrf.
!
!  -----------------------------------------------------------------
!
!  ftrb is a straightforward extension of the subprogram rfftb to
!  handle m simultaneous sequences.  rfftb was originally developed
!  by p. n. swarztrauber of ncar.
!
!***references  p. n. swarztrauber, vectorizing the ffts, in parallel
!               computations, (g. rodrigue, ed.), academic press, 1982,
!               pp. 51-83.
!***end prologue  vftrb
!
!     fftpk, version 3, sept 2000
!
module RealBack
  use Radix2b
  use Radix3b
  use Radix4b
  use Radix5b
  use Radixgb
  implicit none  
  private
  public ftrb
  interface ftrb
    module procedure sftrb,dftrb,svftrb,dvftrb
  end interface
contains
!
!    single precision scalar version
!
  subroutine sftrb(c,wsave)
    integer, parameter :: kr=kind(0.)
    real(kr), dimension(:), intent(inout) :: c
    real(kr), dimension(:), intent(in), target :: wsave
    integer, dimension(15) :: fac
!
    real(kr), parameter :: one=1._kr
    real(kr), dimension(size(c)) :: ch
    real(kr), dimension(:), pointer :: w1,w2,w3,w4
    real(kr) scale
    integer n,nf,na,l1,iw,k1,ip,l2,ido,idl1,ix2,ix3,ix4
!
    n=size(c)
    fac = wsave(n+1:)
!
    if (n==1) return
    nf = fac(2)
    na = 0
    l1 = 1
    iw = 1
    do k1=1,nf
      ip = fac(k1+2)
      l2 = ip*l1
      ido = n/l2
      idl1 = ido*l1
      w1 => wsave(iw:iw+ido-1)
      select case (ip)
      case (4) 
        ix2 = iw+ido
        ix3 = ix2+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        select case(na)
        case (0)
          call radb4 (ido,l1,c,ch,w1,w2,w3)
        case (1)
          call radb4 (ido,l1,ch,c,w1,w2,w3)
        end select
        na = 1-na
      case (2)
        select case (na)
        case (0)
          call radb2 (ido,l1,c,ch,w1)
        case (1)
          call radb2 (ido,l1,ch,c,w1)
        end select
        na = 1-na
      case (3)
        ix2 = iw+ido
        w2 => wsave(ix2:ix2+ido-1)
        select case (na)
        case (0)
          call radb3 (ido,l1,c,ch,w1,w2)
        case (1)
          call radb3 (ido,l1,ch,c,w1,w2)
        end select
        na = 1-na
      case (5)
        ix2 = iw+ido
        ix3 = ix2+ido
        ix4 = ix3+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        w4 => wsave(ix4:ix4+ido-1)
        select case (na)
        case (0)
          call radb5 (ido,l1,c,ch,w1,w2,w3,w4)
        case (1)
          call radb5 (ido,l1,ch,c,w1,w2,w3,w4)
        end select
        na = 1-na
      case default
        select case (na)
        case (0)
          call radbg (ido,ip,l1,c,ch,w1)
        case (1)
          call radbg (ido,ip,l1,ch,c,w1)
        end select
        if (ido==1) na = 1-na
      end select
      l1 = l2
      iw = iw+(ip-1)*ido
    enddo
    scale=one/sqrt(real(n,kr))
    select case (na)
    case (0)
      c= scale*c
    case (1)
      c = scale*ch
    end select
  end subroutine
!
!    double precision scalar version
!
  subroutine dftrb(c,wsave)
    integer, parameter :: kr=kind(0.d0)
    real(kr), dimension(:), intent(inout) :: c
    real(kr), dimension(:), intent(in), target :: wsave
    integer, dimension(15) :: fac
!
    real(kr), parameter :: one=1._kr
    real(kr), dimension(size(c)) :: ch
    real(kr), dimension(:), pointer :: w1,w2,w3,w4
    real(kr) scale
    integer n,nf,na,l1,iw,k1,ip,l2,ido,idl1,ix2,ix3,ix4
!
    n=size(c)
    fac = wsave(n+1:)
!
    if (n==1) return
    nf = fac(2)
    na = 0
    l1 = 1
    iw = 1
    do k1=1,nf
      ip = fac(k1+2)
      l2 = ip*l1
      ido = n/l2
      idl1 = ido*l1
      w1 => wsave(iw:iw+ido-1)
      select case (ip)
      case (4) 
        ix2 = iw+ido
        ix3 = ix2+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        select case(na)
        case (0)
          call radb4 (ido,l1,c,ch,w1,w2,w3)
        case (1)
          call radb4 (ido,l1,ch,c,w1,w2,w3)
        end select
        na = 1-na
      case (2)
        select case (na)
        case (0)
          call radb2 (ido,l1,c,ch,w1)
        case (1)
          call radb2 (ido,l1,ch,c,w1)
        end select
        na = 1-na
      case (3)
        ix2 = iw+ido
        w2 => wsave(ix2:ix2+ido-1)
        select case (na)
        case (0)
          call radb3 (ido,l1,c,ch,w1,w2)
        case (1)
          call radb3 (ido,l1,ch,c,w1,w2)
        end select
        na = 1-na
      case (5)
        ix2 = iw+ido
        ix3 = ix2+ido
        ix4 = ix3+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        w4 => wsave(ix4:ix4+ido-1)
        select case (na)
        case (0)
          call radb5 (ido,l1,c,ch,w1,w2,w3,w4)
        case (1)
          call radb5 (ido,l1,ch,c,w1,w2,w3,w4)
        end select
        na = 1-na
      case default
        select case (na)
        case (0)
          call radbg (ido,ip,l1,c,ch,w1)
        case (1)
          call radbg (ido,ip,l1,ch,c,w1)
        end select
        if (ido==1) na = 1-na
      end select
      l1 = l2
      iw = iw+(ip-1)*ido
    enddo
    scale=one/sqrt(real(n,kr))
    select case (na)
    case (0)
      c= scale*c
    case (1)
      c = scale*ch
    end select
  end subroutine
!
!    single precision vector version
!
  subroutine svftrb(c,wsave)
    integer, parameter :: kr=kind(0.)
    real(kr), dimension(:,:), intent(inout) :: c
    real(kr), dimension(:), intent(in), target :: wsave
    integer, dimension(15) :: fac
!
    real(kr), parameter :: c1=1._kr
    real(kr), dimension(size(c,1),size(c,2)) :: ch
    real(kr), dimension(:), pointer :: w1,w2,w3,w4
    real(kr) scale
    integer m,n,nf,na,l1,iw,k1,ip,l2,ido,idl1,ix2,ix3,ix4
!
    m=size(c,1) 
    n=size(c,2)
    fac = wsave(n+1:)
!
    if (n==1) return
    nf = fac(2)
    na = 0
    l1 = 1
    iw = 1
    do k1=1,nf
      ip = fac(k1+2)
      l2 = ip*l1
      ido = n/l2
      idl1 = ido*l1
      w1 => wsave(iw:iw+ido-1)
      select case (ip)
      case (4) 
        ix2 = iw+ido
        ix3 = ix2+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        select case(na)
        case (0)
          call radb4 (ido,l1,c,ch,w1,w2,w3)
        case (1)
          call radb4 (ido,l1,ch,c,w1,w2,w3)
        end select
        na = 1-na
      case (2)
        select case (na)
        case (0)
          call radb2 (ido,l1,c,ch,w1)
        case (1)
          call radb2 (ido,l1,ch,c,w1)
        end select
        na = 1-na
      case (3)
        ix2 = iw+ido
        w2 => wsave(ix2:ix2+ido-1)
        select case (na)
        case (0)
          call radb3 (ido,l1,c,ch,w1,w2)
        case (1)
          call radb3 (ido,l1,ch,c,w1,w2)
        end select
        na = 1-na
      case (5)
        ix2 = iw+ido
        ix3 = ix2+ido
        ix4 = ix3+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        w4 => wsave(ix4:ix4+ido-1)
        select case (na)
        case (0)
          call radb5 (ido,l1,c,ch,w1,w2,w3,w4)
        case (1)
          call radb5 (ido,l1,ch,c,w1,w2,w3,w4)
        end select
        na = 1-na
      case default
        select case (na)
        case (0)
          call radbg (ido,ip,l1,c,ch,w1)
        case (1)
          call radbg (ido,ip,l1,ch,c,w1)
        end select
        if (ido==1) na = 1-na
      end select
      l1 = l2
      iw = iw+(ip-1)*ido
    enddo
    scale=c1/sqrt(real(n,kr))
    select case (na)
    case (0)
      c= scale*c
    case (1)
      c = scale*ch
    end select
  end subroutine
!
!    double precision vector version
!
  subroutine dvftrb(c,wsave)
    integer, parameter :: kr=kind(0.d0)
    real(kr), dimension(:,:), intent(inout) :: c
    real(kr), dimension(:), intent(in), target :: wsave
    integer, dimension(15) :: fac
!
    real(kr), parameter :: c1=1._kr
    real(kr), dimension(size(c,1),size(c,2)) :: ch
    real(kr), dimension(:), pointer :: w1,w2,w3,w4
    real(kr) scale
    integer m,n,nf,na,l1,iw,k1,ip,l2,ido,idl1,ix2,ix3,ix4
!
    m=size(c,1) 
    n=size(c,2)
    fac = wsave(n+1:)
!
    if (n==1) return
    nf = fac(2)
    na = 0
    l1 = 1
    iw = 1
    do k1=1,nf
      ip = fac(k1+2)
      l2 = ip*l1
      ido = n/l2
      idl1 = ido*l1
      w1 => wsave(iw:iw+ido-1)
      select case (ip)
      case (4) 
        ix2 = iw+ido
        ix3 = ix2+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        select case(na)
        case (0)
          call radb4 (ido,l1,c,ch,w1,w2,w3)
        case (1)
          call radb4 (ido,l1,ch,c,w1,w2,w3)
        end select
        na = 1-na
      case (2)
        select case (na)
        case (0)
          call radb2 (ido,l1,c,ch,w1)
        case (1)
          call radb2 (ido,l1,ch,c,w1)
        end select
        na = 1-na
      case (3)
        ix2 = iw+ido
        w2 => wsave(ix2:ix2+ido-1)
        select case (na)
        case (0)
          call radb3 (ido,l1,c,ch,w1,w2)
        case (1)
          call radb3 (ido,l1,ch,c,w1,w2)
        end select
        na = 1-na
      case (5)
        ix2 = iw+ido
        ix3 = ix2+ido
        ix4 = ix3+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        w4 => wsave(ix4:ix4+ido-1)
        select case (na)
        case (0)
          call radb5 (ido,l1,c,ch,w1,w2,w3,w4)
        case (1)
          call radb5 (ido,l1,ch,c,w1,w2,w3,w4)
        end select
        na = 1-na
      case default
        select case (na)
        case (0)
          call radbg (ido,ip,l1,c,ch,w1)
        case (1)
          call radbg (ido,ip,l1,ch,c,w1)
        end select
        if (ido==1) na = 1-na
      end select
      l1 = l2
      iw = iw+(ip-1)*ido
    enddo
    scale=c1/sqrt(real(n,kr))
    select case (na)
    case (0)
      c= scale*c
    case (1)
      c = scale*ch
    end select
  end subroutine
end module RealBack
