!***begin prologue  ftrf
!***date written   850801   (yymmdd)
!***revision date  900509   (yymmdd)
!***revision date  000928   (yymmdd)
!***category no.  j1a1
!***keywords  fast fourier transform, real periodic transform, 
!             fourier analysis, forward transform, multiple sequences
!***authors  r.a. sweet, l.l. lindgren and r.f boisvert (nist)
!            s.r. clarke (s.clarke@maths.monash.edu.au)
!***purpose  forward real periodic transform, m sequences.
!***description
!
!  subroutine ftrf computes the fourier coefficients (forward 
!  transform) of a number of real periodic sequences.  specifically,
!  for each sequence the subroutine claculates the independent
!  fourier coefficients described below at output parameter r.
!
!  the array wsave which is used by subroutine ftrf must be
!  initialized by calling subroutine fti(wsave).
!
!
!  input parameters
!
!  m       the number of sequences to be transformed.
!
!  n       the length of the sequences to be transformed.  the method
!          is most efficient when n is a product of small primes,
!          however n may be any positive integer.
!
!  c       a real two-dimensional array of size m x n or a vector of
!          length n (i.e. m=1) containing the
!          the sequences to be transformed.  the sequences are stored
!          in the rows of c.  thus, the i-th sequence to be transformed,
!          x(i,j), j=0,1,...,n-1, is stored as
!
!               c(i,j) = x(i,j-1) , j=1, 2, . . . , n.
!
!  wsave   a real one-dimensional work array dimensioned n+15.
!          the wsave array must be initialized by 
!          calling subroutine fti.  a different wsave array must be
!          used for each different value of n.  this initialization does
!          not have to be repeated so long as n remains unchanged.  the
!          same wsave array may be used by ftrf and ftrb.
!
!  output parameters
!
!  c       contains the fourier coefficients f(k) for each of the m 
!          input sequences.  specifically, row i of c, c(i,j), 
!          j=1,2,..,n, contains the independent fourier coefficients
!          f(i,k), for the i-th input sequence stored as
!
!             c(i,1) = real( f(i,0) ),
!                    = sqrt(1/n)*sum(j=0,n-1)[ x(i,j) ],
!
!             c(i,2*k) = real( f(i,k) )
!                      = sqrt(1/n)*sum(j=0,n-1)[x(i,j)*cos(2j*k*pi/n)]
!
!             c(i,2*k+1) = imag( f(i,k) )
!                        =-sqrt(1/n)*sum(j=0,n-1)[x(i,j)*sin(2j*k*pi/n)]
!
!                   for k = 1, 2, . . . , m-1,
!
!              and, when n is even,
!
!              c(i,n) = real( f(i,n/2) ).
!                     = sqrt(1/n)*sum(j=0,n-1)[ (-1)**j*x(i,j) ].
!
!  wsave contains results which must not be destroyed between calls
!          to ftrf or ftrb.
!
!  -----------------------------------------------------------------
!
!  note  -  a call of ftrf followed immediately by a call of
!           of ftrb will return the original sequences c.  thus,
!           ftrb is the correctly normalized inverse of ftrf.
!
!  -----------------------------------------------------------------
!
!  ftrf is a straightforward extension of the subprogram rfftf to
!  handle m simultaneous sequences.  rfftf was originally developed
!  by p. n. swarztrauber of ncar.
!
!
!***references  p. n. swarztrauber, vectorizing the ffts, in parallel
!               computations, (g. rodrigue, ed.), academic press, 1982,
!               pp. 51-83.
!***end prologue  ftrf
!
!     fftpk, version 3, sept 2000
!
module RealFwd
  use Radix2f
  use Radix3f
  use Radix4f
  use Radix5f
  use Radixgf
  implicit none
  private
  public ftrf
  interface ftrf
    module procedure sftrf,dftrf,svftrf,dvftrf
  end interface
contains
!
!     single precision scalar version
!
  subroutine sftrf(c,wsave)
    integer, parameter :: kr=kind(0.)
    real(kr), dimension(:), intent(inout) :: c
    real(kr), dimension(:), intent(in), target :: wsave
    integer, dimension(15) :: fac
!
    real(kr), parameter :: one=1._kr
    real(kr), dimension(size(c)) :: ch
    real(kr), dimension(:), pointer :: w1,w2,w3,w4
    real(kr) scale
    integer n,nf,na,l1,iw,k1,kh,ip,l2,ido,ix2,ix3,ix4
!
    n=size(c)
!
    fac = wsave(n+1:)
    if (n==1) return
    nf = fac(2)
    na = 1
    l2 = n
    iw = n
    do k1=1,nf
      kh = nf-k1
      ip = fac(kh+3)
      l1 = l2/ip
      ido = n/l2
      iw = iw-(ip-1)*ido
      na = 1-na
      w1 => wsave(iw:iw+ido-1)
      select case(ip)
      case (4)
        ix2 = iw+ido
        ix3 = ix2+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        select case (na)
        case (0)
          call radf4 (ido,l1,c,ch,w1,w2,w3)
        case (1)
          call radf4 (ido,l1,ch,c,w1,w2,w3)
        end select
      case (2)
        select case (na)
        case (0)
          call radf2 (ido,l1,c,ch,w1)
        case (1)
          call radf2 (ido,l1,ch,c,w1)
        end select
      case (3)
        ix2 = iw+ido
        w2 => wsave(ix2:ix2+ido-1)
        select case (na)
        case (0)
          call radf3 (ido,l1,c,ch,w1,w2)
        case (1)
          call radf3 (ido,l1,ch,c,w1,w2)
        end select
      case (5)
        ix2 = iw+ido
        ix3 = ix2+ido
        ix4 = ix3+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        w4 => wsave(ix4:ix4+ido-1)
        select case (na)
        case (0)
          call radf5(ido,l1,c,ch,w1,w2,w3,w4)
        case (1)
          call radf5(ido,l1,ch,c,w1,w2,w3,w4)
        end select
      case default
        if (ido==1) na = 1-na
        select case (na)
        case (0)
          call radfg(ido,ip,l1,c,ch,w1)
          na = 1
        case (1)
          call radfg(ido,ip,l1,ch,c,w1)
          na = 0
        end select
      end select
      l2 = l1
    enddo
    scale=one/sqrt(real(n,kr))
    select case (na)
    case (0)
      c = scale*ch
    case (1)
      c = scale*c
    end select
  end subroutine
!
!     double precision vector version
!
  subroutine dftrf(c,wsave)
    integer, parameter :: kr=kind(0.d0)
    real(kr), dimension(:), intent(inout) :: c
    real(kr), dimension(:), intent(in), target :: wsave
    integer, dimension(15) :: fac
!
    real(kr), parameter :: one=1._kr
    real(kr), dimension(size(c)) :: ch
    real(kr), dimension(:), pointer :: w1,w2,w3,w4
    real(kr) scale
    integer n,nf,na,l1,iw,k1,kh,ip,l2,ido,ix2,ix3,ix4
!
    n=size(c)
!
    fac = wsave(n+1:)
    if (n==1) return
    nf = fac(2)
    na = 1
    l2 = n
    iw = n
    do k1=1,nf
      kh = nf-k1
      ip = fac(kh+3)
      l1 = l2/ip
      ido = n/l2
      iw = iw-(ip-1)*ido
      na = 1-na
      w1 => wsave(iw:iw+ido-1)
      select case(ip)
      case (4)
        ix2 = iw+ido
        ix3 = ix2+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        select case (na)
        case (0)
          call radf4 (ido,l1,c,ch,w1,w2,w3)
        case (1)
          call radf4 (ido,l1,ch,c,w1,w2,w3)
        end select
      case (2)
        select case (na)
        case (0)
          call radf2 (ido,l1,c,ch,w1)
        case (1)
          call radf2 (ido,l1,ch,c,w1)
        end select
      case (3)
        ix2 = iw+ido
        w2 => wsave(ix2:ix2+ido-1)
        select case (na)
        case (0)
          call radf3 (ido,l1,c,ch,w1,w2)
        case (1)
          call radf3 (ido,l1,ch,c,w1,w2)
        end select
      case (5)
        ix2 = iw+ido
        ix3 = ix2+ido
        ix4 = ix3+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        w4 => wsave(ix4:ix4+ido-1)
        select case (na)
        case (0)
          call radf5(ido,l1,c,ch,w1,w2,w3,w4)
        case (1)
          call radf5(ido,l1,ch,c,w1,w2,w3,w4)
        end select
      case default
        if (ido==1) na = 1-na
        select case (na)
        case (0)
          call radfg(ido,ip,l1,c,ch,w1)
          na = 1
        case (1)
          call radfg(ido,ip,l1,ch,c,w1)
          na = 0
        end select
      end select
      l2 = l1
    enddo
    scale=one/sqrt(real(n,kr))
    select case (na)
    case (0)
      c = scale*ch
    case (1)
      c = scale*c
    end select
  end subroutine
!
!     single precision vector version
!
  subroutine svftrf(c,wsave)
    integer, parameter :: kr=kind(0.)
    real(kr), dimension(:,:), intent(inout) :: c
    real(kr), dimension(:), intent(in), target :: wsave
    integer, dimension(15) :: fac
!
    real(kr), parameter :: c1=1._kr
    real(kr), dimension(size(c,1),size(c,2)) :: ch
    real(kr), dimension(:), pointer :: w1,w2,w3,w4
    real(kr) scale
    integer m,n,nf,na,l1,iw,k1,kh,ip,l2,ido,ix2,ix3,ix4
!
    n=size(c,2) 
    m=size(c,1)
    fac = wsave(n+1:)
!
    if (n==1) return
    nf = fac(2)
    na = 1
    l2 = n
    iw = n
    do k1=1,nf
      kh = nf-k1
      ip = fac(kh+3)
      l1 = l2/ip
      ido = n/l2
      iw = iw-(ip-1)*ido
      na = 1-na
      w1 => wsave(iw:iw+ido-1)
      select case(ip)
      case (4)
        ix2 = iw+ido
        ix3 = ix2+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        select case (na)
        case (0)
          call radf4 (ido,l1,c,ch,w1,w2,w3)
        case (1)
          call radf4 (ido,l1,ch,c,w1,w2,w3)
        end select
      case (2)
        select case (na)
        case (0)
          call radf2 (ido,l1,c,ch,w1)
        case (1)
          call radf2 (ido,l1,ch,c,w1)
        end select
      case (3)
        ix2 = iw+ido
        w2 => wsave(ix2:ix2+ido-1)
        select case (na)
        case (0)
          call radf3 (ido,l1,c,ch,w1,w2)
        case (1)
          call radf3 (ido,l1,ch,c,w1,w2)
        end select
      case (5)
        ix2 = iw+ido
        ix3 = ix2+ido
        ix4 = ix3+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        w4 => wsave(ix4:ix4+ido-1)
        select case (na)
        case (0)
          call radf5(ido,l1,c,ch,w1,w2,w3,w4)
        case (1)
          call radf5(ido,l1,ch,c,w1,w2,w3,w4)
        end select
      case default
        if (ido==1) na = 1-na
        select case (na)
        case (0)
          call radfg(ido,ip,l1,c,ch,w1)
          na = 1
        case (1)
          call radfg(ido,ip,l1,ch,c,w1)
          na = 0
        end select
      end select
      l2 = l1
    enddo
    scale=c1/sqrt(real(n,kr))
    select case (na)
    case (0)
      c = scale*ch
    case (1)
      c = scale*c
    end select
  end subroutine
!
!     double precision vector version
!
  subroutine dvftrf(c,wsave)
    integer, parameter :: kr=kind(0.d0)
    real(kr), dimension(:,:), intent(inout) :: c
    real(kr), dimension(:), intent(in), target :: wsave
    integer, dimension(15) :: fac
!
    real(kr), parameter :: c1=1._kr
    real(kr), dimension(size(c,1),size(c,2)) :: ch
    real(kr), dimension(:), pointer :: w1,w2,w3,w4
    real(kr) scale
    integer m,n,nf,na,l1,iw,k1,kh,ip,l2,ido,ix2,ix3,ix4
!
    n=size(c,2) 
    m=size(c,1)
    fac = wsave(n+1:)
!
    if (n==1) return
    nf = fac(2)
    na = 1
    l2 = n
    iw = n
    do k1=1,nf
      kh = nf-k1
      ip = fac(kh+3)
      l1 = l2/ip
      ido = n/l2
      iw = iw-(ip-1)*ido
      na = 1-na
      w1 => wsave(iw:iw+ido-1)
      select case(ip)
      case (4)
        ix2 = iw+ido
        ix3 = ix2+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        select case (na)
        case (0)
          call radf4 (ido,l1,c,ch,w1,w2,w3)
        case (1)
          call radf4 (ido,l1,ch,c,w1,w2,w3)
        end select
      case (2)
        select case (na)
        case (0)
          call radf2 (ido,l1,c,ch,w1)
        case (1)
          call radf2 (ido,l1,ch,c,w1)
        end select
      case (3)
        ix2 = iw+ido
        w2 => wsave(ix2:ix2+ido-1)
        select case (na)
        case (0)
          call radf3 (ido,l1,c,ch,w1,w2)
        case (1)
          call radf3 (ido,l1,ch,c,w1,w2)
        end select
      case (5)
        ix2 = iw+ido
        ix3 = ix2+ido
        ix4 = ix3+ido
        w2 => wsave(ix2:ix2+ido-1)
        w3 => wsave(ix3:ix3+ido-1)
        w4 => wsave(ix4:ix4+ido-1)
        select case (na)
        case (0)
          call radf5(ido,l1,c,ch,w1,w2,w3,w4)
        case (1)
          call radf5(ido,l1,ch,c,w1,w2,w3,w4)
        end select
      case default
        if (ido==1) na = 1-na
        select case (na)
        case (0)
          call radfg(ido,ip,l1,c,ch,w1)
          na = 1
        case (1)
          call radfg(ido,ip,l1,ch,c,w1)
          na = 0
        end select
      end select
      l2 = l1
    enddo
    scale=c1/sqrt(real(n,kr))
    select case (na)
    case (0)
      c = scale*ch
    case (1)
      c = scale*c
    end select
  end subroutine
end module RealFwd
