/*
 * load_pud.c
 *
 * crafted - a pud editor for the freecraft project.
 * 
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "load_pud.h"
#include "iolib.h"
#include "crafted.h"
#include "unit.h"

#define DebugLevel1 g_print
#define DebugLevel2 g_print
#define DebugLevel3 g_print

#define TilesetSummer           0
#define TilesetWinter           1
#define TilesetWasteland        2
#define TilesetSwamp            3

static int PudReadHeader(CLFile* input, char *header, long *length)
{
  long len;

  if (CLread(input, header, 4) != 4)
  {
    return 0;
  }
  if (CLread(input, &len, 4) != 4)
  {
    perror("CLread()");
/*    exit(-1);*/
  }
  *length = len;/* ConvertLE32(len);*/
  return 1;
}

void ConvertMTXM(const unsigned short* mtxm)
{
  gint i;

  for (i = 0 ; i<Pud.width*Pud.height ; i++)
    Pud.fields[i] = mtxm[i];
}

static int PudReadWord(CLFile* input)
{
  unsigned short temp_short;

  if (CLread(input, &temp_short, 2) != 2)
  {
    perror("CLread()");
/*    exit(-1);*/
  }
  return temp_short;/*ConvertLE16(temp_short);*/
}

static int PudReadByte(CLFile *input)
{
  unsigned char temp_char;

  if (CLread(input, &temp_char, 1) != 1)
  {
    perror("CLread()");
/*    exit(-1);*/
  }

  return temp_char;
}

char *unit_names[] = {
"Footman",
"Grunt",
"Peasant",
"Peon",
"Ballista",
"Catapult",
"Knight",
"Ogre",
"Archer",
"Axethrower",
"Mage",
"DeathKnight",
"Paladin",
"OgreMage",
"Dwarves",
"GoblinSappers",
"AttackPeasant",
"AttackPeon",
"Ranger",
"Berserker",
"Alleria",
"TeronGorefiend",
"KurdanAndSky_ree",
"Dentarg",
"Khadgar",
"GromHellscream",
"TankerHuman",
"TankerOrc",
"TransportHuman",
"TransportOrc",
"ElvenDestroyer",
"TrollDestroyer",
"Battleship",
"Juggernaught",
"Nothing",
"Deathwing",
"Nothing1",
"Nothing2",
"GnomishSubmarine",
"GiantTurtle",
"GnomishFlyingMachine",
"GoblinZeppelin",
"GryphonRider",
"Dragon",
"Turalyon",
"EyeOfKilrogg",
"Danath",
"KorgathBladefist",
"Nothing3",
"Cho_gall",
"Lothar",
"Gul_dan",
"UtherLightbringer",
"Zuljin",
"Nothing4",
"Skeleton",
"Daemon",
"Critter",
"Farm",
"PigFarm",
"BarracksHuman",
"BarracksOrc",
"Church",
"AltarOfStorms",
"ScoutTowerHuman",
"ScoutTowerOrc",
"Stables",
"OgreMound",
"GnomishInventor",
"GoblinAlchemist",
"GryphonAviary",
"DragonRoost",
"ShipyardHuman",
"ShipyardOrc",
"TownHall",
"GreatHall",
"ElvenLumberMill",
"TrollLumberMill",
"FoundryHuman",
"FoundryOrc",
"MageTower",
"TempleOfTheDamned",
"BlacksmithHuman",
"BlacksmithOrc",
"RefineryHuman",
"RefineryOrc",
"OilPlatformHuman",
"OilPlatformOrc",
"Keep",
"Stronghold",
"Castle",
"Fortress",
"GoldMine",
"OilPatch",
"StartLocationHuman",
"StartLocationOrc",
"GuardTowerHuman",
"GuardTowerOrc",
"CannonTowerHuman",
"CannonTowerOrc",
"CircleofPower",
"DarkPortal",
"Runestone",
"WallHuman",
"WallOrc",
"DeadBody",
"Destroyed1x1Place",
"Destroyed2x2Place",
"Destroyed3x3Place",
"Destroyed4x4Place",
"PeasantWithGold",
"PeonWithGold",
"PeasantWithWood",
"PeonWithWood",
"TankerHumanFull",
"TankerOrcFull" };
        


void load_pud(gchar *filename)
{
  CLFile* input;
  /*unsigned*/ long length;
  char header[5];
  char buf[1024];
  guint width = 0, height = 0;
  int aiopps;
  gchar *pud;

  pud = g_strdup(filename);

  if (Pud.path) g_free(Pud.path);
  Pud.path = g_strdup(pud);

  if (access(Pud.path, W_OK) == -1)
    Pud.readonly = TRUE;
  else
    Pud.readonly = FALSE;

  if (!(input=CLopen(pud)))
  {
    fprintf(stderr, "Try ./path/name\n");
    sprintf(buf, "pud: CLopen(%s)", pud);
    perror(buf);
    exit(-1);
  }
  header[4]='\0';
  if (!PudReadHeader(input,header,&length))
  {
    fprintf(stderr,"LoadPud: %s: invalid pud\n", pud);
    exit(-1);
  }
  if (memcmp(header,"TYPE",4) || length!=16)
  {
    fprintf(stderr,"LoadPud: %s: invalid pud\n", pud);
    exit(-1);
  }
  if (CLread(input,buf,16)!=16)
  { /* Ignore Type */
    perror("CLread()");
    exit(-1);
  }
  if (strcmp(buf,"WAR2 MAP"))
  { /* Only check string */
    fprintf(stderr, "LoadPud: %s: invalid pud\n", pud);
    exit(-1);
  }

  aiopps = width = height = 0;

  /*
   * Parse all sections.
   */
  while (PudReadHeader(input,header,&length))
  {
    if (verbosity) 
      DebugLevel3("\tSection: %4.4s\n", header);

    /*
     * PUD version
     */
    if (!memcmp(header,"VER ",4))
    {
      if (length == 2)
      {
        int v;

        v = PudReadWord(input);
        if (verbosity)
          DebugLevel1("\tVER: %d.%d\n", (v & 0xF0) >> 4 , v & 0xF);
        Pud.version = (v & 0xF0) >> 4;
        Pud.sub_version = v & 0xF;
        continue;
      }
      if (verbosity)
        DebugLevel1("Wrong version length\n");
    }
    /*
     * Map description
     */
    if (!memcmp(header, "DESC" ,4))
    {
      if (CLread(input, buf, length) != length)
      {
        perror("CLread()");
        exit(-1);
      }
      if (verbosity)
        DebugLevel1("\tDESC: %s\n", buf);
      if (Pud.description) g_free(Pud.description);
      Pud.description = g_malloc(length+1);
      strncpy(Pud.description, buf, length);
      Pud.description[length] = 0;
      continue;
    }
    /*
     * Player definitions.
     */
    if (!memcmp(header, "OWNR", 4))
    {
      if (length == 16)
      {
        int i;
        int p;

        for (i = 0; i < 16 ; i++)
        {
          p = PudReadByte(input);
          g_print("OWNR(%i): %X\n", i, p);
          Pud.player[i].owner = p;
/*          if (GameSettings.Opponents != SettingsPresetMapDefault)
          {
            if (p == PlayerComputer)
            {
              if (aiopps < GameSettings.Opponents)
              {
                aiopps++;
              } else
              {
                p = PlayerNobody;
              }
            }
          }
          CreatePlayer("Computer", p);*/
        }
        continue;
      } else
      {
        if (verbosity)
          DebugLevel1("Wrong player length\n");
      }
    }
    /*
     * Terrain type or extended terrain type.
     */
    if (!memcmp(header, "ERA ", 4) || !memcmp(header, "ERAX", 4))
    {
      if (length == 2)
      {
        int t;

        t = PudReadWord(input);
        switch (t)
        {
          case TilesetSummer:
            if (verbosity)
              DebugLevel3("\tTerrain: SUMMER\n");
            break;
          case TilesetWinter:
            if (verbosity)
              DebugLevel3("\tTerrain: WINTER\n");
            break;
          case TilesetWasteland:
            if (verbosity)
              DebugLevel3("\tTerrain: WASTELAND\n");
            break;
          case TilesetSwamp:
            if (verbosity)
              DebugLevel3("\tTerrain: SWAMP\n");
            break;
          default:
            if (verbosity)
              DebugLevel1("Unknown terrain %d\n",t);
            t = TilesetSummer;
            break;
        }
        Pud.terrain = t;
        continue;
      } else
      {
        if (verbosity)
          DebugLevel1("Wrong terrain type length\n");
      }
    }
    /*
     * Dimension
     */
    if (!memcmp(header, "DIM ", 4))
    {
      width  = PudReadWord(input);
      height = PudReadWord(input);
      if (verbosity)
        DebugLevel2("\tMap %d x %d\n", width, height);
      Pud.width  = width;
      Pud.height = height;
      if (Pud.fields) g_free(Pud.fields);
      Pud.fields = g_malloc(width * height * sizeof(gushort));
      continue;
    }
    /*
     * Unit data (optional)
     */
    if (!memcmp(header, "UDTA", 4))
    {
      char *bufp;

      length -= 2;
      if (PudReadWord(input))
      {
        if (verbosity)
          DebugLevel3("\tUsing default data\n");
        CLseek(input, length, SEEK_CUR);
      } else
      {
        if (length < (long)sizeof(buf))
        {
          bufp = buf;
        } else if (!(bufp = alloca(length)))
        {
          perror("alloca()");
          exit(-1);
        }
        if (CLread(input, bufp, length) != length)
        {
          perror("CLread()");
          exit(-1);
        }
/*        ParsePudUDTA(bufp,length);*/
      }
      continue;
    }
    /*
     * Pud restrictions (optional)
     */
    if (!memcmp(header, "ALOW", 4))
    {
      char *bufp;

      if (length < (long)sizeof(buf))
      {
        bufp = buf;
      } else if (!(bufp = alloca(length)))
      {
        perror("alloca()");
        exit(-1);
      }
      if (CLread(input, bufp, length) != length)
      {
        perror("CLread()");
        exit(-1);
      }
/*      ParsePudALOW(bufp, length);*/
      continue;
    }
    /*
     * Upgrade data (optional)
     */
    if (!memcmp(header, "UGRD", 4))
    {
      char *bufp;

      length -= 2;
      if (PudReadWord(input))
      {
        if (verbosity)
          DebugLevel3("\tUsing default data\n");
        CLseek(input, length, SEEK_CUR);
      } else
      {
        if (length < (long)sizeof(buf))
        {
          bufp = buf;
        } else if (!(bufp = alloca(length)))
        {
          perror("alloca()");
          exit(-1);
        }
        if (CLread(input, bufp, length) != length)
        {
          perror("CLread()");
          exit(-1);
        }
/*        ParsePudUGRD(bufp,length);*/
      }
      continue;
    }
    /*
     * Identifies race of each player
     */
    if (!memcmp(header, "SIDE", 4))
    {
      if (length == 16)
      {
        int i;
        int v;

        for (i = 0 ; i < 16 ; i++)
        {
          v = PudReadByte(input);
          Pud.player[i].side = v;
/*          switch (v)
          {
            case PlayerRaceHuman:
            case PlayerRaceOrc:
            case PlayerRaceNeutral:
              break;
            default:
              DebugLevel1("Unknown race %d\n", v);
              v = PlayerRaceNeutral;
              break;
          }
          if (GameSettings.Presets[i].Race == SettingsPresetMapDefault)
          {
            PlayerSetSide(&Players[i], v);
          } else
          {
            PlayerSetSide(&Players[i], GameSettings.Presets[i].Race);
          }*/
        }
        continue;
      } else
      {
        if (verbosity)
          DebugLevel1("Wrong side length\n");
      }
    }
    /*
     * Starting gold
     */
    if (!memcmp(header, "SGLD", 4))
    {
      if (length == 32)
      {
        int i;
        int v;

        for (i = 0 ; i < 16 ; i++)
        {
          v = PudReadWord(input);
          Pud.player[i].gold = v;
        }
        continue;
      } else
      {
        if (verbosity)
          DebugLevel1("Wrong starting gold length\n");
      }
    }
    /*
     * Starting lumber
     */
    if (!memcmp(header, "SLBR", 4))
    {
      if (length == 32)
      {
        int i;
        int v;

        for (i = 0 ; i < 16 ; i++)
        {
          v = PudReadWord(input);
          Pud.player[i].wood = v;
        }
        continue;
      } else
      {
        if (verbosity)
          DebugLevel1("Wrong starting lumber length\n");
      }
    }
    /*
     * Starting oil
     */
    if (!memcmp(header, "SOIL", 4))
    {
      if (length == 32)
      {
        int i;
        int v;

        for (i = 0 ; i < 16 ; i++)
        {
          v = PudReadWord(input);
          Pud.player[i].oil = v;
        }
        continue;
      } else
      {
        if (verbosity)
          DebugLevel1("Wrong starting oil length\n");
      }
    }
    /* FIXME: support the extended resources with puds? */

    /*
     * AI for each player
     */
    if (!memcmp(header, "AIPL", 4))
    {
      if (length == 16)
      {
        int i;
        int v;

        for (i = 0 ; i < 16 ; i++)
        {
          v = PudReadByte(input);
          Pud.player[i].ai = v;
        }
        continue;
      } else
      {
        if (verbosity)
          DebugLevel1("Wrong AI player length\n");
      }
    }
    /*
     * obsolete oil map
     */
    if (!memcmp(header, "OILM", 4))
    {
      CLseek(input, length, SEEK_CUR); /* skip section */
      continue;
    }
    /*
     * Tiles MAP
     */
    if (!memcmp(header, "MTXM", 4))
    {
      unsigned short *mtxm;

      if (length != (long)(width*height*2))
      {
        if (verbosity)
          DebugLevel1("wrong length of MTXM section %ld\n", length);
        exit(-1);
      }
      if (!(mtxm = alloca(length)))
      {
        perror("alloca()");
        exit(-1);
      }
      if (CLread(input, mtxm, length) != length)
      {
        perror("CLread()");
        exit(-1);
      }
      ConvertMTXM(mtxm);
      continue;
    }
    /*
     * Movement MAP
     */
    if (!memcmp(header, "SQM ", 4))
    {
      unsigned short* sqm;

      if (length != (long)(width*height*sizeof(short)))
      {
        if (verbosity)
          DebugLevel1("wrong length of SQM  section %ld\n", length);
        exit(-1);
      }
      if (!(sqm = alloca(length)))
      {
        perror("alloca()");
        exit(-1);
      }
      if (CLread(input, sqm, length) != length)
      {
        perror("CLread()");
        exit(-1);
      }
      /*ConvertSQM(sqm);*/
      continue;
    }
    /*
     * Action MAP
     */
    if (!memcmp(header, "REGM", 4))
    {
      unsigned short* regm;

      if (length != (long)(width*height*sizeof(short)))
      {
        if (verbosity)
          DebugLevel1("wrong length of REGM section %ld\n", length);
        exit(-1);
      }
      if (!(regm = alloca(length)))
      {
        perror("alloca()");
        exit(-1);
      }
      if (CLread(input, regm, length) != length)
      {
        perror("CLread()");
        exit(-1);
      }
      /*ConvertREGM(regm);*/
      continue;
    }
    /*
     * Units
     */
    if (!memcmp(header, "UNIT", 4))
    {
      int x;
      int y;
      int t;
      int o;
      int v;

      while (length >= 8)
      {
        pud_unit_t *pud_unit;
        x = PudReadWord(input);
        y = PudReadWord(input);
        t = PudReadByte(input);
        o = PudReadByte(input);
        v = PudReadWord(input);

        g_print("unit: (%d, %d), %d, %d: %s(%d)\n", x, y, o, v, unit_names[t], t);
        pud_unit = g_malloc(sizeof(pud_unit_t));
        pud_unit->x = x; pud_unit->y = y;
        pud_unit->type = t;
        pud_unit->player_number = o;
        pud_unit->value = v;
        pud_unit->setting_dialog = NULL;
        Pud.units = g_list_append(Pud.units, pud_unit);
        length -= 8;
      }
      continue;
    }
    if (verbosity)
      DebugLevel2("Unsupported Section: %4.4s\n", header);
    CLseek(input, length, SEEK_CUR);
  }
  CLclose(input);

  after_load();
  Pud.is_loaded = TRUE;
}

guchar *load_pud_for_thumbnail(gchar *filename)
{
  guchar *buf = NULL;
  guchar load_buf[1024];
  CLFile *input;
  char header[5];
  /*unsigned*/ long length;
  guint width = 0, height = 0, i, j;

  if (!(input = CLopen(filename)))
    return NULL;

  header[4]='\0';
  if (!PudReadHeader(input, header, &length))
    return NULL;

  if (memcmp(header,"TYPE",4) || length!=16)
    return NULL;

  if (CLread(input, load_buf, 16)!=16)
    return NULL;

  if (strcmp(load_buf, "WAR2 MAP"))
    return NULL;

  /*
   * Parse all sections.
   */
  while (PudReadHeader(input, header, &length))
  {  
    /*
     * skip unwanted sections
     */
    if (!memcmp(header, "OILM", 4) || !memcmp(header, "SQM ", 4) ||
        !memcmp(header, "REGM", 4) || !memcmp(header, "UNIT", 4) ||
        !memcmp(header, "AIPL", 4) || !memcmp(header, "SOIL", 4) ||
        !memcmp(header, "SLBR", 4) || !memcmp(header, "SGLD", 4) ||
        !memcmp(header, "SIDE", 4) || !memcmp(header, "UGRD", 4) ||
        !memcmp(header, "ALOW", 4) || !memcmp(header, "UDTA", 4) || 
        !memcmp(header, "ERA ", 4) || !memcmp(header, "ERAX", 4) || 
        !memcmp(header, "OWNR", 4) || !memcmp(header, "DESC", 4) || 
        !memcmp(header, "VER ", 4)                                   )
    {
      CLseek(input, length, SEEK_CUR); /* skip section */
      continue;
    }
    /*
     * Dimension
     */
    if (!memcmp(header, "DIM ", 4))
    {
      width  = PudReadWord(input);
      height = PudReadWord(input);
      continue;
    }
    /*
     * Tiles MAP
     */
    if (!memcmp(header, "MTXM", 4))
    {
      gushort *mtxm;
      gushort tile;

      if (length != (long)(width*height*2))
        return NULL;

      if (!(mtxm = alloca(length)))
        return NULL;

      if (CLread(input, mtxm, length) != length)
        return NULL;

      buf = g_malloc0(128 * 128 * 3);
      
      for (i = 0 ; i < MIN(128, height) ; i++)
        for (j = 0 ; j < MIN(128, width) ; j++)
        {
          tile = mtxm[i*width+j];
          switch (tile >> 8)
          {
            case 0: /* plain tile */
              switch (tile >> 4)
              {
                case 1: /* light water */
                case 2: /* dark water */
                  buf[3*(i*128+j)+0] = 0;
                  buf[3*(i*128+j)+1] = 128;
                  buf[3*(i*128+j)+2] = 255;
                  break;
                case 3: /* light coast */
                  buf[3*(i*128+j)+0] = 187;
                  buf[3*(i*128+j)+1] = 140;
                  buf[3*(i*128+j)+2] = 102;
                  break;
                case 4: /* dark coast */
                  buf[3*(i*128+j)+0] = 121;
                  buf[3*(i*128+j)+1] = 75;
                  buf[3*(i*128+j)+2] = 38;
                  break;
                case 5: /* light grass */
                  buf[3*(i*128+j)+0] = 32;
                  buf[3*(i*128+j)+1] = 213;
                  buf[3*(i*128+j)+2] = 41;
                  break;
                case 6: /* dark grass */
                  buf[3*(i*128+j)+0] = 23;
                  buf[3*(i*128+j)+1] = 174;
                  buf[3*(i*128+j)+2] = 30;
                  break;
                case 7: /* forest */
                  buf[3*(i*128+j)+0] = 55;
                  buf[3*(i*128+j)+1] = 101;
                  buf[3*(i*128+j)+2] = 57;
                  break;
                case 8: /* rocks */
                  buf[3*(i*128+j)+0] = 192;
                  buf[3*(i*128+j)+1] = 192;
                  buf[3*(i*128+j)+2] = 192;
                  break;
                default:
                  g_print("FIXME!!!\ttile:%X\n", tile);
              }
              break;
            case 1: /* dark water/light water */
              buf[3*(i*128+j)+0] = 0;
              buf[3*(i*128+j)+1] = 128;
              buf[3*(i*128+j)+2] = 255;
              break;
            case 2: /* light water/light coast */
              buf[3*(i*128+j)+0] = 187;
              buf[3*(i*128+j)+1] = 140;
              buf[3*(i*128+j)+2] = 102;
              break;
            case 3: /* dark coast/light coast */
              buf[3*(i*128+j)+0] = 121;
              buf[3*(i*128+j)+1] = 75;
              buf[3*(i*128+j)+2] = 38;
              break;
            case 4: /* rocks/light coast */
              buf[3*(i*128+j)+0] = 192;
              buf[3*(i*128+j)+1] = 192;
              buf[3*(i*128+j)+2] = 192;
              break;
            case 5: /* light coast/light grass */
              buf[3*(i*128+j)+0] = 32;
              buf[3*(i*128+j)+1] = 213;
              buf[3*(i*128+j)+2] = 41;
              break;
            case 6: /* dark grass/light grass */
              buf[3*(i*128+j)+0] = 23;
              buf[3*(i*128+j)+1] = 174;
              buf[3*(i*128+j)+2] = 30;
              break;
            case 7: /* forest/light grass */
              buf[3*(i*128+j)+0] = 55;
              buf[3*(i*128+j)+1] = 101;
              buf[3*(i*128+j)+2] = 57;
              break;
            default:
              g_print("FIXME!!!\ttile:%X\n", tile);
          }
        }
      CLclose(input);
      return buf;
    }
  }
  CLclose(input);
  return NULL;
}

