/*
 * $Id: dapto.c,v 1.69 2001/06/11 21:04:40 tony Exp $
 * Copyright (c) 1994-1998 Anthony M. Sloane
 */

/* This file is part of the Eli Module Library.

The Eli Module Library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

The Eli Module Library is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with the Eli Module Library; see the file COPYING.LIB.
If not, write to the Free Software Foundation, Inc., 59 Temple Place -
Suite 330, Boston, MA 02111-1307, USA.  */

/* As a special exception, when this file is copied by Eli into the
   directory resulting from a :source derivation, you may use that
   created file as a part of that directory without restriction. */

#ifdef MONITOR

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <math.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include "dapto.h"
#include "dapto_dapto.h"
#if defined(__SVR4) || defined(__svr4__)
#include <memory.h>
#endif
#include <sys/ipc.h>
#include <sys/shm.h>
#include "obstack.h"
#include "eliproto.h"

/*
#define DEBUG
*/

#if defined(sun) && !defined(__SVR4) && !defined(__svr4__)
#define USE_ONEXIT
#define DECLARE_SHMAT
#endif

#ifdef DECLARE_SHMAT
#ifdef PROTO_OK
extern char *shmat (int shmid, char *shmaddr, int shmflg);
#else
extern char *shmat ();
#endif
#endif

extern void _dap_interface_init ();

/*
 * Constants and macros
 */

#define MAXMSGSIZE 100000

/*
 * Types
 */

typedef struct command_struct command;
struct command_struct {
    char *name;
    _dapto_cmd_func func;
};

/*
 * Variables
 */

int _dap_noosa_there;
long _dapto_time = 0L, _dapto_entered; 
char *_dapto_cp;
CONST char *_dapto_src;

static int no_stop_at_end = 0, sigusr1, ppid;

static int shmid;
char *_dapto_mbuf;

static Obstack cmd_obstk;
static command *commands;
static int num_commands = 0, doingcmd = 0;

/*
 * Signal tables (entries must match)
 * SIGINT, SIGILL and SIGPIPE aren't here because they may be generated by
 * the user, or dapto-generated things.  If they are here we get infinite
 * loops because we trap them. 
 * SIGQUIT isn't here because it is sometimes useful to be able to get a
 * core dump of the subject if things hang.
 * SIGHUP isn't here because that's what kill generates by default.
 */

static int sigcodes[] = { 
#if !defined(linux)
  SIGEMT,
  SIGBUS,
  SIGSYS,
#endif
  SIGTRAP,
  SIGABRT,
  SIGFPE,
  SIGSEGV,
  SIGTERM,
#if !defined(hpux)
  SIGXCPU,
  SIGXFSZ,
#endif
  0
};

static char *sigdescs[] = {
#if !defined(linux)
    "emulator trap",
    "bus error",
    "bad argument to system call",
#endif
    "trace trap",
    "abort",
    "arithmetic exception",
    "segmentation violation",
    "software termination",
#if !defined(hpux)
    "cpu time limit exceeded",
    "file size limit exceeded"
#endif
};
 
/*
 * init_comms
 */

#ifdef PROTO_OK
static void init_comms (int ppid)
#else
static void init_comms (ppid)
    int ppid;
#endif
{
    if ((shmid = shmget (ppid, MAXMSGSIZE, 0)) == -1) {
	perror ("dapto shmget");
	exit (1);
    }
    if ((_dapto_mbuf = (char *)shmat (shmid, 0, 0)) == (char *) -1) {
	perror ("dapto shmat");
	exit (1);
    }
}

/*
 * finit_comms
 */

static void finit_comms ()
{
    if (shmdt (_dapto_mbuf) == -1) {
	perror ("dapto shmdt");
	exit (1);
    }
}

/*
 * _dap_add_command
 */

#ifdef PROTO_OK
void _dap_add_command (char *name, _dapto_cmd_func func)
#else
void _dap_add_command (name, func)
    char *name;
    _dapto_cmd_func func;
#endif
{
    obstack_blank (&cmd_obstk, sizeof (command));
    commands = (command *) obstack_base (&cmd_obstk);
    commands[num_commands].name = name;
    commands[num_commands].func = func;
    num_commands++;
}

/* 
 * docommand
 * FIXME: uses linear search
 */

#ifdef PROTO_OK
static void docommand (char *cmd)
#else
static void docommand (cmd)
    char *cmd;
#endif
{
    int i;
    char *cp;

#ifdef DEBUG
    fprintf (stderr, "dapto: doing command '%s'\n", cmd);
#endif
    cp = strchr (cmd, ' ');
    if (cp == NULL)
	cp = "";
    else
	*cp++ = '\0';
    for (i = 0; i < num_commands; i++)
	if (strcmp (cmd, commands[i].name) == 0) {
	    (*commands[i].func) (cp);
	    return;
	}
#ifdef DEBUG
    fprintf (stderr, "dapto: unknown command '%s'\n", cmd);
#endif
}

/*
 * _dap_format
 */

#ifdef PROTO_OK
#include <stdarg.h>
char *_dap_format (char *s, ...)
{
    static char buf[1024];
    va_list ap;

    va_start (ap, s);
    vsprintf (buf, s, ap);
    return buf;
}
#else
#include <varargs.h>
char *_dap_format (va_alist) va_dcl
{
    static char buf[1024];
    char *s;
    va_list ap;

    va_start (ap);
    s = va_arg(ap, char *);
    vsprintf (buf, s, ap);
    va_end(ap);
    return buf;
}
#endif

/*
 * print_mesg
 */

#ifdef PROTO_OK
static void print_mesg (char *msg) 
#else
static void print_mesg (msg)
    char *msg;
#endif
{
    char *cp = msg;
    int size, i;
    long int l;

    switch (*cp++) {
    case 'e': fputs ("event ", stderr); break;
    case 'c': fputs ("cmd ", stderr); break;
    case 'r': fputs ("resp ", stderr); break;
    case 'x': fputs ("error ", stderr); break;
    }

    for ( ; *cp != '\0'; cp += size) 
	switch (*cp++) {
	    case 'i': {
                memcpy ((void *)&i, cp, sizeof (int));
		fprintf (stderr, "%d ", i);
		size = sizeof (int);
		break;
	    }

	    case 'l': {
                memcpy ((void *)&l, cp, sizeof (long int));
		fprintf (stderr, "%ld ", l);
		size = sizeof (long int);
		break;
	    }

	    case 's': {
		fprintf (stderr, "'%s' ", cp);	
		size = strlen (cp) + 1;
		break;
	    }

	    default:
		fprintf (stderr, "?%c ", *cp);
                size = 0;
	}
}

/*
 * _dap_send
 */

#ifdef PROTO_OK
void _dap_send (int cont)
#else
void _dap_send (cont)
    int cont;
#endif
{
    sigset_t MaskBuf, *Mask = &MaskBuf;
    int didcmd;

    /* Terminate with nul */
    *_dapto_cp++ = '\0';

#ifdef DEBUG
    fputs ("dapto: send ", stderr);
    print_mesg (_dapto_mbuf);
    fputc ('\n', stderr);
#endif

    /* Signal parent and wait for a response if necessary */
    sigusr1 = 0;
    if (kill (ppid, SIGUSR1) == -1) {
        perror ("dapto kill USR1");
        exit (1);
    }
    if (!cont) 
        do {
            didcmd = 0;
#ifdef DEBUG
            fprintf (stderr, "dapto: waiting for response...\n");
#endif
            if (!sigusr1) {
                if (sigemptyset (Mask) == -1) {
                    perror ("dapto sigemptyset");
                    exit (1);
                }
                if ((sigsuspend (Mask) == -1) && (errno != EINTR)) {
                    perror ("dapto sigsuspend");
                    exit (1);
                }
                sigusr1 = 0;
            }
            if (strcmp (_dapto_mbuf, "cont") == 0) {
#ifdef DEBUG
                fprintf (stderr, "dapto: continuing...\n");
#endif
                break;
            }
            if (*_dapto_mbuf == 'c') {
#ifdef DEBUG
                fprintf (stderr, "dapto: command '%s'\n", _dapto_mbuf);
#endif
                doingcmd = 1;
                docommand (_dapto_mbuf + 2);
                doingcmd = 0;
                didcmd = 1;
            }
        } while (didcmd);
}

/*
 * signal_handler
 */

#ifdef PROTO_OK
#if defined(sun) && !defined(__SVR4) && !defined(__svr4__)
static void signal_handler (int s, int code, struct sigcontext *scp,
			    char *addr)
#else
static void signal_handler (int s)
#endif
#else
#if defined(sun) && !defined(__SVR4) && !defined(__svr4__)
static void signal_handler (s, code, scp, addr)
    int s, code;
    struct sigcontext *scp;
    char *addr;
#else
static void signal_handler (s)
    int s;
#endif
#endif
{
    int i;
    CONST char *cp;
    char buf[256];
    
    for (i = 0; sigcodes[i] != 0; i++)
	if (sigcodes[i] == s)
	    break;
    cp = (sigcodes[i] == 0 ? "unknown": sigdescs[i]);
    if (doingcmd) {
        DAPTO_INIT_ERR;
        DAPTO_RESULT_STR (cp);
        _dap_send (0);
    }
    sprintf (buf, "end \"%s\"", cp);
    _dapto_stopped (buf, 0);
    no_stop_at_end = 1;
    exit (1);
}

/*
 * sigusr1_handler
 */

#ifdef PROTO_OK
#if defined(sun) && !defined(__SVR4) && !defined(__svr4__)
static void sigusr1_handler (int s, int code, struct sigcontext *scp,
			     char *addr)
#else
static void sigusr1_handler (int s)
#endif
#else
#if defined(sun) && !defined(__SVR4) && !defined(__svr4__)
static void sigusr1_handler (s, code, scp, addr)
    int s, code;
    struct sigcontext *scp;
    char *addr;
#else
static void sigusr1_handler (s)
    int s;
#endif
#endif
{
#ifdef DEBUG
    fprintf (stderr, "dapto: got SIGUSR1\n");
#endif
    sigusr1 = 1;
}

/*
 * init_signals
 */

static void init_signals ()
{
    int i;
    struct sigaction actbuf;
    sigset_t MaskBuf, *Mask = &MaskBuf;

    if (sigemptyset (&actbuf.sa_mask) == -1) {
	perror ("dapto sigemptyset init 1");
	exit (1);
    }
    actbuf.sa_flags = 0;
#ifdef __cplusplus
#if defined(sun) && !defined(__SVR4) && !defined(__svr4__)
    actbuf.sa_handler = (void (*)(...)) signal_handler;
#else
    actbuf.sa_handler = (void (*)(int)) signal_handler;
#endif
#else
    actbuf.sa_handler = signal_handler;
#endif

    for (i = 0; sigcodes[i] != 0; i++)
        if (sigaction (sigcodes[i], &actbuf, (struct sigaction *)0)
		== -1) {
	    perror ("dapto sigaction");
	    exit (1);
	}

    if (sigemptyset (Mask) == -1) {
	perror ("dapto sigemptyset init 2");
	exit (1);
    }
    if (sigaddset (Mask, SIGUSR1) == -1) {
	perror ("dapto sigaddset init");
	exit (1);
    }
    if (sigprocmask (SIG_BLOCK, Mask, (sigset_t *)0) == -1) {
        perror ("dapto sigblock SIGUSR1");
        exit (1);
    }
 
#ifdef __cplusplus
#if defined(sun) && !defined(__SVR4) && !defined(__svr4__)
    actbuf.sa_handler = (void (*)(...)) signal_handler;
#else
    actbuf.sa_handler = (void (*)(int)) signal_handler;
#endif
#else
    actbuf.sa_handler = sigusr1_handler;
#endif
    if (sigaction (SIGUSR1, &actbuf, (struct sigaction *)0) == -1) {
	perror ("dapto sigaction SIGUSR1");
	exit (1);
    }
}

/*
 * finit
 */

#ifdef USE_ONEXIT
#ifdef PROTO_OK
static void finit (int status, caddr_t arg)
#else
static void finit (status, arg)
    int status;
    caddr_t arg;
#endif
#else
static void finit ()
#endif
{
    _dapto_finit ();
    if (!no_stop_at_end) {
	_dapto_stopped ("end", 0);
	_dapto_exit ();
	finit_comms ();
    }
}

/*
 * _dapto_setup
 */

#ifdef PROTO_OK
void _dap_init (char *progname)
#else
void _dap_init (progname)
    char *progname;
#endif
{
    char *ppidbuf;

    ppidbuf = getenv ("NOOSA_PRESENT");
    _dap_noosa_there = (ppidbuf != (char *) NULL);
    if (_dap_noosa_there) {
	ppid = atoi (ppidbuf);
	init_comms (ppid);
        obstack_init (&cmd_obstk);
	_dap_interface_init ();
	init_signals ();
#ifdef USE_ONEXIT
	if (on_exit (finit, 0) != 0) {
#else
	if (atexit (finit) != 0) {
#endif
	    fprintf (stderr, "dapto: can't install exit handler\n");
	    exit (1);
	}
	_dapto_init (getpid ());
    }
}

#else
static int _dapto; /* To avoid messages about empty files. */
#endif

