#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/ip_icmp.h>
#include <netinet/ip.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include "global_vars.h"

int in_cksum(u_short *addr, int len);

int pinger_sock;
int pinger_plen;
struct sockaddr_in pinger_to;
char *pinger_pack;
char pinger_ready = 0;

int pinger_get_ip(struct line_t* line)
{ // returns ip in network byte order or 0
    struct hostent *he;
    struct in_addr ia;
    if ( ! line->pinger_hostname ) return(0);
    if ( !(he = gethostbyname(line->pinger_hostname)) )
    {
	syslog(LOG_INFO, "hostname lookup for '%s' failed. Switching off pinger of line '%s'.",
			line->pinger_hostname, line->linename);
	line->pinger = 0;
	return(0);
    }
    line->pinger_ip = *((int*)(he->h_addr_list[0]));
    ia.s_addr = line->pinger_ip;
    syslog(LOG_INFO, "Found ip '%s' for hostname '%s'.", inet_ntoa(ia), line->pinger_hostname);
    return(line->pinger_ip);
}

int pinger_init()
{
    struct protoent *proto;

    if ( !(proto = getprotobyname("icmp")) )
    {
	fprintf(stderr, "Pinger-ERROR: unknown protocol: icmp\n");
	return(-1);
    }
    if ( (pinger_sock = socket(AF_INET, SOCK_RAW, proto->p_proto)) == -1 )
    {
	if ( getuid() != 0 )
	    fprintf(stderr, "Pinger-ERROR: couldn't get raw-sock: need to be root!\n");
	else
	    fprintf(stderr, "Pinger-ERROR: couldn't get raw-sock...\n");
	return(-1);
    }
    /* done */
    fprintf(stdout, "- pinger initialized.\n");
    pinger_ready = 1;
    return(0);
}

int pinger_cleanup()
{
    if ( !pinger_ready ) return(0);
    pinger_ready = 0;
    close(pinger_sock);
    return(0);
}

int pinger_send(struct line_t* line)
{
    struct icmphdr *icmph = (struct icmphdr*)pinger_pack;
    register char *garbage;
    int i;
    if ( !pinger_ready )
    {
	syslog(LOG_ERR, "Pinger-ERROR: not initialized!");
	line->pinger = 0;
	return(-1);
    }
    // check whether we have to do a hostname lookup
    if ( (!line->pinger_ip) && (line->pinger_hostname) )
	if ( !pinger_get_ip(line) )
	{
	    syslog(LOG_WARNING, "Pinger: hostname lookup failure for '%s' on line '%s'",
		    line->pinger_hostname, line->linename);
	    return(-1);
	}
    // init packet
    pinger_plen = line->pinger_datasize + sizeof(struct icmphdr);
    if ( (pinger_pack = malloc(pinger_plen)) == NULL )
    {
	syslog(LOG_ERR, "Pinger-ERROR: malloc(%d) failed.\n", pinger_plen);
	return(-1);
    }
    garbage = pinger_pack + sizeof(struct icmphdr);
    for (i=0;i<line->pinger_datasize;i++)
    {
	*garbage = 1+(int)(60.0*rand());
	garbage++;
    }
    /* init icmp - header */
    icmph = (struct icmphdr*)pinger_pack;
    bzero(icmph, sizeof(struct icmphdr)); 
    icmph->type = ICMP_ECHO;
    icmph->code = 0;
    icmph->un.echo.id = 40312;
    icmph->un.echo.sequence = 0;
    icmph->checksum = in_cksum((u_short *)icmph, pinger_plen);
    /* init target - record */
    pinger_to.sin_family = AF_INET;
    pinger_to.sin_port = 80;
    pinger_to.sin_addr.s_addr = line->pinger_ip;
    // send the stuff
    sendto(pinger_sock, pinger_pack, pinger_plen, 0,
	   (struct sockaddr*)&pinger_to, sizeof(struct sockaddr_in));
		// don't check for errors or such things...
    free(pinger_pack);
    return(0);	// don't even wait for an echo.
}

/* if I remember correctly the following is stolen from the 'ping' utility */
int in_cksum(u_short *addr, int len)
{
    register int nleft = len;
    register u_short *w = addr;
    register int sum = 0;
    u_short answer = 0;
    /*
     * Our algorithm is simple, using a 32 bit accumulator (sum), we add
     * sequential 16 bit words to it, and at the end, fold back all the
     * carry bits from the top 16 bits into the lower 16 bits.
    */
    while (nleft > 1)  {
        sum += *w++;
        nleft -= 2;
    }
    /* mop up an odd byte, if necessary */
    if (nleft == 1) {
        *(u_char *)(&answer) = *(u_char *)w ;
        sum += answer;
    }
    /* add back carry outs from top 16 bits to low 16 bits */
    sum = (sum >> 16) + (sum & 0xffff);     /* add hi 16 to low 16 */
    sum += (sum >> 16);                     /* add carry */
    answer = ~sum;                          /* truncate to 16 bits */
    return(answer);
}
