/***************************************************************************
 $RCSfile: cardfs.cpp,v $
                             -------------------
    cvs         : $Id: cardfs.cpp,v 1.1.1.1 2003/01/25 22:36:44 mali Exp $
    begin       : Thu Aug 15 2002
    copyright   : (C) 2002 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#include <unistd.h>
#include <dirent.h>
#include <fcntl.h>
#include <stdio.h>
#include <utime.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>

#include <lufs/proto.h>
#include <lufs/fs.h>

#include "cardfs.h"

extern "C" {

void*
cardfs_init(struct list_head *cfg, struct dir_cache *cache, struct credentials *cred, void **global_fs){
    return (void*)new CardFS(cfg, cache);
}

void
cardfs_free(void *ctx){
    CardFS *p = (CardFS*)ctx;

    delete p;
}

int 	
cardfs_mount(void *ctx){
    return ((CardFS*)ctx)->do_mount();
}

void 	
cardfs_umount(void *ctx){
//    return ((CardFS*)ctx)->do_umount();
}

int 	
cardfs_readdir(void *ctx, char *dir_name, struct directory *dir){
    return ((CardFS*)ctx)->do_readdir(dir_name, dir);
}

int 	
cardfs_stat(void *ctx, char *name, struct lufs_fattr *fattr){
    return ((CardFS*)ctx)->do_stat(name, fattr);
}

int 	
cardfs_mkdir(void *ctx, char *dir, int mode){
    return ((CardFS*)ctx)->do_mkdir(dir, mode);
}

int 	
cardfs_rmdir(void *ctx, char *dir){
    return ((CardFS*)ctx)->do_rmdir(dir);
}

int 	
cardfs_create(void *ctx, char *file, int mode){
    return ((CardFS*)ctx)->do_create(file, mode);
}

int 	
cardfs_unlink(void *ctx, char *file){
    return ((CardFS*)ctx)->do_unlink(file);
}

int 	
cardfs_rename(void *ctx, char *old_name, char *new_name){
    return ((CardFS*)ctx)->do_rename(old_name, new_name);
}

int 	
cardfs_open(void *ctx, char *file, unsigned mode){
    return ((CardFS*)ctx)->do_open(file, mode);
}

int 	
cardfs_release(void *ctx, char *file){
    return ((CardFS*)ctx)->do_release(file);
}

int 	
cardfs_read(void *ctx, char *file, unsigned long offset, unsigned long count, char *buf){
    return ((CardFS*)ctx)->do_read(file, offset, count, buf);
}

int	
cardfs_write(void *ctx, char *file, unsigned long offset, unsigned long count, char *buf){
    return ((CardFS*)ctx)->do_write(file, offset, count, buf);
}

int 	
cardfs_readlink(void *ctx, char *link, char *buf, int buflen){
    return ((CardFS*)ctx)->do_readlink(link, buf, buflen);
}

int 	
cardfs_link(void *ctx, char *target, char *link){
    return ((CardFS*)ctx)->do_link(target, link);
}

int 	
cardfs_symlink(void *ctx, char *target, char *link){
    return ((CardFS*)ctx)->do_symlink(target, link);
}

int 	
cardfs_setattr(void *ctx, char *file, struct lufs_fattr *fattr){
    return ((CardFS*)ctx)->do_setattr(file, fattr);
}

} /* extern "C" */

CardFS::CardFS(struct list_head *c, struct dir_cache *cache, struct credentials *cred)
{
    cfg = c;
    this->cache = cache;
    this->cred = cred;

    _cid="LUFS:CardFS:"+CTMisc::num2string(getpid());
}

CardFS::~CardFS(){
}

int
CardFS::do_mount(){
    unsigned int pos;
    string tmp;

    try {
        tmp=options.o_host;
        pos=tmp.find("@");
        if (pos!=string::npos) {
            _terminal=tmp.substr(0,pos);
            if (pos+1<tmp.length())
                _host=tmp.substr(pos+1);
        }
        else
            _host=tmp;

        FileClient client(_cid, InetAddress(_host.c_str()),
                          options.o_port,
                          10,
                          TIMEOUT1,
                          TIMEOUT2);
        return client.mount(_terminal,
                            options.o_username,
                            options.o_password);
    }
    catch (Status st) {
        TRACE("Exception: "<<st.statusHistory());
        return 0;
    }
}


void
CardFS::do_umount(){
    FileClient client(_cid, InetAddress(_host.c_str()),
                      options.o_port,
                      10,
                      TIMEOUT1,
                      TIMEOUT2);
    // force unmounting !
    client.unmount(true);
}

struct lufs_fattr CardFS::_entry2fattr(const CTDirEntry &e) {
    struct lufs_fattr fattr;
    int acc;

    acc=0;
    if (e.attributes() & CTDirEntry::Attr_DIR) {
        acc|=S_IFDIR;
        acc|=S_IXUSR;
    }
    else
        acc|=S_IFREG;

    if (e.attributes() & CTDirEntry::Attr_READ) {
        acc|=S_IRUSR;
    }
    if (e.attributes() & CTDirEntry::Attr_WRITE) {
        acc|=S_IWUSR;
    }

    fattr.f_mode = acc;
    fattr.f_nlink = 1;
    fattr.f_uid = getuid();
    fattr.f_gid = getgid();
    fattr.f_size = e.size();
    fattr.f_atime = 0;
    fattr.f_mtime = 0;
    fattr.f_ctime = 0;
    return fattr;
}

int
CardFS::do_readdir(char* d, struct directory *dir){
    struct lufs_fattr fattr;
    int i;
    CTDirEntry entry;
    
    INFO("do_readdir "<<d);
    FileClient client(_cid, InetAddress(_host.c_str()),
                      options.o_port,
                      10,
                      TIMEOUT1,
                      TIMEOUT2);

    // create "." entry
    entry=CTDirEntry(-1,
                     ".",
                     CTDirEntry::Attr_DIR   |
                     CTDirEntry::Attr_READ  |
                     CTDirEntry::Attr_WRITE);
    fattr=_entry2fattr(entry);
    lu_cache_add2dir(dir, (char*)entry.name().c_str(), NULL, &fattr);

    // create ".." entry
    entry=CTDirEntry(-1,
                     "..",
                     CTDirEntry::Attr_DIR   |
                     CTDirEntry::Attr_READ  |
                     CTDirEntry::Attr_WRITE);
    fattr=_entry2fattr(entry);
    lu_cache_add2dir(dir, (char*)entry.name().c_str(), NULL, &fattr);

    i=0;
    while(client.readDir(d,entry,i++)>=0) {
        if (entry.attributes() & CTDirEntry::Attr_USED) {
            fattr=_entry2fattr(entry);
            lu_cache_add2dir(dir, (char*)entry.name().c_str(), NULL, &fattr);
        }
    } // while

    return 0;
}


int
CardFS::do_stat(char *nm, struct lufs_fattr *fattr){
    int result;
    CTDirEntry entry;

    INFO("do_stat "<<nm);
    FileClient client(_cid, InetAddress(_host.c_str()),
                      options.o_port,
                      10,
                      TIMEOUT1,
                      TIMEOUT2);

    result=client.statFile(nm,
                           entry);
    if (result<0)
        return result;
    if (entry.attributes() & CTDirEntry::Attr_USED)
        *fattr=_entry2fattr(entry);
    else
        return -1;
    return 0;
}

unsigned int CardFS::_mode2attribs(int mode) {
    unsigned int att;

    att=0;

    if (mode & S_IRUSR)
        att|=CTDirEntry::Attr_READ;
    if (mode & S_IWUSR)
        att|=CTDirEntry::Attr_WRITE;
    if (mode & S_IFDIR)
        att|=CTDirEntry::Attr_DIR;
    att|=CTDirEntry::Attr_USED;
    return att;
}

int CardFS::do_mkdir(char *d, int mode){
    int result;

    FileClient client(_cid, InetAddress(_host.c_str()),
                      options.o_port,
                      10,
                      TIMEOUT1,
                      TIMEOUT2);
    result=client.mkDir(d,_mode2attribs(mode));

    return result;
}

int
CardFS::do_rmdir(char *d){
    int result;

    FileClient client(_cid, InetAddress(_host.c_str()),
                      options.o_port,
                      10,
                      TIMEOUT1,
                      TIMEOUT2);
    result=client.rmDir(d);
    return result;
}

int
CardFS::do_create(char *fn, int mode){
    int result;

    INFO("CardFS::do_create "<<fn);
    FileClient client(_cid, InetAddress(_host.c_str()),
                      options.o_port,
                      10,
                      TIMEOUT1,
                      TIMEOUT2);
    result=client.createFile(fn,_mode2attribs(mode));
    return result;
}


int
CardFS::do_unlink(char *fn){
    int result;

    FileClient client(_cid, InetAddress(_host.c_str()),
                      options.o_port,
                      10,
                      TIMEOUT1,
                      TIMEOUT2);
    result=client.unlinkFile(fn);
    return result;
}

int
CardFS::do_rename(char *old, char *nnew){
    int result;

    INFO("do_rename "<<old<<"->"<<nnew);

    FileClient client(_cid, InetAddress(_host.c_str()),
                      options.o_port,
                      10,
                      TIMEOUT1,
                      TIMEOUT2);
    result=client.renameFile(old,nnew);

    return result;
}

int
CardFS::do_open(char *fn, unsigned mode){
    struct lufs_fattr fattr;
    INFO("do_open "<<fn);

    return do_stat(fn,&fattr);
}


int
CardFS::do_release(char *file){
    INFO("do_release "<<file);

    return 1;
}

int
CardFS::do_read(char *fn, unsigned long offset, unsigned long count, char *buf){
    int result;
    string tmp;

    INFO("do_read "<<fn);
    FileClient client(_cid, InetAddress(_host.c_str()),
                      options.o_port,
                      10,
                      TIMEOUT1,
                      TIMEOUT2);

    result=client.readFile(fn,
                           offset,
                           count,
                           tmp);
    if (result<0)
        return -1;
    memmove(buf,tmp.c_str(),tmp.length());
    return result;
}

int
CardFS::do_write(char *fn, unsigned long offset, unsigned long count, char *buf){
    FileClient client(_cid, InetAddress(_host.c_str()),
                      options.o_port,
                      10,
                      TIMEOUT1,
                      TIMEOUT2);
    return client.writeFile(fn,offset,count,buf);
}

int
CardFS::do_readlink(char *link, char *buf, int buflen){
    INFO("CardFS::do_readlink "<<link);
    return -1;
}

int
CardFS::do_link(char *old, char *nnew){
    INFO("CardFS::do_link "<<old<<"->"<<nnew);
    return -1;
}

int CardFS::do_symlink(char *old, char *nnew){
    INFO("CardFS::do_symlink "<<old<<"->"<<nnew);
    return -1;
}

int CardFS::do_setattr(char *f, struct lufs_fattr *fattr){

    INFO("CardFS::do_setattr "<<f<<"(ignored)");

    return 0;
}


