/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>

#include <netdude/nd.h>
#include <netdude/nd_capture.h>
#include <netdude/nd_debug.h>
#include <netdude/nd_dialog.h>
#include <netdude/nd_macros.h>
#include <netdude/nd_misc.h>
#include <netdude/nd_tcpdump.h>
#include <netdude/nd_trace_registry.h>
#include "interface.h"
#include "support.h"


static void capture_packet_cb(ND_Capture *capt, gint fd, GdkInputCondition condition);
static void capture_pcap_read_handler(u_char *data, const struct pcap_pkthdr *h,
				      const u_char *pdata);

static char *
capture_read_filter_file(const char *filename)
{
  char *content = NULL;
  int fd;
  struct stat st;

  if (!filename || filename[0] == '\0')
    return NULL;

  if ( (fd = open(filename, O_RDONLY)) < 0)
    return NULL;

  if (fstat(fd, &st) < 0)
    goto exit;

  content = malloc(st.st_size + 1);

  if (read(fd, content, st.st_size) < 0)
    {
      g_free(content);
      content = NULL;
      goto exit;
    }

  content[st.st_size] = '\0';

 exit:
  close(fd);
  return content;
}

static void
capture_update_widget(GtkWidget *widget, GtkWidget *toggle, gboolean enable_if_toggled)
{
  gboolean enable;

  enable = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle));
  enable = (enable_if_toggled ? enable : !enable);
 
  gtk_widget_set_sensitive(widget, enable);
}


ND_Capture *
nd_capture_new(void)
{
  ND_Capture *capt;

  capt = g_new0(ND_Capture, 1);
  D_ASSERT_PTR(capt);
  if (!capt)
    return NULL;

  return capt;
}


void 
nd_capture_free(ND_Capture *capt)
{
  if (!capt)
    return;

  D_ENTER;
  nd_capture_stop(capt);

  g_free(capt->ifname);
  g_free(capt->filter);
  g_free(capt);
  D_RETURN;
}


GtkWidget *
nd_capture_dialog_show(void)
{
  GtkWidget *dialog = NULL;
  GtkWidget *w, *w2;
  GList *ifs;

  if ( (ifs = nd_misc_get_if_names(IFF_UP)) == NULL)
    return NULL;

  dialog = create_capture_dialog();

  /* Populate interface combobox with currently up interfaces: */
  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_ifs_combo");
  D_ASSERT_PTR(w);  
  gtk_combo_set_popdown_strings(GTK_COMBO(w), ifs);
  g_list_foreach(ifs, (GFunc)g_free, NULL);
  g_list_free(ifs);

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_if_prom_checkbutton");
  D_ASSERT_PTR(w);  
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);

  /* Enable/Disble widget according to checkbox settings: */

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_size_lim_checkbutton");
  D_ASSERT_PTR(w);  
  w2 = gtk_object_get_data(GTK_OBJECT(dialog), "capt_size_spinbutton");
  D_ASSERT_PTR(w2);
  capture_update_widget(w2, w, TRUE);

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_filter_file_checkbutton");
  D_ASSERT_PTR(w);  
  w2 = gtk_object_get_data(GTK_OBJECT(dialog), "capt_filter_file_combo");
  D_ASSERT_PTR(w2);
  capture_update_widget(w2, w, TRUE);
  w2 = gtk_object_get_data(GTK_OBJECT(dialog), "capt_filter_file_button");
  D_ASSERT_PTR(w2);
  capture_update_widget(w2, w, TRUE);

  w2 = gtk_object_get_data(GTK_OBJECT(dialog), "capt_filter_entry");
  D_ASSERT_PTR(w2);
  capture_update_widget(w2, w, FALSE);

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_stop_count_checkbutton");
  D_ASSERT_PTR(w);  
  w2 = gtk_object_get_data(GTK_OBJECT(dialog), "capt_stop_count_spinbutton");
  D_ASSERT_PTR(w2);
  capture_update_widget(w2, w, TRUE);

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_stop_size_checkbutton");
  D_ASSERT_PTR(w);  
  w2 = gtk_object_get_data(GTK_OBJECT(dialog), "capt_stop_size_spinbutton");
  D_ASSERT_PTR(w2);
  capture_update_widget(w2, w, TRUE);

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_stop_time_checkbutton");
  D_ASSERT_PTR(w);  
  w2 = gtk_object_get_data(GTK_OBJECT(dialog), "capt_stop_time_spinbutton");
  D_ASSERT_PTR(w2);
  capture_update_widget(w2, w, TRUE);

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_limit_checkbutton");
  D_ASSERT_PTR(w);  
  w2 = gtk_object_get_data(GTK_OBJECT(dialog), "capt_limit_spinbutton");
  D_ASSERT_PTR(w2);
  capture_update_widget(w2, w, TRUE);

  /* Disable "stop capture" button at the beginning */
  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_stop_button");
  D_ASSERT_PTR(w);  
  gtk_widget_set_sensitive(w, FALSE);

  gtk_widget_show(dialog);

  return dialog;
}


void 
nd_capture_dialog_close(GtkWidget *dialog)
{
  gtk_widget_destroy(dialog);
}


void 
nd_capture_dialog_start(GtkWidget *dialog)
{
  ND_Capture *capt;
  GtkWidget *w;

  capt = nd_capture_new();

  /* Fill in capture structure according to current GUI settings: */

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_if_combo_entry");
  D_ASSERT_PTR(w);
  capt->ifname = g_strdup(gtk_entry_get_text(GTK_ENTRY(w)));

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_size_spinbutton");
  D_ASSERT_PTR(w);
  if (GTK_WIDGET_SENSITIVE(w))
    capt->snap_len = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_if_prom_checkbutton");
  D_ASSERT_PTR(w);
  capt->set_promiscuous = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));

  /* Set filter, if applicable: */
  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_filter_entry");
  D_ASSERT_PTR(w);
  if (GTK_WIDGET_SENSITIVE(w))
    capt->filter = g_strdup(gtk_entry_get_text(GTK_ENTRY(w)));

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_filter_file_combo");
  D_ASSERT_PTR(w);
  if (GTK_WIDGET_SENSITIVE(w))
    capt->filter = capture_read_filter_file(gtk_entry_get_text(GTK_ENTRY(w)));
  
  /* Set limiting factors if applicable: */
  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_stop_count_spinbutton");
  D_ASSERT_PTR(w);
  if (GTK_WIDGET_SENSITIVE(w))
    capt->packet_limit = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_stop_size_spinbutton");
  D_ASSERT_PTR(w);
  if (GTK_WIDGET_SENSITIVE(w))
    capt->size_limit = 1024 * gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_stop_time_spinbutton");
  D_ASSERT_PTR(w);
  if (GTK_WIDGET_SENSITIVE(w))
    capt->time_limit = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));

  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_limit_spinbutton");
  D_ASSERT_PTR(w);
  if (GTK_WIDGET_SENSITIVE(w))
    capt->drop_limit = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w));

  /* Enable "stop capture" button */
  w = gtk_object_get_data(GTK_OBJECT(dialog), "capt_stop_button");
  D_ASSERT_PTR(w);  
  gtk_widget_set_sensitive(w, TRUE);

  gtk_object_set_data_full(GTK_OBJECT(dialog), "capture", capt,
			   (GdkDestroyNotify) nd_capture_free);

  capt->dialog = dialog;
			   
  /* And start capture: */
  nd_capture_start(capt);
}


static void
capture_choose_filter_file_cb(const char *filename,
			      void *user_data)
{
  GtkWidget *entry;
  
  if (nd_misc_can_read(filename))
    {
      nd_dialog_filesel_close();
      entry = gtk_object_get_data(GTK_OBJECT(user_data), "capt_filter_combo_entry");
      gtk_entry_set_text(GTK_ENTRY(entry), filename);
    }
}


void 
nd_capture_choose_filter_file(GtkWidget *dialog)
{
  nd_dialog_filesel(_("Choose a filter definition file"), NULL,
		    capture_choose_filter_file_cb, dialog);
}


static void
capture_sync_dialog(ND_Capture *capt)
{
  struct pcap_stat stats;
  char s[128];
  GtkWidget *w;

  pcap_stats(capt->trace->pcap, &stats);

  g_snprintf(s, 128, "%u", stats.ps_recv);
  w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "stats_recv_label");
  D_ASSERT_PTR(w);
  gtk_label_set_text(GTK_LABEL(w), s);

  g_snprintf(s, 128, "%u", stats.ps_drop);
  w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "stats_dropped_label");
  D_ASSERT_PTR(w);
  gtk_label_set_text(GTK_LABEL(w), s);

  g_snprintf(s, 128, "%u", capt->size_cur);
  w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "stats_bytes_label");
  D_ASSERT_PTR(w);
  gtk_label_set_text(GTK_LABEL(w), s);

  g_snprintf(s, 128, "%lu s", capt->time_cur - capt->time_start);
  w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "stats_time_label");
  D_ASSERT_PTR(w);
  gtk_label_set_text(GTK_LABEL(w), s);

  g_snprintf(s, 128, "%.2f", stats.ps_recv / (float) (capt->time_cur - capt->time_start));
  w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "stats_per_sec_label");
  D_ASSERT_PTR(w);
  gtk_label_set_text(GTK_LABEL(w), s);

  g_snprintf(s, 128, "%.2f bytes", capt->size_cur / (float) stats.ps_recv);
  w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "stats_avg_size_label");
  D_ASSERT_PTR(w);
  gtk_label_set_text(GTK_LABEL(w), s);
}


static gint 
capture_timeout_cb(ND_Capture *capt)
{
  char s[128];
  GtkWidget *w;

  capt->time_cur = time(NULL);

  if (capt->dialog)
    {
      g_snprintf(s, 128, "%lu s", capt->time_cur - capt->time_start);
      w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "stats_time_label");
      D_ASSERT_PTR(w);
      gtk_label_set_text(GTK_LABEL(w), s);
    }

  if (capt->time_limit && (capt->time_start + capt->time_limit <= capt->time_cur))
    nd_capture_stop(capt);

  return TRUE;
}


static void
capture_packet_cb(ND_Capture *capt, gint fd, GdkInputCondition condition)
{
  gdk_input_remove(capt->gdk_tag);

  pcap_loop(capt->trace->pcap, 1,
	    capture_pcap_read_handler,
	    (u_char*) capt);  

  return;
  TOUCH(fd);
  TOUCH(condition);
}


static void
capture_pcap_read_handler(u_char *data, const struct pcap_pkthdr *h,
			  const u_char *pdata)
{
  ND_Capture *capt = (ND_Capture *) data;

  nd_packet_pcap_read_handler_gui((u_char *) capt->trace,
				  h, pdata);

  /* Update statistics, so that we know when to stop! */

  capt->packet_cur++;
  capt->size_cur += h->caplen;
  capt->time_cur = time(NULL);

  if ((capt->drop_limit > 0) && (capt->packet_cur > capt->drop_limit))
    nd_packet_delete(capt->trace->pl, TRUE);

  if (capt->dialog)
    capture_sync_dialog(capt);

  /* When we received something, this trace is worth saving. */
  nd_trace_set_dirty(capt->trace, TRUE);
  nd_trace_goto_bottom(capt->trace);

  /* If any abort condition becomes true, reset buttons and 
   * return, not listening to the socket any longer.
   */
  if ((capt->packet_limit && capt->packet_cur >= capt->packet_limit) ||
      (capt->size_limit && capt->size_cur >= capt->size_limit)       ||
      (capt->time_limit && (capt->time_start + capt->time_limit <= capt->time_cur)))
    {
      if (capt->dialog)
	{
	  GtkWidget *w;
	  
	  w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "capt_stop_button");
	  D_ASSERT_PTR(w);  
	  gtk_widget_set_sensitive(w, FALSE);
	  
	  w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "capt_start_button");
	  D_ASSERT_PTR(w);  
	  gtk_widget_set_sensitive(w, TRUE);
	}
      
      return;
    }

  /* We've handled what was available, now listen again */
  capt->gdk_tag = gdk_input_add(pcap_fileno(capt->trace->pcap), GDK_INPUT_READ,
				(GdkInputFunction) capture_packet_cb, capt);

  return;
}


void 
nd_capture_start(ND_Capture *capt)
{
  char      message[MAXPATHLEN];
  char      errbuf[PCAP_ERRBUF_SIZE];
  ND_Trace *trace;
  pcap_t   *pcap;
  struct    bpf_program filter;
  bpf_u_int32 netmask, localnet;

  if (!capt)
    return;

  if (capt->snap_len == 0)
    capt->snap_len = nd_misc_get_if_mtu(capt->ifname);

  D(("Capture on %s with snaplen of %i, prom: %i\n", capt->ifname, capt->snap_len, capt->set_promiscuous));

  if ( (pcap = pcap_open_live(capt->ifname, capt->snap_len,
			    capt->set_promiscuous, 0, errbuf)) == NULL)
    {
      g_snprintf(message, MAXPATHLEN, 
		 _("Cannot listen to the device using pcap.\n"
		   "The error message is '%s'.\n"),
		 errbuf);
      
      nd_dialog_message(_("Pcap problem."), message, TRUE);
      return;
    }

  trace = nd_trace_new(NULL);
  D_ASSERT_PTR(trace);

  capt->trace = trace;
  capt->trace->pcap = pcap;

  nd_trace_init_header(trace, capt->snap_len, pcap_datalink(pcap));

  if (capt->filter)
    {
      D(("Installing filter.\n"));

      if (pcap_lookupnet(capt->ifname, &localnet, &netmask, errbuf) < 0)
	{
	  localnet = netmask = 0;
	  D(("tcpdump: %s\n", errbuf));
	}
      
      if (pcap_compile(pcap, &filter, capt->filter, TRUE, netmask) < 0)
	{
	  g_snprintf(message, MAXPATHLEN, 
		     _("Invalid filter expression. The error message is\n"
		       "'%s'.\n"),
		     pcap_geterr(pcap));
	  
	  nd_dialog_message(_("Pcap problem."), message, TRUE);
	  goto error;
	}

      if (pcap_setfilter(pcap, &filter) < 0)
	{
	  g_snprintf(message, MAXPATHLEN, 
		     _("An error occurred when applying the filter.\n"
		       "The error message is '%s'.\n"),
		     pcap_geterr(pcap));
	  
	  nd_dialog_message(_("Pcap problem."), message, TRUE);
	  goto error;
	}
      
      pcap_freecode(&filter);
    }

  nd_trace_registry_add(trace);

  if (capt->dialog)
    {
      GtkWidget *w;

      w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "capt_start_button");
      D_ASSERT_PTR(w);  
      gtk_widget_set_sensitive(w, FALSE);

      w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "capt_stop_button");
      D_ASSERT_PTR(w);  
      gtk_widget_set_sensitive(w, TRUE);
    }

  capt->time_start = time(NULL);
  capt->finished = FALSE;
  
  capt->gtk_timeout = gtk_timeout_add(1000,
				      (GtkFunction) capture_timeout_cb,
				      capt);

  capt->gdk_tag = gdk_input_add(pcap_fileno(pcap), GDK_INPUT_READ,
				(GdkInputFunction) capture_packet_cb, capt);
  return;
  
 error:
  nd_trace_free(trace);
}


void 
nd_capture_stop(ND_Capture *capt)
{
  if (!capt || capt->finished)
    return;

  capt->finished = TRUE;

  if (capt->dialog)
    {
      GtkWidget *w;
      
      w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "capt_stop_button");
      D_ASSERT_PTR(w);  
      gtk_widget_set_sensitive(w, FALSE);
      
      w = gtk_object_get_data(GTK_OBJECT(capt->dialog), "capt_start_button");
      D_ASSERT_PTR(w);  
      gtk_widget_set_sensitive(w, TRUE);
    }

  if (capt->gtk_timeout > 0)
    {
      gtk_timeout_remove(capt->gtk_timeout);
      capt->gtk_timeout = 0;
    }

  if (capt->gdk_tag > 0)
    gdk_input_remove(capt->gdk_tag);

  nd_trace_init_packets(capt->trace);
}


gboolean 
nd_capture_possible(void)
{
  pcap_t *pcap;
  GList *ifs, *l;
  char *ifname;
  char  errbuf[PCAP_ERRBUF_SIZE];
  gboolean result = TRUE;

  if ( (ifs = nd_misc_get_if_names(IFF_UP)) == NULL)
    return FALSE;

  for (l = ifs; l; l = g_list_next(l))
    {
      ifname = (char *) l->data;

      if ( (pcap = pcap_open_live(ifname, 0, 0, 0, errbuf)))
	{
	  pcap_close(pcap);
	  continue;
	}

      result = FALSE;
      break;
    }

  g_list_foreach(ifs, (GFunc)g_free, NULL);
  g_list_free(ifs);

  return result;
}

