/*
 * example_ia64_smpl.c - example of a sampling output module
 *
 * Copyright (C) 2002-2003 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux/ia64.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */


/*
 * include pfmon main header file. In this file we find the definition
 * of what a sampling output format must provide
 */
#include "pfmon.h"

/*
 * this example is valid for a generic IA-64 PMU, so we need to include
 * to IA-64 generic support files
 */
#include <perfmon/pfmlib_generic_ia64.h>
#include <perfmon/perfmon_default_smpl.h>

/*
 * the name of the output format.
 *
 * You must make sure it is unique and hints the user as to which the format does
 */
#define SMPL_MOD_NAME	"example-ia64"

#define CHECK_VERSION(h)	(PFM_VERSION_MAJOR((h)) != PFM_VERSION_MAJOR(PFM_DEFAULT_SMPL_VERSION))

/*
 * default number of entries in the sampling buffer
 */
#define EXAMPLE_DFL_SMPL_ENTRIES	2048UL

/*
 * contains the module-specific values chosen on the command line
 */
typedef struct {
	unsigned long smpl_entries; /* requested number of entries in the sampling buffer */
} example_options_t;

static example_options_t example_options;

/*
 * forward declaration
 */
pfmon_smpl_module_t example_ia64_smpl_module;

/*
 * The routine which processes the sampling buffer when an overflow if notified to pfmon
 *
 * Argument: a point to a pfmon_smpl_desc_t structure. 
 *  This structures contains:
 * 	- the file descriptor to use for printing (smpl_fp)
 * 	- the address of the sampling buffer (smpl_hdr)
 * 	- the cpu we are sampling on for system-wide mode (cpu)
 * 	- a counter to count the number of entries processed so far (entry_count)
 * 	- a point to a counter in case of aggregation, entry_count not used for aggregation mode (aggr_count)
 * 	- the opaque descriptor to the symbol has table, when resolve-address is used (hash_desc)
 *
 * IMPORTANT:
 * 	the function MUST be thread-safe as it can be called from several pfmon thread at the same time.
 *
 * Return:
 * 	 0: success
 * 	-1: error
 */
static int
example_process_samples(pfmon_smpl_desc_t *csmpl)
{	
	pfm_default_smpl_hdr_t *hdr;
	pfm_default_smpl_entry_t *ent;
	FILE *fp = csmpl->smpl_fp;
	pfm_gen_ia64_pmd_reg_t *reg;
	unsigned long msk, entry, i;
	uint64_t count;
	unsigned int ovfl_pmd, j;
	int ret = 0;

	/*
	 * create a shortcut to the buffer header
	 */
	hdr = csmpl->smpl_hdr;

	/*
	 * point to the first entry in the buffer
	 */
	ent = (pfm_default_smpl_entry_t *)(hdr+1);
	/*
	 * how many entries we have to process
	 */
	count = hdr->hdr_count;

	/*
	 * depending on the aggregation requested by user, we need to adjust the index
	 * of the entries we are going to write
	 *
	 * Even in non-aggregation mode, we keep a total number of samples saved for reference.
	 */
	entry = options.opt_aggr ? *csmpl->aggr_count : csmpl->entry_count;

	/*
	 * how many got recorded in the buffer. the buffer may not necessarily be full
	 */
	fprintf(fp, "entries recorded=%lu\n", count);

	/* 
	 * print the raw value of each PMD in the body of each entry
	 */
	for(i=0; i < count; i++) {
		
		/*
		 * extract which counter overflowed for this entry
		 */
		ovfl_pmd = ent->ovfl_pmd;

		/*
		 * let's print an entry header
		 */
		fprintf(fp, "entry %ld PID:%d CPU:%d STAMP:0x%lx IIP:0x%016lx\n",
			entry,
			ent->pid,
			ent->cpu,
			ent->tstamp,
			ent->ip);

		/*
		 * position just after the entry header
		 */
		reg = (pfm_gen_ia64_pmd_reg_t *)(ent+1);

		/*
		 * the perfmon-2 default sampling buffer format supports variable size entries.
		 * when multiple counters overflow at the same time, one entry is created for
		 * each overflowed counter. To figure out what is contained in the body of an
		 * entry we can use the reverse sampled PMD table provided by pfmon. It can
		 * be indexed by the PMD which overflowed for the entry, it returns a bitmask
		 * of the PMDs recoreded in the body of the entry.
		 */
		msk = options.rev_smpl_pmds[ovfl_pmd];

		/*
		 * now simply print each of the PMD in the body of the entry
		 */
		for(j=0; msk; msk >>=1, j++) {	
			/*
			 * skip the PMD that are not recorded
			 */
			if ((msk & 0x1) == 0) continue;

			ret = fprintf(fp, "PMD%-2u: 0x%016lx\n", j, reg->pmd_val);

			/*
			 * PMDs are stored contiguously in increasing index number
			 */
			reg++;
		}
		/* 
		 * abort if we cannot produce output (such as when the disk is full)
		 */
		if (ret == -1) goto error;

		/*
		 * entries are contiguously stored
		 */
		ent  = (pfm_default_smpl_entry_t *)reg;	
		entry++;
	}
	/*
	 * when aggregation is used, for are guaranteed sequential access to
	 * this routine by higher level lock
	 */
	if (options.opt_aggr) {
		*csmpl->aggr_count += count;
	} else {
		csmpl->entry_count += count;
	}
	/*
	 * keep track of the number of samples we just processed
	 */
	csmpl->last_count = count;

	return 0;
error:
	/*
	 * you cannot quit reason but you can print a warning
	 * and you must return -1
	 */
	warning("cannot write to sampling file: %s\n", strerror(errno));
	/* not reached */
	return -1;
}
/*
 * this routine is invoked prior to lauching the measurement. it can be used
 * by the module to verify that the kernel sampling buffer format uses 
 * a version that is understood by the module.
 *
 * return:
 * 	0 : version is ok
 * 	-1: version mismatch
 */
static int
example_check_version(pfmon_smpl_desc_t *csmpl)
{
	pfm_default_smpl_hdr_t *hdr; 

	/*
	 * extract buffer header pointer
	 */
	hdr   = csmpl->smpl_hdr;

	/*
	 * check that version is as expected
	 */
	if (CHECK_VERSION(hdr->hdr_version)) {
		warning("format %s expect format v%u.x not v%u.%u %u\n", 
				SMPL_MOD_NAME,
				PFM_VERSION_MAJOR(PFM_DEFAULT_SMPL_VERSION),
				PFM_VERSION_MAJOR(hdr->hdr_version),
				PFM_VERSION_MINOR(hdr->hdr_version), hdr->hdr_version);
		return -1;
	}
	return 0;
}

/*
 * routine called prior to process_samples to verify that we are not processing
 * the same set of samples twice. This can happen at the end of a session. pfmon
 * will check if new samples are available. In this case, the buffer may not
 * be full.
 * The way to check that the buffer contains new samples depends on the sampling
 * buffer format used. In this example, we use the default format which includes
 * a overflow counter in its header. The counter is increased each time the buffer
 * overflows. Hence, it no new samples are available, the buffer will have the 
 * same value for this counter.
 *
 * Return:
 * 	0: we have new samples
 *     -1: no new samples 
 */
static int
example_check_new_samples(pfmon_smpl_desc_t *csmpl)
{
	pfm_default_smpl_hdr_t *hdr; 
	uint64_t last_ovfl;

	/*
	 * extract buffer header pointer
	 */
	hdr       = csmpl->smpl_hdr;
	last_ovfl = csmpl->last_ovfl;

	/*
	 * the pfmon_smpl_desc_t structure contains a field which is initialized to ~0UL (infinite) 
	 * by pfmon. The module can store the latest value of the header counter in that field
	 * to keep track of the number of overflows.
	 *
	 * If the buffer reports an overflow count that is greater than the last one we recorded, 
	 * it means we have new samples. The second part of the test is used the first time we get
	 * called.
	 *
	 * The third part of the test ensure we collect the last samples by checking that the number
	 * of entries contained in the buffer is not identical to what was last processed.
	 */
	if (hdr->hdr_overflows <= last_ovfl && last_ovfl != ~0UL && hdr->hdr_count == csmpl->last_count) {

		vbprintf("skipping identical set of samples %lu <= %lu\n", hdr->hdr_overflows, csmpl->last_ovfl); 

		/*
		 * no new samples to process
		 */
		return -1;
	}
	/*
	 * record the new overflow counter value
	 */
	csmpl->last_ovfl = hdr->hdr_overflows;

	/*
	 * we have new samples to process
	 */
	return 0;
}

/*
 * pfmon supports sampling module-specific option on the command line.
 * When the module is selected, the options are added to the other pfmon
 * options. This means that you ALWAYS need to specify the module name PRIOR
 * to the module options if you want to use them. For instance, with this 
 * example, you would type:
 * 	$ pfmon --smpl-module=example-ia64 --smpl-entries=2048 ....
 *
 * Module specific options MUST use option indexes between 500 and 600 (non-inclusive)
 */

/*
 * 000-255   reserved for generic options
 * 400-499   reserved for PMU specific options
 * 500-599   reserved for format specific options
 */
static struct option example_cmd_options[]={
	{ "smpl-entries", 1, 0, 500},
	{ NULL, 0, 0, 0}
};

/*
 * If the module has specific options, the user MUST provide a way for them to be listed
 * by pfmon when the user needs help (pfmon --help).
 * 
 * This routine prints a SINGLE line per option and MUST be preceeded by \t.
 */
static void
example_show_options(void)
{
	printf("\t--smpl-entries=val\t\tset number of entries for sampling buffer (default %lu)\n", EXAMPLE_DFL_SMPL_ENTRIES);
}

/*
 * This routine MUST be provided in case the module has specific options. It is invoke 
 * when pfmon does not recognize an option.
 *
 * Return:
 *    0: means we understand the option
 *   -1: unknown option
 */
static int
example_parse_options(int code, char *optarg, pfmon_lib_param_t *evt)
{
	char *endptr = NULL;

	switch(code) {
		case  500:
			/*
			 * make sure the user does not specify the size twice
			 */
			if (example_options.smpl_entries != EXAMPLE_DFL_SMPL_ENTRIES) 
				fatal_error("smpl-entries already defined\n");

			/*
			 * keep track of the requested size
			 */
			example_options.smpl_entries = strtoul(optarg, &endptr, 0);

			if (*endptr != '\0') 
				fatal_error("invalid number of entries: %s\n", optarg);
			break;
		default:
			/*
			 * the option is not ours
			 */
			return -1;
	}
	/*
	 * the option was ours
	 */
	return 0;

}

/*
 * When the module uses a kernel sampling format that does have options, this function
 * MUST be provided in order for pfmon to allocated enough memory to pass the options
 * along with the pfm_context_t structure when the perfmon context is created
 *
 * Return: size of the kernel sampling format specific argument
 * 	
 */
static size_t
example_get_fmt_arg_size(void)
{
	return sizeof(pfm_default_smpl_ctx_arg_t);
}

/*
 * this module has chosen to use the number of entries as a parameter to determine the
 * size of the sampling buffer. It could have been just a byte size which is what the kernel
 * is expecting. So here we try to figure out the size of the buffer using the number of 
 * entries. We try to approximate the size to make we can at least store the number of entries
 * requested by the user.
 * Arguments:
 * 	             arg: a point to the area reserved for the kernel sampling format specific arguments
 * 	max_pmds_sample : the largest number of PMDS recorded in an entry
 * Return:
 * 	0: success
 *     -1: error
 */
static int
example_initialize_ctx_arg(pfmon_ctx_arg_t *arg, unsigned int max_pmds_sample)
{
#define MAX_PMD_COUNT	 	64
#define FUDGE_FACTOR	 	(sizeof(pfm_default_smpl_entry_t)+(sizeof(unsigned long)*MAX_PMD_COUNT))
#define ENTRY_SIZE(npmd)	(sizeof(pfm_default_smpl_entry_t)+((npmd)*sizeof(unsigned long)))
	pfm_default_smpl_ctx_arg_t *ctx_arg = (pfm_default_smpl_ctx_arg_t *)arg;
	unsigned long entry_size;

	entry_size = ENTRY_SIZE(max_pmds_sample);
	
	/*
	 * the fudge factor allows us to get exactly the number of entries specified by the
	 * user (or the default). The kernel module is such that it will consider the buffer
	 * full if less than PFM_DEFAULT_MAX_ENTRY_SIZE bytes are left in the buffer. Any
	 * entry size is <= PFM_DEFAULT_MAX_ENTRY_SIZE, therefore we will always record
	 * less than the specified number unless we increase the buffer size slightly.
	 */
	ctx_arg->buf_arg.buf_size = sizeof(pfm_default_smpl_hdr_t)
		                  + (FUDGE_FACTOR-entry_size)
				  + example_options.smpl_entries*entry_size;

	/*
	 * copy the uuid of the format we are using
	 */
	memcpy(ctx_arg->ctx_arg.ctx_smpl_buf_id, example_ia64_smpl_module.uuid, sizeof(pfm_uuid_t));

	vbprintf("min buffer entries : %lu max PMD/entry=%u\n", example_options.smpl_entries, max_pmds_sample);

	DPRINT(("max_pmds_sample=%u buf_size=%lu fudge=%lu buffer_header=%lu entry_header=%lu (max)entry_size=%lu\n", 
		max_pmds_sample, 
		ctx_arg->buf_arg.buf_size, 
		FUDGE_FACTOR, 
		sizeof(pfm_default_smpl_hdr_t), 
		sizeof(pfm_default_smpl_entry_t), 
		entry_size));

	return 0;
}

/*
 * this function is invoked once to let the module initialize itself.
 * This is where the module MUST register its options, if it has some.
 *
 * Return:
 * 	0: success
 *     -1: error
 */
static int
example_initialize_module(void)
{
	example_options.smpl_entries = EXAMPLE_DFL_SMPL_ENTRIES;

	return pfmon_register_smpl_mod_options(example_cmd_options, sizeof(example_cmd_options));
}

/*
 * Print explanation about the format of sampling output. This function is optional.
 * The output of this function becomes visible only when the --with-header option
 * is specified. The output is placed after the standard pfmon header.
 *
 * Argument: a point to a pfmon_smpl_desc_t structure. 
 *  This structures contains:
 * 	- the file descriptor to use for printing (smpl_fp)
 * 	- the address of the sampling buffer (smpl_hdr)
 * 	- the cpu we are sampling on for system-wide mode (cpu)
 * 	- a counter to count the number of entries processed so far (entry_count)
 * 	- a point to a counter in case of aggregation, entry_count not used for aggregation mode (aggr_count)
 * 	- the opaque descriptor to the symbol has table, when resolve-address is used (hash_desc)
 *
 * Return:
 * 	 0: if successful
 * 	-1: otherwise
 */
static int
example_print_header(pfmon_smpl_desc_t *csmpl)
{
	FILE *fp = csmpl->smpl_fp;
	int ret;

	fprintf(fp, "# using example-ia64 sampling output format\n");
	ret = fprintf(fp, "# min sampling buffer entries: %lu\n", example_options.smpl_entries);

	return ret > -1 ? 0 : -1;
}

/*
 * structure describing the format which is visible to pfmon_smpl.c
 *
 * The structure MUST be manually added to the smpl_outputs[] table in
 * pfmon_smpl.c
 */
pfmon_smpl_module_t example_ia64_smpl_module ={
	/*
	 * the name of the module
	 */
	.name		   = SMPL_MOD_NAME,
	/* 
	 * Because some formats can be used with more than one PMU models, pmu_mask 
	 * is a bitfield. Each bit represents a PMU model. At least
	 * one bit must be set. use PFMON_PMU_MASK(XX) where XX is the PFMLIB_XX_PMU
	 * from pfmlib.h.
	 */
	.pmu_mask	   = PFMON_PMU_MASK(PFMLIB_GENERIC_IA64_PMU),
	/*
	 * a small description
	 */
	.description	   =  "Sampling output example. Any IA-64 PMU models",
	/*
	 * check kernel sampling format version
	 */
	.check_version	   = example_check_version,
	/*
	 * process the samples
	 */
	.process_samples   = example_process_samples,
	/*
	 * print a format-specific header.
	 */
	.print_header	   = example_print_header,
	/*
	 * initialize the module
	 */
	.initialize_module = example_initialize_module,
	 /*
	  * parse the module-specific options
	  */
	.parse_options     = example_parse_options,
	/*
	 * show module-specific options
	 */
	.show_options      = example_show_options,
	/*
	 * get kernel sampling format specific argument size
	 */
	.get_fmt_arg_size  = example_get_fmt_arg_size,
	/*
	 * routine to initialize the kernel sampling format specific arguments
	 */
	.initialize_ctx_arg= example_initialize_ctx_arg,
	/*
	 * check new samples to avoid double reporting
	 */
	.check_new_samples  = example_check_new_samples,
	/*
	 * the UUID of the kernel sampling format use by this module
	 */
	.uuid		   = PFM_DEFAULT_SMPL_UUID,
};
