/*
 * Resource manager client functions
 *
 * Copyright (C) 2001-2002, Olaf Kirch <okir@lst.de>
 */

#include <string.h>
#include <syslog.h>
#include <stdarg.h>
#include <errno.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdlib.h>
#include "protocol.h"
#include "resmgr.h"

static int	sane(const char *);
static int	wildmatch(const char *, const char *);


static int
rsm_command(int *code, const char *fmt, ...)
{
	struct conn	*conn;
	va_list		ap;
	int		fd = -1, lcode;

	if (!code)
		code = &lcode;
	*code = 0;

	va_start(ap, fmt);
	if (!(conn = rsm_connect(_PATH_RESMGR_SOCKET))
	 || rsm_vprintf(conn, fmt, ap) < 0
	 || (*code = rsm_recv_response(conn)) < 0) {
		syslog(LOG_NOTICE, "resmgr: communication failure: %m");
		goto out;
	}
	if (*code != MSG_OKAY) {
		syslog(LOG_DEBUG, "resmgr: server response code %d", *code);
		goto out;
	}
	if ((fd = conn->passfd) >= 0)
		conn->passfd = -1;
	else
		fd = 0;

out:	rsm_close(conn);
	va_end(ap);
	return fd;
}

char **
rsm_list_devices(const char *wildcard)
{
	struct conn	*conn;
	char		family[128], *s;
	char		buffer[512], *sp, *np, **result;
	int		count = 0, code;

	result = NULL;

	family[0] = '\0';
	if (wildcard && wildcard[0] != '/') {
		strncpy(family, wildcard, sizeof(family)-1);
		family[sizeof(family)-1] = '\0';
		if ((s = strchr(family, ':')) != NULL)
			*s++ = '\0';
		wildcard = s;
	}

	if (!(conn = rsm_connect(_PATH_RESMGR_SOCKET))
	 || rsm_printf(conn, "list %s", family) < 0)
		goto error;

	do {
		if (rsm_recv(conn, buffer, sizeof(buffer)) < 0)
			goto error;
		if (!isdigit(buffer[0])) {
			errno = EINVAL;
			goto error;
		}
		code = strtoul(buffer, &sp, 10);

		/* Skip the flags, get the path name */
		if (code == MSG_OKAY && strtok(sp, " \t\r\n")
		 && (np = strtok(NULL, " \t\r\n"))
		 && wildmatch(wildcard, np)) {
			result = (char **) realloc(result,
					(count+2) * sizeof(char *));
			result[count++] = strdup(np);
			result[count] = NULL;
		}
	} while (*sp == '-');

	if (code != MSG_OKAY) {
		syslog(LOG_DEBUG, "resmgr: server response code %d", code);
		goto error1;
	}

	goto out;

error:	syslog(LOG_NOTICE, "resmgr: communication failure: %m");
error1: while (count--)
		free(result[count]);
	if (result)
		free(result);

out:	rsm_close(conn);
	return result;
}

int
rsm_open_device_as(const char *family, const char *name, int flags)
{
	int	fd, ro = 1;

	/* Are we from a sane family? */
	if (!sane(name) || (family && !sane(family))) {
		errno = EINVAL;
		return -1;
	}

	/* If the name already starts with the family prefix,
	 * drop the family name.
	 */
	if (family) {
		int	n = strlen(family);
		
		if (!strncmp(name, family, n) && name[n] == ':')
			family = NULL;
	}

	if ((flags & O_ACCMODE) == O_RDWR
	 || (flags & O_ACCMODE) == O_WRONLY)
		ro = 0;

	fd = rsm_command(NULL, "open %s %s%s%s",
				ro? "-ro" : "",
				family? family : "",
				family? ":" : "",
				name);

	if (fd >= 0) {
		fcntl(fd, F_SETFL, flags);
	} else {
		errno = EACCES;
	}

	return fd;
}

int
rsm_open_device(const char *pathname, int flags)
{
	return rsm_open_device_as(NULL, pathname, flags);
}


int
rsm_lock_device(const char *pathname)
{
	int	code, res;

	if (pathname[0] != '/' || !sane(pathname)) {
		errno = EINVAL;
		return -1;
	}
	if (rsm_command(&code, "lock %s", pathname) >= 0)
		return 0;
	
	/* Is this a stale lock? */
	if (code == MSG_STALELOCK
	 && rsm_unlock_device(pathname) >= 0
	 && rsm_command(NULL, "lock %s", pathname) >= 0)
	 	return 0;

	errno = EACCES;
	return res;
}

int
rsm_unlock_device(const char *pathname)
{
	int	res;

	if (pathname[0] != '/' || !sane(pathname)) {
		errno = EINVAL;
		return -1;
	}
	if ((res = rsm_command(NULL, "unlock %s", pathname)) < 0)
		errno = EACCES;
	return res;
}

int
rsm_login(const char *user, const char *tty)
{
	if (!sane(user) || !sane(tty)) {
		errno = EINVAL;
		return -1;
	}

	return rsm_command(NULL, "login %s %s", user, tty);
}

int
rsm_logout(const char *tty)
{
	if (!sane(tty)) {
		errno = EINVAL;
		return -1;
	}

	return rsm_command(NULL, "logout %s", tty);
}

int
rsm_grant(const char *user, const char *classname)
{
	if (!sane(user) || !sane(classname)) {
		errno = EINVAL;
		return -1;
	}

	return rsm_command(NULL, "grant %s %s", user, classname);
}

int
rsm_revoke(const char *user, const char *classname)
{
	if (!sane(user) || (classname && !sane(classname))) {
		errno = EINVAL;
		return -1;
	}

	if (classname)
		return rsm_command(NULL, "revoke %s %s", user, classname);
	return rsm_command(NULL, "revoke %s %s", user);
}

int
rsm_add_device(const char *device, const char *classname)
{
	if (!sane(device) || !sane(classname)) {
		errno = EINVAL;
		return -1;
	}

	return rsm_command(NULL, "add %s %s", device, classname);
}

int
rsm_remove_device(const char *device, const char *classname)
{
	if (!sane(device) || (classname && !sane(classname))) {
		errno = EINVAL;
		return -1;
	}

	if (classname)
		return rsm_command(NULL, "remove %s %s", device, classname);
	return rsm_command(NULL, "remove %s", device);
}

/*
 * Utility function - check whether argument is reasonably sane
 */
int
sane(const char *arg)
{
	int	n;

	n = strcspn(arg, " \t\r\n");
	return (arg[n] == '\0');
}

/*
 * Simplified glob matching
 */
int
wildmatch(const char *pattern, const char *name)
{
	const char	*colon;

	if (pattern == NULL)
		return 1;

	/* Skip the resource family if present */
	if ((colon = strchr(name, ':')) != NULL)
		name = colon + 1;

	while ((colon = strchr(name, ':')) != NULL) {
		if (rsm_glob(pattern, name, colon))
			return 1;
		name = colon + 1;
	}

	return rsm_glob(pattern, name, NULL);
}
