/*
*  program name: Rubrica <http://digilander.iol.it/nfragale/download.html>
*  file: account.c
*  
*  Copyright (C) 2000-2001 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <gnome.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>

#include "csv.h"
#include "rubrica.h"
#include "ui.h"
#include "card.h"
#include "import.h"


GtkWidget *csv_list;
GtkWidget *label2;
gint record, model_length;
gboolean file_end = FALSE;
RubricaImportType import_type;

static RubricaCsv mozilla[] = {
  {N_("Display Name"),    TRUE}, {N_("Last Name"),       TRUE},   // 0 - 1
  {N_("First Name"),      TRUE}, {N_("Nickname"),        TRUE},   // 2 - 3 
  {N_("Primary Email"),   TRUE}, {N_("Secondary Email"), TRUE},   // 4 - 5
  {N_("Work Phone"),      TRUE}, {N_("Home Phone"),      TRUE},   // 6 - 7 
  {N_("Fax Number"),      TRUE}, {N_("Pager Number"),    TRUE},   // 8 - 9
  {N_("Cellular Number"), TRUE}, {N_("Home Address"),    TRUE},   // 10 - 11
  {N_("Home Address 2"),  TRUE}, {N_("Home City"),       TRUE},   // 12 - 13  
  {N_("Home State"),      TRUE}, {N_("Home ZipCode"),    TRUE},   // 14 - 15
  {N_("Home Country"),    TRUE}, {N_("Work Address"),    TRUE},   // 16 - 17
  {N_("Work Address 2"),  TRUE}, {N_("Work City"),       TRUE},   // 18 - 19 
  {N_("Work State"),      TRUE}, {N_("Work ZipCode"),    TRUE},   // 20 - 21  
  {N_("Work Country"),    TRUE}, {N_("Job Title"),       TRUE},   // 22 - 23
  {N_("Department"),      TRUE}, {N_("Company"),         TRUE},   // 24 - 25 
  {N_("Web Page 1"),      TRUE}, {N_("Web Page 2"),      TRUE},   // 26 - 27
  {N_("Birth Year"),      TRUE}, {N_("Birth Month"),     TRUE},   // 28 - 29
  {N_("Birth Day"),       TRUE}, {N_("Custom 1"),        TRUE},   // 30 - 31
  {N_("Custom 2"),        TRUE}, {N_("Custom 3"),        TRUE},   // 32 - 33
  {N_("Custom 4"),        TRUE}, {N_("Notes"),           TRUE},   // 34 - 35
  {NULL,                  TRUE}
}; 

static RubricaCsv netscape[] = {
  {N_("Display Name"), TRUE}, {N_("Last Name"),    TRUE},   // 0 - 1 
  {N_("First Name"),   TRUE}, {N_("Notes"),        TRUE},   // 2 - 3  
  {N_("City"),         TRUE}, {N_("State"),        TRUE},   // 4 - 5  
  {N_("Email"),        TRUE}, {N_("Title"),        TRUE},   // 6 - 7  
  {N_("Web Page"),     TRUE}, {N_("Address"),      TRUE},   // 8 - 9  
  {N_("ZipCode"),      TRUE}, {N_("Country"),      TRUE},   // 10 - 11  
  {N_("Work Phone"),   TRUE}, {N_("Fax Number"),   TRUE},   // 12 - 13  
  {N_("Home Phone"),   TRUE}, {N_("Organization"), TRUE},   // 14 - 15  
  {N_("Nickname"),     TRUE}, {N_("Cellular"),     TRUE},   // 16 - 17  
  {N_("Pager"),        TRUE},                               // 18
  {NULL,               TRUE}
};


gboolean rubrica_csv_is_separator(gchar ch)
{
  if ((ch == ',')    ||   /* comma separator */
      (ch == '\t'))       /* tab   separator */
    return TRUE;
  
  return FALSE;
}

gboolean rubrica_csv_is_quote(gchar ch)
{
  if (ch == '"')
    return TRUE;
  
  return FALSE;
}


gchar *rubrica_csv_read_line(FILE *fp)
{
  gchar buffer[2048];  
  gchar *s;
  
  s = fgets(buffer, 2048, fp);

  if (feof(fp))
    file_end = TRUE;
  else
    file_end = FALSE;

  if (!s)
    return NULL;

  return s;
}


void rubrica_csv_parse_line(gchar *line)
{
  gchar ch, *utf8str;
  gint index, n;  
  gboolean closing_quote = FALSE;
  RubricaCsvParseState state;
  GString *buffer = NULL;

  index = 0;
  n = 0; 
  state = RUBRICA_CSV_START_STATE;
  
  while ((ch = line[index++]) != '\0')
    {
      switch (state)
	{
	case RUBRICA_CSV_START_STATE:
	  buffer = g_string_new(NULL);

	  if (rubrica_csv_is_quote(ch))	    
	      state = RUBRICA_CSV_QUOTE_STATE;
	  else if (rubrica_csv_is_separator(ch))
	    {
	      utf8str = g_locale_to_utf8(buffer->str, -1, NULL, NULL, NULL);
	      
	      rubrica_csv_insert_item(utf8str, n++);

	      g_string_free(buffer, TRUE);
	    }
	  else
	    {
	      g_string_append_c(buffer, ch);

	      state = RUBRICA_CSV_BUILD_BUFFER_STATE;
	    }

	  break;
	      
	case RUBRICA_CSV_QUOTE_STATE:	  
	  if (rubrica_csv_is_quote(ch))
	    {
	      if (!closing_quote)
		state = RUBRICA_CSV_DOUBLE_QUOTE_STATE;
	      else
		state = RUBRICA_CSV_BUILD_BUFFER_STATE;
	    }
	  else
	    {
	      g_string_append_c(buffer, ch);
	      
	      state = RUBRICA_CSV_BUILD_BUFFER_STATE;
	    }

	  break;

	case RUBRICA_CSV_DOUBLE_QUOTE_STATE:
	  if (rubrica_csv_is_quote(ch))
	    {
	      closing_quote = TRUE;

	      state = RUBRICA_CSV_QUOTE_STATE;
	    }
	  else
	    {
	      g_string_append_c(buffer, ch); 

	      state = RUBRICA_CSV_BUILD_BUFFER_STATE;	  
	    }
	  
	  break;

	case RUBRICA_CSV_BUILD_BUFFER_STATE:	  	  
	  if (rubrica_csv_is_quote(ch))
	    state = RUBRICA_CSV_QUOTE_STATE;
	  else if (rubrica_csv_is_separator(ch))
	    {
	      utf8str = g_locale_to_utf8(buffer->str, -1, NULL, NULL, NULL);

	      rubrica_csv_insert_item(utf8str, n++);
	      g_string_free(buffer, TRUE);
	      
	      state =  RUBRICA_CSV_START_STATE;
	      closing_quote = FALSE;
	    }
	  else 
	    g_string_append_c(buffer, ch);
	  break;

	default:
	  break;
	}
    }
}


gboolean 
rubrica_csv_import_file(FILE *fin)
{
  GtkWidget *dialog;
  Rubrica *rubrica = NULL;
  gboolean have_rubrica = FALSE;
  gboolean imported = FALSE;
  gchar *buffer;
  gint response;
  gint i;

  record = 0;
  dialog = rubrica_csv_create_import_dialog();
  gtk_widget_show(dialog);

  buffer = g_strdup(rubrica_csv_read_line(fin));
  if (!buffer)
    return FALSE;

  record++;
  rubrica_csv_parse_line(buffer);
  rubrica_csv_display_records(record);
  g_free(buffer);

  while (1)
    {
      response = gtk_dialog_run(GTK_DIALOG(dialog));

      if (!have_rubrica)
	{
	  rubrica_card_add_page_view(NULL);
	  rubrica = rubrica_get_current_rubrica();

	  have_rubrica = TRUE;
	}

      switch (response)
	{
	case GTK_RESPONSE_ACCEPT:  

	  if (record > 0)
	    {
	      record--;
	      rewind(fin);
	      for (i=0; i < record; i++)
		rubrica_csv_read_line(fin);
	    }

	  while(!feof(fin))
	    {
	      rubrica_csv_clean_list();
	      buffer = g_strdup(rubrica_csv_read_line(fin));
	      if (buffer)
		{
		  rubrica_csv_parse_line(buffer);
		  if (import_type == IMPORT_CSV_MOZILLA)
		    rubrica_csv_add_mozilla_address_to_rubrica(rubrica);
		  else
		    rubrica_csv_add_netscape_address_to_rubrica(rubrica);

		  g_free(buffer);

		  record++;
		  rubrica_csv_display_records(record);
		}
	    }
	  imported = TRUE;
	  goto out;
	  break;

	case RUBRICA_CSV_PREV_REC:
	  rubrica_csv_clean_list();

	  rubrica_delete_last_item(rubrica);
	  record--;
	  rubrica_csv_display_records(record);

	  if (record == 0)
	    gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), 
					      RUBRICA_CSV_PREV_REC, FALSE);
	  if (!file_end)
	    gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), 
					      RUBRICA_CSV_NEXT_REC, TRUE); 
	  rewind(fin);
	  for (i=0; i < record-1; i++)
	    rubrica_csv_read_line(fin);

	  buffer = g_strdup(rubrica_csv_read_line(fin));
	  if (buffer)
	    {
	      rubrica_csv_parse_line(buffer); 

	      g_free(buffer);
	    }
	  break;
 
	case RUBRICA_CSV_NEXT_REC:
	  /* store the current address in rubrica */
	  gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), 
					    RUBRICA_CSV_PREV_REC, TRUE);
	  if (import_type == IMPORT_CSV_MOZILLA)
	    rubrica_csv_add_mozilla_address_to_rubrica(rubrica);
	  else
	    rubrica_csv_add_netscape_address_to_rubrica(rubrica);
	  rubrica_csv_clean_list();
	  record++;
	  rubrica_csv_display_records(record);

	  /* read next address from csv file and display it */
	  buffer = g_strdup(rubrica_csv_read_line(fin));
	  if (buffer)
	    {
	      rubrica_csv_parse_line(buffer); 	  
	      
	      g_free(buffer);
	    }

	  if (file_end)
	    gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), 
					      RUBRICA_CSV_NEXT_REC, FALSE); 
	  break;
	  
	case GTK_RESPONSE_CLOSE:
	default:
	  rubrica_set_new(rubrica, FALSE);
	  rubrica_set_empty(rubrica, TRUE);
	  rubrica_set_modified(rubrica, FALSE);	  
	  rubrica_close_rubrica(rubrica);
	  imported = FALSE;
	  goto out;
	  break;
	}
    }
 out:
  gtk_widget_destroy(dialog);  

  return imported;
}


GtkWidget 
*rubrica_csv_create_import_dialog(void)
{
  GtkWidget *dialog;
  GtkWidget *dialog_vbox;
  GtkWidget *hbox1;
  GtkWidget *hbox2;
  GtkWidget *vbox;
  GtkWidget *label;

  GtkWidget *scrollwin;
  GtkWidget *move_up_button;
  GtkWidget *move_down_button;

  GtkListStore *csv_model;


  import_type = rubrica_import_get();

  dialog = gtk_dialog_new();
  gtk_dialog_add_buttons(GTK_DIALOG(dialog), 			 
			 _("Pre_vious record"), RUBRICA_CSV_PREV_REC,
			 _("Ne_xt record"), RUBRICA_CSV_NEXT_REC,
			 GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, 
			 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
			 NULL);
  gtk_window_set_title(GTK_WINDOW(dialog), _("Import csv file"));
  gtk_window_set_resizable (GTK_WINDOW (dialog), TRUE);
  gtk_window_resize(GTK_WINDOW (dialog), 450, 500);
  gtk_widget_show(dialog);

  gtk_dialog_set_has_separator (GTK_DIALOG(dialog), TRUE);
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
  gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog), RUBRICA_CSV_PREV_REC, 
				    FALSE);

  dialog_vbox = GTK_DIALOG (dialog)->vbox;
  gtk_widget_show (dialog_vbox);

  hbox1 = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(dialog_vbox), hbox1, FALSE, FALSE, 5);
  gtk_widget_show(hbox1);  

  label = gtk_label_new(_("Record(s) read:"));
  gtk_box_pack_start(GTK_BOX(hbox1), label, FALSE, TRUE, 2);
  gtk_widget_show(label);

  label2 = gtk_label_new("");
  gtk_box_pack_start(GTK_BOX(hbox1), label2, FALSE, FALSE, 2);
  gtk_widget_show(label2);

  hbox2 = gtk_hbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(dialog_vbox), hbox2, TRUE, TRUE, 5);
  gtk_widget_show(hbox2);

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_box_pack_start(GTK_BOX(hbox2), vbox, FALSE, FALSE, 5);
  gtk_widget_show(vbox);

  move_up_button = gtk_button_new_with_mnemonic(_("Move data _up"));
  gtk_box_pack_start(GTK_BOX(vbox), move_up_button, FALSE, FALSE, 5);
  gtk_widget_show(move_up_button);

  move_down_button = gtk_button_new_with_mnemonic(_("Move data do_wn"));
  gtk_box_pack_start(GTK_BOX(vbox), move_down_button, FALSE, FALSE, 5);
  gtk_widget_show(move_down_button);
 

  /* list
   */
  scrollwin = rubrica_ui_scrolled_window_new(); 
  gtk_box_pack_start(GTK_BOX(hbox2), scrollwin, TRUE, TRUE, 5);

  csv_model = rubrica_csv_create_model();
  if (import_type == IMPORT_CSV_MOZILLA)
    rubrica_csv_populate_model(csv_model, RUBRICA_CSV_MOZILLA);
  else 
    rubrica_csv_populate_model(csv_model, RUBRICA_CSV_NETSCAPE);
 
  csv_list = rubrica_csv_create_list(csv_model);
  gtk_container_add(GTK_CONTAINER(scrollwin), csv_list);

  rubrica_csv_add_column(csv_list);
  g_object_unref(G_OBJECT(csv_model));

  /* callbacks
   */
  g_signal_connect(G_OBJECT(move_up_button), "clicked", 
		   G_CALLBACK(rubrica_csv_move_data), 
		   GINT_TO_POINTER(RUBRICA_CSV_MOVE_DATA_UP));

  g_signal_connect(G_OBJECT(move_down_button), "clicked", 
		   G_CALLBACK(rubrica_csv_move_data), 
		   GINT_TO_POINTER(RUBRICA_CSV_MOVE_DATA_DOWN));

  return  dialog;
}



void 
rubrica_csv_clean_list(void)
{
  gint i;

  for (i=0; i < model_length; i++)
    rubrica_csv_insert_item(NULL, i);
}


void 
rubrica_csv_insert_item(gchar *str, gint pos)
{
  GtkTreeModel *model;  
  GtkTreeIter iter;
  gchar *tmp;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(csv_list));
  tmp = g_strdup_printf("%d", pos);

  if (!gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(model), &iter, tmp))
    {
      g_print("\nimpossibile pulire la lista -- posizione: %d", pos);
      g_free(tmp);

      return;
    }

  g_free(tmp);
  
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
		     RUBRICA_CSV_DATA_COLUMN, str,
		     -1);
}

gchar 
*rubrica_csv_extract_item(gint n)
{
  gchar *str, *buffer;
  gboolean import;
  GtkTreeModel *model;
  GtkTreeIter iter;

  model = gtk_tree_view_get_model(GTK_TREE_VIEW(csv_list));
  str = g_strdup_printf("%d", n);

  if (gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(model), &iter, str))
    {
      gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, 
			 RUBRICA_CSV_DATA_COLUMN, &buffer,
			 RUBRICA_CSV_CHECK_COLUMN, &import, -1);   
      
      if (buffer && import)
	return buffer;
    }
  g_free(str);

  return NULL;
}

void rubrica_csv_add_netscape_address_to_rubrica(Rubrica *rubrica)
{
  GList *email = NULL, *web = NULL, *tel = NULL;
  TelNum *tt;
  gchar *tmp1;
  RubricaItem *r;

  r = rubrica_item_new();
  if (!r) return;

  r->card = ((tmp1 = rubrica_csv_extract_item(0)) ? g_strdup(tmp1) : ""); 
	     
  r->personal.first = ((tmp1 = rubrica_csv_extract_item(2)) ? 
		       g_strdup(tmp1) : "");
  r->personal.middle  = "";
  r->personal.last = ((tmp1 = rubrica_csv_extract_item(1)) ? 
		      g_strdup(tmp1) : "");
  r->personal.profession = "";
  r->personal.prefix     = "";
  r->personal.title      = ((tmp1 = rubrica_csv_extract_item(7)) ? 
			    g_strdup(tmp1) : "");
  r->personal.birthknow = FALSE;
  r->personal.birthdate = 0;

  r->address.street = ((tmp1 = rubrica_csv_extract_item(9)) ? 
		       g_strdup(tmp1) : "");

  r->address.number   = "";
  r->address.zip      = ((tmp1 = rubrica_csv_extract_item(10)) ?
			 g_strdup(tmp1) : "");
  r->address.city     = ((tmp1 = rubrica_csv_extract_item(4)) ? 
			 g_strdup(tmp1) : "");
  r->address.province = "";
  r->address.state    = ((tmp1 = rubrica_csv_extract_item(5)) ? 
			 g_strdup(tmp1) : ""); 
  r->address.country  = ((tmp1 = rubrica_csv_extract_item(11)) ? 
			 g_strdup(tmp1) : "");

  tmp1 = rubrica_csv_extract_item(6);
  if (tmp1)
    email = g_list_append(email, g_strdup(tmp1));  
  r->net.email = email; 

  tmp1 = rubrica_csv_extract_item(8);
  if (tmp1)
    web = g_list_append(web,  g_strdup(tmp1));  
  r->net.web = web; 
  
  tmp1 = rubrica_csv_extract_item(12);
  if (tmp1)  // work telephone
    {
      tt = (TelNum*) g_malloc(sizeof(TelNum));

      tt->number = g_strdup(tmp1);      
      tt->type   = g_strdup("work");
      
      tel = g_list_append(tel, tt);
   }
  
  tmp1 = rubrica_csv_extract_item(14);
  if (tmp1)  // home telephone
    {
      tt = (TelNum*) g_malloc(sizeof(TelNum));

      tt->number = g_strdup(tmp1);      
      tt->type   = g_strdup("home");

      tel = g_list_append(tel, tt);      
    }  

  tmp1 = rubrica_csv_extract_item(13);
  if (tmp1)  // fax telephone
    {
      tt = (TelNum*) g_malloc(sizeof(TelNum));

      tt->number = g_strdup(tmp1);      
      tt->type   = g_strdup("fax");

      tel = g_list_append(tel, tt);     
    }

  tmp1 = rubrica_csv_extract_item(18);
  if (tmp1)  // pager telephone
    {
      tt = (TelNum*) g_malloc(sizeof(TelNum));
      
      tt->number = g_strdup(tmp1);      
      tt->type   = g_strdup("other");

      tel = g_list_append(tel, tt);     
    }
  
  tmp1 = rubrica_csv_extract_item(17);
  if (tmp1)  // cellphone
    {
      tt = (TelNum*) g_malloc(sizeof(TelNum));

      tt->number = g_strdup(tmp1);      
      tt->type   = g_strdup("cellphone");

      tel = g_list_append(tel, tt);      
   }
  r->phone.telephone = tel;

  r->company.name     = "";
  r->company.street   = "";
  r->company.number   = "";
  r->company.zip      = "";
  r->company.city     = "";
  r->company.province = "";
  r->company.country  = "";
  r->company.web      = "";
  r->company.email    = "";
  r->company.operator = "";
  r->company.fax      = "";
  r->company.green    = "";
  r->company.customer_care = "";
  r->company.notes    = "";

  r->work.assigment =  "";
  r->work.organization = ((tmp1 = rubrica_csv_extract_item(15)) ? 
			  g_strdup(tmp1) : "");
  r->work.department = "";
  r->work.subdep     = "";
  r->work.secretary  = "";
  r->work.telephone  = "";

  r->notes.spouse   = "";
  r->notes.children = "";
  r->notes.hobbies  = "";
  r->notes.notes =  ((tmp1 = rubrica_csv_extract_item(3)) ? 
		     g_strdup(tmp1) : "");
  r->notes.pubkey   = "";

  r->created = r->last_change = time(NULL);

  rubrica_append_item(rubrica, r);  
}


void rubrica_csv_add_mozilla_address_to_rubrica(Rubrica *rubrica)
{
  GList *email = NULL, *web = NULL, *tel = NULL;
  TelNum *tt;
  gchar *tmp1, *tmp2;
  RubricaItem *r;

  r = rubrica_item_new();
  if (!r) return;

  r->card = ((tmp1 = rubrica_csv_extract_item(0)) ? g_strdup(tmp1) : ""); 
	     
  r->personal.first = ((tmp1 = rubrica_csv_extract_item(2)) ? 
		       g_strdup(tmp1) : "");
  r->personal.middle  = "";
  r->personal.last = ((tmp1 = rubrica_csv_extract_item(1)) ? 
		      g_strdup(tmp1) : "");
  r->personal.profession = "";
  r->personal.prefix     = "";
  r->personal.title      = "";
  r->personal.birthknow = FALSE;
  r->personal.birthdate = 0;

  tmp1 = rubrica_csv_extract_item(11);
  tmp2 = rubrica_csv_extract_item(12);
  if (tmp1 && tmp2)
    r->address.street  = g_strconcat(tmp1, " ", tmp2, NULL);
  else if (tmp1 && !tmp2)
    r->address.street  = g_strdup(tmp1);
  else if (tmp2 && !tmp1)
    r->address.street  = g_strdup(tmp2);
  else r->address.street = "";

  r->address.number   = "";
  r->address.zip      = ((tmp1 = rubrica_csv_extract_item(15)) ?
			 g_strdup(tmp1) : "");
  r->address.city     = ((tmp1 = rubrica_csv_extract_item(13)) ? 
			 g_strdup(tmp1) : "");
  r->address.province = ((tmp1 = rubrica_csv_extract_item(14)) ? 
			 g_strdup(tmp1) : "");
  r->address.state    = "";
  r->address.country  = ((tmp1 = rubrica_csv_extract_item(16)) ? 
			 g_strdup(tmp1) : "");

  tmp1 = rubrica_csv_extract_item(4);
  if (tmp1)
    email = g_list_append(email, g_strdup(tmp1));

  tmp1 = rubrica_csv_extract_item(5);
  if (tmp1)
    email = g_list_append(email, g_strdup(tmp1));  
  r->net.email = email; 

  tmp1 = rubrica_csv_extract_item(26);
  if (tmp1)
    web = g_list_append(web,  g_strdup(tmp1));  
  r->net.web = web; 
  
  tmp1 = rubrica_csv_extract_item(6);
  if (tmp1)  // work telephone
    {
      tt = (TelNum*) g_malloc(sizeof(TelNum));

      tt->number = g_strdup(tmp1);      
      tt->type   = g_strdup("work");
      
      tel = g_list_append(tel, tt);
   }
  
  tmp1 = rubrica_csv_extract_item(7);
  if (tmp1)  // home telephone
    {
      tt = (TelNum*) g_malloc(sizeof(TelNum));

      tt->number = g_strdup(tmp1);      
      tt->type   = g_strdup("home");

      tel = g_list_append(tel, tt);      
    }  

  tmp1 = rubrica_csv_extract_item(8);
  if (tmp1)  // fax telephone
    {
      tt = (TelNum*) g_malloc(sizeof(TelNum));

      tt->number = g_strdup(tmp1);      
      tt->type   = g_strdup("fax");

      tel = g_list_append(tel, tt);     
    }

  tmp1 = rubrica_csv_extract_item(9);
  if (tmp1)  // pager telephone
    {
      tt = (TelNum*) g_malloc(sizeof(TelNum));
      
      tt->number = g_strdup(tmp1);      
      tt->type   = g_strdup("other");

      tel = g_list_append(tel, tt);     
    }
  
  tmp1 = rubrica_csv_extract_item(10);
  if (tmp1)  // cellphone
    {
      tt = (TelNum*) g_malloc(sizeof(TelNum));

      tt->number = g_strdup(tmp1);      
      tt->type   = g_strdup("cellphone");

      tel = g_list_append(tel, tt);      
   }
  r->phone.telephone = tel;

  r->company.name = ((tmp1 = rubrica_csv_extract_item(25)) ? 
		     g_strdup(tmp1) : "");

  tmp1 = rubrica_csv_extract_item(17);
  tmp2 = rubrica_csv_extract_item(18);
  if (tmp1 && tmp2)
    r->company.street  = g_strconcat(tmp1, " ", tmp2, NULL);
  else if (tmp1 && !tmp2)
    r->company.street  = g_strdup(tmp1);
  else if (tmp2 && !tmp1)
    r->company.street  = g_strdup(tmp2);
  else r->company.street = "";

  r->company.number   = "";
  r->company.zip      = ((tmp1 = rubrica_csv_extract_item(21)) ? 
			 g_strdup(tmp1) : "");
  r->company.city     = ((tmp1 = rubrica_csv_extract_item(19)) ? 
			 g_strdup(tmp1) : "");
  r->company.province = ((tmp1 = rubrica_csv_extract_item(20)) ? 
			 g_strdup(tmp1) : "");
  r->company.country  = ((tmp1 = rubrica_csv_extract_item(22)) ? 
			 g_strdup(tmp1) : "");
  r->company.web      = ((tmp1 = rubrica_csv_extract_item(27)) ? 
			 g_strdup(tmp1) : "");
  r->company.email    = "";
  r->company.operator = "";
  r->company.fax      = "";
  r->company.green    = "";
  r->company.customer_care = "";
  r->company.notes    = "";

  r->work.assigment = ((tmp1 = rubrica_csv_extract_item(23)) ? 
		       g_strdup(tmp1) : "");
  r->work.organization = "";
  r->work.department = ((tmp1 = rubrica_csv_extract_item(24)) ? 
			g_strdup(tmp1) : "");
  r->work.subdep    = "";
  r->work.secretary = "";
  r->work.telephone = "";

  r->notes.spouse   = "";
  r->notes.children = "";
  r->notes.hobbies  = "";
  r->notes.notes = rubrica_csv_extract_item(35);
  r->notes.pubkey   = "";

  r->created = r->last_change = time(NULL);

  rubrica_append_item(rubrica, r);
}


void rubrica_csv_display_records(gint record)
{
  gchar *str;
  
  str = g_strdup_printf("%d", record);
  gtk_label_set_text(GTK_LABEL(label2), str);
}


GtkListStore *rubrica_csv_create_model(void)
{
  GtkListStore *model;
 
  model = gtk_list_store_new(RUBRICA_CSV_COLUMN, 
			     G_TYPE_BOOLEAN,  // check column
			     G_TYPE_STRING,   // field column
			     G_TYPE_STRING);  // data column
  
  return model;
}


void rubrica_csv_populate_model(GtkListStore *model, RubricaCsvModelType type)
{
  GtkTreeIter iter;
  RubricaCsv *list;

  switch(type)
    {
    case RUBRICA_CSV_NETSCAPE:
      model_length = 19;
      list = netscape;
      break;

    case RUBRICA_CSV_MOZILLA:
      model_length = 36;
      list = mozilla;
      break;
      
    default:
      return;
      break;
    }
  
  while (list->label)
    {
      gtk_list_store_append(GTK_LIST_STORE(model), &iter);
      gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
			 RUBRICA_CSV_CHECK_COLUMN, list->import,
			 RUBRICA_CSV_FIELD_COLUMN, _(list->label), 
			 -1);      
      list++;
    }    
}


GtkWidget* rubrica_csv_create_list(GtkListStore *model)
{
  GtkWidget *list;
  
  list = gtk_tree_view_new_with_model(GTK_TREE_MODEL(model)); 
  gtk_widget_show(list);
  
  gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(list), FALSE);
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(list), TRUE);
  gtk_tree_view_columns_autosize(GTK_TREE_VIEW(list));

  return list;
}


void rubrica_csv_add_column(GtkWidget *list)
{
  GtkCellRenderer *cell;

  cell = gtk_cell_renderer_toggle_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(list), -1, 
					      _("Import"), cell, "active", 
					      RUBRICA_CSV_CHECK_COLUMN, 
					      NULL); 
  
  g_signal_connect(G_OBJECT(cell), "toggled", 
		   G_CALLBACK(rubrica_csv_manage_columns), NULL);
  
  cell = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(list), -1, 
					      _("Field"), cell, "text",
					      RUBRICA_CSV_FIELD_COLUMN, 
					      NULL);    
  
  cell = gtk_cell_renderer_text_new();
  gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(list), -1, 
					      _("Data"), cell, "text", 
					      RUBRICA_CSV_DATA_COLUMN, 
					      NULL);  
}


void rubrica_csv_move_data(GtkWidget *button, gpointer user_data)
{
  RubricaCsvMoveType move = (RubricaCsvMoveType) user_data;
  GtkTreeModel *model;
  GtkTreeSelection *select;
  GtkTreeIter iter, new_iter;
  GtkTreePath *path, *new_path;
  gchar *data, *new_data;
  gint depth;

  model  = gtk_tree_view_get_model(GTK_TREE_VIEW(csv_list));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(csv_list));

  if (!gtk_tree_selection_get_selected(select, NULL, &iter))
    {
      g_print("\nselect an item");
      
      return;
    }

  gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, 
		     RUBRICA_CSV_DATA_COLUMN, &data, -1);

  new_path = path = gtk_tree_model_get_path(model, &iter);

  switch (move)
    {
    case RUBRICA_CSV_MOVE_DATA_UP:
      gtk_tree_path_prev(new_path);
      break;
      
    case RUBRICA_CSV_MOVE_DATA_DOWN:
      depth = gtk_tree_path_get_indices(new_path)[0];

      if (depth < model_length-1)
	gtk_tree_path_next(new_path);
      break;

    default:
      break;
    }
  
  gtk_tree_model_get_iter(model, &new_iter, new_path);  
  gtk_tree_model_get(GTK_TREE_MODEL(model), &new_iter, 
		     RUBRICA_CSV_DATA_COLUMN, &new_data, 
		     -1);
  
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
		     RUBRICA_CSV_DATA_COLUMN, new_data,
		     -1);
  
  gtk_list_store_set(GTK_LIST_STORE(model), &new_iter, 
		     RUBRICA_CSV_DATA_COLUMN, data,
		     -1);
  
  gtk_tree_selection_unselect_all(select);
  gtk_tree_selection_select_path(select, new_path);      
  gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(csv_list), 
			       new_path, NULL, TRUE, 0.5, 0.0);    
}



void rubrica_csv_manage_columns(GtkCellRendererToggle *cell, gchar *path_str, 
				gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreePath *path;
  gboolean status;      /* check status */

  model  = gtk_tree_view_get_model(GTK_TREE_VIEW(csv_list));
  path   = gtk_tree_path_new_from_string(path_str);

  gtk_tree_model_get_iter (model, &iter, path);
  gtk_tree_model_get (model, &iter,
		      RUBRICA_CSV_CHECK_COLUMN, &status, 
		      -1);
  status ^= 1;
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
		     RUBRICA_CSV_CHECK_COLUMN, status, 
		     -1);
}
