/*
*  Rubrica
*  file: druid.c
*  
*  Copyright (C) 2000-2001 Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif


#include <gnome.h>

#include "druid.h"
#include "ui.h"
#include "log.h"
#include "dialog.h"
#include "prefer.h"
#include "main.h"
#include "utils.h"

GtkWidget *druid_window;
gboolean install_script = FALSE;


GtkWidget* rubrica_druid_create_druid (void)
{
  GtkWidget *druid;
  GdkPixbuf *pixbuf;
  gchar *home;

  home = g_strdup(g_get_home_dir());

  druid_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width (GTK_CONTAINER (druid_window), 5);
  gtk_window_set_title (GTK_WINDOW (druid_window), _("Rubrica Druid"));

  druid = gnome_druid_new ();
  gtk_container_add (GTK_CONTAINER (druid_window), druid);
  gtk_container_set_border_width (GTK_CONTAINER (druid), 5);
  gtk_widget_show (druid);

  pixbuf = rubrica_ui_pixbuf_new("rubrica.jpg");

  rubrica_druid_start_page(druid, pixbuf); 
  rubrica_druid_first_page(druid, pixbuf, home);
  rubrica_druid_second_page(druid, pixbuf, home);
  rubrica_druid_finish_page(druid, pixbuf);

  g_free(home);

  g_signal_connect(G_OBJECT(druid), "cancel", 
		   G_CALLBACK(on_druid_cancel), NULL);
  
  return druid_window;
}


void rubrica_druid_start_page(GtkWidget *druid, GdkPixbuf *pixbuf)
{
  GtkWidget *druidpage;
  
  druidpage = gnome_druid_page_edge_new (GNOME_EDGE_START);
  gnome_druid_append_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druidpage));
  gnome_druid_page_edge_set_title (GNOME_DRUID_PAGE_EDGE (druidpage),
                                   _("Rubrica Configuration"));
  gnome_druid_page_edge_set_text (GNOME_DRUID_PAGE_EDGE (druidpage),
                                  _("It seems to be the first time that "
				    "you have run Rubrica.\nClick on "
				    "Forward button to start rubrica's "
				    "configuration."));
  if (pixbuf)
    gnome_druid_page_edge_set_logo (GNOME_DRUID_PAGE_EDGE (druidpage), pixbuf);

  gtk_widget_show (druidpage);
}


void rubrica_druid_first_page(GtkWidget *druid, GdkPixbuf *pixbuf, gchar *home)
{
  GtkWidget *druidpage;
  GtkWidget *druid_vbox_file;

  GtkWidget *frame_book;
  GtkWidget *book_vbox;
  GtkWidget *check_book;
  GtkWidget *fileentry_book;
  GtkWidget *entry_book;

  GtkWidget *frame_folder;
  GtkWidget *folder_vbox;
  GtkWidget *check_folder;
  GtkWidget *fileentry_folder;
  GtkWidget *entry_folder;

  gchar *addbook, *folder;


  addbook  = g_strdup_printf("%s/.rubrica/rubrica.rub", home);
  folder   = g_strdup_printf("%s/.rubrica", home);

  druidpage = gnome_druid_page_standard_new ();
  gnome_druid_append_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druidpage));
  if (pixbuf)
    gnome_druid_page_standard_set_logo(GNOME_DRUID_PAGE_STANDARD(druidpage),
				       pixbuf);
  gtk_widget_show_all (druidpage);
 
  druid_vbox_file = GNOME_DRUID_PAGE_STANDARD (druidpage)->vbox;
  gtk_container_set_border_width (GTK_CONTAINER (druid_vbox_file), 16);
  gtk_widget_show (druid_vbox_file);

  /*    Addressbook
  */
  frame_book = rubrica_ui_frame_new(_("Default addressbook"));
  gtk_box_pack_start (GTK_BOX (druid_vbox_file), frame_book, TRUE, TRUE, 5);
  gtk_widget_show (frame_book);

  book_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame_book), book_vbox);
  gtk_container_set_border_width (GTK_CONTAINER (book_vbox), 4);
  gtk_widget_show (book_vbox);

  check_book = gtk_check_button_new_with_mnemonic(_("Rubrica will _load this "
						    "addressbook at start"));
  gtk_box_pack_start (GTK_BOX (book_vbox), check_book, FALSE, FALSE, 0);
  gtk_widget_show (check_book);

  fileentry_book = gnome_file_entry_new (NULL, NULL);
  gtk_box_pack_start (GTK_BOX (book_vbox), fileentry_book, FALSE, FALSE, 0);
  gtk_widget_show (fileentry_book);

  entry_book = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (fileentry_book));
  gtk_widget_show (entry_book);

  gtk_entry_set_text(GTK_ENTRY(entry_book), addbook);
  rubrica_preferences_set_book(addbook);
  gtk_widget_set_sensitive(GTK_WIDGET(fileentry_book), 
			   gtk_toggle_button_get_active
			   (GTK_TOGGLE_BUTTON(check_book)));

  /*     Folder
  */
  frame_folder = rubrica_ui_frame_new(_("Default folder"));
  gtk_box_pack_start (GTK_BOX (druid_vbox_file), frame_folder, TRUE, TRUE, 5);
  gtk_widget_show (frame_folder);

  folder_vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame_folder), folder_vbox);
  gtk_container_set_border_width (GTK_CONTAINER (folder_vbox), 4);
  gtk_widget_show (folder_vbox);

  check_folder = gtk_check_button_new_with_mnemonic (_("Rubrica will _save "
						       "addressbooks "
						       "in this folder"));
  gtk_widget_show (check_folder);
  gtk_box_pack_start (GTK_BOX (folder_vbox), check_folder, FALSE, FALSE, 0);

  fileentry_folder = gnome_file_entry_new (NULL, NULL);
  gtk_widget_show (fileentry_folder);
  gtk_box_pack_start(GTK_BOX (folder_vbox), fileentry_folder, FALSE, FALSE, 0);

  entry_folder = gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY
					    (fileentry_folder));
  gtk_widget_show (entry_folder);

  gtk_entry_set_text (GTK_ENTRY (entry_folder), folder);
  rubrica_preferences_set_folder(folder);

  gtk_widget_set_sensitive(GTK_WIDGET(fileentry_folder), 
			   gtk_toggle_button_get_active
			   (GTK_TOGGLE_BUTTON(check_folder)));


  g_signal_connect (G_OBJECT(check_book), "toggled", 
		    G_CALLBACK(on_check_book_toggled), fileentry_book);
  g_signal_connect (G_OBJECT(check_folder), "toggled", 
		    G_CALLBACK(on_check_folder_toggled), fileentry_folder);

  g_signal_connect (G_OBJECT(entry_book), "activate",
		    G_CALLBACK(on_entry_book_activate), NULL); 
  g_signal_connect (G_OBJECT(entry_book), "changed", 
		    G_CALLBACK(on_entry_book_changed), NULL); 
  
  g_signal_connect (G_OBJECT(entry_folder), "activate", 
		    G_CALLBACK (on_entry_folder_activate), NULL); 
  g_signal_connect (G_OBJECT(entry_folder), "changed", 
		    G_CALLBACK(on_entry_folder_changed), NULL); 
  
  g_free(addbook);
  g_free(folder);
}


void rubrica_druid_second_page(GtkWidget *druid, GdkPixbuf *pixbuf, 
			       gchar *home)
{
  GtkWidget *druidpage;

  GtkWidget *frame;
  GtkWidget *check;
  GtkWidget *druid_box;
  GtkWidget *label;
  GtkWidget *box;


  druidpage = gnome_druid_page_standard_new (); 
  gnome_druid_append_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druidpage)); 
  gtk_widget_show_all (druidpage);  
 
  druid_box = GNOME_DRUID_PAGE_STANDARD (druidpage)->vbox; 
  gtk_container_set_border_width (GTK_CONTAINER (druid_box), 16); 
  gtk_widget_show (druid_box); 

  frame = rubrica_ui_frame_new(_("Nautilus script"));
  gtk_box_pack_start (GTK_BOX (druid_box), frame, TRUE, TRUE, 5);
  gtk_widget_show (frame);

  box = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame), box);
  gtk_container_set_border_width (GTK_CONTAINER (box), 4);
  gtk_widget_show (box);

  label = gtk_label_new(_("If you are using the Nautilus file manager, "
			  "rubrica can\ninstall a script, in nautilus "
			  "scripts directory, to\nexport rubrica's "
			  "addressbook to HTML")); 
  gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 5);
  gtk_widget_show(label);

  check = gtk_check_button_new_with_mnemonic (_("_Install Rubrica script "
						"for Nautilus"));
  gtk_box_pack_start (GTK_BOX (box), check, FALSE, FALSE, 0);
  gtk_widget_show (check);

  g_signal_connect (G_OBJECT(check), "toggled", 
		    G_CALLBACK(on_check_script_toggled), NULL);
}




void rubrica_druid_finish_page(GtkWidget *druid, GdkPixbuf *pixbuf)
{
  GtkWidget *druidpage;

  druidpage = gnome_druid_page_edge_new (GNOME_EDGE_FINISH);
  gnome_druid_append_page (GNOME_DRUID (druid), GNOME_DRUID_PAGE (druidpage));
  gnome_druid_page_edge_set_text (GNOME_DRUID_PAGE_EDGE (druidpage),
                                  _("Click on apply to finish configuration\n"
				    "and run Rubrica."));
  if (pixbuf)
    gnome_druid_page_edge_set_logo (GNOME_DRUID_PAGE_EDGE (druidpage), pixbuf);
  
  g_signal_connect (G_OBJECT(druidpage), "finish",
		    G_CALLBACK(on_druid_finish), NULL);

  gtk_widget_show (druidpage);
}



/*   CALLBACKS
*/
void 
on_druid_finish (GnomeDruid  *gnomedruid, gpointer data)
{ 
  gchar *home;

  home = g_strdup(g_get_home_dir()); 
  rubrica_preferences_set_home(home); 

  rubrica_utils_make_rubrica_home(home);
  rubrica_utils_make_dictionary_home(home);
  rubrica_log_init_log_file();
  g_free(home);

  if (install_script)
    rubrica_utils_copy_script();

  if (rubrica_preferences_get_have_default_folder())
    {
      gchar *folder;

      folder = rubrica_preferences_get_default_folder();
      
      if (!g_file_exists(folder))
	rubrica_utils_make_user_rubrica_home(folder);
    }

  rubrica_preferences_set_first_run(FALSE);
  gtk_widget_destroy(druid_window);

  rubrica_begin();
}


void on_druid_cancel (GnomeDruid *gnomedruid, gpointer data)
{
  gtk_widget_destroy(druid_window);  

  rubrica_begin();
}


void on_check_book_toggled(GtkToggleButton *togglebutton, gpointer  user_data)
{
  GtkWidget *fileentry = user_data;
  gboolean bool;

  bool = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutton));  

  gtk_widget_set_sensitive(fileentry, bool);
  rubrica_preferences_set_have_default_book(bool);
}


void on_check_folder_toggled (GtkToggleButton *togglebutton, 
			      gpointer user_data)
{
  GtkWidget *fileentry = user_data;
  gboolean bool;

  bool = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutton));
  
  gtk_widget_set_sensitive(fileentry, bool);
  rubrica_preferences_set_have_default_folder(bool);  
}

void on_check_script_toggled(GtkToggleButton *togglebutton, gpointer user_data)
{
  gboolean bool;

  bool = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutton));
  
  install_script = bool;
}


void
on_entry_book_activate (GtkEditable *editable, gpointer user_data)
{
  on_entry_book_changed(editable, NULL);
}


void on_entry_book_changed (GtkEditable *editable, gpointer user_data)
{
  gchar *text;
  
  g_return_if_fail(editable != NULL);
  
  text = gtk_editable_get_chars(GTK_EDITABLE(editable), 0, -1);
  rubrica_preferences_set_book(text); 
  
  g_free(text);  
}



void on_entry_folder_activate (GtkEditable *editable, gpointer user_data)
{
  on_entry_folder_changed(editable, NULL);  
}


void on_entry_folder_changed (GtkEditable *editable, gpointer user_data)
{
  gchar *text;
  
  g_return_if_fail(editable != NULL);
  
  text = gtk_editable_get_chars(GTK_EDITABLE(editable), 0, -1);
  rubrica_preferences_set_folder(text); 
  
  g_free(text);  
}
