/*
 * dviutils.c
 * by Hirotsugu Kakugawa
 *
 */
/*
 * Copyright (C) 1999  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This file is part of the DVIlib Library.  This library is free
 * software; you can redistribute it and/or modify it under the terms of
 * the GNU Library General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  This library is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_MALLOC_H
#  include <malloc.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#  include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_SYS_STAT_H
#  include <sys/stat.h>
#endif
#if defined(HAVE_STRING_H)
#  include  <string.h>
#endif
#if defined(HAVE_STRINGS_H)
#  include  <strings.h>
#endif
#include <ctype.h>
#include <sys/errno.h>
#include <sys/param.h>

#include "dvi-2_6.h"
#include "defs.h"



struct paper_info_struct {
  char     name[16];   /* paper name */
  double   w, h;       /* width, height, in inch */
  struct paper_info_struct  *next;
};

struct mode_info_struct {
  char     name[16];        /* device mode name */
  double   ppi, ppi_v;      /* pixels per inch */
  char    *desc;            /* description */ 
  struct mode_info_struct  *next;
};


Private struct paper_info_struct  paper_list;
Private struct mode_info_struct   mode_list;




/** 
 ** DVI_find_dvi_path
 **/
Public char*
DVI_find_dvi_path(char *f)
{
  static char  *suff[] = { 
    "", ".dvi", ".DVI", "dvi", "DVI", NULL };

  return  DVI_find_dvi_path2(f, suff);
}


/** 
 ** DVI_find_dvi_path2
 **/
Public char*
DVI_find_dvi_path2(char *f, char **suff)
{
  char  *path;
  int    i, n;
#ifdef HAVE_STAT
  struct stat  st;
#endif
  static char  *suff2[] = { "", NULL };
  
  if (f == NULL)
    return  NULL;

  if (suff == NULL)
    suff = suff2;

  n = 0;
  for (i = 0; suff[i] != NULL; i++){
    if (strlen(suff[i]) > n)
      n = strlen(suff[i]);
  }
  if ((path = (char*)malloc(strlen(f)+n+1)) == NULL)
    return  NULL;

  for (i = 0; suff[i] != NULL; i++){
    sprintf(path, "%s%s", f, suff[i]);
    if (access(path, R_OK) == 0){
#ifdef HAVE_STAT
      if (stat(path, &st) < 0)
	continue;
# ifndef __linux__
      if ((st.st_mode & S_IFREG) == 0)
	continue;
# else
      if (!S_ISREG(st.st_mode))
	continue;
# endif
#endif
      return  path;
    }
  } 

  free(path);
  return  NULL;
}



/**
 ** Read Paper Size Database
 **  (This should be called when DVIlib is initialized.)
 **/
Public int
DVI_read_paper_size_db(void)
{
  FILE   *fp;
  char   *f; 
  char    buff[BUFSIZ], n[BUFSIZ];
  double  w, h; 
  struct paper_info_struct  *p, *p0;

  f = DVILIBDIR PAPERS_CNF;
  if ((fp = fopen(f, "r")) == NULL){
    fprintf(stderr, "Can't open %s\n", f);
    exit(1);
    return -1;
  }
  
  paper_list.next = NULL;
  p0 = &paper_list;
  while (fgets(buff, sizeof(buff), fp) != NULL){
    if (buff[0] == '#')
      continue;
    if (sscanf(buff, "%s %lf %lf", n, &w, &h) != 3)
      continue;
    p = (struct paper_info_struct *)malloc(sizeof(struct paper_info_struct));
    if (p == NULL)
      break;
    p0->next = p;
    strncpy(p->name, n, sizeof(p->name));
    p->w = w;
    p->h = h;
    p->next = NULL;
    p0 = p;
#if 0
    printf("%8s %3.3f %3.3f\n", p->name, p->w, p->h);
#endif
  }

  fclose(fp);

  return 0;
}


/** 
 ** Parse Paper Name
 **/
Public int
DVI_parse_paper_size(char *paper_name, double *widthp, double *heightp)
{
  char  *s, *t;
  double w, h; 
  struct paper_info_struct  *p;

  /* check if "name" */
  for (p = paper_list.next; p != NULL; p = p->next){
    s = p->name;
    t = paper_name;
    while (toupper((int) *s) == toupper((int)*t)){
      if ((*s == '\0') || (*t == '\0'))
	break;
      s++;
      t++;
    }
    if ((*s == '\0') && (*t == '\0')){
      if (widthp != NULL)
	*widthp = p->w;
      if (heightp != NULL)
	*heightp = p->h;
      return 0;
    }
  }

  /* check if "size" */
  s = paper_name;
  w = h = strtod(s, &t);
  if (strncmp(t, "in", 2) == 0){
    ;
  } else if (strncmp(t, "cm", 2) == 0){
    w /= 2.54; h /= 2.54; t += 2;
  } else if (strncmp(t, "mm", 2) == 0){
    w /= 25.4; h /= 25.4; t += 2;
  } else if (strncmp(t, "pt", 2) == 0){
    w /= 72.0; h /= 72.0; t += 2;
  }

  if ((s = strchr(t, ',')) != NULL){
    s = s + 1;
    h = strtod(s, &t);
    if (strncmp(t, "in", 2) == 0){
      ;
    } else if (strncmp(t, "cm", 2) == 0){
      h /= 2.54;
    } else if (strncmp(t, "mm", 2) == 0){
      h /= 25.4;
    } else if (strncmp(t, "pt", 2) == 0){
      h /= 72.0;
    }
  }  

  if (widthp != NULL)
    *widthp  = w;
  if (heightp != NULL)
  *heightp = h;

  return -1;
}


/**
 ** Get Paper Size List
 **/
Public char**
DVI_get_paper_size_list(void)
{
  char  **papers; 
  int     n, i;
  struct paper_info_struct  *p;

  n = 0;
  for (p = paper_list.next; p != NULL; p = p->next)
    n++;

  if ((papers = (char**)calloc(n+1, sizeof(char*))) == NULL)
    return NULL;

  for (i = 0; i < n+1; i++)
    papers[i] = NULL;

  i = 0;
  for (p = paper_list.next; p != NULL; p = p->next, i++){
    if ((papers[i] = (char*)malloc(strlen(p->name)+1)) == NULL)
      break;
    strcpy(papers[i], p->name);
  }
  papers[i] = NULL;

  return papers;
}



/**
 ** Read Device Mode Database
 **  (This should be called when DVIlib is initialized.)
 **/
Public int
DVI_read_device_mode_db(void)
{
  FILE   *fp;
  char   *f, *s; 
  char    buff[BUFSIZ], n[BUFSIZ], d[BUFSIZ];
  double  ppi, ppi_v; 
  struct mode_info_struct  *p, *p0;

  mode_list.next = NULL;

  f = DVILIBDIR DEVMODES_CNF;
  if ((fp = fopen(f, "r")) == NULL)
    return -1;

  p0 = &mode_list;
  while (fgets(buff, sizeof(buff), fp) != NULL){
    if ((s = strchr(buff, '\n')) != NULL)
      *s = '\0';
    if (buff[0] == '#')
      continue;
    if (sscanf(buff, "%s %lf %lf", n, &ppi, &ppi_v) != 3)
      continue;
    strcpy(d, "");
    if ((s = strchr(buff, ':')) != NULL){
      s++;
      while (isspace((int)*s) && (*s != '\0'))
	s++;
      strcpy(d, s); 
    }	
    p = (struct mode_info_struct *)malloc(sizeof(struct mode_info_struct));
    if (p == NULL)
      break;
    p0->next = p;
    strncpy(p->name, n, sizeof(p->name));
    p->ppi   = ppi;
    p->ppi_v = ppi_v;
    if ((p->desc = (char*)malloc(strlen(d)+1)) != NULL)
      strcpy(p->desc, d);
    p->next  = NULL;
    p0 = p;
  }
  fclose(fp);

  return 0;
}


/** 
 ** Parse Device Mode Name
 **/
Public int
DVI_parse_device_mode(char *mode_name, double *ppip, double *ppi_vp, 
		      char **descp)
{
  char  *s, *t;
  struct mode_info_struct  *p;

  for (p = mode_list.next; p != NULL; p = p->next){
    s = p->name;
    t = mode_name;
    while (toupper((int)*s) == toupper((int)*t)){
      if ((*s == '\0') || (*t == '\0'))
	break;
      s++;
      t++;
    }
    if ((*s == '\0') && (*t == '\0')){
      if (ppip != NULL)
	*ppip = p->ppi;
      if (ppi_vp != NULL)
	*ppi_vp = p->ppi_v;
      if (descp != NULL)
	*descp = p->desc;
      return 0;
    }
  }

  return -1;
}


/**
 ** Get Device Mode List
 **/
Public char**
DVI_get_device_mode_list(void)
{
  char  **modes; 
  int     n, i;
  struct mode_info_struct  *p;

  n = 0;
  for (p = mode_list.next; p != NULL; p = p->next)
    n++;

  if ((modes = (char**)calloc(n+1, sizeof(char*))) == NULL)
    return NULL;

  for (i = 0; i < n+1; i++)
    modes[i] = NULL;

  i = 0;
  for (p = mode_list.next; p != NULL; p = p->next, i++){
    if ((modes[i] = (char*)malloc(strlen(p->name)+1)) == NULL)
      break;
    strcpy(modes[i], p->name);
  }
  modes[i] = NULL;

  return modes;
}
