#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library: Actions.
 *   
 *   This module defines the Action classes.  An Action is an abstract
 *   object representing a command to be performed.  
 */

#include "adv3.h"
#include "tok.h"

/* ------------------------------------------------------------------------ */
/*
 *   Object associations lists.  We use this object to store some lookup
 *   tables that we build during preinitialization to relate object usages
 *   (DirectObject, IndirectObject) to certain properties. 
 */
objectRelations: PreinitObject
    /* preinitialization - build the lookup tables */
    execute()
    {
        /* build the default pre-condition properties table */
        preCondDefaultProps[DirectObject] = &preCondDobjDefault;
        preCondDefaultProps[IndirectObject] = &preCondIobjDefault;

        /* build the catch-all pre-conditions properties table */
        preCondAllProps[DirectObject] = &preCondDobjAll;
        preCondAllProps[IndirectObject] = &preCondIobjAll;

        /* build the default verification properties table */
        verifyDefaultProps[DirectObject] = &verifyDobjDefault;
        verifyDefaultProps[IndirectObject] = &verifyIobjDefault;

        /* build the catch-all verification properties table */
        verifyAllProps[DirectObject] = &verifyDobjAll;
        verifyAllProps[IndirectObject] = &verifyIobjAll;

        /* build the default check properties table */
        checkDefaultProps[DirectObject] = &checkDobjDefault;
        checkDefaultProps[IndirectObject] = &checkIobjDefault;

        /* build the catch-all check properties table */
        checkAllProps[DirectObject] = &checkDobjAll;
        checkAllProps[IndirectObject] = &checkIobjAll;

        /* build the default action properties table */
        actionDefaultProps[DirectObject] = &actionDobjDefault;
        actionDefaultProps[IndirectObject] = &actionIobjDefault;

        /* build the catch-all check properties table */
        actionAllProps[DirectObject] = &actionDobjAll;
        actionAllProps[IndirectObject] = &actionIobjAll;
    }

    /* lookup table for default precondition properties */
    preCondDefaultProps = static new LookupTable()

    /* lookup table for catch-all precondition properties */
    preCondAllProps = static new LookupTable()

    /* lookup table for default verification properties */
    verifyDefaultProps = static new LookupTable()

    /* lookup table for catch-all verification properties */
    verifyAllProps = static new LookupTable()

    /* lookup table for default check properties */
    checkDefaultProps = static new LookupTable()

    /* lookup table for catch-all check properties */
    checkAllProps = static new LookupTable()

    /* lookup table for default action properties */
    actionDefaultProps = static new LookupTable()

    /* lookup table for catch-all action properties */
    actionAllProps = static new LookupTable()
;

/* ------------------------------------------------------------------------ */
/*
 *   Basic Action class.  An Action is the language-independent definition
 *   of the abstract action of a command.  
 */
class Action: BasicProd
    /*
     *   Reset the action in preparation for re-execution.  This should
     *   discard any scoped context from any past execution of the
     *   command, such as cached scope information.  
     */
    resetAction() { }

    /*
     *   Create an instance of this action, for use by a recursive or
     *   programmatically-generated command.
     *   
     *   The generic actions defined in the library are always subclassed
     *   by language-specific library modules, because the language
     *   modules have to define the grammar rules for the verbs - we can't
     *   define the grammar rules generically because the verbs wouldn't
     *   be reusable for non-English translations if we did.  As a result,
     *   library code referring to one of the library verbs by name, say
     *   TakeAction, doesn't get a language-specific subclass of the verb,
     *   but just gets the language-independent base class.
     *   
     *   However, to make full use of an Action object in a recursive
     *   command, we do need a final language-specific subclass - without
     *   this, we won't be able to generate text describing the command,
     *   for example.  This method bridges this gap by finding a suitable
     *   language-specific subclass of the given action, then creating an
     *   instance of that subclass rather than an instance of the base
     *   class.
     *   
     *   By default, we'll take any subclass of this action that is itself
     *   a class.  However, if any subclass has the property
     *   defaultForRecursion set to true, we'll use that class
     *   specifically - this lets the language module designate a
     *   particular subclass to use as the default for recursive commands,
     *   which might be desirable in cases where the language module
     *   defines more than one subclass of an action.  
     */
    createActionInstance()
    {
        local found;

        /* 
         *   Iterate over our subclasses.  Initialize 'found' to this base
         *   class, so that if we fail to find any subclasses, we'll at
         *   least be able to create an instance of the generic base
         *   class.  
         */
        for (local cur = firstObj(self, ObjClasses), found = self ;
             cur != nil ;
             cur = nextObj(cur, self, ObjClasses))
        {
            /* take this as the best so far */
            found = cur;

            /* 
             *   if this one has the 'defaultForRecursion' flag, use it
             *   without looking any further 
             */
            if (cur.defaultForRecursion)
                break;
        }

        /* return a new instance of what we found */
        return found.createInstance();
    }

    /*
     *   Create an instance of this action based on another action.  We'll
     *   copy the basic properties of the original action. 
     */
    createActionFrom(orig)
    {
        local action;
        
        /* create a new instance of this action */
        action = createActionInstance();

        /* copy the token list information to the new action */
        action.tokenList = orig.tokenList;
        action.firstTokenIndex = orig.firstTokenIndex;
        action.lastTokenIndex = orig.lastTokenIndex;

        /* the new action is implicit if the original was */
        if (orig.isImplicit)
            action.setImplicit(orig.implicitMsg);

        /* return the new action */
        return action;
    }

    /*
     *   Mark the command as implicit.  'msgProp' is the property (of
     *   libMessages) to use to announce the implicit command.  
     */
    setImplicit(msgProp)
    {
        /* mark ourselves as implicit */
        isImplicit = true;

        /* do not show default reports for implicit commands */
        showDefaultReports = nil;

        /* all implicit commands are nested */
        setNested();

        /* 
         *   do not include implicit commands in undo - since an implicit
         *   command is only a subpart of an explicit command, an implicit
         *   command is not undone individually but only as part of the
         *   enclosing explicit command 
         */
        includeInUndo = nil;

        /* remember the implicit command announcement message property */
        implicitMsg = msgProp;
    }

    /*
     *   Mark the command as nested, noting the parent action (which we
     *   take as the global current action). 
     */
    setNested()
    {
        /* remember the parent action */
        parentAction = gAction;
    }

    /* 
     *   Determine if I'm nested within the given action.  Returns true if
     *   the given action is my parent action, or if my parent action is
     *   nested within the given action. 
     */
    isNestedIn(action)
    {
        /* if my parent action is the given action, I'm nested in it */
        if (parentAction == action)
            return true;

        /* 
         *   if I have a parent action, and it's nested in the given
         *   action, then I'm nested in the given action because I'm
         *   nested in anything my parent is nested in 
         */
        if (parentAction != nil && parentAction.isNestedIn(action))
            return true;

        /* we're not nested in the given action */
        return nil;
    }

    /*
     *   Set the "original" action.  An action with an original action is
     *   effectively part of the original action for the purposes of its
     *   reported results. 
     */
    setOriginalAction(action)
    {
        /* remember my original action */
        originalAction = action;
    }

    /*
     *   Determine if this action is "part of" the given action.  I'm part
     *   of the given action if I am the given action, or the given action
     *   is my "original" action, or my original action is part of the
     *   given action.
     */
    isPartOf(action)
    {
        /* if I'm the same as the given action, I'm obviously part of it */
        if (action == self)
            return true;

        /* if my original action is part of the action, I'm part of it */
        if (originalAction != nil && originalAction.isPartOf(action))
            return true;

        /* I'm not part of the given action */
        return nil;
    }

    /*
     *   Mark the action as "remapped."  This indicates that the action
     *   was explicitly remapped to a different action during the remap()
     *   phase.  
     */
    setRemapped(orig) { remappedFrom = gAction; }

    /* determine if I'm remapped, and get the original action if so */
    isRemapped() { return remappedFrom != nil; }
    getRemappedFrom() { return remappedFrom; }

    /* 
     *   the defaultForRecursion flag must be explicitly set in subclasses
     *   when desired - by default we'll use any language-specific
     *   subclass of an Action for recursive commands 
     */
    defaultForRecursion = nil

    /* 
     *   Flag: the command is implicit.  An implicit command is one that
     *   is performed as an implied enabling step of another command - for
     *   example, if an actor wants to throw something, the actor must be
     *   holding the object, so will implicitly try to take the object.
     */
    isImplicit = nil

    /*
     *   The parent action.  If the command is performed programmatically
     *   in the course of executing another command, this is set to the
     *   enclosing action.
     *   
     *   Note that while all implicit commands are nested, not all nested
     *   commands are implicit.  A nested command may simply be a
     *   component of another command, or another command may be handled
     *   entirely by running a different command as a nested command.  In
     *   any case, a nested but non-implicit command does not appear to
     *   the player as a separate command; it is simply part of the
     *   original command.  
     */
    parentAction = nil

    /*
     *   The original action we were remapped from.  This is valid when
     *   the action was explicitly remapped during the remap() phase to a
     *   different action. 
     */
    remappedFrom = nil

    /* 
     *   the original action - we are effectively part of the original
     *   action for reporting purposes 
     */
    originalAction = nil

    /* the libMessage property, if any, to announce the implicit command */
    implicitMsg = nil

    /* 
     *   Flag: we are to show default reports for this action.  In most
     *   cases we will do so, but for some types of commands (such as
     *   implicit commands), we suppress default reports. 
     */
    showDefaultReports = true

    /*
     *   Get a message parameter object for the action.  Each action
     *   subclass defines this to return its objects according to its own
     *   classifications.  The default action has no objects, but
     *   recognizes 'actor' as the current command's actor.  
     */
    getMessageParam(objName)
    {
        switch(objName)
        {
        case 'actor':
            /* return the current actor */
            return gActor;

        default:
            /* 
             *   if we have an extra message parameters table, look up the
             *   parameter name in the table 
             */
            if (extraMessageParams != nil)
                return extraMessageParams[objName];

            /* we don't recognize other names */
            return nil;
        }
    }

    /*
     *   Define an extra message-specific parameter.  Message processors
     *   can use this to add their own special parameters, so that they
     *   can refer to parameters that aren't involved directly in the
     *   command.  For example, a message for "take <dobj>" might want to
     *   refer to the object containing the direct object.
     */
    setMessageParam(objName, obj)
    {
        /* 
         *   if we don't yet have an extra message parameters table,
         *   create a small lookup table for it 
         */
        if (extraMessageParams == nil)
            extraMessageParams = new LookupTable(8, 8);

        /* add the parameter to the table, indexing by the parameter name */
        extraMessageParams[objName] = obj;
    }

    /*
     *   For convenience, this method allows setting any number of
     *   name/value pairs for message parameters. 
     */
    setMessageParams([lst])
    {
        /* set each pair from the argument list */
        for (local i = 1, local len = lst.length() ; i+1 <= len ; i += 2)
            setMessageParam(lst[i], lst[i+1]);
    }

    /*
     *   Extra message parameters.  If a message processor wants to add
     *   special message parameters of its own, we'll create a lookup
     *   table for the extra parameters.  Message processors might want to
     *   add their own special parameters to allow referring to objects
     *   other than the main objects of the command.  
     */
    extraMessageParams = nil

    /*
     *   Flag: this command is repeatable with 'again'.  Before executing
     *   a command, we'll save it for use by the 'again' command if this
     *   flag is true.
     */
    isRepeatable = true

    /*
     *   Flag: this command should be included in the undo records.  This
     *   is true for almost every command, but a few special commands
     *   (undo, save) are not subject to undo.  
     */
    includeInUndo = true

    /*
     *   Get the object "role" identifier for the given index.  This
     *   returns the identifier (DirectObject, IndirectObject, etc.) for
     *   the object at the given slot index, as used in
     *   setResolvedObjects().  The first object is at index 1.  
     */
    getRoleFromIndex(idx) { return nil; }

    /* get the resolved object in a given role */
    getObjectForRole(role) { return nil; }

    /* get the match tree for the noun phrase in the given role */
    getMatchForRole(role) { return nil; }

    /* get the 'verify' property for a given object role */
    getVerifyPropForRole(role)
    {
        return nil;
    }

    /* get the 'preCond' property for a given object role */
    getPreCondPropForRole(role)
    {
        return nil;
    }

    /* get the 'remap' property for a given object role */
    getRemapPropForRole(role)
    {
        return nil;
    }

    /*
     *   Explicitly set the resolved objects.  This should be overridden
     *   in each subclass for the number of objects specific to the action
     *   (a simple transitive action takes one argument, an action with
     *   both a direct and indirect object takes two arguments, and so
     *   on).  The base action doesn't have any objects at all, so this
     *   takes no arguments.
     *   
     *   This method is used to set up an action to be performed
     *   programmatically, rather than based on parsed input.  Since
     *   there's no parsed input in such cases, the objects are specified
     *   directly by the programmatic caller.  
     */
    setResolvedObjects() { }

    /*
     *   Explicitly set the object match trees.  This sets the
     *   pre-resolved production match trees, using the same order for
     *   object roles as setResolvedObjects().  
     */
    setObjectMatches() { }

    /*
     *   Check that the resolved objects are all in scope.  Returns true
     *   if so, nil if not. 
     */
    resolvedObjectsInScope()
    {
        /* we have no objects at all, so there is nothing out of scope */
        return true;
    }

    /*
     *   Convert an object or list of objects to a ResolveInfo list 
     */
    makeResolveInfoList(val)
    {
        /* if we have a non-list collection, make it a list */
        if (dataType(val) == TypeObject && val.ofKind(Collection))
            val = val.toList();

        /* if it's nil or an empty list, return an empty list */
        if (val == nil || val == [])
            return [];

        /* see what we have */
        if (dataType(val) == TypeList)
        {
            /* it's a list - make a ResolveInfo for each item */
            return val.mapAll({x: new ResolveInfo(x, 0)});
        }
        else 
        {
            /* it's not a list - return a one-element ResolveInfo list */
            return [new ResolveInfo(val, 0)];
        }
    }

    /*
     *   Perform this action.  Throughout execution of the action, the
     *   global parser variables that specify the current actor and action
     *   are valid.  
     */
    doAction(issuingActor, targetActor, targetActorPhrase,
             countsAsIssuerTurn)
    {
        local oldActor;
        local oldIssuer;
        local oldAction;
        local oldResults;

        /* 
         *   save the current parser globals, for restoration when we
         *   finish this command - if this command is nested within
         *   another, this will let us ensure that everything is restored
         *   properly when we finish with this command 
         */
        oldActor = gActor;
        oldIssuer = gIssuingActor;
        oldAction = gAction;
        oldResults = gVerifyResults;

        /* 
         *   set the new globals (note that there are no verification
         *   results or command reports objects yet - these are valid only
         *   while we're running the corresponding command phases) 
         */
        gActor = targetActor;
        gIssuingActor = issuingActor;
        gAction = self;
        gVerifyResults = nil;

        /* 
         *   If the command is repeatable, save it for use by 'again'.
         *   Note that nested commands are never repeatable with 'again',
         *   since no one ever typed them in.
         *   
         *   "Again" is strictly for the player's use, so only save the
         *   command if this is a player turn.  This is a player command
         *   only if it is issued by the player character (which means it
         *   came from the player), and either it's directed to the player
         *   character or it counts as a turn for the player character.  
         */
        if (isRepeatable
            && parentAction == nil
            && (issuingActor.isPlayerChar()
                && (targetActor.isPlayerChar() || countsAsIssuerTurn)))
            AgainAction.saveForAgain(issuingActor, targetActor,
                                     targetActorPhrase, self);

        /* make sure we restore globals on our way out */
        try
        {
            local pc;
            
            /* start a new command visually if this isn't a nested action */
            if (parentAction == nil)
                gTranscript.addCommandSep();

            /* have the player character note initial conditions */
            pc = gPlayerChar;
            pc.noteConditionsBefore();

            /* run the before routine for the entire action */
            beforeActionMain();

            /* run the subclass-specific processing */
            doActionMain();

            /* run the after routine for the entire action */
            afterActionMain();

            /* 
             *   If this is a top-level action, show the command results.
             *   Don't show results for a nested action, since we want to
             *   wait and let the top-level action show the results after
             *   it has the full set of results.  
             */
            if (parentAction == nil)
            {
                /* 
                 *   If the player character didn't change, ask the player
                 *   character to note any condition changes.  If the
                 *   player character did change to a new actor,
                 *   presumably the command will have displayed a specific
                 *   message, since this would be an unusual development
                 *   for which we can generate no generic message.  
                 */
                if (gPlayerChar == pc)
                    pc.noteConditionsAfter();
            }
        }
        finally
        {
            /* restore the parser globals to how we found them */
            gActor = oldActor;
            gIssuingActor = oldIssuer;
            gAction = oldAction;
            gVerifyResults = oldResults;
        }
    }

    /*
     *   Perform processing before running the action.  This is called
     *   just once per action, even if the action will be iterated for a
     *   list of objects. 
     */
    beforeActionMain()
    {
    }

    /*
     *   Perform processing after running the entire action.  This is
     *   called just once per action, even if the action was iterated for
     *   a list of objects. 
     */
    afterActionMain()
    {
        /* 
         *   Mark ourselves as busy for the amount of time this action
         *   takes.  Don't count the time taken for implicit moves,
         *   though, since these are meant to be zero-time sub-actions
         *   performed as part of the main action and thus don't have a
         *   separate time cost.
         *   
         *   Note that we add our busy time in the main after-action
         *   processing because we only want to count our time cost once
         *   for the whole command, even if we're performing the command
         *   on multiple objects.  
         */
        if (!isImplicit)
            gActor.addBusyTime(self, actionTime);
    }

    /* 
     *   the amount of time on the game clock that the action consumes -
     *   by default, each action consumes one unit, but actions can
     *   override this to consume more or less game time 
     */
    actionTime = 1

    /*
     *   Execute the action for a single set of objects.  This runs the
     *   full execution sequence for the current set of objects.
     *   
     *   Subclasses generally won't override this method, but will instead
     *   override the methods that implement the individual steps in the
     *   execution sequence.
     */
    doActionOnce()
    {
        /*
         *   Perform the sequence of operations to execute the action.  If
         *   an ExitSignal is thrown during the sequence, skip to the
         *   end-of-turn processing.  
         */
        try
        {
            local result;

            /*
             *   Before doing any actual execution, check the command for
             *   remapping.  If we end up doing any remapping, the
             *   remapping routine will simply replace the current command,
             *   so we the remapping call will terminate the current action
             *   with 'exit' and thus never return here.  
             */
            checkRemapping();

            /*
             *   If this is an implicit action, check for danger: we never
             *   try a command implicitly when the command is obviously
             *   dangerous.
             */
            if (isImplicit)
            {
                /* 
                 *   verify the action for an implicit command, checking
                 *   for danger - we never try a command implicitly when
                 *   the command is (or should be) obviously dangerous 
                 */
                result = verifyAction();

                /* if we found a 'dangerous' result, abort the command */
                if (result != nil && result.isDangerous)
                    abortImplicit;
            }

            /*
             *   If this is an implicit command, display a message
             *   indicating that we're performing the command.
             */
            if (isImplicit && implicitMsg != nil)
            {
                /* describe the implicit command we're performing */
                gTranscript.announceImplicit(implicitMsg);
            }

            /*
             *   Make one or two passes through verifications and
             *   preconditions.  If any precondition performs an implicit
             *   command, we must run everything a second time to ensure
             *   that the implicit command or commands did not invalidate
             *   any earlier precondition or a verification.
             *   
             *   Run verifications before preconditions, because there
             *   would be no point in applying implicit commands from
             *   preconditions if the command verifies as illogical in the
             *   first place.  
             */
            for (local iter = 1 ; iter <= 2 ; ++iter)
            {
                /* verify the action */
                result = verifyAction();
                
                /* 
                 *   if verification doesn't allow the command to proceed,
                 *   show the reason and end the command 
                 */
                if (result != nil && !result.allowAction)
                {
                    /* show the result message */
                    result.showMessage();
                    
                    /* mark the command as a failure */
                    gTranscript.isFailure = true;
                    
                    /* terminate the command */
                    exit;
                }

                /* 
                 *   Check preconditions of the action.  If we don't invoke
                 *   any implicit commands, we can stop here: nothing in
                 *   the game state will have changed, so there is no need
                 *   to re-verify or re-check preconditions.
                 *   
                 *   Only allow implicit actions on the first pass.  Do not
                 *   perform implicit actions on the second pass, because
                 *   if we did so we could get into an infinite loop of
                 *   conflicting preconditions, where each pass would
                 *   reverse the state from the last pass.  
                 */
                if (!checkPreConditions(iter == 1))
                    break;
            }

#ifdef SENSE_CACHE
            /* 
             *   Disable sense caching once we start the action phase -
             *   once we start making changes to game state, it's too much
             *   work to figure out when to invalidate the cache, so simply
             *   turn off caching entirely.
             *   
             *   Note that the sense cache will already be disabled if we
             *   executed any implied commands, because the first implied
             *   command will have disabled the cache as soon as it reached
             *   its execution phase, and no one will have turned caching
             *   back on.  It does no harm to disable it again here.  
             */
            libGlobal.disableSenseCache();
#endif

            /* run the before-action processing */
            beforeAction();
                
            /* 
             *   notify the actor's containers that an action is about to
             *   take place within them 
             */
            gActor.forEachContainer(callRoomBeforeAction);

            /* call beforeAction for each object in the notify list */
            notifyBeforeAfter(&beforeAction);
                
            /* 
             *   Invoke the action's execution method.  Catch any "exit
             *   action" exceptions - these indicate that the action is
             *   finished but that the rest of the command processing is to
             *   proceed as normal.  
             */
            try
            {
                /* notify the actor of what we're about to do */
                gActor.actorAction();
                
                /* execute the action */
                execAction();
            }
            catch (ExitActionSignal eaSig)
            {
                /* 
                 *   an exit action signal was thrown - since we've now
                 *   skipped past any remaining action processing, simply
                 *   continue with the rest of the command processing as
                 *   normal 
                 */
            }
            
            /* call afterAction for each object in the notify list */
            notifyBeforeAfter(&afterAction);
            
            /* notify the actor's containers of the completed action */
            gActor.forEachContainer(callRoomAfterAction);
            
            /* run the after-action processing */
            afterAction();
        }
        catch (ExitSignal exc)
        {
            /* the execution sequence is finished - simply stop here */
        }
    }

    /*
     *   Reset the message generation context for a sense change.  This
     *   can be called when something substantial happens in the midst of
     *   a command, and we might need different message generation rules
     *   before and after the change.  For example, this is used when a
     *   non-player character moves from one location to another, because
     *   the NPC might want to generate leaving and arriving messages
     *   differently in the two locations.
     */
    recalcSenseContext()
    {
        /* tell the sense context capturer to recalculate the context */
        senseContext.recalcSenseContext();
    }

    /*
     *   Announce the object of an action.  This should be used for each
     *   iteration of a command that takes objects to announce the objects
     *   on this iteration.
     *   
     *   We announce an object under several circumstances:
     *   
     *   - If we are iterating through multiple objects, we'll show the
     *   current object to show the player the individual step in the
     *   command being performed.
     *   
     *   - If 'all' was used to specify the object, we'll announce it even
     *   if only one object is involved, to make it clear to the player
     *   exactly what we chose as a match.
     *   
     *   - If we are executing the command on a single object, and the
     *   object was chosen through disambiguation of a set of ambiguous
     *   choices, and some of the discarded possibilities were logical but
     *   less so than the chosen object, we'll show the assumption we
     *   made.  In such cases, our assumption is not necessarily correct,
     *   so we'll tell the user about our choice explicitly by way of
     *   confirmation - this gives the user a better chance of noticing
     *   quickly if our assumption was incorrect.
     *   
     *   - If we supplied a default for a missing noun phrase in the
     *   player's command, we'll show what we chose.  Since the player
     *   didn't say what they meant, we'll make it plain that we're
     *   providing an assumption about what we thought they must have
     *   meant.
     *   
     *   'info' is the ResolveInfo object describing this resolved object,
     *   and 'numberInList' is the total number of objects we're iterating
     *   over for this object function (direct object, indirect object,
     *   etc).  'whichObj' is one of the object function constants
     *   (DirectObject, IndirectObject, etc) describing which object we're
     *   mentioning; the language-specific message generator might use
     *   this in conjunction with the action to include a preposition with
     *   the displayed phrase, for example, or choose an appropriate
     *   inflection.
     */
    announceActionObject(info, numberInList, whichObj)
    {
        /* 
         *   Show prefix announcements only if the player character is
         *   performing the action.  For NPC's, we don't use the prefix
         *   format, because it doesn't work as well for NPC result
         *   reports; instead, the NPC versions of the library messages
         *   tend to use sufficiently detailed reports that the prefix
         *   isn't required (for example, "Bob takes the iron key" rather
         *   than just "Taken").  
         */
        if (gActor.isPlayerChar)
        {
            /* check for the various announcements */
            if (numberInList > 1 || (info.flags_ & AlwaysAnnounce) != 0)
            {
                /* show the current object of a multi-object action */
                gTranscript.announceMultiActionObject(
                    info.obj_, whichObj);
            }
            else if ((info.flags_ & UnclearDisambig) != 0)
            {
                /* show the object, since we're not certain it's right */
                gTranscript.announceAmbigActionObject(info.obj_, whichObj);
            }
            else if ((info.flags_ & DefaultObject) != 0
                     && !(info.flags_ & AnnouncedDefaultObject))
            {
                /*   
                 *   Show the object, since we supplied it as a default.
                 *   At this stage in the command, we have resolved
                 *   everything.  
                 */
                gTranscript.announceDefaultObject(
                    info.obj_, whichObj, self, true);

                /* note that we've announced this object */
                info.flags_ |= AnnouncedDefaultObject;
            }
        }
    }

    /*
     *   Announce a defaulted object list, if appropriate.  We'll announce
     *   the object if we have a single object in the given resolution
     *   list, it was defaulted, and it hasn't yet been announced.
     */
    maybeAnnounceDefaultObject(lst, which, allResolved)
    {
        /* 
         *   if the list has exactly one element, and it's marked as a
         *   defaulted object, and it hasn't yet been announced, announce
         *   it 
         */
        if (lst != nil
            && lst.length() == 1
            && (lst[1].flags_ & DefaultObject) != 0
            && !(lst[1].flags_ & AnnouncedDefaultObject))
        {
            /* announce the object */
            gTranscript.announceDefaultObject(
                lst[1].obj_, which, self, allResolved);

            /* 
             *   we've now announced the object; mark it as announced so
             *   we don't show the same announcement again 
             */
            lst[1].flags_ |= AnnouncedDefaultObject;
        }
    }

    /*
     *   "Pre-announce" a common object for a command that might involve
     *   iteration over other objects.  For example, in "put all in box",
     *   the box is common to all iterations of the command, so we would
     *   want to preannounce it, if it needs to be announced at all,
     *   before the iterations of the command.
     *   
     *   We'll announce the object only if it's marked as defaulted or
     *   unclearly disambiguated, and then only if the other list will be
     *   announcing its objects as multi-action objects.  However, we do
     *   not pre-announce anything for a remapped action, because we'll
     *   show the full action description for each individually announced
     *   object, so we don't need or want a separate announcement for the
     *   group.
     *   
     *   Returns true if we did any pre-announcing, nil otherwise.  If we
     *   return true, the caller should not re-announce this object during
     *   the iteration, since our pre-announcement is common to all
     *   iterations.  
     */
    preAnnounceActionObject(info, mainList, whichObj)
    {
        /* do not pre-announce anything for a remapped action */
        if (isRemapped())
            return nil;

        /* 
         *   determine if the main list will be announcing its objects -
         *   it will if it has more than one object, or if its one object
         *   is marked as "always announce" 
         */
        if (mainList.length() > 1
            || (mainList[1].flags_ & AlwaysAnnounce) != 0)
        {
            /* 
             *   we will be announcing the main list object or objects, so
             *   definitely pre-announce this object if appropriate 
             */
            announceActionObject(info, 1, whichObj);

            /* tell the caller we pre-announced the object */
            return true;
        }

        /* we didn't pre-announce the object */
        return nil;
    }

    /*
     *   Run our action-specific pre-processing.  By default, we do
     *   nothing here.  
     */
    beforeAction()
    {
    }

    /*
     *   Execute the action.  This must be overridden by each subclass.
     *   
     *   Intransitive actions must do all of their work in this routine.
     *   In most cases, transitive actions will delegate processing to one
     *   or more of the objects involved in the command - for example,
     *   most single-object commands will call a method in the direct
     *   object to carry out the command.  
     */
    execAction()
    {
        /* by default, just show the 'no can do' message */
        mainReport(&cannotDoThat);
    }

    /*
     *   Run our action-specific post-processing.  By default, we do
     *   nothing here. 
     */
    afterAction()
    {
    }

    /*
     *   Verify the action.  Action subclasses with one or more objects
     *   should call object verification routines here.  Returns a
     *   VerifyResultList with the results, or nil if there are no
     *   verification results at all.  A nil return should be taken as
     *   success, not failure, because it means that we found no objection
     *   to the command.  
     */
    verifyAction()
    {
        /* 
         *   there are no objects in the default action, but we might have
         *   pre-condition verifiers 
         */
        return callVerifyPrecond(nil);
    }

    /*
     *   Initialize tentative resolutions for other noun phrases besides
     *   the one indicated. 
     */
    initTentative(issuingActor, targetActor, whichObj)
    {
        /* by default, we have no noun phrases to tentatively resolve */
    }

    /*
     *   Check for remapping the action.  This should check with each
     *   resolved object involved in the command to see if the object wants
     *   to remap the action to a new action; if it does, the object must
     *   replace the current action (using replaceAction or equivalent).
     *   Note that replacing the action must use 'exit' to terminate the
     *   original action, so this will never return if remapping actually
     *   does occur.  
     */
    checkRemapping()
    {
        /* by default, we have no objects, so we do nothing here */
    }

    /*
     *   Invoke a callback with a verify results list in gVerifyResults,
     *   using the existing results list or creating a new one if there is
     *   no existing one.  Returns the results list used. 
     */
    withVerifyResults(resultsSoFar, obj, func)
    {
        local oldResults;

        /* if we don't already have a result list, create one */
        if (resultsSoFar == nil)
            resultsSoFar = new VerifyResultList(obj);

        /* remember the old global results list */
        oldResults = gVerifyResults;

        /* install the new one */
        gVerifyResults = resultsSoFar;

        /* make sure we restore the old result list on the way out */
        try
        {
            /* invoke the callback */
            (func)();
        }
        finally
        {
            /* restore the old result list */
            gVerifyResults = oldResults;
        }

        /* return the result list */
        return resultsSoFar;
    }

    /*
     *   Verify the non-object-related pre-conditions.  This runs
     *   verification on each of the pre-condition objects defined for the
     *   action.  
     */
    callVerifyPrecond(resultSoFar)
    {
        /* look at each of our preconditions */
        foreach (local cond in preCond)
        {
            /* 
             *   If this precondition defines a verifier, call it.  Check
             *   to see if we have a verifier defined first so that we can
             *   avoid creating a result list if we won't have any use for
             *   it. 
             */
            if (cond.propDefined(&verifyPreCondition))
            {
                /* 
                 *   invoke the pre-condition verifier - this is an
                 *   action-level verifier, so there's no object involved 
                 */
                resultSoFar = withVerifyResults(resultSoFar, nil,
                    {: cond.verifyPreCondition(nil) });
            }
        }

        /* return the latest result list */
        return resultSoFar;
    }

    /*
     *   Call a catch-all property on the given object.
     *   
     *   actionProp is the custom per-object/per-action property that we
     *   normally invoke to process the action.  For example, if we're
     *   processing verification for the direct object of Take, this would
     *   be &verifyDobjTake.
     *   
     *   defProp is the default property that corresponds to actionProp.
     *   This is the per-object/default-action property that we invoke
     *   when the object doesn't provide a "more specialized" version of
     *   actionProp - that is, if the object doesn't define or inherit
     *   actionProp at a point in its class hierarchy that is more
     *   specialized than the point at which it defines defProp, we'll
     *   call defProp.  If there is a more specialized definition of
     *   actionProp for the object, it effectively overrides the default
     *   handler, so we do not invoke the default handler.
     *   
     *   allProp is the catch-all property corresponding to actionProp.
     *   We invoke this property in all cases.  
     */
    callCatchAllProp(obj, actionProp, defProp, allProp)
    {
        /* 
         *   invoke the default property only if the object doesn't have a
         *   "more specialized" version of actionProp 
         */
        if (!obj.propHidesProp(actionProp, defProp))
            obj.(defProp)();

        /* always invoke the catch-all property */
        obj.(allProp)();
    }

    /*
     *   Call a verification routine.  This creates a results object and
     *   makes it active, then invokes the given verification routine on
     *   the given object.
     *   
     *   We call verification directly on the object, and we also call
     *   verification on the object's preconditions.
     *   
     *   If resultSoFar is non-nil, it is a VerifyResultList that has the
     *   results so far - this can be used for multi-object verifications
     *   to gather all of the verification results for all of the objects
     *   into a single result list.  If resultSoFar is nil, we'll create a
     *   new result list.  
     */
    callVerifyProp(obj, verProp, preCondProp, remapProp,
                   resultSoFar, whichObj)
    {
        local remapInfo;

        /* check for remapping */
        if ((remapInfo = obj.(remapProp)()) != nil)
            return remapVerify(whichObj, resultSoFar, remapInfo);

        /* initialize tentative resolutions for other noun phrases */
        initTentative(gIssuingActor, gActor, whichObj);

        /* 
         *   run the verifiers in the presence of a results list, and
         *   return the result list 
         */
        return withVerifyResults(resultSoFar, obj, new function()
        {
            local lst;
            
            /* 
             *   check the object for a default or catch-all verifier, and
             *   call it if present 
             */
            callCatchAllProp(obj, verProp,
                             objectRelations.verifyDefaultProps[whichObj],
                             objectRelations.verifyAllProps[whichObj]);

            /* 
             *   invoke the verifier method - this will update the global
             *   verification results object with the appropriate status
             *   for this action being performed on this object 
             */
            obj.(verProp)();

            /*
             *   Check the pre-conditions defined for this action on this
             *   object.  For each one that has a verifier, invoke the
             *   verifier.  
             */
            lst = getObjPreConditions(obj, preCondProp, whichObj);
            if (lst != nil)
                foreach (local cur in lst)
                    cur.verifyPreCondition(obj);
        });
    }

    /*
     *   Get the precondition list for an object.  whichObj is the object
     *   role of the object whose preconditions we're retrieving; this is
     *   nil if we're looking for action-level preconditions.  
     */
    getObjPreConditions(obj, preCondProp, whichObj)
    {
        local allPreProp;
        local defPreProp;
        local pre;

        /* 
         *   if we're looking for action preconditions, there are no
         *   default or catch-all properties, so simply get the
         *   preconditions from the action
         */
        if (whichObj == nil)
            return obj.(preCondProp);

        /* get the default-action and catch-all precondition properties */
        defPreProp = objectRelations.preCondDefaultProps[whichObj];
        allPreProp = objectRelations.preCondAllProps[whichObj];
        
        /*
         *   Check for an "overriding" default-action handler.  If we have
         *   a default-action handler that hides the specific handler for
         *   this action, use the default handler's precondition list
         *   instead.  Otherwise, use the specific action preconditions.  
         */
        if (obj.propHidesProp(defPreProp, preCondProp))
            pre = obj.(defPreProp);
        else
            pre = obj.(preCondProp);

        /* if we have catch-all preconditions, add them to the list */
        if (obj.propDefined(allPreProp))
            pre += obj.(allPreProp);

        /* return the precondition list */
        return pre;
    }

    /*
     *   Verify that the an action method ("doXxx") is defined on at least
     *   one of the given objects.  If it's not, we'll add an "illogical"
     *   status to the verification results to indicate that the action is
     *   not defined on this object.
     *   
     *   Each entry of objList must have a corresponding entry in propList
     *   which gives the action property for that object.  
     */
    verifyActionProp(objList, propList, result)
    {
        /* 
         *   check each object to see if any define their corresponding
         *   action property 
         */
        for (local i = 1, local len = objList.length() ; i <= len ; ++i)
        {
            /* 
             *   if this object defines the corresponding action property,
             *   note that we've found a definition 
             */
            if (objList[i].propDefined(propList[i]))
            {
                /* 
                 *   we've found a definition, so there's no need to add
                 *   an illogical status to the result list - just return
                 *   the original result list unchanged 
                 */
                return result;
            }
        }
        
        /* 
         *   None of the objects defines an appropriate action method, so
         *   this verifies as illogical.  If there's no result list so
         *   far, create one.  
         */
        if (result == nil)
            result = new VerifyResultList(obj);

        /* add an "illogical" status to the results */
        result.addResult(new IllogicalVerifyResult(&cannotDoThat));

        /* return the result */
        return result;
    }
    
    /*
     *   Call the beforeAction or afterAction method for each object in
     *   the notification list. 
     */
    notifyBeforeAfter(prop)
    {
        local lst;

        /* get a table of potential notification receivers */
        lst = getNotifyTable();

        /* go through the table and notify each object */
        lst.forEachAssoc({obj, val: obj.(prop)()});
    }

    /*
     *   Get the list of objects to notify before or after the action has
     *   been performed.  
     */
    getNotifyTable()
    {
        local tab;
        local curObjs;
        local actor;

        /* stash the current actor in a local for faster reference */
        actor = gActor;
        
        /* start with everything connected by containment to the actor */
        tab = actor.connectionTable();

        /* add the actor's explicitly registered notification list */
        foreach (local cur in actor.getActorNotifyList())
            tab[cur] = true;

        /* add the items explicitly registered in the actor's location(s) */
        actor.forEachContainer(
            {loc: loc.getRoomNotifyList().forEach(
                {obj: tab[obj] = true})
            });

        /* get the list of objects directly involved in the command */
        curObjs = getCurrentObjects();

        /* 
         *   add any objects explicitly registered with the objects
         *   directly involved in the command 
         */
        foreach (local cur in curObjs)
        {
            /* add each item in the object's notify list */
            foreach (local ncur in cur.getObjectNotifyList())
                tab[ncur] = true;
        }

        /*
         *   Remove from the list all of the actor's containers.  These
         *   will be notified via the more specific room notification
         *   methods, so we don't want to send them generic notifiers as
         *   well. 
         */
        tab.forEachAssoc(new function(obj, val)
        {
            if (actor.isIn(obj))
                tab.removeElement(obj);
        });

        /* return the final table */
        return tab;
    }

    /*
     *   Get the list of all of the objects (direct object, indirect
     *   object, and any additional objects for actions with three or more
     *   object roles) involved in the current execution.  This is valid
     *   only during a call to doActionOnce(), since that's the only time
     *   a particular set of objects are selected for the action.
     *   
     *   By default, an action has no objects roles at all, so we'll just
     *   return an empty list.  
     */
    getCurrentObjects()
    {
        return [];
    }

    /*
     *   Set the current objects.  This takes a list of the same form
     *   returned by getCurrentObjects(). 
     */
    setCurrentObjects(lst) { }

    /*
     *   Check any pre-conditions for the action.
     *   
     *   This should check all of the conditions that must be met for the
     *   action to proceed.  If any pre-condition can be met by running an
     *   implicit command first, that implicit command should be executed
     *   here.  If any pre-condition cannot be met, this routine should
     *   notify the actor and throw an ExitSignal.
     *   
     *   Returns true if any implicit commands are executed, nil if not.
     *   Implicit commands can only be attempted if allowImplicit is true;
     *   if this is nil, a precondition must simply fail (by displaying an
     *   appropriate failure report and using 'exit') without attempting
     *   an implicit command if its assertion does not hold.  
     */
    checkPreConditions(allowImplicit)
    {
        /* check each condition in our action preconditions */
        return callPreConditions(self, &preCond, nil, allowImplicit, nil);
    }

    /*
     *   Call a method on all of the precondition objects in the
     *   precondition list obtained from the given property of the given
     *   object. 
     */
    callPreConditions(obj, preCondProp, checkArg, allowImplicit, whichObj)
    {
        local lst;
        local ret;
        
        /* presume we won't call any implicit commands */
        ret = nil;

        /* get the precondition list from the object */
        lst = getObjPreConditions(obj, preCondProp, whichObj);

        /* if it's not nil, invoke the given check method */
        if (lst != nil)
        {
            /* invoke the check method for each condition in the list */
            foreach (local cur in lst)
            {
                /* 
                 *   call this precondition; if it runs an implicit
                 *   command, note that we have run an implicit command 
                 */
                if (cur.checkPreCondition(checkArg, allowImplicit))
                    ret = true;
            }
        }

        /* return true if any implicit commands were executed */
        return ret;
    }

    /* 
     *   Our list of action-level pre-condition objects.  These are the
     *   conditions that apply to the overall action, not to the
     *   individual objects involved.  (Object-level pre-conditions are
     *   attached to the objects, not to the action.)  
     */
    preCond = []

    /*
     *   Get a list of verification results for the given ResolveInfo
     *   objects, sorted from best to worst.  Each entry in the returned
     *   list will be a VerifyResultList object whose obj_ property is set
     *   to the ResolveInfo object for which it was generated.  
     */
    getSortedVerifyResults(lst, verProp, preCondProp, remapProp, whichObj)
    {
        local results;

        /* if there's nothing in the list, we're done */
        if (lst == [])
            return lst;

        /* create a vector to hold the verification results */        
        results = new Vector(lst.length());

        /*
         *   Call the given verifier method on each object, noting each
         *   result.
         */
        foreach (local cur in lst)
        {
            local curResult;

            /* call the verifier method and note the current result */
            curResult = callVerifyProp(cur.obj_, verProp, preCondProp,
                                       remapProp, nil, whichObj);

            /* 
             *   save the ResolveInfo in the verify result list object, so
             *   that we can figure out later (after sorting the results)
             *   which original ResolveInfo this verification result
             *   applies to 
             */
            curResult.obj_ = cur;

            /* add this verify result to our result vector */
            results.append(curResult);
        }

        /* 
         *   Sort the results in descending order of logicalness, and
         *   return the sorted list.
         */
        return results.toList().sort(SortDesc, {x, y: x.compareTo(y)});
    }

    /*
     *   Filter an ambiguous object list using the given verification
     *   method.  We call the given verification method on each object,
     *   noting the result, then find the best (most logical) result in
     *   the list.  We reduce the set to the objects that all have the
     *   same best value - everything else in the list is less logical, so
     *   we discard it.  This gives us a set of objects that are all of
     *   equivalent likelihood and all of the best likelihood of all the
     *   objects.
     *   
     *   This is the typical way that we disambiguate a list of objects,
     *   but this is merely a service routine, so individual actions can
     *   choose to use this or other mechanisms as appropriate.  
     */
    filterAmbiguousWithVerify(lst, requiredNum, verProp,
                              preCondProp, remapProp, whichObj)
    {
        local results;
        local discards;
        local keepers;
        local bestResult;
        local uniqueKeepers;

        /* if there's nothing in the list, there's nothing to do */
        if (lst == [])
            return lst;

        /* 
         *   Call the verifier method on each object, and sort the results
         *   from best to worst.  
         */
        results = getSortedVerifyResults(lst, verProp, preCondProp,
                                         remapProp, whichObj);

        /* note the best result value */
        bestResult = results[1];

        /* 
         *   Reduce the list to the most logical elements - in other
         *   words, keep only the elements that are exactly as logical as
         *   the first element, which we know to have the best logicalness
         *   ranking in the list by virtue of having sorted the list in
         *   descending order of logicalness.  
         */
        keepers = results.subset({x: x.compareTo(bestResult) == 0});

        /*
         *   Count the number of unique keepers - this is the number of
         *   items in the keepers list that don't have any equivalents in
         *   the keepers list.
         *   
         *   To calculate this number, start with the total number of
         *   items in the list, and reduce it by one for each item with an
         *   earlier equivalent in the list.  Note that we only ignore
         *   items with unique equivalents *earlier* in the list so that
         *   we keep exactly one of each equivalent - if we ignored every
         *   element with a unique equivalent elsewhere in the list, we'd
         *   ignore every equivalent item, so we'd only count the items
         *   with no equivalents at all.  
         */
        uniqueKeepers = keepers.length();
        for (local i = 1, local cnt = keepers.length() ; i <= cnt ; ++i)
        {
            local eqIdx;
            
            /* check for a unique equivalent earlier in the list */
            eqIdx = keepers.indexWhich(
                {x: x.obj_.obj_.isVocabEquivalent(keepers[i].obj_.obj_)});
            if (eqIdx != nil && eqIdx < i)
            {
                /* 
                 *   this one has an earlier equivalent, so don't include
                 *   it in the unique item count 
                 */
                --uniqueKeepers;
            }
        }

        /*
         *   If we found more items to keep than were required by the
         *   caller, we were not able to reduce the set to an unambiguous
         *   subset.  In this case, keep *all* of the items that are
         *   logical. 
         */
        if (uniqueKeepers > requiredNum)
        {
            local allAllowed;
            
            /* filter so that we keep all of the logical results */
            allAllowed = results.subset({x: x.allowAction});

            /* if that list is non-empty, use it as the keepers */
            if (allAllowed.length() != 0)
                keepers = allAllowed;
        }

        /* 
         *   Get the list of discards - this is the balance of the
         *   original result list after removing the best ones.  
         */
        discards = results - keepers;

        /* 
         *   We now have the set of objects we want, but the entries in
         *   the list are all VerifyResultList instances, and we just want
         *   the objects - pull out a list of just the ResolveInfo, and
         *   return that.  
         */
        keepers = keepers.mapAll({x: x.obj_});

        /*
         *   Check to see if we should set flags in the results.  If we
         *   eliminated any objects, flag the remaining objects as having
         *   been selected through disambiguation.  If the best of the
         *   discarded objects were logical, flag the survivors as
         *   "unclear," because we only selected them as better than the
         *   discards, and not because they were the only possible
         *   choices.  
         */
        if (discards != [])
        {
            local addedFlags;
            
            /* 
             *   We have reduced the set.  If the best of the discards was
             *   ranked as highly as the survivors at the coarsest level,
             *   flag the survivors as having been "unclearly"
             *   disambiguated; otherwise, mark them as "clearly"
             *   disambiguated.  
             */
            if (keepers.indexOf(bestResult.obj_) != nil
                && (discards[1].getEffectiveResult().resultRank ==
                    bestResult.getEffectiveResult().resultRank))
            {
                /* 
                 *   we had to make a choice that discarded possibilities
                 *   that were valid, though not as good as the one we
                 *   chose - mark the objects as being not perfectly clear 
                 */
                addedFlags = UnclearDisambig;

                /* 
                 *   if the keepers and the rejects are all basic
                 *   equivalents, don't bother flagging this as unclear,
                 *   since there's no point in mentioning that we chose
                 *   one basic equivalent over another, as they all have
                 *   the same name 
                 */
                if (BasicResolveResults.filterWithDistinguisher(
                    keepers + discards.mapAll({x: x.obj_}),
                    basicDistinguisher).length() == 1)
                {
                    /* they're all basic equivalents - mark as clear */
                    addedFlags = ClearDisambig;
                }
            }
            else
            {
                /* the choice is clear */
                addedFlags = ClearDisambig;
            }

            /* add the flags to each survivor */
            foreach (local cur in keepers)
                cur.flags_ |= addedFlags;
        }

        /* return the results */
        return keepers;
    }

    /*
     *   Filter a plural list with a verification method.  We'll reduce
     *   the list to the subset of objects that verify as logical, if
     *   there are any.  If there are no logical objects in the list,
     *   we'll simply return the entire original list.  
     */
    filterPluralWithVerify(lst, verProp, preCondProp, remapProp, whichObj)
    {
        local results;

        /* if there's nothing in the list, there's nothing to do */
        if (lst == [])
            return lst;

        /* 
         *   Call the verifier method on each object, and sort the results
         *   from best to worst.  
         */
        results = getSortedVerifyResults(lst, verProp, preCondProp,
                                         remapProp, whichObj);

        /*
         *   If the best (and thus first) result allows the action, filter
         *   the list to keep only the elements that are equally as
         *   logical as the first.  Otherwise, there are no allowed
         *   results, so return the original list.  
         */
        if (results[1].allowAction)
        {
            local bestResult;

            /* 
             *   the results are ordered in descending logicalness, so the
             *   first result in the list is the best one 
             */
            bestResult = results[1];

            /* filter to keep only the items as good as the best result */
            results = results.subset({x: x.compareTo(bestResult) == 0});
        }

        /* return the resolve results objects from the list */
        return results.mapAll({x: x.obj_});
    }

    /*
     *   Get a default object using the given verification method.  We'll
     *   start with the 'all' list, then use the verification method to
     *   reduce the list to the most likely candidates.  If we find a
     *   unique most likely candidate, we'll return a ResolveInfo list
     *   with that result; otherwise, we'll return nothing, since there is
     *   no suitable default.  
     */
    getDefaultWithVerify(resolver, verProp, preCondProp, remapProp, whichObj)
    {
        local lst;
        local results;
        local bestResult;
        
        /* 
         *   Start with the 'all' list for this noun phrase.  This is the
         *   set of every object that we consider obviously suitable for
         *   the command, so it's a good starting point to guess at a
         *   default object. 
         */
        lst = resolver.getAllDefaults();

        /* 
         *   reduce equivalent items to a single instance of each
         *   equivalent - if we have several equivalent items that are
         *   equally good as defaults, we can pick just one 
         */
        lst = resolver.filterAmbiguousEquivalents(lst);

        /* if we have no entries in the list, there is no default */
        if (lst == [])
            return nil;

        /* 
         *   get the verification results for these objects, sorted from
         *   best to worst 
         */
        results = getSortedVerifyResults(lst, verProp, preCondProp,
                                         remapProp, whichObj);

        /* note the best result */
        bestResult = results[1];

        /* 
         *   if the best item is not allowed as an implied object, there
         *   is no default 
         */
        if (!bestResult.allowImplicit)
            return nil;

        /* 
         *   The best item must be uniquely logical in order to be a
         *   default - if more than one item is equally good, it makes no
         *   sense to assume anything about what the user meant.  So, if
         *   we have more than one item, and the second item is equally as
         *   logical as the first item, we cannot supply a default.  (The
         *   second item cannot be better than the first item, because of
         *   the sorting - at most, it can be equally good.)  
         */
        if (results.length() != 1 && bestResult.compareTo(results[2]) == 0)
            return nil;

        /* 
         *   We have a uniquely logical item, so we can assume the user
         *   must have been referring to this item.  Return the
         *   ResolveInfo for this item (which the verify result sorter
         *   stashed in the obj_ property of the verify result object).
         *   
         *   Before returning the list, clear the 'all' flag in the result
         *   (getAll() set that flag), and replace it with the 'default'
         *   flag, since the object is an implied default.  
         */
        bestResult.obj_.flags_ &= ~MatchedAll;
        bestResult.obj_.flags_ |= DefaultObject;

        /* return the result list */
        return [bestResult.obj_];
    }

    /* 
     *   List of objects that verified okay on a prior pass.  This is a
     *   scratch-pad for use by verifier routines, to keep track of work
     *   they've already done.  A few verifiers use this as a way to
     *   detect when an implicit action actually finished the entire job,
     *   which would in many cases result in a verify failure if not
     *   checked (because a command that effects conditions that already
     *   hold is normally illogical); by tracking that the verification
     *   previously succeeded, the verifier can know that the action
     *   should be allowed to proceed and do nothing.  
     */
    verifiedOkay = []
;

/*
 *   Call the roomBeforeAction method on a given room's containing rooms,
 *   then on the room itself.  
 */
callRoomBeforeAction(room)
{
    /* first, call roomBeforeAction on the room's containers */
    room.forEachContainer(callRoomBeforeAction);

    /* call roomBeforeAction on this room */
    room.roomBeforeAction();
}

/*
 *   Call the roomAfterAction method on a given room, then on the room's
 *   containing rooms.  
 */
callRoomAfterAction(room)
{
    /* first, call roomAfterAction on this room */
    room.roomAfterAction();

    /* next, call roomAfterAction on the room's containers */
    room.forEachContainer(callRoomAfterAction);
}

/* ------------------------------------------------------------------------ */
/*
 *   Intransitive Action class - this is an action that takes no objects. 
 */
class IAction: Action
    /* 
     *   resolve my noun phrases to objects 
     */
    resolveNouns(issuingActor, targetActor, results)
    {
        /* we have no objects to resolve, so there's nothing to do */
        return;
    }

    /*
     *   Execute the action.  
     */
    doActionMain()
    {
        /* 
         *   we have no objects to iterate, so simply run through the
         *   execution sequence once 
         */
        doActionOnce();
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Transitive Action class - this is an action that takes a direct
 *   object.
 *   
 *   For simplicity, this object is its own object resolver - we really
 *   don't need a separate resolver object because we have only one object
 *   list for this verb.  (In contrast, an action with both a direct and
 *   indirect object might need separate resolution rules for the two
 *   objects, and hence would need separate resolver objects for the two.)
 *   
 *   The advantage of implementing the Resolver behavior in this object,
 *   rather than using a separate object, is that it's less trouble to
 *   override object resolution rules - simply override the resolver
 *   methods in the subclass where you define the grammar rule for the
 *   action.  
 */
class TAction: Action, Resolver
    construct()
    {
        /* inherit only the Action constructor */
        inherited Action.construct();
    }

    resetAction()
    {
        /* inherit default handling */
        inherited();
        
        /* discard our cached resolver */
        dobjResolver_ = nil;
    }
    
    /*
     *   Create an action for retrying an original action with changes. 
     */
    createForRetry(orig)
    {
        local action;
        
        /* create the new action based on the original action */
        action = createActionFrom(orig);

        /* 
         *   do not include the new command in undo, since it's merely
         *   part of the enclosing explicit command 
         */
        action.includeInUndo = nil;

        /* mark the action as nested */
        action.setNested();

        /* 
         *   set the original command's action time to zero - the original
         *   command's only action is to invoke the new command, so we
         *   only want to count the time of the new command 
         */
        orig.actionTime = 0;

        /*
         *   Even if the original action is implicit, don't announce this
         *   action as implicit, because it's good enough to have
         *   announced the original implicit action.  Now, this new
         *   command actually still is implicit if the original was - we
         *   simply don't want to announce it as such.  To suppress the
         *   extra announcement while still retaining the rest of the
         *   desired implicitness, simply set the implicitMsg property of
         *   the new action to nil; when there's no implicitMsg, there's
         *   no announcement. 
         */
        action.implicitMsg = nil;

        /* return the new action */
        return action;
    }

    /*
     *   Retry an intransitive action as a single-object action.  We'll
     *   obtain a indirect object using the normal means (first looking
     *   for a default, then prompting the player if we can't find a
     *   suitable default).  'orig' is the original zero-object action.
     *   
     *   This routine terminates with 'exit' if it doesn't throw some
     *   other error.  
     */
    retryWithMissingDobj(orig)
    {
        local action;

        /* create the action for a retry */
        action = createForRetry(orig);

        /* use an empty noun phrase for the new action's direct object */
        action.dobjMatch = new EmptyNounPhraseProd();
        action.dobjMatch.responseProd = action.askDobjResponseProd;

        /* 
         *   execute the new command, using the same target and issuing
         *   actors as the original action 
         */
        executeAction(gActor, nil, gIssuingActor, nil, action);

        /* the invoking command is done */
        exit;
    }

    /*
     *   The root production to use to parse missing direct object
     *   responses.  By default, this is nounList, but individual actions
     *   can override this as appropriate.
     *   
     *   Note that language modules might want to override this to allow
     *   for special responses.  For example, in English, some verbs might
     *   want to override this with a specialized production that allows
     *   the appropriate preposition in the response.  
     */
    askDobjResponseProd = nounList

    /*
     *   Resolve objects.  This is called at the start of command
     *   execution to resolve noun phrases in the command to specific
     *   objects.  
     */
    resolveNouns(issuingActor, targetActor, results)
    {
        /* 
         *   Ask the direct object noun phrase list to resolve itself, and
         *   store the resulting object list.  Since we're starting a
         *   resolution pass through our objects, reset the resolver if
         *   we're reusing it.  
         */
        dobjList_ = dobjMatch.resolveNouns(
            getDobjResolver(issuingActor, targetActor, true), results);
    }

    /* get the role of an object */
    getRoleFromIndex(idx)
    {
        /* we only take a single object role - the direct object */
        return (idx == 1 ? DirectObject : inherited(idx));
    }

    /* get the resolved object in a given role */
    getObjectForRole(role)
    {
        /* return the direct object if requested */
        return (role == DirectObject ? getDobj() : inherited(role));
    }

    /* get the match tree for the noun phrase in the given role */
    getMatchForRole(role)
    {
        /* return the direct object match tree if requested */
        return (role == DirectObject ? dobjMatch : inherited(role));
    }

    /* get the 'verify' property for a given object role */
    getVerifyPropForRole(role)
    {
        return (role == DirectObject ? verDobjProp : inherited(role));
    }

    /* get the 'preCond' property for a given object role */
    getPreCondPropForRole(role)
    {
        return (role == DirectObject ? preCondDobjProp : inherited(role));
    }

    /* get the 'remap' property for a given object role */
    getRemapPropForRole(role)
    {
        return (role == DirectObject ? remapDobjProp : inherited(role));
    }

    /* manually set the resolved objects - we'll set the direct object */
    setResolvedObjects(dobj)
    {
        /* 
         *   set the direct object tree to a fake grammar tree that
         *   resolves to our single direct object, in case we're asked to
         *   resolve explicitly 
         */
        dobjMatch = new PreResolvedProd(dobj);

        /* 
         *   Set the resolved direct object list to the single object or
         *   to the list of objects, depending on what we received.
         */
        dobjList_ = makeResolveInfoList(dobj);
    }

    /* manually set the unresolved object noun phrase match trees */
    setObjectMatches(dobj)
    {
        /* note the new direct object match tree */
        dobjMatch = dobj;
    }

    /* check that the resolved objects are in scope */
    resolvedObjectsInScope()
    {
        /* check the direct object */
        return getDobjResolver(gIssuingActor, gActor, true)
            .objInScope(dobjList_[1].obj_);
    }

    /*
     *   Get a message parameter object for the action.  We define 'dobj'
     *   as the direct object, in addition to any inherited targets.  
     */
    getMessageParam(objName)
    {
        switch(objName)
        {
        case 'dobj':
            /* return the current direct object */
            return dobjCur_;

        default:
            /* inherit default handling */
            return inherited(objName);
        }
    }

    /*
     *   Execute the action.  We'll run through the execution sequence
     *   once for each resolved direct object.  
     */
    doActionMain()
    {
        /* 
         *   Set the direct object list as the antecedent, using the
         *   game-specific pronoun setter.  Don't set pronouns for a
         *   nested command, because the player didn't necessarily refer
         *   to the objects in a nested command.  
         */
        if (parentAction == nil)
            setPronoun(dobjList_);

        /* run through the sequence once for each direct object */
        for (local i = 1, local len = dobjList_.length() ; i <= len ; ++i)
        {
            /* make this object our current direct object */
            dobjCur_ = dobjList_[i].obj_;

            /* announce the object if appropriate */
            announceActionObject(dobjList_[i], len, whichObject);

            /* run the execution sequence for the current direct object */
            doActionOnce();

            /* if we're top-level, count the iteration in the transcript */
            if (parentAction == nil)
                gTranscript.newIter();
        }
    }

    /*
     *   Check any pre-conditions for the action.  We'll check our
     *   action-level conditions, plus any preconditions associated with
     *   the direct object. 
     */
    checkPreConditions(allowImplicit)
    {
        local ret1;
        local ret2;
        
        /* inherit default handling to check our action-level conditions */
        ret1 = inherited(allowImplicit);

        /* check conditions on the direct object, if any */
        ret2 = callPreConditions(dobjCur_, preCondDobjProp, dobjCur_,
                                 allowImplicit, DirectObject);

        /* return true if any implicit commands were executed */
        return ret1 || ret2;
    }

    /*
     *   Get the list of active objects.  We have only a direct object, so
     *   we'll return a list with the current direct object. 
     */
    getCurrentObjects()
    {
        return [dobjCur_];
    }

    /* set the current objects */
    setCurrentObjects(lst)
    {
        dobjCur_ = lst[1];
    }

    /*
     *   Verify the action.
     */
    verifyAction()
    {
        local result;
        
        /* invoke any general (non-object) pre-condition verifiers */
        result = callVerifyPrecond(nil);

        /* 
         *   invoke the verifier routine ("verXxx") on the current direct
         *   object and return the result
         */
        result = callVerifyProp(dobjCur_, verDobjProp, preCondDobjProp,
                                remapDobjProp, result, DirectObject);

        /* verify that the action routine ("doXxx") exists on the object */
        return verifyActionProp([dobjCur_], [actionDobjProp], result);
    }

    /* initialize tentative resolutions for other noun phrases */
    initTentative(issuingActor, targetActor, whichObj)
    {
        /* 
         *   we have only one noun phrase, so there's nothing else to
         *   tentatively resolve 
         */
    }

    /*
     *   Check for remapping 
     */
    checkRemapping()
    {
        local remapInfo;
        
        /* check for remapping in the direct object */
        if ((remapInfo = dobjCur_.(remapDobjProp)) != nil)
        {
            /* 
             *   we have a remapping, so apply it - note that this won't
             *   return, since this will completely replace the command
             *   and thus terminate the old command with 'exit' 
             */
            remapAction(nil, DirectObject, remapInfo);
        }
    }

    /*
     *   Execute the command. 
     */
    execAction()
    {
        /* call the catch-all 'check' properties */
        callCatchAllProp(dobjCur_, checkDobjProp,
                         &checkDobjDefault, &checkDobjAll);

        /* call the direct object's check routine (its "checkXxx") method */
        dobjCur_.(checkDobjProp)();

        /* call the catch-all 'action' properties */
        callCatchAllProp(dobjCur_, actionDobjProp,
                         &actionDobjDefault, &actionDobjAll);

        /* call the direct object's action routine (its "doXxx" method) */
        dobjCur_.(actionDobjProp)();
    }

    /*
     *   The direct object preconditions, verifier, remapper, check, and
     *   action methods for this action.  Each concrete action must define
     *   these appropriately.  By convention, the methods are named like
     *   so:
     *   
     *.  preconditions: preCondDobjAction
     *.  verifier: verDobjAction
     *.  remap: remapDobjAction
     *.  check: checkDobjAction
     *.  action: actionDobjAction
     *   
     *   where the 'Action' suffix is replaced by the name of the action.
     *   The DefineTAction macro applies this convention, so in most cases
     *   library and game authors will never have to create all of those
     *   property names manually.  
     */
    verDobjProp = nil
    preCondDobjProp = nil
    remapDobjProp = nil
    checkDobjProp = nil
    actionDobjProp = nil

    /*
     *   Get my direct object resolver.  If I don't already have one,
     *   create one and cache it; if I've already cached one, return it.
     *   Note that we cache the resolver because it can sometimes take a
     *   bit of work to set one up (the scope list can in some cases be
     *   complicated to calculate).  We use the resolver only during the
     *   object resolution phase; since game state can't change during
     *   this phase, it's safe to keep a cached copy.  
     */
    getDobjResolver(issuingActor, targetActor, reset)
    {
        /* create a new resolver if we don't already have one cached */
        if (dobjResolver_ == nil)
            dobjResolver_ = createDobjResolver(issuingActor, targetActor);

        /* reset the resolver if desired */
        if (reset)
            dobjResolver_.resetResolver();

        /* return it */
        return dobjResolver_;
    }

    /*
     *   Create a resolver for the direct object.  By default, we are our
     *   own resolver.  Some actions might want to override this to create
     *   and return a specialized resolver instance if special resolution
     *   rules are needed.  
     */
    createDobjResolver(issuingActor, targetActor)
    {
        /* initialize myself as a resolver */
        initResolver(issuingActor, targetActor);

        /* return myself */
        return self;
    }

    /*
     *   Resolve 'all' for the direct object, given a list of everything
     *   in scope.  By default, we'll simply return everything in scope;
     *   some actions might want to override this to return a more
     *   specific list of objects suitable for 'all'.  
     */
    getAllDobj(actor, scopeList)
    {
        return scopeList;
    }

    /* filter an ambiguous direct object noun phrase */
    filterAmbiguousDobj(lst, requiredNum)
    {
        /* filter using the direct object verifier method */
        return filterAmbiguousWithVerify(lst, requiredNum, verDobjProp,
                                         preCondDobjProp, remapDobjProp,
                                         DirectObject);
    }

    /* filter a plural phrase */
    filterPluralDobj(lst)
    {
        /* filter using the direct object verifier method */
        return filterPluralWithVerify(lst, verDobjProp, preCondDobjProp,
                                      remapDobjProp, DirectObject);
    }

    /* get the default direct object */
    getDefaultDobj(resolver)
    {
        /* get a default direct object using the verify method */
        return getDefaultWithVerify(resolver, verDobjProp, preCondDobjProp,
                                    remapDobjProp, DirectObject);
    }

    /* get the current direct object of the command */
    getDobj() { return dobjCur_; }

    /* the predicate must assign the direct object production tree here */
    dobjMatch = nil

    /* my resolved list of direct objects */
    dobjList_ = []

    /* 
     *   The resolved direct object on which we're currently executing the
     *   command.  To execute the command, we iterate through the direct
     *   object list, calling the execution sequence for each object in
     *   the list.  We set this to the current object in each iteration.  
     */
    dobjCur_ = nil

    /* my cached direct object resolver */
    dobjResolver_ = nil

    /* -------------------------------------------------------------------- */
    /*
     *   Resolver interface implementation - for the moment, we don't need
     *   any special definitions here, since the basic Resolver
     *   implementation (which we inherit) is suitable for a single-object
     *   action.  
     */

    /* -------------------------------------------------------------------- */
    /*
     *   private Resolver implementation details 
     */

    /*
     *   Initialize me as a resolver.  
     */
    initResolver(issuingActor, targetActor)
    {
        /* cache the actor's default scope list */
        scope_ = targetActor.scopeList();

        /* remember the actors */
        issuer_ = issuingActor;
        actor_ = targetActor;

        /* I'm the action as well as the resolver */
        action_ = self;
    }

    /* resolver scope list */
    scope_ = []

    /* issuing actor */
    issuer_ = nil

    /* target actor */
    actor_ = nil

    /* 
     *   By default, our resolver is for a direct object.  This is always
     *   the case for a one-object action, but we make this overridable so
     *   that we're a more flexible base class for multi-object actions. 
     */
    whichObject = DirectObject
;


/* ------------------------------------------------------------------------ */
/*
 *   "Tentative" noun resolver results gather.  This type of results
 *   gatherer is used to perform a tentative pre-resolution of an object
 *   of a multi-object action.
 *   
 *   Consider what happens when we resolve a two-object action, such as
 *   "put <dobj> in <iobj>".  Since we have two objects, we obviously must
 *   resolve one object or the other first; but this means that we must
 *   resolve one object with no knowledge of the resolution of the other
 *   object.  This often makes it very difficult to resolve that first
 *   object intelligently, because we'd really like to know something
 *   about the other object.  For example, if we first resolve the iobj of
 *   "put <dobj> in <iobj>", it would be nice to know which dobj we're
 *   talking about, since we could reduce the likelihood that the iobj is
 *   the dobj's present container.
 *   
 *   Tentative resolution addresses this need by giving us some
 *   information about a later-resolved object while resolving an
 *   earlier-resolved object, even though we obviously can't have fully
 *   resolved the later-resolved object.  In tentative resolution, we
 *   perform the resolution of the later-resolved object, completely in
 *   the dark about the earlier-resolved object(s), and come up with as
 *   much information as we can.  The important thing about this stage of
 *   resolution is that we don't ask any interactive questions and we
 *   don't count anything for ranking purposes - we simply do the best we
 *   can and note the results, leaving any ranking or interaction for the
 *   true resolution phase that we'll perform later.  
 */
class TentativeResolveResults: ResolveResults
    construct(target, issuer) { setActors(target, issuer); }

    /* 
     *   ignore most resolution problems, since this is only a tentative
     *   resolution pass 
     */
    noMatch(txt) { }
    noVocabMatch(txt) { }
    noMatchForAll() { }
    noteEmptyBut() { }
    noMatchForAllBut() { }
    noMatchForListBut() { }
    noMatchForPronoun(typ, txt) { }
    noMatchForPossessive(owner, txt) { }
    noMatchForLocation(loc, txt) { }
    noteBadPrep() { }
    nothingInLocation(loc) { }
    unknownNounPhrase(match, resolver) { return []; }
    noteLiteral(txt) { }
    emptyNounPhrase(resolver) { return []; }
    zeroQuantity(txt) { }
    insufficientQuantity(txt, matchList, requiredNum) { }
    uniqueObjectRequired(txt, matchList) { }
    noteAdjEnding() { }
    noteMiscWordList() { }
    notePronoun() { }
    noteMatches(matchList) { }
    incCommandCount() { }
    noteActorSpecified() { }
    allowActionRemapping = nil

    /* 
     *   during the tentative phase, keep all equivalents - we don't want
     *   to make any arbitrary choices among equivalents during this
     *   phase, because doing so could improperly force a choice among
     *   otherwise ambiguous resolutions to the other phrase 
     */
    allowEquivalentFiltering = nil

    /* 
     *   for ambiguous results, don't attempt to narrow things down - just
     *   keep the entire list 
     */
    ambiguousNounPhrase(keeper, txt, matchList, fullMatchList, scopeList,
                        requiredNum, resolver)
    {
        return matchList;
    }

    /* 
     *   no interaction is allowed, so return nothing if we need to ask
     *   for a missing object 
     */
    askMissingObject(resolver, responseProd)
    {
        /* note that we have a missing noun phrase */
        npMissing = true;

        /* return nothing */
        return nil;
    }

    /* 
     *   no interaction is allowed, so return no tokens if we need to ask
     *   for a literal 
     */
    askMissingLiteral(action, which) { return []; }

    /* no interaction is allowed during tentative resolution */
    canResolveInteractively() { return nil; }

    /* 
     *   flag: the noun phrase we're resolving is a missing noun phrase,
     *   which means that we'll ask for it to be filled in when we get
     *   around to resolving it for real 
     */
    npMissing = nil
;

/*
 *   A dummy object that we use for the *tentative* resolution of a noun
 *   phrase when the noun phrase doesn't match anything.  This lets us
 *   distinguish cases where we have a noun phrase that has an error from a
 *   noun phrase that's simply missing.  
 */
dummyTentativeObject: object
;

dummyTentativeInfo: ResolveInfo
    obj_ = dummyTentativeObject
    flags_ = 0
;

/* ------------------------------------------------------------------------ */
/* 
/*
 *   Transitive-with-indirect Action class - this is an action that takes
 *   both a direct and indirect object.  We subclass the basic one-object
 *   action to add the indirect object handling.  
 */
class TIAction: TAction
    resetAction()
    {
        /* inherit defaulting handling */
        inherited();

        /* discard our cached iobj resolver */
        iobjResolver_ = nil;
    }
    
    /*
     *   Retry a single-object action as a two-object action.  We'll treat
     *   the original action's direct object list as our direct object
     *   list, and obtain an indirect object using the normal means (first
     *   looking for a default, then prompting the player if we can't find
     *   a suitable default).  'orig' is the original single-object action.
     *   
     *   This routine terminates with 'exit' if it doesn't throw some
     *   other error.  
     */
    retryWithMissingIobj(orig)
    {
        local action;
        
        /* create the new action based on the original action */
        action = createForRetry(orig);

        /* 
         *   take the original action's direct object list as the new
         *   action's direct object list 
         */
        action.dobjMatch = new PreResolvedProd(orig.getDobj());

        /* use an empty noun phrase for the new action's indirect object */
        action.iobjMatch = new EmptyNounPhraseProd();
        action.iobjMatch.responseProd = action.askIobjResponseProd;

        /* 
         *   execute the new command, using the same target and issuing
         *   actors as the original action 
         */
        executeAction(gActor, nil, gIssuingActor, nil, action);

        /* the invoking command is done */
        exit;
    }

    /*
     *   The root production to use to parse missing indirect object
     *   responses.  By default, this is singleNoun, but individual
     *   actions can override this as appropriate.
     *   
     *   Note that language modules might want to override this to allow
     *   for special responses.  For example, in English, most verbs will
     *   want to override this with a specialized production that allows
     *   the appropriate preposition in the response.  
     */
    askIobjResponseProd = singleNoun

    /*
     *   Resolution order - returns DirectObject or IndirectObject to
     *   indicate which noun phrase to resolve first in resolveNouns().
     *   By default, we'll resolve the indirect object first, but
     *   individual actions can override this to resolve in a non-default
     *   order.  
     */
    resolveFirst = IndirectObject

    /*
     *   Empty phrase resolution order.  This is similar to the standard
     *   resolution order (resolveFirst), but is used only when both the
     *   direct and indirect objects are empty phrases.
     *   
     *   When both phrases are empty, we will either use a default or
     *   prompt interactively for the missing phrase.  In most cases, it
     *   is desirable to prompt interactively for a missing direct object
     *   first, regardless of the usual resolution order.  
     */
    resolveFirstEmpty = DirectObject
    
    /*
     *   Determine which object to call first for action processing.  By
     *   default, we execute in the same order as we resolve, but this can
     *   be overridden if necessary.  
     */
    execFirst = (resolveFirst)

    /*
     *   resolve our noun phrases to objects 
     */
    resolveNouns(issuingActor, targetActor, results)
    {
        local first;
        local objMatch1, objMatch2;
        local objList1, objList2;
        local getResolver1, getResolver2;
        local objCur1;
        local remapProp;

        /* we have no current known direct or indirect object yet */
        dobjCur_ = nil;
        iobjCur_ = nil;

        /* 
         *   Determine which object we want to resolve first.  If both
         *   phrases are empty, use the special all-empty ordering;
         *   otherwise, use the standard ordering for this verb.  
         */
        if (dobjMatch.isEmptyPhrase && iobjMatch.isEmptyPhrase)
        {
            /* both phrases are empty - use the all-empty ordering */
            first = resolveFirstEmpty;
        }
        else
        {
            /* 
             *   we have at least one non-empty phrase, so use our
             *   standard ordering 
             */
            first = resolveFirst;
        }

        /*
         *   The resolution process is symmetrical for the two possible
         *   resolution orders (direct object first or indirect object
         *   first); all we need to do is to set up the parameters we'll
         *   need according to which order we're using.
         *   
         *   This parameterized approach makes the code further below a
         *   little mind-boggling, because it's using so much indirection.
         *   But the alternative is worse: the alternative is to
         *   essentially make two copies of the code below (one for the
         *   dobj-first case and one for the iobj-first case), which is
         *   prone to maintenance problems because of the need to keep the
         *   two copies synchronized when making any future changes.  
         */
        if (first == DirectObject)
        {
            objMatch1 = dobjMatch;
            objMatch2 = iobjMatch;
            objList1 = &dobjList_;
            objList2 = &iobjList_;
            getResolver1 = &getDobjResolver;
            getResolver2 = &getIobjResolver;
            objCur1 = &dobjCur_;
            remapProp = remapDobjProp;
        }
        else
        {
            objMatch1 = iobjMatch;
            objMatch2 = dobjMatch;
            objList1 = &iobjList_;
            objList2 = &dobjList_;
            getResolver1 = &getIobjResolver;
            getResolver2 = &getDobjResolver;
            objCur1 = &iobjCur_;
            remapProp = remapIobjProp;
        }

        /* 
         *   Get the unfiltered second-resolved object list - this will
         *   give the first-resolved object resolver access to some
         *   minimal information about the possible second-resolved object
         *   or objects.
         *   
         *   Note that we're not *really* resolving the second object here
         *   - it is the second-resolved object, after all.  What we're
         *   doing is figuring out the *potential* set of objects that
         *   could resolve to the second phrase, with minimal
         *   disambiguation, so that the first object resolver is not
         *   totally in the dark about the second object's potential
         *   resolution.  
         */
        initTentative(issuingActor, targetActor, first);
            
        /* resolve the first-resolved object */
        self.(objList1) = objMatch1.resolveNouns(
            self.(getResolver1)(issuingActor, targetActor, true), results);

        /* 
         *   if the first-resolved phrase resolves to just one object, we
         *   can immediately set the current resolved object, so that we
         *   can use it while resolving the second-resolved object list 
         */
        if (self.(objList1).length() == 1)
        {
            /* set the current first-resolved object */
            self.(objCur1) = self.(objList1)[1].obj_;

            /* if remapping is allowed at this point, look for a remapping */
            if (results.allowActionRemapping)
            {
                withParserGlobals(targetActor, self, new function()
                {
                    local remapInfo;
                    
                    /* check for a remapping */
                    if ((remapInfo = self.(objCur1).(remapProp)) != nil)
                    {
                        /* 
                         *   we have a remapping, so apply it - note that
                         *   we're still in the process of resolving noun
                         *   phrases (since we've only resolved one of our
                         *   two noun phrases so far), so pass 'true' for
                         *   the inResolve parameter 
                         */
                        remapAction(true, first, remapInfo);
                    }
                });
            }
        }

        /* resolve the second-resolved object */
        self.(objList2) = objMatch2.resolveNouns(
            self.(getResolver2)(issuingActor, targetActor, true), results);
    }

    /* get an object role */
    getRoleFromIndex(idx)
    {
        /* 
         *   the second object is always our indirect object; for other
         *   roles, defer to the inherited behavior 
         */
        return (idx == 2 ? IndirectObject : inherited(idx));
    }

    /* get the resolved object in a given role */
    getObjectForRole(role)
    {
        /* return the indirect object if requested; otherwise inherit */
        return (role == IndirectObject ? getIobj() : inherited(role));
    }

    /* get the match tree for the noun phrase in the given role */
    getMatchForRole(role)
    {
        /* return the indirect object match tree if requested; else inherit */
        return (role == IndirectObject ? iobjMatch : inherited(role));
    }

    /* get the 'verify' property for a given object role */
    getVerifyPropForRole(role)
    {
        return (role == IndirectObject ? verIobjProp : inherited(role));
    }

    /* get the 'preCond' property for a given object role */
    getPreCondPropForRole(role)
    {
        return (role == IndirectObject ? preCondIobjProp : inherited(role));
    }

    /* get the 'remap' property for a given object role */
    getRemapPropForRole(role)
    {
        return (role == IndirectObject ? remapIobjProp : inherited(role));
    }

    /*
     *   Manually set the resolved objects.  We'll set our direct and
     *   indirect objects.  
     */
    setResolvedObjects(dobj, iobj)
    {
        /* inherit the base class handling to set the direct object */
        inherited(dobj);

        /* build a pre-resolved production for the indirect object */
        iobjMatch = new PreResolvedProd(iobj);

        /* set the resolved indirect object */
        iobjList_ = makeResolveInfoList(iobj);
    }

    /* manually set the unresolved object noun phrase match trees */
    setObjectMatches(dobj, iobj)
    {
        /* inherit default handling */
        inherited(dobj);
        
        /* note the new indirect object match tree */
        iobjMatch = iobj;
    }

    /* check that the resolved objects are in scope */
    resolvedObjectsInScope()
    {
        /* check the direct and indirect objects */
        return (getDobjResolver(gIssuingActor, gActor, true)
                .objInScope(dobjList_[1].obj_))
            && (getIobjResolver(gIssuingActor, gActor, true)
                .objInScope(iobjList_[1].obj_));
    }

    /* 
     *   get our indirect object resolver, or create one if we haven't
     *   already cached one 
     */
    getIobjResolver(issuingActor, targetActor, reset)
    {
        /* if we don't already have one cached, create a new one */
        if (iobjResolver_ == nil)
            iobjResolver_ = createIobjResolver(issuingActor, targetActor);

        /* reset the resolver if desired */
        if (reset)
            iobjResolver_.resetResolver();
        
        /* return the cached resolver */
        return iobjResolver_;
    }

    /*
     *   Create our indirect object resolver.  By default, we'll use a
     *   basic indirect object resolver.
     */
    createIobjResolver(issuingActor, targetActor)
    {
        /* create and return a new basic indirect object resolver */
        return new IobjResolver(self, issuingActor, targetActor);
    }
    
    /* 
     *   Resolve 'all' for the indirect object.  By default, we'll return
     *   everything in the scope list.  
     */
    getAllIobj(actor, scopeList)
    {
        return scopeList;
    }

    /* filter an ambiguous indirect object noun phrase */
    filterAmbiguousIobj(lst, requiredNum)
    {
        /* filter using the indirect object verifier method */
        return filterAmbiguousWithVerify(lst, requiredNum, verIobjProp,
                                         preCondIobjProp, remapIobjProp,
                                         IndirectObject);
    }

    /* filter a plural phrase */
    filterPluralIobj(lst)
    {
        /* filter using the direct object verifier method */
        return filterPluralWithVerify(lst, verIobjProp, preCondIobjProp,
                                      remapIobjProp, IndirectObject);
    }

    /* get the default indirect object */
    getDefaultIobj(resolver)
    {
        /* get a default indirect object using the verify method */
        return getDefaultWithVerify(resolver, verIobjProp, preCondIobjProp,
                                    remapIobjProp, IndirectObject);
    }

    /*
     *   Execute the action.  We'll run through the execution sequence
     *   once for each resolved object in our direct or indirect object
     *   list, depending on which one is the list and which one is the
     *   singleton.  
     */
    doActionMain()
    {
        local lst;
        local preAnnouncedDobj;
        local preAnnouncedIobj;
        
        /* 
         *   Get the list of resolved objects for the multiple object.  If
         *   neither has multiple objects, it doesn't matter which is
         *   iterated, since we'll just do the command once anyway.  
         */
        lst = (iobjList_.length() > 1 ? iobjList_ : dobjList_);

        /* 
         *   Set the pronoun antecedents, using the game-specific pronoun
         *   setter.  Don't set an antecedent for a nested command.
         */
        if (parentAction == nil)
        {
            /*
             *   Set both direct and indirect objects as antecedents, in
             *   resolution order - this way, if both are antecedents for
             *   the same pronoun, the one that we resolve last will be
             *   the one that sticks, since the last antecedent set is
             *   always the winning one.  
             */
            if (resolveFirst == DirectObject)
            {
                setPronoun(dobjList_);
                setPronoun(iobjList_);
            }
            else
            {
                setPronoun(iobjList_);
                setPronoun(dobjList_);
            }
        }

        /* 
         *   pre-announce the non-list object if appropriate - this will
         *   provide a common pre-announcement if we iterate through
         *   several announcements of the main list objects 
         */
        if (lst == dobjList_)
        {
            /* pre-announce the single indirect object if needed */
            preAnnouncedIobj = preAnnounceActionObject(
                iobjList_[1], dobjList_, IndirectObject);

            /* we haven't announced the direct object yet */
            preAnnouncedDobj = nil;
        }
        else
        {
            /* pre-announce the single direct object if needed */
            preAnnouncedDobj = preAnnounceActionObject(
                dobjList_[1], iobjList_, DirectObject);

            /* we haven't announced the indirect object yet */
            preAnnouncedIobj = nil;
        }

        /* iterate over the resolved list for the multiple object */
        for (local i = 1, local len = lst.length() ; i <= len ; ++i)
        {
            local dobjInfo;
            local iobjInfo;

            /* 
             *   make the current list item the direct or indirect object,
             *   as appropriate 
             */
            if (lst == dobjList_)
            {
                /* the direct object is the multiple object */
                dobjInfo = lst[i];
                iobjInfo = iobjList_[1];
            }
            else
            {
                /* the indirect object is the multiple object */
                dobjInfo = dobjList_[1];
                iobjInfo = lst[i];
            }

            /* get the current dobj and iobj from the resolve info */
            dobjCur_ = dobjInfo.obj_;
            iobjCur_ = iobjInfo.obj_;

            /* 
             *   if the action was remapped, and we need to announce
             *   anything, announce the entire action 
             */
            if (isRemapped())
            {
                /*
                 *   We were remapped.  The entire phrasing of the new
                 *   action might have changed from what the player typed,
                 *   so it might be nonsensical to show the objects as we
                 *   usually would, as sentence fragments that are meant
                 *   to combine with what the player actually typed.  So,
                 *   instead of showing the usual sentence fragments, show
                 *   the entire phrasing of the command.
                 *   
                 *   Only show the announcement if we have a reason to: we
                 *   have unclear disambiguation in one of the objects, or
                 *   one of the objects is defaulted.  
                 */
                if (needRemappedAnnouncement(dobjInfo)
                    || needRemappedAnnouncement(iobjInfo))
                    gTranscript.announceRemappedAction();
            }
            else
            {
                /* announce the direct object if appropriate */
                if (!preAnnouncedDobj)
                    announceActionObject(dobjInfo, dobjList_.length(),
                                         DirectObject);

                /* announce the indirect object if appropriate */
                if (!preAnnouncedIobj)
                    announceActionObject(iobjInfo, iobjList_.length(),
                                         IndirectObject);
            }

            /* run the execution sequence for the current direct object */
            doActionOnce();

            /* if we're top-level, count the iteration in the transcript */
            if (parentAction == nil)
                gTranscript.newIter();
        }
    }

    /*
     *   Determine if we need to announce this action when the action was
     *   remapped, based on the resolution information for one of our
     *   objects.  We need to announce a remapped action when either
     *   object had unclear disambiguation or was defaulted. 
     */
    needRemappedAnnouncement(info)
    {
        /* 
         *   if it's a defaulted object that hasn't been announced, or it
         *   was unclearly disambiguated, we need an announcement 
         */
        return (((info.flags_ & DefaultObject) != 0
                 && (info.flags_ & AnnouncedDefaultObject) == 0)
                || (info.flags_ & UnclearDisambig) != 0);
    }

    /*
     *   Verify the action.
     */
    verifyAction()
    {
        local result;
        
        /* invoke any general (non-object) pre-condition verifiers */
        result = callVerifyPrecond(nil);

        /* check the direct object */
        result = callVerifyProp(dobjCur_, verDobjProp, preCondDobjProp,
                                remapDobjProp, result, DirectObject);

        /* 
         *   Check the indirect object, combining the results with the
         *   direct object results.  We combine the results for the two
         *   objects because we're simply looking for any reason that we
         *   can't perform the command.  
         */
        result = callVerifyProp(iobjCur_, verIobjProp, preCondIobjProp,
                                remapIobjProp, result, IndirectObject);

        /* 
         *   check that the action method ("doXxx") is defined on at least
         *   one of the objects 
         */
        return verifyActionProp([dobjCur_, iobjCur_],
                                [actionDobjProp, actionIobjProp],
                                result);
    }

    /* initialize tentative resolutions for other noun phrases */
    initTentative(issuingActor, targetActor, whichObj)
    {
        local tRes;
        local ti, td;

        /* 
         *   remember the old tentative direct and indirect objects, then
         *   set them to empty lists - this will ensure that we don't
         *   recursively try to get a tentative resolution for the object
         *   we're working on right now, which would cause infinite
         *   recursion 
         */
        td = tentativeDobj_;
        ti = tentativeIobj_;

        /* set them to empty lists to indicate they don't need resolving */
        tentativeDobj_ = [];
        tentativeIobj_ = [];

        /* make sure we set things back when we're done */
        try
        {
            /* initialize the other noun phrase */
            if (whichObj == DirectObject && ti == nil)
            {
                /* tentatively resolve the indirect object */
                tRes = new TentativeResolveResults(targetActor, issuingActor);
                ti = iobjMatch.resolveNouns(
                    getIobjResolver(issuingActor, targetActor, true), tRes);

                /* 
                 *   if the list is empty, and we didn't have a missing
                 *   noun phrase, use a dummy object as the tentative
                 *   resolution - this distinguishes erroneous noun phrases
                 *   from those that are simply missing 
                 */
                if (ti == [] && !tRes.npMissing)
                    ti = [dummyTentativeInfo];
            }
            else if (whichObj == IndirectObject && td == nil)
            {
                /* tentatively resolve the direct object */
                tRes = new TentativeResolveResults(targetActor, issuingActor);
                td = dobjMatch.resolveNouns(
                    getDobjResolver(issuingActor, targetActor, true), tRes);

                /* use a dummy object if appropriate */
                if (td == [] && !tRes.npMissing)
                    td = [dummyTentativeInfo];
            }
        }
        finally
        {
            /* set the original (or updated) tentative lists */
            tentativeDobj_ = td;
            tentativeIobj_ = ti;
        }
    }

    /*
     *   Check for remapping 
     */
    checkRemapping()
    {
        local remapInfo;
        local role;

        /* presume we'll find remapping for the first-resolved object */
        role = resolveFirst;
        
        /* check remapping for each object, in the resolution order */
        if (resolveFirst == DirectObject)
        {
            /* the direct object is resolved first, so try it first */
            if ((remapInfo = dobjCur_.(remapDobjProp)) == nil)
            {
                remapInfo = iobjCur_.(remapIobjProp);
                role = IndirectObject;
            }
        }
        else
        {
            /* the indirect object is resolved first, so remap it first */
            if ((remapInfo = iobjCur_.(remapIobjProp)) == nil)
            {
                remapInfo = dobjCur_.(remapDobjProp);
                role = DirectObject;
            }
        }

        /* if we found a remapping, apply it */
        if (remapInfo != nil)
            remapAction(nil, role, remapInfo);
    }

    /*
     *   Execute the command. 
     */
    execAction()
    {
        /* invoke the catch-all 'check' method on each object */
        callCatchAllProp(iobjCur_, checkIobjProp,
                         &checkIobjDefault, &checkIobjAll);
        callCatchAllProp(dobjCur_, checkDobjProp,
                         &checkDobjDefault, &checkDobjAll);

        /* invoke the check method on each object */
        dobjCur_.(checkDobjProp)();
        iobjCur_.(checkIobjProp)();

        /* invoke the catch-all 'action' method on each object */
        callCatchAllProp(iobjCur_, actionIobjProp,
                         &actionIobjDefault, &actionIobjAll);
        callCatchAllProp(dobjCur_, actionDobjProp,
                         &actionDobjDefault, &actionDobjAll);

        /* 
         *   Invoke the action method on each object, starting with the
         *   non-list object.  
         */
        if (execFirst == DirectObject)
        {
            dobjCur_.(actionDobjProp)();
            iobjCur_.(actionIobjProp)();
        }
        else
        {
            iobjCur_.(actionIobjProp)();
            dobjCur_.(actionDobjProp)();
        }
    }

    /*
     *   Check any pre-conditions for the action.  We'll check our
     *   action-level conditions, plus any preconditions associated with
     *   the direct and indirect objects.  
     */
    checkPreConditions(allowImplicit)
    {
        local ret1;
        local ret2;
        
        /* 
         *   inherit default handling to check our action-level conditions
         *   and our direct-object conditions 
         */
        ret1 = inherited(allowImplicit);

        /* check conditions on the indirect object, if any */
        ret2 = callPreConditions(iobjCur_, preCondIobjProp, iobjCur_,
                                 allowImplicit, IndirectObject);

        /* return true if any implicit commands were executed */
        return ret1 || ret2;
    }

    /*
     *   Get a message parameter object for the action.  We define 'dobj'
     *   as the direct object and 'iobj' as the indirect object, in
     *   addition to any inherited targets.  
     */
    getMessageParam(objName)
    {
        switch(objName)
        {
        case 'iobj':
            /* return the current indirect object */
            return iobjCur_;

        default:
            /* inherit default handling */
            return inherited(objName);
        }
    }

    /* get the current indirect object being executed */
    getIobj() { return iobjCur_; }

    /*
     *   Get the list of active objects.  We have a direct and indirect
     *   object.  
     */
    getCurrentObjects()
    {
        return [dobjCur_, iobjCur_];
    }

    /* set the current objects */
    setCurrentObjects(lst)
    {
        dobjCur_ = lst[1];
        iobjCur_ = lst[2];
    }

    /*
     *   Copy one tentative object list to the other.  This is useful when
     *   an object's verifier for one TIAction wants to forward the call
     *   to the other object verifier for a different TIAction.  
     */
    copyTentativeObjs()
    {
        /* copy whichever tentative list is populated to the other slot */
        if (tentativeDobj_ != nil)
            tentativeDobj_ = tentativeIobj_;
        else
            tentativeIobj_ = tentativeDobj_;
    }

    /*
     *   Get the tentative direct/indirect object resolution lists.  A
     *   tentative list is available for the later-resolved object while
     *   resolving the earlier-resolved object. 
     */
    getTentativeDobj() { return tentativeDobj_; }
    getTentativeIobj() { return tentativeIobj_; }

    /* 
     *   the predicate grammar must assign the indirect object production
     *   tree to iobjMatch 
     */
    iobjMatch = nil

    /* the indirect object list */
    iobjList_ = []

    /* current indirect object being executed */
    iobjCur_ = nil

    /* my cached indirect object resolver */
    iobjResolver_ = nil

    /*
     *   The tentative direct and indirect object lists.  A tentative list
     *   is available for the later-resolved object while resolving the
     *   earlier-resolved object.  
     */
    tentativeDobj_ = nil
    tentativeIobj_ = nil

    /*
     *   Verification and action properties for the indirect object.  By
     *   convention, the verification method for the indirect object of a
     *   two-object action is verIobjXxx; the check method is
     *   checkIobjXxx; and the action method is actionIobjXxx.  
     */
    verIobjProp = nil
    preCondIobjProp = nil
    checkIobjProp = nil
    actionIobjProp = nil

    /*
     *   Action-remap properties for the indirect object.  By convention,
     *   the remapper properties are named remapDobjAction and
     *   remapIobjAction, for the direct and indirect objects,
     *   respectively, where Action is replaced by the root name of the
     *   action.  
     */
    remapIobjProp = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   An action with a literal phrase as its only object, such as "say <any
 *   text>".  We'll accept anything as the literal phrase - a number, a
 *   quoted string, or arbitrary words - and treat them all simply as text.
 *   
 *   The grammar rules that produce these actions must set literalMatch to
 *   the literal phrase's match tree.  
 */
class LiteralAction: Action
    /*
     *   Resolve objects.  We don't actually have any objects to resolve,
     *   but we do have to get the text for the literal phrase.  
     */
    resolveNouns(issuingActor, targetActor, results)
    {
        /* 
         *   "Resolve" our literal phrase.  The literal phrase doesn't
         *   resolve to an object list the way a regular noun phrase would,
         *   but rather just resolves to a text string giving the original
         *   literal contents of the phrase.  
         */
        literalMatch.resolveLiteral(results);

        /* retrieve the text of the phrase, exactly as the player typed it */
        text_ = literalMatch.getLiteralText(results, self, DirectObject);
    }

    /* manually set the resolved objects */
    setResolvedObjects(txt)
    {
        /* remember the literal text */
        text_ = txt;
    }

    /* manually set the pre-resolved match trees */
    setObjectMatches(lit)
    {
        /* note the new literal match tree */
        literalMatch = lit;
    }

    /* 
     *   Get a message parameter.  We define 'literal' as the text of the
     *   literal phrase, in addition to inherited targets.  
     */
    getMessageParam(objName)
    {
        switch(objName)
        {
        case 'literal':
            /* return the text of the literal phrase */
            return text_;

        default:
            /* inherit default handling */
            return inherited(objName);
        }
    }

    /* get the current literal text */
    getLiteral() { return text_; }

    /* the text of the literal phrase */
    text_ = nil
;


/* ------------------------------------------------------------------------ */
/*
 *   An action with a direct object and a literal, such as "turn dial to
 *   <setting>" or "type <string> on keypad".  We'll accept anything as the
 *   literal phrase - a number, a quoted string, or arbitrary words - and
 *   treat them all simply as text.
 *   
 *   The grammar rules that produce these actions must set dobjMatch to the
 *   resolvable object of the command, and must set literalMatch to the
 *   literal phrase's match tree.  Note that we use dobjMatch as the
 *   resolvable object even if the object serves grammatically as the
 *   indirect object - this is a simplification, and the true grammatical
 *   purpose of the object isn't important since there's only one true
 *   object in the command.
 *   
 *   Each subclass must set the property whichLiteral to the object role
 *   (DirectObject, IndirectObject) the literal phrase plays grammatically,
 *   so that message generators will know how to phrase messages involving
 *   the literal phrase.  
 */
class LiteralTAction: TAction
    /*
     *   Resolve objects.  
     */
    resolveNouns(issuingActor, targetActor, results)
    {
        /* 
         *   If the literal phrase serves as the direct object
         *   grammatically, ask for its literal text first, so that we ask
         *   for it interactively first.  Otherwise, get the tentative
         *   literal text, in case it's useful to have in resolving the
         *   other object 
         */
        if (whichLiteral == DirectObject)
            text_ = literalMatch.getLiteralText(results, self, DirectObject);
        else
            text_ = literalMatch.getTentativeLiteralText();

        /* resolve the direct object */
        dobjList_ = dobjMatch.resolveNouns(
            getDobjResolver(issuingActor, targetActor, true), results);

        /* 
         *   "Resolve" the literal, ignoring the result - we call this so
         *   that the literal can do any scoring it wants to do during
         *   resolution; but since we're treating it literally, it
         *   obviously has no actual resolved value.  
         */
        literalMatch.resolveLiteral(results);

        /* the literal phrase resolves to the text only */
        text_ = literalMatch.getLiteralText(results, self, whichLiteral);
    }

    /* get an object role */
    getRoleFromIndex(idx)
    {
        /* 
         *   our literal doesn't have a formal role, so just inherit the
         *   default handling 
         */
        return inherited(idx);
    }

    /* manually set the resolved objects */
    setResolvedObjects(dobj, txt)
    {
        /* inherit default handling for the direct object */
        inherited(dobj);

        /* remember the literal text */
        text_ = txt;
    }

    /* manually set the pre-resolved match trees */
    setObjectMatches(dobj, lit)
    {
        /* inherit default handling for the direct object */
        inherited(dobj);
        
        /* note the new literal match tree */
        literalMatch = lit;
    }

    /*
     *   Get a message parameter.  We define 'literal' as the text of the
     *   literal phrase, in addition to inherited targets. 
     */
    getMessageParam(objName)
    {
        switch(objName)
        {
        case 'literal':
            /* return the text of the literal phrase */
            return text_;

        default:
            /* inherit default handling */
            return inherited(objName);
        }
    }

    /* get the current literal text */
    getLiteral() { return text_; }

    /* 
     *   Get a list of the current objects.  We include only the direct
     *   object here, since the literal text is not a resolved object but
     *   simply literal text. 
     */
    getCurrentObjects() { return [dobjCur_]; }

    /* set the current objects */
    setCurrentObjects(lst) { dobjCur_ = lst[1]; }

    /* the text of the literal phrase */
    text_ = nil

    /* object role played by the literal phrase */
    whichLiteral = nil

    /* -------------------------------------------------------------------- */
    /*
     *   Direct Object Resolver implementation.  We serve as our own
     *   direct object resolver, so we define any special resolver
     *   behavior here.  
     */

    /* 
     *   What we call our direct object might actually be playing the
     *   grammatical role of the indirect object - in order to inherit
     *   easily from TAction, we call our resolved object our direct
     *   object, regardless of which grammatical role it actually plays.
     *   For the most part it doesn't matter which is which; but for the
     *   purposes of our resolver, we actually do care about its real
     *   role.  So, override the resolver method whichObject so that it
     *   returns whichever role is NOT served by the topic object.  
     */
    whichObject =
        (whichLiteral == DirectObject ? IndirectObject : DirectObject)
;

/* ------------------------------------------------------------------------ */
/*
 *   An Action with a direct object and a topic, such as "ask <actor>
 *   about <topic>".  Topics differ from ordinary noun phrases in scope:
 *   rather than resolving to simulation objects based on location, we
 *   resolve these based on the actor's knowledge.
 *   
 *   The grammar rules that produce these actions must set dobjMatch to
 *   the resolvable object of the command (the <actor> in "ask <actor>
 *   about <topic>"), and must set topicMatch to the topic match tree
 *   object, which must be a TopicProd object.  Note that, in some cases,
 *   the phrasing might make the dobjMatch the indirect object,
 *   grammatically speaking: "type <topic> on <object>"; even in such
 *   cases, use dobjMatch for the resolvable object.
 *   
 *   When we resolve the topic, we will always resolve it to a single
 *   object of class ResolvedTopic.  This contains the literal tokens of
 *   the original command plus a list of simulation objects matching the
 *   topic name, ordered from best to worst.  This is different from the
 *   way most commands work, since we do not resolve the topic to a simple
 *   game world object.  We keep all of this extra information because we
 *   don't want to perform disambiguation in the normal fashion, but
 *   instead resolve as much as we can with what we're given, and then
 *   give the specialized action code as much information as we can to let
 *   the action code figure out how to respond to the topic.  
 */
class TopicAction: TAction
    /* 
     *   reset the action 
     */
    resetAction()
    {
        /* reset the inherited state */
        inherited();

        /* forget our topic resolver */
        topicResolver_ = nil;
    }
    
    /*
     *   resolve our noun phrases to objects 
     */
    resolveNouns(issuingActor, targetActor, results)
    {
        /* resolve the direct object, if we have one */
        if (dobjMatch != nil)
        {
            /* resolve the direct object */
            dobjList_ = dobjMatch.resolveNouns(
                getDobjResolver(issuingActor, targetActor, true), results);
        }

        /* resolve the topic */
        topicList_ = topicMatch.resolveNouns(
            getTopicResolver(issuingActor, targetActor, true), results);

        /* 
         *   if the topic is other than a single ResolvedTopic object, run
         *   it through the topic resolver's ambiguous noun phrase filter
         *   to get it into that canonical form 
         */
        if (topicList_.length() > 1
            || (topicList_.length() == 1
                && !topicList_[1].obj_.ofKind(ResolvedTopic)))
        {
            /* it's not in canonical form, so get it in canonical form now */
            topicList_ = getTopicResolver(issuingActor, targetActor, true)
                         .filterAmbiguousNounPhrase(topicList_, 1);
        }
    }

    /* get an object role */
    getRoleFromIndex(idx)
    {
        /* 
         *   our topic phrase doesn't have a formal role, so just inherit
         *   the default handling 
         */
        return inherited(idx);
    }

    /*
     *   Manually set the resolved objects.  We'll set our direct and
     *   indirect objects.  
     */
    setResolvedObjects(dobj, topic)
    {
        /* inherit default handling for the direct object */
        inherited(dobj);

        /* set the topic to a pre-resolved object tree */
        topicMatch = new PreResolvedProd(topic);

        /* set the indirect object list */
        topicList_ = makeResolveInfoList(topic);
    }

    /* manually set the pre-resolved match trees */
    setObjectMatches(dobj, topic)
    {
        /* inherit default handling for the direct object */
        inherited(dobj);
        
        /* note the new literal match tree */
        topicMatch = topic;
    }

    /*
     *   get the topic resolver 
     */
    getTopicResolver(issuingActor, targetActor, reset)
    {
        /* create one if we don't already have one */
        if (topicResolver_ == nil)
            topicResolver_ = createTopicResolver(issuingActor, targetActor);

        /* reset the resolver if desired */
        if (reset)
            topicResolver_.resetResolver();

        /* return the cached resolver */
        return topicResolver_;
    }

    /*
     *   Create the topic resolver.  
     */
    createTopicResolver(issuingActor, targetActor)
    {
        return new TopicResolver(self, issuingActor, targetActor,
                                 topicMatch, whichTopic);
    }

    getMessageParam(objName)
    {
        switch(objName)
        {
        case 'topic':
            /* return the current topic object */
            return getTopic();

        default:
            /* inherit default handling */
            return inherited(objName);
        }
    }

    /* get the current topic */
    getTopic()
    {
        /* 
         *   Because the topic list always contains one entry (a
         *   ResolvedTopic object encapsulating the topic information),
         *   our current topic is always simply the first and only element
         *   of the topic list. 
         */
        return topicList_[1].obj_;
    }

    /*
     *   Get the list of active objects.  We return only our direct
     *   object, since our topic isn't actually a simulation object.  
     */
    getCurrentObjects()
    {
        return [dobjCur_];
    }

    /* set the current objects */
    setCurrentObjects(lst)
    {
        dobjCur_ = lst[1];
    }

    /* the resolved topic object list */
    topicList_ = nil

    /* my cached topic resolver */
    topicResolver_ = nil

    /* grammatical role played by topic phrase */
    whichTopic = nil

    /* -------------------------------------------------------------------- */
    /*
     *   Direct Object Resolver implementation.  We serve as our own
     *   direct object resolver, so we define any special resolver
     *   behavior here.  
     */

    /* 
     *   What we call our direct object might actually be playing the
     *   grammatical role of the indirect object - in order to inherit
     *   easily from TAction, we call our resolved object our direct
     *   object, regardless of which grammatical role it actually plays.
     *   For the most part it doesn't matter which is which; but for the
     *   purposes of our resolver, we actually do care about its real
     *   role.  So, override the resolver method whichObject so that it
     *   returns whichever role is NOT served by the topic object.  
     */
    whichObject = (whichTopic == DirectObject ? IndirectObject : DirectObject)
;

/*
 *   Resolved Topic object.  The topic of a TopicAction always resolves to
 *   one of these objects. 
 */
class ResolvedTopic: object
    construct(inScope, likely, others, prod)
    {
        /*
         *   Remember our lists of objects.  We keep three separate lists,
         *   so that the action knows how we've classified the objects
         *   matching our phrase.  We keep a list of objects that are in
         *   scope; a list of objects that aren't in scope but which the
         *   actor thinks are likely topics; and a list of all of the
         *   other matches.
         *   
         *   We keep only the simulation objects from the lists - we don't
         *   keep the full ResolveInfo data.  
         */
        inScopeList = inScope.mapAll({x: x.obj_});
        likelyList = likely.mapAll({x: x.obj_});
        otherList = others.mapAll({x: x.obj_});

        /* keep the production match tree */
        topicProd = prod;
    }

    /*
     *   Get the best object match to the topic.  This is a default
     *   implementation that can be changed by game authors or library
     *   extensions to implement different topic-matching strategies.  This
     *   implementation simply picks an object arbitrarily from the
     *   "strongest" of the three lists we build: if there's anything in
     *   the inScopeList, we choose an object from that list; otherwise, if
     *   there's anything in the likelyList, we choose an object from that
     *   list; otherwise we choose an object from the otherList.
     */
    getBestMatch()
    {
        /* if there's an in-scope match, pick one */
        if (inScopeList.length() != 0)
            return inScopeList[1];

        /* nothing's in scope, so try to pick a likely match */
        if (likelyList.length() != 0)
            return likelyList[1];

        /* 
         *   there's not even anything likely, so pick an "other", if
         *   there's anything in that list 
         */
        if (otherList.length() != 0)
            return otherList[1];

        /* there are no matches at all - return nil */
        return nil;
    }

    /* get the original text of the topic phrase */
    getTopicText() { return topicProd.getOrigText(); }

    /* 
     *   get the original tokens of the topic phrase, in canonical
     *   tokenizer format 
     */
    getTopicTokens() { return topicProd.getOrigTokenList(); }

    /*
     *   get the original word strings of the topic phrase - this is
     *   simply a list of the original word strings (in their original
     *   case), without any of the extra information of the more
     *   complicated canonical tokenizer format 
     */
    getTopicWords()
    {
        /* return just the original text strings from the token list */
        return topicProd.getOrigTokenList().mapAll({x: getTokOrig(x)});
    }

    /*
     *   Our lists of resolved objects matching the topic phrase,
     *   separated by classification.  
     */
    inScopeList = nil
    likelyList = nil
    otherList = nil

    /*
     *   The production match tree object that matched the topic phrase in
     *   the command.  This can be used to obtain the original tokens of
     *   the command or the original text of the phrase.  
     */
    topicProd = nil
;

/*
 *   Topic Resolver
 */
class TopicResolver: Resolver
    construct(action, issuingActor, targetActor, prod, which)
    {
        /* inherit the base class constructor */
        inherited(action, issuingActor, targetActor);

        /* remember the grammatical role of our object in the command */
        whichObject = which;

        /* remember our topic match tree */
        topicProd = prod;
    }

    /*
     *   Determine if an object is in scope.  We'll accept anything that
     *   is in "physical" scope, and we'll also accept any topic object
     *   that the actor knows about. 
     */
    objInScope(obj)
    {
        /* if it's in the physical scope list, it's in scope as a topic */
        if (inherited(obj))
            return true;

        /* 
         *   it's not in physical scope; if the actor knows about it as a
         *   topic, it's in scope 
         */
        return actor_.knowsTopic(obj);
    }

    /*
     *   Filter an ambiguous noun list.
     *   
     *   It is almost always undesirable from a user interface perspective
     *   to ask for help disambiguating a topic phrase.  In the first
     *   place, since all topics tend to be in scope all the time, we
     *   might reveal too much about the inner model of the story if we
     *   were to enumerate all of the topic matches to a phrase.  In the
     *   second place, topics are used in conversational contexts, so it
     *   almost never make sense for the parser to ask for clarification -
     *   the other member of the conversation might ask, but not the
     *   parser.  So, we'll always filter the list to the required number,
     *   even if it means we choose arbitrarily.
     *   
     *   As a first cut, we prefer objects that are physically in scope to
     *   those not in scope: if the player is standing next to a control
     *   panel and types "ask bob about control panel," it makes little
     *   sense to consider any other control panels in the simulation.
     *   
     *   As a second cut, we'll ask the actor to filter the list.  Games
     *   that keep track of the actor's knowledge can use this to filter
     *   according to topics the actor is likely to know about.  
     */
    filterAmbiguousNounPhrase(lst, requiredNum)
    {
        local inScope;
        local actorPrefs;
        local rt;

        /* 
         *   First, get the subset of items that are in scope - we'll
         *   consider this the best set of matches.
         */
        inScope = lst.subset({x: scope_.indexOf(x.obj_) != nil});

        /* 
         *   eliminate the in-scope items from the list, so we can
         *   consider only what remains 
         */
        lst -= inScope;

        /* ask the actor to pick out the most likely set of topics */
        actorPrefs = lst.subset({x: actor_.isLikelyTopic(x.obj_)});

        /* eliminate those items from the list */
        lst -= actorPrefs;

        /* create our ResolvedTopic object for the results */
        rt = new ResolvedTopic(inScope, actorPrefs, lst, topicProd);

        /* return our special resolved topic object with the topic list */
        return [new ResolveInfo(rt, 0)];
    }

    /*
     *   Resolve an unknown phrase.  We allow unknown words to be used in
     *   topics; we simply return a ResolvedTopic that doesn't refer to
     *   any simulation objects at all.  
     */
    resolveUnknownNounPhrase(tokList)
    {
        local rt;
        
        /* 
         *   Create our ResolvedTopic object for the results.  We have
         *   words we don't know, so we're not referring to any objects,
         *   so our underlying simulation object list is empty.  
         */
        rt = new ResolvedTopic([], [], [], topicProd);

        /* return a resolved topic object with the empty list */
        return [new ResolveInfo(rt, 0)];
    }

    /*
     *   Filter a plural.
     */
    filterPluralPhrase(lst)
    {
        /*
         *   Handle this the same way we handle an ambiguous noun phrase,
         *   so that we yield only one object.  Topics are inherently
         *   singular; we'll allow asking about a grammatically plural
         *   term, but we'll turn it into a single topic result.  
         */
        return filterAmbiguousNounPhrase(lst, 1);
    }

    /* get a default object */
    getDefaultObject()
    {
        /* there is never a default for a topic */
        return nil;
    }

    /* the production match tree for the topic phrase we're resolving */
    topicProd = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   System action.  These actions are for out-of-game meta-verbs (save,
 *   restore, undo).  These verbs take no objects, must be performed by
 *   the player (thus by the player character, not an NPC), and consume no
 *   game clock time.  
 */
class SystemAction: IAction
    execAction()
    {
        /* 
         *   Conceptually, system actions are performed by the player
         *   directly, not by any character in the game (not even the
         *   player character).  However, we don't distinguish in the
         *   command-entry user interface between a command the player is
         *   performing and a command to the player character, hence we
         *   can merely ensure that the command is not directed to a
         *   non-player character. 
         */
        if (!gActor.isPlayerChar)
        {
            libMessages.systemActionToNPC();
            exit;
        }

        /* 
         *   system actions sometimes need to prompt for interactive
         *   responses, so deactivate the report list - this will allow
         *   interactive prompts to be shown immediately, not treated as
         *   reports to be deferred until the command is finished 
         */
        gTranscript.showReports(true);
        gTranscript.clearReports();

        /* perform our specific action */
        execSystemAction();

        /* re-activate the transcript for the next command */
        gTranscript.activate();
    }

    /* each subclass must override this to perform its actual action */
    execSystemAction() { }

    /*
     *   Ask for an input file.  Freezes the real-time event clock for the
     *   duration of reading the event, and sets our property
     *   origElapsedTime to the elapsed time when we started processing
     *   the interaction.  
     */
    getInputFile(prompt, dialogType, fileType, flags)
    {
        local result;
        
        /* 
         *   note the game elapsed time before we start - we want to
         *   freeze the real-time clock while we're waiting for the user
         *   to respond, since this system verb exists outside of the
         *   usual time flow of the game 
         */
        origElapsedTime = realTimeManager.getElapsedTime();
        
        /* ask for a file */
        result = inputFile(prompt, dialogType, fileType, flags);

        /* 
         *   restore the game real-time counter to what it was before we
         *   started the interactive response 
         */
        realTimeManager.setElapsedTime(origElapsedTime);

        /* return the result from inputFile */
        return result;
    }

    /* system actions consume no game time */
    actionTime = 0

    /* real-time event clock at start of getInputFile() */
    origElapsedTime = nil
;

