#ifdef RCSID
static char RCSid[] =
"$Header$";
#endif

/* Copyright (c) 1987, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
Name
  vmconsol.cpp - TADS 3 console input reader and output formatter
Function
  Provides console input and output for the TADS 3 built-in function set
  for the T3 VM, including the output formatter.

  T3 uses the UTF-8 character set to represent character strings.  The OS
  functions use the local character set.  We perform the mapping between
  UTF-8 and the local character set within this module, so that OS routines
  see local characters only, not UTF-8.

  This code is based on the TADS 2 output formatter, but has been
  substantially reworked for C++, Unicode, and the slightly different
  TADS 3 formatting model.
Notes

Returns
  None
Modified
  08/25/99 MJRoberts  - created from TADS 2 output formatter
*/

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include "wchar.h"

#include "os.h"
#include "t3std.h"
#include "utf8.h"
#include "charmap.h"
#include "vmuni.h"
#include "vmconsol.h"
#include "vmglob.h"
#include "vmhash.h"


/* ------------------------------------------------------------------------ */
/*
 *   Log-file formatter subclass implementation 
 */

/*
 *   Open a new log file 
 */
int CVmFormatterLog::open_log_file(const char *fname, int initial_html_mode)
{
    /* close any existing log file */
    if (close_log_file())
        return 1;
    
    /* reinitialize */
    init();

    /* save the filename for later (we'll need it when we close the file) */
    strcpy(logfname_, fname);

    /* open the new file */
    logfp_ = osfopwt(fname, OSFTLOG);

    /* set our initial HTML mode */
    html_mode_ = initial_html_mode;

    /* return success if we successfully opened the file, failure otherwise */
    return (logfp_ == 0);
}

/*
 *   Close the log file 
 */
int CVmFormatterLog::close_log_file()
{
    /* if we have a file, close it */
    if (logfp_ != 0)
    {
        /* close the handle */
        osfcls(logfp_);

        /* set the system file type to "log file" */
        os_settype(logfname_, OSFTLOG);

        /* forget about our log file handle */
        logfp_ = 0;
    }

    /* success */
    return 0;
}


/* ------------------------------------------------------------------------ */
/*
 *   Base Formatter 
 */

/*
 *   deletion
 */
CVmFormatter::~CVmFormatter()
{
    /* if we have a table of horizontal tabs, delete it */
    if (tabs_ != 0)
        delete tabs_;
}

/*
 *   Write out a line.  Text we receive is in the UTF-8 character set.
 */
void CVmFormatter::write_text(VMG_ const wchar_t *txt,
                              const vmcon_color_t *colors, vm_nl_type nl)
{
    /* 
     *   Check the "script quiet" mode - this indicates that we're reading
     *   a script and not echoing output to the display.  If this mode is
     *   on, and we're writing to the display, suppress this write.  If
     *   the mode is off, or we're writing to a non-display stream (such
     *   as a log file stream), show the output as normal.  
     */
    if (!console_->is_quiet_script() || !is_disp_stream_)
    {
        char local_buf[128];
        char *dst;
        size_t rem;

        /*
         *   Check to see if we've reached the end of the screen, and if so
         *   run the MORE prompt.  Note that we don't make this check at all
         *   if USE_MORE is undefined, since this means that the OS layer
         *   code is taking responsibility for pagination issues.  We also
         *   don't display a MORE prompt when reading from a script file.
         *   
         *   Note that we suppress the MORE prompt if we're showing a
         *   continuation of a line already partially shown.  We only want to
         *   show a MORE prompt at the start of a new line.
         *   
         *   Skip the MORE prompt if this stream doesn't use it.  
         */
        if (use_more_mode()
            && !console_->is_reading_script()
            && console_->is_more_mode()
            && linecol_ - linepos_ == 0
            && linecnt_ + 1 >= console_->get_page_length())
        {
            /* set the standard text color */
            set_os_text_color(OS_COLOR_P_TEXT, OS_COLOR_P_TEXTBG);
            set_os_text_attr(0);

            /* display the MORE prompt */
            console_->show_more_prompt(vmg0_);

            /* restore the current color scheme */
            set_os_text_color(os_color_.fg, os_color_.bg);
            set_os_text_attr(os_color_.hilite ? OS_ATTR_HILITE : 0);
        }

        /* count the line if a newline follows */
        if (nl != VM_NL_NONE && nl != VM_NL_NONE_INTERNAL)
            ++linecnt_;

        /* convert and display the text */
        for (dst = local_buf, rem = sizeof(local_buf) - 1 ; *txt != '\0' ; )
        {
            size_t cur;
            size_t old_rem;
            
            /* 
             *   if this character is in a new color, write out the OS-level
             *   color switch code 
             */
            if (colors != 0 && !colors->equals(&os_color_))
            {
                /* 
                 *   null-terminate and display what's in the buffer so far,
                 *   so that we close out all of the remaining text in the
                 *   old color and attributes
                 */
                *dst = '\0';
                print_to_os(local_buf);

                /* reset to the start of the local output buffer */
                dst = local_buf;
                rem = sizeof(local_buf) - 1;

                /* set the highlighting mode, if it changed */
                if (colors->hilite != os_color_.hilite)
                    set_os_text_attr(colors->hilite ? OS_ATTR_HILITE : 0);

                /* set the color, if it changed */
                if (colors->fg != os_color_.fg
                    || colors->bg != os_color_.bg)
                    set_os_text_color(colors->fg, colors->bg);

                /* 
                 *   Whatever happened, set our new color internally as the
                 *   last color we sent to the OS.  Even if we didn't
                 *   actually do anything, we'll at least know we won't have
                 *   to do anything more until we find another new color. 
                 */
                os_color_ = *colors;
            }

            /* try storing another character */
            old_rem = rem;
            cur = G_cmap_to_ui->map(*txt, &dst, &rem);

            /* if that failed, flush the buffer and try again */
            if (cur > old_rem)
            {
                /* null-terminate the buffer */
                *dst = '\0';
                
                /* display the text */
                print_to_os(local_buf);

                /* reset to the start of the local output buffer */
                dst = local_buf;
                rem = sizeof(local_buf) - 1;
            }
            else
            {
                /* we've now consumed this character of input */
                ++txt;
                if (colors)
                    ++colors;
            }
        }

        /* if we have a partially-filled buffer, display it */
        if (dst > local_buf)
        {
            /* null-terminate and display the buffer */
            *dst = '\0';
            print_to_os(local_buf);
        }

        /* write the appropriate type of line termination */
        switch(nl)
        {
        case VM_NL_NONE:
        case VM_NL_NONE_INTERNAL:
            /* no line termination is needed */
            break;

        case VM_NL_NEWLINE:
            /* write a newline */
            print_to_os("\n");
            break;

        case VM_NL_OSNEWLINE:
            /* 
             *   the OS will provide a newline, but add a space to make it
             *   explicit that we can break the line here 
             */
            print_to_os(" ");
            break;
        }
    }
}

/* ------------------------------------------------------------------------ */
/*
 *   Flush the current line to the display, using the given type of line
 *   termination.
 *   
 *   VM_NL_NONE: flush the current line but do not start a new line; more
 *   text will follow on the current line.  This is used, for example, to
 *   flush text after displaying a prompt and before waiting for user
 *   input.
 *   
 *   VM_NL_NONE_INTERNAL: same as VM_NL_NONE, but doesn't flush at the OS
 *   level.  This is used when we're only flushing our buffers in order to
 *   clear out space internally, not because we want the underlying OS
 *   renderer to display things immediately.  This distinction is
 *   important in HTML mode, since it ensures that the HTML parser only
 *   sees well-formed strings when flushing.
 *   
 *   VM_NL_NEWLINE: flush the line and start a new line by writing out a
 *   newline character.
 *   
 *   VM_NL_OSNEWLINE: flush the line as though starting a new line, but
 *   don't add an actual newline character to the output, since the
 *   underlying OS display code will handle this.  Instead, add a space
 *   after the line to indicate to the OS code that a line break is
 *   possible there.  (This differs from VM_NL_NONE in that VM_NL_NONE
 *   doesn't add anything at all after the line.)  
 */

/* flush a given output stream */
void CVmFormatter::flush(VMG_ vm_nl_type nl)
{
    int i;
    vm_nl_type write_nl;

    /* null-terminate the current output line buffer */
    linebuf_[linepos_] = '\0';

    /* 
     *   Expand any pending tab.  Allow "anonymous" tabs only if we're
     *   flushing because we're ending the line normally; if we're not
     *   ending the line, we can't handle tabs that depend on the line
     *   ending. 
     */
    expand_pending_tab(vmg_ nl == VM_NL_NEWLINE);

    /* note the position of the last character to display */
    i = linepos_ - 1;

    /* if we're adding anything, remove trailing spaces */
    if (nl != VM_NL_NONE && nl != VM_NL_NONE_INTERNAL)
    {
        /* look for last non-space character */
        for ( ; i >= 0 && linebuf_[i] == ' ' ; --i) ;
    }

    /* null-terminate the buffer at the current position */
    linebuf_[++i] = '\0';

    /* check the newline mode */
    switch(nl)
    {
    case VM_NL_NONE:
    case VM_NL_NONE_INTERNAL:
        /* no newline - just flush out what we have */
        write_nl = VM_NL_NONE;
        break;

    case VM_NL_NEWLINE:
        /* 
         *   We're adding a newline.  We want to suppress redundant
         *   newlines -- we reduce any run of consecutive vertical
         *   whitespace to a single newline.  So, if we have anything in
         *   this line, or we didn't already just write a newline, write
         *   out a newline now; otherwise, write nothing.  
         */
        if (linecol_ != 0 || !just_did_nl_)
        {
            /* add the newline */
            write_nl = VM_NL_NEWLINE;
        }
        else
        {
            /* 
             *   Don't write out a newline after all - the line buffer is
             *   empty, and we just wrote a newline, so this is a
             *   redundant newline that we wish to suppress (so that we
             *   collapse a run of vertical whitespace down to a single
             *   newline).  
             */
            write_nl = VM_NL_NONE;
        }
        break;

    case VM_NL_OSNEWLINE:
        /* 
         *   we're going to depend on the underlying OS output layer to do
         *   line breaking, so we won't add a newline, but we will add a
         *   space, so that the underlying OS layer knows we have a word
         *   break here 
         */
        write_nl = VM_NL_OSNEWLINE;
        break;
    }

    /* 
     *   display the line, as long as we have something buffered to
     *   display; even if we don't, display it if our column is non-zero
     *   and we didn't just do a newline, since this must mean that we've
     *   flushed a partial line and are just now doing the newline 
     */
    if (linebuf_[0] != '\0'
        || (linecol_ != 0 && !just_did_nl_))
    {
        /* write it out */
        write_text(vmg_ linebuf_, colorbuf_, write_nl);
    }

    /* generate an HTML line break if necessary */
    if (nl == VM_NL_NEWLINE && html_mode_ && html_target_)
        write_text(vmg_ L"<BR HEIGHT=0>", 0, VM_NL_NONE);

    /* check the line ending */
    if (nl == VM_NL_NONE)
    {
        /* we're not displaying a newline, so flush what we have */
        flush_to_os();
    }
    else
    {
        /* we displayed a newline, so reset the column position */
        linecol_ = 0;
    }

    /* reset the line output buffer position */
    linepos_ = 0;

    /* 
     *   If we just output a newline, note it.  If we didn't just output a
     *   newline, but we did write out anything else, note that we're no
     *   longer at the start of a line on the underlying output device.  
     */
    if (nl == VM_NL_NEWLINE)
        just_did_nl_ = TRUE;
    else if (linebuf_[0] != '\0')
        just_did_nl_ = FALSE;

    /* 
     *   if the current buffering color doesn't match the current osifc-layer
     *   color, then we must need to flush just the new color/attribute
     *   settings (this can happen when we have changed the attributes in
     *   preparation for reading input, since we won't have any actual text
     *   to write after the color change) 
     */
    if (!cur_color_.equals(&os_color_))
    {
        /* set the highlighting mode in the OS window, if it changed */
        if (cur_color_.hilite != os_color_.hilite)
            set_os_text_attr(cur_color_.hilite ? OS_ATTR_HILITE : 0);

        /* set the color in the OS window, if it changed */
        if (cur_color_.fg != os_color_.fg
            || cur_color_.bg != os_color_.bg)
            set_os_text_color(cur_color_.fg, cur_color_.bg);

        /* set the new osifc color */
        os_color_ = cur_color_;
    }
}

/* ------------------------------------------------------------------------ */
/*
 *   Clear out our buffers 
 */
void CVmFormatter::empty_buffers(VMG0_)
{
    /* reset our buffer pointers */
    linepos_ = 0;
    linecol_ = 0;
    linebuf_[0] = '\0';
    just_did_nl_ = FALSE;

    /* there's no pending tab now */
    pending_tab_align_ = VMFMT_TAB_NONE;

    /* start out at the first line */
    linecnt_ = 0;
}

/* ------------------------------------------------------------------------ */
/*
 *   Immediately update the display window 
 */
void CVmFormatter::update_display(VMG0_)
{
    /* update the display window at the OS layer */
    os_update_display();
}

/* ------------------------------------------------------------------------ */
/*
 *   Display a blank line to the stream
 */
void CVmFormatter::write_blank_line(VMG0_)
{
    /* flush the stream */
    flush(vmg_ VM_NL_NEWLINE);

    /* if generating for an HTML display target, add an HTML line break */
    if (html_mode_ && html_target_)
        write_text(vmg_ L"<BR>", 0, VM_NL_NONE);

    /* write out a blank line */
    write_text(vmg_ L"", 0, VM_NL_NEWLINE);
}

/* ------------------------------------------------------------------------ */
/*
 *   Generate a tab for a "\t" sequence in the game text, or a <TAB
 *   MULTIPLE> or <TAB INDENT> sequence parsed in our mini-parser.
 *   
 *   Standard (non-HTML) version: we'll generate enough spaces to take us to
 *   the next tab stop.
 *   
 *   HTML version: if we're in native HTML mode, we'll just generate the
 *   equivalent HTML; if we're not in HTML mode, we'll generate a hard tab
 *   character, which the HTML formatter will interpret as a <TAB
 *   MULTIPLE=4>.  
 */
void CVmFormatter::write_tab(VMG_ int indent, int multiple)
{
    int maxcol;

    /* check to see what the underlying system is expecting */
    if (html_target_)
    {
        /* the underlying system is HTML - check for HTML mode */
        if (html_mode_)
        {
            char buf[40];

            /* HTML mode - generate the appropriate <TAB> sequence */
            sprintf(buf, "<TAB %s=%d>",
                    indent != 0 ? "INDENT" : "MULTIPLE",
                    indent != 0 ? indent : multiple);
            
            /* write it out */
            buffer_string(vmg_ buf);
        }
        else
        {
            /* we're not in HTML mode, so generate a hard tab character */
            buffer_expchar(vmg_ QTAB);
        }
    }
    else if (multiple != 0)
    {
        /* get the maximum column */
        maxcol = get_buffer_maxcol();

        /*
         *   We're not in HTML mode, and we have a tab to an every-N stop -
         *   expand the tab with spaces.  Keep going until we reach the next
         *   tab stop of the given multiple.
         */
        do
        {
            /* add another space */
            linebuf_[linepos_] = ' ';
            colorbuf_[linepos_] = cur_color_;

            /* advance one character in the buffer */
            ++linepos_;

            /* advance the column counter */
            ++linecol_;
        } while (linecol_ < maxcol && (linecol_ + 1) % multiple != 0);
    }
    else if (indent != 0)
    {
        /* 
         *   We're not in HTML mode, and we just want to add a given number
         *   of spaces.  Simply write out the given number of spaces, up to
         *   our maximum column limit.  
         */
        for (maxcol = get_buffer_maxcol() ;
             indent != 0 && linecol_ < maxcol ; --indent)
        {
            /* add another space */
            linebuf_[linepos_] = ' ';
            colorbuf_[linepos_] = cur_color_;

            /* advance one character in the buffer and one column */
            ++linepos_;
            ++linecol_;
        }
    }
}


/* ------------------------------------------------------------------------ */
/*
 *   Flush a line 
 */
void CVmFormatter::flush_line(VMG_ int padding)
{
    /* 
     *   check to see if we're using the underlying display layer's line
     *   wrapping 
     */
    if (os_line_wrap_)
    {
        /*
         *   In the HTML version, we don't need the normal *MORE*
         *   processing, since the HTML layer will handle that.
         *   Furthermore, we don't need to provide actual newline breaks
         *   -- that happens after the HTML is parsed, so we don't have
         *   enough information here to figure out actual line breaks.
         *   So, we'll just flush out our buffer whenever it fills up, and
         *   suppress newlines.
         *   
         *   Similarly, if we have OS-level line wrapping, don't try to
         *   figure out where the line breaks go -- just flush our buffer
         *   without a trailing newline whenever the buffer is full, and
         *   let the OS layer worry about formatting lines and paragraphs.
         *   
         *   If we're using padding, use newline mode VM_NL_OSNEWLINE.  If
         *   we don't want padding (which is the case if we completely
         *   fill up the buffer without finding any word breaks), write
         *   out in mode VM_NL_NONE, which just flushes the buffer exactly
         *   like it is.  
         */
        flush(vmg_ padding ? VM_NL_OSNEWLINE : VM_NL_NONE_INTERNAL);
    }
    else
    {
        /*
         *   Normal mode - we process the *MORE* prompt ourselves, and we
         *   are responsible for figuring out where the actual line breaks
         *   go.  Use flush() to generate an actual newline whenever we
         *   flush out our buffer.  
         */
        flush(vmg_ VM_NL_NEWLINE);
    }
}


/* ------------------------------------------------------------------------ */
/*
 *   Write a character to an output stream.  The character is provided to us
 *   as a wide Unicode character.  
 */
void CVmFormatter::buffer_char(VMG_ wchar_t c)
{
    const wchar_t *exp;
    size_t exp_len;

    /* check for a display expansion */
    exp = G_cmap_to_ui->get_expansion(c, &exp_len);
    if (exp != 0)
    {
        /* write each character of the expansion */
        for ( ; exp_len != 0 ; ++exp, --exp_len)
            buffer_expchar(vmg_ *exp);
    }
    else
    {
        /* there's no expansion - buffer the character as-is */
        buffer_expchar(vmg_ c);
    }
}

/*
 *   Write an expanded character to an output stream.  
 */
void CVmFormatter::buffer_expchar(VMG_ wchar_t c)
{
    wchar_t brkchar;
    int i;
    int qspace;
    
    /* check for the special quoted space character */
    if (c == QSPACE)
    {
        /* it's a quoted space - note it and convert it to a regular space */
        qspace = TRUE;
        c = ' ';
    }
    else if (c == QTAB)
    {
        /* 
         *   it's a hard tab - put a hard tab character in the buffer for
         *   passing to the underlying renderer (the only time we use this
         *   character is for rendering a tab to an underlying HTML
         *   processor; in any other case we will already have expanded
         *   any tabs into runs of spaces) 
         */
        c = '\t';
        qspace = FALSE;
    }
    else
    {
        /* 
         *   Translate any whitespace character to a regular space
         *   character.  Note that, once this is done, we don't need to
         *   worry about calling t3_is_space() any more - we can use the
         *   faster is_space() from now on, because we know we have a
         *   regular space (or possibly a regular tab, from above). 
         */
        if (t3_is_space(c))
            c = ' ';
        
        /* 
         *   it's not a quoted space - treat it as normal space unless
         *   we're in obey-whitespace mode, in which case treat it as a
         *   non-breaking space 
         */
        qspace = obey_whitespace_;
    }

    /* check for the caps/nocaps flags */
    if ((capsflag_ || allcapsflag_) && t3_is_alpha(c))
    {
        /* capsflag is set, so capitalize this character */
        c = t3_to_upper(c);

        /* okay, we've capitalized something; clear flag */
        capsflag_ = FALSE;
    }
    else if (nocapsflag_ && t3_is_alpha(c))
    {
        /* nocapsflag is set, so minisculize this character */
        c = t3_to_lower(c);

        /* clear the flag now that we've done the job */
        nocapsflag_ = FALSE;
    }

    /* add the character to out output buffer, flushing as needed */
    if (linecol_ + 1 < get_buffer_maxcol())
    {
        /* 
         *   there's room for this character, so add it to the buffer 
         */
        
        /* 
         *   Ignore non-quoted space at start of line.  (Note that we use
         *   is_space() rather than t3_is_space() - we know we've already
         *   translated this character to a regular space or a regular
         *   tab, so we don't need to make the more extensive and
         *   expensive check that t3_is_space() makes; is_space() is
         *   perfectly adequate for our needs here.) 
         */
        if (is_space(c) && c != '\t' && linecol_ == 0 && !qspace)
            return;

        /* is this a non-quoted space not at the start of the line? */
        if (is_space(c) && c != '\t' && linecol_ != 0 && !qspace)
        {
            int pos1;
            wchar_t prv;

            /* check the previous character */
            pos1 = linepos_ - 1;
            prv = linebuf_[pos1];

            /* ignore repeated spaces - collapse into a single space */
            if (is_space(prv))
                return;

            /*
             *   Certain punctuation requires a double space: a period, a
             *   question mark, an exclamation mark, or a colon; or any of
             *   these characters followed by any number of single and/or
             *   double quotes.  First, scan back to before any quotes, if
             *   are on one now, then check the preceding character; if
             *   it's one of the punctuation marks requiring a double
             *   space, add this space a second time.  (In addition to
             *   scanning back past quotes, scan past parentheses,
             *   brackets, and braces.)  Don't double the spacing if we're
             *   not in the normal doublespace mode; some people may
             *   prefer single spacing after punctuation, so we make this
             *   a run-time option.  
             */
            if (console_->get_doublespace())
            {
                /* 
                 *   Find the previous relevant punctuation character.
                 *   Skip certain punctuation - in particular, skip
                 *   quotes, close parentheses, braces, and brackets,
                 *   since we want the effect of any double-spacing
                 *   punctuation to transcend these grouping marks. 
                 */
                while (pos1 != 0 &&
                       (prv == '"' || prv == '\'' || prv == ')'
                        || prv == ']' || prv == '}'))
                {
                    /* go back one position */
                    prv = linebuf_[--pos1];
                }

                /* check for double-spacing punctuation */
                if (prv == '.' || prv == '?' || prv == '!' || prv == ':')
                {
                    /* a double-space is required after this character */
                    linebuf_[linepos_] = c;
                    colorbuf_[linepos_] = cur_color_;

                    /* advance one position */
                    ++linepos_;
                    ++linecol_;
                }
            }
        }

        /* add this character to the buffer */
        linebuf_[linepos_] = c;
        colorbuf_[linepos_] = cur_color_;

        /* advance one character in the buffer */
        ++linepos_;

        /* advance the output column position */
        ++linecol_;
        return;
    }

    /*
     *   The line would overflow if this character were added.  If we're
     *   trying to output a space, we'll just add it to the line buffer for
     *   now; we'll come back later and figure out where to break at the next
     *   non-space character - this ensures that we don't carry trailing
     *   space to the start of the next line if we have an explicit newline
     *   before the next non-space character.  
     */
    if (is_space(c) && c != '\t' && !qspace)
    {
        /* 
         *   We're adding a space, so we'll figure out the breaking later,
         *   when we output a non-space character.  If the buffer doesn't
         *   already end with a space, add this space to the buffer.  (Don't
         *   add it if we already have a trailing space, in keeping with our
         *   general policy of compressing runs of contiguous whitespace.) 
         */
        if (!is_space(linebuf_[linepos_ - 1]))
        {
            /* add a space */
            linebuf_[linepos_] = ' ';
            colorbuf_[linepos_] = cur_color_;

            /* advance to the next character position */
            ++linepos_;
            ++linecol_;
        }

        /* done for now */
        return;
    }
    
    /*
     *   Find the most recent word break: look for a space or dash, starting
     *   at the end of the line.  
     *   
     *   If we're about to write a hyphen, we want to skip all contiguous
     *   hyphens, because we want to keep them together as a single
     *   punctuation mark; then keep going in the normal manner, which will
     *   keep the hyphens plus the word they're attached to together as a
     *   single unit.  If spaces precede the sequence of hyphens, include
     *   the prior word as well.  
     */
    i = linepos_ - 1;
    if (c == '-')
    {
        /* skip any contiguous hyphens at the end of the line */
        for ( ; i >= 0 && linebuf_[i] == '-' ; --i) ;
        
        /* skip any spaces preceding the sequence of hyphens */
        for ( ; i >= 0 && is_space(linebuf_[i]) ; --i) ;
    }

    /* 
     *   Now find the preceding space.  If we're doing our own wrapping
     *   (i.e., we're not using OS line wrapping), then look for the
     *   nearest hyphen as well. 
     */
    for ( ; i >= 0 && !is_space(linebuf_[i])
          && !(!os_line_wrap_ && linebuf_[i] == '-') ; --i) ;

    /* check to see if we found a good place to break */
    if (i < 0)
    {
        /*
         *   We didn't find a good place to break.  If the underlying
         *   console allows overrunning the line width, simply add the
         *   character, even though it overflows; otherwise, force a break
         *   at the line width, even though it doesn't occur at a natural
         *   breaking point.
         *   
         *   In any case, don't let our buffer fill up beyond its maximum
         *   size.  
         */
        if (!console_->allow_overrun() || linepos_ + 1 >= OS_MAXWIDTH)
        {
            /* 
             *   we didn't find any good place to break, and we have a
             *   non-zero console width - flush the entire line as-is,
             *   breaking arbitrarily in the middle of a word 
             */
            flush_line(vmg_ FALSE);
            
            /* 
             *   we've completely cleared out the line buffer, so reset all
             *   of the line buffer counters 
             */
            linepos_ = 0;
            linecol_ = 0;
            linebuf_[0] = '\0';
        }
    }
    else
    {
        wchar_t tmpbuf[OS_MAXWIDTH];
        vmcon_color_t tmpcolor[OS_MAXWIDTH];
        size_t tmpchars;

        /* remember word-break character */        
        brkchar = linebuf_[i];

        /* null-terminate the line buffer */        
        linebuf_[linepos_] = '\0';

        /* the next line starts after the break - save a copy */
        tmpchars = wcslen(&linebuf_[i+1]);
        memcpy(tmpbuf, &linebuf_[i+1], tmpchars*sizeof(tmpbuf[0]));
        memcpy(tmpcolor, &colorbuf_[i+1], tmpchars*sizeof(tmpcolor[0]));

        /* 
         *   terminate the buffer at the space or after the hyphen,
         *   depending on where we broke 
         */
        if (is_space(brkchar))
            linebuf_[i] = '\0';
        else
            linebuf_[i+1] = '\0';

        /* write out everything up to the word break */
        flush_line(vmg_ TRUE);

        /* move the saved start of the next line into the line buffer */
        memcpy(linebuf_, tmpbuf, tmpchars*sizeof(tmpbuf[0]));
        memcpy(colorbuf_, tmpcolor, tmpchars*sizeof(tmpcolor[0]));
        linepos_ = tmpchars;

        /* 
         *   figure what column we're now in - count all of the printable
         *   characters in the new line 
         */
        for (linecol_ = 0, i = 0 ; i < linepos_ ; ++i)
        {
            /* if it's printable, count it */
            if (linebuf_[i] >= 26)
                ++linecol_;
        }
    }
    
    /* add the new character to buffer */
    linebuf_[linepos_] = c;
    colorbuf_[linepos_] = cur_color_;

    /* advance the buffer index */
    ++linepos_;

    /* advance the column counter */
    ++linecol_;
}

/* ------------------------------------------------------------------------ */
/* 
 *   write out a UTF-8 string
 */
void CVmFormatter::buffer_string(VMG_ const char *txt)
{
    /* write out each character in the string */
    for ( ; utf8_ptr::s_getch(txt) != 0 ; txt += utf8_ptr::s_charsize(*txt))
        buffer_char(vmg_ utf8_ptr::s_getch(txt));
}

/*
 *   write out a wide unicode string 
 */
void CVmFormatter::buffer_wstring(VMG_ const wchar_t *txt)
{
    /* write out each wide character */
    for ( ; *txt != '\0' ; ++txt)
        buffer_char(vmg_ *txt);
}


/* ------------------------------------------------------------------------ */
/*
 *   Get the next wide unicode character in a UTF8-encoded string, and
 *   update the string pointer and remaining length.  Returns zero if no
 *   more characters are available in the string.  
 */
wchar_t CVmFormatter::next_wchar(const char **s, size_t *len)
{
    wchar_t ret;
    size_t charsize;

    /* if there's nothing left, return a null terminator */
    if (*len == 0)
        return 0;

    /* get this character */
    ret = utf8_ptr::s_getch(*s);

    /* advance the string pointer and length counter */
    charsize = utf8_ptr::s_charsize(**s);
    *len -= charsize;
    *s += charsize;

    /* return the result */
    return ret;
}

/* ------------------------------------------------------------------------ */
/*
 *   Turn HTML mode on or off 
 */
void CVmFormatter::set_html_mode(VMG_ int html_mode)
{
    /* 
     *   if we're not changing the mode, ignore it - make this check before
     *   doing any work so that we don't unnecessarily flush any buffers 
     */
    if ((html_mode && html_mode_)
        || (!html_mode && !html_mode_))
    {
        /* the state isn't changing, so there's nothing for us to do */
        return;
    }

    /*
     *   If we have an HTML target, flush buffers and notify the target of
     *   the new mode. 
     */
    if (html_target_)
    {
        /* 
         *   flush the underlying stream, so that the target processes all
         *   of the text up to this point in the old mode 
         */
        flush(vmg_ VM_NL_NONE);

        /* turn HTML mode on or off in the underlying OS-level stream */
        if (html_mode)
            start_html_in_os();
        else
            end_html_in_os();
    }

    /* remember the new mode internally */
    html_mode_ = html_mode;
}

/* ------------------------------------------------------------------------ */
/*
 *   Display a string of a given length.  The text is encoded as UTF-8
 *   characters.  
 */
int CVmFormatter::format_text(VMG_ const char *s, size_t slen)
{
    wchar_t c;
    int done = FALSE;

    /* get the first character */
    c = next_wchar(&s, &slen);

    /* if we have anything to show, show it */
    while (c != '\0')
    {
        /* 
         *   first, process the character through our built-in text-only HTML
         *   mini-parser, if our HTML mini-parser state indicates that we're
         *   in the midst of parsing a tag 
         */
        if (html_mode_flag_ != HTML_MODE_NORMAL)
        {
            /* run our HTML parsing until we finish the tag */
            c = resume_html_parsing(vmg_ c, &s, &slen);

            /* proceed with the next character */
            continue;
        }

        /* check for special characters */
        switch(c)
        {
        case 10:
            /* newline */
            flush(vmg_ VM_NL_NEWLINE);
            break;
                    
        case 9:
            /* tab - write an ordinary every-4-columns tab */
            write_tab(vmg_ 0, 4);
            break;

        case 0x000B:
            /* \b - blank line */
            write_blank_line(vmg0_);
            break;
                    
        case 0x0015:
            /* non-breaking space */
            if (html_target_ && html_mode_)
            {
                /* 
                 *   we're generating for an HTML target and we're in HTML
                 *   mode - generate the HTML non-breaking space 
                 */
                buffer_string(vmg_ "&nbsp;");
            }
            else
            {
                /* 
                 *   we're not in HTML mode - generate our internal quoted
                 *   space character 
                 */
                buffer_expchar(vmg_ QSPACE);
            }
            break;

        case 0x000F:
            /* capitalize next character */
            capsflag_ = TRUE;
            nocapsflag_ = FALSE;
            break;

        case 0x000E:
            /* un-capitalize next character */
            nocapsflag_ = TRUE;
            capsflag_ = FALSE;
            break;

        case '<':
        case '&':
            /* HTML markup-start character - process it */
            if (!html_target_ && html_mode_)
            {
                /*
                 *   We're in HTML mode, but the underlying target does
                 *   not accept HTML sequences.  It appears we're at the
                 *   start of an "&" entity or a tag sequence, so parse
                 *   it, remove it, and replace it (if possible) with a
                 *   text-only equivalent.  
                 */
                c = parse_html_markup(vmg_ c, &s, &slen);

                /* go back and process the next character */
                continue;
            }
            else
            {
                /* 
                 *   we're either not in HTML mode, or the underlying OS
                 *   renderer interprets HTML mark-up sequences -- in
                 *   either case, we don't need to perform any
                 *   interpretation; simply pass through the character as
                 *   though it were any other 
                 */
                goto normal_char;
            }
            break;

        default:
        normal_char:
            /* normal character - write it out */
            buffer_char(vmg_ c);
            break;
        }

        /* move on to the next character, unless we're finished */
        if (done)
            c = '\0';
        else
            c = next_wchar(&s, &slen);
    }

    /* success */
    return 0;
}

/* ------------------------------------------------------------------------ */
/*
 *   Initialize the display object 
 */
CVmConsole::CVmConsole()
{
    /* no script file yet */
    script_fp_ = 0;
    quiet_script_ = FALSE;

    /* no command log file yet */
    command_fp_ = 0;

    /* assume we'll double-space after each period */
    doublespace_ = TRUE;

    /* clear the debug flags */
    outwxflag_ = FALSE;

    /* presume we'll have no log stream */
    log_str_ = 0;
    log_enabled_ = FALSE;
}

/*
 *   Delete the display object 
 */
CVmConsole::~CVmConsole()
{
    /* close any active script file */
    if (script_fp_ != 0)
        osfcls(script_fp_);

    /* close any active command log file */
    close_command_log();

    /* delete the log stream if we have one */
    if (log_str_ != 0)
        delete log_str_;
}

/* ------------------------------------------------------------------------ */
/*
 *   Display a string of a given byte length 
 */
int CVmConsole::format_text(VMG_ const char *p, size_t len)
{
    int ret;

    /* presume we'll return success */
    ret = 0;

    /* if the debugger is showing watchpoints, suppress all output */
    if (outwxflag_)
        return ret;

    /* display the string */
    disp_str_->format_text(vmg_ p, len);

    /* if there's a log file, write to the log file as well */
    if (log_enabled_)
        log_str_->format_text(vmg_ p, len);

    /* return the result from displaying to the screen */
    return ret;
}

/* ------------------------------------------------------------------------ */
/*
 *   Set the text color 
 */
void CVmConsole::set_text_color(VMG_ os_color_t fg, os_color_t bg)
{
    /* set the color in our main display stream */
    disp_str_->set_text_color(vmg_ fg, bg);
}

/*
 *   Set the body color 
 */
void CVmConsole::set_body_color(VMG_ os_color_t color)
{
    /* set the color in the main display stream */
    disp_str_->set_os_body_color(color);
}

/* ------------------------------------------------------------------------ */
/*
 *   Display a blank line 
 */
void CVmConsole::write_blank_line(VMG0_)
{
    /* generate the newline to the standard display */
    disp_str_->write_blank_line(vmg0_);

    /* if we're logging, generate the newline to the log file as well */
    if (log_enabled_)
        log_str_->write_blank_line(vmg0_);
}


/* ------------------------------------------------------------------------ */
/*
 *   outcaps() - sets an internal flag which makes the next letter output
 *   a capital, whether it came in that way or not.  Set the same state in
 *   both formatters (standard and log).  
 */
void CVmConsole::caps()
{
    disp_str_->caps();
    if (log_enabled_)
        log_str_->caps();
}

/*
 *   outnocaps() - sets the next letter to a miniscule, whether it came in
 *   that way or not.  
 */
void CVmConsole::nocaps()
{
    disp_str_->nocaps();
    if (log_enabled_)
        log_str_->nocaps();
}

/*
 *   obey_whitespace() - sets the obey-whitespace mode 
 */
int CVmConsole::set_obey_whitespace(int f)
{
    int ret;

    /* note the original display stream status */
    ret = disp_str_->get_obey_whitespace();

    /* set the stream status */
    disp_str_->set_obey_whitespace(f);
    if (log_enabled_)
        log_str_->set_obey_whitespace(f);

    /* return the original status of the display stream */
    return ret;
}

/* ------------------------------------------------------------------------ */
/*
 *   Open a log file 
 */
int CVmConsole::open_log_file(const char *fname)
{
    /* if there's no log stream, we can't open a log file */
    if (log_str_ == 0)
        return 1;

    /* 
     *   Tell the log stream to open the file.  Set the log file's HTML
     *   source mode flag to the same value as is currently being used in
     *   the main display stream, so that it will interpret source markups
     *   the same way that the display stream is going to.  
     */
    return log_str_->open_log_file(fname, disp_str_->is_html_mode());
}

/*
 *   Close the log file 
 */
int CVmConsole::close_log_file()
{
    /* if there's no log stream, there's obviously no file open */
    if (log_str_ == 0)
        return 1;

    /* tell the log stream to close its file */
    return log_str_->close_log_file();
}

#if 0 //$$$
/*
 *   This code is currently unused.  However, I'm leaving it in for now -
 *   the algorithm takes a little thought, so it would be nicer to be able
 *   to uncomment the existing code should we ever need it in the future.  
 */

/* ------------------------------------------------------------------------ */
/*
 *   Write UTF-8 text explicitly to the log file.  This can be used to add
 *   special text (such as prompt text) that would normally be suppressed
 *   from the log file.  When more mode is turned off, we don't
 *   automatically copy text to the log file; any text that the caller
 *   knows should be in the log file during times when more mode is turned
 *   off can be explicitly added with this function.
 *   
 *   If nl is true, we'll add a newline at the end of this text.  The
 *   caller should not include any newlines in the text being displayed
 *   here.  
 */
void CVmConsole::write_to_logfile(VMG_ const char *txt, int nl)
{
    /* if there's no log file, there's nothing to do */
    if (logfp_ == 0)
        return;

    /* 
     *   convert the text from UTF-8 to the local character set and write
     *   the converted text to the log file 
     */
    while (*txt != '\0')
    {
        char local_buf[128];
        size_t src_bytes_used;
        size_t out_bytes;

        /* convert as much as we can (leaving room for a null terminator) */
        out_bytes = G_cmap_to_ui->map_utf8(local_buf, sizeof(local_buf) - 1,
                                           txt, strlen(txt), &src_bytes_used);

        /* null-terminate the result */
        local_buf[out_bytes] = '\0';
        
        /* write the converted text */
        os_fprintz(logfp_, local_buf);

        /* skip the text we were able to convert */
        txt += src_bytes_used;
    }

    /* add a newline if desired */
    if (nl)
    {
        /* add a normal newline */
        os_fprintz(logfp_, "\n");

        /* if the logfile is an html target, write an HTML line break */
        if (log_str_ != 0
            && log_str_->is_html_target()
            && log_str_->is_html_mode())
            os_fprintz(logfp_, "<BR HEIGHT=0>\n");
    }
}
#endif /* 0 */


/* ------------------------------------------------------------------------ */
/*
 *   Reset the MORE line counter.  This should be called whenever user
 *   input is read, since stopping to read user input makes it unnecessary
 *   to show another MORE prompt until the point at which input was
 *   solicited scrolls off the screen.  
 */
void CVmConsole::reset_line_count()
{
    /* reset the MORE counter in the display stream */
    disp_str_->reset_line_count();
}

/* ------------------------------------------------------------------------ */
/*
 *   Determine if HTML mode is active.  Returns true if so, false if not.
 *   Note that this merely indicates whether an "\H+" sequence is
 *   currently active -- this will return true after an "\H+" sequence,
 *   even on text-only interpreters.  
 */
int CVmConsole::is_html_mode()
{
    /* return the current HTML mode flag for the standard display stream */
    return disp_str_->is_html_mode();
}


/* ------------------------------------------------------------------------ */
/*
 *   Flush the output line.  We'll write to both the standard display and
 *   the log file, as needed.  
 */
void CVmConsole::flush(VMG_ vm_nl_type nl)
{
    /* flush the display stream */
    disp_str_->flush(vmg_ nl);

    /* flush the log stream, if we have an open log file */
    if (log_enabled_)
        log_str_->flush(vmg_ nl);
}

/* ------------------------------------------------------------------------ */
/*
 *   Clear our buffers
 */
void CVmConsole::empty_buffers(VMG0_)
{
    /* tell the formatter to clear its buffer */
    disp_str_->empty_buffers(vmg0_);

    /* same with the log stream, if applicable */
    if (log_enabled_)
        log_str_->empty_buffers(vmg0_);
}

/* ------------------------------------------------------------------------ */
/*
 *   Immediately update the display 
 */
void CVmConsole::update_display(VMG0_)
{
    /* update the display for the main display stream */
    disp_str_->update_display(vmg0_);
}

/* ------------------------------------------------------------------------ */
/*
 *   Open a script file 
 */
void CVmConsole::open_script_file(const char *fname, int quiet,
                                  int script_more_mode)
{
    /* close any existing script file */
    close_script_file();

    /* open the new file */
    script_fp_ = osfoprt(fname, OSFTCMD);
    
    /* 
     *   if we successfully opened the file, remember the quiet setting;
     *   otherwise, we're definitely not reading a quiet script because
     *   we're not reading a script at all 
     */
    quiet_script_ = (script_fp_ != 0 && quiet);

    /*
     *   If we successfully opened a script file, remember the original
     *   MORE mode, and set the MORE mode that the caller wants in effect
     *   while processing the script.  If we didn't successfully open a
     *   script, don't make any change to the MORE mode.  
     */
    if (script_fp_ != 0)
        pre_script_more_mode_ = set_more_state(script_more_mode);
}

/*
 *   Close the script file 
 */
int CVmConsole::close_script_file()
{
    /* if we have a file, close it */
    if (script_fp_ != 0)
    {
        /* close the file */
        osfcls(script_fp_);

        /* forget the script file */
        script_fp_ = 0;

        /* 
         *   we're not reading any script any more, so forget any
         *   quiet-script mode that's in effect 
         */
        quiet_script_ = FALSE;

        /* 
         *   return the MORE mode in effect before we started reading the
         *   script file 
         */
        return pre_script_more_mode_;
    }
    else
    {
        /* 
         *   there's no script file - just return the current MORE mode,
         *   since we're not making any changes 
         */
        return is_more_mode();
    }
}

/* ------------------------------------------------------------------------ */
/*
 *   Open a command log file 
 */
int CVmConsole::open_command_log(const char *fname)
{
    /* close any existing command log file */
    close_command_log();
    
    /* remember the filename */
    strcpy(command_fname_, fname);

    /* open the file */
    command_fp_ = osfopwt(fname, OSFTCMD);

    /* return success if we successfully opened the file */
    return (command_fp_ == 0);
}

/* 
 *   close the active command log file 
 */
int CVmConsole::close_command_log()
{
    /* if there's a command log file, close it */
    if (command_fp_ != 0)
    {
        /* close the file */
        osfcls(command_fp_);

        /* set its file type */
        os_settype(command_fname_, OSFTCMD);

        /* forget the file */
        command_fp_ = 0;
    }

    /* success */
    return 0;
}


/* ------------------------------------------------------------------------ */
/*
 *   Read a line of input from the console.  Fills in the buffer with a
 *   null-terminated string in the UTF-8 character set.  Returns zero on
 *   success, non-zero on end-of-file.  
 */
int CVmConsole::read_line(VMG_ char *buf, size_t buflen)
{
    /* cancel any previous interrupted input */
    read_line_cancel(vmg_ TRUE);

try_again:
    /* use the timeout version, with no timeout specified */
    switch(read_line_timeout(vmg_ buf, buflen, 0, FALSE))
    {
    case OS_EVT_LINE:
        /* success */
        return 0;

    case VMCON_EVT_END_SCRIPT:
        /* 
         *   end of script - we have no way to communicate this result back
         *   to our caller, so simply ignore the result and ask for another
         *   line 
         */
        goto try_again;

    default:
        /* anything else is an error */
        return 1;
    }
}


/* ------------------------------------------------------------------------ */
/*
 *   Static variables for input state.  We keep these statically, because we
 *   might need to use the values across a series of read_line_timeout calls
 *   if timeouts occur. 
 */

/* original 'more' mode, before input began */
static int S_old_more_mode;

/* flag: input is pending from an interrupted read_line_timeout invocation */
static int S_read_in_progress;

/* local buffer for reading input lines */
static char S_read_buf[256];


/*
 *   Read a line of input from the console, with an optional timeout value. 
 */
int CVmConsole::read_line_timeout(VMG_ char *buf, size_t buflen,
                                  unsigned long timeout, int use_timeout)
{
    int echo_text;
    char *outp;
    size_t outlen;
    int evt;
    int resuming;

    /* 
     *   presume we won't echo the text to the display; in most cases, it
     *   will be echoed to the display in the course of reading it from
     *   the keyboard 
     */
    echo_text = FALSE;

    /*
     *   If we're not resuming an interrupted read already in progress,
     *   initialize some display settings. 
     */
    if (!S_read_in_progress)
    {
        /* 
         *   Turn off MORE mode if it's on - we don't want a MORE prompt
         *   showing up in the midst of user input.  
         */
        S_old_more_mode = set_more_state(FALSE);

        /* 
         *   flush the output; don't start a new line, since we might have
         *   displayed a prompt that is to be on the same line with the user
         *   input 
         */
        flush_all(vmg0_);

        /* 
         *   reset the MORE line counter, since we're reading user input at
         *   the current point and shouldn't pause for a MORE prompt until
         *   the text we're reading has scrolled off the screen 
         */
        reset_line_count();

        /* if there's a script file, read from it */
        if (script_fp_ != 0)
        {
            /* try reading a line from the script file */
            if (read_line_from_script(S_read_buf, sizeof(S_read_buf)))
            {
                int was_quiet;

                /* note whether or not we were in quiet mode */
                was_quiet = quiet_script_;

                /* 
                 *   End of script file - return to reading from the
                 *   keyboard.  The return value from close_script_file() is
                 *   the MORE mode that was in effect before we started
                 *   reading the script file; use this when we restore the
                 *   enclosing MORE mode so that we restore the pre-script
                 *   MORE mode when we return.  
                 */
                S_old_more_mode = close_script_file();
                
                /* turn off MORE mode, in case we read from the keyboard */
                set_more_state(FALSE);
                
                /* flush any output we generated while reading the script */
                flush(vmg_ VM_NL_NONE);
                
                /* 
                 *   reset the MORE line counter, since we might have
                 *   generated output while reading the script file 
                 */
                reset_line_count();

                /* 
                 *   If we were in quiet mode, let the caller know we've
                 *   finished reading a script, so that the caller can set
                 *   up the display properly for reading from the keyboard.
                 *   
                 *   If we weren't in quiet mode, we'll simply proceed to
                 *   the normal keyboard reading; when not in quiet mode, no
                 *   special display fixup is needed.  
                 */
                if (was_quiet)
                {
                    /* return to the old MORE mode */
                    set_more_state(S_old_more_mode);

                    /* add a blank line to the log file, if necessary */
                    if (log_enabled_)
                        log_str_->print_to_os("\n");

                    /* note in the streams that we've read an input line */
                    disp_str_->note_input_line();
                    if (log_str_ != 0)
                        log_str_->note_input_line();

                    /* 
                     *   generate a synthetic "end of script" event to let
                     *   the caller know we're switching back to regular
                     *   keyboard reading 
                     */
                    return VMCON_EVT_END_SCRIPT;
                }
            }
            else
            {
                /* 
                 *   we got a line from the script file - if we're not in
                 *   quiet mode, make a note to echo the text to the display 
                 */
                if (!quiet_script_)
                    echo_text = TRUE;

                /* we've read a line */
                evt = OS_EVT_LINE;
            }
        }
    }

    /* 
     *   if reading was already in progress, we're resuming a previously
     *   interrupted read operation 
     */
    resuming = S_read_in_progress;

    /* reading is now in progress */
    S_read_in_progress = TRUE;

    /* 
     *   if we don't have a script file, or we're resuming an interrupted
     *   read operation, read from the keyboard 
     */
    if (script_fp_ == 0 || resuming)
    {
        /* read a line from the keyboard */
        evt = os_gets_timeout((uchar *)S_read_buf, sizeof(S_read_buf),
                              timeout, use_timeout);

        /*
         *   If that failed because timeout is not supported on this
         *   platform, and the caller didn't actually want to use a timeout,
         *   try again with an ordinary os_gets().  If they wanted to use a
         *   timeout, simply return the NOTIMEOUT indication to our caller.  
         */
        if (evt == OS_EVT_NOTIMEOUT && !use_timeout)
        {
            /* perform an ordinary untimed input */
            if (os_gets((uchar *)S_read_buf, sizeof(S_read_buf)) != 0)
            {
                /* success */
                evt = OS_EVT_LINE;
            }
            else
            {
                /* error reading input */
                evt = OS_EVT_EOF;
            }
        }

        /* 
         *   If we actually read a line, notify the display stream that we
         *   read text from the console - it might need to make some
         *   internal bookkeeping adjustments to account for the fact that
         *   we moved the write position around on the display.
         *   
         *   Don't note the input if we timed out, since we haven't finished
         *   reading the line yet in this case.  
         */
        if (evt == OS_EVT_LINE)
        {
            disp_str_->note_input_line();
            if (log_str_ != 0)
                log_str_->note_input_line();
        }
    }

    /* if we got an error, return it */
    if (evt == OS_EVT_EOF)
    {
        set_more_state(S_old_more_mode);
        return evt;
    }

    /* if we finished reading the line, do our line-finishing work */
    if (evt == OS_EVT_LINE)
        read_line_done(vmg0_);

    /* 
     *   Convert the text from the local UI character set to UTF-8.
     *   Reserve space in the output buffer for the null terminator. 
     */
    outp = buf;
    outlen = buflen - 1;
    G_cmap_from_ui->map(&outp, &outlen, S_read_buf, strlen(S_read_buf));

    /* add the null terminator */
    *outp = '\0';

    /* 
     *   If we need to echo the text (because we read it from a script
     *   file), do so now.
     */
    if (echo_text)
    {
        /* show the text */
        format_text(vmg_ buf);

        /* add a newline */
        format_text(vmg_ "\n");
    }

    /* return the event code */
    return evt;
}

/*
 *   Cancel an interrupted input. 
 */
void CVmConsole::read_line_cancel(VMG_ int reset)
{
    /* reset the underling OS layer */
    os_gets_cancel(reset);

    /* do our line-ending work */
    read_line_done(vmg0_);
}

/*
 *   Perform line-ending work.  This is used when we finish reading a line
 *   in read_line_timeout(), or when we cancel an interrupted line, thus
 *   finishing the line, in read_line_cancel(). 
 */
void CVmConsole::read_line_done(VMG0_)
{
    /* if we have a line in progress, finish it off */
    if (S_read_in_progress)
    {
        /* set the original 'more' mode */
        set_more_state(S_old_more_mode);

        /* 
         *   Write the input line, followed by a newline, to the log file.
         *   Note that the text is still in the local character set, so we
         *   can write it directly to the log file.  
         */
        if (log_enabled_)
        {
            log_str_->print_to_os(S_read_buf);
            log_str_->print_to_os("\n");
        }
        
        /*
         *   If we have a command log file, log the command (preceded by a
         *   ">" character, to indicate that it's a command, and followed by
         *   a newline) to the command log.  
         */
        if (command_fp_ != 0)
        {
            os_fprintz(command_fp_, ">");
            os_fprintz(command_fp_, S_read_buf);
            os_fprintz(command_fp_, "\n");
        }

        /* note in the streams that we've read an input line */
        disp_str_->note_input_line();
        if (log_str_ != 0)
            log_str_->note_input_line();

        /* clear the in-progress flag */
        S_read_in_progress = FALSE;
    }
}

/*
 *   Read a line of text from the script file, if there is one.  Returns
 *   zero on success, non-zero if we reach the end of the script file or
 *   encounter any other error. 
 */
int CVmConsole::read_line_from_script(char *buf, size_t buflen)
{
    /* if there's no script file, return failure */
    if (script_fp_ == 0)
        return 1;

    /* keep going until we find a line that we like */
    for (;;)
    {
        char c;

        /* 
         *   Read the next character of input.  If it's not a newline,
         *   there's more text on the same line, so read the rest and
         *   determine what to do. 
         */
        c = osfgetc(script_fp_);
        if (c != '\n' && c != '\r')
        {
            /* there's more on this line - read the rest of the line */
            if (osfgets(buf, buflen, script_fp_) == 0)
            {
                /* end of file - return failure */
                return 1;
            }

            /* 
             *   if the line starts with '>', it's a command line;
             *   otherwise, it's a comment or something else, in which
             *   case we'll ignore it and keep looking for a line starting
             *   with '>' 
             */
            if (c == '>')
            {
                size_t len;
                
                /* 
                 *   if there are any trailing newline characters in the
                 *   buffer, remove them
                 */
                len = strlen(buf);
                while (len > 0
                       && (buf[len - 1] == '\n' || buf[len - 1] == '\r'))
                    buf[--len] = '\0';

                /* return success */
                return 0;
            }
        }
        else if (c == EOF)
        {
            /* end of file - return failure */
            return 1;
        }
    }
}

/* ------------------------------------------------------------------------ */
/*
 *   Main System Console 
 */

/*
 *   create 
 */
CVmConsoleMain::CVmConsoleMain()
{
    /* create the system banner manager */
    banner_manager_ = new CVmBannerManager();

    /* create and initialize our display stream */
    main_disp_str_ = new CVmFormatterMain(this, 256);
    main_disp_str_->init();

    /* initially send text to the main display stream */
    disp_str_ = main_disp_str_;

    /* 
     *   Create and initialize our log stream.  The main console always has a
     *   log stream, even when it's not in use, so that we can keep the log
     *   stream's state synchronized with the display stream in preparation
     *   for activation.  
     */
    log_str_ = new CVmFormatterLog(this);
    log_str_->init();

    /* 
     *   the log stream is initially enabled (this is separate from the log
     *   file being opened; it merely indicates that we send output
     *   operations to the log stream for processing) 
     */
    log_enabled_ = TRUE;

    /* we don't have a statusline formatter until asked for one */
    statline_str_ = 0;
}

/*
 *   delete 
 */
CVmConsoleMain::~CVmConsoleMain()
{
    /* delete the system banner manager */
    delete banner_manager_;

    /* delete the display stream */
    delete main_disp_str_;

    /* delete the statusline stream, if we have one */
    if (statline_str_ != 0)
        delete statline_str_;
}

/*
 *   Set statusline mode 
 */
void CVmConsoleMain::set_statusline_mode(VMG_ int mode)
{
    CVmFormatterDisp *str;

    /* 
     *   if we're switching into statusline mode, and we don't have a
     *   statusline stream yet, create one 
     */
    if (mode && statline_str_ == 0)
    {
        /* create and initialize the statusline stream */
        statline_str_ = new CVmFormatterStatline(this);
        statline_str_->init();
    }

    /* get the stream selected by the new mode */
    if (mode)
        str = statline_str_;
    else
        str = main_disp_str_;

    /* if this is already the active stream, we have nothing more to do */
    if (str == disp_str_)
        return;

    /* 
     *   always transfer the HTML mode from the outgoing stream to the
     *   incoming stream 
     */
    str->copy_html_mode_from(disp_str_);

    /* make the new stream current */
    disp_str_ = str;

    /* 
     *   check which mode we're switching to, so we can do some extra work
     *   specific to each mode 
     */
    if (mode)
    {
        /* 
         *   we're switching to the status line, so disable the log stream -
         *   statusline text is never sent to the log, since the log reflects
         *   only what was displayed in the main text area 
         */
        log_enabled_ = FALSE;
    }
    else
    {
        /*
         *   we're switching back to the main stream, so flush the statusline
         *   so we're sure the statusline text is displayed 
         */

        /* end the line */
        statline_str_->format_text(vmg_ "\n", 1);

        /* flush output */
        statline_str_->flush(vmg_ VM_NL_NONE);

        /* re-enable the log stream, if we have one */
        if (log_str_ != 0)
            log_enabled_ = TRUE;
    }

    /* switch at the OS layer */
    os_status(mode);
}

/*
 *   Flush everything 
 */
void CVmConsoleMain::flush_all(VMG0_)
{
    /* flush our primary console */
    flush(vmg_ VM_NL_NONE);

    /* flush each banner we're controlling */
    banner_manager_->flush_all(vmg0_);
}


/* ------------------------------------------------------------------------ */
/*
 *   Banner manager 
 */

/* initialize */
CVmBannerManager::CVmBannerManager()
{
    size_t i;

    /* allocate an initial array of banner window pointers */
    banners_max_ = 32;
    banners_ = (CVmBannerItem **)t3malloc(banners_max_ * sizeof(*banners_));

    /* none of the banner pointers are valid yet */
    for (i = 0 ; i < banners_max_ ; ++i)
        banners_[i] = 0;
}

/* delete */
CVmBannerManager::~CVmBannerManager()
{
    size_t i;

    /* delete each remaining banner */
    for (i = 0 ; i < banners_max_ ; ++i)
    {
        /* if this banner is still valid, delete it */
        if (banners_[i] != 0)
            delete_or_orphan_banner(i + 1, TRUE);
    }

    /* delete the banner pointer array */
    t3free(banners_);
}

/*
 *   Create a banner 
 */
int CVmBannerManager::create_banner(VMG_ int where, int other_id,
                                    int wintype, int align,
                                    int siz, int siz_units,
                                    unsigned long style)
{
    size_t slot;
    void *handle;
    void *other_handle;
    CVmBannerItem *item;

    /* get the 'other' handle, if we need it for the 'where' */
    switch(where)
    {
    case OS_BANNER_BEFORE:
    case OS_BANNER_AFTER:
        /* retrieve the handle for the other_id */
        other_handle = get_os_handle(other_id);
        break;

    default:
        /* we don't need 'other' for other 'where' modes */
        other_handle = 0;
        break;
    }

    /* try creating the OS-level banner window */
    handle = os_banner_create(where, other_handle, wintype,
                              align, siz, siz_units, style);

    /* if we couldn't create the OS-level window, return failure */
    if (handle == 0)
        return 0;

    /* create the new banner item to keep track of the banner */
    item = new CVmBannerItem(handle, wintype);

    /* find a free slot to store our banner item */
    for (slot = 0 ; slot < banners_max_ ; ++slot)
    {
        /* if this one is free, use it */
        if (banners_[slot] == 0)
            break;
    }

    /* if we didn't find a free slot, extend the array */
    if (slot == banners_max_)
    {
        size_t i;

        /* allocate a larger array */
        banners_max_ += 32;
        banners_ = (CVmBannerItem **)
                   t3realloc(banners_, banners_max_ * sizeof(*banners_));

        /* clear out the newly-allocated slots */
        for (i = slot ; i < banners_max_ ; ++i)
            banners_[i] = 0;
    }

    /* store the new banner item in our pointer array */
    banners_[slot] = item;

    /* 
     *   return the banner window handle - this is simply the index in the
     *   banner arrray of the slot incremented by one, since 0 is reserved as
     *   the invalid handle 
     */
    return (int)slot + 1;
}

/*
 *   Delete or orphan a banner window.  Deleting and orphaning both sever
 *   all ties from the banner manager (and thus from the T3 program) to the
 *   banner.  Deleting a banner actually gets deletes it at the OS level;
 *   orphaning the banner severs our ties, but hands the banner over to the
 *   OS to do with as it pleases.  On some implementations, the OS will
 *   continue to display the banner after it's orphaned to allow the final
 *   display configuration to remain visible even after the program has
 *   terminated.  
 */
void CVmBannerManager::delete_or_orphan_banner(int banner, int orphan)
{
    CVmBannerItem **slotp;
    void *handle;

    /* if the banner is invalid, ignore the request */
    if (banner < 1
        || (size_t)banner > banners_max_
        || banners_[banner - 1] == 0)
        return;

    /* get our slot pointer */
    slotp = &banners_[banner - 1];

    /* get the OS-level banner handle */
    handle = (*slotp)->get_os_handle();

    /* delete the banner item */
    delete *slotp;

    /* clear the slot */
    *slotp = 0;

    /* delete the OS-level banner */
    if (orphan)
        os_banner_orphan(handle);
    else
        os_banner_delete(handle);
}

/*
 *   Get the OS-level handle for the given banner 
 */
void *CVmBannerManager::get_os_handle(int banner)
{
    /* if the banner is invalid, return failure */
    if (banner < 1
        || (size_t)banner > banners_max_
        || banners_[banner - 1] == 0)
        return 0;

    /* return the handle from the slot */
    return banners_[banner - 1]->get_os_handle();
}

/*
 *   Get the CVmConsole object for the given banner 
 */
CVmConsoleBanner *CVmBannerManager::get_console(int banner)
{
    /* if the banner is invalid, return failure */
    if (banner < 1
        || (size_t)banner > banners_max_
        || banners_[banner - 1] == 0)
        return 0;

    /* return the handle from the slot */
    return banners_[banner - 1]->get_console();
}

/*
 *   Flush all banners 
 */
void CVmBannerManager::flush_all(VMG0_)
{
    size_t slot;

    /* flush each banner */
    for (slot = 0 ; slot < banners_max_ ; ++slot)
    {
        /* if this slot has a valid banner, flush it */
        if (banners_[slot] != 0)
            banners_[slot]->get_console()->flush(vmg_ VM_NL_NONE);
    }

}

/* ------------------------------------------------------------------------ */
/*
 *   Banner Manager Item 
 */

/* create */
CVmBannerItem::CVmBannerItem(void *os_handle, int win_type)
{
    /* create the console */
    console_ = new CVmConsoleBanner(os_handle, win_type);
}

/* delete */
CVmBannerItem::~CVmBannerItem()
{
    /* delete our underlying console */
    delete console_;
}

/* get my OS-level banner handle */
void *CVmBannerItem::get_os_handle() const
{
    /* retrieve the banner from my console object */
    return console_->get_banner_handle();
}

/* ------------------------------------------------------------------------ */
/*
 *   Banner Window Console 
 */
CVmConsoleBanner::CVmConsoleBanner(void *banner_handle, int win_type)
{
    CVmFormatterBanner *str;
    os_banner_info_t info;

    /* remember our OS-level banner handle */
    banner_ = banner_handle;

    /* get osifc-level information on the banner */
    if (!os_banner_getinfo(banner_, &info))
        info.os_line_wrap = FALSE;

    /* 
     *   If it's a text grid window, don't do any line wrapping.  Text grids
     *   simply don't have any line wrapping, so we don't want to impose any
     *   at the formatter level.  Set the formatter to "os line wrap" mode,
     *   to indicate that the formatter doesn't do wrapping - even though
     *   the underlying OS banner window won't do any wrapping either, the
     *   lack of line wrapping counts as OS handling of line wrapping.  
     */
    if (win_type == OS_BANNER_TYPE_TEXTGRID)
        info.os_line_wrap = TRUE;

    /* create and initialize our display stream */
    disp_str_ = str = new CVmFormatterBanner(banner_handle, this, win_type);
    str->init(info.os_line_wrap);

    /* remember our window type */
    win_type_ = win_type;
}

/*
 *   Deletion 
 */
CVmConsoleBanner::~CVmConsoleBanner()
{
    /* delete our display stream */
    delete disp_str_;
}

/*
 *   Get banner information 
 */
int CVmConsoleBanner::get_banner_info(os_banner_info_t *info)
{
    int ret;
    
    /* get the OS-level information */
    ret = os_banner_getinfo(banner_, info);

    /* make some adjustments if we got valid information back */
    if (ret)
    {
        /* 
         *   check the window type for further adjustments we might need to
         *   make to the data returned from the OS layer 
         */
        switch(win_type_)
        {
        case OS_BANNER_TYPE_TEXTGRID:
            /* 
             *   text grids don't support <TAB> alignment, even if the
             *   underlying OS banner says we do, because we simply don't
             *   support <TAB> (or any other HTML markups) in a text grid
             *   window 
             */
            info->style &= ~OS_BANNER_STYLE_TAB_ALIGN;
            break;

        default:
            /* other types don't require any adjustments */
            break;
        }
    }

    /* return the success indication */
    return ret;
}

