// Copyright 2017 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef ASH_SYSTEM_PALETTE_PALETTE_WELCOME_BUBBLE_H_
#define ASH_SYSTEM_PALETTE_PALETTE_WELCOME_BUBBLE_H_

#include "ash/ash_export.h"
#include "ash/session/session_observer.h"
#include "base/macros.h"
#include "base/optional.h"
#include "ui/views/pointer_watcher.h"
#include "ui/views/widget/widget_observer.h"

class PrefRegistrySimple;
class PrefService;

namespace views {
class Widget;
}  // namespace views

namespace ash {
class PaletteTray;

// The PaletteWelcomeBubble handles displaying a warm welcome bubble letting
// users know about the PaletteTray the first time a stylus is ejected, or if an
// external stylus is detected. PaletteTray controls the visibility of the
// bubble.
class ASH_EXPORT PaletteWelcomeBubble : public SessionObserver,
                                        public views::WidgetObserver,
                                        public views::PointerWatcher {
 public:
  explicit PaletteWelcomeBubble(PaletteTray* tray);
  ~PaletteWelcomeBubble() override;

  static void RegisterProfilePrefs(PrefRegistrySimple* registry);

  // Show the welcome bubble iff it has not been shown before. |shown_by_stylus|
  // is true if the call is generated by a stylus-down event.
  void ShowIfNeeded(bool shown_by_stylus);

  // Set the pref which stores whether the bubble has been shown before as true.
  // The bubble will not be shown after this is called.
  void MarkAsShown();

  bool bubble_shown() { return bubble_view_ != nullptr; }

  // SessionObserver:
  void OnActiveUserPrefServiceChanged(PrefService* pref_service) override;

  // views::WidgetObserver:
  void OnWidgetClosing(views::Widget* widget) override;

  // Returns the bounds of the bubble view if it exists.
  base::Optional<gfx::Rect> GetBubbleBoundsForTest();

 private:
  friend class PaletteWelcomeBubbleTest;
  class WelcomeBubbleView;

  void Show(bool shown_by_stylus);

  // Hides the welcome bubble.
  void Hide();

  // views::PointerWatcher:
  void OnPointerEventObserved(const ui::PointerEvent& event,
                              const gfx::Point& location_in_screen,
                              gfx::NativeView target) override;

  // Flag which determines whether a pointer event should be ignored. This is
  // set when this is first registered as a pointer watcher, so the same event
  // which caused the bubble to show does not also cause the bubble to be
  // hidden.
  bool ignore_stylus_event_ = false;

  // The PaletteTray this bubble is associated with. Serves as the anchor for
  // the bubble. Not owned.
  PaletteTray* tray_ = nullptr;

  PrefService* active_user_pref_service_ = nullptr;  // Not owned.

  WelcomeBubbleView* bubble_view_ = nullptr;

  DISALLOW_COPY_AND_ASSIGN(PaletteWelcomeBubble);
};

}  // namespace ash

#endif  // ASH_SYSTEM_PALETTE_PALETTE_WELCOME_BUBBLE_H_
